//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

#include <string>
#include <sstream>
#include <iomanip>
#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>
#include "shaders.h"
#include "teapot.h"
#include "sphere.h"
#include "materials.h"
#include "text.h"

//////////////////////////////////////////////////////////////////////
// rozmiary bryy obcinania
//////////////////////////////////////////////////////////////////////
GLfloat left = -4.0f;
GLfloat right = 4.0f;
GLfloat bottom = -4.0f;
GLfloat top = 4.0f;
GLfloat near = 6.0f;
GLfloat far = 14.0f;

//////////////////////////////////////////////////////////////////////
// wektory normalne wierzchokw trjktw tworzcych ciany pokoju
//////////////////////////////////////////////////////////////////////
GLfloat roomNormal[5*6*3] =
{
    // podoga
    0.0f, 1.0f, 0.0f,
    0.0f, 1.0f, 0.0f,
    0.0f, 1.0f, 0.0f,
    0.0f, 1.0f, 0.0f,
    0.0f, 1.0f, 0.0f,
    0.0f, 1.0f, 0.0f,

    // lewa ciana
    1.0f, 0.0f, 0.0f,
    1.0f, 0.0f, 0.0f,
    1.0f, 0.0f, 0.0f,
    1.0f, 0.0f, 0.0f,
    1.0f, 0.0f, 0.0f,
    1.0f, 0.0f, 0.0f,

    // prawa ciana
    -1.0f, 0.0f, 0.0f,
    -1.0f, 0.0f, 0.0f,
    -1.0f, 0.0f, 0.0f,
    -1.0f, 0.0f, 0.0f,
    -1.0f, 0.0f, 0.0f,
    -1.0f, 0.0f, 0.0f,

    // sufit
    0.0f, -1.0f, 0.0f,
    0.0f, -1.0f, 0.0f,
    0.0f, -1.0f, 0.0f,
    0.0f, -1.0f, 0.0f,
    0.0f, -1.0f, 0.0f,
    0.0f, -1.0f, 0.0f,

    // tylna ciana
    0.0f, 0.0f, 1.0f,
    0.0f, 0.0f, 1.0f,
    0.0f, 0.0f, 1.0f,
    0.0f, 0.0f, 1.0f,
    0.0f, 0.0f, 1.0f,
    0.0f, 0.0f, 1.0f
};

//////////////////////////////////////////////////////////////////////
// wsprzdne wierzchokw trjktw tworzcych ciany pokoju;
// wsprzdne w przestrzeni obserwatora
//////////////////////////////////////////////////////////////////////
GLfloat roomPosition[5*6*4] =
{
    // podoga
    left, bottom, -near, 1.0f,
    right, bottom, -near, 1.0f,
    right, bottom, -far + 1.0f, 1.0f,
    left, bottom, -near, 1.0f,
    right, bottom, -far + 1.0f, 1.0f,
    left, bottom, -far + 1.0f, 1.0f,

    // lewa ciana
    left, bottom, -near, 1.0f,
    left, bottom, -far + 1.0f, 1.0f,
    left, top, -far + 1.0f, 1.0f,
    left, bottom, -near, 1.0f,
    left, top, -far + 1.0f, 1.0f,
    left, top, -near, 1.0f,

    // prawa ciana
    right, bottom, -near, 1.0f,
    right, top, -near, 1.0f,
    right, top, -far + 1.0f, 1.0f,
    right, bottom, -near, 1.0f,
    right, top, -far + 1.0f, 1.0f,
    right, bottom, -far + 1.0f, 1.0f,

    // sufit
    left, top, -near, 1.0f,
    left, top, -far + 1.0f, 1.0f,
    right, top, -far + 1.0f, 1.0f,
    left, top, -near, 1.0f,
    right, top, -far + 1.0f, 1.0f,
    right, top, -near, 1.0f,

    // tylna ciana
    left, bottom, -far + 1.0f, 1.0f,
    right, bottom, -far + 1.0f, 1.0f,
    right, top, -far + 1.0f, 1.0f,
    left, bottom, -far + 1.0f, 1.0f,
    right, top, -far + 1.0f, 1.0f,
    left, top, -far + 1.0f, 1.0f
};

//////////////////////////////////////////////////////////////////////
// macierz rzutowania
//////////////////////////////////////////////////////////////////////
glm::mat4x4 projectionMatrix;

//////////////////////////////////////////////////////////////////////
// wspczynniki skalowania obiektu
//////////////////////////////////////////////////////////////////////
GLfloat scale = 0.75f;

//////////////////////////////////////////////////////////////////////
// kty obrotu obiektu
//////////////////////////////////////////////////////////////////////
GLfloat rotateX = 0.0f;
GLfloat rotateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// przesunicie obiektu
//////////////////////////////////////////////////////////////////////
GLfloat translateX = 0.0f;
GLfloat translateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// pooenie rda wiata punktowego we wsprzdnych obserwatora
//////////////////////////////////////////////////////////////////////
glm::vec4 lightPosition( 3.0f, 3.0f, 3.0f - (near+far)/2.0f, 1.0f );

//////////////////////////////////////////////////////////////////////
// wybrany materia
//////////////////////////////////////////////////////////////////////
int material = MTL_BRASS;

//////////////////////////////////////////////////////////////////////
// numeracja obiektw programu
//////////////////////////////////////////////////////////////////////
enum
{
    OBJECT,                 // czajnik i ciany
    OBJECT_SHADOW,          // czajnik i ciany w cieniu
    DEPTH_PASS_SHADOW,      // cie metod depth-pass
    DEPTH_FAIL_SHADOW,      // cie metod depth-fail
    LIGHT_POINT,            // pooenie rda wiata
    PROGRAM_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw programu
//////////////////////////////////////////////////////////////////////
GLuint program[PROGRAM_SIZE];

//////////////////////////////////////////////////////////////////////
// numeracja obiektw bufora wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    TEAPOT_POSITION,
    TEAPOT_NORMAL,
    LIGHT_POINT_POSITION,
    ROOM_POSITION,
    ROOM_NORMAL,
    VERTEX_BUFFER_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw bufora z danymi tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexBuffer[VERTEX_BUFFER_SIZE];

//////////////////////////////////////////////////////////////////////
// numeracja obiektw tablic wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    TEAPOT,
    // LIGHT_POINT,
    ROOM = 2,
    TEAPOT_ADJACENCY,
    VERTEX_ARRAY_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexArray[VERTEX_ARRAY_SIZE];

//////////////////////////////////////////////////////////////////////
// numeracja obiektw bufora wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    TEAPOT_INDICES,
    TEAPOT_ADJACENCY_INDICES,
    LIGHT_POINT_INDICES,
    INDICES_BUFFER_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw bufora z danymi tablic
// indeksw wierzchokw obiektw
//////////////////////////////////////////////////////////////////////
GLuint indicesBuffer[INDICES_BUFFER_SIZE];

//////////////////////////////////////////////////////////////////////
// stae okrelajce tryb renderingu bryy cienia
//////////////////////////////////////////////////////////////////////
enum
{
    DEPTH_PASS_SHADOW_VOLUME,
    DEPTH_FAIL_SHADOW_VOLUME
    // DEPTH_PASS_SHADOW
    // DEPTH_FAIL_SHADOW
};

//////////////////////////////////////////////////////////////////////
// tryb renderingu cienia
//////////////////////////////////////////////////////////////////////
int shadowMode = DEPTH_PASS_SHADOW;

//////////////////////////////////////////////////////////////////////
// numery pooenia poszczeglnych atrybutw wierzchokw
//////////////////////////////////////////////////////////////////////
#define POSITION 0
#define NORMAL   1

//////////////////////////////////////////////////////////////////////
// rysowanie czajnika
// prg - numer obiektu programu z tablicy program
// modelViewMatrix - macierz modelu-widoku
//////////////////////////////////////////////////////////////////////
void DrawTeapot( const GLuint prg, glm::mat4x4 &modelViewMatrix )
{
    // odwrcona macierz modelu-widoku niezbdna do przeksztace
    // do ukadu wsprzdnych obiektu
    glm::mat4x4 modelViewMatrixInverse( glm::inverse( modelViewMatrix ) );

    // przeksztacenie pooenia obserwatora do ukadu wsprzdnych obiektu
    glm::vec4 eyePosition( 0.0f, 0.0f, 0.0f, 1.0f );
    eyePosition = modelViewMatrixInverse * eyePosition;

    // przeksztacenie pooenia rda wiata do ukadu wsprzdnych obiektu
    glm::vec4 inverseLightPosition = modelViewMatrixInverse * lightPosition;

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[TEAPOT] );

    // wczenie programu
    glUseProgram( program[prg] );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    glm::mat4x4 modelViewProjectionMatrix = projectionMatrix * modelViewMatrix;
    glUniformMatrix4fv( glGetUniformLocation( program[prg], "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // zaadowanie pooenia rda wiata i pooenia obserwatora w ukadzie wsprzdnych obiektu
    glUniform4fv( glGetUniformLocation( program[prg], "lightSource[0].position" ), 1, glm::value_ptr( inverseLightPosition ) );
    glUniform4fv( glGetUniformLocation( program[prg], "eyePosition" ), 1, glm::value_ptr( eyePosition ) );

    // zaadowanie numeru wybranego materiau
    glUniform1i( glGetUniformLocation( program[prg], "material" ), material );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawElements( GL_TRIANGLES, TEAPOT_INDICES_COUNT * 3, GL_UNSIGNED_INT, NULL );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wyczenie programu
    glUseProgram( 0 );
}

//////////////////////////////////////////////////////////////////////
// rysowanie cian pokoju
// prg - numer obiektu programu z tablicy program
//////////////////////////////////////////////////////////////////////
void DrawRoom( const GLuint prg )
{
    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[ROOM] );

    // wczenie programu
    glUseProgram( program[prg] );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    // z uwagi na to, e wsprzdne wierzchokw trjktw tworzcych ciany
    // pokoju s ju w przestrzeni obserwatora, nie ma potrzeby mnoenia macierzy
    // rzutowania przez macierz modelu-widoku
    glUniformMatrix4fv( glGetUniformLocation( program[prg], "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( projectionMatrix ) );

    // pooenie obserwatora w ukadzie wsprzdnych obserwatora
    GLfloat eyePosition[4] = { 0.0f, 0.0f, 0.0f, 1.0f };

    // zaadowanie pooenia rda wiata i pooenia obserwatora w ukadzie wsprzdnych obserwatora
    glUniform4fv( glGetUniformLocation( program[prg], "lightSource[0].position" ), 1, glm::value_ptr( lightPosition ) );
    glUniform4fv( glGetUniformLocation( program[prg], "eyePosition" ), 1, eyePosition );

    // zaadowanie numeru wybranego materiau
    glUniform1i( glGetUniformLocation( program[prg], "material" ), MTL_DEFAULT );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawArrays( GL_TRIANGLES, 0, 30 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wyczenie programu
    glUseProgram( 0 );
}

//////////////////////////////////////////////////////////////////////
// rysowanie bryy cienia
// prg - numer obiektu programu z tablicy program
// modelViewMatrix - macierz modelu-widoku
//////////////////////////////////////////////////////////////////////
void DrawShadowVolume( const GLuint prg, glm::mat4x4 &modelViewMatrix )
{
    // odwrcona macierz modelu-widoku niezbdna do przeksztace
    // do ukadu wsprzdnych obiektu
    glm::mat4x4 modelViewMatrixInverse( glm::inverse( modelViewMatrix ) );

    // przeksztacenie pooenia obserwatora do ukadu wsprzdnych obiektu
    glm::vec4 eyePosition( 0.0f, 0.0f, 0.0f, 1.0f );
    eyePosition = modelViewMatrixInverse * eyePosition;

    // przeksztacenie pooenia rda wiata do ukadu wsprzdnych obiektu
    glm::vec4 inverseLightPosition = modelViewMatrixInverse * lightPosition;

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[TEAPOT_ADJACENCY] );

    // wczenie programu
    glUseProgram( program[prg] );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    glm::mat4x4 modelViewProjectionMatrix = projectionMatrix * modelViewMatrix;
    glUniformMatrix4fv( glGetUniformLocation( program[prg], "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // zaadowanie pooenia rda wiata w ukadzie wsprzdnych obiektu
    glUniform4fv( glGetUniformLocation( program[prg], "lightPosition" ), 1, glm::value_ptr( inverseLightPosition ) );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawElements( GL_TRIANGLES_ADJACENCY, TEAPOT_INDICES_COUNT * 6, GL_UNSIGNED_INT, NULL );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wyczenie programu
    glUseProgram( 0 );
}

//////////////////////////////////////////////////////////////////////
// rysowanie rda wiata
//////////////////////////////////////////////////////////////////////
void DrawLight()
{
    // transformacje pooenia rda wiata punktowego
    glm::mat4x4 modelViewMatrix = glm::mat4x4( 1.0 );
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( lightPosition ) );
    modelViewMatrix = glm::scale( modelViewMatrix, glm::vec3( 0.2f, 0.2f, 0.2f ) );

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[LIGHT_POINT] );

    // wczenie programu
    glUseProgram( program[LIGHT_POINT] );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    glm::mat4x4 modelViewProjectionMatrix = projectionMatrix * modelViewMatrix;
    glUniformMatrix4fv( glGetUniformLocation( program[LIGHT_POINT], "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawElements( GL_TRIANGLES, SPHERE_LOW_INDICES_COUNT * 3, GL_UNSIGNED_INT, NULL );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );
}

//////////////////////////////////////////////////////////////////////
// generowanie sceny 3D z cieniem metod depth-pass
// modelViewMatrix - macierz modelu-widoku
//////////////////////////////////////////////////////////////////////
void DepthPassShadow( glm::mat4x4 &modelViewMatrix )
{
    // narysowanie sceny przy wczonym owietleniu otaczajcym
    DrawRoom( OBJECT_SHADOW );
    DrawTeapot( OBJECT_SHADOW, modelViewMatrix );

    // narysowanie przednich i tylnych cian bryy cienia przy
    // wczonym tecie bufora szablonu i wyczonym zapisie
    // do bufora gbokoci oraz bufora kolorw; przy pozytywnym
    // tecie bufora gbokoci dla przedniej ciany zawarto
    // bufora szablonu zwikszana jest o 1, a przy pozytywnym
    // tecie dla tylnej ciany zawarto zmniejszana jest o 1
    glColorMask( GL_FALSE, GL_FALSE, GL_FALSE, GL_FALSE );
    glEnable( GL_STENCIL_TEST );
    glDepthMask( GL_FALSE );
    glStencilFuncSeparate( GL_FRONT_AND_BACK, GL_ALWAYS, 0, ~0 );
    glStencilOpSeparate( GL_FRONT, GL_KEEP, GL_KEEP, GL_INCR_WRAP );
    glStencilOpSeparate( GL_BACK, GL_KEEP, GL_KEEP, GL_DECR_WRAP );
    DrawShadowVolume( DEPTH_PASS_SHADOW, modelViewMatrix );

    // na koniec narysowanie waciwej sceny 3D z penym owietleniem;
    // nie s rysowane jedynie te elementy sceny, ktre pokrywa cie
    glDepthMask( GL_TRUE );
    glColorMask( GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE );
    glDepthFunc( GL_LEQUAL );
    glStencilOpSeparate( GL_FRONT_AND_BACK, GL_KEEP, GL_KEEP, GL_KEEP );
    glStencilFuncSeparate( GL_FRONT_AND_BACK, GL_EQUAL, 0, ~0 );
    DrawRoom( OBJECT );
    DrawTeapot( OBJECT, modelViewMatrix );
    glDepthFunc( GL_LESS );
    glDisable( GL_STENCIL_TEST );
}

//////////////////////////////////////////////////////////////////////
// generowanie sceny 3D z cieniem metod depth-fail
// modelViewMatrix - macierz modelu-widoku
//////////////////////////////////////////////////////////////////////
void DepthFailShadow( glm::mat4x4 &modelViewMatrix )
{
    // narysowanie sceny przy wczonym owietleniu otaczajcym
    DrawRoom( OBJECT_SHADOW );
    DrawTeapot( OBJECT_SHADOW, modelViewMatrix );

    // narysowanie tylnych i przednich cian bryy cienia przy
    // wczonym tecie bufora szablonu i wyczonym zapisie
    // do bufora gbokoci; przy negatywnym tecie bufora gbokoci
    // dla tylnej ciany zawarto bufora szablonu zwikszana
    // jest o 1; przy negatywnym tecie bufora gbokoci zawarto
    // bufora szablonu zmniejszana jest o 1
    glColorMask( GL_FALSE, GL_FALSE, GL_FALSE, GL_FALSE );
    glEnable( GL_STENCIL_TEST );
    glDepthMask( GL_FALSE );
    glStencilFuncSeparate( GL_FRONT_AND_BACK, GL_ALWAYS, 0, ~0 );
    glStencilOpSeparate( GL_FRONT, GL_KEEP, GL_DECR_WRAP, GL_KEEP );
    glStencilOpSeparate( GL_BACK, GL_KEEP, GL_INCR_WRAP, GL_KEEP );
    glEnable( GL_DEPTH_CLAMP );
    DrawShadowVolume( DEPTH_FAIL_SHADOW, modelViewMatrix );
    glDisable( GL_DEPTH_CLAMP );

    // na koniec narysowanie waciwej sceny 3D z penym owietleniem;
    // nie s rysowane jedynie te elementy sceny, ktre pokrywa cie
    glDepthMask( GL_TRUE );
    glColorMask( GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE );
    glDepthFunc( GL_LEQUAL);
    glStencilOpSeparate( GL_FRONT_AND_BACK, GL_KEEP, GL_KEEP, GL_KEEP );
    glStencilFuncSeparate( GL_FRONT_AND_BACK, GL_EQUAL, 0, ~0 );
    DrawRoom( OBJECT );
    DrawTeapot( OBJECT, modelViewMatrix );
    glDepthFunc( GL_LESS );
    glDisable( GL_STENCIL_TEST );
}

//////////////////////////////////////////////////////////////////////
// funkcja generujca scen 3D
//////////////////////////////////////////////////////////////////////
void DisplayScene()
{
    // czyszczenie bufora koloru i bufora gbokoci
    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT );

    // macierz modelu-widoku = macierz jednostkowa
    glm::mat4x4 modelViewMatrix = glm::mat4x4( 1.0 );

    // przesunicie obserwatora tak, aby ukad wsprzdnych obiektu by w rodku bryy obcinania
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( 0.0f, 0.0f, -(near+far)/2.0f ) );

    // skalowanie obiektu
    modelViewMatrix = glm::scale( modelViewMatrix, glm::vec3( scale, scale, scale ) );

    // obroty obiektu
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateX, glm::vec3( 1.0f, 0.0f, 0.0f ) );
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateY, glm::vec3( 0.0f, 1.0f, 0.0f ) );

    // renderin bryy cienia
    switch( shadowMode )
    {
        // cie metod depth-pass
        case DEPTH_PASS_SHADOW:
            DepthPassShadow( modelViewMatrix );
            break;

        // cie metod depth-fail
        case DEPTH_FAIL_SHADOW:
            DepthFailShadow( modelViewMatrix );
            break;

        // brya cienia metod depth-pass
        case DEPTH_PASS_SHADOW_VOLUME:
            DrawRoom( OBJECT );
            glPolygonMode( GL_FRONT_AND_BACK, GL_LINE );
            DrawShadowVolume( DEPTH_PASS_SHADOW, modelViewMatrix );
            glPolygonMode( GL_FRONT_AND_BACK, GL_FILL );
            break;

        // brya cienia metod depth-fail
        case DEPTH_FAIL_SHADOW_VOLUME:
            DrawRoom( OBJECT );
            glPolygonMode( GL_FRONT_AND_BACK, GL_LINE );
            DrawShadowVolume( DEPTH_FAIL_SHADOW, modelViewMatrix );
            glPolygonMode( GL_FRONT_AND_BACK, GL_FILL );
            break;
    }

    // rysowanie rda wiata
    DrawLight();

    // wyczenie testu gbokoci
    glDisable( GL_DEPTH_TEST );

    // wypisanie pooenia rda wiata punktowego
    const glm::vec4 clGreen( 0.000000f, 0.501961f, 0.000000f, 1.000000f );
    std::ostringstream txt;
    txt << std::setprecision( 1 ) << std::fixed
        << "pooenie rda wiata: (" << lightPosition[0] << ";"
        << lightPosition[1] << ";" << lightPosition[2] + (near+far)/2.0f << ")";
    DrawText8x16( 3, 3, txt.str(), clGreen );

    // wypisanie nazwy materiau
    DrawText8x16( 3, 21, std::string( "materia: " ) + GetMaterialName( material ), clGreen );

    // wypisanie trybu renderingu bryy cienia
    switch( shadowMode )
    {
        // cie metod depth-pass
        case DEPTH_PASS_SHADOW:
        case DEPTH_PASS_SHADOW_VOLUME:
            DrawText8x16( 3, 39, "algorytm depth-pass", clGreen );
            break;

        // cie metod depth-fail
        case DEPTH_FAIL_SHADOW:
        case DEPTH_FAIL_SHADOW_VOLUME:
            DrawText8x16( 3, 39, "algorytm depth-fail", clGreen );
            break;
    }

    // wczenie testu gbokoci
    glEnable( GL_DEPTH_TEST );
}

//////////////////////////////////////////////////////////////////////
// zmiana wielkoci okna
//////////////////////////////////////////////////////////////////////
void Reshape( int width, int height )
{
    // obszar renderingu - cae okno
    glViewport( 0, 0, width, height );

    // parametry bryy obcinania - rzutowanie perspektywiczne
    projectionMatrix = glm::frustum( left, right, bottom, top, near, far );

    // alternatywna definicja bryy obcinania z nieskoczenie
    // odleg tyln paszczyn obcinania
    //projectionMatrix = glm::frustum( 2.0f * near / (right - left), 0.0f, (right + left) / (right - left), 0.0f,
    //                                 0.0f, 2.0f * near / (top - bottom), (top + bottom) / (top - bottom), 0.0f,
    //                                 0.0f, 0.0f, -1.0f, -2.0f * near,
    //                                 0.0f, 0.0f, -1.0f, 0.0f );
}

//////////////////////////////////////////////////////////////////////
// inicjalizacja staych elementw maszyny stanu OpenGL
//////////////////////////////////////////////////////////////////////
void InitScene()
{
    // kolor ta - zawarto bufora koloru
    glClearColor( 1.0f, 1.0f, 1.0f, 1.0f );

    // shadery pomocnicze do materiaw, owietlenia i bry cieni
    GLuint shaders[5];
    enum { MATERIALS, LIGHT_MODEL, BLINN_PHONG_LIGHT, VOLUME_VS, VOLUME_FS };
    shaders[MATERIALS] = LoadShader( GL_FRAGMENT_SHADER, "../../common/materials_static.glsl" );
    shaders[LIGHT_MODEL] = LoadShader( GL_FRAGMENT_SHADER, "../../common/light_model_static.glsl" );
    shaders[BLINN_PHONG_LIGHT] = LoadShader( GL_FRAGMENT_SHADER, "../../common/blinn_phong_light.glsl" );
    shaders[VOLUME_VS] = LoadShader( GL_VERTEX_SHADER, "bryly_cieni_vs.glsl" );
    shaders[VOLUME_FS] = LoadShader( GL_FRAGMENT_SHADER, "bryly_cieni_fs.glsl" );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[OBJECT] = glCreateProgram();
    glAttachShader( program[OBJECT], LoadShader( GL_VERTEX_SHADER, "swiatlo_fragment_vs.glsl" ) );
    glAttachShader( program[OBJECT], shaders[MATERIALS] );
    glAttachShader( program[OBJECT], shaders[LIGHT_MODEL] );
    glAttachShader( program[OBJECT], shaders[BLINN_PHONG_LIGHT] );
    glAttachShader( program[OBJECT], LoadShader( GL_FRAGMENT_SHADER, "swiatlo_fragment_fs.glsl" ) );
    LinkValidateProgram( program[OBJECT] );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[OBJECT_SHADOW] = glCreateProgram();
    glAttachShader( program[OBJECT_SHADOW], LoadShader( GL_VERTEX_SHADER, "cien_fragment_vs.glsl" ) );
    glAttachShader( program[OBJECT_SHADOW], shaders[MATERIALS] );
    glAttachShader( program[OBJECT_SHADOW], shaders[LIGHT_MODEL] );
    glAttachShader( program[OBJECT_SHADOW], shaders[BLINN_PHONG_LIGHT] );
    glAttachShader( program[OBJECT_SHADOW], LoadShader( GL_FRAGMENT_SHADER, "cien_fragment_fs.glsl" ) );
    LinkValidateProgram( program[OBJECT_SHADOW] );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[LIGHT_POINT] = glCreateProgram();
    glAttachShader( program[LIGHT_POINT], LoadShader( GL_VERTEX_SHADER, "punkt_swiatla_vs.glsl" ) );
    glAttachShader( program[LIGHT_POINT], LoadShader( GL_FRAGMENT_SHADER, "punkt_swiatla_fs.glsl" ) );
    LinkValidateProgram( program[LIGHT_POINT] );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[DEPTH_PASS_SHADOW] = glCreateProgram();
    glAttachShader( program[DEPTH_PASS_SHADOW], shaders[VOLUME_VS] );
    glAttachShader( program[DEPTH_PASS_SHADOW], LoadShader( GL_GEOMETRY_SHADER, "bryly_cieni_depth_pass_gs.glsl" ) );
    glAttachShader( program[DEPTH_PASS_SHADOW], shaders[VOLUME_FS] );
    LinkValidateProgram( program[DEPTH_PASS_SHADOW] );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[DEPTH_FAIL_SHADOW] = glCreateProgram();
    glAttachShader( program[DEPTH_FAIL_SHADOW], shaders[VOLUME_VS] );
    glAttachShader( program[DEPTH_FAIL_SHADOW], LoadShader( GL_GEOMETRY_SHADER, "bryly_cieni_depth_fail_gs.glsl" ) );
    glAttachShader( program[DEPTH_FAIL_SHADOW], shaders[VOLUME_FS] );
    LinkValidateProgram( program[DEPTH_FAIL_SHADOW] );

    // generowanie identyfikatorw obiektw tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray[TEAPOT_ADJACENCY] );

    // utworzenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[TEAPOT_ADJACENCY] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[TEAPOT_POSITION] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[TEAPOT_POSITION] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( teapotPosition ), teapotPosition, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[TEAPOT_NORMAL] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[TEAPOT_NORMAL] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( teapotNormal ), teapotNormal, GL_STATIC_DRAW );
    glVertexAttribPointer( NORMAL, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( NORMAL );

    // bufor na dane indeksw wierzchokw z wierzchokami przylegymi
    GLuint teapotAdjacencyIndices[TEAPOT_INDICES_COUNT * 6];
    for( int v = 0; v < TEAPOT_INDICES_COUNT; v++ )
    {
        // wierzchoki podstawowe trjkta
        const GLuint v0 = teapotAdjacencyIndices[v*6 + 0] = teapotIndices[v*3 + 0];
        const GLuint v1 = teapotAdjacencyIndices[v*6 + 2] = teapotIndices[v*3 + 1];
        const GLuint v2 = teapotAdjacencyIndices[v*6 + 4] = teapotIndices[v*3 + 2];

        // wartoci specjalne do przyspieszonego testu wyjcia z ptli wewntrznej
        #define END_TEST 0xFFFFFFFF
        teapotAdjacencyIndices[v*6 + 1] = END_TEST;
        teapotAdjacencyIndices[v*6 + 3] = END_TEST;
        teapotAdjacencyIndices[v*6 + 5] = END_TEST;

        // wyszukiwanie wierzchokw trjkta przylegego do trjkta v0,v1,v2
        for( int w = 0; w < TEAPOT_INDICES_COUNT; w++ )
        {
            // pomijamy biecy trjkt
            if( w == v ) continue;

            // sprawdzenie, czy mamy ju znalezione wszystkie indeksy
            // wierzchokw trjktw przylegych
            if( teapotAdjacencyIndices[v*6 + 1] != END_TEST && 
                teapotAdjacencyIndices[v*6 + 3] != END_TEST &&
                teapotAdjacencyIndices[v*6 + 5] != END_TEST )
                break;

            // wierzchoki sprawdzanego trjkta
            const GLuint w0 = teapotIndices[w*3 + 0];
            const GLuint w1 = teapotIndices[w*3 + 1];
            const GLuint w2 = teapotIndices[w*3 + 2];

            // testy zgodnoci wierzchokw v0 i v1
            if( ( v0 == w0 && v1 == w1 ) || ( v0 == w1 && v1 == w0 ) )
            {
                teapotAdjacencyIndices[v*6 + 1] = w2;
                continue;
            }
            if( ( v0 == w1 && v1 == w2 ) || ( v0 == w2 && v1 == w1 ) )
            {
                teapotAdjacencyIndices[v*6 + 1] = w0;
                continue;
            }
            if( ( v0 == w2 && v1 == w0 ) || ( v0 == w0 && v1 == w2 ) )
            {
                teapotAdjacencyIndices[v*6 + 1] = w1;
                continue;
            }

            // testy zgodnoci wierzchokw v1 i v2
            if( ( v1 == w0 && v2 == w1 ) || ( v1 == w1 && v2 == w0 ) )
            {
                teapotAdjacencyIndices[v*6 + 3] = w2;
                continue;
            }
            if( ( v1 == w1 && v2 == w2 ) || ( v1 == w2 && v2 == w1 ) )
            {
                teapotAdjacencyIndices[v*6 + 3] = w0;
                continue;
            }
            if( ( v1 == w2 && v2 == w0 ) || ( v1 == w0 && v2 == w2 ) )
            {
                teapotAdjacencyIndices[v*6 + 3] = w1;
                continue;
            }

            // testy zgodnoci wierzchokw v0 i v2
            if( ( v0 == w0 && v2 == w1 ) || ( v0 == w1 && v2 == w0 ) )
            {
                teapotAdjacencyIndices[v*6 + 5] = w2;
                continue;
            }
            if( ( v0 == w1 && v2 == w2 ) || ( v0 == w2 && v2 == w1 ) )
            {
                teapotAdjacencyIndices[v*6 + 5] = w0;
                continue;
            }
            if( ( v0 == w2 && v2 == w0 ) || ( v0 == w0 && v2 == w2 ) )
            {
                teapotAdjacencyIndices[v*6 + 5] = w1;
                continue;
            }
        }

        // sprawdzenie, czy wszystkie trjkty maj indeksy
        // wierzchokw trjktw przylegych, jeeli nie,
        // to tworzone s trjkty przylege rwne odwrconym
        // trjktom podstawowym
        if( teapotAdjacencyIndices[v*6 + 1] == END_TEST )
            teapotAdjacencyIndices[v*6 + 1] = v2;
        if( teapotAdjacencyIndices[v*6 + 3] == END_TEST )
            teapotAdjacencyIndices[v*6 + 3] = v0;
        if( teapotAdjacencyIndices[v*6 + 5] == END_TEST )
            teapotAdjacencyIndices[v*6 + 5] = v1;
    }

    // utworzenie obiektu bufora indeksw wierzchokw i zaadowanie danych
    glGenBuffers( 1, &indicesBuffer[TEAPOT_ADJACENCY_INDICES] );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, indicesBuffer[TEAPOT_ADJACENCY_INDICES] );
    glBufferData( GL_ELEMENT_ARRAY_BUFFER, sizeof( teapotAdjacencyIndices ), teapotAdjacencyIndices, GL_STATIC_DRAW );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // generowanie identyfikatorw obiektw tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray[TEAPOT] );

    // utworzenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[TEAPOT] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[TEAPOT_POSITION] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( teapotPosition ), teapotPosition, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[TEAPOT_NORMAL] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( teapotNormal ), teapotNormal, GL_STATIC_DRAW );
    glVertexAttribPointer( NORMAL, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( NORMAL );

    // utworzenie obiektu bufora indeksw wierzchokw i zaadowanie danych
    glGenBuffers( 1, &indicesBuffer[TEAPOT_INDICES] );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, indicesBuffer[TEAPOT_INDICES] );
    glBufferData( GL_ELEMENT_ARRAY_BUFFER, sizeof( teapotIndices ), teapotIndices, GL_STATIC_DRAW );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // generowanie identyfikatora obiektu tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray[LIGHT_POINT] );

    // utworzenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[LIGHT_POINT] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[LIGHT_POINT_POSITION] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[LIGHT_POINT_POSITION] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( sphereLowPositionNormal ), sphereLowPositionNormal, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );

    // utworzenie obiektu bufora indeksw wierzchokw i zaadowanie danych
    glGenBuffers( 1, &indicesBuffer[LIGHT_POINT_INDICES] );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, indicesBuffer[LIGHT_POINT_INDICES] );
    glBufferData( GL_ELEMENT_ARRAY_BUFFER, sizeof( sphereLowIndices ), sphereLowIndices, GL_STATIC_DRAW );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // generowanie identyfikatora obiektu tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray[ROOM] );

    // utworzenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[ROOM] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[ROOM_POSITION] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[ROOM_POSITION] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( roomPosition ), roomPosition, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 4, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[ROOM_NORMAL] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[ROOM_NORMAL] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( roomNormal ), roomNormal, GL_STATIC_DRAW );
    glVertexAttribPointer( NORMAL, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( NORMAL );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wczenie testu bufora gbokoci
    glEnable( GL_DEPTH_TEST );

    // wczenie mechanizmw uywanych podczas renderingu tekstu
    InitDrawText();
}

//////////////////////////////////////////////////////////////////////
// usunicie obiektw OpenGL
//////////////////////////////////////////////////////////////////////
void DeleteScene()
{
    // porzdki
    glDeleteProgram( program[OBJECT] );
    glDeleteProgram( program[OBJECT_SHADOW] );
    glDeleteProgram( program[LIGHT_POINT] );
    glDeleteProgram( program[DEPTH_PASS_SHADOW] );
    glDeleteProgram( program[DEPTH_FAIL_SHADOW] );
    glDeleteBuffers( VERTEX_BUFFER_SIZE, vertexBuffer );
    glDeleteVertexArrays( VERTEX_ARRAY_SIZE, vertexArray );
    glDeleteBuffers( INDICES_BUFFER_SIZE, indicesBuffer );

    // usunicie mechanizmw uywanych podczas renderingu tekstu
    DeleteDrawText();
}
