//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

#include <string>
#include <cstring>
#include <sstream>
#include <iostream>
#include <fstream>
#include <cstdlib>
#include <iomanip>
#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>
#include "shaders.h"
#include "text.h"
#include "textures.h"

//////////////////////////////////////////////////////////////////////
// rozmiary bryy obcinania
//////////////////////////////////////////////////////////////////////
GLfloat left = -2.0f;
GLfloat right = 2.0f;
GLfloat bottom = -2.0f;
GLfloat top = 2.0f;
GLfloat near = 3.0f;
GLfloat far = 7.0f;

//////////////////////////////////////////////////////////////////////
// macierz rzutowania
//////////////////////////////////////////////////////////////////////
glm::mat4x4 projectionMatrix;

//////////////////////////////////////////////////////////////////////
// wspczynniki skalowania obiektu
//////////////////////////////////////////////////////////////////////
GLfloat scale = 1.0f;

//////////////////////////////////////////////////////////////////////
// kty obrotu obiektu
//////////////////////////////////////////////////////////////////////
GLfloat rotateX = 0.0f;
GLfloat rotateY = 180.0f;

//////////////////////////////////////////////////////////////////////
// przesunicie obiektu
//////////////////////////////////////////////////////////////////////
GLfloat translateX = 0.0f;
GLfloat translateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu programu
//////////////////////////////////////////////////////////////////////
GLuint program;

//////////////////////////////////////////////////////////////////////
// numeracja obiektw bufora wierzchokw
// numery indeksw poszczeglnych atrybutw wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    POSITION,
    TEX_COORD,
    VERTEX_BUFFER_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw bufora z danymi tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexBuffer[VERTEX_BUFFER_SIZE];

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexArray;

//////////////////////////////////////////////////////////////////////
// wsprzdne wierzchokw trjktw skadajcych si na kwadrat
//////////////////////////////////////////////////////////////////////
GLfloat position [4*3] =
{
    -1.0f, -1.0f, -1.0f,
    1.0f, -1.0f, -1.0f,
    -1.0f, 1.0f, -1.0f,
    1.0f, 1.0f, -1.0f
};

//////////////////////////////////////////////////////////////////////
// wsprzdne tekstury w wierzchokach trjktw skadajcych si
// na kwadrat; wsprzdna t jest obliczana w shaderze wierzchokw
//////////////////////////////////////////////////////////////////////
GLfloat texCoord [4*3] =
{
    0.0f, 0.0f, 0.0f,
    1.0f, 0.0f, 0.0f,
    0.0f, 0.0f, 1.0f,
    1.0f, 0.0f, 1.0f
};

//////////////////////////////////////////////////////////////////////
// numeracja obiektw tekstury
//////////////////////////////////////////////////////////////////////
enum
{
    TEXTURE_3D_0,
    TEXTURE_3D_1,
    TEXTURE_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw tekstur
//////////////////////////////////////////////////////////////////////
GLuint texture[TEXTURE_SIZE];

//////////////////////////////////////////////////////////////////////
// numer wybranego obiektu tekstury
//////////////////////////////////////////////////////////////////////
int texNumber = TEXTURE_3D_0;

//////////////////////////////////////////////////////////////////////
// liczba generowanych instancji - warstw rysowanego obiektu 3D
//////////////////////////////////////////////////////////////////////
GLint instCount = 512;

//////////////////////////////////////////////////////////////////////
// warto referencyjna skadowej alfa fragmentu uywana do usuwania
// fragmentw z potoku renderingu
//////////////////////////////////////////////////////////////////////
GLfloat refAlpha = 0.3f;

//////////////////////////////////////////////////////////////////////
// funkcja generujca scen 3D
//////////////////////////////////////////////////////////////////////
void DisplayScene()
{
    // czyszczenie bufora koloru
    glClear( GL_COLOR_BUFFER_BIT );

    // macierz modelu-widoku = macierz jednostkowa
    glm::mat4x4 modelViewMatrix = glm::mat4x4( 1.0 );

    // przesunicie obserwatora tak, aby ukad wsprzdnych obiektu by w rodku bryy obcinania
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( translateX, translateY, -(near+far)/2.0f ) );

    // skalowanie obiektu
    modelViewMatrix = glm::scale( modelViewMatrix, glm::vec3( scale, scale, scale ) );

    // obroty obiektu realizowane jako obroty wsprzdnych tekstury
    glm::mat4x4 textureMatrix = glm::mat4x4( 1.0 );
    textureMatrix = glm::rotate( textureMatrix, rotateX, glm::vec3( -1.0f, 0.0f, 0.0f ) );
    textureMatrix = glm::rotate( textureMatrix, rotateY, glm::vec3( 0.0f, 0.0f, 1.0f ) );

    // wybr obiektu tekstury
    glBindTexture( GL_TEXTURE_3D, texture[texNumber] );

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray );

    // wczenie programu
    glUseProgram( program );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    glm::mat4x4 modelViewProjectionMatrix = projectionMatrix * modelViewMatrix;
    glUniformMatrix4fv( glGetUniformLocation( program, "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // zaadowanie zmiennej jednorodnej - macierzy tekstury
    glUniformMatrix4fv( glGetUniformLocation( program, "textureMatrix" ), 1, GL_FALSE, glm::value_ptr( textureMatrix ) );

    // zmiana wartoci zmiennej jednorodnej - numeru jednostki teksturujcej
    glUniform1i( glGetUniformLocation( program ,"tex" ), 0 );

    // zaadowanie zmiennej jednorodnej - liczba instancji
    glUniform1f( glGetUniformLocation( program, "instCount" ), static_cast<GLfloat>( instCount ) );

    // zaadowanie zmiennej jednorodnej - wartoci referencyjnej skadowej
    // alfa fragmentu uywana do usuwania fragmentw z potoku renderingu
    glUniform1f( glGetUniformLocation( program, "refAlpha" ), refAlpha );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawArraysInstanced( GL_TRIANGLE_STRIP, 0, 4, instCount );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wyczenie obiektu tekstury
    glBindTexture( GL_TEXTURE_3D, 0 );

    // wypisanie wartoci referencyjnej testu alfa
    std::ostringstream txt;
    txt << std::setprecision( 2 ) << std::fixed << "warto referencyjna testu alfa: " << refAlpha;
    DrawText8x16( 3, 3, txt.str() );

    // wypisanie liczby warstw (instancji)
    txt.str( "" );
    txt << "liczba warstw (instancji): " << instCount;
    DrawText8x16( 3, 21, txt.str() );
}

//////////////////////////////////////////////////////////////////////
// zmiana wielkoci okna
//////////////////////////////////////////////////////////////////////
void Reshape( int width, int height )
{
    // obszar renderingu - cae okno
    glViewport( 0, 0, width, height );

    // parametry bryy obcinania - rzutowanie perspektywiczne
    // wysoko okna wiksza od szerokoci okna
    if( width < height && width > 0 )
         projectionMatrix = glm::frustum( left, right, bottom*height/width, top*height/width, near, far );
    else
        // szeroko okna wiksza lub rwna wysokoci okna
        if (width >= height && height > 0)
            projectionMatrix = glm::frustum( left*width/height, right*width/height, bottom, top, near, far );
        else
            projectionMatrix = glm::frustum( left, right, bottom, top, near, far );
}

//////////////////////////////////////////////////////////////////////
// inicjalizacja staych elementw maszyny stanu OpenGL
//////////////////////////////////////////////////////////////////////
void InitScene()
{
    // kolor ta - zawarto bufora koloru
    glClearColor( 1.0f, 1.0f, 1.0f, 1.0f );

    // pobranie maksymalnego rozmiaru tekstury 3D i sprawdzenie,
    // czy implementacja obsuguje potrzebn wielko tekstury 3D
    GLint max3dTextureSize;
    glGetIntegerv( GL_MAX_3D_TEXTURE_SIZE, &max3dTextureSize );
    if( max3dTextureSize < 256 )
    {
        std::cout << "GL_MAX_3D_TEXTURE_SIZE mniejsze od 256" << std::endl;
        exit( 0 );
    }

    // generowanie dwch identyfikatorw obiektw tekstury
    glGenTextures( TEXTURE_SIZE, texture );

    // utworzenie pierwszego obiektu tekstury
    glBindTexture( GL_TEXTURE_3D, texture[TEXTURE_3D_0] );

    // odczyt danych tekstury 3D, dane tekstury z kursu:
    // Programming with OpenGL: Advanced Techniques; SIGGRAPH `97
    // http://www.opengl.org/resources/code/samples/advanced/advanced97/slides/slides.html
    // http://www.opengl.org/resources/code/samples/advanced/advanced97/programs/volume.c
    // http://www.opengl.org/resources/code/samples/advanced/advanced97/programs/advanced97_data.zip

    // nazwy tekstur
    const int layers = 69;
    const char *fileNames[layers] =
    {
        "../../media/www.opengl.org/skull/skull0.la",
        "../../media/www.opengl.org/skull/skull1.la",
        "../../media/www.opengl.org/skull/skull2.la",
        "../../media/www.opengl.org/skull/skull3.la",
        "../../media/www.opengl.org/skull/skull4.la",
        "../../media/www.opengl.org/skull/skull5.la",
        "../../media/www.opengl.org/skull/skull6.la",
        "../../media/www.opengl.org/skull/skull7.la",
        "../../media/www.opengl.org/skull/skull8.la",
        "../../media/www.opengl.org/skull/skull9.la",
        "../../media/www.opengl.org/skull/skull10.la",
        "../../media/www.opengl.org/skull/skull11.la",
        "../../media/www.opengl.org/skull/skull12.la",
        "../../media/www.opengl.org/skull/skull13.la",
        "../../media/www.opengl.org/skull/skull14.la",
        "../../media/www.opengl.org/skull/skull15.la",
        "../../media/www.opengl.org/skull/skull16.la",
        "../../media/www.opengl.org/skull/skull17.la",
        "../../media/www.opengl.org/skull/skull18.la",
        "../../media/www.opengl.org/skull/skull19.la",
        "../../media/www.opengl.org/skull/skull20.la",
        "../../media/www.opengl.org/skull/skull21.la",
        "../../media/www.opengl.org/skull/skull22.la",
        "../../media/www.opengl.org/skull/skull23.la",
        "../../media/www.opengl.org/skull/skull24.la",
        "../../media/www.opengl.org/skull/skull25.la",
        "../../media/www.opengl.org/skull/skull26.la",
        "../../media/www.opengl.org/skull/skull27.la",
        "../../media/www.opengl.org/skull/skull28.la",
        "../../media/www.opengl.org/skull/skull29.la",
        "../../media/www.opengl.org/skull/skull30.la",
        "../../media/www.opengl.org/skull/skull31.la",
        "../../media/www.opengl.org/skull/skull32.la",
        "../../media/www.opengl.org/skull/skull33.la",
        "../../media/www.opengl.org/skull/skull34.la",
        "../../media/www.opengl.org/skull/skull35.la",
        "../../media/www.opengl.org/skull/skull36.la",
        "../../media/www.opengl.org/skull/skull37.la",
        "../../media/www.opengl.org/skull/skull38.la",
        "../../media/www.opengl.org/skull/skull39.la",
        "../../media/www.opengl.org/skull/skull40.la",
        "../../media/www.opengl.org/skull/skull41.la",
        "../../media/www.opengl.org/skull/skull42.la",
        "../../media/www.opengl.org/skull/skull43.la",
        "../../media/www.opengl.org/skull/skull44.la",
        "../../media/www.opengl.org/skull/skull45.la",
        "../../media/www.opengl.org/skull/skull46.la",
        "../../media/www.opengl.org/skull/skull47.la",
        "../../media/www.opengl.org/skull/skull48.la",
        "../../media/www.opengl.org/skull/skull49.la",
        "../../media/www.opengl.org/skull/skull50.la",
        "../../media/www.opengl.org/skull/skull51.la",
        "../../media/www.opengl.org/skull/skull52.la",
        "../../media/www.opengl.org/skull/skull53.la",
        "../../media/www.opengl.org/skull/skull54.la",
        "../../media/www.opengl.org/skull/skull55.la",
        "../../media/www.opengl.org/skull/skull56.la",
        "../../media/www.opengl.org/skull/skull57.la",
        "../../media/www.opengl.org/skull/skull58.la",
        "../../media/www.opengl.org/skull/skull59.la",
        "../../media/www.opengl.org/skull/skull60.la",
        "../../media/www.opengl.org/skull/skull61.la",
        "../../media/www.opengl.org/skull/skull62.la",
        "../../media/www.opengl.org/skull/skull63.la",
        "../../media/www.opengl.org/skull/skull64.la",
        "../../media/www.opengl.org/skull/skull65.la",
        "../../media/www.opengl.org/skull/skull66.la",
        "../../media/www.opengl.org/skull/skull67.la",
        "../../media/www.opengl.org/skull/skull68.la"
    };

    // wczytanie tekstur
    if( !LoadTextures( fileNames, GL_TEXTURE_3D, layers ) )
    {
        std::cout << "Niepoprawny odczyt pliku" << std::endl;
        exit( 0 );
    }

    // generownanie mipmap
    glGenerateMipmap( GL_TEXTURE_3D );

    // parametry zawijania wsprzdnych tekstury
    glTexParameteri( GL_TEXTURE_3D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_BORDER );
    glTexParameteri( GL_TEXTURE_3D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_BORDER );
    glTexParameteri( GL_TEXTURE_3D, GL_TEXTURE_WRAP_R, GL_CLAMP_TO_BORDER );

    // parametry filtracji tekstury
    glTexParameteri( GL_TEXTURE_3D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR );
    glTexParameteri( GL_TEXTURE_3D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );

    // utworzenie drugiego obiektu tekstury
    glBindTexture( GL_TEXTURE_3D, texture[TEXTURE_3D_1] );

    // rozmiary tekstury 3D
    const int width = 256;
    const int height = 256;
    const int depth = 128;    // faktycznie tekstura ma tylko 74 warstwy
    const int realDepth = 74;

    // tablica na dane tekstury 3D w formacie GL_RED
    GLushort *shortPix = new GLushort [width*height*depth];
    memset( shortPix, 0, width*height*depth * sizeof( GLushort ) );

    // odczyt danych tekstury 3D, dane tekstury z dema RadeonVolVis dotpnego pod adresem:
    // http://developer.amd.com/gpu/radeon/RadeonSampleCode/VolumeVisualizationonRadeon/Pages/default.aspx
    // http://download-developer.amd.com/GPU/zip/RadeonVolVis.zip
    // http://download-developer.amd.com/GPU/zip/raw.zip
    std::ifstream file;
    file.open( "../../media/developer.amd.com/mr01.raw", std::ios::binary );
    if( file.bad() )
    {
        std::cout << "Brak pliku: mr.raw" << std::endl;
        exit( 0 );
    }
    file.read( reinterpret_cast<char *>( &shortPix[((depth-realDepth)/2)*width*height] ), realDepth*width*height*sizeof( GLushort ) );
    file.close();

    // zdefiniowanie tekstury 3D
    glTexImage3D( GL_TEXTURE_3D, 0, GL_RED, width, height, depth, 0, GL_RED, GL_UNSIGNED_SHORT, shortPix );

    // generownanie mipmap
    glGenerateMipmap( GL_TEXTURE_3D );

    // parametry zawijania wsprzdnych tekstury
    glTexParameteri( GL_TEXTURE_3D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_BORDER );
    glTexParameteri( GL_TEXTURE_3D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_BORDER );
    glTexParameteri( GL_TEXTURE_3D, GL_TEXTURE_WRAP_R, GL_CLAMP_TO_BORDER );

    // parametry filtracji tekstury
    glTexParameteri( GL_TEXTURE_3D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR );
    glTexParameteri( GL_TEXTURE_3D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );

    // porzdki
    delete [] shortPix;

    // przeczenie na tekstur domyln
    glBindTexture( GL_TEXTURE_3D, 0 );

    // wczytanie shaderw i przygotowanie obsugi programu
    program = glCreateProgram();
    glAttachShader( program, LoadShader( GL_VERTEX_SHADER, "tekstura3d_vs.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "tekstura3d_fs.glsl" ) );
    LinkValidateProgram( program );

    // utworzenie obiektu tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray );
    glBindVertexArray( vertexArray );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[POSITION] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( position ), position, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[TEX_COORD] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[TEX_COORD] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( texCoord ), texCoord, GL_STATIC_DRAW );
    glVertexAttribPointer( TEX_COORD, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( TEX_COORD );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wczenie mieszania kolorw
    glEnable( GL_BLEND );
    glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );

    // wczenie mechanizmw uywanych podczas renderingu tekstu
    InitDrawText();
}

//////////////////////////////////////////////////////////////////////
// usunicie obiektw OpenGL
//////////////////////////////////////////////////////////////////////
void DeleteScene()
{
    // porzdki
    glDeleteProgram( program );
    glDeleteBuffers( VERTEX_BUFFER_SIZE, vertexBuffer );
    glDeleteVertexArrays( 1, &vertexArray );
    glDeleteTextures( TEXTURE_SIZE, texture );

    // usunicie mechanizmw uywanych podczas renderingu tekstu
    DeleteDrawText();
}
