//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

#include "textures.h"
#include "FreeImage.h"
#include "gli/gli.hpp"
#define KTX_OPENGL 1
#include "ktx.h"

//////////////////////////////////////////////////////////////////////
// zaadowanie tekstury z pliku graficznego
// fileName - nazwa pliku graficznego z tekstur
// target - rodzaj tworzonej tekstury
// level - poziom tworzonego obrazu tekstury, domylna warto 0
//////////////////////////////////////////////////////////////////////
bool LoadTexture( const char *fileName, GLenum target, GLint level )
{
    // inicjalizacja biblioteki FreeImage w wersji statycznej
#ifdef FREEIMAGE_LIB
    FreeImage_Initialise();
#endif

    // sprawdzenie sygnatury pliku i ustalenie jego formatu
    FREE_IMAGE_FORMAT fif = FreeImage_GetFileType( fileName, 0 );

    // format pliku nieznany, ustalenie formatu za pomoc rozszerzenia pliku
    if( fif == FIF_UNKNOWN )
        fif = FreeImage_GetFIFFromFilename( fileName );

    // format pliku nieustalony, sprawdzenie pliku KTX (tylko tekstury 2D)
    if( fif == FIF_UNKNOWN )
    {
        // pobranie identyfikatora biecego obiektu tekstury
        GLuint gnTexture;
        GLint nTexture;
        glGetIntegerv( GL_TEXTURE_BINDING_2D, &nTexture );
        gnTexture = nTexture;

        // odczyt pliku KTX
        KTX_dimensions dimensions;
        GLenum glerror;
        KTX_error_code ktxerror = ktxLoadTextureN( fileName, &gnTexture, &target, &dimensions, NULL, &glerror, NULL , NULL );

        // test poprawnoci odczytu pliku KTX
        return ktxerror == KTX_SUCCESS && dimensions.width > 0 && dimensions.height > 0 &&
               glerror == GL_NO_ERROR && target == GL_TEXTURE_2D;
    }

    // pliki DDS obsugiwane s przez bibliotek GLI
    if( fif == FIF_DDS )
    {
        // odczyt pliku DDS
        gli::texture2D texture( gli::load_dds( fileName ) );

        // sprawdzenie poprawnoci odczytu pliku
        if( texture.empty() )
        {
            return false;
        }

        // format skompresowany
        if( gli::is_compressed( texture.format() ) )
        {
            glCompressedTexImage2D( target, level, GLenum( gli::internal_format( texture.format() ) ),
                                    GLsizei( texture[0].dimensions().x ), GLsizei( texture[0].dimensions().y ),
                                    0, texture[0].size(), texture[0].data() ); 
            return true;
        }

        // format nieskompesowany
        else
        {
            glTexImage2D( target, level, GLenum( gli::internal_format( texture.format() ) ),
                          GLsizei( texture[0].dimensions().x ), GLsizei( texture[0].dimensions().y ),
                          0, GLenum( gli::external_format( texture.format() ) ),
                          GLenum( gli::type_format( texture.format() ) ), texture[0].data() );
            return true;
        }
    }

    // odczyt pliku
    FIBITMAP *dib;
    if( FreeImage_FIFSupportsReading( fif ) )
        dib = FreeImage_Load( fif, fileName );

    // bd odczytu pliku
    if( !dib )
    {
        return false;
    }

    // obrcenie obrazu
    FreeImage_FlipVertical( dib );

    // format danych obrazu
    GLenum format;
    GLenum internalformat;
    switch( FreeImage_GetColorType( dib ) )
    {
        // odcienie szaroci
        case FIC_MINISWHITE:
        case FIC_MINISBLACK:
            format = GL_RED;
            internalformat = GL_RED;
            break;

        // obraz z map kolorw - konwersja na RGB
        case FIC_PALETTE:
            {
                FIBITMAP *dibTmp = FreeImage_ConvertTo24Bits( dib );
                FreeImage_Unload( dib );
                dib = dibTmp;
            }

        // RGB
        case FIC_RGB:
#if FREEIMAGE_COLORORDER == FREEIMAGE_COLORORDER_RGB
            format = GL_RGB;
#else
            format = GL_BGR;
#endif
            internalformat = GL_RGB;
            break;

        // RGBA
        case FIC_RGBALPHA:
#if FREEIMAGE_COLORORDER == FREEIMAGE_COLORORDER_RGB
            format = GL_RGBA;
#else
            format = GL_BGRA;
#endif
            internalformat = GL_RGBA;
            break;

        // nieobsugiwany format pliku
        default:
            return false;
    }

    // typ danych pikseli obrazu
    GLenum type;
    if( FreeImage_GetImageType( dib ) == FIT_BITMAP )
        switch( FreeImage_GetBPP( dib ) )
        {
            case 24:
                if( internalformat == GL_RGB )
                    type = GL_UNSIGNED_BYTE;
                break;
            case 32:
                if( internalformat == GL_RGBA )
                    type = GL_UNSIGNED_BYTE;
                else
                if( internalformat == GL_RGB )
                {
                    type = GL_UNSIGNED_BYTE;
                    internalformat = GL_RGBA;
#if FREEIMAGE_COLORORDER == FREEIMAGE_COLORORDER_RGB
                    format = GL_RGBA;
#else
                    format = GL_BGRA;
#endif
                }
                break;
            case 8:
                if( internalformat == GL_RED )
                    type = GL_UNSIGNED_BYTE;
                break;

            // nieobsugiwany format pliku
            default:
                return false;
        }

    // tryb upakowania bajtw danych tekstury
    glPixelStorei( GL_UNPACK_ALIGNMENT, 4 );

    // definiowanie obrazu tekstury
    glTexImage2D( target, level, internalformat, FreeImage_GetWidth( dib ), FreeImage_GetHeight( dib ),
                  0, format, type, FreeImage_GetBits( dib ) );

    // porzdki
    FreeImage_Unload( dib );

    // deinicjalizacja biblioteki FreeImage w wersji statycznej
#ifdef FREEIMAGE_LIB
    FreeImage_DeInitialise();
#endif

    // sukces
    return true;
}

//////////////////////////////////////////////////////////////////////
// zaadowanie tablicy tekstur z plikw graficznych
// fileNames - nazwy plikw graficznych z teksturami
// target - rodzaj tworzonej tekstury
// layers - liczba warstw tekstury
//////////////////////////////////////////////////////////////////////
bool LoadTextures( const char **fileNames, GLenum target, GLint layers )
{
    // sprawdzenie liczby warstw tekstury
    if( layers <= 0 )
        return false;

    // inicjalizacja biblioteki FreeImage w wersji statycznej
#ifdef FREEIMAGE_LIB
    FreeImage_Initialise();
#endif

    // odczyt poszczeglnych warstw tekstury
    for( int i = 0; i < layers; i++ )
    {
        // sprawdzenie sygnatury pliku i ustalenie jego formatu
        FREE_IMAGE_FORMAT fif = FreeImage_GetFileType( fileNames[i], 0 );

        // format pliku nieznany, ustalenie formatu za pomoc rozszerzenia pliku
        if( fif == FIF_UNKNOWN )
            fif = FreeImage_GetFIFFromFilename( fileNames[i] );

        // format pliku nieustalony
        if( fif == FIF_UNKNOWN )
            return false;

        // odczyt pliku
        FIBITMAP *dib;
        if( FreeImage_FIFSupportsReading( fif ) )
            dib = FreeImage_Load( fif, fileNames[i] );

        // bd odczytu pliku
        if( !dib )
        {
            return false;
        }

        // obrcenie obrazu
        FreeImage_FlipVertical( dib );

        // format danych obrazu
        GLenum format;
        GLenum internalformat;
        switch( FreeImage_GetColorType( dib ) )
        {
            // odcienie szaroci
            case FIC_MINISWHITE:
            case FIC_MINISBLACK:
                format = GL_RED;
                internalformat = GL_RED;
                break;

            // obraz z map kolorw - konwersja na RGB
            case FIC_PALETTE:
                {
                    FIBITMAP *dibTmp = FreeImage_ConvertTo24Bits( dib );
                    FreeImage_Unload( dib );
                    dib = dibTmp;
                }

            // RGB
            case FIC_RGB:
#if FREEIMAGE_COLORORDER == FREEIMAGE_COLORORDER_RGB
                format = GL_RGB;
#else
                format = GL_BGR;
#endif
                internalformat = GL_RGB;
                break;

            // RGBA
            case FIC_RGBALPHA:
#if FREEIMAGE_COLORORDER == FREEIMAGE_COLORORDER_RGB
                format = GL_RGBA;
#else
                format = GL_BGRA;
#endif
                internalformat = GL_RGBA;
                break;

            // nieobsugiwany format pliku
            default:
                return false;
        }

        // typ danych pikseli obrazu
        GLenum type;
        if( FreeImage_GetImageType( dib ) == FIT_BITMAP )
            switch( FreeImage_GetBPP( dib ) )
            {
                case 24:
                    if( internalformat == GL_RGB )
                        type = GL_UNSIGNED_BYTE;
                    break;
                case 32:
                    if( internalformat == GL_RGBA )
                        type = GL_UNSIGNED_BYTE;
                    break;
                case 8:
                    if( internalformat == GL_RED )
                        type = GL_UNSIGNED_BYTE;
                    break;

                // nieobsugiwany format pliku
                default:
                    return false;
            }

        // tryb upakowania bajtw danych tekstury
        glPixelStorei( GL_UNPACK_ALIGNMENT, 4 );

        // definiowanie pustego obrazu tekstury
        if( i == 0 )
            glTexImage3D( target, 0, internalformat,
                          FreeImage_GetWidth( dib ), FreeImage_GetHeight( dib ), layers,
                          0, format, type, NULL );

        // definiowanie obrazu warstwy tekstury
        glTexSubImage3D( target, 0, 0, 0, i,
                         FreeImage_GetWidth( dib ), FreeImage_GetHeight( dib ), 1,
                         format, type, FreeImage_GetBits( dib ) );

        // porzdki
        FreeImage_Unload( dib );
    }

    // deinicjalizacja biblioteki FreeImage w wersji statycznej
#ifdef FREEIMAGE_LIB
    FreeImage_DeInitialise();
#endif

    // sukces
    return true;
}

//////////////////////////////////////////////////////////////////////
// zapis tekstury szeciennej do plikw graficznych w formacie PNG
// fileNames - nazwy plikw graficznych z teksturami
// level - poziom zapisywanych obrazw tekstury, domylna warto 0
//////////////////////////////////////////////////////////////////////
bool SaveCubeTexture( const char **fileNames, GLint level )
{
    // inicjalizacja biblioteki FreeImage w wersji statycznej
#ifdef FREEIMAGE_LIB
    FreeImage_Initialise();
#endif

    // tryb upakowania bajtw danych tekstury
    glPixelStorei( GL_PACK_ALIGNMENT, 1 );

    // pobranie rozmiarw tekstury
    GLint width,height;
    glGetTexLevelParameteriv( GL_TEXTURE_CUBE_MAP_POSITIVE_X, level, GL_TEXTURE_WIDTH, &width );
    glGetTexLevelParameteriv( GL_TEXTURE_CUBE_MAP_POSITIVE_X, level, GL_TEXTURE_HEIGHT, &height );

    // utworzenie bitmapy
    FIBITMAP *dib = FreeImage_Allocate( width, height, 24 );
    if( dib )
    {
        // bufor na dane poszczeglnych stron tekstury szeciennej
        GLubyte *buf = new GLubyte[width * height * 3];

        // zapis kolejnych tekstur
        for( int i = 0; i < 6; i++ )
        {
            // pobranie danych tektury
            glGetTexImage( GL_TEXTURE_CUBE_MAP_POSITIVE_X + i, 0, GL_RGB, GL_UNSIGNED_BYTE, buf );

            // kopiowanie danych pikseli
            GLubyte *pbuf = buf;
            unsigned int pitch = FreeImage_GetPitch( dib );
            BYTE *bits = (BYTE*)FreeImage_GetBits( dib );
            for( int y = 0; y < height; y++ )
            {
                BYTE *pixel = (BYTE*)bits;
                for( int x = 0; x < width; x++ )
                {
                    pixel[FI_RGBA_RED] = *pbuf++;
                    pixel[FI_RGBA_GREEN] = *pbuf++;
                    pixel[FI_RGBA_BLUE] = *pbuf++;
                    pixel += 3;
                }

                // nastpny wiersz
                bits += pitch;
            }

            // obrcenie obrazu
            FreeImage_FlipVertical( dib );

            // zapis pliku
            if( FreeImage_Save( FIF_PNG, dib, fileNames[i], 0 ) == false )
            {
                // porzdki
                delete[] buf;
                FreeImage_Unload( dib );
                return false;
            }
        }

        // porzdki
        delete[] buf;
        FreeImage_Unload( dib );
    }
    else
        return false;

    // deinicjalizacja biblioteki FreeImage w wersji statycznej
#ifdef FREEIMAGE_LIB
    FreeImage_DeInitialise();
#endif

    // sukces
    return true;
}
