//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////
// GLSL 4.30
//////////////////////////////////////////////////////////////////////
#version 430

//////////////////////////////////////////////////////////////////////
// definicje numerw pooenia zmiennych jednorodnych
// i powizania uchwytw tekstur
//////////////////////////////////////////////////////////////////////
#define AMBIENT     0
#define DIFFUSE     1
#define SPECULAR    2
#define SHININESS   3
#define HEIGHT      4
#define NORMAL      5
#define TEXTURE_LOC 4

//////////////////////////////////////////////////////////////////////
// tablica znacznikw materiaw zawartych w teksturze
//////////////////////////////////////////////////////////////////////
layout( location = TEXTURE_LOC ) uniform bool textures[6] =
        bool[6]( false, false, false, false, false, false );

//////////////////////////////////////////////////////////////////////
// wspczynnik odbicia wiata otoczenia; oznaczenie Ma
//////////////////////////////////////////////////////////////////////
layout( location = AMBIENT ) uniform vec4 ambient = vec4( 0.2, 0.2, 0.2, 1.0 );

//////////////////////////////////////////////////////////////////////
// wspczynnik odbicia wiata rozproszonego; oznaczenie Md
//////////////////////////////////////////////////////////////////////
layout( location = DIFFUSE ) uniform vec4 diffuse = vec4( 0.8, 0.8, 0.8, 1.0 );

//////////////////////////////////////////////////////////////////////
// wspczynnik odbicia wiata zwierciadlanego; oznaczenie Ms
//////////////////////////////////////////////////////////////////////
layout( location = SPECULAR ) uniform vec4 specular = vec4( 0.0, 0.0, 0.0, 1.0 );

//////////////////////////////////////////////////////////////////////
// wykadnik wspczynnika funkcji rozbysku;
// zakres wartoci <0;128>; oznaczenie Msh
//////////////////////////////////////////////////////////////////////
layout( location = SHININESS ) uniform float shininess = 0.0;

//////////////////////////////////////////////////////////////////////
// uchwyt tekstury 2D
// wspczynnik odbicia wiata otoczenia; oznaczenie Ma
//////////////////////////////////////////////////////////////////////
layout( binding = AMBIENT ) uniform sampler2D texAmbient;

//////////////////////////////////////////////////////////////////////
// uchwyt tekstury 2D
// wspczynnik odbicia wiata rozproszonego; oznaczenie Md
//////////////////////////////////////////////////////////////////////
layout( binding = DIFFUSE ) uniform sampler2D texDiffuse;

//////////////////////////////////////////////////////////////////////
// uchwyt tekstury 2D
// wspczynnik odbicia wiata zwierciadlanego; oznaczenie Ms
//////////////////////////////////////////////////////////////////////
layout( binding = SPECULAR ) uniform sampler2D texSpecular;

//////////////////////////////////////////////////////////////////////
// uchwyt tekstury 2D
// wykadnik wspczynnika funkcji rozbysku;
// zakres wartoci <0;128>; oznaczenie Msh
//////////////////////////////////////////////////////////////////////
layout( binding = SHININESS ) uniform sampler2D texShininess;

//////////////////////////////////////////////////////////////////////
// uchwyt tekstury 2D
// mapa wektorw normalnych
//////////////////////////////////////////////////////////////////////
layout( binding = NORMAL ) uniform sampler2D texNormal;

//////////////////////////////////////////////////////////////////////
// interpolowane wsprzdne wierzchoka
//////////////////////////////////////////////////////////////////////
in vec3 inoutPosition;

//////////////////////////////////////////////////////////////////////
// interpolowane wsprzdne wektora normalnego
//////////////////////////////////////////////////////////////////////
in vec3 inoutNormal;

//////////////////////////////////////////////////////////////////////
// interpolowane wsprzdne wektora stycznego
//////////////////////////////////////////////////////////////////////
in vec3 inoutTangent;

//////////////////////////////////////////////////////////////////////
// interpolowane wsprzdne wektora bistycznego
//////////////////////////////////////////////////////////////////////
in vec3 inoutBitangent;

//////////////////////////////////////////////////////////////////////
// interpolowane wsprzdne tekstury
//////////////////////////////////////////////////////////////////////
in vec2 inoutTexCoord;

//////////////////////////////////////////////////////////////////////
// prototypy funkcji
//////////////////////////////////////////////////////////////////////
vec4 BlinnPhongLight( const int light, const vec3 normal, const vec3 position,
                      const vec4 ambient, const vec4 diffuse,
                      const vec4 specular, const float shininess );
vec4 PhongLight( const int light, const vec3 normal, const vec3 position,
                 const vec4 ambient, const vec4 diffuse,
                 const vec4 specular, const float shininess );
vec4 LambertLight( const int light, const vec3 normal, const vec3 position,
                   const vec4 ambient, const vec4 diffuse );
vec4 BlinnPhongLight( const int light, const vec3 normal, const vec3 position,
                      const vec4 ambient, const vec4 diffuse,
                      const vec4 specular, const float shininess,
                      const mat3 TBN );
vec4 PhongLight( const int light, const vec3 normal, const vec3 position,
                 const vec4 ambient, const vec4 diffuse,
                 const vec4 specular, const float shininess,
                 const mat3 TBN );
vec4 LambertLight( const int light, const vec3 normal, const vec3 position,
                   const vec4 ambient, const vec4 diffuse, const mat3 TBN );

//////////////////////////////////////////////////////////////////////
// deklaracja typu podprogramu obsugujcego model owietlenia
//////////////////////////////////////////////////////////////////////
subroutine vec4 lightSubroutine( const vec3 normal, const vec3 position,
                                 const vec4 ambient, const vec4 diffuse,
                                 const vec4 specular, const float shininess );

//////////////////////////////////////////////////////////////////////
// podprogram obsugujcy model owietlenia Phonga
//////////////////////////////////////////////////////////////////////
subroutine( lightSubroutine ) vec4 Phong( const vec3 normal, const vec3 position,
                                          const vec4 ambient, const vec4 diffuse,
                                          const vec4 specular, const float shininess )
{
    return PhongLight( 0, normal, position, ambient, diffuse, specular, shininess );
}

//////////////////////////////////////////////////////////////////////
// podprogram obsugujcy model owietlenia Blinna-Phonga
//////////////////////////////////////////////////////////////////////
subroutine( lightSubroutine ) vec4 BlinnPhong( const vec3 normal, const vec3 position,
                                                const vec4 ambient, const vec4 diffuse,
                                                const vec4 specular, const float shininess )
{
    return BlinnPhongLight( 0, normal, position, ambient, diffuse, specular, shininess );
}

//////////////////////////////////////////////////////////////////////
// podprogram obsugujcy model owietlenia Lamberta
//////////////////////////////////////////////////////////////////////
subroutine( lightSubroutine ) vec4 Lambert( const vec3 normal, const vec3 position,
                                            const vec4 ambient, const vec4 diffuse,
                                            const vec4 specular, const float shininess )
{
    return LambertLight( 0, normal, position, ambient, diffuse );
}

//////////////////////////////////////////////////////////////////////
// podprogram obsugujcy model owietlenia Phonga
// z mapowaniem wektorw normalnych
//////////////////////////////////////////////////////////////////////
subroutine( lightSubroutine ) vec4 PhongNormalMap( const vec3 normal, const vec3 position,
                                                   const vec4 ambient, const vec4 diffuse,
                                                   const vec4 specular, const float shininess )
{
    // obliczenia zwizane z mapowaniem nierwnoci
    if( textures[NORMAL] )
    {
        // obliczenie macierzy przeksztacenia
        vec3 tangent = normalize( inoutTangent );
        vec3 bitangent = normalize( inoutBitangent );
        mat3 TBN = transpose( mat3( tangent, bitangent, normal ) );

        // rekonstrukcja skadowej z wektora normalnego z mapy wektorw normalnych
        vec3 bumpNormal;
        bumpNormal.rg = texture( texNormal, inoutTexCoord ).xy;
        bumpNormal.rg = 2.0 * bumpNormal.rg - 1.0;
        bumpNormal.b = 0.5 * (sqrt( 1.0 - dot( bumpNormal.rg, bumpNormal.rg ) ) + 1.0);

        // kolor fragmentu
        return PhongLight( 0, bumpNormal, position, ambient, diffuse, specular, shininess, TBN );
    }
    else
        // brak mapy wektorw normalnych - standardowe owietlenie
        return PhongLight( 0, normal, position, ambient, diffuse, specular, shininess );
}

//////////////////////////////////////////////////////////////////////
// podprogram obsugujcy model owietlenia Blinna-Phonga
// z mapowaniem wektorw normalnych
//////////////////////////////////////////////////////////////////////
subroutine( lightSubroutine ) vec4 BlinnPhongNormalMap( const vec3 normal, const vec3 position,
                                                        const vec4 ambient, const vec4 diffuse,
                                                        const vec4 specular, const float shininess )
{
    // obliczenia zwizane z mapowaniem nierwnoci
    if( textures[NORMAL] )
    {
        // obliczenie macierzy przeksztacenia
        vec3 tangent = normalize( inoutTangent );
        vec3 bitangent = normalize( inoutBitangent );
        mat3 TBN = transpose( mat3( tangent, bitangent, normal ) );

        // rekonstrukcja skadowej z wektora normalnego z mapy wektorw normalnych
        vec3 bumpNormal;
        bumpNormal.rg = texture( texNormal, inoutTexCoord ).xy;
        bumpNormal.rg = 2.0 * bumpNormal.rg - 1.0;
        bumpNormal.b = 0.5 * (sqrt( 1.0 - dot( bumpNormal.rg, bumpNormal.rg ) ) + 1.0);

        // kolor fragmentu
        return BlinnPhongLight( 0, bumpNormal, position, ambient, diffuse, specular, shininess, TBN );
    }
    else
        // brak mapy wektorw normalnych - standardowe owietlenie
        return BlinnPhongLight( 0, normal, position, ambient, diffuse, specular, shininess );
}

//////////////////////////////////////////////////////////////////////
// podprogram obsugujcy model owietlenia Lamberta
// z mapowaniem wektorw normalnych
//////////////////////////////////////////////////////////////////////
subroutine( lightSubroutine ) vec4 LambertNormalMap( const vec3 normal, const vec3 position,
                                                     const vec4 ambient, const vec4 diffuse,
                                                     const vec4 specular, const float shininess )
{
    // obliczenia zwizane z mapowaniem nierwnoci
    if( textures[NORMAL] )
    {
        // obliczenie macierzy przeksztacenia
        vec3 tangent = normalize( inoutTangent );
        vec3 bitangent = normalize( inoutBitangent );
        mat3 TBN = transpose( mat3( tangent, bitangent, normal ) );

        // rekonstrukcja skadowej z wektora normalnego z mapy wektorw normalnych
        vec3 bumpNormal;
        bumpNormal.rg = texture( texNormal, inoutTexCoord ).xy;
        bumpNormal.rg = 2.0 * bumpNormal.rg - 1.0;
        bumpNormal.b = 0.5 * (sqrt( 1.0 - dot( bumpNormal.rg, bumpNormal.rg ) ) + 1.0);

        // kolor fragmentu
        return LambertLight( 0, bumpNormal, position, ambient, diffuse, TBN );
    }
    else
        // brak mapy wektorw normalnych - standardowe owietlenie
        return LambertLight( 0, normal, position, ambient, diffuse );
}

//////////////////////////////////////////////////////////////////////
// wybr rodzaju modelu owietlenia
//////////////////////////////////////////////////////////////////////
subroutine uniform lightSubroutine lightType;

//////////////////////////////////////////////////////////////////////
// wyjciowy kolor fragmentu
//////////////////////////////////////////////////////////////////////
out vec4 outColor;

//////////////////////////////////////////////////////////////////////
// program gwny
//////////////////////////////////////////////////////////////////////
void main()
{
    // normalizacja wektora normalnego zalena od strony wielokta
    vec3 normal = gl_FrontFacing ? normalize( inoutNormal ) : -normalize( inoutNormal );

    // pobranie materiaw
    vec4 matAmbient = textures[AMBIENT] ? texture( texAmbient, inoutTexCoord ) : ambient;
    vec4 matDiffuse = textures[DIFFUSE] ? texture( texDiffuse, inoutTexCoord ) : diffuse;
    vec4 matSpecular = textures[SPECULAR] ? texture( texSpecular, inoutTexCoord ) : specular;
    float matShininess = textures[SHININESS] ? texture( texShininess, inoutTexCoord ).x : shininess;

    // kolor fragmentu zaleny od modelu owietlenia
    outColor = vec4( lightType( normal, inoutPosition, matAmbient, matDiffuse, matSpecular, matShininess ).xyz, diffuse.a );
}
