//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

#include <iostream>
#include <string>
#include <sstream>
#include <cstdlib>
#include <ctime>
#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>
#include "shaders.h"
#include "text.h"
#include "textures.h"

//////////////////////////////////////////////////////////////////////
// rozmiary bryy obcinania
//////////////////////////////////////////////////////////////////////
GLfloat left = -2.0f;
GLfloat right = 2.0f;
GLfloat bottom = -2.0f;
GLfloat top = 2.0f;
GLfloat near = 3.0f;
GLfloat far = 7.0f;

//////////////////////////////////////////////////////////////////////
// macierz rzutowania
//////////////////////////////////////////////////////////////////////
glm::mat4x4 projectionMatrix;

//////////////////////////////////////////////////////////////////////
// kty obrotu obiektu
//////////////////////////////////////////////////////////////////////
GLfloat rotateX = 0.0f;
GLfloat rotateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// numeracja identyfikatorw obiektu programu
//////////////////////////////////////////////////////////////////////
enum
{
    POINT_SPRITE,
    GS_SPRITE,
    PROGRAM_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektu programu
//////////////////////////////////////////////////////////////////////
GLuint program[PROGRAM_SIZE];

//////////////////////////////////////////////////////////////////////
// rodzaj sprajta - punkty/prostokty
//////////////////////////////////////////////////////////////////////
int spriteType = GS_SPRITE;

//////////////////////////////////////////////////////////////////////
// liczba tekstur z patkami niegu
//////////////////////////////////////////////////////////////////////
const int MAX_SNOW_FLAKES = 6;

//////////////////////////////////////////////////////////////////////
// maksymalna liczba punktw
//////////////////////////////////////////////////////////////////////
const int MAX_VERTEX = 20000;

//////////////////////////////////////////////////////////////////////
// tablica ze wsprzdnymi punktw oraz numerem tekstury
// z patkiem niegu
//////////////////////////////////////////////////////////////////////
GLfloat position[MAX_VERTEX * 4];

//////////////////////////////////////////////////////////////////////
// liczba wywietlanych obiektw
//////////////////////////////////////////////////////////////////////
int spriteCount = 1000;

//////////////////////////////////////////////////////////////////////
// bieca wielko punktu
//////////////////////////////////////////////////////////////////////
GLfloat pointSize = 10.0;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu bufora z danymi tablicy wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexBuffer;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektw tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexArray;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu tekstury
//////////////////////////////////////////////////////////////////////
GLuint texture;

//////////////////////////////////////////////////////////////////////
// numery indeksw poszczeglnych atrybutw wierzchokw
//////////////////////////////////////////////////////////////////////
#define POSITION 0

//////////////////////////////////////////////////////////////////////
// funkcja generujca scen 3D
//////////////////////////////////////////////////////////////////////
void DisplayScene()
{
    // czyszczenie bufora koloru i bufora gbokoci
    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

    // macierz modelu-widoku = macierz jednostkowa
    glm::mat4x4 modelViewMatrix = glm::mat4x4( 1.0 );

    // przesunicie ukadu wsprzdnych obiektu do rodka bryy obcinania
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( 0.0f, 0.0f, -(near+far)/2.0f ) );

    // obroty obiektu
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateX, glm::vec3( 1.0f, 0.0f, 0.0f ) );
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateY, glm::vec3( 0.0f, 1.0f, 0.0f ) );

    // aktualizacja wsprzdnych obiektw
    // oraz generowanie numerw tekstury patkw niegu
    for( int i = 0; i < spriteCount; i++ )
    {
        // wsprzdne
        position[i*4 + 1] -= 0.005f + rand() / ( float )RAND_MAX * 0.01f;
        position[i*4 + 0] += 0.001f * sin( position[i*3 + 1] );
        position[i*4 + 2] += 0.001f * cos( position[i*3 + 1] );

        // generowanie nowego patka
        if( position[i*4 + 1] < 2.0*bottom )
        {
            // wsprzdne
            position[i*4 + 0] = 2.0f*(rand() / ( float )RAND_MAX) * (right - left) + 2.0f*left;
            position[i*4 + 1] = 2.0f*(rand() / ( float )RAND_MAX) * (top - bottom) + 2.0f*top;
            position[i*4 + 2] = (rand() / ( float )RAND_MAX) * (far - near) - near;

            // numer tekstury patka niegu
            position[i*4 + 3] = (rand() / ( float )RAND_MAX) * MAX_SNOW_FLAKES;
        }
    }

    // aktualizacja magazynu danych bufora
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer );
    glBufferSubData( GL_ARRAY_BUFFER, 0, 4 * spriteCount * sizeof( GLfloat ), position );
    glBindBuffer( GL_ARRAY_BUFFER, 0 );

    // wczenie obiektu tekstury
    glBindTexture( GL_TEXTURE_2D_ARRAY, texture );

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray );

    // wczenie programu
    glUseProgram( program[spriteType] );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    glm::mat4x4 modelViewProjectionMatrix = projectionMatrix * modelViewMatrix;
    glUniformMatrix4fv( glGetUniformLocation( program[spriteType], "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // zaadowanie zmiennej jednorodnej - numeru jednostki teksturujcej
    glUniform1i( glGetUniformLocation( program[spriteType], "tex" ), 0 );

    // zaadowanie zmiennej jednorodnej - wielkoci punktu
    glUniform1f( glGetUniformLocation( program[spriteType], "pointSize" ), pointSize );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawArrays( GL_POINTS, 0, spriteCount );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wyczenie obiektu tekstury
    glBindTexture( GL_TEXTURE_2D_ARRAY, 0 );

    // wyczenie testu gbokoci
    glDisable( GL_DEPTH_TEST );

    // wypisanie informacji o liczbie obiektw
    std::ostringstream txt;
    txt << "liczba czstek: " << spriteCount;
    DrawText8x16( 3, 3, txt.str(), glm::vec4( 1.0f ) );

    // wypisanie informacji o rodzaju sprajta
    if( spriteType == POINT_SPRITE )
        DrawText8x16( 3, 20, "rendering czstek: sprajty punktowe", glm::vec4( 1.0f ) );
    else
        DrawText8x16( 3, 20, "rendering czstek: prostokty (shadery geometrii)", glm::vec4( 1.0f ) );

    // wczenie testu gbokoci
    glEnable( GL_DEPTH_TEST );
}

//////////////////////////////////////////////////////////////////////
// zmiana wielkoci okna
//////////////////////////////////////////////////////////////////////
void Reshape( int width, int height )
{
    // obszar renderingu - cae okno
    glViewport( 0, 0, width, height );

    // parametry bryy obcinania - rzutowanie perspektywiczne
    // wysoko okna wiksza od szerokoci okna
    if( width < height && width > 0 )
         projectionMatrix = glm::frustum( left, right, bottom*height/width, top*height/width, near, far );
    else
        // szeroko okna wiksza lub rwna wysokoci okna
        if (width >= height && height > 0)
            projectionMatrix = glm::frustum( left*width/height, right*width/height, bottom, top, near, far );
        else
            projectionMatrix = glm::frustum( left, right, bottom, top, near, far );
}

//////////////////////////////////////////////////////////////////////
// inicjalizacja staych elementw maszyny stanu OpenGL
//////////////////////////////////////////////////////////////////////
void InitScene()
{
    // kolor ta - zawarto bufora koloru
    glClearColor( 0.0f, 0.0f, 0.545098f, 1.0f );

    // utworzenie obiektu tekstury
    glGenTextures( 1, &texture );
    glBindTexture( GL_TEXTURE_2D_ARRAY, texture );

    // nazwy tekstur z patkami niegu
    const char *fileNames[MAX_SNOW_FLAKES] =
    {
        "../../media/snowflakebentley.com/Snowflake1.jpg",
        "../../media/snowflakebentley.com/Snowflake2.jpg",
        "../../media/snowflakebentley.com/Snowflake3.jpg",
        "../../media/snowflakebentley.com/Snowflake4.jpg",
        "../../media/snowflakebentley.com/Snowflake5.jpg",
        "../../media/snowflakebentley.com/Snowflake6.jpg"
    };

    // wczytanie tekstur
    if( !LoadTextures( fileNames, GL_TEXTURE_2D_ARRAY, MAX_SNOW_FLAKES ) )
    {
        std::cout << "Niepoprawny odczyt pliku" << std::endl;
        exit( 0 );
    }

    // generownanie mipmap
    glGenerateMipmap( GL_TEXTURE_2D_ARRAY );

    // filtr pomniejszajcy
    glTexParameteri( GL_TEXTURE_2D_ARRAY, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR );

    // wyczenie obiektu tekstury
    glBindTexture( GL_TEXTURE_2D_ARRAY, 0 );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[POINT_SPRITE] = glCreateProgram();
    glAttachShader( program[POINT_SPRITE], LoadShader( GL_VERTEX_SHADER, "snieg_punkty_vs.glsl" ) );
    glAttachShader( program[POINT_SPRITE], LoadShader( GL_FRAGMENT_SHADER, "snieg_punkty_fs.glsl" ) );
    LinkValidateProgram( program[POINT_SPRITE] );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[GS_SPRITE] = glCreateProgram();
    glAttachShader( program[GS_SPRITE], LoadShader( GL_VERTEX_SHADER, "snieg_prostokaty_vs.glsl" ) );
    glAttachShader( program[GS_SPRITE], LoadShader( GL_GEOMETRY_SHADER, "snieg_prostokaty_gs.glsl" ) );
    glAttachShader( program[GS_SPRITE], LoadShader( GL_FRAGMENT_SHADER, "snieg_prostokaty_fs.glsl" ) );
    LinkValidateProgram( program[GS_SPRITE] );

    // utworzenie obiektu tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray );
    glBindVertexArray( vertexArray );

    // okrelenie zarodka dla cigu liczb pseudolosowych
    srand( static_cast<unsigned int>( time( NULL ) ) );

    // generowanie pocztkowych wsprzdnych punktw
    // oraz numerw tekstury patkw niegu
    for( int i = 0; i < MAX_VERTEX; i++ )
    {
        // wsprzdne punktw
        position[i*4 + 0] = 2.0f*(rand() / ( float )RAND_MAX) * (right - left) + 2.0f*left;
        position[i*4 + 1] = 2.0f*(rand() / ( float )RAND_MAX) * (top - bottom) + 2.0f*top;
        position[i*4 + 2] = (rand() / ( float )RAND_MAX) * (far - near) - near;

        // numer tekstury patka niegu
        position[i*4 + 3] = (rand() / ( float )RAND_MAX) * MAX_SNOW_FLAKES;
    }

    // utworzenie obiektw bufora wierzchokw (VBO) i zaadowania danych
    glGenBuffers( 1, &vertexBuffer );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer );
    glBufferData( GL_ARRAY_BUFFER, sizeof( position ), position, GL_DYNAMIC_DRAW );
    glVertexAttribPointer( POSITION, 4, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wczenie generowania wielkoci punktu w shaderze
    glEnable( GL_PROGRAM_POINT_SIZE );

    // wczenie testu bufora gbokoci
    glEnable( GL_DEPTH_TEST );

    // wczenie mechanizmw uywanych podczas renderingu tekstu
    InitDrawText();
}

//////////////////////////////////////////////////////////////////////
// usunicie obiektw OpenGL
//////////////////////////////////////////////////////////////////////
void DeleteScene()
{
    // porzdki
    glDeleteProgram( program[POINT_SPRITE] );
    glDeleteProgram( program[GS_SPRITE] );
    glDeleteBuffers( 1, &vertexBuffer );
    glDeleteVertexArrays( 1, &vertexArray );
    glDeleteTextures( 1, &texture );

    // usunicie mechanizmw uywanych podczas renderingu tekstu
    DeleteDrawText();
}
