//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

#define _USE_MATH_DEFINES
#include <cmath>
#include <string>
#include <iostream>
#include <sstream>
#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>
#include "shaders.h"
#include "text.h"
#include "textures.h"

//////////////////////////////////////////////////////////////////////
// rozmiary bryy obcinania
//////////////////////////////////////////////////////////////////////
GLfloat left = -4.0f;
GLfloat right = 4.0f;
GLfloat bottom = -4.0f;
GLfloat top = 4.0f;
GLfloat near = 10.0f;
GLfloat far = 18.0f;

//////////////////////////////////////////////////////////////////////
// macierz rzutowania
//////////////////////////////////////////////////////////////////////
glm::mat4x4 projectionMatrix;

//////////////////////////////////////////////////////////////////////
// wspczynniki skalowania obiektu
//////////////////////////////////////////////////////////////////////
GLfloat scale = 3.0f;

//////////////////////////////////////////////////////////////////////
// kty obrotu obiektu
//////////////////////////////////////////////////////////////////////
GLfloat rotateX = 0.0f;
GLfloat rotateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// przesunicie obiektu
//////////////////////////////////////////////////////////////////////
GLfloat translateX = 0.0f;
GLfloat translateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// kty obrotu kierunku wiata
//////////////////////////////////////////////////////////////////////
GLfloat rotateLightX = 0.0f;
GLfloat rotateLightY = 0.0f;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu programu
//////////////////////////////////////////////////////////////////////
GLuint program;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu bufora z danymi tablicy wierzchokw ze
// wsprzdnymi wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexBuffer;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexArray;

//////////////////////////////////////////////////////////////////////
// parametr regulujcy jako generowanej powierzchni
//////////////////////////////////////////////////////////////////////
float DENSITY = 12;

//////////////////////////////////////////////////////////////////////
// numeracja obiektw tekstury
//////////////////////////////////////////////////////////////////////
enum
{
    VENUS,
    EARTH,
    MARS,
    TEXTURE_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw tekstury
//////////////////////////////////////////////////////////////////////
GLuint textures[TEXTURE_SIZE];

//////////////////////////////////////////////////////////////////////
// numer wybranego obiektu tekstury
//////////////////////////////////////////////////////////////////////
int texNumber = VENUS;

//////////////////////////////////////////////////////////////////////
// wsprzdne wierzchokw trjktw skadajcych si na kwadrat
//////////////////////////////////////////////////////////////////////
GLfloat position[4*2] =
{
     0.0f, 0.0f,
     static_cast<GLfloat>( 1.0f * M_PI ), 0.0f,
     static_cast<GLfloat>( 1.0f * M_PI ), static_cast<GLfloat>( 2.0f * M_PI ),
     0.0f, static_cast<GLfloat>( 2.0f * M_PI )
};

//////////////////////////////////////////////////////////////////////
// numery indeksw poszczeglnych atrybutw wierzchokw
//////////////////////////////////////////////////////////////////////
#define POSITION 0

//////////////////////////////////////////////////////////////////////
// funkcja generujca scen 3D
//////////////////////////////////////////////////////////////////////
void DisplayScene()
{
    // czyszczenie bufora koloru i bufora gbokoci
    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

    // macierz modelu-widoku = macierz jednostkowa
    glm::mat4x4 modelViewMatrix = glm::mat4x4( 1.0 );

    // przesunicie obserwatora tak, aby ukad wsprzdnych obiektu by w rodku bryy obcinania
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( 0.0f, 0.0f, -(near+far)/2.0f ) );

    // skalowanie obiektu
    modelViewMatrix = glm::scale( modelViewMatrix, glm::vec3( scale, scale, scale ) );

    // przesunicie obiektu
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( translateX, translateY, 0.0f ) );

    // obroty obiektu
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateX, glm::vec3( 1.0f, 0.0f, 0.0f ) );
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateY, glm::vec3( 0.0f, 1.0f, 0.0f ) );

    // obroty wektora kierunku wiata
    glm::mat4x4 rotateLightDir = glm::mat4x4( 1.0 );
    rotateLightDir = glm::rotate( rotateLightDir, rotateLightX, glm::vec3( 1.0f, 0.0f, 0.0f ) );
    rotateLightDir = glm::rotate( rotateLightDir, rotateLightY, glm::vec3( 0.0f, 1.0f, 0.0f ) );
    glm::vec4 lightPosition( 0.0f, 0.0f, 1.0f, 0.0f );
    lightPosition = rotateLightDir * lightPosition;
    lightPosition = glm::normalize( lightPosition );

    // odwrcona macierz modelu-widoku niezbdna do przeksztace
    // do ukadu wsprzdnych obiektu
    glm::mat4x4 modelViewMatrixInverse( glm::inverse( modelViewMatrix ) );

    // transformacja kierunku wiata do ukadu wsprzdnych obiektu
    glm::vec4 inverseLightPosition( modelViewMatrixInverse * lightPosition );
    inverseLightPosition = glm::normalize( inverseLightPosition );

    // przeksztacenie pooenia obserwatora do ukadu wsprzdnych obiektu
    glm::vec4 eyePosition( 0.0f, 0.0f, 0.0f, 1.0f );
    eyePosition = modelViewMatrixInverse * eyePosition;

    // okrelenie poziomw teselacji
    GLfloat tessLevel[4] = { DENSITY, DENSITY, DENSITY, DENSITY };
    glPatchParameterfv( GL_PATCH_DEFAULT_OUTER_LEVEL, tessLevel );
    glPatchParameterfv( GL_PATCH_DEFAULT_INNER_LEVEL, tessLevel );

    // wybr obiektu tekstury
    glBindTexture( GL_TEXTURE_2D, textures[texNumber] );

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray );

    // wczenie programu
    glUseProgram( program );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    glm::mat4x4 modelViewProjectionMatrix = projectionMatrix * modelViewMatrix;
    glUniformMatrix4fv( glGetUniformLocation( program, "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // zaadowanie kierunku rda wiata i pooenia obserwatora w ukadzie wsprzdnych obiektu
    glUniform4fv( glGetUniformLocation( program, "lightSource[0].position" ), 1, glm::value_ptr( inverseLightPosition ) );
    glUniform4fv( glGetUniformLocation( program, "eyePosition" ), 1, glm::value_ptr( eyePosition ) );

    // zaadowanie zmiennej jednorodnej - numeru jednostki teksturujcej
    glUniform1i( glGetUniformLocation( program, "tex" ), 0 );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawArrays( GL_PATCHES, 0, 4 );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wypisanie stopnia teselacji
    std::ostringstream txt;
    txt << "poziom teselacji: " << (int)DENSITY << "" << (int)DENSITY;
    DrawText8x16( 3, 21, txt.str() );

    // wypisanie nazwy tekstury
    switch( texNumber )
    {
        case 0: DrawText8x16( 3, 3, "Wenus - zdjcia sondy Magellan" ); break;
        case 1: DrawText8x16( 3, 3, "Ziemia - mapa hipsometryczna (fizyczna)" ); break;
        case 2: DrawText8x16( 3, 3, "Mars - zdjcia sond Viking" ); break;
    };
}

//////////////////////////////////////////////////////////////////////
// zmiana wielkoci okna
//////////////////////////////////////////////////////////////////////
void Reshape( int width, int height )
{
    // obszar renderingu - cae okno
    glViewport( 0, 0, width, height );

    // parametry bryy obcinania - rzutowanie perspektywiczne
    // wysoko okna wiksza od szerokoci okna
    if( width < height && width > 0 )
         projectionMatrix = glm::frustum( left, right, bottom*height/width, top*height/width, near, far );
    else
        // szeroko okna wiksza lub rwna wysokoci okna
        if (width >= height && height > 0)
            projectionMatrix = glm::frustum( left*width/height, right*width/height, bottom, top, near, far );
        else
            projectionMatrix = glm::frustum( left, right, bottom, top, near, far );
}

//////////////////////////////////////////////////////////////////////
// inicjalizacja staych elementw maszyny stanu OpenGL
//////////////////////////////////////////////////////////////////////
void InitScene()
{
    // kolor ta - zawarto bufora koloru
    glClearColor( 1.0f, 1.0f, 1.0f, 1.0f );

    // wczytanie shaderw i przygotowanie obsugi programu
    program = glCreateProgram();
    glAttachShader( program, LoadShader( GL_VERTEX_SHADER, "elipsoida_vs.glsl" ) );
    glAttachShader( program, LoadShader( GL_TESS_EVALUATION_SHADER, "elipsoida_tes.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "../../common/materials_static.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "../../common/light_model_static.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "../../common/blinn_phong_light.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "elipsoida_fs.glsl" ) );
    LinkValidateProgram( program );

    // generowanie identyfikatorw obiektw tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray );

    // utworzenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer );
    glBufferData( GL_ARRAY_BUFFER, sizeof( position ), position, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 2, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // nazwy plikw z obrazami trzech tekstur http://nasa3d.arc.nasa.gov/
    // tekstura Wenus utworzonej ze zdj sondy Magellan
    // tekstura z hipsometryczn (fizyczn) map Ziemi
    // tekstura Marsa utworzonej ze zdj sond Viking
    const char *fileNames[TEXTURE_SIZE] = 
    {
        "../../media/nasa3d.arc.nasa.gov/ven0aaa2.tif",
        "../../media/nasa3d.arc.nasa.gov/ear0xuu2.tif",
        "../../media/nasa3d.arc.nasa.gov/mar0kuu2.tif"
    };

    // generowanie trzech identyfikatorw obiektw tekstury
    glGenTextures( TEXTURE_SIZE, textures );

    // utworzenie trzech tekstur
    for( int i = 0; i < TEXTURE_SIZE; i++ )
    {
        // utworzenie obiektu tekstury
        glBindTexture( GL_TEXTURE_2D, textures[i] );

        // odczyt kolejnych plikw i ewentualny komunikat o bdzie
        if( !LoadTexture( fileNames[i], GL_TEXTURE_2D ) )
        {
            std::cout << "Niepoprawny odczyt pliku " << fileNames[i] << std::endl;
            exit( 0 );
        }

        // generownanie mipmap
        glGenerateMipmap( GL_TEXTURE_2D );

        // filtr pomniejszajcy
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR );

        // filtr powikszajcy
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );

        // parametry zawijania tekstury
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE );
    }

    // przeczenie na tekstur domyln
    glBindTexture( GL_TEXTURE_2D, 0 );

    // wczenie testu bufora gbokoci
    glEnable( GL_DEPTH_TEST );

    // wielko wejciowego pata wierzchokw
    glPatchParameteri( GL_PATCH_VERTICES, 4 );

    // wczenie mechanizmw uywanych podczas renderingu tekstu
    InitDrawText();
}

//////////////////////////////////////////////////////////////////////
// usunicie obiektw OpenGL
//////////////////////////////////////////////////////////////////////
void DeleteScene()
{
    // porzdki
    glDeleteProgram( program );
    glDeleteBuffers( 1, &vertexBuffer );
    glDeleteVertexArrays( 1, &vertexArray );
    glDeleteTextures( TEXTURE_SIZE, textures );

    // usunicie mechanizmw uywanych podczas renderingu tekstu
    DeleteDrawText();
}
