//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>
#include "shaders.h"
#include "teapot.h"

//////////////////////////////////////////////////////////////////////
// rozmiary bryy obcinania
//////////////////////////////////////////////////////////////////////
GLfloat left = -2.0f;
GLfloat right = 2.0f;
GLfloat bottom = -2.0f;
GLfloat top = 2.0f;
GLfloat near = 3.0f;
GLfloat far = 7.0f;

//////////////////////////////////////////////////////////////////////
// macierz rzutowania
//////////////////////////////////////////////////////////////////////
glm::mat4x4 projectionMatrix;

//////////////////////////////////////////////////////////////////////
// wspczynniki skalowania obiektu
//////////////////////////////////////////////////////////////////////
GLfloat scale = 1.0f;

//////////////////////////////////////////////////////////////////////
// kty obrotu obiektu
//////////////////////////////////////////////////////////////////////
GLfloat rotateX = 15.0f;
GLfloat rotateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// przesunicie obiektu
//////////////////////////////////////////////////////////////////////
GLfloat translateX = 0.0f;
GLfloat translateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// numeracja obiektw programu i obiektw tablic wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    TEAPOT,         // czajnik
    FLOOR,          // powierzchnia odbijajca
    PROGRAM_SIZE,
    VERTEX_ARRAY_SIZE = PROGRAM_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw programu
//////////////////////////////////////////////////////////////////////
GLuint program[PROGRAM_SIZE];

//////////////////////////////////////////////////////////////////////
// numeracja obiektw bufora wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    POSITION_TEAPOT,
    NORMAL_TEAPOT,
    POSITION_FLOOR,
    COLOR_FLOOR,
    VERTEX_BUFFER_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw bufora z danymi tablicy wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexBuffer[VERTEX_BUFFER_SIZE];

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu bufora z danymi tablic indeksw wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint indicesBuffer;

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexArray[VERTEX_ARRAY_SIZE];

//////////////////////////////////////////////////////////////////////
// wsprzdne wierzchokw trjktw skadajcych si na obiekt
//////////////////////////////////////////////////////////////////////
GLfloat position[8*3*3] =
{
    -0.5f, 0.0f, 0.0f,
    -0.5f, 0.0f, 0.5f,
    0.0f, 0.0f, 0.5f,

    -0.5f, 0.0f, 0.0f,
    0.0f, 0.0f, 0.5f,
    0.0f, 0.0f, 0.0f,

    0.0f, 0.0f, 0.0f,
    0.0f, 0.0f, 0.5f,
    0.5f, 0.0f, 0.5f,

    0.0f, 0.0f, 0.0f,
    0.5f, 0.0f, 0.5f,
    0.5f, 0.0f, 0.0f,

    -0.5f, 0.0f, -0.5f,
    -0.5f, 0.0f, 0.0f,
    0.0f, 0.0f, 0.0f,

    -0.5f, 0.0f, -0.5f,
    0.0f, 0.0f, 0.0f,
    0.0f, 0.0f, -0.5f,

    0.0f, 0.0f, -0.5f,
    0.0f, 0.0f, 0.0f,
    0.5f, 0.0f, 0.0f,

    0.0f, 0.0f, -0.5f,
    0.5f, 0.0f, 0.0f,
    0.5f, 0.0f, -0.5f
};

//////////////////////////////////////////////////////////////////////
// skadowe kolorw wierzchokw trjktw skadajcych si na obiekt
//////////////////////////////////////////////////////////////////////
const GLfloat color[8*3*4] =
{
    0.000000f, 0.000000f, 1.000000f, 0.500000f,
    0.000000f, 0.000000f, 1.000000f, 0.500000f,
    0.000000f, 0.000000f, 1.000000f, 0.500000f,

    0.000000f, 0.000000f, 1.000000f, 0.500000f,
    0.000000f, 0.000000f, 1.000000f, 0.500000f,
    0.000000f, 0.000000f, 1.000000f, 0.500000f,

    0.752941f, 0.752941f, 0.752941f, 0.700000f,
    0.752941f, 0.752941f, 0.752941f, 0.700000f,
    0.752941f, 0.752941f, 0.752941f, 0.700000f,

    0.752941f, 0.752941f, 0.752941f, 0.700000f,
    0.752941f, 0.752941f, 0.752941f, 0.700000f,
    0.752941f, 0.752941f, 0.752941f, 0.700000f,

    0.752941f, 0.752941f, 0.752941f, 0.700000f,
    0.752941f, 0.752941f, 0.752941f, 0.700000f,
    0.752941f, 0.752941f, 0.752941f, 0.700000f,

    0.752941f, 0.752941f, 0.752941f, 0.700000f,
    0.752941f, 0.752941f, 0.752941f, 0.700000f,
    0.752941f, 0.752941f, 0.752941f, 0.700000f,

    0.000000f, 0.000000f, 1.000000f, 0.500000f,
    0.000000f, 0.000000f, 1.000000f, 0.500000f,
    0.000000f, 0.000000f, 1.000000f, 0.500000f,

    0.000000f, 0.000000f, 1.000000f, 0.500000f,
    0.000000f, 0.000000f, 1.000000f, 0.500000f,
    0.000000f, 0.000000f, 1.000000f, 0.500000f
};

//////////////////////////////////////////////////////////////////////
// numery indeksw poszczeglnych atrybutw wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    POSITION,
    NORMAL,
    COLOR
};

//////////////////////////////////////////////////////////////////////
// wywietlenie paszczyzny odbicia
//////////////////////////////////////////////////////////////////////
void DisplayFloor()
{
    // macierz modelu-widoku = macierz jednostkowa
    glm::mat4x4 modelViewMatrix = glm::mat4x4( 1.0 );

    // przesunicie ukadu wsprzdnych obiektu do rodka bryy obcinania
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( 0.0f, 0.0f, -(near+far)/2.0f ) );

    // skalowanie obiektu
    modelViewMatrix = glm::scale( modelViewMatrix, glm::vec3( scale, scale, scale ) );

    // przesunicie obiektu
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( translateX, translateY, 0.0f ) );

    // obroty obiektu
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateX, glm::vec3( 1.0f, 0.0f, 0.0f ) );
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateY, glm::vec3( 0.0f, 1.0f, 0.0f ) );

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[FLOOR] );

    // wczenie programu
    glUseProgram( program[FLOOR] );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    glm::mat4x4 modelViewProjectionMatrix = projectionMatrix * modelViewMatrix;
    glUniformMatrix4fv( glGetUniformLocation( program[FLOOR], "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawArraysInstanced( GL_TRIANGLES, 0, 8 * 3, 16 );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );
}

//////////////////////////////////////////////////////////////////////
// wywietlenie obiektw wiata
// addTranslate - dodatkowy wektor przesunicia wsprzdnych
//                obiektw wiata
// addScale - dodatkowy wektor skalowania (odbicia) wsprzdnych
//            obiektw wiata i wsprzdnych wektora normalnego
//////////////////////////////////////////////////////////////////////
void DisplayWorld( const glm::vec3 addTranslate, const glm::vec3 addScale )
{
    // macierz modelu-widoku = macierz jednostkowa
    glm::mat4x4 modelViewMatrix = glm::mat4x4( 1.0 );

    // przesunicie ukadu wsprzdnych obiektu do rodka bryy obcinania
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( 0.0f, 0.0f, -(near+far)/2.0f ) );

    // skalowanie obiektu
    modelViewMatrix = glm::scale( modelViewMatrix, 0.3f * glm::vec3( scale, scale, scale ) );

    // przesunicie obiektu
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( translateX, translateY, 0.0f ) );

    // obroty obiektu
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateX, glm::vec3( 1.0f, 0.0f, 0.0f ) );
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateY, glm::vec3( 0.0f, 1.0f, 0.0f ) );

    // dodatkowe przesunicie obiektu
    modelViewMatrix = glm::translate( modelViewMatrix, addTranslate );

    // dodatkowe skalowanie obiektu (odbicie)
    modelViewMatrix = glm::scale( modelViewMatrix, addScale );

    // odwrcona macierz modelu-widoku niezbdna do przeksztace
    // do ukadu wsprzdnych obiektu
    glm::mat4x4 modelViewMatrixInverse( glm::inverse( modelViewMatrix ) );

    // transformacja kierunku wiata do ukadu wsprzdnych obiektu
    glm::vec4 lightPosition( 0.0f, 0.0f, 1.0f, 0.0f );
    lightPosition = modelViewMatrixInverse * lightPosition;
    lightPosition = glm::normalize( lightPosition );

    // przeksztacenie pooenia obserwatora do ukadu wsprzdnych obiektu
    glm::vec4 eyePosition( 0.0f, 0.0f, 0.0f, 1.0f );
    eyePosition = modelViewMatrixInverse * eyePosition;

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[TEAPOT] );

    // wczenie programu
    glUseProgram( program[TEAPOT] );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    glm::mat4x4 modelViewProjectionMatrix = projectionMatrix * modelViewMatrix;
    glUniformMatrix4fv( glGetUniformLocation( program[TEAPOT], "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // zaadowanie kierunku rda wiata i pooenia obserwatora w ukadzie wsprzdnych obiektu
    glUniform4fv( glGetUniformLocation( program[TEAPOT], "lightSource[0].position" ), 1, glm::value_ptr( lightPosition ) );
    glUniform4fv( glGetUniformLocation( program[TEAPOT], "eyePosition" ), 1, glm::value_ptr( eyePosition ) );

    // zaadowanie dodatkowego wektora skalowania (odbicia) wsprzdnych wektora normalnego
    glUniform3fv( glGetUniformLocation( program[TEAPOT], "addScale" ), 1, glm::value_ptr( addScale ) );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawElements( GL_TRIANGLES, TEAPOT_INDICES_COUNT * 3, GL_UNSIGNED_INT, NULL );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );
}

//////////////////////////////////////////////////////////////////////
// funkcja generujca scen 3D
//////////////////////////////////////////////////////////////////////
void DisplayScene()
{
    // czyszczenie bufora koloru, bufora gbokoci i bufora szablonu
    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT );

    // wczenie nierysowania tylnej strony wieloktw
    glEnable( GL_CULL_FACE );

    // wyczenie bufora gbokoci
    glDisable( GL_DEPTH_TEST );

    // wczenie bufora szablonu
    glEnable( GL_STENCIL_TEST );

    // test bufora szablonu
    glStencilFunc( GL_ALWAYS, 0x01, 0xFF );

    // okrelenie operacji na buforze szablonu
    glStencilOp( GL_REPLACE, GL_REPLACE, GL_REPLACE );

    // wyczenie zapisu skadowych RGBA do bufora kolorw
    glColorMask( GL_FALSE, GL_FALSE, GL_FALSE, GL_FALSE );

    // wywietlenie paszczyzny odbicia przy wczonym buforze
    // szablonu w celu  utworzenia szablonu, ktry zostanie
    // wykorzystany do narysowania obiektw odbitych
    DisplayFloor();

    // wczenie zapisu skadowych RGBA do bufora kolorw
    glColorMask( GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE );

    // wczenie testu bufora gbokoci
    glEnable( GL_DEPTH_TEST );

    // wyczenie nierysowania tylnej strony wieloktw
    glDisable( GL_CULL_FACE );

    // test bufora szablonu
    glStencilFunc( GL_EQUAL, 0x01, 0xFF );

    // okrelenie operacji na buforze szablonu
    glStencilOp( GL_KEEP, GL_KEEP, GL_KEEP );

    // w wiecie lustrzanym trzeba zmieni orientacj wierzchokw prymityww
    glFrontFace( GL_CW );

    // wywietlenie obiektw wiata "lustrzanego"
    DisplayWorld( glm::vec3( 0.0f, -3.0f, 0.0f ), glm::vec3( 1.0f, -1.0f, 1.0f ) );

    // powrt do normalnej orientacji wierzchokw prymityww
    glFrontFace( GL_CCW );

    // wyczenie bufora szablonu
    glDisable( GL_STENCIL_TEST );

    // wczenie mieszania kolorw
    glEnable( GL_BLEND );

    // wspczynniki rwnania mieszania kolorw
    glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );

    // wywietlenie paszczyzny odbicia
    DisplayFloor();

    // wyczenie mieszania kolorw
    glDisable( GL_BLEND );

    // wywietlenie obiektw wiata "normalnego"
    DisplayWorld( glm::vec3( 0.0f, 3.0f, 0.0f ), glm::vec3( 1.0f, 1.0f, 1.0f ) );
}

//////////////////////////////////////////////////////////////////////
// zmiana wielkoci okna
//////////////////////////////////////////////////////////////////////
void Reshape( int width, int height )
{
    // obszar renderingu - cae okno
    glViewport( 0, 0, width, height );

    // parametry bryy obcinania - rzutowanie perspektywiczne
    // wysoko okna wiksza od szerokoci okna
    if( width < height && width > 0 )
         projectionMatrix = glm::frustum( left, right, bottom*height/width, top*height/width, near, far );
    else
        // szeroko okna wiksza lub rwna wysokoci okna
        if (width >= height && height > 0)
            projectionMatrix = glm::frustum( left*width/height, right*width/height, bottom, top, near, far );
        else
            projectionMatrix = glm::frustum( left, right, bottom, top, near, far );
}

//////////////////////////////////////////////////////////////////////
// inicjalizacja staych elementw maszyny stanu OpenGL
//////////////////////////////////////////////////////////////////////
void InitScene()
{
    // kolor ta - zawarto bufora koloru
    glClearColor( 1.0f, 1.0f, 1.0f, 1.0f );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[FLOOR] = glCreateProgram();
    glAttachShader( program[FLOOR], LoadShader( GL_VERTEX_SHADER, "odbicie_powierzchnia_vs.glsl" ) );
    glAttachShader( program[FLOOR], LoadShader( GL_FRAGMENT_SHADER, "odbicie_powierzchnia_fs.glsl" ) );
    LinkValidateProgram( program[FLOOR] );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[TEAPOT] = glCreateProgram();
    glAttachShader( program[TEAPOT], LoadShader( GL_VERTEX_SHADER, "odbicie_obiekt_vs.glsl" ) );
    glAttachShader( program[TEAPOT], LoadShader( GL_FRAGMENT_SHADER, "../../common/light_model_static.glsl" ) );
    glAttachShader( program[TEAPOT], LoadShader( GL_FRAGMENT_SHADER, "../../common/materials_static.glsl" ) );
    glAttachShader( program[TEAPOT], LoadShader( GL_FRAGMENT_SHADER, "../../common/blinn_phong_light.glsl" ) );
    glAttachShader( program[TEAPOT], LoadShader( GL_FRAGMENT_SHADER, "odbicie_obiekt_fs.glsl" ) );
    LinkValidateProgram( program[TEAPOT] );

    // utworzenie obiektu tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray[FLOOR] );
    glBindVertexArray( vertexArray[FLOOR] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[POSITION_FLOOR] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION_FLOOR] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( position ), position, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[COLOR_FLOOR] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[COLOR_FLOOR] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( color ), color, GL_STATIC_DRAW );
    glVertexAttribPointer( COLOR, 4, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( COLOR );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // generowanie identyfikatora obiektu tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray[TEAPOT] );

    // utworzenie pierwszego obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[TEAPOT] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[POSITION_TEAPOT] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION_TEAPOT] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( teapotPosition ), teapotPosition, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[NORMAL_TEAPOT] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[NORMAL_TEAPOT] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( teapotNormal ), teapotNormal, GL_STATIC_DRAW );
    glVertexAttribPointer( NORMAL, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( NORMAL );

    // utworzenie obiektu bufora indeksw wierzchokw i zaadowanie danych
    glGenBuffers( 1, &indicesBuffer );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, indicesBuffer );
    glBufferData( GL_ELEMENT_ARRAY_BUFFER, sizeof( teapotIndices ), teapotIndices, GL_STATIC_DRAW );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wczenie testu bufora gbokoci
    glEnable( GL_DEPTH_TEST );
}

//////////////////////////////////////////////////////////////////////
// usunicie obiektw OpenGL
//////////////////////////////////////////////////////////////////////
void DeleteScene()
{
    // porzdki
    glDeleteProgram( program[FLOOR] );
    glDeleteProgram( program[TEAPOT] );
    glDeleteBuffers( VERTEX_BUFFER_SIZE, vertexBuffer );
    glDeleteBuffers( 1, &indicesBuffer );
    glDeleteVertexArrays( VERTEX_ARRAY_SIZE, vertexArray );
}
