//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>
#include "text.h"
#include "shaders.h"
#include "cp1250.h"

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu programu
//////////////////////////////////////////////////////////////////////
GLuint textProgram = 0;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu bufora z danymi tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint textBuffer = 0;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint textVertexArray = 0;

//////////////////////////////////////////////////////////////////////
// numery pooenia poszczeglnych atrybutw wierzchokw
//////////////////////////////////////////////////////////////////////
#define POSITION 0

//////////////////////////////////////////////////////////////////////
// pooenie danych wsprzdnych punktw poszczeglnych znakw
// w buforze; dane znakw zaczynaj si od spacji
//////////////////////////////////////////////////////////////////////
int textPositions[224];

//////////////////////////////////////////////////////////////////////
// liczba danych wsprzdnych punktw poszczeglnych znakw
// w buforze; dane znakw zaczynaj si od spacji
//////////////////////////////////////////////////////////////////////
int textSizes[224];

//////////////////////////////////////////////////////////////////////
// inicjacja mechanizmw uywanych podczas renderingu tekstu
//////////////////////////////////////////////////////////////////////
void InitDrawText()
{
    // wczytanie i przygotowanie obsugi programu rysujcego napisy
    if( !textProgram )
    {
        textProgram = glCreateProgram();
        glAttachShader( textProgram, LoadShader( GL_VERTEX_SHADER, "../../common/text_vs.glsl" ) );
        glAttachShader( textProgram, LoadShader( GL_FRAGMENT_SHADER, "../../common/text_fs.glsl" ) );
        LinkValidateProgram( textProgram );
    }

    // obiekt tablic wierzchokw z buforem tablic wierzchokw
    // na wsprzdne punktw z tekstem (bez adowania danych)
    if( !textBuffer && !textVertexArray )
    {
        // maksymalnia pojemno bufora na dane wsprzdnych punktw
        const int MAX_BUF = 8912;

        // bufor wsprzdnych punktw
        GLfloat buf[MAX_BUF];

        // pooenie biecych danych w buforze wsprzdnych punktw
        int bufPos = 0;

        // generowanie poszczeglnych liter, poczwszy od spacji
        for( unsigned int i = 0; i < 224 && bufPos < MAX_BUF; i++ )
        {
            // pooenie danych w buforze; warto dzielimy przez 2,
            // bowiem pooenie kadego wierzchoka opisuj dwie wsprzdne
            textPositions[i] = bufPos/2;

            // przygotowanie danych pojedynczego znaku
            for( int row = 0; row < 16; row++ )
            {
                // pobranie danych pojedynczego wiersza pikseli w znaku
                // (dane znakw zaczynaj si od spacji)
                const unsigned int chr = font8x16CP1250[i][row];
                for( int col = 0; col < 8; col++ )
                {
                    // test obecnoci poszczeglnych bitw oraz test pojemoci bufora
                    if( ((chr >> col) & 0x01) == 0x01 && bufPos < MAX_BUF - 2 )
                    {
                        buf[bufPos + 0] = static_cast< GLfloat >( 7 - col );
                        buf[bufPos + 1] = static_cast< GLfloat >( row );
                        bufPos += 2;
                    }
                }
            }

            // rozmiar danych w buforze; warto dzielimy przez 2,
            // bowiem pooenie kadego wierzchoka opisuj dwie wsprzdne
            textSizes[i] = bufPos/2 - textPositions[i];
        }

        // generowanie identyfikatora obiektu tablic wierzchokw
        glGenVertexArrays( 1, &textVertexArray );

        // utworzenie obiektu tablic wierzchokw
        glBindVertexArray( textVertexArray );

        // generowanie VBO
        glGenBuffers( 1, &textBuffer );
        glBindBuffer( GL_ARRAY_BUFFER, textBuffer );
        glBufferData( GL_ARRAY_BUFFER, bufPos * sizeof( GLfloat ), buf, GL_STATIC_DRAW );
        glVertexAttribPointer( POSITION, 2, GL_FLOAT, GL_FALSE, 0, NULL );
        glEnableVertexAttribArray( POSITION );

        // wyczenie obiektu tablic wierzchokw
        glBindVertexArray( 0 );
    }
}

//////////////////////////////////////////////////////////////////////
// funkcja rysujca napis w wybranym miejscu
// x, y - wsprzdne okienkowe pocztku napisu
// str - tekst napisu
// color - kolor napisu, domylnie czarny
//////////////////////////////////////////////////////////////////////
void DrawText8x16( const int x, const int y, const std::string &str,
                   const glm::vec4 &color )
{
    // pobranie rozmiarw okna renderingu
    GLint viewport[4];
    glGetIntegerv( GL_VIEWPORT, viewport );

    // przygotowanie macierzy rzutowania prostoktnego 2D; wykorzystujemy
    // wycznie macierz rzutowania, bowiem macierz modelu-widoku jest
    // w tym przypadku macierz jednostkow
    glm::mat4x4 projectionMatrix( glm::ortho( 0.0f, static_cast<GLfloat>( viewport[2] ), 0.0f, static_cast<GLfloat>( viewport[3] ) ) );

    // sprawdzenie czy wczone jest generowanie wielkoci punktu w shaderze
    GLboolean programPointSize = glIsEnabled( GL_PROGRAM_POINT_SIZE );
    if( programPointSize )
        glDisable( GL_PROGRAM_POINT_SIZE );

    // rozmiar punktw
    glPointSize( 1.5f );

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( textVertexArray );

    // wczenie programu
    glUseProgram( textProgram );

    // zaadowanie zmiennych jednorodnych - iloczynu macierzy 
    // modelu-widoku i rzutowania oraz koloru punktu
    glUniformMatrix4fv( glGetUniformLocation( textProgram, "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( projectionMatrix ) );
    glUniform4fv( glGetUniformLocation( textProgram, "color" ), 1, glm::value_ptr( color ) );

    // rysowanie poszczeglnych liter
    for( unsigned int i = 0; i < str.length(); i++ )
    {
        // pobranie znaku
        const unsigned char ch = static_cast<unsigned char>( str[i] );

        // pomijamy spacj i znaki sterujce
        if( ch > 32 )
        {
            // pooenie znaku
            glUniform2f( glGetUniformLocation( textProgram, "pos" ), static_cast<GLfloat>( x + i * 8 ), static_cast<GLfloat>( y ) );

            // narysowanie danych zawartych w tablicach wierzchokw
            glDrawArrays( GL_POINTS, textPositions[ch -32], textSizes[ch - 32] );
        }
    }

    // wyczenie programu
    glUseProgram( 0 );

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // w razie potrzeby wczamy generowanie wielkoci punktu w shaderze
    if( programPointSize )
        glEnable( GL_PROGRAM_POINT_SIZE );
}

//////////////////////////////////////////////////////////////////////
// usunicie mechanizmw uywanych podczas renderingu tekstu
//////////////////////////////////////////////////////////////////////
void DeleteDrawText()
{
    // porzdki
    glDeleteProgram( textProgram );
    glDeleteBuffers( 1, &textBuffer );
    glDeleteVertexArrays( 1, &textVertexArray );

    // wyzerowanie identyfikatorw i pozostaych zmiennych
    textProgram = 0;
    textBuffer = 0;
    textVertexArray = 0;
}

