//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

#include <string>
#include <sstream>
#include <iomanip>
#include <cstring>
#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>
#include "shaders.h"
#include "teapot.h"
#include "materials.h"
#include "text.h"

//////////////////////////////////////////////////////////////////////
// rozmiary bryy obcinania
//////////////////////////////////////////////////////////////////////
GLfloat left = -2.0f;
GLfloat right = 2.0f;
GLfloat bottom = -2.0f;
GLfloat top = 2.0f;
GLfloat near = 3.0f;
GLfloat far = 7.0f;

//////////////////////////////////////////////////////////////////////
// macierz rzutowania
//////////////////////////////////////////////////////////////////////
glm::mat4x4 projectionMatrix;

//////////////////////////////////////////////////////////////////////
// wspczynniki skalowania obiektu
//////////////////////////////////////////////////////////////////////
GLfloat scale = 0.5f;

//////////////////////////////////////////////////////////////////////
// kty obrotu obiektu
//////////////////////////////////////////////////////////////////////
GLfloat rotateX = 0.0f;
GLfloat rotateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// przesunicie obiektu
//////////////////////////////////////////////////////////////////////
GLfloat translateX = 0.0f;
GLfloat translateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// wybrany materia
//////////////////////////////////////////////////////////////////////
int material = MTL_BRASS;

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw programu
//////////////////////////////////////////////////////////////////////
GLuint program;

//////////////////////////////////////////////////////////////////////
// numeracja obiektw bufora wierzchokw
// numery pooenia poszczeglnych atrybutw wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    POSITION,
    NORMAL,
    VERTEX_BUFFER_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw bufora z danymi tablic wierzchokw:
// wsprzdnymi wierzchokw i wektorw normalnych
//////////////////////////////////////////////////////////////////////
GLuint vertexBuffer[VERTEX_BUFFER_SIZE];

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw bufora z danymi tablic
// indeksw wierzchokw obiektw
//////////////////////////////////////////////////////////////////////
GLuint indicesBuffer;

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexArray;

//////////////////////////////////////////////////////////////////////
// funkcja generujca scen 3D
//////////////////////////////////////////////////////////////////////
void DisplayScene()
{
    // czyszczenie bufora koloru i bufora gbokoci
    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

    // macierz modelu-widoku = macierz jednostkowa
    glm::mat4x4 modelViewMatrix = glm::mat4x4( 1.0 );

    // przesunicie obserwatora tak, aby ukad wsprzdnych obiektu by w rodku bryy obcinania
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( 0.0f, 0.0f, -(near+far)/2.0f ) );

    // skalowanie obiektu
    modelViewMatrix = glm::scale( modelViewMatrix, glm::vec3( scale, scale, scale ) );

    // przesunicie obiektu
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( translateX, translateY, 0.0f ) );

    // obroty obiektu
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateX, glm::vec3( 1.0f, 0.0f, 0.0f ) );
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateY, glm::vec3( 0.0f, 1.0f, 0.0f ) );

    // odwrcona macierz modelu-widoku niezbdna do przeksztace
    // do ukadu wsprzdnych obiektu
    glm::mat4x4 modelViewMatrixInverse( glm::inverse( modelViewMatrix ) );

    // transformacja kierunku wiata do ukadu wsprzdnych obiektu
    glm::vec4 lightPosition( 0.0f, 0.0f, 1.0f, 0.0f );
    lightPosition = modelViewMatrixInverse * lightPosition;
    lightPosition = glm::normalize( lightPosition );

    // przeksztacenie pooenia obserwatora do ukadu wsprzdnych obiektu
    glm::vec4 eyePosition( 0.0f, 0.0f, 0.0f, 1.0f );
    eyePosition = modelViewMatrixInverse * eyePosition;

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray );

    // wczenie programu
    glUseProgram( program );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    glm::mat4x4 modelViewProjectionMatrix = projectionMatrix * modelViewMatrix;
    glUniformMatrix4fv( glGetUniformLocation( program, "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // zaadowanie zmiennej jednorodnej - rodzaj materiau
    glUniform1i( glGetUniformLocation( program, "material" ), material );

    // zaadowanie kierunku rda wiata i pooenia obserwatora w ukadzie wsprzdnych obiektu
    glUniform4fv( glGetUniformLocation( program, "lightSource[0].position" ), 1, glm::value_ptr( lightPosition ) );
    glUniform4fv( glGetUniformLocation( program, "eyePosition" ), 1, glm::value_ptr( eyePosition ) );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawElements( GL_TRIANGLES, TEAPOT_INDICES_COUNT * 3, GL_UNSIGNED_INT, NULL );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wywietlenie wartoci zmiennych stanu GL_SAMPLE_BUFFERS i GL_SAMPLES
    GLint sampleBuffers, samples;
    glGetIntegerv( GL_SAMPLE_BUFFERS, &sampleBuffers );
    glGetIntegerv( GL_SAMPLES, &samples );
    std::ostringstream txt;
    txt << "GL_SAMPLE_BUFFERS: " << sampleBuffers;
    DrawText8x16( 3, 3, txt.str() );
    txt.str( "" );
    txt << "GL_SAMPLES: " << samples;
    DrawText8x16( 3, 20, txt.str() );
}

//////////////////////////////////////////////////////////////////////
// zmiana wielkoci okna
//////////////////////////////////////////////////////////////////////
void Reshape( int width, int height )
{
    // obszar renderingu - cae okno
    glViewport( 0, 0, width, height );

    // parametry bryy obcinania - rzutowanie perspektywiczne
    // wysoko okna wiksza od szerokoci okna
    if( width < height && width > 0 )
         projectionMatrix = glm::frustum( left, right, bottom*height/width, top*height/width, near, far );
    else
        // szeroko okna wiksza lub rwna wysokoci okna
        if (width >= height && height > 0)
            projectionMatrix = glm::frustum( left*width/height, right*width/height, bottom, top, near, far );
        else
            projectionMatrix = glm::frustum( left, right, bottom, top, near, far );
}

//////////////////////////////////////////////////////////////////////
// inicjalizacja staych elementw maszyny stanu OpenGL
//////////////////////////////////////////////////////////////////////
void InitScene()
{
    // kolor ta - zawarto bufora koloru
    glClearColor( 1.0f, 1.0f, 1.0f, 1.0f );

    // wczytanie shaderw i przygotowanie obsugi programu
    program = glCreateProgram();
    glAttachShader( program, LoadShader( GL_VERTEX_SHADER, "wieloprobkowanie_na_probke_vs.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "../../common/materials_static.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "../../common/light_model_static.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "../../common/blinn_phong_light.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "wieloprobkowanie_na_probke_fs.glsl" ) );
    LinkValidateProgram( program );

    // generowanie identyfikatora obiektu tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray );

    // utworzenie pierwszego obiektu tablic wierzchokw
    glBindVertexArray( vertexArray );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[POSITION] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( teapotPosition ), teapotPosition, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO)
    glGenBuffers( 1, &vertexBuffer[NORMAL] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[NORMAL] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( teapotNormal ), teapotNormal, GL_STATIC_DRAW );
    glVertexAttribPointer( NORMAL, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( NORMAL );

    // utworzenie obiektu bufora indeksw wierzchokw i zaadowanie danych
    glGenBuffers( 1, &indicesBuffer );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, indicesBuffer );
    glBufferData( GL_ELEMENT_ARRAY_BUFFER, sizeof( teapotIndices ), teapotIndices, GL_STATIC_DRAW );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wczenie testu bufora gbokoci
    glEnable( GL_DEPTH_TEST );

    // wczenie mechanizmw uywanych podczas renderingu tekstu
    InitDrawText();

    // wczenie wieloprbkowania
    glEnable( GL_MULTISAMPLE );

    // generowanie danych na poszczeglne prbki
    glMinSampleShading( 1.0f );
    glEnable( GL_SAMPLE_SHADING );
}

//////////////////////////////////////////////////////////////////////
// usunicie obiektw OpenGL
//////////////////////////////////////////////////////////////////////
void DeleteScene()
{
    // porzdki
    glDeleteProgram( program );
    glDeleteBuffers( VERTEX_BUFFER_SIZE, vertexBuffer );
    glDeleteBuffers( 1, &indicesBuffer );
    glDeleteVertexArrays( 1, &vertexArray );

    // usunicie mechanizmw uywanych podczas renderingu tekstu
    DeleteDrawText();
}
