//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

#include <sstream>
#include <iostream>
#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>
#include "shaders.h"
#include "text.h"

//////////////////////////////////////////////////////////////////////
// rozmiary bryy obcinania
//////////////////////////////////////////////////////////////////////
GLfloat left = -2.0f;
GLfloat right = 2.0f;
GLfloat bottom = -2.0f;
GLfloat top = 2.0f;
GLfloat near = 3.0f;
GLfloat far = 7.0f;

//////////////////////////////////////////////////////////////////////
// liczba warstw renderingu
//////////////////////////////////////////////////////////////////////
const int FRAME_LAYERS = 16;

//////////////////////////////////////////////////////////////////////
// macierz rzutowania
//////////////////////////////////////////////////////////////////////
glm::mat4x4 projectionMatrix;

//////////////////////////////////////////////////////////////////////
// tablica macierzy modelu-widoku
//////////////////////////////////////////////////////////////////////
glm::mat4x4 modelViewMatrix[FRAME_LAYERS];

//////////////////////////////////////////////////////////////////////
// wspczynniki skalowania obiektu
//////////////////////////////////////////////////////////////////////
GLfloat scale = 1.0f;

//////////////////////////////////////////////////////////////////////
// kty obrotu obiektu
//////////////////////////////////////////////////////////////////////
GLfloat rotateX = 0.0f;
GLfloat rotateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// przesunicie obiektu
//////////////////////////////////////////////////////////////////////
GLfloat translateX = 0.0f;
GLfloat translateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// numeracja obiektw programu
//////////////////////////////////////////////////////////////////////
enum
{
    MOTION_BLUR_FLAT_SHADING,       // efekt rozmycia ruchu - cieniowanie paskie
    MOTION_BLUR_SMOOTH_SHADING,     // efekt rozmycia ruchu - cieniowanie gadkie
    POSTPROCESSING_RECT,            // przetwarzanie kocowe (postprocesing)
    PROGRAM_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw programu
//////////////////////////////////////////////////////////////////////
GLuint program[PROGRAM_SIZE];

//////////////////////////////////////////////////////////////////////
// numeracja obiektw bufora wierzchokw oraz
// numery indeksw poszczeglnych atrybutw wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    POSITION,
    COLOR,
    TEX_COORD,
    POSITION_RECT = TEX_COORD,
    TEXCOORD_RECT,
    VERTEX_BUFFER_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw bufora z danymi tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexBuffer[VERTEX_BUFFER_SIZE];

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu bufora z danymi tablicy
// indeksw wierzchokw obiektu
//////////////////////////////////////////////////////////////////////
GLuint indicesBuffer;

//////////////////////////////////////////////////////////////////////
// numeracja obiektw tablic wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    CUBE,
    RECTANGLE,
    VERTEX_ARRAY_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexArray[VERTEX_ARRAY_SIZE];

//////////////////////////////////////////////////////////////////////
// wsprzdne wierzchokw trjktw skadajcych si na szecian
//////////////////////////////////////////////////////////////////////
GLfloat position[8*3] =
{
    1.0f, 1.0f, 1.0f,
    -1.0f, 1.0f, 1.0f,
    -1.0f, -1.0f, 1.0f,
    1.0f, -1.0f, 1.0f,
    1.0f, 1.0f, -1.0f,
    -1.0f, 1.0f, -1.0f,
    -1.0f, -1.0f, -1.0f,
    1.0f, -1.0f, -1.0f
};

//////////////////////////////////////////////////////////////////////
// skadowe kolorw wierzchokw trjktw skadajcych si na szecian
//////////////////////////////////////////////////////////////////////
const GLfloat color[8*3] =
{
    1.0f, 1.0f, 1.0f,
    1.0f, 1.0f, 0.0f,
    0.0f, 1.0f, 0.0f,
    0.0f, 1.0f, 1.0f,
    1.0f, 0.0f, 1.0f,
    1.0f, 0.0f, 0.0f,
    0.0f, 0.0f, 0.0f,
    0.0f, 0.0f, 1.0f
};

//////////////////////////////////////////////////////////////////////
// dane indeksw wierzchokw trjktw skadajcych si na szecian
//////////////////////////////////////////////////////////////////////
GLuint indices[12*3] =
{
    5, 1, 0,
    5, 0, 4,
    2, 3, 0,
    2, 0, 1,
    7, 4, 0,
    7, 0, 3,
    3, 2, 6,
    3, 6, 7,
    1, 6, 2,
    1, 5, 6,
    4, 6, 5,
    4, 7, 6
};

//////////////////////////////////////////////////////////////////////
// wsprzdne wierzchokw trjktw skadajcych si na kwadrat
//////////////////////////////////////////////////////////////////////
GLfloat positionRect [4*2] =
{
    -1.0f, -1.0f,
    1.0f, -1.0f,
    -1.0f, 1.0f,
    1.0f, 1.0f
};

//////////////////////////////////////////////////////////////////////
// wsprzdne tekstury w wierzchokach trjktw
// skadajcych si na kwadrat
//////////////////////////////////////////////////////////////////////
GLfloat texCoordRect [4*2] =
{
    0.0f, 0.0f,
    1.0f, 0.0f,
    0.0f, 1.0f,
    1.0f, 1.0f
};

//////////////////////////////////////////////////////////////////////
// wybr wierzchoka podstawowego
//////////////////////////////////////////////////////////////////////
GLenum provokingVertex = GL_LAST_VERTEX_CONVENTION;

//////////////////////////////////////////////////////////////////////
// interpolacja kolorw wierzchokw wczona/wyczona
//////////////////////////////////////////////////////////////////////
GLboolean shading = MOTION_BLUR_SMOOTH_SHADING;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu bufora ramki
//////////////////////////////////////////////////////////////////////
GLuint frameBuffer;

//////////////////////////////////////////////////////////////////////
// rozmiary obszarw renderingu do tekstury
//////////////////////////////////////////////////////////////////////
const int FRAME_WIDTH = 4*256;
const int FRAME_HEIGHT = 4*256;

//////////////////////////////////////////////////////////////////////
// numeracja obiektw tekstury
//////////////////////////////////////////////////////////////////////
enum
{
    COLOR0,
    DEPTH,
    RENDER_TEXTURE_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw tekstury
//////////////////////////////////////////////////////////////////////
GLuint renderTexture[RENDER_TEXTURE_SIZE];

//////////////////////////////////////////////////////////////////////
// funkcja generujca scen 3D
//////////////////////////////////////////////////////////////////////
void DisplayScene()
{
    // bufor ramki do zapisu = obiekt bufora ramki
    glBindFramebuffer( GL_DRAW_FRAMEBUFFER, frameBuffer );

    // czyszczenie bufora koloru i bufora gbokoci
    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

    // przepisanie danych w tablicy macierzy modelu-widoku
    for( int layer = FRAME_LAYERS - 2; layer > 0 ; layer-- )
    {
        modelViewMatrix[layer] = modelViewMatrix[layer-1];
    }

    // macierz modelu-widoku = macierz jednostkowa
    modelViewMatrix[0] = glm::mat4x4( 1.0 );

    // przesunicie ukadu wsprzdnych obiektu do rodka bryy obcinania
    modelViewMatrix[0] = glm::translate( modelViewMatrix[0], glm::vec3( 0.0f, 0.0f, -(near+far)/2.0f ) );

    // skalowanie obiektu
    modelViewMatrix[0] = glm::scale( modelViewMatrix[0], glm::vec3( scale, scale, scale ) );

    // przesunicie obiektu
    modelViewMatrix[0] = glm::translate( modelViewMatrix[0], glm::vec3( translateX, translateY, 0.0f ) );

    // wymuszenie obrotw obiektu
    rotateX += 1.2f;
    rotateY -= 1.7f;

    // obroty obiektu
    modelViewMatrix[0] = glm::rotate( modelViewMatrix[0], rotateX, glm::vec3( 1.0f, 0.0f, 0.0f ) );
    modelViewMatrix[0] = glm::rotate( modelViewMatrix[0], rotateY, glm::vec3( 0.0f, 1.0f, 0.0f ) );

    // wybr wierzchoka podstawowego
    glProvokingVertex( provokingVertex );

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[CUBE] );

    // wczenie programu
    glUseProgram( program[shading] );

    // zaadowanie zmiennej jednorodnej - iloczynw macierzy modelu-widoku i rzutowania
    for( int layer = 0; layer < FRAME_LAYERS; layer++ )
    {
        std::ostringstream txt;
        txt << "modelViewProjectionMatrix[" << layer << "]";
        glm::mat4x4 modelViewProjectionMatrix = projectionMatrix * modelViewMatrix[layer];
        glUniformMatrix4fv( glGetUniformLocation( program[shading], txt.str().c_str() ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );
    }

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawElements( GL_TRIANGLES, 12 * 3, GL_UNSIGNED_INT, NULL );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // bufor ramki do zapisu = domylny bufor ramki
    glBindFramebuffer( GL_DRAW_FRAMEBUFFER, 0 );

    // czyszczenie bufora koloru
    glClear( GL_COLOR_BUFFER_BIT );

    // wyczenie testu bufora gbokoci
    glDisable( GL_DEPTH_TEST );

    // wybr tablicy tekstur 2D
    glBindTexture( GL_TEXTURE_2D_ARRAY, renderTexture[COLOR0] );

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[RECTANGLE] );

    // wczenie programu
    glUseProgram( program[POSTPROCESSING_RECT] );

    // zaadowanie zmiennej jednorodnej - uchwyt tekstury
    glUniform1i( glGetUniformLocation( program[POSTPROCESSING_RECT], "tex" ), 0 );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawArrays( GL_TRIANGLE_STRIP, 0, 4 );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wyczenie tablicy tekstur 2D
    glBindTexture( GL_TEXTURE_2D_ARRAY, 0 );

    // wczenie testu bufora gbokoci
    glEnable( GL_DEPTH_TEST );

    // wyczenie tesktu gbokoci
    glDisable( GL_DEPTH_TEST );

    // wypisanie informacji o wierzchoku podstawowym
    DrawText8x16( 3, 3, provokingVertex == GL_FIRST_VERTEX_CONVENTION ?
                        "Wierzchoek podstawowy: GL_FIRST_VERTEX_CONVENTION" :
                        "Wierzchoek podstawowy: GL_LAST_VERTEX_CONVENTION" );

    // wypisanie informacji o rodzaju cieniowania
    DrawText8x16( 3, 21, shading == MOTION_BLUR_SMOOTH_SHADING ?
                         "Cieniowanie gadkie (smooth)" :
                         "Cieniowanie paskie (flat)" );

    // wczenie testu gbokoci
    glEnable( GL_DEPTH_TEST );
}

//////////////////////////////////////////////////////////////////////
// zmiana wielkoci okna
//////////////////////////////////////////////////////////////////////
void Reshape( int width, int height )
{
    // obszar renderingu - cae okno
    glViewportIndexedf( 0, 0.0, 0.0, static_cast<GLfloat>( width ), static_cast<GLfloat>( height ) );

    // parametry bryy obcinania - rzutowanie perspektywiczne
    // wysoko okna wiksza od szerokoci okna
    if( width < height && width > 0 )
         projectionMatrix = glm::frustum( left, right, bottom*height/width, top*height/width, near, far );
    else
        // szeroko okna wiksza lub rwna wysokoci okna
        if (width >= height && height > 0)
            projectionMatrix = glm::frustum( left*width/height, right*width/height, bottom, top, near, far );
        else
            projectionMatrix = glm::frustum( left, right, bottom, top, near, far );
}

//////////////////////////////////////////////////////////////////////
// inicjalizacja staych elementw maszyny stanu OpenGL
//////////////////////////////////////////////////////////////////////
void InitScene()
{
    // kolor ta - zawarto bufora koloru
    glClearColor( 1.0f, 1.0f, 1.0f, 1.0f );

    // utworzenie obiektw tekstur - na dane bufora koloru i bufora gbokoci
    glGenTextures( RENDER_TEXTURE_SIZE, renderTexture );
    glBindTexture( GL_TEXTURE_2D_ARRAY, renderTexture[COLOR0] );
    glTexImage3D( GL_TEXTURE_2D_ARRAY, 0, GL_RGBA, FRAME_WIDTH, FRAME_HEIGHT, FRAME_LAYERS, 0, GL_RGBA, GL_UNSIGNED_BYTE, NULL );
    glTexParameteri( GL_TEXTURE_2D_ARRAY, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
    glTexParameteri( GL_TEXTURE_2D_ARRAY, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE );
    glTexParameteri( GL_TEXTURE_2D_ARRAY, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE );
    glBindTexture( GL_TEXTURE_2D_ARRAY, renderTexture[DEPTH] );
    glTexImage3D( GL_TEXTURE_2D_ARRAY, 0, GL_DEPTH_COMPONENT, FRAME_WIDTH, FRAME_HEIGHT, FRAME_LAYERS, 0, GL_DEPTH_COMPONENT, GL_FLOAT, NULL );
    glTexParameteri( GL_TEXTURE_2D_ARRAY, GL_TEXTURE_MIN_FILTER, GL_LINEAR );

    // utworzenie obiektu bufora ramki i doczenie dwch obiektw bufora renderingu
    glGenFramebuffers( 1, &frameBuffer );
    glBindFramebuffer( GL_DRAW_FRAMEBUFFER, frameBuffer );
    glFramebufferTexture( GL_DRAW_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, renderTexture[COLOR0], 0 );
    glFramebufferTexture( GL_DRAW_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, renderTexture[DEPTH], 0 );

    // sprawdzenie poprawnoci i kompletnoci obiektu bufora ramki
    GLenum error = glCheckFramebufferStatus( GL_DRAW_FRAMEBUFFER );
    if( error != GL_FRAMEBUFFER_COMPLETE )
    {
        std::cout << "Niepoprawny obiekt bufora ramki" << std::endl;
        exit( 0 );
    }

    // obszar renderingu - caa powierzchnia bufora ramki
    glViewportIndexedf( 1, 0.0, 0.0, static_cast<GLfloat>( FRAME_WIDTH ), static_cast<GLfloat>( FRAME_HEIGHT ) );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[MOTION_BLUR_FLAT_SHADING] = glCreateProgram();
    GLuint vertShader = LoadShader( GL_VERTEX_SHADER, "rozmycie_ruchu_vs.glsl" );
    glAttachShader( program[MOTION_BLUR_FLAT_SHADING], vertShader );
    glAttachShader( program[MOTION_BLUR_FLAT_SHADING], LoadShader( GL_GEOMETRY_SHADER, "rozmycie_ruchu_flat_gs.glsl" ) );
    glAttachShader( program[MOTION_BLUR_FLAT_SHADING], LoadShader( GL_FRAGMENT_SHADER, "rozmycie_ruchu_flat_fs.glsl" ) );
    LinkValidateProgram( program[MOTION_BLUR_FLAT_SHADING] );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[MOTION_BLUR_SMOOTH_SHADING] = glCreateProgram();
    glAttachShader( program[MOTION_BLUR_SMOOTH_SHADING], vertShader );
    glAttachShader( program[MOTION_BLUR_SMOOTH_SHADING], LoadShader( GL_GEOMETRY_SHADER, "rozmycie_ruchu_smooth_gs.glsl" ) );
    glAttachShader( program[MOTION_BLUR_SMOOTH_SHADING], LoadShader( GL_FRAGMENT_SHADER, "rozmycie_ruchu_smooth_fs.glsl" ) );
    LinkValidateProgram( program[MOTION_BLUR_SMOOTH_SHADING] );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[POSTPROCESSING_RECT] = glCreateProgram();
    glAttachShader( program[POSTPROCESSING_RECT], LoadShader( GL_VERTEX_SHADER, "prostakat_vs.glsl" ) );
    glAttachShader( program[POSTPROCESSING_RECT], LoadShader( GL_FRAGMENT_SHADER, "prostakat_fs.glsl" ) );
    LinkValidateProgram( program[POSTPROCESSING_RECT] );

    // generowanie identyfikatora obiektu tablic wierzchokw
    glGenVertexArrays( VERTEX_ARRAY_SIZE, vertexArray );

    // utworzenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[CUBE] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[POSITION] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( position ), position, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[COLOR] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[COLOR] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( color ), color, GL_STATIC_DRAW );
    glVertexAttribPointer( COLOR, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( COLOR );

    // utworzenie obiektu bufora indeksw wierzchokw i zaadowanie danych
    glGenBuffers( 1, &indicesBuffer );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, indicesBuffer );
    glBufferData( GL_ELEMENT_ARRAY_BUFFER, sizeof( indices ), indices, GL_STATIC_DRAW );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // utworzenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[RECTANGLE] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[POSITION_RECT] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION_RECT] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( positionRect ), positionRect, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 2, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[TEXCOORD_RECT] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[TEXCOORD_RECT] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( texCoordRect ), texCoordRect, GL_STATIC_DRAW );
    glVertexAttribPointer( TEX_COORD, 2, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( TEX_COORD );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // zaadowanie pocztkowych ustawie tablicy z macierzami modelu-widoku
    modelViewMatrix[0] = glm::mat4x4( 1.0 );
    modelViewMatrix[0] = glm::translate( modelViewMatrix[0], glm::vec3( 0.0f, 0.0f, -(near+far)/2.0f ) );
    modelViewMatrix[0] = glm::scale( modelViewMatrix[0], glm::vec3( scale, scale, scale ) );
    modelViewMatrix[0] = glm::translate( modelViewMatrix[0], glm::vec3( translateX, translateY, 0.0f ) );
    for( int layer = 1; layer < FRAME_LAYERS; layer++ )
    {
        modelViewMatrix[layer] = modelViewMatrix[0];
    }

    // wczenie testu bufora gbokoci
    glEnable( GL_DEPTH_TEST );

    // wczenie mechanizmw uywanych podczas renderingu tekstu
    InitDrawText();
}

//////////////////////////////////////////////////////////////////////
// usunicie obiektw OpenGL
//////////////////////////////////////////////////////////////////////
void DeleteScene()
{
    // porzdki
    glDeleteProgram( program[MOTION_BLUR_FLAT_SHADING] );
    glDeleteProgram( program[MOTION_BLUR_SMOOTH_SHADING] );
    glDeleteProgram( program[POSTPROCESSING_RECT] );
    glDeleteBuffers( VERTEX_BUFFER_SIZE, vertexBuffer );
    glDeleteBuffers( 1, &indicesBuffer );
    glDeleteVertexArrays( VERTEX_ARRAY_SIZE, vertexArray );
    glDeleteTextures( RENDER_TEXTURE_SIZE, renderTexture );
    glDeleteFramebuffers( 1, &frameBuffer );

    // usunicie mechanizmw uywanych podczas renderingu tekstu
    DeleteDrawText();
}
