//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>
#include "shaders.h"
#include "text.h"

//////////////////////////////////////////////////////////////////////
// rozmiary bryy obcinania
//////////////////////////////////////////////////////////////////////
GLfloat left = -2.0f;
GLfloat right = 2.0f;
GLfloat bottom = -2.0f;
GLfloat top = 2.0f;
GLfloat near = 3.0f;
GLfloat far = 7.0f;

//////////////////////////////////////////////////////////////////////
// macierz rzutowania
//////////////////////////////////////////////////////////////////////
glm::mat4x4 projectionMatrix;

//////////////////////////////////////////////////////////////////////
// wspczynniki skalowania obiektu
//////////////////////////////////////////////////////////////////////
GLfloat scale = 1.0f;

//////////////////////////////////////////////////////////////////////
// kty obrotu obiektu
//////////////////////////////////////////////////////////////////////
GLfloat rotateX = 0.0f;
GLfloat rotateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// przesunicie obiektu
//////////////////////////////////////////////////////////////////////
GLfloat translateX = 0.0f;
GLfloat translateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// numeracja obiektw programu
//////////////////////////////////////////////////////////////////////
enum
{
    FLAT_SHADING,       // cieniowanie paskie
    SMOOTH_SHADING,     // cieniowanie gadkie
    PROGRAM_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw programu
//////////////////////////////////////////////////////////////////////
GLuint program[PROGRAM_SIZE];

//////////////////////////////////////////////////////////////////////
// numeracja obiektw bufora wierzchokw oraz
// numery indeksw poszczeglnych atrybutw wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    POSITION,
    COLOR,
    VERTEX_BUFFER_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw bufora z danymi tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexBuffer[VERTEX_BUFFER_SIZE];

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu bufora z danymi tablicy
// indeksw wierzchokw obiektu
//////////////////////////////////////////////////////////////////////
GLuint indicesBuffer;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexArray;

//////////////////////////////////////////////////////////////////////
// wsprzdne wierzchokw trjktw skadajcych si na szecian
//////////////////////////////////////////////////////////////////////
GLfloat position[8*3] =
{
    1.0f, 1.0f, 1.0f,
    -1.0f, 1.0f, 1.0f,
    -1.0f, -1.0f, 1.0f,
    1.0f, -1.0f, 1.0f,
    1.0f, 1.0f, -1.0f,
    -1.0f, 1.0f, -1.0f,
    -1.0f, -1.0f, -1.0f,
    1.0f, -1.0f, -1.0f
};

//////////////////////////////////////////////////////////////////////
// skadowe kolorw wierzchokw trjktw skadajcych si na szecian
//////////////////////////////////////////////////////////////////////
const GLfloat color[8*3] =
{
    1.0f, 1.0f, 1.0f,
    1.0f, 1.0f, 0.0f,
    0.0f, 1.0f, 0.0f,
    0.0f, 1.0f, 1.0f,
    1.0f, 0.0f, 1.0f,
    1.0f, 0.0f, 0.0f,
    0.0f, 0.0f, 0.0f,
    0.0f, 0.0f, 1.0f
};

//////////////////////////////////////////////////////////////////////
// dane indeksw wierzchokw trjktw skadajcych si na szecian
//////////////////////////////////////////////////////////////////////
GLuint indices[12*3] =
{
    5, 1, 0,
    5, 0, 4,
    2, 3, 0,
    2, 0, 1,
    7, 4, 0,
    7, 0, 3,
    3, 2, 6,
    3, 6, 7,
    1, 6, 2,
    1, 5, 6,
    4, 6, 5,
    4, 7, 6
};

//////////////////////////////////////////////////////////////////////
// wybr wierzchoka podstawowego
//////////////////////////////////////////////////////////////////////
GLenum provokingVertex = GL_LAST_VERTEX_CONVENTION;

//////////////////////////////////////////////////////////////////////
// interpolacja kolorw wierzchokw wczona/wyczona
//////////////////////////////////////////////////////////////////////
GLboolean shading = SMOOTH_SHADING;

//////////////////////////////////////////////////////////////////////
// funkcja generujca scen 3D
//////////////////////////////////////////////////////////////////////
void DisplayScene()
{
    // macierz modelu-widoku = macierz jednostkowa
    glm::mat4x4 modelViewMatrix = glm::mat4x4( 1.0 );

    // przesunicie ukadu wsprzdnych obiektu do rodka bryy obcinania
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( 0.0f, 0.0f, -(near+far)/2.0f ) );

    // skalowanie obiektu
    modelViewMatrix = glm::scale( modelViewMatrix, glm::vec3( scale, scale, scale ) );

    // przesunicie obiektu
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( translateX, translateY, 0.0f ) );

    // obroty obiektu
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateX, glm::vec3( 1.0f, 0.0f, 0.0f ) );
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateY, glm::vec3( 0.0f, 1.0f, 0.0f ) );

    // czyszczenie bufora koloru i bufora gbokoci
    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

    // wybr wierzchoka podstawowego
    glProvokingVertex( provokingVertex );

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray );

    // wczenie programu
    glUseProgram( program[shading] );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    glm::mat4x4 modelViewProjectionMatrix = projectionMatrix * modelViewMatrix;
    glUniformMatrix4fv( glGetUniformLocation( program[shading], "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawElements( GL_TRIANGLES, 12 * 3, GL_UNSIGNED_INT, NULL );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wypisanie informacji o wierzchoku podstawowym
    DrawText8x16( 3, 3, provokingVertex == GL_FIRST_VERTEX_CONVENTION ?
                        "Wierzchoek podstawowy: GL_FIRST_VERTEX_CONVENTION" :
                        "Wierzchoek podstawowy: GL_LAST_VERTEX_CONVENTION" );

    // wypisanie informacji o rodzaju cieniowania
    DrawText8x16( 3, 21, shading == SMOOTH_SHADING ?
                         "Cieniowanie gadkie (smooth)" :
                         "Cieniowanie paskie (flat)" );
}

//////////////////////////////////////////////////////////////////////
// zmiana wielkoci okna
//////////////////////////////////////////////////////////////////////
void Reshape( int width, int height )
{
    // obszar renderingu - cae okno
    glViewport( 0, 0, width, height );

    // parametry bryy obcinania - rzutowanie perspektywiczne
    // wysoko okna wiksza od szerokoci okna
    if( width < height && width > 0 )
         projectionMatrix = glm::frustum( left, right, bottom*height/width, top*height/width, near, far );
    else
        // szeroko okna wiksza lub rwna wysokoci okna
        if (width >= height && height > 0)
            projectionMatrix = glm::frustum( left*width/height, right*width/height, bottom, top, near, far );
        else
            projectionMatrix = glm::frustum( left, right, bottom, top, near, far );
}

//////////////////////////////////////////////////////////////////////
// inicjalizacja staych elementw maszyny stanu OpenGL
//////////////////////////////////////////////////////////////////////
void InitScene()
{
    // kolor ta - zawarto bufora koloru
    glClearColor( 1.0f, 1.0f, 1.0f, 1.0f );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[FLAT_SHADING] = glCreateProgram();
    glAttachShader( program[FLAT_SHADING], LoadShader( GL_VERTEX_SHADER, "szescian_rgb_flat_vs.glsl" ) );
    glAttachShader( program[FLAT_SHADING], LoadShader( GL_FRAGMENT_SHADER, "szescian_rgb_flat_fs.glsl" ) );
    LinkValidateProgram( program[FLAT_SHADING] );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[SMOOTH_SHADING] = glCreateProgram();
    glAttachShader( program[SMOOTH_SHADING], LoadShader( GL_VERTEX_SHADER, "szescian_rgb_smooth_vs.glsl" ) );
    glAttachShader( program[SMOOTH_SHADING], LoadShader( GL_FRAGMENT_SHADER, "szescian_rgb_smooth_fs.glsl" ) );
    LinkValidateProgram( program[SMOOTH_SHADING] );

    // generowanie identyfikatora obiektu tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray );

    // utworzenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[POSITION] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( position ), position, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[COLOR] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[COLOR] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( color ), color, GL_STATIC_DRAW );
    glVertexAttribPointer( COLOR, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( COLOR );

    // utworzenie obiektu bufora indeksw wierzchokw i zaadowanie danych
    glGenBuffers( 1, &indicesBuffer );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, indicesBuffer );
    glBufferData( GL_ELEMENT_ARRAY_BUFFER, sizeof( indices ), indices, GL_STATIC_DRAW );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wczenie testu bufora gbokoci
    glEnable( GL_DEPTH_TEST );

    // wczenie mechanizmw uywanych podczas renderingu tekstu
    InitDrawText();
}

//////////////////////////////////////////////////////////////////////
// usunicie obiektw OpenGL
//////////////////////////////////////////////////////////////////////
void DeleteScene()
{
    // porzdki
    glDeleteProgram( program[FLAT_SHADING] );
    glDeleteProgram( program[SMOOTH_SHADING] );
    glDeleteBuffers( VERTEX_BUFFER_SIZE, vertexBuffer );
    glDeleteBuffers( 1, &indicesBuffer );
    glDeleteVertexArrays( 1, &vertexArray );

    // usunicie mechanizmw uywanych podczas renderingu tekstu
    DeleteDrawText();
}
