#include <iostream>
#include <string>
#include <string_view>
#include <fstream>

#include "ZipFile.h"
#include "utils/stream_utils.h"

#ifdef USE_BOOST_FILESYSTEM
#  include <boost/filesystem/path.hpp>
#  include <boost/filesystem/operations.hpp>
namespace fs = boost::filesystem;
#else
#  include <filesystem>
#  ifdef FILESYSTEM_EXPERIMENTAL
namespace fs = std::experimental::filesystem;
#  else
namespace fs = std::filesystem;
#  endif
#endif

void compress(
    fs::path const & source,
    fs::path const & archive,
    std::string_view password,
    std::function<void(std::string_view)> reporter)
{
    if (fs::is_regular_file(source))
    {
        if (reporter) reporter("Pakowanie " + source.string());
        ZipFile::AddEncryptedFile(
            archive.string(),
            source.string(),
            source.filename().string(),
            password.data(),
            LzmaMethod::Create());
    }
    else
    {
        for (auto const & p : fs::recursive_directory_iterator(source))
        {
            if (reporter) reporter("Pakowanie " + p.path().string());

            if (fs::is_directory(p))
            {
                auto zipArchive = ZipFile::Open(archive.string());
                auto entry = zipArchive->CreateEntry(p.path().string());
                entry->SetAttributes(ZipArchiveEntry::Attributes::Directory);
                ZipFile::SaveAndClose(zipArchive, archive.string());
            }
            else if (fs::is_regular_file(p))
            {
                ZipFile::AddEncryptedFile(
                    archive.string(),
                    p.path().string(),
                    p.path().filename().string(),
                    password.data(),
                    LzmaMethod::Create());
            }
        }
    }
}

void ensure_directory_exists(fs::path const & dir)
{
    if (!fs::exists(dir))
    {
#ifdef USE_BOOST_FILESYSTEM
        boost::system::error_code err;
#else
        std::error_code err;
#endif
        fs::create_directories(dir, err);
    }
}

void decompress(
    fs::path const & destination,
    fs::path const & archive,
    std::string_view password,
    std::function<void(std::string_view)> reporter)
{
    ensure_directory_exists(destination);

    auto zipArchive = ZipFile::Open(archive.string());

    for (size_t i = 0; i < zipArchive->GetEntriesCount(); ++i)
    {
        auto entry = zipArchive->GetEntry(i);
        if (entry)
        {
            auto filepath = destination / fs::path{ entry->GetFullName() }.relative_path();
            if (reporter) reporter("Tworzenie " + filepath.string());

            if (entry->IsPasswordProtected())
                entry->SetPassword(password.data());

            if (entry->IsDirectory())
            {
                ensure_directory_exists(filepath);
            }
            else
            {
                ensure_directory_exists(filepath.parent_path());

                std::ofstream destFile;
                destFile.open(
                    filepath.string().c_str(),
                    std::ios::binary | std::ios::trunc);

                if (!destFile.is_open())
                {
                    if (reporter)
                        reporter("Nie mona utworzy pliku docelowego!");
                }

                auto dataStream = entry->GetDecompressionStream();
                if (dataStream)
                {
                    utils::stream::copy(*dataStream, destFile);
                }
            }
        }
    }
}

int main()
{
    setlocale(LC_ALL, "polish");

    char option = 0;
    std::cout << "Wybierz [p]akowanie/[w]ypakowywanie?";
    std::cin >> option;

    if (option == 'p')
    {
        std::string archivepath;
        std::string inputpath;
        std::string password;
        std::cout << "Podaj nazw pliku lub katalogu do spakowania:";
        std::cin >> inputpath;
        std::cout << "Podaj ciek do archiwum:";
        std::cin >> archivepath;
        std::cout << "Podaj haso:";
        std::cin >> password;

        compress(inputpath, archivepath, password, [](std::string_view message) {std::cout << message << std::endl; });
    }
    else if (option == 'w')
    {
        std::string archivepath;
        std::string outputpath;
        std::string password;
        std::cout << "Podaj katalog do wypakowania:";
        std::cin >> outputpath;
        std::cout << "Podaj ciek do archiwum:";
        std::cin >> archivepath;
        std::cout << "Podaj haso:";
        std::cin >> password;

        decompress(outputpath, archivepath, password, [](std::string_view message) {std::cout << message << std::endl; });
    }
    else
    {
        std::cout << "bdna opcja" << std::endl;
    }
}
