(function(root) {

	/** Use a single `load` function */
	var load = typeof require == 'function' ? require : root.load,

	/** The unit testing framework */
	QUnit =
		root.QUnit ||
		(root.QUnit = load('../vendor/qunit/qunit/qunit.js') || root.QUnit) &&
		(load('../vendor/qunit-clib/qunit-clib.js'), root.QUnit),

	/** The `punycode` object to test */
	punycode = root.punycode || load('../punycode.js') || root.punycode,

	/** Data that will be used in the tests */
	testData = {
		'strings': [
			{
				'description': 'a single basic code point',
				'decoded': 'Bach',
				'encoded': 'Bach-'
			},
			{
				'description': 'a single non-ASCII character',
				'decoded': '\xfc', // 'ü'
				'encoded': 'tda'
			},
			{
				'description': 'multiple non-ASCII characters',
				'decoded': '\xfc\xeb\xe4\xf6\u2665', // üëäö♥
				'encoded': '4can8av2009b'
			},
			{
				'description': 'mix of ASCII and non-ASCII characters',
				'decoded': 'b\xfccher', // 'bücher'
				'encoded': 'bcher-kva'
			},
			{
				'description': 'long string with both ASCII and non-ASCII characters',
				'decoded': 'Willst du die Bl\xfcthe des fr\xfchen, die Fr\xfcchte des sp\xe4teren Jahres',
				'encoded': 'Willst du die Blthe des frhen, die Frchte des spteren Jahres-x9e96lkal'
			},
			// http://tools.ietf.org/html/rfc3492#section-7.1
			{
				'description': 'Arabic (Egyptian)',
				'decoded': '\u0644\u064a\u0647\u0645\u0627\u0628\u062a\u0643\u0644\u0645\u0648\u0634\u0639\u0631\u0628\u064a\u061f',
				'encoded': 'egbpdaj6bu4bxfgehfvwxn'
			},
			{
				'description': 'Chinese (simplified)',
				'decoded': '\u4ed6\u4eec\u4e3a\u4ec0\u4e48\u4e0d\u8bf4\u4e2d\u6587',
				'encoded': 'ihqwcrb4cv8a8dqg056pqjye'
			},
			{
				'description': 'Chinese (traditional)',
				'decoded': '\u4ed6\u5011\u7232\u4ec0\u9ebd\u4e0d\u8aaa\u4e2d\u6587',
				'encoded': 'ihqwctvzc91f659drss3x8bo0yb'
			},
			{
				'description': 'Czech',
				'decoded': 'Pro\u010dprost\u011bnemluv\xed\u010desky',
				'encoded': 'Proprostnemluvesky-uyb24dma41a'
			},
			{
				'description': 'Hebrew',
				'decoded': '\u05dc\u05de\u05d4\u05d4\u05dd\u05e4\u05e9\u05d5\u05d8\u05dc\u05d0\u05de\u05d3\u05d1\u05e8\u05d9\u05dd\u05e2\u05d1\u05e8\u05d9\u05ea',
				'encoded': '4dbcagdahymbxekheh6e0a7fei0b'
			},
			{
				'description': 'Hindi (Devanagari)',
				'decoded': '\u092f\u0939\u0932\u094b\u0917\u0939\u093f\u0928\u094d\u0926\u0940\u0915\u094d\u092f\u094b\u0902\u0928\u0939\u0940\u0902\u092c\u094b\u0932\u0938\u0915\u0924\u0947\u0939\u0948\u0902',
				'encoded': 'i1baa7eci9glrd9b2ae1bj0hfcgg6iyaf8o0a1dig0cd'
			},
			{
				'description': 'Japanese (kanji and hiragana)',
				'decoded': '\u306a\u305c\u307f\u3093\u306a\u65e5\u672c\u8a9e\u3092\u8a71\u3057\u3066\u304f\u308c\u306a\u3044\u306e\u304b',
				'encoded': 'n8jok5ay5dzabd5bym9f0cm5685rrjetr6pdxa'
			},
			{
				'description': 'Korean (Hangul syllables)',
				'decoded': '\uc138\uacc4\uc758\ubaa8\ub4e0\uc0ac\ub78c\ub4e4\uc774\ud55c\uad6d\uc5b4\ub97c\uc774\ud574\ud55c\ub2e4\uba74\uc5bc\ub9c8\ub098\uc88b\uc744\uae4c',
				'encoded': '989aomsvi5e83db1d2a355cv1e0vak1dwrv93d5xbh15a0dt30a5jpsd879ccm6fea98c'
			},
			/**
			 * As there’s no way to do it in JavaScript, Punycode.js doesn’t support
			 * mixed-case annotation (which is entirely optional as per the RFC).
			 * So, while the RFC sample string encodes to:
			 * `b1abfaaepdrnnbgefbaDotcwatmq2g4l`
			 * Without mixed-case annotation it has to encode to:
			 * `b1abfaaepdrnnbgefbadotcwatmq2g4l`
			 * https://github.com/bestiejs/punycode.js/issues/3
			 */
			{
				'description': 'Russian (Cyrillic)',
				'decoded': '\u043f\u043e\u0447\u0435\u043c\u0443\u0436\u0435\u043e\u043d\u0438\u043d\u0435\u0433\u043e\u0432\u043e\u0440\u044f\u0442\u043f\u043e\u0440\u0443\u0441\u0441\u043a\u0438',
				'encoded': 'b1abfaaepdrnnbgefbadotcwatmq2g4l'
			},
			{
				'description': 'Spanish',
				'decoded': 'Porqu\xe9nopuedensimplementehablarenEspa\xf1ol',
				'encoded': 'PorqunopuedensimplementehablarenEspaol-fmd56a'
			},
			{
				'description': 'Vietnamese',
				'decoded': 'T\u1ea1isaoh\u1ecdkh\xf4ngth\u1ec3ch\u1ec9n\xf3iti\u1ebfngVi\u1ec7t',
				'encoded': 'TisaohkhngthchnitingVit-kjcr8268qyxafd2f1b9g'
			},
			{
				'decoded': '3\u5e74B\u7d44\u91d1\u516b\u5148\u751f',
				'encoded': '3B-ww4c5e180e575a65lsy2b'
			},
			{
				'decoded': '\u5b89\u5ba4\u5948\u7f8e\u6075-with-SUPER-MONKEYS',
				'encoded': '-with-SUPER-MONKEYS-pc58ag80a8qai00g7n9n'
			},
			{
				'decoded': 'Hello-Another-Way-\u305d\u308c\u305e\u308c\u306e\u5834\u6240',
				'encoded': 'Hello-Another-Way--fc4qua05auwb3674vfr0b'
			},
			{
				'decoded': '\u3072\u3068\u3064\u5c4b\u6839\u306e\u4e0b2',
				'encoded': '2-u9tlzr9756bt3uc0v'
			},
			{
				'decoded': 'Maji\u3067Koi\u3059\u308b5\u79d2\u524d',
				'encoded': 'MajiKoi5-783gue6qz075azm5e'
			},
			{
				'decoded': '\u30d1\u30d5\u30a3\u30fcde\u30eb\u30f3\u30d0',
				'encoded': 'de-jg4avhby1noc0d'
			},
			{
				'decoded': '\u305d\u306e\u30b9\u30d4\u30fc\u30c9\u3067',
				'encoded': 'd9juau41awczczp'
			},
			/**
			 * This example is an ASCII string that breaks the existing rules for host
			 * name labels. (It’s not a realistic example for IDNA, because IDNA never
			 * encodes pure ASCII labels.)
			 */
			{
				'description': 'ASCII string that breaks the existing rules for host-name labels',
				'decoded': '-> $1.00 <-',
				'encoded': '-> $1.00 <--'
			}
		],
		'ucs2': [
			{
				'description': 'printable ASCII characters',
				'decoded': [32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99, 100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126],
				'encoded': ' !"#$%&\'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~'
			},
			{
				'description': 'Apple logo (private range)',
				'decoded': [63743],
				'encoded': '\uf8ff' // ''
			},
			{
				'decoded': [102, 3232, 246, 32, 98, 97, 114, 32, 98, 229, 122, 32, 113, 252, 120],
				'encoded': 'f\u0ca0\xf6 bar b\xe5\x7a q\xfcx' // 'fಠö bar båz qüx'
			},
			{
				'decoded': [127829, 119808, 119558, 119638],
				'encoded': '\ud83c\udf55\ud835\udc00\ud834\udf06\ud834\udf56' // ''
			},
			{
				'description': 'Unicode character category [Cc]: other, control',
				'decoded': [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 127, 128, 129, 130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142, 143, 144, 145, 146, 147, 148, 149, 150, 151, 152, 153, 154, 155, 156, 157, 158, 159],
				'encoded': '\x00\x01\x02\x03\x04\x05\x06\x07\b\t\n\x0b\f\r\x0e\x0f\x10\x11\x12\x13\x14\x15\x16\x17\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f\x7f\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f'
			},
			{
				'description': 'Unicode character category [Cf]: other, format',
				'decoded': [173, 1536, 1537, 1538, 1539, 1757, 1807, 6068, 6069, 8203, 32, 8205, 8206, 8207, 8234, 8235, 8236, 8237, 8238, 8288, 8289, 8290, 8291, 8292, 8298, 8299, 8300, 8301, 8302, 8303, 65279, 65529, 65530, 65531, 69821, 119155, 119156, 119157, 119158, 119159, 119160, 119161, 119162, 917505, 917536, 917537, 917538, 917539, 917540, 917541, 917542, 917543, 917544, 917545, 917546, 917547, 917548, 917549, 917550, 917551, 917552, 917553, 917554, 917555, 917556, 917557, 917558, 917559, 917560, 917561, 917562, 917563, 917564, 917565, 917566, 917567, 917568, 917569, 917570, 917571, 917572, 917573, 917574, 917575, 917576, 917577, 917578, 917579, 917580, 917581, 917582, 917583, 917584, 917585, 917586, 917587, 917588, 917589, 917590, 917591, 917592, 917593, 917594, 917595, 917596, 917597, 917598, 917599, 917600, 917601, 917602, 917603, 917604, 917605, 917606, 917607, 917608, 917609, 917610, 917611, 917612, 917613, 917614, 917615, 917616, 917617, 917618, 917619, 917620, 917621, 917622, 917623, 917624, 917625, 917626, 917627, 917628, 917629, 917630, 917631],
				'encoded': '\xad\u0600\u0601\u0602\u0603\u06dd\u070f\u17b4\u17b5\u200b \u200d\u200e\u200f\u202a\u202b\u202c\u202d\u202e\u2060\u2061\u2062\u2063\u2064\u206a\u206b\u206c\u206d\u206e\u206f\ufeff\ufff9\ufffa\ufffb\ud804\udcbd\ud834\udd73\ud834\udd74\ud834\udd75\ud834\udd76\ud834\udd77\ud834\udd78\ud834\udd79\ud834\udd7a\udb40\udc01\udb40\udc20\udb40\udc21\udb40\udc22\udb40\udc23\udb40\udc24\udb40\udc25\udb40\udc26\udb40\udc27\udb40\udc28\udb40\udc29\udb40\udc2a\udb40\udc2b\udb40\udc2c\udb40\udc2d\udb40\udc2e\udb40\udc2f\udb40\udc30\udb40\udc31\udb40\udc32\udb40\udc33\udb40\udc34\udb40\udc35\udb40\udc36\udb40\udc37\udb40\udc38\udb40\udc39\udb40\udc3a\udb40\udc3b\udb40\udc3c\udb40\udc3d\udb40\udc3e\udb40\udc3f\udb40\udc40\udb40\udc41\udb40\udc42\udb40\udc43\udb40\udc44\udb40\udc45\udb40\udc46\udb40\udc47\udb40\udc48\udb40\udc49\udb40\udc4a\udb40\udc4b\udb40\udc4c\udb40\udc4d\udb40\udc4e\udb40\udc4f\udb40\udc50\udb40\udc51\udb40\udc52\udb40\udc53\udb40\udc54\udb40\udc55\udb40\udc56\udb40\udc57\udb40\udc58\udb40\udc59\udb40\udc5a\udb40\udc5b\udb40\udc5c\udb40\udc5d\udb40\udc5e\udb40\udc5f\udb40\udc60\udb40\udc61\udb40\udc62\udb40\udc63\udb40\udc64\udb40\udc65\udb40\udc66\udb40\udc67\udb40\udc68\udb40\udc69\udb40\udc6a\udb40\udc6b\udb40\udc6c\udb40\udc6d\udb40\udc6e\udb40\udc6f\udb40\udc70\udb40\udc71\udb40\udc72\udb40\udc73\udb40\udc74\udb40\udc75\udb40\udc76\udb40\udc77\udb40\udc78\udb40\udc79\udb40\udc7a\udb40\udc7b\udb40\udc7c\udb40\udc7d\udb40\udc7e\udb40\udc7f'
			},
			{
				// This category contains no characters
				'description': 'Unicode character category [Cn]: other, not assigned',
				'decoded': [],
				'encoded': ''
			},
			{
				'description': 'Unicode character category [Co]: other, private use',
				'decoded': [57344, 63744, 983040, 1048574, 1048576, 1114110],
				'encoded': '\ue000\uf900\udb80\udc00\udbbf\udffe\udbc0\udc00\udbff\udffe' // '豈'
			},
			{
				// This category contains no characters
				'description': 'Unicode character category [LC]: letter, cased',
				'decoded': [],
				'encoded': ''
			},
			{
				'description': 'Unicode character category [Ll]: letter, lowercase',
				'decoded': [97, 98, 99, 100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 170, 181, 186, 223, 224, 225, 226, 227, 228, 229, 230, 231, 232, 233, 234, 235, 236, 237, 238, 239, 240, 241, 242, 243, 244, 245, 246, 248, 249, 250, 251, 252, 253, 254, 255, 257, 259, 261, 263, 265, 267, 269, 271, 273, 275, 277, 279, 281, 283, 285, 287, 289, 291, 293, 295, 297, 299, 301, 303, 305, 307, 309, 311, 312, 314, 316, 318, 320, 322, 324, 326, 328, 329, 331, 333, 335, 337, 339, 341, 343, 345, 347, 349, 351, 353, 355, 357, 359, 361, 363, 365, 367, 369, 371, 373, 375, 378, 380, 382, 383, 384, 387, 389, 392, 396, 397, 402, 405, 409, 410, 411, 414, 417, 419, 421, 424, 426, 427, 429, 432, 436, 438, 441, 442, 445, 446, 447, 454, 457, 460, 462, 464, 466, 468, 470, 472, 474, 476, 477, 479, 481, 483, 485, 487, 489, 491, 493, 495, 496, 499, 501, 505, 507, 509, 511, 513, 515, 517, 519, 521, 523, 525, 527, 529, 531, 533, 535, 537, 539, 541, 543, 545, 547, 549, 551, 553, 555, 557, 559, 561, 563, 564, 565, 566, 567, 568, 569, 572, 575, 576, 578, 583, 585, 587, 589, 591, 592, 593, 594, 595, 596, 597, 598, 599, 600, 601, 602, 603, 604, 605, 606, 607, 608, 609, 610, 611, 612, 613, 614, 615, 616, 617, 618, 619, 620, 621, 622, 623, 624, 625, 626, 627, 628, 629, 630, 631, 632, 633, 634, 635, 636, 637, 638, 639, 640, 641, 642, 643, 644, 645, 646, 647, 648, 649, 650, 651, 652, 653, 654, 655, 656, 657, 658, 659, 661, 662, 663, 664, 665, 666, 667, 668, 669, 670, 671, 672, 673, 674, 675, 676, 677, 678, 679, 680, 681, 682, 683, 684, 685, 686, 687, 881, 883, 887, 891, 892, 893, 912, 940, 941, 942, 943, 944, 945, 946, 947, 948, 949, 950, 951, 952, 953, 954, 955, 956, 957, 958, 959, 960, 961, 962, 963, 964, 965, 966, 967, 968, 969, 970, 971, 972, 973, 974, 976, 977, 981, 982, 983, 985, 987, 989, 991, 993, 995, 997, 999, 1001, 1003, 1005, 1007, 1008, 1009, 1010, 1011, 1013, 1016, 1019, 1020, 1072, 1073, 1074, 1075, 1076, 1077, 1078, 1079, 1080, 1081, 1082, 1083, 1084, 1085, 1086, 1087, 1088, 1089, 1090, 1091, 1092, 1093, 1094, 1095, 1096, 1097, 1098, 1099, 1100, 1101, 1102, 1103, 1104, 1105, 1106, 1107, 1108, 1109, 1110, 1111, 1112, 1113, 1114, 1115, 1116, 1117, 1118, 1119, 1121, 1123, 1125, 1127, 1129, 1131, 1133, 1135, 1137, 1139, 1141, 1143, 1145, 1147, 1149, 1151, 1153, 1163, 1165, 1167, 1169, 1171, 1173, 1175, 1177, 1179, 1181, 1183, 1185, 1187, 1189, 1191, 1193, 1195, 1197, 1199, 1201, 1203, 1205, 1207, 1209, 1211, 1213, 1215, 1218, 1220, 1222, 1224, 1226, 1228, 1230, 1231, 1233, 1235, 1237, 1239, 1241, 1243, 1245, 1247, 1249, 1251, 1253, 1255, 1257, 1259, 1261, 1263, 1265, 1267, 1269, 1271, 1273, 1275, 1277, 1279, 1281, 1283, 1285, 1287, 1289, 1291, 1293, 1295, 1297, 1299, 1301, 1303, 1305, 1307, 1309, 1311, 1313, 1315, 1317, 1319, 1377, 1378, 1379, 1380, 1381, 1382, 1383, 1384, 1385, 1386, 1387, 1388, 1389, 1390, 1391, 1392, 1393, 1394, 1395, 1396, 1397, 1398, 1399, 1400, 1401, 1402, 1403, 1404, 1405, 1406, 1407, 1408, 1409, 1410, 1411, 1412, 1413, 1414, 1415, 7424, 7425, 7426, 7427, 7428, 7429, 7430, 7431, 7432, 7433, 7434, 7435, 7436, 7437, 7438, 7439, 7440, 7441, 7442, 7443, 7444, 7445, 7446, 7447, 7448, 7449, 7450, 7451, 7452, 7453, 7454, 7455, 7456, 7457, 7458, 7459, 7460, 7461, 7462, 7463, 7464, 7465, 7466, 7467, 7522, 7523, 7524, 7525, 7526, 7527, 7528, 7529, 7530, 7531, 7532, 7533, 7534, 7535, 7536, 7537, 7538, 7539, 7540, 7541, 7542, 7543, 7545, 7546, 7547, 7548, 7549, 7550, 7551, 7552, 7553, 7554, 7555, 7556, 7557, 7558, 7559, 7560, 7561, 7562, 7563, 7564, 7565, 7566, 7567, 7568, 7569, 7570, 7571, 7572, 7573, 7574, 7575, 7576, 7577, 7578, 7681, 7683, 7685, 7687, 7689, 7691, 7693, 7695, 7697, 7699, 7701, 7703, 7705, 7707, 7709, 7711, 7713, 7715, 7717, 7719, 7721, 7723, 7725, 7727, 7729, 7731, 7733, 7735, 7737, 7739, 7741, 7743, 7745, 7747, 7749, 7751, 7753, 7755, 7757, 7759, 7761, 7763, 7765, 7767, 7769, 7771, 7773, 7775, 7777, 7779, 7781, 7783, 7785, 7787, 7789, 7791, 7793, 7795, 7797, 7799, 7801, 7803, 7805, 7807, 7809, 7811, 7813, 7815, 7817, 7819, 7821, 7823, 7825, 7827, 7829, 7830, 7831, 7832, 7833, 7834, 7835, 7836, 7837, 7839, 7841, 7843, 7845, 7847, 7849, 7851, 7853, 7855, 7857, 7859, 7861, 7863, 7865, 7867, 7869, 7871, 7873, 7875, 7877, 7879, 7881, 7883, 7885, 7887, 7889, 7891, 7893, 7895, 7897, 7899, 7901, 7903, 7905, 7907, 7909, 7911, 7913, 7915, 7917, 7919, 7921, 7923, 7925, 7927, 7929, 7931, 7933, 7935, 7936, 7937, 7938, 7939, 7940, 7941, 7942, 7943, 7952, 7953, 7954, 7955, 7956, 7957, 7968, 7969, 7970, 7971, 7972, 7973, 7974, 7975, 7984, 7985, 7986, 7987, 7988, 7989, 7990, 7991, 8000, 8001, 8002, 8003, 8004, 8005, 8016, 8017, 8018, 8019, 8020, 8021, 8022, 8023, 8032, 8033, 8034, 8035, 8036, 8037, 8038, 8039, 8048, 940, 8050, 941, 8052, 942, 8054, 943, 8056, 972, 8058, 973, 8060, 974, 8064, 8065, 8066, 8067, 8068, 8069, 8070, 8071, 8080, 8081, 8082, 8083, 8084, 8085, 8086, 8087, 8096, 8097, 8098, 8099, 8100, 8101, 8102, 8103, 8112, 8113, 8114, 8115, 8116, 8118, 8119, 953, 8130, 8131, 8132, 8134, 8135, 8144, 8145, 8146, 912, 8150, 8151, 8160, 8161, 8162, 944, 8164, 8165, 8166, 8167, 8178, 8179, 8180, 8182, 8183, 8458, 8462, 8463, 8467, 8495, 8500, 8505, 8508, 8509, 8518, 8519, 8520, 8521, 8526, 8580, 11312, 11313, 11314, 11315, 11316, 11317, 11318, 11319, 11320, 11321, 11322, 11323, 11324, 11325, 11326, 11327, 11328, 11329, 11330, 11331, 11332, 11333, 11334, 11335, 11336, 11337, 11338, 11339, 11340, 11341, 11342, 11343, 11344, 11345, 11346, 11347, 11348, 11349, 11350, 11351, 11352, 11353, 11354, 11355, 11356, 11357, 11358, 11361, 11365, 11366, 11368, 11370, 11372, 11377, 11379, 11380, 11382, 11383, 11384, 11385, 11386, 11387, 11388, 11393, 11395, 11397, 11399, 11401, 11403, 11405, 11407, 11409, 11411, 11413, 11415, 11417, 11419, 11421, 11423, 11425, 11427, 11429, 11431, 11433, 11435, 11437, 11439, 11441, 11443, 11445, 11447, 11449, 11451, 11453, 11455, 11457, 11459, 11461, 11463, 11465, 11467, 11469, 11471, 11473, 11475, 11477, 11479, 11481, 11483, 11485, 11487, 11489, 11491, 11492, 11500, 11502, 11520, 11521, 11522, 11523, 11524, 11525, 11526, 11527, 11528, 11529, 11530, 11531, 11532, 11533, 11534, 11535, 11536, 11537, 11538, 11539, 11540, 11541, 11542, 11543, 11544, 11545, 11546, 11547, 11548, 11549, 11550, 11551, 11552, 11553, 11554, 11555, 11556, 11557, 42561, 42563, 42565, 42567, 42569, 42571, 42573, 42575, 42577, 42579, 42581, 42583, 42585, 42587, 42589, 42591, 42593, 42595, 42597, 42599, 42601, 42603, 42605, 42625, 42627, 42629, 42631, 42633, 42635, 42637, 42639, 42641, 42643, 42645, 42647, 42787, 42789, 42791, 42793, 42795, 42797, 42799, 42800, 42801, 42803, 42805, 42807, 42809, 42811, 42813, 42815, 42817, 42819, 42821, 42823, 42825, 42827, 42829, 42831, 42833, 42835, 42837, 42839, 42841, 42843, 42845, 42847, 42849, 42851, 42853, 42855, 42857, 42859, 42861, 42863, 42865, 42866, 42867, 42868, 42869, 42870, 42871, 42872, 42874, 42876, 42879, 42881, 42883, 42885, 42887, 42892, 42894, 42897, 42913, 42915, 42917, 42919, 42921, 43002, 64256, 64257, 64258, 64259, 64260, 64261, 64262, 64275, 64276, 64277, 64278, 64279, 65345, 65346, 65347, 65348, 65349, 65350, 65351, 65352, 65353, 65354, 65355, 65356, 65357, 65358, 65359, 65360, 65361, 65362, 65363, 65364, 65365, 65366, 65367, 65368, 65369, 65370, 66600, 66601, 66602, 66603, 66604, 66605, 66606, 66607, 66608, 66609, 66610, 66611, 66612, 66613, 66614, 66615, 66616, 66617, 66618, 66619, 66620, 66621, 66622, 66623, 66624, 66625, 66626, 66627, 66628, 66629, 66630, 66631, 66632, 66633, 66634, 66635, 66636, 66637, 66638, 66639, 119834, 119835, 119836, 119837, 119838, 119839, 119840, 119841, 119842, 119843, 119844, 119845, 119846, 119847, 119848, 119849, 119850, 119851, 119852, 119853, 119854, 119855, 119856, 119857, 119858, 119859, 119886, 119887, 119888, 119889, 119890, 119891, 119892, 119894, 119895, 119896, 119897, 119898, 119899, 119900, 119901, 119902, 119903, 119904, 119905, 119906, 119907, 119908, 119909, 119910, 119911, 119938, 119939, 119940, 119941, 119942, 119943, 119944, 119945, 119946, 119947, 119948, 119949, 119950, 119951, 119952, 119953, 119954, 119955, 119956, 119957, 119958, 119959, 119960, 119961, 119962, 119963, 119990, 119991, 119992, 119993, 119995, 119997, 119998, 119999, 120000, 120001, 120002, 120003, 120005, 120006, 120007, 120008, 120009, 120010, 120011, 120012, 120013, 120014, 120015, 120042, 120043, 120044, 120045, 120046, 120047, 120048, 120049, 120050, 120051, 120052, 120053, 120054, 120055, 120056, 120057, 120058, 120059, 120060, 120061, 120062, 120063, 120064, 120065, 120066, 120067, 120094, 120095, 120096, 120097, 120098, 120099, 120100, 120101, 120102, 120103, 120104, 120105, 120106, 120107, 120108, 120109, 120110, 120111, 120112, 120113, 120114, 120115, 120116, 120117, 120118, 120119, 120146, 120147, 120148, 120149, 120150, 120151, 120152, 120153, 120154, 120155, 120156, 120157, 120158, 120159, 120160, 120161, 120162, 120163, 120164, 120165, 120166, 120167, 120168, 120169, 120170, 120171, 120198, 120199, 120200, 120201, 120202, 120203, 120204, 120205, 120206, 120207, 120208, 120209, 120210, 120211, 120212, 120213, 120214, 120215, 120216, 120217, 120218, 120219, 120220, 120221, 120222, 120223, 120250, 120251, 120252, 120253, 120254, 120255, 120256, 120257, 120258, 120259, 120260, 120261, 120262, 120263, 120264, 120265, 120266, 120267, 120268, 120269, 120270, 120271, 120272, 120273, 120274, 120275, 120302, 120303, 120304, 120305, 120306, 120307, 120308, 120309, 120310, 120311, 120312, 120313, 120314, 120315, 120316, 120317, 120318, 120319, 120320, 120321, 120322, 120323, 120324, 120325, 120326, 120327, 120354, 120355, 120356, 120357, 120358, 120359, 120360, 120361, 120362, 120363, 120364, 120365, 120366, 120367, 120368, 120369, 120370, 120371, 120372, 120373, 120374, 120375, 120376, 120377, 120378, 120379, 120406, 120407, 120408, 120409, 120410, 120411, 120412, 120413, 120414, 120415, 120416, 120417, 120418, 120419, 120420, 120421, 120422, 120423, 120424, 120425, 120426, 120427, 120428, 120429, 120430, 120431, 120458, 120459, 120460, 120461, 120462, 120463, 120464, 120465, 120466, 120467, 120468, 120469, 120470, 120471, 120472, 120473, 120474, 120475, 120476, 120477, 120478, 120479, 120480, 120481, 120482, 120483, 120484, 120485, 120514, 120515, 120516, 120517, 120518, 120519, 120520, 120521, 120522, 120523, 120524, 120525, 120526, 120527, 120528, 120529, 120530, 120531, 120532, 120533, 120534, 120535, 120536, 120537, 120538, 120540, 120541, 120542, 120543, 120544, 120545, 120572, 120573, 120574, 120575, 120576, 120577, 120578, 120579, 120580, 120581, 120582, 120583, 120584, 120585, 120586, 120587, 120588, 120589, 120590, 120591, 120592, 120593, 120594, 120595, 120596, 120598, 120599, 120600, 120601, 120602, 120603, 120630, 120631, 120632, 120633, 120634, 120635, 120636, 120637, 120638, 120639, 120640, 120641, 120642, 120643, 120644, 120645, 120646, 120647, 120648, 120649, 120650, 120651, 120652, 120653, 120654, 120656, 120657, 120658, 120659, 120660, 120661, 120688, 120689, 120690, 120691, 120692, 120693, 120694, 120695, 120696, 120697, 120698, 120699, 120700, 120701, 120702, 120703, 120704, 120705, 120706, 120707, 120708, 120709, 120710, 120711, 120712, 120714, 120715, 120716, 120717, 120718, 120719, 120746, 120747, 120748, 120749, 120750, 120751, 120752, 120753, 120754, 120755, 120756, 120757, 120758, 120759, 120760, 120761, 120762, 120763, 120764, 120765, 120766, 120767, 120768, 120769, 120770, 120772, 120773, 120774, 120775, 120776, 120777, 120779],
				'encoded': 'abcdefghijklmnopqrstuvwxyz\xaa\xb5\xba\xdf\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7\xe8\xe9\xea\xeb\xec\xed\xee\xef\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff\u0101\u0103\u0105\u0107\u0109\u010b\u010d\u010f\u0111\u0113\u0115\u0117\u0119\u011b\u011d\u011f\u0121\u0123\u0125\u0127\u0129\u012b\u012d\u012f\u0131\u0133\u0135\u0137\u0138\u013a\u013c\u013e\u0140\u0142\u0144\u0146\u0148\u0149\u014b\u014d\u014f\u0151\u0153\u0155\u0157\u0159\u015b\u015d\u015f\u0161\u0163\u0165\u0167\u0169\u016b\u016d\u016f\u0171\u0173\u0175\u0177\u017a\u017c\u017e\u017f\u0180\u0183\u0185\u0188\u018c\u018d\u0192\u0195\u0199\u019a\u019b\u019e\u01a1\u01a3\u01a5\u01a8\u01aa\u01ab\u01ad\u01b0\u01b4\u01b6\u01b9\u01ba\u01bd\u01be\u01bf\u01c6\u01c9\u01cc\u01ce\u01d0\u01d2\u01d4\u01d6\u01d8\u01da\u01dc\u01dd\u01df\u01e1\u01e3\u01e5\u01e7\u01e9\u01eb\u01ed\u01ef\u01f0\u01f3\u01f5\u01f9\u01fb\u01fd\u01ff\u0201\u0203\u0205\u0207\u0209\u020b\u020d\u020f\u0211\u0213\u0215\u0217\u0219\u021b\u021d\u021f\u0221\u0223\u0225\u0227\u0229\u022b\u022d\u022f\u0231\u0233\u0234\u0235\u0236\u0237\u0238\u0239\u023c\u023f\u0240\u0242\u0247\u0249\u024b\u024d\u024f\u0250\u0251\u0252\u0253\u0254\u0255\u0256\u0257\u0258\u0259\u025a\u025b\u025c\u025d\u025e\u025f\u0260\u0261\u0262\u0263\u0264\u0265\u0266\u0267\u0268\u0269\u026a\u026b\u026c\u026d\u026e\u026f\u0270\u0271\u0272\u0273\u0274\u0275\u0276\u0277\u0278\u0279\u027a\u027b\u027c\u027d\u027e\u027f\u0280\u0281\u0282\u0283\u0284\u0285\u0286\u0287\u0288\u0289\u028a\u028b\u028c\u028d\u028e\u028f\u0290\u0291\u0292\u0293\u0295\u0296\u0297\u0298\u0299\u029a\u029b\u029c\u029d\u029e\u029f\u02a0\u02a1\u02a2\u02a3\u02a4\u02a5\u02a6\u02a7\u02a8\u02a9\u02aa\u02ab\u02ac\u02ad\u02ae\u02af\u0371\u0373\u0377\u037b\u037c\u037d\u0390\u03ac\u03ad\u03ae\u03af\u03b0\u03b1\u03b2\u03b3\u03b4\u03b5\u03b6\u03b7\u03b8\u03b9\u03ba\u03bb\u03bc\u03bd\u03be\u03bf\u03c0\u03c1\u03c2\u03c3\u03c4\u03c5\u03c6\u03c7\u03c8\u03c9\u03ca\u03cb\u03cc\u03cd\u03ce\u03d0\u03d1\u03d5\u03d6\u03d7\u03d9\u03db\u03dd\u03df\u03e1\u03e3\u03e5\u03e7\u03e9\u03eb\u03ed\u03ef\u03f0\u03f1\u03f2\u03f3\u03f5\u03f8\u03fb\u03fc\u0430\u0431\u0432\u0433\u0434\u0435\u0436\u0437\u0438\u0439\u043a\u043b\u043c\u043d\u043e\u043f\u0440\u0441\u0442\u0443\u0444\u0445\u0446\u0447\u0448\u0449\u044a\u044b\u044c\u044d\u044e\u044f\u0450\u0451\u0452\u0453\u0454\u0455\u0456\u0457\u0458\u0459\u045a\u045b\u045c\u045d\u045e\u045f\u0461\u0463\u0465\u0467\u0469\u046b\u046d\u046f\u0471\u0473\u0475\u0477\u0479\u047b\u047d\u047f\u0481\u048b\u048d\u048f\u0491\u0493\u0495\u0497\u0499\u049b\u049d\u049f\u04a1\u04a3\u04a5\u04a7\u04a9\u04ab\u04ad\u04af\u04b1\u04b3\u04b5\u04b7\u04b9\u04bb\u04bd\u04bf\u04c2\u04c4\u04c6\u04c8\u04ca\u04cc\u04ce\u04cf\u04d1\u04d3\u04d5\u04d7\u04d9\u04db\u04dd\u04df\u04e1\u04e3\u04e5\u04e7\u04e9\u04eb\u04ed\u04ef\u04f1\u04f3\u04f5\u04f7\u04f9\u04fb\u04fd\u04ff\u0501\u0503\u0505\u0507\u0509\u050b\u050d\u050f\u0511\u0513\u0515\u0517\u0519\u051b\u051d\u051f\u0521\u0523\u0525\u0527\u0561\u0562\u0563\u0564\u0565\u0566\u0567\u0568\u0569\u056a\u056b\u056c\u056d\u056e\u056f\u0570\u0571\u0572\u0573\u0574\u0575\u0576\u0577\u0578\u0579\u057a\u057b\u057c\u057d\u057e\u057f\u0580\u0581\u0582\u0583\u0584\u0585\u0586\u0587\u1d00\u1d01\u1d02\u1d03\u1d04\u1d05\u1d06\u1d07\u1d08\u1d09\u1d0a\u1d0b\u1d0c\u1d0d\u1d0e\u1d0f\u1d10\u1d11\u1d12\u1d13\u1d14\u1d15\u1d16\u1d17\u1d18\u1d19\u1d1a\u1d1b\u1d1c\u1d1d\u1d1e\u1d1f\u1d20\u1d21\u1d22\u1d23\u1d24\u1d25\u1d26\u1d27\u1d28\u1d29\u1d2a\u1d2b\u1d62\u1d63\u1d64\u1d65\u1d66\u1d67\u1d68\u1d69\u1d6a\u1d6b\u1d6c\u1d6d\u1d6e\u1d6f\u1d70\u1d71\u1d72\u1d73\u1d74\u1d75\u1d76\u1d77\u1d79\u1d7a\u1d7b\u1d7c\u1d7d\u1d7e\u1d7f\u1d80\u1d81\u1d82\u1d83\u1d84\u1d85\u1d86\u1d87\u1d88\u1d89\u1d8a\u1d8b\u1d8c\u1d8d\u1d8e\u1d8f\u1d90\u1d91\u1d92\u1d93\u1d94\u1d95\u1d96\u1d97\u1d98\u1d99\u1d9a\u1e01\u1e03\u1e05\u1e07\u1e09\u1e0b\u1e0d\u1e0f\u1e11\u1e13\u1e15\u1e17\u1e19\u1e1b\u1e1d\u1e1f\u1e21\u1e23\u1e25\u1e27\u1e29\u1e2b\u1e2d\u1e2f\u1e31\u1e33\u1e35\u1e37\u1e39\u1e3b\u1e3d\u1e3f\u1e41\u1e43\u1e45\u1e47\u1e49\u1e4b\u1e4d\u1e4f\u1e51\u1e53\u1e55\u1e57\u1e59\u1e5b\u1e5d\u1e5f\u1e61\u1e63\u1e65\u1e67\u1e69\u1e6b\u1e6d\u1e6f\u1e71\u1e73\u1e75\u1e77\u1e79\u1e7b\u1e7d\u1e7f\u1e81\u1e83\u1e85\u1e87\u1e89\u1e8b\u1e8d\u1e8f\u1e91\u1e93\u1e95\u1e96\u1e97\u1e98\u1e99\u1e9a\u1e9b\u1e9c\u1e9d\u1e9f\u1ea1\u1ea3\u1ea5\u1ea7\u1ea9\u1eab\u1ead\u1eaf\u1eb1\u1eb3\u1eb5\u1eb7\u1eb9\u1ebb\u1ebd\u1ebf\u1ec1\u1ec3\u1ec5\u1ec7\u1ec9\u1ecb\u1ecd\u1ecf\u1ed1\u1ed3\u1ed5\u1ed7\u1ed9\u1edb\u1edd\u1edf\u1ee1\u1ee3\u1ee5\u1ee7\u1ee9\u1eeb\u1eed\u1eef\u1ef1\u1ef3\u1ef5\u1ef7\u1ef9\u1efb\u1efd\u1eff\u1f00\u1f01\u1f02\u1f03\u1f04\u1f05\u1f06\u1f07\u1f10\u1f11\u1f12\u1f13\u1f14\u1f15\u1f20\u1f21\u1f22\u1f23\u1f24\u1f25\u1f26\u1f27\u1f30\u1f31\u1f32\u1f33\u1f34\u1f35\u1f36\u1f37\u1f40\u1f41\u1f42\u1f43\u1f44\u1f45\u1f50\u1f51\u1f52\u1f53\u1f54\u1f55\u1f56\u1f57\u1f60\u1f61\u1f62\u1f63\u1f64\u1f65\u1f66\u1f67\u1f70\u03ac\u1f72\u03ad\u1f74\u03ae\u1f76\u03af\u1f78\u03cc\u1f7a\u03cd\u1f7c\u03ce\u1f80\u1f81\u1f82\u1f83\u1f84\u1f85\u1f86\u1f87\u1f90\u1f91\u1f92\u1f93\u1f94\u1f95\u1f96\u1f97\u1fa0\u1fa1\u1fa2\u1fa3\u1fa4\u1fa5\u1fa6\u1fa7\u1fb0\u1fb1\u1fb2\u1fb3\u1fb4\u1fb6\u1fb7\u03b9\u1fc2\u1fc3\u1fc4\u1fc6\u1fc7\u1fd0\u1fd1\u1fd2\u0390\u1fd6\u1fd7\u1fe0\u1fe1\u1fe2\u03b0\u1fe4\u1fe5\u1fe6\u1fe7\u1ff2\u1ff3\u1ff4\u1ff6\u1ff7\u210a\u210e\u210f\u2113\u212f\u2134\u2139\u213c\u213d\u2146\u2147\u2148\u2149\u214e\u2184\u2c30\u2c31\u2c32\u2c33\u2c34\u2c35\u2c36\u2c37\u2c38\u2c39\u2c3a\u2c3b\u2c3c\u2c3d\u2c3e\u2c3f\u2c40\u2c41\u2c42\u2c43\u2c44\u2c45\u2c46\u2c47\u2c48\u2c49\u2c4a\u2c4b\u2c4c\u2c4d\u2c4e\u2c4f\u2c50\u2c51\u2c52\u2c53\u2c54\u2c55\u2c56\u2c57\u2c58\u2c59\u2c5a\u2c5b\u2c5c\u2c5d\u2c5e\u2c61\u2c65\u2c66\u2c68\u2c6a\u2c6c\u2c71\u2c73\u2c74\u2c76\u2c77\u2c78\u2c79\u2c7a\u2c7b\u2c7c\u2c81\u2c83\u2c85\u2c87\u2c89\u2c8b\u2c8d\u2c8f\u2c91\u2c93\u2c95\u2c97\u2c99\u2c9b\u2c9d\u2c9f\u2ca1\u2ca3\u2ca5\u2ca7\u2ca9\u2cab\u2cad\u2caf\u2cb1\u2cb3\u2cb5\u2cb7\u2cb9\u2cbb\u2cbd\u2cbf\u2cc1\u2cc3\u2cc5\u2cc7\u2cc9\u2ccb\u2ccd\u2ccf\u2cd1\u2cd3\u2cd5\u2cd7\u2cd9\u2cdb\u2cdd\u2cdf\u2ce1\u2ce3\u2ce4\u2cec\u2cee\u2d00\u2d01\u2d02\u2d03\u2d04\u2d05\u2d06\u2d07\u2d08\u2d09\u2d0a\u2d0b\u2d0c\u2d0d\u2d0e\u2d0f\u2d10\u2d11\u2d12\u2d13\u2d14\u2d15\u2d16\u2d17\u2d18\u2d19\u2d1a\u2d1b\u2d1c\u2d1d\u2d1e\u2d1f\u2d20\u2d21\u2d22\u2d23\u2d24\u2d25\ua641\ua643\ua645\ua647\ua649\ua64b\ua64d\ua64f\ua651\ua653\ua655\ua657\ua659\ua65b\ua65d\ua65f\ua661\ua663\ua665\ua667\ua669\ua66b\ua66d\ua681\ua683\ua685\ua687\ua689\ua68b\ua68d\ua68f\ua691\ua693\ua695\ua697\ua723\ua725\ua727\ua729\ua72b\ua72d\ua72f\ua730\ua731\ua733\ua735\ua737\ua739\ua73b\ua73d\ua73f\ua741\ua743\ua745\ua747\ua749\ua74b\ua74d\ua74f\ua751\ua753\ua755\ua757\ua759\ua75b\ua75d\ua75f\ua761\ua763\ua765\ua767\ua769\ua76b\ua76d\ua76f\ua771\ua772\ua773\ua774\ua775\ua776\ua777\ua778\ua77a\ua77c\ua77f\ua781\ua783\ua785\ua787\ua78c\ua78e\ua791\ua7a1\ua7a3\ua7a5\ua7a7\ua7a9\ua7fa\ufb00\ufb01\ufb02\ufb03\ufb04\ufb05\ufb06\ufb13\ufb14\ufb15\ufb16\ufb17\uff41\uff42\uff43\uff44\uff45\uff46\uff47\uff48\uff49\uff4a\uff4b\uff4c\uff4d\uff4e\uff4f\uff50\uff51\uff52\uff53\uff54\uff55\uff56\uff57\uff58\uff59\uff5a\ud801\udc28\ud801\udc29\ud801\udc2a\ud801\udc2b\ud801\udc2c\ud801\udc2d\ud801\udc2e\ud801\udc2f\ud801\udc30\ud801\udc31\ud801\udc32\ud801\udc33\ud801\udc34\ud801\udc35\ud801\udc36\ud801\udc37\ud801\udc38\ud801\udc39\ud801\udc3a\ud801\udc3b\ud801\udc3c\ud801\udc3d\ud801\udc3e\ud801\udc3f\ud801\udc40\ud801\udc41\ud801\udc42\ud801\udc43\ud801\udc44\ud801\udc45\ud801\udc46\ud801\udc47\ud801\udc48\ud801\udc49\ud801\udc4a\ud801\udc4b\ud801\udc4c\ud801\udc4d\ud801\udc4e\ud801\udc4f\ud835\udc1a\ud835\udc1b\ud835\udc1c\ud835\udc1d\ud835\udc1e\ud835\udc1f\ud835\udc20\ud835\udc21\ud835\udc22\ud835\udc23\ud835\udc24\ud835\udc25\ud835\udc26\ud835\udc27\ud835\udc28\ud835\udc29\ud835\udc2a\ud835\udc2b\ud835\udc2c\ud835\udc2d\ud835\udc2e\ud835\udc2f\ud835\udc30\ud835\udc31\ud835\udc32\ud835\udc33\ud835\udc4e\ud835\udc4f\ud835\udc50\ud835\udc51\ud835\udc52\ud835\udc53\ud835\udc54\ud835\udc56\ud835\udc57\ud835\udc58\ud835\udc59\ud835\udc5a\ud835\udc5b\ud835\udc5c\ud835\udc5d\ud835\udc5e\ud835\udc5f\ud835\udc60\ud835\udc61\ud835\udc62\ud835\udc63\ud835\udc64\ud835\udc65\ud835\udc66\ud835\udc67\ud835\udc82\ud835\udc83\ud835\udc84\ud835\udc85\ud835\udc86\ud835\udc87\ud835\udc88\ud835\udc89\ud835\udc8a\ud835\udc8b\ud835\udc8c\ud835\udc8d\ud835\udc8e\ud835\udc8f\ud835\udc90\ud835\udc91\ud835\udc92\ud835\udc93\ud835\udc94\ud835\udc95\ud835\udc96\ud835\udc97\ud835\udc98\ud835\udc99\ud835\udc9a\ud835\udc9b\ud835\udcb6\ud835\udcb7\ud835\udcb8\ud835\udcb9\ud835\udcbb\ud835\udcbd\ud835\udcbe\ud835\udcbf\ud835\udcc0\ud835\udcc1\ud835\udcc2\ud835\udcc3\ud835\udcc5\ud835\udcc6\ud835\udcc7\ud835\udcc8\ud835\udcc9\ud835\udcca\ud835\udccb\ud835\udccc\ud835\udccd\ud835\udcce\ud835\udccf\ud835\udcea\ud835\udceb\ud835\udcec\ud835\udced\ud835\udcee\ud835\udcef\ud835\udcf0\ud835\udcf1\ud835\udcf2\ud835\udcf3\ud835\udcf4\ud835\udcf5\ud835\udcf6\ud835\udcf7\ud835\udcf8\ud835\udcf9\ud835\udcfa\ud835\udcfb\ud835\udcfc\ud835\udcfd\ud835\udcfe\ud835\udcff\ud835\udd00\ud835\udd01\ud835\udd02\ud835\udd03\ud835\udd1e\ud835\udd1f\ud835\udd20\ud835\udd21\ud835\udd22\ud835\udd23\ud835\udd24\ud835\udd25\ud835\udd26\ud835\udd27\ud835\udd28\ud835\udd29\ud835\udd2a\ud835\udd2b\ud835\udd2c\ud835\udd2d\ud835\udd2e\ud835\udd2f\ud835\udd30\ud835\udd31\ud835\udd32\ud835\udd33\ud835\udd34\ud835\udd35\ud835\udd36\ud835\udd37\ud835\udd52\ud835\udd53\ud835\udd54\ud835\udd55\ud835\udd56\ud835\udd57\ud835\udd58\ud835\udd59\ud835\udd5a\ud835\udd5b\ud835\udd5c\ud835\udd5d\ud835\udd5e\ud835\udd5f\ud835\udd60\ud835\udd61\ud835\udd62\ud835\udd63\ud835\udd64\ud835\udd65\ud835\udd66\ud835\udd67\ud835\udd68\ud835\udd69\ud835\udd6a\ud835\udd6b\ud835\udd86\ud835\udd87\ud835\udd88\ud835\udd89\ud835\udd8a\ud835\udd8b\ud835\udd8c\ud835\udd8d\ud835\udd8e\ud835\udd8f\ud835\udd90\ud835\udd91\ud835\udd92\ud835\udd93\ud835\udd94\ud835\udd95\ud835\udd96\ud835\udd97\ud835\udd98\ud835\udd99\ud835\udd9a\ud835\udd9b\ud835\udd9c\ud835\udd9d\ud835\udd9e\ud835\udd9f\ud835\uddba\ud835\uddbb\ud835\uddbc\ud835\uddbd\ud835\uddbe\ud835\uddbf\ud835\uddc0\ud835\uddc1\ud835\uddc2\ud835\uddc3\ud835\uddc4\ud835\uddc5\ud835\uddc6\ud835\uddc7\ud835\uddc8\ud835\uddc9\ud835\uddca\ud835\uddcb\ud835\uddcc\ud835\uddcd\ud835\uddce\ud835\uddcf\ud835\uddd0\ud835\uddd1\ud835\uddd2\ud835\uddd3\ud835\uddee\ud835\uddef\ud835\uddf0\ud835\uddf1\ud835\uddf2\ud835\uddf3\ud835\uddf4\ud835\uddf5\ud835\uddf6\ud835\uddf7\ud835\uddf8\ud835\uddf9\ud835\uddfa\ud835\uddfb\ud835\uddfc\ud835\uddfd\ud835\uddfe\ud835\uddff\ud835\ude00\ud835\ude01\ud835\ude02\ud835\ude03\ud835\ude04\ud835\ude05\ud835\ude06\ud835\ude07\ud835\ude22\ud835\ude23\ud835\ude24\ud835\ude25\ud835\ude26\ud835\ude27\ud835\ude28\ud835\ude29\ud835\ude2a\ud835\ude2b\ud835\ude2c\ud835\ude2d\ud835\ude2e\ud835\ude2f\ud835\ude30\ud835\ude31\ud835\ude32\ud835\ude33\ud835\ude34\ud835\ude35\ud835\ude36\ud835\ude37\ud835\ude38\ud835\ude39\ud835\ude3a\ud835\ude3b\ud835\ude56\ud835\ude57\ud835\ude58\ud835\ude59\ud835\ude5a\ud835\ude5b\ud835\ude5c\ud835\ude5d\ud835\ude5e\ud835\ude5f\ud835\ude60\ud835\ude61\ud835\ude62\ud835\ude63\ud835\ude64\ud835\ude65\ud835\ude66\ud835\ude67\ud835\ude68\ud835\ude69\ud835\ude6a\ud835\ude6b\ud835\ude6c\ud835\ude6d\ud835\ude6e\ud835\ude6f\ud835\ude8a\ud835\ude8b\ud835\ude8c\ud835\ude8d\ud835\ude8e\ud835\ude8f\ud835\ude90\ud835\ude91\ud835\ude92\ud835\ude93\ud835\ude94\ud835\ude95\ud835\ude96\ud835\ude97\ud835\ude98\ud835\ude99\ud835\ude9a\ud835\ude9b\ud835\ude9c\ud835\ude9d\ud835\ude9e\ud835\ude9f\ud835\udea0\ud835\udea1\ud835\udea2\ud835\udea3\ud835\udea4\ud835\udea5\ud835\udec2\ud835\udec3\ud835\udec4\ud835\udec5\ud835\udec6\ud835\udec7\ud835\udec8\ud835\udec9\ud835\udeca\ud835\udecb\ud835\udecc\ud835\udecd\ud835\udece\ud835\udecf\ud835\uded0\ud835\uded1\ud835\uded2\ud835\uded3\ud835\uded4\ud835\uded5\ud835\uded6\ud835\uded7\ud835\uded8\ud835\uded9\ud835\udeda\ud835\udedc\ud835\udedd\ud835\udede\ud835\udedf\ud835\udee0\ud835\udee1\ud835\udefc\ud835\udefd\ud835\udefe\ud835\udeff\ud835\udf00\ud835\udf01\ud835\udf02\ud835\udf03\ud835\udf04\ud835\udf05\ud835\udf06\ud835\udf07\ud835\udf08\ud835\udf09\ud835\udf0a\ud835\udf0b\ud835\udf0c\ud835\udf0d\ud835\udf0e\ud835\udf0f\ud835\udf10\ud835\udf11\ud835\udf12\ud835\udf13\ud835\udf14\ud835\udf16\ud835\udf17\ud835\udf18\ud835\udf19\ud835\udf1a\ud835\udf1b\ud835\udf36\ud835\udf37\ud835\udf38\ud835\udf39\ud835\udf3a\ud835\udf3b\ud835\udf3c\ud835\udf3d\ud835\udf3e\ud835\udf3f\ud835\udf40\ud835\udf41\ud835\udf42\ud835\udf43\ud835\udf44\ud835\udf45\ud835\udf46\ud835\udf47\ud835\udf48\ud835\udf49\ud835\udf4a\ud835\udf4b\ud835\udf4c\ud835\udf4d\ud835\udf4e\ud835\udf50\ud835\udf51\ud835\udf52\ud835\udf53\ud835\udf54\ud835\udf55\ud835\udf70\ud835\udf71\ud835\udf72\ud835\udf73\ud835\udf74\ud835\udf75\ud835\udf76\ud835\udf77\ud835\udf78\ud835\udf79\ud835\udf7a\ud835\udf7b\ud835\udf7c\ud835\udf7d\ud835\udf7e\ud835\udf7f\ud835\udf80\ud835\udf81\ud835\udf82\ud835\udf83\ud835\udf84\ud835\udf85\ud835\udf86\ud835\udf87\ud835\udf88\ud835\udf8a\ud835\udf8b\ud835\udf8c\ud835\udf8d\ud835\udf8e\ud835\udf8f\ud835\udfaa\ud835\udfab\ud835\udfac\ud835\udfad\ud835\udfae\ud835\udfaf\ud835\udfb0\ud835\udfb1\ud835\udfb2\ud835\udfb3\ud835\udfb4\ud835\udfb5\ud835\udfb6\ud835\udfb7\ud835\udfb8\ud835\udfb9\ud835\udfba\ud835\udfbb\ud835\udfbc\ud835\udfbd\ud835\udfbe\ud835\udfbf\ud835\udfc0\ud835\udfc1\ud835\udfc2\ud835\udfc4\ud835\udfc5\ud835\udfc6\ud835\udfc7\ud835\udfc8\ud835\udfc9\ud835\udfcb'
			},
			{
				'description': 'Unicode character category [Lm]: letter, modifier',
				'decoded': [688, 689, 690, 691, 692, 693, 694, 695, 696, 697, 698, 699, 700, 701, 702, 703, 704, 705, 710, 711, 712, 713, 714, 715, 716, 717, 718, 719, 720, 721, 736, 737, 738, 739, 740, 748, 750, 884, 890, 1369, 1600, 1765, 1766, 2036, 2037, 2042, 2074, 2084, 2088, 2417, 3654, 3782, 4348, 6103, 6211, 6823, 7288, 7289, 7290, 7291, 7292, 7293, 7468, 7469, 7470, 7471, 7472, 7473, 7474, 7475, 7476, 7477, 7478, 7479, 7480, 7481, 7482, 7483, 7484, 7485, 7486, 7487, 7488, 7489, 7490, 7491, 7492, 7493, 7494, 7495, 7496, 7497, 7498, 7499, 7500, 7501, 7502, 7503, 7504, 7505, 7506, 7507, 7508, 7509, 7510, 7511, 7512, 7513, 7514, 7515, 7516, 7517, 7518, 7519, 7520, 7521, 7544, 7579, 7580, 7581, 7582, 7583, 7584, 7585, 7586, 7587, 7588, 7589, 7590, 7591, 7592, 7593, 7594, 7595, 7596, 7597, 7598, 7599, 7600, 7601, 7602, 7603, 7604, 7605, 7606, 7607, 7608, 7609, 7610, 7611, 7612, 7613, 7614, 7615, 8305, 8319, 8336, 8337, 8338, 8339, 8340, 8341, 8342, 8343, 8344, 8345, 8346, 8347, 8348, 11389, 11631, 11823, 12293, 12337, 12338, 12339, 12340, 12341, 12347, 12445, 12446, 12540, 12541, 12542, 40981, 42232, 42233, 42234, 42235, 42236, 42237, 42508, 42623, 42775, 42776, 42777, 42778, 42779, 42780, 42781, 42782, 42783, 42864, 42888, 43471, 43632, 43741, 65392, 65438, 65439],
				'encoded': '\u02b0\u02b1\u02b2\u02b3\u02b4\u02b5\u02b6\u02b7\u02b8\u02b9\u02ba\u02bb\u02bc\u02bd\u02be\u02bf\u02c0\u02c1\u02c6\u02c7\u02c8\u02c9\u02ca\u02cb\u02cc\u02cd\u02ce\u02cf\u02d0\u02d1\u02e0\u02e1\u02e2\u02e3\u02e4\u02ec\u02ee\u0374\u037a\u0559\u0640\u06e5\u06e6\u07f4\u07f5\u07fa\u081a\u0824\u0828\u0971\u0e46\u0ec6\u10fc\u17d7\u1843\u1aa7\u1c78\u1c79\u1c7a\u1c7b\u1c7c\u1c7d\u1d2c\u1d2d\u1d2e\u1d2f\u1d30\u1d31\u1d32\u1d33\u1d34\u1d35\u1d36\u1d37\u1d38\u1d39\u1d3a\u1d3b\u1d3c\u1d3d\u1d3e\u1d3f\u1d40\u1d41\u1d42\u1d43\u1d44\u1d45\u1d46\u1d47\u1d48\u1d49\u1d4a\u1d4b\u1d4c\u1d4d\u1d4e\u1d4f\u1d50\u1d51\u1d52\u1d53\u1d54\u1d55\u1d56\u1d57\u1d58\u1d59\u1d5a\u1d5b\u1d5c\u1d5d\u1d5e\u1d5f\u1d60\u1d61\u1d78\u1d9b\u1d9c\u1d9d\u1d9e\u1d9f\u1da0\u1da1\u1da2\u1da3\u1da4\u1da5\u1da6\u1da7\u1da8\u1da9\u1daa\u1dab\u1dac\u1dad\u1dae\u1daf\u1db0\u1db1\u1db2\u1db3\u1db4\u1db5\u1db6\u1db7\u1db8\u1db9\u1dba\u1dbb\u1dbc\u1dbd\u1dbe\u1dbf\u2071\u207f\u2090\u2091\u2092\u2093\u2094\u2095\u2096\u2097\u2098\u2099\u209a\u209b\u209c\u2c7d\u2d6f\u2e2f\u3005\u3031\u3032\u3033\u3034\u3035\u303b\u309d\u309e\u30fc\u30fd\u30fe\ua015\ua4f8\ua4f9\ua4fa\ua4fb\ua4fc\ua4fd\ua60c\ua67f\ua717\ua718\ua719\ua71a\ua71b\ua71c\ua71d\ua71e\ua71f\ua770\ua788\ua9cf\uaa70\uaadd\uff70\uff9e\uff9f'
			},
			{
				'description': 'Unicode character category [Lo]: letter, other',
				'decoded': [443, 448, 449, 450, 451, 660, 1488, 1489, 1490, 1491, 1492, 1493, 1494, 1495, 1496, 1497, 1498, 1499, 1500, 1501, 1502, 1503, 1504, 1505, 1506, 1507, 1508, 1509, 1510, 1511, 1512, 1513, 1514, 1520, 1521, 1522, 1568, 1569, 1570, 1571, 1572, 1573, 1574, 1575, 1576, 1577, 1578, 1579, 1580, 1581, 1582, 1583, 1584, 1585, 1586, 1587, 1588, 1589, 1590, 1591, 1592, 1593, 1594, 1595, 1596, 1597, 1598, 1599, 1601, 1602, 1603, 1604, 1605, 1606, 1607, 1608, 1609, 1610, 1646, 1647, 1649, 1650, 1651, 1652, 1653, 1654, 1655, 1656, 1657, 1658, 1659, 1660, 1661, 1662, 1663, 1664, 1665, 1666, 1667, 1668, 1669, 1670, 1671, 1672, 1673, 1674, 1675, 1676, 1677, 1678, 1679, 1680, 1681, 1682, 1683, 1684, 1685, 1686, 1687, 1688, 1689, 1690, 1691, 1692, 1693, 1694, 1695, 1696, 1697, 1698, 1699, 1700, 1701, 1702, 1703, 1704, 1705, 1706, 1707, 1708, 1709, 1710, 1711, 1712, 1713, 1714, 1715, 1716, 1717, 1718, 1719, 1720, 1721, 1722, 1723, 1724, 1725, 1726, 1727, 1728, 1729, 1730, 1731, 1732, 1733, 1734, 1735, 1736, 1737, 1738, 1739, 1740, 1741, 1742, 1743, 1744, 1745, 1746, 1747, 1749, 1774, 1775, 1786, 1787, 1788, 1791, 1808, 1810, 1811, 1812, 1813, 1814, 1815, 1816, 1817, 1818, 1819, 1820, 1821, 1822, 1823, 1824, 1825, 1826, 1827, 1828, 1829, 1830, 1831, 1832, 1833, 1834, 1835, 1836, 1837, 1838, 1839, 1869, 1870, 1871, 1872, 1873, 1874, 1875, 1876, 1877, 1878, 1879, 1880, 1881, 1882, 1883, 1884, 1885, 1886, 1887, 1888, 1889, 1890, 1891, 1892, 1893, 1894, 1895, 1896, 1897, 1898, 1899, 1900, 1901, 1902, 1903, 1904, 1905, 1906, 1907, 1908, 1909, 1910, 1911, 1912, 1913, 1914, 1915, 1916, 1917, 1918, 1919, 1920, 1921, 1922, 1923, 1924, 1925, 1926, 1927, 1928, 1929, 1930, 1931, 1932, 1933, 1934, 1935, 1936, 1937, 1938, 1939, 1940, 1941, 1942, 1943, 1944, 1945, 1946, 1947, 1948, 1949, 1950, 1951, 1952, 1953, 1954, 1955, 1956, 1957, 1969, 1994, 1995, 1996, 1997, 1998, 1999, 2000, 2001, 2002, 2003, 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016, 2017, 2018, 2019, 2020, 2021, 2022, 2023, 2024, 2025, 2026, 2048, 2049, 2050, 2051, 2052, 2053, 2054, 2055, 2056, 2057, 2058, 2059, 2060, 2061, 2062, 2063, 2064, 2065, 2066, 2067, 2068, 2069, 2112, 2113, 2114, 2115, 2116, 2117, 2118, 2119, 2120, 2121, 2122, 2123, 2124, 2125, 2126, 2127, 2128, 2129, 2130, 2131, 2132, 2133, 2134, 2135, 2136, 2308, 2309, 2310, 2311, 2312, 2313, 2314, 2315, 2316, 2317, 2318, 2319, 2320, 2321, 2322, 2323, 2324, 2325, 2326, 2327, 2328, 2329, 2330, 2331, 2332, 2333, 2334, 2335, 2336, 2337, 2338, 2339, 2340, 2341, 2342, 2343, 2344, 2345, 2346, 2347, 2348, 2349, 2350, 2351, 2352, 2353, 2354, 2355, 2356, 2357, 2358, 2359, 2360, 2361, 2365, 2384, 2325, 2364, 2326, 2364, 2327, 2364, 2332, 2364, 2337, 2364, 2338, 2364, 2347, 2364, 2351, 2364, 2400, 2401, 2418, 2419, 2420, 2421, 2422, 2423, 2425, 2426, 2427, 2428, 2429, 2430, 2431, 2437, 2438, 2439, 2440, 2441, 2442, 2443, 2444, 2447, 2448, 2451, 2452, 2453, 2454, 2455, 2456, 2457, 2458, 2459, 2460, 2461, 2462, 2463, 2464, 2465, 2466, 2467, 2468, 2469, 2470, 2471, 2472, 2474, 2475, 2476, 2477, 2478, 2479, 2480, 2482, 2486, 2487, 2488, 2489, 2493, 2510, 2465, 2492, 2466, 2492, 2479, 2492, 2528, 2529, 2544, 2545, 2565, 2566, 2567, 2568, 2569, 2570, 2575, 2576, 2579, 2580, 2581, 2582, 2583, 2584, 2585, 2586, 2587, 2588, 2589, 2590, 2591, 2592, 2593, 2594, 2595, 2596, 2597, 2598, 2599, 2600, 2602, 2603, 2604, 2605, 2606, 2607, 2608, 2610, 2610, 2620, 2613, 2616, 2620, 2616, 2617, 2582, 2620, 2583, 2620, 2588, 2620, 2652, 2603, 2620, 2674, 2675, 2676, 2693, 2694, 2695, 2696, 2697, 2698, 2699, 2700, 2701, 2703, 2704, 2705, 2707, 2708, 2709, 2710, 2711, 2712, 2713, 2714, 2715, 2716, 2717, 2718, 2719, 2720, 2721, 2722, 2723, 2724, 2725, 2726, 2727, 2728, 2730, 2731, 2732, 2733, 2734, 2735, 2736, 2738, 2739, 2741, 2742, 2743, 2744, 2745, 2749, 2768, 2784, 2785, 2821, 2822, 2823, 2824, 2825, 2826, 2827, 2828, 2831, 2832, 2835, 2836, 2837, 2838, 2839, 2840, 2841, 2842, 2843, 2844, 2845, 2846, 2847, 2848, 2849, 2850, 2851, 2852, 2853, 2854, 2855, 2856, 2858, 2859, 2860, 2861, 2862, 2863, 2864, 2866, 2867, 2869, 2870, 2871, 2872, 2873, 2877, 2849, 2876, 2850, 2876, 2911, 2912, 2913, 2929, 2947, 2949, 2950, 2951, 2952, 2953, 2954, 2958, 2959, 2960, 2962, 2963, 2964, 2965, 2969, 2970, 2972, 2974, 2975, 2979, 2980, 2984, 2985, 2986, 2990, 2991, 2992, 2993, 2994, 2995, 2996, 2997, 2998, 2999, 3000, 3001, 3024, 3077, 3078, 3079, 3080, 3081, 3082, 3083, 3084, 3086, 3087, 3088, 3090, 3091, 3092, 3093, 3094, 3095, 3096, 3097, 3098, 3099, 3100, 3101, 3102, 3103, 3104, 3105, 3106, 3107, 3108, 3109, 3110, 3111, 3112, 3114, 3115, 3116, 3117, 3118, 3119, 3120, 3121, 3122, 3123, 3125, 3126, 3127, 3128, 3129, 3133, 3160, 3161, 3168, 3169, 3205, 3206, 3207, 3208, 3209, 3210, 3211, 3212, 3214, 3215, 3216, 3218, 3219, 3220, 3221, 3222, 3223, 3224, 3225, 3226, 3227, 3228, 3229, 3230, 3231, 3232, 3233, 3234, 3235, 3236, 3237, 3238, 3239, 3240, 3242, 3243, 3244, 3245, 3246, 3247, 3248, 3249, 3250, 3251, 3253, 3254, 3255, 3256, 3257, 3261, 3294, 3296, 3297, 3313, 3314, 3333, 3334, 3335, 3336, 3337, 3338, 3339, 3340, 3342, 3343, 3344, 3346, 3347, 3348, 3349, 3350, 3351, 3352, 3353, 3354, 3355, 3356, 3357, 3358, 3359, 3360, 3361, 3362, 3363, 3364, 3365, 3366, 3367, 3368, 3369, 3370, 3371, 3372, 3373, 3374, 3375, 3376, 3377, 3378, 3379, 3380, 3381, 3382, 3383, 3384, 3385, 3386, 3389, 3406, 3424, 3425, 3450, 3451, 3452, 3453, 3454, 3455, 3461, 3462, 3463, 3464, 3465, 3466, 3467, 3468, 3469, 3470, 3471, 3472, 3473, 3474, 3475, 3476, 3477, 3478, 3482, 3483, 3484, 3485, 3486, 3487, 3488, 3489, 3490, 3491, 3492, 3493, 3494, 3495, 3496, 3497, 3498, 3499, 3500, 3501, 3502, 3503, 3504, 3505, 3507, 3508, 3509, 3510, 3511, 3512, 3513, 3514, 3515, 3517, 3520, 3521, 3522, 3523, 3524, 3525, 3526, 3585, 3586, 3587, 3588, 3589, 3590, 3591, 3592, 3593, 3594, 3595, 3596, 3597, 3598, 3599, 3600, 3601, 3602, 3603, 3604, 3605, 3606, 3607, 3608, 3609, 3610, 3611, 3612, 3613, 3614, 3615, 3616, 3617, 3618, 3619, 3620, 3621, 3622, 3623, 3624, 3625, 3626, 3627, 3628, 3629, 3630, 3631, 3632, 3634, 3635, 3648, 3649, 3650, 3651, 3652, 3653, 3713, 3714, 3716, 3719, 3720, 3722, 3725, 3732, 3733, 3734, 3735, 3737, 3738, 3739, 3740, 3741, 3742, 3743, 3745, 3746, 3747, 3749, 3751, 3754, 3755, 3757, 3758, 3759, 3760, 3762, 3763, 3773, 3776, 3777, 3778, 3779, 3780, 3804, 3805, 3840, 3904, 3905, 3906, 3906, 4023, 3908, 3909, 3910, 3911, 3913, 3914, 3915, 3916, 3916, 4023, 3918, 3919, 3920, 3921, 3921, 4023, 3923, 3924, 3925, 3926, 3926, 4023, 3928, 3929, 3930, 3931, 3931, 4023, 3933, 3934, 3935, 3936, 3937, 3938, 3939, 3940, 3941, 3942, 3943, 3944, 3904, 4021, 3946, 3947, 3948, 3976, 3977, 3978, 3979, 3980, 4096, 4097, 4098, 4099, 4100, 4101, 4102, 4103, 4104, 4105, 4106, 4107, 4108, 4109, 4110, 4111, 4112, 4113, 4114, 4115, 4116, 4117, 4118, 4119, 4120, 4121, 4122, 4123, 4124, 4125, 4126, 4127, 4128, 4129, 4130, 4131, 4132, 4133, 4134, 4135, 4136, 4137, 4138, 4159, 4176, 4177, 4178, 4179, 4180, 4181, 4186, 4187, 4188, 4189, 4193, 4197, 4198, 4206, 4207, 4208, 4213, 4214, 4215, 4216, 4217, 4218, 4219, 4220, 4221, 4222, 4223, 4224, 4225, 4238, 4304, 4305, 4306, 4307, 4308, 4309, 4310, 4311, 4312, 4313, 4314, 4315, 4316, 4317, 4318, 4319, 4320, 4321, 4322, 4323, 4324, 4325, 4326, 4327, 4328, 4329, 4330, 4331, 4332, 4333, 4334, 4335, 4336, 4337, 4338, 4339, 4340, 4341, 4342, 4343, 4344, 4345, 4346, 4352, 4353, 4354, 4355, 4356, 4357, 4358, 4359, 4360, 4361, 4362, 4363, 4364, 4365, 4366, 4367, 4368, 4369, 4370, 4371, 4372, 4373, 4374, 4375, 4376, 4377, 4378, 4379, 4380, 4381, 4382, 4383, 4384, 4385, 4386, 4387, 4388, 4389, 4390, 4391, 4392, 4393, 4394, 4395, 4396, 4397, 4398, 4399, 4400, 4401, 4402, 4403, 4404, 4405, 4406, 4407, 4408, 4409, 4410, 4411, 4412, 4413, 4414, 4415, 4416, 4417, 4418, 4419, 4420, 4421, 4422, 4423, 4424, 4425, 4426, 4427, 4428, 4429, 4430, 4431, 4432, 4433, 4434, 4435, 4436, 4437, 4438, 4439, 4440, 4441, 4442, 4443, 4444, 4445, 4446, 4447, 4448, 4449, 4450, 4451, 4452, 4453, 4454, 4455, 4456, 4457, 4458, 4459, 4460, 4461, 4462, 4463, 4464, 4465, 4466, 4467, 4468, 4469, 4470, 4471, 4472, 4473, 4474, 4475, 4476, 4477, 4478, 4479, 4480, 4481, 4482, 4483, 4484, 4485, 4486, 4487, 4488, 4489, 4490, 4491, 4492, 4493, 4494, 4495, 4496, 4497, 4498, 4499, 4500, 4501, 4502, 4503, 4504, 4505, 4506, 4507, 4508, 4509, 4510, 4511, 4512, 4513, 4514, 4515, 4516, 4517, 4518, 4519, 4520, 4521, 4522, 4523, 4524, 4525, 4526, 4527, 4528, 4529, 4530, 4531, 4532, 4533, 4534, 4535, 4536, 4537, 4538, 4539, 4540, 4541, 4542, 4543, 4544, 4545, 4546, 4547, 4548, 4549, 4550, 4551, 4552, 4553, 4554, 4555, 4556, 4557, 4558, 4559, 4560, 4561, 4562, 4563, 4564, 4565, 4566, 4567, 4568, 4569, 4570, 4571, 4572, 4573, 4574, 4575, 4576, 4577, 4578, 4579, 4580, 4581, 4582, 4583, 4584, 4585, 4586, 4587, 4588, 4589, 4590, 4591, 4592, 4593, 4594, 4595, 4596, 4597, 4598, 4599, 4600, 4601, 4602, 4603, 4604, 4605, 4606, 4607, 4608, 4609, 4610, 4611, 4612, 4613, 4614, 4615, 4616, 4617, 4618, 4619, 4620, 4621, 4622, 4623, 4624, 4625, 4626, 4627, 4628, 4629, 4630, 4631, 4632, 4633, 4634, 4635, 4636, 4637, 4638, 4639, 4640, 4641, 4642, 4643, 4644, 4645, 4646, 4647, 4648, 4649, 4650, 4651, 4652, 4653, 4654, 4655, 4656, 4657, 4658, 4659, 4660, 4661, 4662, 4663, 4664, 4665, 4666, 4667, 4668, 4669, 4670, 4671, 4672, 4673, 4674, 4675, 4676, 4677, 4678, 4679, 4680, 4682, 4683, 4684, 4685, 4688, 4689, 4690, 4691, 4692, 4693, 4694, 4696, 4698, 4699, 4700, 4701, 4704, 4705, 4706, 4707, 4708, 4709, 4710, 4711, 4712, 4713, 4714, 4715, 4716, 4717, 4718, 4719, 4720, 4721, 4722, 4723, 4724, 4725, 4726, 4727, 4728, 4729, 4730, 4731, 4732, 4733, 4734, 4735, 4736, 4737, 4738, 4739, 4740, 4741, 4742, 4743, 4744, 4746, 4747, 4748, 4749, 4752, 4753, 4754, 4755, 4756, 4757, 4758, 4759, 4760, 4761, 4762, 4763, 4764, 4765, 4766, 4767, 4768, 4769, 4770, 4771, 4772, 4773, 4774, 4775, 4776, 4777, 4778, 4779, 4780, 4781, 4782, 4783, 4784, 4786, 4787, 4788, 4789, 4792, 4793, 4794, 4795, 4796, 4797, 4798, 4800, 4802, 4803, 4804, 4805, 4808, 4809, 4810, 4811, 4812, 4813, 4814, 4815, 4816, 4817, 4818, 4819, 4820, 4821, 4822, 4824, 4825, 4826, 4827, 4828, 4829, 4830, 4831, 4832, 4833, 4834, 4835, 4836, 4837, 4838, 4839, 4840, 4841, 4842, 4843, 4844, 4845, 4846, 4847, 4848, 4849, 4850, 4851, 4852, 4853, 4854, 4855, 4856, 4857, 4858, 4859, 4860, 4861, 4862, 4863, 4864, 4865, 4866, 4867, 4868, 4869, 4870, 4871, 4872, 4873, 4874, 4875, 4876, 4877, 4878, 4879, 4880, 4882, 4883, 4884, 4885, 4888, 4889, 4890, 4891, 4892, 4893, 4894, 4895, 4896, 4897, 4898, 4899, 4900, 4901, 4902, 4903, 4904, 4905, 4906, 4907, 4908, 4909, 4910, 4911, 4912, 4913, 4914, 4915, 4916, 4917, 4918, 4919, 4920, 4921, 4922, 4923, 4924, 4925, 4926, 4927, 4928, 4929, 4930, 4931, 4932, 4933, 4934, 4935, 4936, 4937, 4938, 4939, 4940, 4941, 4942, 4943, 4944, 4945, 4946, 4947, 4948, 4949, 4950, 4951, 4952, 4953, 4954, 4992, 4993, 4994, 4995, 4996, 4997, 4998, 4999, 5000, 5001, 5002, 5003, 5004, 5005, 5006, 5007, 5024, 5025, 5026, 5027, 5028, 5029, 5030, 5031, 5032, 5033, 5034, 5035, 5036, 5037, 5038, 5039, 5040, 5041, 5042, 5043, 5044, 5045, 5046, 5047, 5048, 5049, 5050, 5051, 5052, 5053, 5054, 5055, 5056, 5057, 5058, 5059, 5060, 5061, 5062, 5063, 5064, 5065, 5066, 5067, 5068, 5069, 5070, 5071, 5072, 5073, 5074, 5075, 5076, 5077, 5078, 5079, 5080, 5081, 5082, 5083, 5084, 5085, 5086, 5087, 5088, 5089, 5090, 5091, 5092, 5093, 5094, 5095, 5096, 5097, 5098, 5099, 5100, 5101, 5102, 5103, 5104, 5105, 5106, 5107, 5108, 5121, 5122, 5123, 5124, 5125, 5126, 5127, 5128, 5129, 5130, 5131, 5132, 5133, 5134, 5135, 5136, 5137, 5138, 5139, 5140, 5141, 5142, 5143, 5144, 5145, 5146, 5147, 5148, 5149, 5150, 5151, 5152, 5153, 5154, 5155, 5156, 5157, 5158, 5159, 5160, 5161, 5162, 5163, 5164, 5165, 5166, 5167, 5168, 5169, 5170, 5171, 5172, 5173, 5174, 5175, 5176, 5177, 5178, 5179, 5180, 5181, 5182, 5183, 5184, 5185, 5186, 5187, 5188, 5189, 5190, 5191, 5192, 5193, 5194, 5195, 5196, 5197, 5198, 5199, 5200, 5201, 5202, 5203, 5204, 5205, 5206, 5207, 5208, 5209, 5210, 5211, 5212, 5213, 5214, 5215, 5216, 5217, 5218, 5219, 5220, 5221, 5222, 5223, 5224, 5225, 5226, 5227, 5228, 5229, 5230, 5231, 5232, 5233, 5234, 5235, 5236, 5237, 5238, 5239, 5240, 5241, 5242, 5243, 5244, 5245, 5246, 5247, 5248, 5249, 5250, 5251, 5252, 5253, 5254, 5255, 5256, 5257, 5258, 5259, 5260, 5261, 5262, 5263, 5264, 5265, 5266, 5267, 5268, 5269, 5270, 5271, 5272, 5273, 5274, 5275, 5276, 5277, 5278, 5279, 5280, 5281, 5282, 5283, 5284, 5285, 5286, 5287, 5288, 5289, 5290, 5291, 5292, 5293, 5294, 5295, 5296, 5297, 5298, 5299, 5300, 5301, 5302, 5303, 5304, 5305, 5306, 5307, 5308, 5309, 5310, 5311, 5312, 5313, 5314, 5315, 5316, 5317, 5318, 5319, 5320, 5321, 5322, 5323, 5324, 5325, 5326, 5327, 5328, 5329, 5330, 5331, 5332, 5333, 5334, 5335, 5336, 5337, 5338, 5339, 5340, 5341, 5342, 5343, 5344, 5345, 5346, 5347, 5348, 5349, 5350, 5351, 5352, 5353, 5354, 5355, 5356, 5357, 5358, 5359, 5360, 5361, 5362, 5363, 5364, 5365, 5366, 5367, 5368, 5369, 5370, 5371, 5372, 5373, 5374, 5375, 5376, 5377, 5378, 5379, 5380, 5381, 5382, 5383, 5384, 5385, 5386, 5387, 5388, 5389, 5390, 5391, 5392, 5393, 5394, 5395, 5396, 5397, 5398, 5399, 5400, 5401, 5402, 5403, 5404, 5405, 5406, 5407, 5408, 5409, 5410, 5411, 5412, 5413, 5414, 5415, 5416, 5417, 5418, 5419, 5420, 5421, 5422, 5423, 5424, 5425, 5426, 5427, 5428, 5429, 5430, 5431, 5432, 5433, 5434, 5435, 5436, 5437, 5438, 5439, 5440, 5441, 5442, 5443, 5444, 5445, 5446, 5447, 5448, 5449, 5450, 5451, 5452, 5453, 5454, 5455, 5456, 5457, 5458, 5459, 5460, 5461, 5462, 5463, 5464, 5465, 5466, 5467, 5468, 5469, 5470, 5471, 5472, 5473, 5474, 5475, 5476, 5477, 5478, 5479, 5480, 5481, 5482, 5483, 5484, 5485, 5486, 5487, 5488, 5489, 5490, 5491, 5492, 5493, 5494, 5495, 5496, 5497, 5498, 5499, 5500, 5501, 5502, 5503, 5504, 5505, 5506, 5507, 5508, 5509, 5510, 5511, 5512, 5513, 5514, 5515, 5516, 5517, 5518, 5519, 5520, 5521, 5522, 5523, 5524, 5525, 5526, 5527, 5528, 5529, 5530, 5531, 5532, 5533, 5534, 5535, 5536, 5537, 5538, 5539, 5540, 5541, 5542, 5543, 5544, 5545, 5546, 5547, 5548, 5549, 5550, 5551, 5552, 5553, 5554, 5555, 5556, 5557, 5558, 5559, 5560, 5561, 5562, 5563, 5564, 5565, 5566, 5567, 5568, 5569, 5570, 5571, 5572, 5573, 5574, 5575, 5576, 5577, 5578, 5579, 5580, 5581, 5582, 5583, 5584, 5585, 5586, 5587, 5588, 5589, 5590, 5591, 5592, 5593, 5594, 5595, 5596, 5597, 5598, 5599, 5600, 5601, 5602, 5603, 5604, 5605, 5606, 5607, 5608, 5609, 5610, 5611, 5612, 5613, 5614, 5615, 5616, 5617, 5618, 5619, 5620, 5621, 5622, 5623, 5624, 5625, 5626, 5627, 5628, 5629, 5630, 5631, 5632, 5633, 5634, 5635, 5636, 5637, 5638, 5639, 5640, 5641, 5642, 5643, 5644, 5645, 5646, 5647, 5648, 5649, 5650, 5651, 5652, 5653, 5654, 5655, 5656, 5657, 5658, 5659, 5660, 5661, 5662, 5663, 5664, 5665, 5666, 5667, 5668, 5669, 5670, 5671, 5672, 5673, 5674, 5675, 5676, 5677, 5678, 5679, 5680, 5681, 5682, 5683, 5684, 5685, 5686, 5687, 5688, 5689, 5690, 5691, 5692, 5693, 5694, 5695, 5696, 5697, 5698, 5699, 5700, 5701, 5702, 5703, 5704, 5705, 5706, 5707, 5708, 5709, 5710, 5711, 5712, 5713, 5714, 5715, 5716, 5717, 5718, 5719, 5720, 5721, 5722, 5723, 5724, 5725, 5726, 5727, 5728, 5729, 5730, 5731, 5732, 5733, 5734, 5735, 5736, 5737, 5738, 5739, 5740, 5743, 5744, 5745, 5746, 5747, 5748, 5749, 5750, 5751, 5752, 5753, 5754, 5755, 5756, 5757, 5758, 5759, 5761, 5762, 5763, 5764, 5765, 5766, 5767, 5768, 5769, 5770, 5771, 5772, 5773, 5774, 5775, 5776, 5777, 5778, 5779, 5780, 5781, 5782, 5783, 5784, 5785, 5786, 5792, 5793, 5794, 5795, 5796, 5797, 5798, 5799, 5800, 5801, 5802, 5803, 5804, 5805, 5806, 5807, 5808, 5809, 5810, 5811, 5812, 5813, 5814, 5815, 5816, 5817, 5818, 5819, 5820, 5821, 5822, 5823, 5824, 5825, 5826, 5827, 5828, 5829, 5830, 5831, 5832, 5833, 5834, 5835, 5836, 5837, 5838, 5839, 5840, 5841, 5842, 5843, 5844, 5845, 5846, 5847, 5848, 5849, 5850, 5851, 5852, 5853, 5854, 5855, 5856, 5857, 5858, 5859, 5860, 5861, 5862, 5863, 5864, 5865, 5866, 5888, 5889, 5890, 5891, 5892, 5893, 5894, 5895, 5896, 5897, 5898, 5899, 5900, 5902, 5903, 5904, 5905, 5920, 5921, 5922, 5923, 5924, 5925, 5926, 5927, 5928, 5929, 5930, 5931, 5932, 5933, 5934, 5935, 5936, 5937, 5952, 5953, 5954, 5955, 5956, 5957, 5958, 5959, 5960, 5961, 5962, 5963, 5964, 5965, 5966, 5967, 5968, 5969, 5984, 5985, 5986, 5987, 5988, 5989, 5990, 5991, 5992, 5993, 5994, 5995, 5996, 5998, 5999, 6000, 6016, 6017, 6018, 6019, 6020, 6021, 6022, 6023, 6024, 6025, 6026, 6027, 6028, 6029, 6030, 6031, 6032, 6033, 6034, 6035, 6036, 6037, 6038, 6039, 6040, 6041, 6042, 6043, 6044, 6045, 6046, 6047, 6048, 6049, 6050, 6051, 6052, 6053, 6054, 6055, 6056, 6057, 6058, 6059, 6060, 6061, 6062, 6063, 6064, 6065, 6066, 6067, 6108, 6176, 6177, 6178, 6179, 6180, 6181, 6182, 6183, 6184, 6185, 6186, 6187, 6188, 6189, 6190, 6191, 6192, 6193, 6194, 6195, 6196, 6197, 6198, 6199, 6200, 6201, 6202, 6203, 6204, 6205, 6206, 6207, 6208, 6209, 6210, 6212, 6213, 6214, 6215, 6216, 6217, 6218, 6219, 6220, 6221, 6222, 6223, 6224, 6225, 6226, 6227, 6228, 6229, 6230, 6231, 6232, 6233, 6234, 6235, 6236, 6237, 6238, 6239, 6240, 6241, 6242, 6243, 6244, 6245, 6246, 6247, 6248, 6249, 6250, 6251, 6252, 6253, 6254, 6255, 6256, 6257, 6258, 6259, 6260, 6261, 6262, 6263, 6272, 6273, 6274, 6275, 6276, 6277, 6278, 6279, 6280, 6281, 6282, 6283, 6284, 6285, 6286, 6287, 6288, 6289, 6290, 6291, 6292, 6293, 6294, 6295, 6296, 6297, 6298, 6299, 6300, 6301, 6302, 6303, 6304, 6305, 6306, 6307, 6308, 6309, 6310, 6311, 6312, 6314, 6320, 6321, 6322, 6323, 6324, 6325, 6326, 6327, 6328, 6329, 6330, 6331, 6332, 6333, 6334, 6335, 6336, 6337, 6338, 6339, 6340, 6341, 6342, 6343, 6344, 6345, 6346, 6347, 6348, 6349, 6350, 6351, 6352, 6353, 6354, 6355, 6356, 6357, 6358, 6359, 6360, 6361, 6362, 6363, 6364, 6365, 6366, 6367, 6368, 6369, 6370, 6371, 6372, 6373, 6374, 6375, 6376, 6377, 6378, 6379, 6380, 6381, 6382, 6383, 6384, 6385, 6386, 6387, 6388, 6389, 6400, 6401, 6402, 6403, 6404, 6405, 6406, 6407, 6408, 6409, 6410, 6411, 6412, 6413, 6414, 6415, 6416, 6417, 6418, 6419, 6420, 6421, 6422, 6423, 6424, 6425, 6426, 6427, 6428, 6480, 6481, 6482, 6483, 6484, 6485, 6486, 6487, 6488, 6489, 6490, 6491, 6492, 6493, 6494, 6495, 6496, 6497, 6498, 6499, 6500, 6501, 6502, 6503, 6504, 6505, 6506, 6507, 6508, 6509, 6512, 6513, 6514, 6515, 6516, 6528, 6529, 6530, 6531, 6532, 6533, 6534, 6535, 6536, 6537, 6538, 6539, 6540, 6541, 6542, 6543, 6544, 6545, 6546, 6547, 6548, 6549, 6550, 6551, 6552, 6553, 6554, 6555, 6556, 6557, 6558, 6559, 6560, 6561, 6562, 6563, 6564, 6565, 6566, 6567, 6568, 6569, 6570, 6571, 6593, 6594, 6595, 6596, 6597, 6598, 6599, 6656, 6657, 6658, 6659, 6660, 6661, 6662, 6663, 6664, 6665, 6666, 6667, 6668, 6669, 6670, 6671, 6672, 6673, 6674, 6675, 6676, 6677, 6678, 6688, 6689, 6690, 6691, 6692, 6693, 6694, 6695, 6696, 6697, 6698, 6699, 6700, 6701, 6702, 6703, 6704, 6705, 6706, 6707, 6708, 6709, 6710, 6711, 6712, 6713, 6714, 6715, 6716, 6717, 6718, 6719, 6720, 6721, 6722, 6723, 6724, 6725, 6726, 6727, 6728, 6729, 6730, 6731, 6732, 6733, 6734, 6735, 6736, 6737, 6738, 6739, 6740, 6917, 6918, 6919, 6920, 6921, 6922, 6923, 6924, 6925, 6926, 6927, 6928, 6929, 6930, 6931, 6932, 6933, 6934, 6935, 6936, 6937, 6938, 6939, 6940, 6941, 6942, 6943, 6944, 6945, 6946, 6947, 6948, 6949, 6950, 6951, 6952, 6953, 6954, 6955, 6956, 6957, 6958, 6959, 6960, 6961, 6962, 6963, 6981, 6982, 6983, 6984, 6985, 6986, 6987, 7043, 7044, 7045, 7046, 7047, 7048, 7049, 7050, 7051, 7052, 7053, 7054, 7055, 7056, 7057, 7058, 7059, 7060, 7061, 7062, 7063, 7064, 7065, 7066, 7067, 7068, 7069, 7070, 7071, 7072, 7086, 7087, 7104, 7105, 7106, 7107, 7108, 7109, 7110, 7111, 7112, 7113, 7114, 7115, 7116, 7117, 7118, 7119, 7120, 7121, 7122, 7123, 7124, 7125, 7126, 7127, 7128, 7129, 7130, 7131, 7132, 7133, 7134, 7135, 7136, 7137, 7138, 7139, 7140, 7141, 7168, 7169, 7170, 7171, 7172, 7173, 7174, 7175, 7176, 7177, 7178, 7179, 7180, 7181, 7182, 7183, 7184, 7185, 7186, 7187, 7188, 7189, 7190, 7191, 7192, 7193, 7194, 7195, 7196, 7197, 7198, 7199, 7200, 7201, 7202, 7203, 7245, 7246, 7247, 7258, 7259, 7260, 7261, 7262, 7263, 7264, 7265, 7266, 7267, 7268, 7269, 7270, 7271, 7272, 7273, 7274, 7275, 7276, 7277, 7278, 7279, 7280, 7281, 7282, 7283, 7284, 7285, 7286, 7287, 7401, 7402, 7403, 7404, 7406, 7407, 7408, 7409, 8501, 8502, 8503, 8504, 11568, 11569, 11570, 11571, 11572, 11573, 11574, 11575, 11576, 11577, 11578, 11579, 11580, 11581, 11582, 11583, 11584, 11585, 11586, 11587, 11588, 11589, 11590, 11591, 11592, 11593, 11594, 11595, 11596, 11597, 11598, 11599, 11600, 11601, 11602, 11603, 11604, 11605, 11606, 11607, 11608, 11609, 11610, 11611, 11612, 11613, 11614, 11615, 11616, 11617, 11618, 11619, 11620, 11621, 11648, 11649, 11650, 11651, 11652, 11653, 11654, 11655, 11656, 11657, 11658, 11659, 11660, 11661, 11662, 11663, 11664, 11665, 11666, 11667, 11668, 11669, 11670, 11680, 11681, 11682, 11683, 11684, 11685, 11686, 11688, 11689, 11690, 11691, 11692, 11693, 11694, 11696, 11697, 11698, 11699, 11700, 11701, 11702, 11704, 11705, 11706, 11707, 11708, 11709, 11710, 11712, 11713, 11714, 11715, 11716, 11717, 11718, 11720, 11721, 11722, 11723, 11724, 11725, 11726, 11728, 11729, 11730, 11731, 11732, 11733, 11734, 11736, 11737, 11738, 11739, 11740, 11741, 11742, 12294, 12348, 12353, 12354, 12355, 12356, 12357, 12358, 12359, 12360, 12361, 12362, 12363, 12364, 12365, 12366, 12367, 12368, 12369, 12370, 12371, 12372, 12373, 12374, 12375, 12376, 12377, 12378, 12379, 12380, 12381, 12382, 12383, 12384, 12385, 12386, 12387, 12388, 12389, 12390, 12391, 12392, 12393, 12394, 12395, 12396, 12397, 12398, 12399, 12400, 12401, 12402, 12403, 12404, 12405, 12406, 12407, 12408, 12409, 12410, 12411, 12412, 12413, 12414, 12415, 12416, 12417, 12418, 12419, 12420, 12421, 12422, 12423, 12424, 12425, 12426, 12427, 12428, 12429, 12430, 12431, 12432, 12433, 12434, 12435, 12436, 12437, 12438, 12447, 12449, 12450, 12451, 12452, 12453, 12454, 12455, 12456, 12457, 12458, 12459, 12460, 12461, 12462, 12463, 12464, 12465, 12466, 12467, 12468, 12469, 12470, 12471, 12472, 12473, 12474, 12475, 12476, 12477, 12478, 12479, 12480, 12481, 12482, 12483, 12484, 12485, 12486, 12487, 12488, 12489, 12490, 12491, 12492, 12493, 12494, 12495, 12496, 12497, 12498, 12499, 12500, 12501, 12502, 12503, 12504, 12505, 12506, 12507, 12508, 12509, 12510, 12511, 12512, 12513, 12514, 12515, 12516, 12517, 12518, 12519, 12520, 12521, 12522, 12523, 12524, 12525, 12526, 12527, 12528, 12529, 12530, 12531, 12532, 12533, 12534, 12535, 12536, 12537, 12538, 12543, 12549, 12550, 12551, 12552, 12553, 12554, 12555, 12556, 12557, 12558, 12559, 12560, 12561, 12562, 12563, 12564, 12565, 12566, 12567, 12568, 12569, 12570, 12571, 12572, 12573, 12574, 12575, 12576, 12577, 12578, 12579, 12580, 12581, 12582, 12583, 12584, 12585, 12586, 12587, 12588, 12589, 12593, 12594, 12595, 12596, 12597, 12598, 12599, 12600, 12601, 12602, 12603, 12604, 12605, 12606, 12607, 12608, 12609, 12610, 12611, 12612, 12613, 12614, 12615, 12616, 12617, 12618, 12619, 12620, 12621, 12622, 12623, 12624, 12625, 12626, 12627, 12628, 12629, 12630, 12631, 12632, 12633, 12634, 12635, 12636, 12637, 12638, 12639, 12640, 12641, 12642, 12643, 12644, 12645, 12646, 12647, 12648, 12649, 12650, 12651, 12652, 12653, 12654, 12655, 12656, 12657, 12658, 12659, 12660, 12661, 12662, 12663, 12664, 12665, 12666, 12667, 12668, 12669, 12670, 12671, 12672, 12673, 12674, 12675, 12676, 12677, 12678, 12679, 12680, 12681, 12682, 12683, 12684, 12685, 12686, 12704, 12705, 12706, 12707, 12708, 12709, 12710, 12711, 12712, 12713, 12714, 12715, 12716, 12717, 12718, 12719, 12720, 12721, 12722, 12723, 12724, 12725, 12726, 12727, 12728, 12729, 12730, 12784, 12785, 12786, 12787, 12788, 12789, 12790, 12791, 12792, 12793, 12794, 12795, 12796, 12797, 12798, 12799, 13312, 19893, 19968, 40907, 40960, 40961, 40962, 40963, 40964, 40965, 40966, 40967, 40968, 40969, 40970, 40971, 40972, 40973, 40974, 40975, 40976, 40977, 40978, 40979, 40980, 40982, 40983, 40984, 40985, 40986, 40987, 40988, 40989, 40990, 40991, 40992, 40993, 40994, 40995, 40996, 40997, 40998, 40999, 41000, 41001, 41002, 41003, 41004, 41005, 41006, 41007, 41008, 41009, 41010, 41011, 41012, 41013, 41014, 41015, 41016, 41017, 41018, 41019, 41020, 41021, 41022, 41023, 41024, 41025, 41026, 41027, 41028, 41029, 41030, 41031, 41032, 41033, 41034, 41035, 41036, 41037, 41038, 41039, 41040, 41041, 41042, 41043, 41044, 41045, 41046, 41047, 41048, 41049, 41050, 41051, 41052, 41053, 41054, 41055, 41056, 41057, 41058, 41059, 41060, 41061, 41062, 41063, 41064, 41065, 41066, 41067, 41068, 41069, 41070, 41071, 41072, 41073, 41074, 41075, 41076, 41077, 41078, 41079, 41080, 41081, 41082, 41083, 41084, 41085, 41086, 41087, 41088, 41089, 41090, 41091, 41092, 41093, 41094, 41095, 41096, 41097, 41098, 41099, 41100, 41101, 41102, 41103, 41104, 41105, 41106, 41107, 41108, 41109, 41110, 41111, 41112, 41113, 41114, 41115, 41116, 41117, 41118, 41119, 41120, 41121, 41122, 41123, 41124, 41125, 41126, 41127, 41128, 41129, 41130, 41131, 41132, 41133, 41134, 41135, 41136, 41137, 41138, 41139, 41140, 41141, 41142, 41143, 41144, 41145, 41146, 41147, 41148, 41149, 41150, 41151, 41152, 41153, 41154, 41155, 41156, 41157, 41158, 41159, 41160, 41161, 41162, 41163, 41164, 41165, 41166, 41167, 41168, 41169, 41170, 41171, 41172, 41173, 41174, 41175, 41176, 41177, 41178, 41179, 41180, 41181, 41182, 41183, 41184, 41185, 41186, 41187, 41188, 41189, 41190, 41191, 41192, 41193, 41194, 41195, 41196, 41197, 41198, 41199, 41200, 41201, 41202, 41203, 41204, 41205, 41206, 41207, 41208, 41209, 41210, 41211, 41212, 41213, 41214, 41215, 41216, 41217, 41218, 41219, 41220, 41221, 41222, 41223, 41224, 41225, 41226, 41227, 41228, 41229, 41230, 41231, 41232, 41233, 41234, 41235, 41236, 41237, 41238, 41239, 41240, 41241, 41242, 41243, 41244, 41245, 41246, 41247, 41248, 41249, 41250, 41251, 41252, 41253, 41254, 41255, 41256, 41257, 41258, 41259, 41260, 41261, 41262, 41263, 41264, 41265, 41266, 41267, 41268, 41269, 41270, 41271, 41272, 41273, 41274, 41275, 41276, 41277, 41278, 41279, 41280, 41281, 41282, 41283, 41284, 41285, 41286, 41287, 41288, 41289, 41290, 41291, 41292, 41293, 41294, 41295, 41296, 41297, 41298, 41299, 41300, 41301, 41302, 41303, 41304, 41305, 41306, 41307, 41308, 41309, 41310, 41311, 41312, 41313, 41314, 41315, 41316, 41317, 41318, 41319, 41320, 41321, 41322, 41323, 41324, 41325, 41326, 41327, 41328, 41329, 41330, 41331, 41332, 41333, 41334, 41335, 41336, 41337, 41338, 41339, 41340, 41341, 41342, 41343, 41344, 41345, 41346, 41347, 41348, 41349, 41350, 41351, 41352, 41353, 41354, 41355, 41356, 41357, 41358, 41359, 41360, 41361, 41362, 41363, 41364, 41365, 41366, 41367, 41368, 41369, 41370, 41371, 41372, 41373, 41374, 41375, 41376, 41377, 41378, 41379, 41380, 41381, 41382, 41383, 41384, 41385, 41386, 41387, 41388, 41389, 41390, 41391, 41392, 41393, 41394, 41395, 41396, 41397, 41398, 41399, 41400, 41401, 41402, 41403, 41404, 41405, 41406, 41407, 41408, 41409, 41410, 41411, 41412, 41413, 41414, 41415, 41416, 41417, 41418, 41419, 41420, 41421, 41422, 41423, 41424, 41425, 41426, 41427, 41428, 41429, 41430, 41431, 41432, 41433, 41434, 41435, 41436, 41437, 41438, 41439, 41440, 41441, 41442, 41443, 41444, 41445, 41446, 41447, 41448, 41449, 41450, 41451, 41452, 41453, 41454, 41455, 41456, 41457, 41458, 41459, 41460, 41461, 41462, 41463, 41464, 41465, 41466, 41467, 41468, 41469, 41470, 41471, 41472, 41473, 41474, 41475, 41476, 41477, 41478, 41479, 41480, 41481, 41482, 41483, 41484, 41485, 41486, 41487, 41488, 41489, 41490, 41491, 41492, 41493, 41494, 41495, 41496, 41497, 41498, 41499, 41500, 41501, 41502, 41503, 41504, 41505, 41506, 41507, 41508, 41509, 41510, 41511, 41512, 41513, 41514, 41515, 41516, 41517, 41518, 41519, 41520, 41521, 41522, 41523, 41524, 41525, 41526, 41527, 41528, 41529, 41530, 41531, 41532, 41533, 41534, 41535, 41536, 41537, 41538, 41539, 41540, 41541, 41542, 41543, 41544, 41545, 41546, 41547, 41548, 41549, 41550, 41551, 41552, 41553, 41554, 41555, 41556, 41557, 41558, 41559, 41560, 41561, 41562, 41563, 41564, 41565, 41566, 41567, 41568, 41569, 41570, 41571, 41572, 41573, 41574, 41575, 41576, 41577, 41578, 41579, 41580, 41581, 41582, 41583, 41584, 41585, 41586, 41587, 41588, 41589, 41590, 41591, 41592, 41593, 41594, 41595, 41596, 41597, 41598, 41599, 41600, 41601, 41602, 41603, 41604, 41605, 41606, 41607, 41608, 41609, 41610, 41611, 41612, 41613, 41614, 41615, 41616, 41617, 41618, 41619, 41620, 41621, 41622, 41623, 41624, 41625, 41626, 41627, 41628, 41629, 41630, 41631, 41632, 41633, 41634, 41635, 41636, 41637, 41638, 41639, 41640, 41641, 41642, 41643, 41644, 41645, 41646, 41647, 41648, 41649, 41650, 41651, 41652, 41653, 41654, 41655, 41656, 41657, 41658, 41659, 41660, 41661, 41662, 41663, 41664, 41665, 41666, 41667, 41668, 41669, 41670, 41671, 41672, 41673, 41674, 41675, 41676, 41677, 41678, 41679, 41680, 41681, 41682, 41683, 41684, 41685, 41686, 41687, 41688, 41689, 41690, 41691, 41692, 41693, 41694, 41695, 41696, 41697, 41698, 41699, 41700, 41701, 41702, 41703, 41704, 41705, 41706, 41707, 41708, 41709, 41710, 41711, 41712, 41713, 41714, 41715, 41716, 41717, 41718, 41719, 41720, 41721, 41722, 41723, 41724, 41725, 41726, 41727, 41728, 41729, 41730, 41731, 41732, 41733, 41734, 41735, 41736, 41737, 41738, 41739, 41740, 41741, 41742, 41743, 41744, 41745, 41746, 41747, 41748, 41749, 41750, 41751, 41752, 41753, 41754, 41755, 41756, 41757, 41758, 41759, 41760, 41761, 41762, 41763, 41764, 41765, 41766, 41767, 41768, 41769, 41770, 41771, 41772, 41773, 41774, 41775, 41776, 41777, 41778, 41779, 41780, 41781, 41782, 41783, 41784, 41785, 41786, 41787, 41788, 41789, 41790, 41791, 41792, 41793, 41794, 41795, 41796, 41797, 41798, 41799, 41800, 41801, 41802, 41803, 41804, 41805, 41806, 41807, 41808, 41809, 41810, 41811, 41812, 41813, 41814, 41815, 41816, 41817, 41818, 41819, 41820, 41821, 41822, 41823, 41824, 41825, 41826, 41827, 41828, 41829, 41830, 41831, 41832, 41833, 41834, 41835, 41836, 41837, 41838, 41839, 41840, 41841, 41842, 41843, 41844, 41845, 41846, 41847, 41848, 41849, 41850, 41851, 41852, 41853, 41854, 41855, 41856, 41857, 41858, 41859, 41860, 41861, 41862, 41863, 41864, 41865, 41866, 41867, 41868, 41869, 41870, 41871, 41872, 41873, 41874, 41875, 41876, 41877, 41878, 41879, 41880, 41881, 41882, 41883, 41884, 41885, 41886, 41887, 41888, 41889, 41890, 41891, 41892, 41893, 41894, 41895, 41896, 41897, 41898, 41899, 41900, 41901, 41902, 41903, 41904, 41905, 41906, 41907, 41908, 41909, 41910, 41911, 41912, 41913, 41914, 41915, 41916, 41917, 41918, 41919, 41920, 41921, 41922, 41923, 41924, 41925, 41926, 41927, 41928, 41929, 41930, 41931, 41932, 41933, 41934, 41935, 41936, 41937, 41938, 41939, 41940, 41941, 41942, 41943, 41944, 41945, 41946, 41947, 41948, 41949, 41950, 41951, 41952, 41953, 41954, 41955, 41956, 41957, 41958, 41959, 41960, 41961, 41962, 41963, 41964, 41965, 41966, 41967, 41968, 41969, 41970, 41971, 41972, 41973, 41974, 41975, 41976, 41977, 41978, 41979, 41980, 41981, 41982, 41983, 41984, 41985, 41986, 41987, 41988, 41989, 41990, 41991, 41992, 41993, 41994, 41995, 41996, 41997, 41998, 41999, 42000, 42001, 42002, 42003, 42004, 42005, 42006, 42007, 42008, 42009, 42010, 42011, 42012, 42013, 42014, 42015, 42016, 42017, 42018, 42019, 42020, 42021, 42022, 42023, 42024, 42025, 42026, 42027, 42028, 42029, 42030, 42031, 42032, 42033, 42034, 42035, 42036, 42037, 42038, 42039, 42040, 42041, 42042, 42043, 42044, 42045, 42046, 42047, 42048, 42049, 42050, 42051, 42052, 42053, 42054, 42055, 42056, 42057, 42058, 42059, 42060, 42061, 42062, 42063, 42064, 42065, 42066, 42067, 42068, 42069, 42070, 42071, 42072, 42073, 42074, 42075, 42076, 42077, 42078, 42079, 42080, 42081, 42082, 42083, 42084, 42085, 42086, 42087, 42088, 42089, 42090, 42091, 42092, 42093, 42094, 42095, 42096, 42097, 42098, 42099, 42100, 42101, 42102, 42103, 42104, 42105, 42106, 42107, 42108, 42109, 42110, 42111, 42112, 42113, 42114, 42115, 42116, 42117, 42118, 42119, 42120, 42121, 42122, 42123, 42124, 42192, 42193, 42194, 42195, 42196, 42197, 42198, 42199, 42200, 42201, 42202, 42203, 42204, 42205, 42206, 42207, 42208, 42209, 42210, 42211, 42212, 42213, 42214, 42215, 42216, 42217, 42218, 42219, 42220, 42221, 42222, 42223, 42224, 42225, 42226, 42227, 42228, 42229, 42230, 42231, 42240, 42241, 42242, 42243, 42244, 42245, 42246, 42247, 42248, 42249, 42250, 42251, 42252, 42253, 42254, 42255, 42256, 42257, 42258, 42259, 42260, 42261, 42262, 42263, 42264, 42265, 42266, 42267, 42268, 42269, 42270, 42271, 42272, 42273, 42274, 42275, 42276, 42277, 42278, 42279, 42280, 42281, 42282, 42283, 42284, 42285, 42286, 42287, 42288, 42289, 42290, 42291, 42292, 42293, 42294, 42295, 42296, 42297, 42298, 42299, 42300, 42301, 42302, 42303, 42304, 42305, 42306, 42307, 42308, 42309, 42310, 42311, 42312, 42313, 42314, 42315, 42316, 42317, 42318, 42319, 42320, 42321, 42322, 42323, 42324, 42325, 42326, 42327, 42328, 42329, 42330, 42331, 42332, 42333, 42334, 42335, 42336, 42337, 42338, 42339, 42340, 42341, 42342, 42343, 42344, 42345, 42346, 42347, 42348, 42349, 42350, 42351, 42352, 42353, 42354, 42355, 42356, 42357, 42358, 42359, 42360, 42361, 42362, 42363, 42364, 42365, 42366, 42367, 42368, 42369, 42370, 42371, 42372, 42373, 42374, 42375, 42376, 42377, 42378, 42379, 42380, 42381, 42382, 42383, 42384, 42385, 42386, 42387, 42388, 42389, 42390, 42391, 42392, 42393, 42394, 42395, 42396, 42397, 42398, 42399, 42400, 42401, 42402, 42403, 42404, 42405, 42406, 42407, 42408, 42409, 42410, 42411, 42412, 42413, 42414, 42415, 42416, 42417, 42418, 42419, 42420, 42421, 42422, 42423, 42424, 42425, 42426, 42427, 42428, 42429, 42430, 42431, 42432, 42433, 42434, 42435, 42436, 42437, 42438, 42439, 42440, 42441, 42442, 42443, 42444, 42445, 42446, 42447, 42448, 42449, 42450, 42451, 42452, 42453, 42454, 42455, 42456, 42457, 42458, 42459, 42460, 42461, 42462, 42463, 42464, 42465, 42466, 42467, 42468, 42469, 42470, 42471, 42472, 42473, 42474, 42475, 42476, 42477, 42478, 42479, 42480, 42481, 42482, 42483, 42484, 42485, 42486, 42487, 42488, 42489, 42490, 42491, 42492, 42493, 42494, 42495, 42496, 42497, 42498, 42499, 42500, 42501, 42502, 42503, 42504, 42505, 42506, 42507, 42512, 42513, 42514, 42515, 42516, 42517, 42518, 42519, 42520, 42521, 42522, 42523, 42524, 42525, 42526, 42527, 42538, 42539, 42606, 42656, 42657, 42658, 42659, 42660, 42661, 42662, 42663, 42664, 42665, 42666, 42667, 42668, 42669, 42670, 42671, 42672, 42673, 42674, 42675, 42676, 42677, 42678, 42679, 42680, 42681, 42682, 42683, 42684, 42685, 42686, 42687, 42688, 42689, 42690, 42691, 42692, 42693, 42694, 42695, 42696, 42697, 42698, 42699, 42700, 42701, 42702, 42703, 42704, 42705, 42706, 42707, 42708, 42709, 42710, 42711, 42712, 42713, 42714, 42715, 42716, 42717, 42718, 42719, 42720, 42721, 42722, 42723, 42724, 42725, 43003, 43004, 43005, 43006, 43007, 43008, 43009, 43011, 43012, 43013, 43015, 43016, 43017, 43018, 43020, 43021, 43022, 43023, 43024, 43025, 43026, 43027, 43028, 43029, 43030, 43031, 43032, 43033, 43034, 43035, 43036, 43037, 43038, 43039, 43040, 43041, 43042, 43072, 43073, 43074, 43075, 43076, 43077, 43078, 43079, 43080, 43081, 43082, 43083, 43084, 43085, 43086, 43087, 43088, 43089, 43090, 43091, 43092, 43093, 43094, 43095, 43096, 43097, 43098, 43099, 43100, 43101, 43102, 43103, 43104, 43105, 43106, 43107, 43108, 43109, 43110, 43111, 43112, 43113, 43114, 43115, 43116, 43117, 43118, 43119, 43120, 43121, 43122, 43123, 43138, 43139, 43140, 43141, 43142, 43143, 43144, 43145, 43146, 43147, 43148, 43149, 43150, 43151, 43152, 43153, 43154, 43155, 43156, 43157, 43158, 43159, 43160, 43161, 43162, 43163, 43164, 43165, 43166, 43167, 43168, 43169, 43170, 43171, 43172, 43173, 43174, 43175, 43176, 43177, 43178, 43179, 43180, 43181, 43182, 43183, 43184, 43185, 43186, 43187, 43250, 43251, 43252, 43253, 43254, 43255, 43259, 43274, 43275, 43276, 43277, 43278, 43279, 43280, 43281, 43282, 43283, 43284, 43285, 43286, 43287, 43288, 43289, 43290, 43291, 43292, 43293, 43294, 43295, 43296, 43297, 43298, 43299, 43300, 43301, 43312, 43313, 43314, 43315, 43316, 43317, 43318, 43319, 43320, 43321, 43322, 43323, 43324, 43325, 43326, 43327, 43328, 43329, 43330, 43331, 43332, 43333, 43334, 43360, 43361, 43362, 43363, 43364, 43365, 43366, 43367, 43368, 43369, 43370, 43371, 43372, 43373, 43374, 43375, 43376, 43377, 43378, 43379, 43380, 43381, 43382, 43383, 43384, 43385, 43386, 43387, 43388, 43396, 43397, 43398, 43399, 43400, 43401, 43402, 43403, 43404, 43405, 43406, 43407, 43408, 43409, 43410, 43411, 43412, 43413, 43414, 43415, 43416, 43417, 43418, 43419, 43420, 43421, 43422, 43423, 43424, 43425, 43426, 43427, 43428, 43429, 43430, 43431, 43432, 43433, 43434, 43435, 43436, 43437, 43438, 43439, 43440, 43441, 43442, 43520, 43521, 43522, 43523, 43524, 43525, 43526, 43527, 43528, 43529, 43530, 43531, 43532, 43533, 43534, 43535, 43536, 43537, 43538, 43539, 43540, 43541, 43542, 43543, 43544, 43545, 43546, 43547, 43548, 43549, 43550, 43551, 43552, 43553, 43554, 43555, 43556, 43557, 43558, 43559, 43560, 43584, 43585, 43586, 43588, 43589, 43590, 43591, 43592, 43593, 43594, 43595, 43616, 43617, 43618, 43619, 43620, 43621, 43622, 43623, 43624, 43625, 43626, 43627, 43628, 43629, 43630, 43631, 43633, 43634, 43635, 43636, 43637, 43638, 43642, 43648, 43649, 43650, 43651, 43652, 43653, 43654, 43655, 43656, 43657, 43658, 43659, 43660, 43661, 43662, 43663, 43664, 43665, 43666, 43667, 43668, 43669, 43670, 43671, 43672, 43673, 43674, 43675, 43676, 43677, 43678, 43679, 43680, 43681, 43682, 43683, 43684, 43685, 43686, 43687, 43688, 43689, 43690, 43691, 43692, 43693, 43694, 43695, 43697, 43701, 43702, 43705, 43706, 43707, 43708, 43709, 43712, 43714, 43739, 43740, 43777, 43778, 43779, 43780, 43781, 43782, 43785, 43786, 43787, 43788, 43789, 43790, 43793, 43794, 43795, 43796, 43797, 43798, 43808, 43809, 43810, 43811, 43812, 43813, 43814, 43816, 43817, 43818, 43819, 43820, 43821, 43822, 43968, 43969, 43970, 43971, 43972, 43973, 43974, 43975, 43976, 43977, 43978, 43979, 43980, 43981, 43982, 43983, 43984, 43985, 43986, 43987, 43988, 43989, 43990, 43991, 43992, 43993, 43994, 43995, 43996, 43997, 43998, 43999, 44000, 44001, 44002, 44032, 55203, 55216, 55217, 55218, 55219, 55220, 55221, 55222, 55223, 55224, 55225, 55226, 55227, 55228, 55229, 55230, 55231, 55232, 55233, 55234, 55235, 55236, 55237, 55238, 55243, 55244, 55245, 55246, 55247, 55248, 55249, 55250, 55251, 55252, 55253, 55254, 55255, 55256, 55257, 55258, 55259, 55260, 55261, 55262, 55263, 55264, 55265, 55266, 55267, 55268, 55269, 55270, 55271, 55272, 55273, 55274, 55275, 55276, 55277, 55278, 55279, 55280, 55281, 55282, 55283, 55284, 55285, 55286, 55287, 55288, 55289, 55290, 55291, 35912, 26356, 36554, 36040, 28369, 20018, 21477, 40860, 40860, 22865, 37329, 21895, 22856, 25078, 30313, 32645, 34367, 34746, 35064, 37007, 27138, 27931, 28889, 29662, 33853, 37226, 39409, 20098, 21365, 27396, 29211, 34349, 40478, 23888, 28651, 34253, 35172, 25289, 33240, 34847, 24266, 26391, 28010, 29436, 37070, 20358, 20919, 21214, 25796, 27347, 29200, 30439, 32769, 34310, 34396, 36335, 38706, 39791, 40442, 30860, 31103, 32160, 33737, 37636, 40575, 35542, 22751, 24324, 31840, 32894, 29282, 30922, 36034, 38647, 22744, 23650, 27155, 28122, 28431, 32047, 32311, 38475, 21202, 32907, 20956, 20940, 31260, 32190, 33777, 38517, 35712, 25295, 27138, 35582, 20025, 23527, 24594, 29575, 30064, 21271, 30971, 20415, 24489, 19981, 27852, 25976, 32034, 21443, 22622, 30465, 33865, 35498, 27578, 36784, 27784, 25342, 33509, 25504, 30053, 20142, 20841, 20937, 26753, 31975, 33391, 35538, 37327, 21237, 21570, 22899, 24300, 26053, 28670, 31018, 38317, 39530, 40599, 40654, 21147, 26310, 27511, 36706, 24180, 24976, 25088, 25754, 28451, 29001, 29833, 31178, 32244, 32879, 36646, 34030, 36899, 37706, 21015, 21155, 21693, 28872, 35010, 35498, 24265, 24565, 25467, 27566, 31806, 29557, 20196, 22265, 23527, 23994, 24604, 29618, 29801, 32666, 32838, 37428, 38646, 38728, 38936, 20363, 31150, 37300, 38584, 24801, 20102, 20698, 23534, 23615, 26009, 27138, 29134, 30274, 34044, 36988, 40845, 26248, 38446, 21129, 26491, 26611, 27969, 28316, 29705, 30041, 30827, 32016, 39006, 20845, 25134, 38520, 20523, 23833, 28138, 36650, 24459, 24900, 26647, 29575, 38534, 21033, 21519, 23653, 26131, 26446, 26792, 27877, 29702, 30178, 32633, 35023, 35041, 37324, 38626, 21311, 28346, 21533, 29136, 29848, 34298, 38563, 40023, 40607, 26519, 28107, 33256, 31435, 31520, 31890, 29376, 28825, 35672, 20160, 33590, 21050, 20999, 24230, 25299, 31958, 23429, 27934, 26292, 36667, 34892, 38477, 35211, 24275, 20800, 21952, 64014, 64015, 22618, 64017, 26228, 64019, 64020, 20958, 29482, 30410, 31036, 31070, 31077, 31119, 38742, 31934, 32701, 64031, 34322, 64033, 35576, 64035, 64036, 36920, 37117, 64039, 64040, 64041, 39151, 39164, 39208, 40372, 20398, 20711, 20813, 21193, 21220, 21329, 21917, 22022, 22120, 22592, 22696, 23652, 23662, 24724, 24936, 24974, 25074, 25935, 26082, 26257, 26757, 28023, 28186, 28450, 29038, 29227, 29730, 30865, 31038, 31049, 31048, 31056, 31062, 31069, 31117, 31118, 31296, 31361, 31680, 32244, 32265, 32321, 32626, 32773, 33261, 33401, 33401, 33879, 35088, 35222, 35585, 35641, 36051, 36104, 36790, 36920, 38627, 38911, 38971, 24693, 148206, 33304, 20006, 20917, 20840, 20352, 20805, 20864, 21191, 21242, 21917, 21845, 21913, 21986, 22618, 22707, 22852, 22868, 23138, 23336, 24274, 24281, 24425, 24493, 24792, 24910, 24840, 24974, 24928, 25074, 25140, 25540, 25628, 25682, 25942, 26228, 26391, 26395, 26454, 27513, 27578, 27969, 28379, 28363, 28450, 28702, 29038, 30631, 29237, 29359, 29482, 29809, 29958, 30011, 30237, 30239, 30410, 30427, 30452, 30538, 30528, 30924, 31409, 31680, 31867, 32091, 32244, 32574, 32773, 33618, 33775, 34681, 35137, 35206, 35222, 35519, 35576, 35531, 35585, 35582, 35565, 35641, 35722, 36104, 36664, 36978, 37273, 37494, 38524, 38627, 38742, 38875, 38911, 38923, 38971, 39698, 40860, 141386, 141380, 144341, 15261, 16408, 16441, 152137, 154832, 163539, 40771, 40846, 1497, 1460, 1522, 1463, 64288, 64289, 64290, 64291, 64292, 64293, 64294, 64295, 64296, 1513, 1473, 1513, 1474, 1513, 1468, 1473, 1513, 1468, 1474, 1488, 1463, 1488, 1464, 1488, 1468, 1489, 1468, 1490, 1468, 1491, 1468, 1492, 1468, 1493, 1468, 1494, 1468, 1496, 1468, 1497, 1468, 1498, 1468, 1499, 1468, 1500, 1468, 1502, 1468, 1504, 1468, 1505, 1468, 1507, 1468, 1508, 1468, 1510, 1468, 1511, 1468, 1512, 1468, 1513, 1468, 1514, 1468, 1493, 1465, 1489, 1471, 1499, 1471, 1508, 1471, 64335, 64336, 64337, 64338, 64339, 64340, 64341, 64342, 64343, 64344, 64345, 64346, 64347, 64348, 64349, 64350, 64351, 64352, 64353, 64354, 64355, 64356, 64357, 64358, 64359, 64360, 64361, 64362, 64363, 64364, 64365, 64366, 64367, 64368, 64369, 64370, 64371, 64372, 64373, 64374, 64375, 64376, 64377, 64378, 64379, 64380, 64381, 64382, 64383, 64384, 64385, 64386, 64387, 64388, 64389, 64390, 64391, 64392, 64393, 64394, 64395, 64396, 64397, 64398, 64399, 64400, 64401, 64402, 64403, 64404, 64405, 64406, 64407, 64408, 64409, 64410, 64411, 64412, 64413, 64414, 64415, 64416, 64417, 64418, 64419, 64420, 64421, 64422, 64423, 64424, 64425, 64426, 64427, 64428, 64429, 64430, 64431, 64432, 64433, 64467, 64468, 64469, 64470, 64471, 64472, 64473, 64474, 64475, 64476, 64477, 64478, 64479, 64480, 64481, 64482, 64483, 64484, 64485, 64486, 64487, 64488, 64489, 64490, 64491, 64492, 64493, 64494, 64495, 64496, 64497, 64498, 64499, 64500, 64501, 64502, 64503, 64504, 64505, 64506, 64507, 64508, 64509, 64510, 64511, 64512, 64513, 64514, 64515, 64516, 64517, 64518, 64519, 64520, 64521, 64522, 64523, 64524, 64525, 64526, 64527, 64528, 64529, 64530, 64531, 64532, 64533, 64534, 64535, 64536, 64537, 64538, 64539, 64540, 64541, 64542, 64543, 64544, 64545, 64546, 64547, 64548, 64549, 64550, 64551, 64552, 64553, 64554, 64555, 64556, 64557, 64558, 64559, 64560, 64561, 64562, 64563, 64564, 64565, 64566, 64567, 64568, 64569, 64570, 64571, 64572, 64573, 64574, 64575, 64576, 64577, 64578, 64579, 64580, 64581, 64582, 64583, 64584, 64585, 64586, 64587, 64588, 64589, 64590, 64591, 64592, 64593, 64594, 64595, 64596, 64597, 64598, 64599, 64600, 64601, 64602, 64603, 64604, 64605, 64606, 64607, 64608, 64609, 64610, 64611, 64612, 64613, 64614, 64615, 64616, 64617, 64618, 64619, 64620, 64621, 64622, 64623, 64624, 64625, 64626, 64627, 64628, 64629, 64630, 64631, 64632, 64633, 64634, 64635, 64636, 64637, 64638, 64639, 64640, 64641, 64642, 64643, 64644, 64645, 64646, 64647, 64648, 64649, 64650, 64651, 64652, 64653, 64654, 64655, 64656, 64657, 64658, 64659, 64660, 64661, 64662, 64663, 64664, 64665, 64666, 64667, 64668, 64669, 64670, 64671, 64672, 64673, 64674, 64675, 64676, 64677, 64678, 64679, 64680, 64681, 64682, 64683, 64684, 64685, 64686, 64687, 64688, 64689, 64690, 64691, 64692, 64693, 64694, 64695, 64696, 64697, 64698, 64699, 64700, 64701, 64702, 64703, 64704, 64705, 64706, 64707, 64708, 64709, 64710, 64711, 64712, 64713, 64714, 64715, 64716, 64717, 64718, 64719, 64720, 64721, 64722, 64723, 64724, 64725, 64726, 64727, 64728, 64729, 64730, 64731, 64732, 64733, 64734, 64735, 64736, 64737, 64738, 64739, 64740, 64741, 64742, 64743, 64744, 64745, 64746, 64747, 64748, 64749, 64750, 64751, 64752, 64753, 64754, 64755, 64756, 64757, 64758, 64759, 64760, 64761, 64762, 64763, 64764, 64765, 64766, 64767, 64768, 64769, 64770, 64771, 64772, 64773, 64774, 64775, 64776, 64777, 64778, 64779, 64780, 64781, 64782, 64783, 64784, 64785, 64786, 64787, 64788, 64789, 64790, 64791, 64792, 64793, 64794, 64795, 64796, 64797, 64798, 64799, 64800, 64801, 64802, 64803, 64804, 64805, 64806, 64807, 64808, 64809, 64810, 64811, 64812, 64813, 64814, 64815, 64816, 64817, 64818, 64819, 64820, 64821, 64822, 64823, 64824, 64825, 64826, 64827, 64828, 64829, 64848, 64849, 64850, 64851, 64852, 64853, 64854, 64855, 64856, 64857, 64858, 64859, 64860, 64861, 64862, 64863, 64864, 64865, 64866, 64867, 64868, 64869, 64870, 64871, 64872, 64873, 64874, 64875, 64876, 64877, 64878, 64879, 64880, 64881, 64882, 64883, 64884, 64885, 64886, 64887, 64888, 64889, 64890, 64891, 64892, 64893, 64894, 64895, 64896, 64897, 64898, 64899, 64900, 64901, 64902, 64903, 64904, 64905, 64906, 64907, 64908, 64909, 64910, 64911, 64914, 64915, 64916, 64917, 64918, 64919, 64920, 64921, 64922, 64923, 64924, 64925, 64926, 64927, 64928, 64929, 64930, 64931, 64932, 64933, 64934, 64935, 64936, 64937, 64938, 64939, 64940, 64941, 64942, 64943, 64944, 64945, 64946, 64947, 64948, 64949, 64950, 64951, 64952, 64953, 64954, 64955, 64956, 64957, 64958, 64959, 64960, 64961, 64962, 64963, 64964, 64965, 64966, 64967, 65008, 65009, 65010, 65011, 65012, 65013, 65014, 65015, 65016, 65017, 65018, 65019, 65136, 65137, 65138, 65139, 65140, 65142, 65143, 65144, 65145, 65146, 65147, 65148, 65149, 65150, 65151, 65152, 65153, 65154, 65155, 65156, 65157, 65158, 65159, 65160, 65161, 65162, 65163, 65164, 65165, 65166, 65167, 65168, 65169, 65170, 65171, 65172, 65173, 65174, 65175, 65176, 65177, 65178, 65179, 65180, 65181, 65182, 65183, 65184, 65185, 65186, 65187, 65188, 65189, 65190, 65191, 65192, 65193, 65194, 65195, 65196, 65197, 65198, 65199, 65200, 65201, 65202, 65203, 65204, 65205, 65206, 65207, 65208, 65209, 65210, 65211, 65212, 65213, 65214, 65215, 65216, 65217, 65218, 65219, 65220, 65221, 65222, 65223, 65224, 65225, 65226, 65227, 65228, 65229, 65230, 65231, 65232, 65233, 65234, 65235, 65236, 65237, 65238, 65239, 65240, 65241, 65242, 65243, 65244, 65245, 65246, 65247, 65248, 65249, 65250, 65251, 65252, 65253, 65254, 65255, 65256, 65257, 65258, 65259, 65260, 65261, 65262, 65263, 65264, 65265, 65266, 65267, 65268, 65269, 65270, 65271, 65272, 65273, 65274, 65275, 65276, 65382, 65383, 65384, 65385, 65386, 65387, 65388, 65389, 65390, 65391, 65393, 65394, 65395, 65396, 65397, 65398, 65399, 65400, 65401, 65402, 65403, 65404, 65405, 65406, 65407, 65408, 65409, 65410, 65411, 65412, 65413, 65414, 65415, 65416, 65417, 65418, 65419, 65420, 65421, 65422, 65423, 65424, 65425, 65426, 65427, 65428, 65429, 65430, 65431, 65432, 65433, 65434, 65435, 65436, 65437, 65440, 65441, 65442, 65443, 65444, 65445, 65446, 65447, 65448, 65449, 65450, 65451, 65452, 65453, 65454, 65455, 65456, 65457, 65458, 65459, 65460, 65461, 65462, 65463, 65464, 65465, 65466, 65467, 65468, 65469, 65470, 65474, 65475, 65476, 65477, 65478, 65479, 65482, 65483, 65484, 65485, 65486, 65487, 65490, 65491, 65492, 65493, 65494, 65495, 65498, 65499, 65500, 65536, 65537, 65538, 65539, 65540, 65541, 65542, 65543, 65544, 65545, 65546, 65547, 65549, 65550, 65551, 65552, 65553, 65554, 65555, 65556, 65557, 65558, 65559, 65560, 65561, 65562, 65563, 65564, 65565, 65566, 65567, 65568, 65569, 65570, 65571, 65572, 65573, 65574, 65576, 65577, 65578, 65579, 65580, 65581, 65582, 65583, 65584, 65585, 65586, 65587, 65588, 65589, 65590, 65591, 65592, 65593, 65594, 65596, 65597, 65599, 65600, 65601, 65602, 65603, 65604, 65605, 65606, 65607, 65608, 65609, 65610, 65611, 65612, 65613, 65616, 65617, 65618, 65619, 65620, 65621, 65622, 65623, 65624, 65625, 65626, 65627, 65628, 65629, 65664, 65665, 65666, 65667, 65668, 65669, 65670, 65671, 65672, 65673, 65674, 65675, 65676, 65677, 65678, 65679, 65680, 65681, 65682, 65683, 65684, 65685, 65686, 65687, 65688, 65689, 65690, 65691, 65692, 65693, 65694, 65695, 65696, 65697, 65698, 65699, 65700, 65701, 65702, 65703, 65704, 65705, 65706, 65707, 65708, 65709, 65710, 65711, 65712, 65713, 65714, 65715, 65716, 65717, 65718, 65719, 65720, 65721, 65722, 65723, 65724, 65725, 65726, 65727, 65728, 65729, 65730, 65731, 65732, 65733, 65734, 65735, 65736, 65737, 65738, 65739, 65740, 65741, 65742, 65743, 65744, 65745, 65746, 65747, 65748, 65749, 65750, 65751, 65752, 65753, 65754, 65755, 65756, 65757, 65758, 65759, 65760, 65761, 65762, 65763, 65764, 65765, 65766, 65767, 65768, 65769, 65770, 65771, 65772, 65773, 65774, 65775, 65776, 65777, 65778, 65779, 65780, 65781, 65782, 65783, 65784, 65785, 65786, 66176, 66177, 66178, 66179, 66180, 66181, 66182, 66183, 66184, 66185, 66186, 66187, 66188, 66189, 66190, 66191, 66192, 66193, 66194, 66195, 66196, 66197, 66198, 66199, 66200, 66201, 66202, 66203, 66204, 66208, 66209, 66210, 66211, 66212, 66213, 66214, 66215, 66216, 66217, 66218, 66219, 66220, 66221, 66222, 66223, 66224, 66225, 66226, 66227, 66228, 66229, 66230, 66231, 66232, 66233, 66234, 66235, 66236, 66237, 66238, 66239, 66240, 66241, 66242, 66243, 66244, 66245, 66246, 66247, 66248, 66249, 66250, 66251, 66252, 66253, 66254, 66255, 66256, 66304, 66305, 66306, 66307, 66308, 66309, 66310, 66311, 66312, 66313, 66314, 66315, 66316, 66317, 66318, 66319, 66320, 66321, 66322, 66323, 66324, 66325, 66326, 66327, 66328, 66329, 66330, 66331, 66332, 66333, 66334, 66352, 66353, 66354, 66355, 66356, 66357, 66358, 66359, 66360, 66361, 66362, 66363, 66364, 66365, 66366, 66367, 66368, 66370, 66371, 66372, 66373, 66374, 66375, 66376, 66377, 66432, 66433, 66434, 66435, 66436, 66437, 66438, 66439, 66440, 66441, 66442, 66443, 66444, 66445, 66446, 66447, 66448, 66449, 66450, 66451, 66452, 66453, 66454, 66455, 66456, 66457, 66458, 66459, 66460, 66461, 66464, 66465, 66466, 66467, 66468, 66469, 66470, 66471, 66472, 66473, 66474, 66475, 66476, 66477, 66478, 66479, 66480, 66481, 66482, 66483, 66484, 66485, 66486, 66487, 66488, 66489, 66490, 66491, 66492, 66493, 66494, 66495, 66496, 66497, 66498, 66499, 66504, 66505, 66506, 66507, 66508, 66509, 66510, 66511, 66640, 66641, 66642, 66643, 66644, 66645, 66646, 66647, 66648, 66649, 66650, 66651, 66652, 66653, 66654, 66655, 66656, 66657, 66658, 66659, 66660, 66661, 66662, 66663, 66664, 66665, 66666, 66667, 66668, 66669, 66670, 66671, 66672, 66673, 66674, 66675, 66676, 66677, 66678, 66679, 66680, 66681, 66682, 66683, 66684, 66685, 66686, 66687, 66688, 66689, 66690, 66691, 66692, 66693, 66694, 66695, 66696, 66697, 66698, 66699, 66700, 66701, 66702, 66703, 66704, 66705, 66706, 66707, 66708, 66709, 66710, 66711, 66712, 66713, 66714, 66715, 66716, 66717, 67584, 67585, 67586, 67587, 67588, 67589, 67592, 67594, 67595, 67596, 67597, 67598, 67599, 67600, 67601, 67602, 67603, 67604, 67605, 67606, 67607, 67608, 67609, 67610, 67611, 67612, 67613, 67614, 67615, 67616, 67617, 67618, 67619, 67620, 67621, 67622, 67623, 67624, 67625, 67626, 67627, 67628, 67629, 67630, 67631, 67632, 67633, 67634, 67635, 67636, 67637, 67639, 67640, 67644, 67647, 67648, 67649, 67650, 67651, 67652, 67653, 67654, 67655, 67656, 67657, 67658, 67659, 67660, 67661, 67662, 67663, 67664, 67665, 67666, 67667, 67668, 67669, 67840, 67841, 67842, 67843, 67844, 67845, 67846, 67847, 67848, 67849, 67850, 67851, 67852, 67853, 67854, 67855, 67856, 67857, 67858, 67859, 67860, 67861, 67872, 67873, 67874, 67875, 67876, 67877, 67878, 67879, 67880, 67881, 67882, 67883, 67884, 67885, 67886, 67887, 67888, 67889, 67890, 67891, 67892, 67893, 67894, 67895, 67896, 67897, 68096, 68112, 68113, 68114, 68115, 68117, 68118, 68119, 68121, 68122, 68123, 68124, 68125, 68126, 68127, 68128, 68129, 68130, 68131, 68132, 68133, 68134, 68135, 68136, 68137, 68138, 68139, 68140, 68141, 68142, 68143, 68144, 68145, 68146, 68147, 68192, 68193, 68194, 68195, 68196, 68197, 68198, 68199, 68200, 68201, 68202, 68203, 68204, 68205, 68206, 68207, 68208, 68209, 68210, 68211, 68212, 68213, 68214, 68215, 68216, 68217, 68218, 68219, 68220, 68352, 68353, 68354, 68355, 68356, 68357, 68358, 68359, 68360, 68361, 68362, 68363, 68364, 68365, 68366, 68367, 68368, 68369, 68370, 68371, 68372, 68373, 68374, 68375, 68376, 68377, 68378, 68379, 68380, 68381, 68382, 68383, 68384, 68385, 68386, 68387, 68388, 68389, 68390, 68391, 68392, 68393, 68394, 68395, 68396, 68397, 68398, 68399, 68400, 68401, 68402, 68403, 68404, 68405, 68416, 68417, 68418, 68419, 68420, 68421, 68422, 68423, 68424, 68425, 68426, 68427, 68428, 68429, 68430, 68431, 68432, 68433, 68434, 68435, 68436, 68437, 68448, 68449, 68450, 68451, 68452, 68453, 68454, 68455, 68456, 68457, 68458, 68459, 68460, 68461, 68462, 68463, 68464, 68465, 68466, 68608, 68609, 68610, 68611, 68612, 68613, 68614, 68615, 68616, 68617, 68618, 68619, 68620, 68621, 68622, 68623, 68624, 68625, 68626, 68627, 68628, 68629, 68630, 68631, 68632, 68633, 68634, 68635, 68636, 68637, 68638, 68639, 68640, 68641, 68642, 68643, 68644, 68645, 68646, 68647, 68648, 68649, 68650, 68651, 68652, 68653, 68654, 68655, 68656, 68657, 68658, 68659, 68660, 68661, 68662, 68663, 68664, 68665, 68666, 68667, 68668, 68669, 68670, 68671, 68672, 68673, 68674, 68675, 68676, 68677, 68678, 68679, 68680, 69635, 69636, 69637, 69638, 69639, 69640, 69641, 69642, 69643, 69644, 69645, 69646, 69647, 69648, 69649, 69650, 69651, 69652, 69653, 69654, 69655, 69656, 69657, 69658, 69659, 69660, 69661, 69662, 69663, 69664, 69665, 69666, 69667, 69668, 69669, 69670, 69671, 69672, 69673, 69674, 69675, 69676, 69677, 69678, 69679, 69680, 69681, 69682, 69683, 69684, 69685, 69686, 69687, 69763, 69764, 69765, 69766, 69767, 69768, 69769, 69770, 69771, 69772, 69773, 69774, 69775, 69776, 69777, 69778, 69779, 69780, 69781, 69782, 69783, 69784, 69785, 69786, 69787, 69788, 69789, 69790, 69791, 69792, 69793, 69794, 69795, 69796, 69797, 69798, 69799, 69800, 69801, 69802, 69803, 69804, 69805, 69806, 69807, 73728, 73729, 73730, 73731, 73732, 73733, 73734, 73735, 73736, 73737, 73738, 73739, 73740, 73741, 73742, 73743, 73744, 73745, 73746, 73747, 73748, 73749, 73750, 73751, 73752, 73753, 73754, 73755, 73756, 73757, 73758, 73759, 73760, 73761, 73762, 73763, 73764, 73765, 73766, 73767, 73768, 73769, 73770, 73771, 73772, 73773, 73774, 73775, 73776, 73777, 73778, 73779, 73780, 73781, 73782, 73783, 73784, 73785, 73786, 73787, 73788, 73789, 73790, 73791, 73792, 73793, 73794, 73795, 73796, 73797, 73798, 73799, 73800, 73801, 73802, 73803, 73804, 73805, 73806, 73807, 73808, 73809, 73810, 73811, 73812, 73813, 73814, 73815, 73816, 73817, 73818, 73819, 73820, 73821, 73822, 73823, 73824, 73825, 73826, 73827, 73828, 73829, 73830, 73831, 73832, 73833, 73834, 73835, 73836, 73837, 73838, 73839, 73840, 73841, 73842, 73843, 73844, 73845, 73846, 73847, 73848, 73849, 73850, 73851, 73852, 73853, 73854, 73855, 73856, 73857, 73858, 73859, 73860, 73861, 73862, 73863, 73864, 73865, 73866, 73867, 73868, 73869, 73870, 73871, 73872, 73873, 73874, 73875, 73876, 73877, 73878, 73879, 73880, 73881, 73882, 73883, 73884, 73885, 73886, 73887, 73888, 73889, 73890, 73891, 73892, 73893, 73894, 73895, 73896, 73897, 73898, 73899, 73900, 73901, 73902, 73903, 73904, 73905, 73906, 73907, 73908, 73909, 73910, 73911, 73912, 73913, 73914, 73915, 73916, 73917, 73918, 73919, 73920, 73921, 73922, 73923, 73924, 73925, 73926, 73927, 73928, 73929, 73930, 73931, 73932, 73933, 73934, 73935, 73936, 73937, 73938, 73939, 73940, 73941, 73942, 73943, 73944, 73945, 73946, 73947, 73948, 73949, 73950, 73951, 73952, 73953, 73954, 73955, 73956, 73957, 73958, 73959, 73960, 73961, 73962, 73963, 73964, 73965, 73966, 73967, 73968, 73969, 73970, 73971, 73972, 73973, 73974, 73975, 73976, 73977, 73978, 73979, 73980, 73981, 73982, 73983, 73984, 73985, 73986, 73987, 73988, 73989, 73990, 73991, 73992, 73993, 73994, 73995, 73996, 73997, 73998, 73999, 74000, 74001, 74002, 74003, 74004, 74005, 74006, 74007, 74008, 74009, 74010, 74011, 74012, 74013, 74014, 74015, 74016, 74017, 74018, 74019, 74020, 74021, 74022, 74023, 74024, 74025, 74026, 74027, 74028, 74029, 74030, 74031, 74032, 74033, 74034, 74035, 74036, 74037, 74038, 74039, 74040, 74041, 74042, 74043, 74044, 74045, 74046, 74047, 74048, 74049, 74050, 74051, 74052, 74053, 74054, 74055, 74056, 74057, 74058, 74059, 74060, 74061, 74062, 74063, 74064, 74065, 74066, 74067, 74068, 74069, 74070, 74071, 74072, 74073, 74074, 74075, 74076, 74077, 74078, 74079, 74080, 74081, 74082, 74083, 74084, 74085, 74086, 74087, 74088, 74089, 74090, 74091, 74092, 74093, 74094, 74095, 74096, 74097, 74098, 74099, 74100, 74101, 74102, 74103, 74104, 74105, 74106, 74107, 74108, 74109, 74110, 74111, 74112, 74113, 74114, 74115, 74116, 74117, 74118, 74119, 74120, 74121, 74122, 74123, 74124, 74125, 74126, 74127, 74128, 74129, 74130, 74131, 74132, 74133, 74134, 74135, 74136, 74137, 74138, 74139, 74140, 74141, 74142, 74143, 74144, 74145, 74146, 74147, 74148, 74149, 74150, 74151, 74152, 74153, 74154, 74155, 74156, 74157, 74158, 74159, 74160, 74161, 74162, 74163, 74164, 74165, 74166, 74167, 74168, 74169, 74170, 74171, 74172, 74173, 74174, 74175, 74176, 74177, 74178, 74179, 74180, 74181, 74182, 74183, 74184, 74185, 74186, 74187, 74188, 74189, 74190, 74191, 74192, 74193, 74194, 74195, 74196, 74197, 74198, 74199, 74200, 74201, 74202, 74203, 74204, 74205, 74206, 74207, 74208, 74209, 74210, 74211, 74212, 74213, 74214, 74215, 74216, 74217, 74218, 74219, 74220, 74221, 74222, 74223, 74224, 74225, 74226, 74227, 74228, 74229, 74230, 74231, 74232, 74233, 74234, 74235, 74236, 74237, 74238, 74239, 74240, 74241, 74242, 74243, 74244, 74245, 74246, 74247, 74248, 74249, 74250, 74251, 74252, 74253, 74254, 74255, 74256, 74257, 74258, 74259, 74260, 74261, 74262, 74263, 74264, 74265, 74266, 74267, 74268, 74269, 74270, 74271, 74272, 74273, 74274, 74275, 74276, 74277, 74278, 74279, 74280, 74281, 74282, 74283, 74284, 74285, 74286, 74287, 74288, 74289, 74290, 74291, 74292, 74293, 74294, 74295, 74296, 74297, 74298, 74299, 74300, 74301, 74302, 74303, 74304, 74305, 74306, 74307, 74308, 74309, 74310, 74311, 74312, 74313, 74314, 74315, 74316, 74317, 74318, 74319, 74320, 74321, 74322, 74323, 74324, 74325, 74326, 74327, 74328, 74329, 74330, 74331, 74332, 74333, 74334, 74335, 74336, 74337, 74338, 74339, 74340, 74341, 74342, 74343, 74344, 74345, 74346, 74347, 74348, 74349, 74350, 74351, 74352, 74353, 74354, 74355, 74356, 74357, 74358, 74359, 74360, 74361, 74362, 74363, 74364, 74365, 74366, 74367, 74368, 74369, 74370, 74371, 74372, 74373, 74374, 74375, 74376, 74377, 74378, 74379, 74380, 74381, 74382, 74383, 74384, 74385, 74386, 74387, 74388, 74389, 74390, 74391, 74392, 74393, 74394, 74395, 74396, 74397, 74398, 74399, 74400, 74401, 74402, 74403, 74404, 74405, 74406, 74407, 74408, 74409, 74410, 74411, 74412, 74413, 74414, 74415, 74416, 74417, 74418, 74419, 74420, 74421, 74422, 74423, 74424, 74425, 74426, 74427, 74428, 74429, 74430, 74431, 74432, 74433, 74434, 74435, 74436, 74437, 74438, 74439, 74440, 74441, 74442, 74443, 74444, 74445, 74446, 74447, 74448, 74449, 74450, 74451, 74452, 74453, 74454, 74455, 74456, 74457, 74458, 74459, 74460, 74461, 74462, 74463, 74464, 74465, 74466, 74467, 74468, 74469, 74470, 74471, 74472, 74473, 74474, 74475, 74476, 74477, 74478, 74479, 74480, 74481, 74482, 74483, 74484, 74485, 74486, 74487, 74488, 74489, 74490, 74491, 74492, 74493, 74494, 74495, 74496, 74497, 74498, 74499, 74500, 74501, 74502, 74503, 74504, 74505, 74506, 74507, 74508, 74509, 74510, 74511, 74512, 74513, 74514, 74515, 74516, 74517, 74518, 74519, 74520, 74521, 74522, 74523, 74524, 74525, 74526, 74527, 74528, 74529, 74530, 74531, 74532, 74533, 74534, 74535, 74536, 74537, 74538, 74539, 74540, 74541, 74542, 74543, 74544, 74545, 74546, 74547, 74548, 74549, 74550, 74551, 74552, 74553, 74554, 74555, 74556, 74557, 74558, 74559, 74560, 74561, 74562, 74563, 74564, 74565, 74566, 74567, 74568, 74569, 74570, 74571, 74572, 74573, 74574, 74575, 74576, 74577, 74578, 74579, 74580, 74581, 74582, 74583, 74584, 74585, 74586, 74587, 74588, 74589, 74590, 74591, 74592, 74593, 74594, 74595, 74596, 74597, 74598, 74599, 74600, 74601, 74602, 74603, 74604, 74605, 74606, 77824, 77825, 77826, 77827, 77828, 77829, 77830, 77831, 77832, 77833, 77834, 77835, 77836, 77837, 77838, 77839, 77840, 77841, 77842, 77843, 77844, 77845, 77846, 77847, 77848, 77849, 77850, 77851, 77852, 77853, 77854, 77855, 77856, 77857, 77858, 77859, 77860, 77861, 77862, 77863, 77864, 77865, 77866, 77867, 77868, 77869, 77870, 77871, 77872, 77873, 77874, 77875, 77876, 77877, 77878, 77879, 77880, 77881, 77882, 77883, 77884, 77885, 77886, 77887, 77888, 77889, 77890, 77891, 77892, 77893, 77894, 77895, 77896, 77897, 77898, 77899, 77900, 77901, 77902, 77903, 77904, 77905, 77906, 77907, 77908, 77909, 77910, 77911, 77912, 77913, 77914, 77915, 77916, 77917, 77918, 77919, 77920, 77921, 77922, 77923, 77924, 77925, 77926, 77927, 77928, 77929, 77930, 77931, 77932, 77933, 77934, 77935, 77936, 77937, 77938, 77939, 77940, 77941, 77942, 77943, 77944, 77945, 77946, 77947, 77948, 77949, 77950, 77951, 77952, 77953, 77954, 77955, 77956, 77957, 77958, 77959, 77960, 77961, 77962, 77963, 77964, 77965, 77966, 77967, 77968, 77969, 77970, 77971, 77972, 77973, 77974, 77975, 77976, 77977, 77978, 77979, 77980, 77981, 77982, 77983, 77984, 77985, 77986, 77987, 77988, 77989, 77990, 77991, 77992, 77993, 77994, 77995, 77996, 77997, 77998, 77999, 78000, 78001, 78002, 78003, 78004, 78005, 78006, 78007, 78008, 78009, 78010, 78011, 78012, 78013, 78014, 78015, 78016, 78017, 78018, 78019, 78020, 78021, 78022, 78023, 78024, 78025, 78026, 78027, 78028, 78029, 78030, 78031, 78032, 78033, 78034, 78035, 78036, 78037, 78038, 78039, 78040, 78041, 78042, 78043, 78044, 78045, 78046, 78047, 78048, 78049, 78050, 78051, 78052, 78053, 78054, 78055, 78056, 78057, 78058, 78059, 78060, 78061, 78062, 78063, 78064, 78065, 78066, 78067, 78068, 78069, 78070, 78071, 78072, 78073, 78074, 78075, 78076, 78077, 78078, 78079, 78080, 78081, 78082, 78083, 78084, 78085, 78086, 78087, 78088, 78089, 78090, 78091, 78092, 78093, 78094, 78095, 78096, 78097, 78098, 78099, 78100, 78101, 78102, 78103, 78104, 78105, 78106, 78107, 78108, 78109, 78110, 78111, 78112, 78113, 78114, 78115, 78116, 78117, 78118, 78119, 78120, 78121, 78122, 78123, 78124, 78125, 78126, 78127, 78128, 78129, 78130, 78131, 78132, 78133, 78134, 78135, 78136, 78137, 78138, 78139, 78140, 78141, 78142, 78143, 78144, 78145, 78146, 78147, 78148, 78149, 78150, 78151, 78152, 78153, 78154, 78155, 78156, 78157, 78158, 78159, 78160, 78161, 78162, 78163, 78164, 78165, 78166, 78167, 78168, 78169, 78170, 78171, 78172, 78173, 78174, 78175, 78176, 78177, 78178, 78179, 78180, 78181, 78182, 78183, 78184, 78185, 78186, 78187, 78188, 78189, 78190, 78191, 78192, 78193, 78194, 78195, 78196, 78197, 78198, 78199, 78200, 78201, 78202, 78203, 78204, 78205, 78206, 78207, 78208, 78209, 78210, 78211, 78212, 78213, 78214, 78215, 78216, 78217, 78218, 78219, 78220, 78221, 78222, 78223, 78224, 78225, 78226, 78227, 78228, 78229, 78230, 78231, 78232, 78233, 78234, 78235, 78236, 78237, 78238, 78239, 78240, 78241, 78242, 78243, 78244, 78245, 78246, 78247, 78248, 78249, 78250, 78251, 78252, 78253, 78254, 78255, 78256, 78257, 78258, 78259, 78260, 78261, 78262, 78263, 78264, 78265, 78266, 78267, 78268, 78269, 78270, 78271, 78272, 78273, 78274, 78275, 78276, 78277, 78278, 78279, 78280, 78281, 78282, 78283, 78284, 78285, 78286, 78287, 78288, 78289, 78290, 78291, 78292, 78293, 78294, 78295, 78296, 78297, 78298, 78299, 78300, 78301, 78302, 78303, 78304, 78305, 78306, 78307, 78308, 78309, 78310, 78311, 78312, 78313, 78314, 78315, 78316, 78317, 78318, 78319, 78320, 78321, 78322, 78323, 78324, 78325, 78326, 78327, 78328, 78329, 78330, 78331, 78332, 78333, 78334, 78335, 78336, 78337, 78338, 78339, 78340, 78341, 78342, 78343, 78344, 78345, 78346, 78347, 78348, 78349, 78350, 78351, 78352, 78353, 78354, 78355, 78356, 78357, 78358, 78359, 78360, 78361, 78362, 78363, 78364, 78365, 78366, 78367, 78368, 78369, 78370, 78371, 78372, 78373, 78374, 78375, 78376, 78377, 78378, 78379, 78380, 78381, 78382, 78383, 78384, 78385, 78386, 78387, 78388, 78389, 78390, 78391, 78392, 78393, 78394, 78395, 78396, 78397, 78398, 78399, 78400, 78401, 78402, 78403, 78404, 78405, 78406, 78407, 78408, 78409, 78410, 78411, 78412, 78413, 78414, 78415, 78416, 78417, 78418, 78419, 78420, 78421, 78422, 78423, 78424, 78425, 78426, 78427, 78428, 78429, 78430, 78431, 78432, 78433, 78434, 78435, 78436, 78437, 78438, 78439, 78440, 78441, 78442, 78443, 78444, 78445, 78446, 78447, 78448, 78449, 78450, 78451, 78452, 78453, 78454, 78455, 78456, 78457, 78458, 78459, 78460, 78461, 78462, 78463, 78464, 78465, 78466, 78467, 78468, 78469, 78470, 78471, 78472, 78473, 78474, 78475, 78476, 78477, 78478, 78479, 78480, 78481, 78482, 78483, 78484, 78485, 78486, 78487, 78488, 78489, 78490, 78491, 78492, 78493, 78494, 78495, 78496, 78497, 78498, 78499, 78500, 78501, 78502, 78503, 78504, 78505, 78506, 78507, 78508, 78509, 78510, 78511, 78512, 78513, 78514, 78515, 78516, 78517, 78518, 78519, 78520, 78521, 78522, 78523, 78524, 78525, 78526, 78527, 78528, 78529, 78530, 78531, 78532, 78533, 78534, 78535, 78536, 78537, 78538, 78539, 78540, 78541, 78542, 78543, 78544, 78545, 78546, 78547, 78548, 78549, 78550, 78551, 78552, 78553, 78554, 78555, 78556, 78557, 78558, 78559, 78560, 78561, 78562, 78563, 78564, 78565, 78566, 78567, 78568, 78569, 78570, 78571, 78572, 78573, 78574, 78575, 78576, 78577, 78578, 78579, 78580, 78581, 78582, 78583, 78584, 78585, 78586, 78587, 78588, 78589, 78590, 78591, 78592, 78593, 78594, 78595, 78596, 78597, 78598, 78599, 78600, 78601, 78602, 78603, 78604, 78605, 78606, 78607, 78608, 78609, 78610, 78611, 78612, 78613, 78614, 78615, 78616, 78617, 78618, 78619, 78620, 78621, 78622, 78623, 78624, 78625, 78626, 78627, 78628, 78629, 78630, 78631, 78632, 78633, 78634, 78635, 78636, 78637, 78638, 78639, 78640, 78641, 78642, 78643, 78644, 78645, 78646, 78647, 78648, 78649, 78650, 78651, 78652, 78653, 78654, 78655, 78656, 78657, 78658, 78659, 78660, 78661, 78662, 78663, 78664, 78665, 78666, 78667, 78668, 78669, 78670, 78671, 78672, 78673, 78674, 78675, 78676, 78677, 78678, 78679, 78680, 78681, 78682, 78683, 78684, 78685, 78686, 78687, 78688, 78689, 78690, 78691, 78692, 78693, 78694, 78695, 78696, 78697, 78698, 78699, 78700, 78701, 78702, 78703, 78704, 78705, 78706, 78707, 78708, 78709, 78710, 78711, 78712, 78713, 78714, 78715, 78716, 78717, 78718, 78719, 78720, 78721, 78722, 78723, 78724, 78725, 78726, 78727, 78728, 78729, 78730, 78731, 78732, 78733, 78734, 78735, 78736, 78737, 78738, 78739, 78740, 78741, 78742, 78743, 78744, 78745, 78746, 78747, 78748, 78749, 78750, 78751, 78752, 78753, 78754, 78755, 78756, 78757, 78758, 78759, 78760, 78761, 78762, 78763, 78764, 78765, 78766, 78767, 78768, 78769, 78770, 78771, 78772, 78773, 78774, 78775, 78776, 78777, 78778, 78779, 78780, 78781, 78782, 78783, 78784, 78785, 78786, 78787, 78788, 78789, 78790, 78791, 78792, 78793, 78794, 78795, 78796, 78797, 78798, 78799, 78800, 78801, 78802, 78803, 78804, 78805, 78806, 78807, 78808, 78809, 78810, 78811, 78812, 78813, 78814, 78815, 78816, 78817, 78818, 78819, 78820, 78821, 78822, 78823, 78824, 78825, 78826, 78827, 78828, 78829, 78830, 78831, 78832, 78833, 78834, 78835, 78836, 78837, 78838, 78839, 78840, 78841, 78842, 78843, 78844, 78845, 78846, 78847, 78848, 78849, 78850, 78851, 78852, 78853, 78854, 78855, 78856, 78857, 78858, 78859, 78860, 78861, 78862, 78863, 78864, 78865, 78866, 78867, 78868, 78869, 78870, 78871, 78872, 78873, 78874, 78875, 78876, 78877, 78878, 78879, 78880, 78881, 78882, 78883, 78884, 78885, 78886, 78887, 78888, 78889, 78890, 78891, 78892, 78893, 78894, 92160, 92161, 92162, 92163, 92164, 92165, 92166, 92167, 92168, 92169, 92170, 92171, 92172, 92173, 92174, 92175, 92176, 92177, 92178, 92179, 92180, 92181, 92182, 92183, 92184, 92185, 92186, 92187, 92188, 92189, 92190, 92191, 92192, 92193, 92194, 92195, 92196, 92197, 92198, 92199, 92200, 92201, 92202, 92203, 92204, 92205, 92206, 92207, 92208, 92209, 92210, 92211, 92212, 92213, 92214, 92215, 92216, 92217, 92218, 92219, 92220, 92221, 92222, 92223, 92224, 92225, 92226, 92227, 92228, 92229, 92230, 92231, 92232, 92233, 92234, 92235, 92236, 92237, 92238, 92239, 92240, 92241, 92242, 92243, 92244, 92245, 92246, 92247, 92248, 92249, 92250, 92251, 92252, 92253, 92254, 92255, 92256, 92257, 92258, 92259, 92260, 92261, 92262, 92263, 92264, 92265, 92266, 92267, 92268, 92269, 92270, 92271, 92272, 92273, 92274, 92275, 92276, 92277, 92278, 92279, 92280, 92281, 92282, 92283, 92284, 92285, 92286, 92287, 92288, 92289, 92290, 92291, 92292, 92293, 92294, 92295, 92296, 92297, 92298, 92299, 92300, 92301, 92302, 92303, 92304, 92305, 92306, 92307, 92308, 92309, 92310, 92311, 92312, 92313, 92314, 92315, 92316, 92317, 92318, 92319, 92320, 92321, 92322, 92323, 92324, 92325, 92326, 92327, 92328, 92329, 92330, 92331, 92332, 92333, 92334, 92335, 92336, 92337, 92338, 92339, 92340, 92341, 92342, 92343, 92344, 92345, 92346, 92347, 92348, 92349, 92350, 92351, 92352, 92353, 92354, 92355, 92356, 92357, 92358, 92359, 92360, 92361, 92362, 92363, 92364, 92365, 92366, 92367, 92368, 92369, 92370, 92371, 92372, 92373, 92374, 92375, 92376, 92377, 92378, 92379, 92380, 92381, 92382, 92383, 92384, 92385, 92386, 92387, 92388, 92389, 92390, 92391, 92392, 92393, 92394, 92395, 92396, 92397, 92398, 92399, 92400, 92401, 92402, 92403, 92404, 92405, 92406, 92407, 92408, 92409, 92410, 92411, 92412, 92413, 92414, 92415, 92416, 92417, 92418, 92419, 92420, 92421, 92422, 92423, 92424, 92425, 92426, 92427, 92428, 92429, 92430, 92431, 92432, 92433, 92434, 92435, 92436, 92437, 92438, 92439, 92440, 92441, 92442, 92443, 92444, 92445, 92446, 92447, 92448, 92449, 92450, 92451, 92452, 92453, 92454, 92455, 92456, 92457, 92458, 92459, 92460, 92461, 92462, 92463, 92464, 92465, 92466, 92467, 92468, 92469, 92470, 92471, 92472, 92473, 92474, 92475, 92476, 92477, 92478, 92479, 92480, 92481, 92482, 92483, 92484, 92485, 92486, 92487, 92488, 92489, 92490, 92491, 92492, 92493, 92494, 92495, 92496, 92497, 92498, 92499, 92500, 92501, 92502, 92503, 92504, 92505, 92506, 92507, 92508, 92509, 92510, 92511, 92512, 92513, 92514, 92515, 92516, 92517, 92518, 92519, 92520, 92521, 92522, 92523, 92524, 92525, 92526, 92527, 92528, 92529, 92530, 92531, 92532, 92533, 92534, 92535, 92536, 92537, 92538, 92539, 92540, 92541, 92542, 92543, 92544, 92545, 92546, 92547, 92548, 92549, 92550, 92551, 92552, 92553, 92554, 92555, 92556, 92557, 92558, 92559, 92560, 92561, 92562, 92563, 92564, 92565, 92566, 92567, 92568, 92569, 92570, 92571, 92572, 92573, 92574, 92575, 92576, 92577, 92578, 92579, 92580, 92581, 92582, 92583, 92584, 92585, 92586, 92587, 92588, 92589, 92590, 92591, 92592, 92593, 92594, 92595, 92596, 92597, 92598, 92599, 92600, 92601, 92602, 92603, 92604, 92605, 92606, 92607, 92608, 92609, 92610, 92611, 92612, 92613, 92614, 92615, 92616, 92617, 92618, 92619, 92620, 92621, 92622, 92623, 92624, 92625, 92626, 92627, 92628, 92629, 92630, 92631, 92632, 92633, 92634, 92635, 92636, 92637, 92638, 92639, 92640, 92641, 92642, 92643, 92644, 92645, 92646, 92647, 92648, 92649, 92650, 92651, 92652, 92653, 92654, 92655, 92656, 92657, 92658, 92659, 92660, 92661, 92662, 92663, 92664, 92665, 92666, 92667, 92668, 92669, 92670, 92671, 92672, 92673, 92674, 92675, 92676, 92677, 92678, 92679, 92680, 92681, 92682, 92683, 92684, 92685, 92686, 92687, 92688, 92689, 92690, 92691, 92692, 92693, 92694, 92695, 92696, 92697, 92698, 92699, 92700, 92701, 92702, 92703, 92704, 92705, 92706, 92707, 92708, 92709, 92710, 92711, 92712, 92713, 92714, 92715, 92716, 92717, 92718, 92719, 92720, 92721, 92722, 92723, 92724, 92725, 92726, 92727, 92728, 110592, 110593, 131072, 173782, 173824, 177972, 177984, 178205, 20029, 20024, 20033, 131362, 20320, 20398, 20411, 20482, 20602, 20633, 20711, 20687, 13470, 132666, 20813, 20820, 20836, 20855, 132380, 13497, 20839, 20877, 132427, 20887, 20900, 20172, 20908, 20917, 168415, 20981, 20995, 13535, 21051, 21062, 21106, 21111, 13589, 21191, 21193, 21220, 21242, 21253, 21254, 21271, 21321, 21329, 21338, 21363, 21373, 21375, 21375, 21375, 133676, 28784, 21450, 21471, 133987, 21483, 21489, 21510, 21662, 21560, 21576, 21608, 21666, 21750, 21776, 21843, 21859, 21892, 21892, 21913, 21931, 21939, 21954, 22294, 22022, 22295, 22097, 22132, 20999, 22766, 22478, 22516, 22541, 22411, 22578, 22577, 22700, 136420, 22770, 22775, 22790, 22810, 22818, 22882, 136872, 136938, 23020, 23067, 23079, 23000, 23142, 14062, 14076, 23304, 23358, 23358, 137672, 23491, 23512, 23527, 23539, 138008, 23551, 23558, 24403, 23586, 14209, 23648, 23662, 23744, 23693, 138724, 23875, 138726, 23918, 23915, 23932, 24033, 24034, 14383, 24061, 24104, 24125, 24169, 14434, 139651, 14460, 24240, 24243, 24246, 24266, 172946, 24318, 140081, 140081, 33281, 24354, 24354, 14535, 144056, 156122, 24418, 24427, 14563, 24474, 24525, 24535, 24569, 24705, 14650, 14620, 24724, 141012, 24775, 24904, 24908, 24910, 24908, 24954, 24974, 25010, 24996, 25007, 25054, 25074, 25078, 25104, 25115, 25181, 25265, 25300, 25424, 142092, 25405, 25340, 25448, 25475, 25572, 142321, 25634, 25541, 25513, 14894, 25705, 25726, 25757, 25719, 14956, 25935, 25964, 143370, 26083, 26360, 26185, 15129, 26257, 15112, 15076, 20882, 20885, 26368, 26268, 32941, 17369, 26391, 26395, 26401, 26462, 26451, 144323, 15177, 26618, 26501, 26706, 26757, 144493, 26766, 26655, 26900, 15261, 26946, 27043, 27114, 27304, 145059, 27355, 15384, 27425, 145575, 27476, 15438, 27506, 27551, 27578, 27579, 146061, 138507, 146170, 27726, 146620, 27839, 27853, 27751, 27926, 27966, 28023, 27969, 28009, 28024, 28037, 146718, 27956, 28207, 28270, 15667, 28363, 28359, 147153, 28153, 28526, 147294, 147342, 28614, 28729, 28702, 28699, 15766, 28746, 28797, 28791, 28845, 132389, 28997, 148067, 29084, 148395, 29224, 29237, 29264, 149000, 29312, 29333, 149301, 149524, 29562, 29579, 16044, 29605, 16056, 16056, 29767, 29788, 29809, 29829, 29898, 16155, 29988, 150582, 30014, 150674, 30064, 139679, 30224, 151457, 151480, 151620, 16380, 16392, 30452, 151795, 151794, 151833, 151859, 30494, 30495, 30495, 30538, 16441, 30603, 16454, 16534, 152605, 30798, 30860, 30924, 16611, 153126, 31062, 153242, 153285, 31119, 31211, 16687, 31296, 31306, 31311, 153980, 154279, 154279, 31470, 16898, 154539, 31686, 31689, 16935, 154752, 31954, 17056, 31976, 31971, 32000, 155526, 32099, 17153, 32199, 32258, 32325, 17204, 156200, 156231, 17241, 156377, 32634, 156478, 32661, 32762, 32773, 156890, 156963, 32864, 157096, 32880, 144223, 17365, 32946, 33027, 17419, 33086, 23221, 157607, 157621, 144275, 144284, 33281, 33284, 36766, 17515, 33425, 33419, 33437, 21171, 33457, 33459, 33469, 33510, 158524, 33509, 33565, 33635, 33709, 33571, 33725, 33767, 33879, 33619, 33738, 33740, 33756, 158774, 159083, 158933, 17707, 34033, 34035, 34070, 160714, 34148, 159532, 17757, 17761, 159665, 159954, 17771, 34384, 34396, 34407, 34409, 34473, 34440, 34574, 34530, 34681, 34600, 34667, 34694, 17879, 34785, 34817, 17913, 34912, 34915, 161383, 35031, 35038, 17973, 35066, 13499, 161966, 162150, 18110, 18119, 35488, 35565, 35722, 35925, 162984, 36011, 36033, 36123, 36215, 163631, 133124, 36299, 36284, 36336, 133342, 36564, 36664, 165330, 165357, 37012, 37105, 37137, 165678, 37147, 37432, 37591, 37592, 37500, 37881, 37909, 166906, 38283, 18837, 38327, 167287, 18918, 38595, 23986, 38691, 168261, 168474, 19054, 19062, 38880, 168970, 19122, 169110, 38923, 38923, 38953, 169398, 39138, 19251, 39209, 39335, 39362, 39422, 19406, 170800, 39698, 40000, 40189, 19662, 19693, 40295, 172238, 19704, 172293, 172558, 172689, 40635, 19798, 40697, 40702, 40709, 40719, 40726, 40763, 173568],
				'encoded': '\u01bb\u01c0\u01c1\u01c2\u01c3\u0294\u05d0\u05d1\u05d2\u05d3\u05d4\u05d5\u05d6\u05d7\u05d8\u05d9\u05da\u05db\u05dc\u05dd\u05de\u05df\u05e0\u05e1\u05e2\u05e3\u05e4\u05e5\u05e6\u05e7\u05e8\u05e9\u05ea\u05f0\u05f1\u05f2\u0620\u0621\u0622\u0623\u0624\u0625\u0626\u0627\u0628\u0629\u062a\u062b\u062c\u062d\u062e\u062f\u0630\u0631\u0632\u0633\u0634\u0635\u0636\u0637\u0638\u0639\u063a\u063b\u063c\u063d\u063e\u063f\u0641\u0642\u0643\u0644\u0645\u0646\u0647\u0648\u0649\u064a\u066e\u066f\u0671\u0672\u0673\u0674\u0675\u0676\u0677\u0678\u0679\u067a\u067b\u067c\u067d\u067e\u067f\u0680\u0681\u0682\u0683\u0684\u0685\u0686\u0687\u0688\u0689\u068a\u068b\u068c\u068d\u068e\u068f\u0690\u0691\u0692\u0693\u0694\u0695\u0696\u0697\u0698\u0699\u069a\u069b\u069c\u069d\u069e\u069f\u06a0\u06a1\u06a2\u06a3\u06a4\u06a5\u06a6\u06a7\u06a8\u06a9\u06aa\u06ab\u06ac\u06ad\u06ae\u06af\u06b0\u06b1\u06b2\u06b3\u06b4\u06b5\u06b6\u06b7\u06b8\u06b9\u06ba\u06bb\u06bc\u06bd\u06be\u06bf\u06c0\u06c1\u06c2\u06c3\u06c4\u06c5\u06c6\u06c7\u06c8\u06c9\u06ca\u06cb\u06cc\u06cd\u06ce\u06cf\u06d0\u06d1\u06d2\u06d3\u06d5\u06ee\u06ef\u06fa\u06fb\u06fc\u06ff\u0710\u0712\u0713\u0714\u0715\u0716\u0717\u0718\u0719\u071a\u071b\u071c\u071d\u071e\u071f\u0720\u0721\u0722\u0723\u0724\u0725\u0726\u0727\u0728\u0729\u072a\u072b\u072c\u072d\u072e\u072f\u074d\u074e\u074f\u0750\u0751\u0752\u0753\u0754\u0755\u0756\u0757\u0758\u0759\u075a\u075b\u075c\u075d\u075e\u075f\u0760\u0761\u0762\u0763\u0764\u0765\u0766\u0767\u0768\u0769\u076a\u076b\u076c\u076d\u076e\u076f\u0770\u0771\u0772\u0773\u0774\u0775\u0776\u0777\u0778\u0779\u077a\u077b\u077c\u077d\u077e\u077f\u0780\u0781\u0782\u0783\u0784\u0785\u0786\u0787\u0788\u0789\u078a\u078b\u078c\u078d\u078e\u078f\u0790\u0791\u0792\u0793\u0794\u0795\u0796\u0797\u0798\u0799\u079a\u079b\u079c\u079d\u079e\u079f\u07a0\u07a1\u07a2\u07a3\u07a4\u07a5\u07b1\u07ca\u07cb\u07cc\u07cd\u07ce\u07cf\u07d0\u07d1\u07d2\u07d3\u07d4\u07d5\u07d6\u07d7\u07d8\u07d9\u07da\u07db\u07dc\u07dd\u07de\u07df\u07e0\u07e1\u07e2\u07e3\u07e4\u07e5\u07e6\u07e7\u07e8\u07e9\u07ea\u0800\u0801\u0802\u0803\u0804\u0805\u0806\u0807\u0808\u0809\u080a\u080b\u080c\u080d\u080e\u080f\u0810\u0811\u0812\u0813\u0814\u0815\u0840\u0841\u0842\u0843\u0844\u0845\u0846\u0847\u0848\u0849\u084a\u084b\u084c\u084d\u084e\u084f\u0850\u0851\u0852\u0853\u0854\u0855\u0856\u0857\u0858\u0904\u0905\u0906\u0907\u0908\u0909\u090a\u090b\u090c\u090d\u090e\u090f\u0910\u0911\u0912\u0913\u0914\u0915\u0916\u0917\u0918\u0919\u091a\u091b\u091c\u091d\u091e\u091f\u0920\u0921\u0922\u0923\u0924\u0925\u0926\u0927\u0928\u0929\u092a\u092b\u092c\u092d\u092e\u092f\u0930\u0931\u0932\u0933\u0934\u0935\u0936\u0937\u0938\u0939\u093d\u0950\u0915\u093c\u0916\u093c\u0917\u093c\u091c\u093c\u0921\u093c\u0922\u093c\u092b\u093c\u092f\u093c\u0960\u0961\u0972\u0973\u0974\u0975\u0976\u0977\u0979\u097a\u097b\u097c\u097d\u097e\u097f\u0985\u0986\u0987\u0988\u0989\u098a\u098b\u098c\u098f\u0990\u0993\u0994\u0995\u0996\u0997\u0998\u0999\u099a\u099b\u099c\u099d\u099e\u099f\u09a0\u09a1\u09a2\u09a3\u09a4\u09a5\u09a6\u09a7\u09a8\u09aa\u09ab\u09ac\u09ad\u09ae\u09af\u09b0\u09b2\u09b6\u09b7\u09b8\u09b9\u09bd\u09ce\u09a1\u09bc\u09a2\u09bc\u09af\u09bc\u09e0\u09e1\u09f0\u09f1\u0a05\u0a06\u0a07\u0a08\u0a09\u0a0a\u0a0f\u0a10\u0a13\u0a14\u0a15\u0a16\u0a17\u0a18\u0a19\u0a1a\u0a1b\u0a1c\u0a1d\u0a1e\u0a1f\u0a20\u0a21\u0a22\u0a23\u0a24\u0a25\u0a26\u0a27\u0a28\u0a2a\u0a2b\u0a2c\u0a2d\u0a2e\u0a2f\u0a30\u0a32\u0a32\u0a3c\u0a35\u0a38\u0a3c\u0a38\u0a39\u0a16\u0a3c\u0a17\u0a3c\u0a1c\u0a3c\u0a5c\u0a2b\u0a3c\u0a72\u0a73\u0a74\u0a85\u0a86\u0a87\u0a88\u0a89\u0a8a\u0a8b\u0a8c\u0a8d\u0a8f\u0a90\u0a91\u0a93\u0a94\u0a95\u0a96\u0a97\u0a98\u0a99\u0a9a\u0a9b\u0a9c\u0a9d\u0a9e\u0a9f\u0aa0\u0aa1\u0aa2\u0aa3\u0aa4\u0aa5\u0aa6\u0aa7\u0aa8\u0aaa\u0aab\u0aac\u0aad\u0aae\u0aaf\u0ab0\u0ab2\u0ab3\u0ab5\u0ab6\u0ab7\u0ab8\u0ab9\u0abd\u0ad0\u0ae0\u0ae1\u0b05\u0b06\u0b07\u0b08\u0b09\u0b0a\u0b0b\u0b0c\u0b0f\u0b10\u0b13\u0b14\u0b15\u0b16\u0b17\u0b18\u0b19\u0b1a\u0b1b\u0b1c\u0b1d\u0b1e\u0b1f\u0b20\u0b21\u0b22\u0b23\u0b24\u0b25\u0b26\u0b27\u0b28\u0b2a\u0b2b\u0b2c\u0b2d\u0b2e\u0b2f\u0b30\u0b32\u0b33\u0b35\u0b36\u0b37\u0b38\u0b39\u0b3d\u0b21\u0b3c\u0b22\u0b3c\u0b5f\u0b60\u0b61\u0b71\u0b83\u0b85\u0b86\u0b87\u0b88\u0b89\u0b8a\u0b8e\u0b8f\u0b90\u0b92\u0b93\u0b94\u0b95\u0b99\u0b9a\u0b9c\u0b9e\u0b9f\u0ba3\u0ba4\u0ba8\u0ba9\u0baa\u0bae\u0baf\u0bb0\u0bb1\u0bb2\u0bb3\u0bb4\u0bb5\u0bb6\u0bb7\u0bb8\u0bb9\u0bd0\u0c05\u0c06\u0c07\u0c08\u0c09\u0c0a\u0c0b\u0c0c\u0c0e\u0c0f\u0c10\u0c12\u0c13\u0c14\u0c15\u0c16\u0c17\u0c18\u0c19\u0c1a\u0c1b\u0c1c\u0c1d\u0c1e\u0c1f\u0c20\u0c21\u0c22\u0c23\u0c24\u0c25\u0c26\u0c27\u0c28\u0c2a\u0c2b\u0c2c\u0c2d\u0c2e\u0c2f\u0c30\u0c31\u0c32\u0c33\u0c35\u0c36\u0c37\u0c38\u0c39\u0c3d\u0c58\u0c59\u0c60\u0c61\u0c85\u0c86\u0c87\u0c88\u0c89\u0c8a\u0c8b\u0c8c\u0c8e\u0c8f\u0c90\u0c92\u0c93\u0c94\u0c95\u0c96\u0c97\u0c98\u0c99\u0c9a\u0c9b\u0c9c\u0c9d\u0c9e\u0c9f\u0ca0\u0ca1\u0ca2\u0ca3\u0ca4\u0ca5\u0ca6\u0ca7\u0ca8\u0caa\u0cab\u0cac\u0cad\u0cae\u0caf\u0cb0\u0cb1\u0cb2\u0cb3\u0cb5\u0cb6\u0cb7\u0cb8\u0cb9\u0cbd\u0cde\u0ce0\u0ce1\u0cf1\u0cf2\u0d05\u0d06\u0d07\u0d08\u0d09\u0d0a\u0d0b\u0d0c\u0d0e\u0d0f\u0d10\u0d12\u0d13\u0d14\u0d15\u0d16\u0d17\u0d18\u0d19\u0d1a\u0d1b\u0d1c\u0d1d\u0d1e\u0d1f\u0d20\u0d21\u0d22\u0d23\u0d24\u0d25\u0d26\u0d27\u0d28\u0d29\u0d2a\u0d2b\u0d2c\u0d2d\u0d2e\u0d2f\u0d30\u0d31\u0d32\u0d33\u0d34\u0d35\u0d36\u0d37\u0d38\u0d39\u0d3a\u0d3d\u0d4e\u0d60\u0d61\u0d7a\u0d7b\u0d7c\u0d7d\u0d7e\u0d7f\u0d85\u0d86\u0d87\u0d88\u0d89\u0d8a\u0d8b\u0d8c\u0d8d\u0d8e\u0d8f\u0d90\u0d91\u0d92\u0d93\u0d94\u0d95\u0d96\u0d9a\u0d9b\u0d9c\u0d9d\u0d9e\u0d9f\u0da0\u0da1\u0da2\u0da3\u0da4\u0da5\u0da6\u0da7\u0da8\u0da9\u0daa\u0dab\u0dac\u0dad\u0dae\u0daf\u0db0\u0db1\u0db3\u0db4\u0db5\u0db6\u0db7\u0db8\u0db9\u0dba\u0dbb\u0dbd\u0dc0\u0dc1\u0dc2\u0dc3\u0dc4\u0dc5\u0dc6\u0e01\u0e02\u0e03\u0e04\u0e05\u0e06\u0e07\u0e08\u0e09\u0e0a\u0e0b\u0e0c\u0e0d\u0e0e\u0e0f\u0e10\u0e11\u0e12\u0e13\u0e14\u0e15\u0e16\u0e17\u0e18\u0e19\u0e1a\u0e1b\u0e1c\u0e1d\u0e1e\u0e1f\u0e20\u0e21\u0e22\u0e23\u0e24\u0e25\u0e26\u0e27\u0e28\u0e29\u0e2a\u0e2b\u0e2c\u0e2d\u0e2e\u0e2f\u0e30\u0e32\u0e33\u0e40\u0e41\u0e42\u0e43\u0e44\u0e45\u0e81\u0e82\u0e84\u0e87\u0e88\u0e8a\u0e8d\u0e94\u0e95\u0e96\u0e97\u0e99\u0e9a\u0e9b\u0e9c\u0e9d\u0e9e\u0e9f\u0ea1\u0ea2\u0ea3\u0ea5\u0ea7\u0eaa\u0eab\u0ead\u0eae\u0eaf\u0eb0\u0eb2\u0eb3\u0ebd\u0ec0\u0ec1\u0ec2\u0ec3\u0ec4\u0edc\u0edd\u0f00\u0f40\u0f41\u0f42\u0f42\u0fb7\u0f44\u0f45\u0f46\u0f47\u0f49\u0f4a\u0f4b\u0f4c\u0f4c\u0fb7\u0f4e\u0f4f\u0f50\u0f51\u0f51\u0fb7\u0f53\u0f54\u0f55\u0f56\u0f56\u0fb7\u0f58\u0f59\u0f5a\u0f5b\u0f5b\u0fb7\u0f5d\u0f5e\u0f5f\u0f60\u0f61\u0f62\u0f63\u0f64\u0f65\u0f66\u0f67\u0f68\u0f40\u0fb5\u0f6a\u0f6b\u0f6c\u0f88\u0f89\u0f8a\u0f8b\u0f8c\u1000\u1001\u1002\u1003\u1004\u1005\u1006\u1007\u1008\u1009\u100a\u100b\u100c\u100d\u100e\u100f\u1010\u1011\u1012\u1013\u1014\u1015\u1016\u1017\u1018\u1019\u101a\u101b\u101c\u101d\u101e\u101f\u1020\u1021\u1022\u1023\u1024\u1025\u1026\u1027\u1028\u1029\u102a\u103f\u1050\u1051\u1052\u1053\u1054\u1055\u105a\u105b\u105c\u105d\u1061\u1065\u1066\u106e\u106f\u1070\u1075\u1076\u1077\u1078\u1079\u107a\u107b\u107c\u107d\u107e\u107f\u1080\u1081\u108e\u10d0\u10d1\u10d2\u10d3\u10d4\u10d5\u10d6\u10d7\u10d8\u10d9\u10da\u10db\u10dc\u10dd\u10de\u10df\u10e0\u10e1\u10e2\u10e3\u10e4\u10e5\u10e6\u10e7\u10e8\u10e9\u10ea\u10eb\u10ec\u10ed\u10ee\u10ef\u10f0\u10f1\u10f2\u10f3\u10f4\u10f5\u10f6\u10f7\u10f8\u10f9\u10fa\u1100\u1101\u1102\u1103\u1104\u1105\u1106\u1107\u1108\u1109\u110a\u110b\u110c\u110d\u110e\u110f\u1110\u1111\u1112\u1113\u1114\u1115\u1116\u1117\u1118\u1119\u111a\u111b\u111c\u111d\u111e\u111f\u1120\u1121\u1122\u1123\u1124\u1125\u1126\u1127\u1128\u1129\u112a\u112b\u112c\u112d\u112e\u112f\u1130\u1131\u1132\u1133\u1134\u1135\u1136\u1137\u1138\u1139\u113a\u113b\u113c\u113d\u113e\u113f\u1140\u1141\u1142\u1143\u1144\u1145\u1146\u1147\u1148\u1149\u114a\u114b\u114c\u114d\u114e\u114f\u1150\u1151\u1152\u1153\u1154\u1155\u1156\u1157\u1158\u1159\u115a\u115b\u115c\u115d\u115e\u115f\u1160\u1161\u1162\u1163\u1164\u1165\u1166\u1167\u1168\u1169\u116a\u116b\u116c\u116d\u116e\u116f\u1170\u1171\u1172\u1173\u1174\u1175\u1176\u1177\u1178\u1179\u117a\u117b\u117c\u117d\u117e\u117f\u1180\u1181\u1182\u1183\u1184\u1185\u1186\u1187\u1188\u1189\u118a\u118b\u118c\u118d\u118e\u118f\u1190\u1191\u1192\u1193\u1194\u1195\u1196\u1197\u1198\u1199\u119a\u119b\u119c\u119d\u119e\u119f\u11a0\u11a1\u11a2\u11a3\u11a4\u11a5\u11a6\u11a7\u11a8\u11a9\u11aa\u11ab\u11ac\u11ad\u11ae\u11af\u11b0\u11b1\u11b2\u11b3\u11b4\u11b5\u11b6\u11b7\u11b8\u11b9\u11ba\u11bb\u11bc\u11bd\u11be\u11bf\u11c0\u11c1\u11c2\u11c3\u11c4\u11c5\u11c6\u11c7\u11c8\u11c9\u11ca\u11cb\u11cc\u11cd\u11ce\u11cf\u11d0\u11d1\u11d2\u11d3\u11d4\u11d5\u11d6\u11d7\u11d8\u11d9\u11da\u11db\u11dc\u11dd\u11de\u11df\u11e0\u11e1\u11e2\u11e3\u11e4\u11e5\u11e6\u11e7\u11e8\u11e9\u11ea\u11eb\u11ec\u11ed\u11ee\u11ef\u11f0\u11f1\u11f2\u11f3\u11f4\u11f5\u11f6\u11f7\u11f8\u11f9\u11fa\u11fb\u11fc\u11fd\u11fe\u11ff\u1200\u1201\u1202\u1203\u1204\u1205\u1206\u1207\u1208\u1209\u120a\u120b\u120c\u120d\u120e\u120f\u1210\u1211\u1212\u1213\u1214\u1215\u1216\u1217\u1218\u1219\u121a\u121b\u121c\u121d\u121e\u121f\u1220\u1221\u1222\u1223\u1224\u1225\u1226\u1227\u1228\u1229\u122a\u122b\u122c\u122d\u122e\u122f\u1230\u1231\u1232\u1233\u1234\u1235\u1236\u1237\u1238\u1239\u123a\u123b\u123c\u123d\u123e\u123f\u1240\u1241\u1242\u1243\u1244\u1245\u1246\u1247\u1248\u124a\u124b\u124c\u124d\u1250\u1251\u1252\u1253\u1254\u1255\u1256\u1258\u125a\u125b\u125c\u125d\u1260\u1261\u1262\u1263\u1264\u1265\u1266\u1267\u1268\u1269\u126a\u126b\u126c\u126d\u126e\u126f\u1270\u1271\u1272\u1273\u1274\u1275\u1276\u1277\u1278\u1279\u127a\u127b\u127c\u127d\u127e\u127f\u1280\u1281\u1282\u1283\u1284\u1285\u1286\u1287\u1288\u128a\u128b\u128c\u128d\u1290\u1291\u1292\u1293\u1294\u1295\u1296\u1297\u1298\u1299\u129a\u129b\u129c\u129d\u129e\u129f\u12a0\u12a1\u12a2\u12a3\u12a4\u12a5\u12a6\u12a7\u12a8\u12a9\u12aa\u12ab\u12ac\u12ad\u12ae\u12af\u12b0\u12b2\u12b3\u12b4\u12b5\u12b8\u12b9\u12ba\u12bb\u12bc\u12bd\u12be\u12c0\u12c2\u12c3\u12c4\u12c5\u12c8\u12c9\u12ca\u12cb\u12cc\u12cd\u12ce\u12cf\u12d0\u12d1\u12d2\u12d3\u12d4\u12d5\u12d6\u12d8\u12d9\u12da\u12db\u12dc\u12dd\u12de\u12df\u12e0\u12e1\u12e2\u12e3\u12e4\u12e5\u12e6\u12e7\u12e8\u12e9\u12ea\u12eb\u12ec\u12ed\u12ee\u12ef\u12f0\u12f1\u12f2\u12f3\u12f4\u12f5\u12f6\u12f7\u12f8\u12f9\u12fa\u12fb\u12fc\u12fd\u12fe\u12ff\u1300\u1301\u1302\u1303\u1304\u1305\u1306\u1307\u1308\u1309\u130a\u130b\u130c\u130d\u130e\u130f\u1310\u1312\u1313\u1314\u1315\u1318\u1319\u131a\u131b\u131c\u131d\u131e\u131f\u1320\u1321\u1322\u1323\u1324\u1325\u1326\u1327\u1328\u1329\u132a\u132b\u132c\u132d\u132e\u132f\u1330\u1331\u1332\u1333\u1334\u1335\u1336\u1337\u1338\u1339\u133a\u133b\u133c\u133d\u133e\u133f\u1340\u1341\u1342\u1343\u1344\u1345\u1346\u1347\u1348\u1349\u134a\u134b\u134c\u134d\u134e\u134f\u1350\u1351\u1352\u1353\u1354\u1355\u1356\u1357\u1358\u1359\u135a\u1380\u1381\u1382\u1383\u1384\u1385\u1386\u1387\u1388\u1389\u138a\u138b\u138c\u138d\u138e\u138f\u13a0\u13a1\u13a2\u13a3\u13a4\u13a5\u13a6\u13a7\u13a8\u13a9\u13aa\u13ab\u13ac\u13ad\u13ae\u13af\u13b0\u13b1\u13b2\u13b3\u13b4\u13b5\u13b6\u13b7\u13b8\u13b9\u13ba\u13bb\u13bc\u13bd\u13be\u13bf\u13c0\u13c1\u13c2\u13c3\u13c4\u13c5\u13c6\u13c7\u13c8\u13c9\u13ca\u13cb\u13cc\u13cd\u13ce\u13cf\u13d0\u13d1\u13d2\u13d3\u13d4\u13d5\u13d6\u13d7\u13d8\u13d9\u13da\u13db\u13dc\u13dd\u13de\u13df\u13e0\u13e1\u13e2\u13e3\u13e4\u13e5\u13e6\u13e7\u13e8\u13e9\u13ea\u13eb\u13ec\u13ed\u13ee\u13ef\u13f0\u13f1\u13f2\u13f3\u13f4\u1401\u1402\u1403\u1404\u1405\u1406\u1407\u1408\u1409\u140a\u140b\u140c\u140d\u140e\u140f\u1410\u1411\u1412\u1413\u1414\u1415\u1416\u1417\u1418\u1419\u141a\u141b\u141c\u141d\u141e\u141f\u1420\u1421\u1422\u1423\u1424\u1425\u1426\u1427\u1428\u1429\u142a\u142b\u142c\u142d\u142e\u142f\u1430\u1431\u1432\u1433\u1434\u1435\u1436\u1437\u1438\u1439\u143a\u143b\u143c\u143d\u143e\u143f\u1440\u1441\u1442\u1443\u1444\u1445\u1446\u1447\u1448\u1449\u144a\u144b\u144c\u144d\u144e\u144f\u1450\u1451\u1452\u1453\u1454\u1455\u1456\u1457\u1458\u1459\u145a\u145b\u145c\u145d\u145e\u145f\u1460\u1461\u1462\u1463\u1464\u1465\u1466\u1467\u1468\u1469\u146a\u146b\u146c\u146d\u146e\u146f\u1470\u1471\u1472\u1473\u1474\u1475\u1476\u1477\u1478\u1479\u147a\u147b\u147c\u147d\u147e\u147f\u1480\u1481\u1482\u1483\u1484\u1485\u1486\u1487\u1488\u1489\u148a\u148b\u148c\u148d\u148e\u148f\u1490\u1491\u1492\u1493\u1494\u1495\u1496\u1497\u1498\u1499\u149a\u149b\u149c\u149d\u149e\u149f\u14a0\u14a1\u14a2\u14a3\u14a4\u14a5\u14a6\u14a7\u14a8\u14a9\u14aa\u14ab\u14ac\u14ad\u14ae\u14af\u14b0\u14b1\u14b2\u14b3\u14b4\u14b5\u14b6\u14b7\u14b8\u14b9\u14ba\u14bb\u14bc\u14bd\u14be\u14bf\u14c0\u14c1\u14c2\u14c3\u14c4\u14c5\u14c6\u14c7\u14c8\u14c9\u14ca\u14cb\u14cc\u14cd\u14ce\u14cf\u14d0\u14d1\u14d2\u14d3\u14d4\u14d5\u14d6\u14d7\u14d8\u14d9\u14da\u14db\u14dc\u14dd\u14de\u14df\u14e0\u14e1\u14e2\u14e3\u14e4\u14e5\u14e6\u14e7\u14e8\u14e9\u14ea\u14eb\u14ec\u14ed\u14ee\u14ef\u14f0\u14f1\u14f2\u14f3\u14f4\u14f5\u14f6\u14f7\u14f8\u14f9\u14fa\u14fb\u14fc\u14fd\u14fe\u14ff\u1500\u1501\u1502\u1503\u1504\u1505\u1506\u1507\u1508\u1509\u150a\u150b\u150c\u150d\u150e\u150f\u1510\u1511\u1512\u1513\u1514\u1515\u1516\u1517\u1518\u1519\u151a\u151b\u151c\u151d\u151e\u151f\u1520\u1521\u1522\u1523\u1524\u1525\u1526\u1527\u1528\u1529\u152a\u152b\u152c\u152d\u152e\u152f\u1530\u1531\u1532\u1533\u1534\u1535\u1536\u1537\u1538\u1539\u153a\u153b\u153c\u153d\u153e\u153f\u1540\u1541\u1542\u1543\u1544\u1545\u1546\u1547\u1548\u1549\u154a\u154b\u154c\u154d\u154e\u154f\u1550\u1551\u1552\u1553\u1554\u1555\u1556\u1557\u1558\u1559\u155a\u155b\u155c\u155d\u155e\u155f\u1560\u1561\u1562\u1563\u1564\u1565\u1566\u1567\u1568\u1569\u156a\u156b\u156c\u156d\u156e\u156f\u1570\u1571\u1572\u1573\u1574\u1575\u1576\u1577\u1578\u1579\u157a\u157b\u157c\u157d\u157e\u157f\u1580\u1581\u1582\u1583\u1584\u1585\u1586\u1587\u1588\u1589\u158a\u158b\u158c\u158d\u158e\u158f\u1590\u1591\u1592\u1593\u1594\u1595\u1596\u1597\u1598\u1599\u159a\u159b\u159c\u159d\u159e\u159f\u15a0\u15a1\u15a2\u15a3\u15a4\u15a5\u15a6\u15a7\u15a8\u15a9\u15aa\u15ab\u15ac\u15ad\u15ae\u15af\u15b0\u15b1\u15b2\u15b3\u15b4\u15b5\u15b6\u15b7\u15b8\u15b9\u15ba\u15bb\u15bc\u15bd\u15be\u15bf\u15c0\u15c1\u15c2\u15c3\u15c4\u15c5\u15c6\u15c7\u15c8\u15c9\u15ca\u15cb\u15cc\u15cd\u15ce\u15cf\u15d0\u15d1\u15d2\u15d3\u15d4\u15d5\u15d6\u15d7\u15d8\u15d9\u15da\u15db\u15dc\u15dd\u15de\u15df\u15e0\u15e1\u15e2\u15e3\u15e4\u15e5\u15e6\u15e7\u15e8\u15e9\u15ea\u15eb\u15ec\u15ed\u15ee\u15ef\u15f0\u15f1\u15f2\u15f3\u15f4\u15f5\u15f6\u15f7\u15f8\u15f9\u15fa\u15fb\u15fc\u15fd\u15fe\u15ff\u1600\u1601\u1602\u1603\u1604\u1605\u1606\u1607\u1608\u1609\u160a\u160b\u160c\u160d\u160e\u160f\u1610\u1611\u1612\u1613\u1614\u1615\u1616\u1617\u1618\u1619\u161a\u161b\u161c\u161d\u161e\u161f\u1620\u1621\u1622\u1623\u1624\u1625\u1626\u1627\u1628\u1629\u162a\u162b\u162c\u162d\u162e\u162f\u1630\u1631\u1632\u1633\u1634\u1635\u1636\u1637\u1638\u1639\u163a\u163b\u163c\u163d\u163e\u163f\u1640\u1641\u1642\u1643\u1644\u1645\u1646\u1647\u1648\u1649\u164a\u164b\u164c\u164d\u164e\u164f\u1650\u1651\u1652\u1653\u1654\u1655\u1656\u1657\u1658\u1659\u165a\u165b\u165c\u165d\u165e\u165f\u1660\u1661\u1662\u1663\u1664\u1665\u1666\u1667\u1668\u1669\u166a\u166b\u166c\u166f\u1670\u1671\u1672\u1673\u1674\u1675\u1676\u1677\u1678\u1679\u167a\u167b\u167c\u167d\u167e\u167f\u1681\u1682\u1683\u1684\u1685\u1686\u1687\u1688\u1689\u168a\u168b\u168c\u168d\u168e\u168f\u1690\u1691\u1692\u1693\u1694\u1695\u1696\u1697\u1698\u1699\u169a\u16a0\u16a1\u16a2\u16a3\u16a4\u16a5\u16a6\u16a7\u16a8\u16a9\u16aa\u16ab\u16ac\u16ad\u16ae\u16af\u16b0\u16b1\u16b2\u16b3\u16b4\u16b5\u16b6\u16b7\u16b8\u16b9\u16ba\u16bb\u16bc\u16bd\u16be\u16bf\u16c0\u16c1\u16c2\u16c3\u16c4\u16c5\u16c6\u16c7\u16c8\u16c9\u16ca\u16cb\u16cc\u16cd\u16ce\u16cf\u16d0\u16d1\u16d2\u16d3\u16d4\u16d5\u16d6\u16d7\u16d8\u16d9\u16da\u16db\u16dc\u16dd\u16de\u16df\u16e0\u16e1\u16e2\u16e3\u16e4\u16e5\u16e6\u16e7\u16e8\u16e9\u16ea\u1700\u1701\u1702\u1703\u1704\u1705\u1706\u1707\u1708\u1709\u170a\u170b\u170c\u170e\u170f\u1710\u1711\u1720\u1721\u1722\u1723\u1724\u1725\u1726\u1727\u1728\u1729\u172a\u172b\u172c\u172d\u172e\u172f\u1730\u1731\u1740\u1741\u1742\u1743\u1744\u1745\u1746\u1747\u1748\u1749\u174a\u174b\u174c\u174d\u174e\u174f\u1750\u1751\u1760\u1761\u1762\u1763\u1764\u1765\u1766\u1767\u1768\u1769\u176a\u176b\u176c\u176e\u176f\u1770\u1780\u1781\u1782\u1783\u1784\u1785\u1786\u1787\u1788\u1789\u178a\u178b\u178c\u178d\u178e\u178f\u1790\u1791\u1792\u1793\u1794\u1795\u1796\u1797\u1798\u1799\u179a\u179b\u179c\u179d\u179e\u179f\u17a0\u17a1\u17a2\u17a3\u17a4\u17a5\u17a6\u17a7\u17a8\u17a9\u17aa\u17ab\u17ac\u17ad\u17ae\u17af\u17b0\u17b1\u17b2\u17b3\u17dc\u1820\u1821\u1822\u1823\u1824\u1825\u1826\u1827\u1828\u1829\u182a\u182b\u182c\u182d\u182e\u182f\u1830\u1831\u1832\u1833\u1834\u1835\u1836\u1837\u1838\u1839\u183a\u183b\u183c\u183d\u183e\u183f\u1840\u1841\u1842\u1844\u1845\u1846\u1847\u1848\u1849\u184a\u184b\u184c\u184d\u184e\u184f\u1850\u1851\u1852\u1853\u1854\u1855\u1856\u1857\u1858\u1859\u185a\u185b\u185c\u185d\u185e\u185f\u1860\u1861\u1862\u1863\u1864\u1865\u1866\u1867\u1868\u1869\u186a\u186b\u186c\u186d\u186e\u186f\u1870\u1871\u1872\u1873\u1874\u1875\u1876\u1877\u1880\u1881\u1882\u1883\u1884\u1885\u1886\u1887\u1888\u1889\u188a\u188b\u188c\u188d\u188e\u188f\u1890\u1891\u1892\u1893\u1894\u1895\u1896\u1897\u1898\u1899\u189a\u189b\u189c\u189d\u189e\u189f\u18a0\u18a1\u18a2\u18a3\u18a4\u18a5\u18a6\u18a7\u18a8\u18aa\u18b0\u18b1\u18b2\u18b3\u18b4\u18b5\u18b6\u18b7\u18b8\u18b9\u18ba\u18bb\u18bc\u18bd\u18be\u18bf\u18c0\u18c1\u18c2\u18c3\u18c4\u18c5\u18c6\u18c7\u18c8\u18c9\u18ca\u18cb\u18cc\u18cd\u18ce\u18cf\u18d0\u18d1\u18d2\u18d3\u18d4\u18d5\u18d6\u18d7\u18d8\u18d9\u18da\u18db\u18dc\u18dd\u18de\u18df\u18e0\u18e1\u18e2\u18e3\u18e4\u18e5\u18e6\u18e7\u18e8\u18e9\u18ea\u18eb\u18ec\u18ed\u18ee\u18ef\u18f0\u18f1\u18f2\u18f3\u18f4\u18f5\u1900\u1901\u1902\u1903\u1904\u1905\u1906\u1907\u1908\u1909\u190a\u190b\u190c\u190d\u190e\u190f\u1910\u1911\u1912\u1913\u1914\u1915\u1916\u1917\u1918\u1919\u191a\u191b\u191c\u1950\u1951\u1952\u1953\u1954\u1955\u1956\u1957\u1958\u1959\u195a\u195b\u195c\u195d\u195e\u195f\u1960\u1961\u1962\u1963\u1964\u1965\u1966\u1967\u1968\u1969\u196a\u196b\u196c\u196d\u1970\u1971\u1972\u1973\u1974\u1980\u1981\u1982\u1983\u1984\u1985\u1986\u1987\u1988\u1989\u198a\u198b\u198c\u198d\u198e\u198f\u1990\u1991\u1992\u1993\u1994\u1995\u1996\u1997\u1998\u1999\u199a\u199b\u199c\u199d\u199e\u199f\u19a0\u19a1\u19a2\u19a3\u19a4\u19a5\u19a6\u19a7\u19a8\u19a9\u19aa\u19ab\u19c1\u19c2\u19c3\u19c4\u19c5\u19c6\u19c7\u1a00\u1a01\u1a02\u1a03\u1a04\u1a05\u1a06\u1a07\u1a08\u1a09\u1a0a\u1a0b\u1a0c\u1a0d\u1a0e\u1a0f\u1a10\u1a11\u1a12\u1a13\u1a14\u1a15\u1a16\u1a20\u1a21\u1a22\u1a23\u1a24\u1a25\u1a26\u1a27\u1a28\u1a29\u1a2a\u1a2b\u1a2c\u1a2d\u1a2e\u1a2f\u1a30\u1a31\u1a32\u1a33\u1a34\u1a35\u1a36\u1a37\u1a38\u1a39\u1a3a\u1a3b\u1a3c\u1a3d\u1a3e\u1a3f\u1a40\u1a41\u1a42\u1a43\u1a44\u1a45\u1a46\u1a47\u1a48\u1a49\u1a4a\u1a4b\u1a4c\u1a4d\u1a4e\u1a4f\u1a50\u1a51\u1a52\u1a53\u1a54\u1b05\u1b06\u1b07\u1b08\u1b09\u1b0a\u1b0b\u1b0c\u1b0d\u1b0e\u1b0f\u1b10\u1b11\u1b12\u1b13\u1b14\u1b15\u1b16\u1b17\u1b18\u1b19\u1b1a\u1b1b\u1b1c\u1b1d\u1b1e\u1b1f\u1b20\u1b21\u1b22\u1b23\u1b24\u1b25\u1b26\u1b27\u1b28\u1b29\u1b2a\u1b2b\u1b2c\u1b2d\u1b2e\u1b2f\u1b30\u1b31\u1b32\u1b33\u1b45\u1b46\u1b47\u1b48\u1b49\u1b4a\u1b4b\u1b83\u1b84\u1b85\u1b86\u1b87\u1b88\u1b89\u1b8a\u1b8b\u1b8c\u1b8d\u1b8e\u1b8f\u1b90\u1b91\u1b92\u1b93\u1b94\u1b95\u1b96\u1b97\u1b98\u1b99\u1b9a\u1b9b\u1b9c\u1b9d\u1b9e\u1b9f\u1ba0\u1bae\u1baf\u1bc0\u1bc1\u1bc2\u1bc3\u1bc4\u1bc5\u1bc6\u1bc7\u1bc8\u1bc9\u1bca\u1bcb\u1bcc\u1bcd\u1bce\u1bcf\u1bd0\u1bd1\u1bd2\u1bd3\u1bd4\u1bd5\u1bd6\u1bd7\u1bd8\u1bd9\u1bda\u1bdb\u1bdc\u1bdd\u1bde\u1bdf\u1be0\u1be1\u1be2\u1be3\u1be4\u1be5\u1c00\u1c01\u1c02\u1c03\u1c04\u1c05\u1c06\u1c07\u1c08\u1c09\u1c0a\u1c0b\u1c0c\u1c0d\u1c0e\u1c0f\u1c10\u1c11\u1c12\u1c13\u1c14\u1c15\u1c16\u1c17\u1c18\u1c19\u1c1a\u1c1b\u1c1c\u1c1d\u1c1e\u1c1f\u1c20\u1c21\u1c22\u1c23\u1c4d\u1c4e\u1c4f\u1c5a\u1c5b\u1c5c\u1c5d\u1c5e\u1c5f\u1c60\u1c61\u1c62\u1c63\u1c64\u1c65\u1c66\u1c67\u1c68\u1c69\u1c6a\u1c6b\u1c6c\u1c6d\u1c6e\u1c6f\u1c70\u1c71\u1c72\u1c73\u1c74\u1c75\u1c76\u1c77\u1ce9\u1cea\u1ceb\u1cec\u1cee\u1cef\u1cf0\u1cf1\u2135\u2136\u2137\u2138\u2d30\u2d31\u2d32\u2d33\u2d34\u2d35\u2d36\u2d37\u2d38\u2d39\u2d3a\u2d3b\u2d3c\u2d3d\u2d3e\u2d3f\u2d40\u2d41\u2d42\u2d43\u2d44\u2d45\u2d46\u2d47\u2d48\u2d49\u2d4a\u2d4b\u2d4c\u2d4d\u2d4e\u2d4f\u2d50\u2d51\u2d52\u2d53\u2d54\u2d55\u2d56\u2d57\u2d58\u2d59\u2d5a\u2d5b\u2d5c\u2d5d\u2d5e\u2d5f\u2d60\u2d61\u2d62\u2d63\u2d64\u2d65\u2d80\u2d81\u2d82\u2d83\u2d84\u2d85\u2d86\u2d87\u2d88\u2d89\u2d8a\u2d8b\u2d8c\u2d8d\u2d8e\u2d8f\u2d90\u2d91\u2d92\u2d93\u2d94\u2d95\u2d96\u2da0\u2da1\u2da2\u2da3\u2da4\u2da5\u2da6\u2da8\u2da9\u2daa\u2dab\u2dac\u2dad\u2dae\u2db0\u2db1\u2db2\u2db3\u2db4\u2db5\u2db6\u2db8\u2db9\u2dba\u2dbb\u2dbc\u2dbd\u2dbe\u2dc0\u2dc1\u2dc2\u2dc3\u2dc4\u2dc5\u2dc6\u2dc8\u2dc9\u2dca\u2dcb\u2dcc\u2dcd\u2dce\u2dd0\u2dd1\u2dd2\u2dd3\u2dd4\u2dd5\u2dd6\u2dd8\u2dd9\u2dda\u2ddb\u2ddc\u2ddd\u2dde\u3006\u303c\u3041\u3042\u3043\u3044\u3045\u3046\u3047\u3048\u3049\u304a\u304b\u304c\u304d\u304e\u304f\u3050\u3051\u3052\u3053\u3054\u3055\u3056\u3057\u3058\u3059\u305a\u305b\u305c\u305d\u305e\u305f\u3060\u3061\u3062\u3063\u3064\u3065\u3066\u3067\u3068\u3069\u306a\u306b\u306c\u306d\u306e\u306f\u3070\u3071\u3072\u3073\u3074\u3075\u3076\u3077\u3078\u3079\u307a\u307b\u307c\u307d\u307e\u307f\u3080\u3081\u3082\u3083\u3084\u3085\u3086\u3087\u3088\u3089\u308a\u308b\u308c\u308d\u308e\u308f\u3090\u3091\u3092\u3093\u3094\u3095\u3096\u309f\u30a1\u30a2\u30a3\u30a4\u30a5\u30a6\u30a7\u30a8\u30a9\u30aa\u30ab\u30ac\u30ad\u30ae\u30af\u30b0\u30b1\u30b2\u30b3\u30b4\u30b5\u30b6\u30b7\u30b8\u30b9\u30ba\u30bb\u30bc\u30bd\u30be\u30bf\u30c0\u30c1\u30c2\u30c3\u30c4\u30c5\u30c6\u30c7\u30c8\u30c9\u30ca\u30cb\u30cc\u30cd\u30ce\u30cf\u30d0\u30d1\u30d2\u30d3\u30d4\u30d5\u30d6\u30d7\u30d8\u30d9\u30da\u30db\u30dc\u30dd\u30de\u30df\u30e0\u30e1\u30e2\u30e3\u30e4\u30e5\u30e6\u30e7\u30e8\u30e9\u30ea\u30eb\u30ec\u30ed\u30ee\u30ef\u30f0\u30f1\u30f2\u30f3\u30f4\u30f5\u30f6\u30f7\u30f8\u30f9\u30fa\u30ff\u3105\u3106\u3107\u3108\u3109\u310a\u310b\u310c\u310d\u310e\u310f\u3110\u3111\u3112\u3113\u3114\u3115\u3116\u3117\u3118\u3119\u311a\u311b\u311c\u311d\u311e\u311f\u3120\u3121\u3122\u3123\u3124\u3125\u3126\u3127\u3128\u3129\u312a\u312b\u312c\u312d\u3131\u3132\u3133\u3134\u3135\u3136\u3137\u3138\u3139\u313a\u313b\u313c\u313d\u313e\u313f\u3140\u3141\u3142\u3143\u3144\u3145\u3146\u3147\u3148\u3149\u314a\u314b\u314c\u314d\u314e\u314f\u3150\u3151\u3152\u3153\u3154\u3155\u3156\u3157\u3158\u3159\u315a\u315b\u315c\u315d\u315e\u315f\u3160\u3161\u3162\u3163\u3164\u3165\u3166\u3167\u3168\u3169\u316a\u316b\u316c\u316d\u316e\u316f\u3170\u3171\u3172\u3173\u3174\u3175\u3176\u3177\u3178\u3179\u317a\u317b\u317c\u317d\u317e\u317f\u3180\u3181\u3182\u3183\u3184\u3185\u3186\u3187\u3188\u3189\u318a\u318b\u318c\u318d\u318e\u31a0\u31a1\u31a2\u31a3\u31a4\u31a5\u31a6\u31a7\u31a8\u31a9\u31aa\u31ab\u31ac\u31ad\u31ae\u31af\u31b0\u31b1\u31b2\u31b3\u31b4\u31b5\u31b6\u31b7\u31b8\u31b9\u31ba\u31f0\u31f1\u31f2\u31f3\u31f4\u31f5\u31f6\u31f7\u31f8\u31f9\u31fa\u31fb\u31fc\u31fd\u31fe\u31ff\u3400\u4db5\u4e00\u9fcb\ua000\ua001\ua002\ua003\ua004\ua005\ua006\ua007\ua008\ua009\ua00a\ua00b\ua00c\ua00d\ua00e\ua00f\ua010\ua011\ua012\ua013\ua014\ua016\ua017\ua018\ua019\ua01a\ua01b\ua01c\ua01d\ua01e\ua01f\ua020\ua021\ua022\ua023\ua024\ua025\ua026\ua027\ua028\ua029\ua02a\ua02b\ua02c\ua02d\ua02e\ua02f\ua030\ua031\ua032\ua033\ua034\ua035\ua036\ua037\ua038\ua039\ua03a\ua03b\ua03c\ua03d\ua03e\ua03f\ua040\ua041\ua042\ua043\ua044\ua045\ua046\ua047\ua048\ua049\ua04a\ua04b\ua04c\ua04d\ua04e\ua04f\ua050\ua051\ua052\ua053\ua054\ua055\ua056\ua057\ua058\ua059\ua05a\ua05b\ua05c\ua05d\ua05e\ua05f\ua060\ua061\ua062\ua063\ua064\ua065\ua066\ua067\ua068\ua069\ua06a\ua06b\ua06c\ua06d\ua06e\ua06f\ua070\ua071\ua072\ua073\ua074\ua075\ua076\ua077\ua078\ua079\ua07a\ua07b\ua07c\ua07d\ua07e\ua07f\ua080\ua081\ua082\ua083\ua084\ua085\ua086\ua087\ua088\ua089\ua08a\ua08b\ua08c\ua08d\ua08e\ua08f\ua090\ua091\ua092\ua093\ua094\ua095\ua096\ua097\ua098\ua099\ua09a\ua09b\ua09c\ua09d\ua09e\ua09f\ua0a0\ua0a1\ua0a2\ua0a3\ua0a4\ua0a5\ua0a6\ua0a7\ua0a8\ua0a9\ua0aa\ua0ab\ua0ac\ua0ad\ua0ae\ua0af\ua0b0\ua0b1\ua0b2\ua0b3\ua0b4\ua0b5\ua0b6\ua0b7\ua0b8\ua0b9\ua0ba\ua0bb\ua0bc\ua0bd\ua0be\ua0bf\ua0c0\ua0c1\ua0c2\ua0c3\ua0c4\ua0c5\ua0c6\ua0c7\ua0c8\ua0c9\ua0ca\ua0cb\ua0cc\ua0cd\ua0ce\ua0cf\ua0d0\ua0d1\ua0d2\ua0d3\ua0d4\ua0d5\ua0d6\ua0d7\ua0d8\ua0d9\ua0da\ua0db\ua0dc\ua0dd\ua0de\ua0df\ua0e0\ua0e1\ua0e2\ua0e3\ua0e4\ua0e5\ua0e6\ua0e7\ua0e8\ua0e9\ua0ea\ua0eb\ua0ec\ua0ed\ua0ee\ua0ef\ua0f0\ua0f1\ua0f2\ua0f3\ua0f4\ua0f5\ua0f6\ua0f7\ua0f8\ua0f9\ua0fa\ua0fb\ua0fc\ua0fd\ua0fe\ua0ff\ua100\ua101\ua102\ua103\ua104\ua105\ua106\ua107\ua108\ua109\ua10a\ua10b\ua10c\ua10d\ua10e\ua10f\ua110\ua111\ua112\ua113\ua114\ua115\ua116\ua117\ua118\ua119\ua11a\ua11b\ua11c\ua11d\ua11e\ua11f\ua120\ua121\ua122\ua123\ua124\ua125\ua126\ua127\ua128\ua129\ua12a\ua12b\ua12c\ua12d\ua12e\ua12f\ua130\ua131\ua132\ua133\ua134\ua135\ua136\ua137\ua138\ua139\ua13a\ua13b\ua13c\ua13d\ua13e\ua13f\ua140\ua141\ua142\ua143\ua144\ua145\ua146\ua147\ua148\ua149\ua14a\ua14b\ua14c\ua14d\ua14e\ua14f\ua150\ua151\ua152\ua153\ua154\ua155\ua156\ua157\ua158\ua159\ua15a\ua15b\ua15c\ua15d\ua15e\ua15f\ua160\ua161\ua162\ua163\ua164\ua165\ua166\ua167\ua168\ua169\ua16a\ua16b\ua16c\ua16d\ua16e\ua16f\ua170\ua171\ua172\ua173\ua174\ua175\ua176\ua177\ua178\ua179\ua17a\ua17b\ua17c\ua17d\ua17e\ua17f\ua180\ua181\ua182\ua183\ua184\ua185\ua186\ua187\ua188\ua189\ua18a\ua18b\ua18c\ua18d\ua18e\ua18f\ua190\ua191\ua192\ua193\ua194\ua195\ua196\ua197\ua198\ua199\ua19a\ua19b\ua19c\ua19d\ua19e\ua19f\ua1a0\ua1a1\ua1a2\ua1a3\ua1a4\ua1a5\ua1a6\ua1a7\ua1a8\ua1a9\ua1aa\ua1ab\ua1ac\ua1ad\ua1ae\ua1af\ua1b0\ua1b1\ua1b2\ua1b3\ua1b4\ua1b5\ua1b6\ua1b7\ua1b8\ua1b9\ua1ba\ua1bb\ua1bc\ua1bd\ua1be\ua1bf\ua1c0\ua1c1\ua1c2\ua1c3\ua1c4\ua1c5\ua1c6\ua1c7\ua1c8\ua1c9\ua1ca\ua1cb\ua1cc\ua1cd\ua1ce\ua1cf\ua1d0\ua1d1\ua1d2\ua1d3\ua1d4\ua1d5\ua1d6\ua1d7\ua1d8\ua1d9\ua1da\ua1db\ua1dc\ua1dd\ua1de\ua1df\ua1e0\ua1e1\ua1e2\ua1e3\ua1e4\ua1e5\ua1e6\ua1e7\ua1e8\ua1e9\ua1ea\ua1eb\ua1ec\ua1ed\ua1ee\ua1ef\ua1f0\ua1f1\ua1f2\ua1f3\ua1f4\ua1f5\ua1f6\ua1f7\ua1f8\ua1f9\ua1fa\ua1fb\ua1fc\ua1fd\ua1fe\ua1ff\ua200\ua201\ua202\ua203\ua204\ua205\ua206\ua207\ua208\ua209\ua20a\ua20b\ua20c\ua20d\ua20e\ua20f\ua210\ua211\ua212\ua213\ua214\ua215\ua216\ua217\ua218\ua219\ua21a\ua21b\ua21c\ua21d\ua21e\ua21f\ua220\ua221\ua222\ua223\ua224\ua225\ua226\ua227\ua228\ua229\ua22a\ua22b\ua22c\ua22d\ua22e\ua22f\ua230\ua231\ua232\ua233\ua234\ua235\ua236\ua237\ua238\ua239\ua23a\ua23b\ua23c\ua23d\ua23e\ua23f\ua240\ua241\ua242\ua243\ua244\ua245\ua246\ua247\ua248\ua249\ua24a\ua24b\ua24c\ua24d\ua24e\ua24f\ua250\ua251\ua252\ua253\ua254\ua255\ua256\ua257\ua258\ua259\ua25a\ua25b\ua25c\ua25d\ua25e\ua25f\ua260\ua261\ua262\ua263\ua264\ua265\ua266\ua267\ua268\ua269\ua26a\ua26b\ua26c\ua26d\ua26e\ua26f\ua270\ua271\ua272\ua273\ua274\ua275\ua276\ua277\ua278\ua279\ua27a\ua27b\ua27c\ua27d\ua27e\ua27f\ua280\ua281\ua282\ua283\ua284\ua285\ua286\ua287\ua288\ua289\ua28a\ua28b\ua28c\ua28d\ua28e\ua28f\ua290\ua291\ua292\ua293\ua294\ua295\ua296\ua297\ua298\ua299\ua29a\ua29b\ua29c\ua29d\ua29e\ua29f\ua2a0\ua2a1\ua2a2\ua2a3\ua2a4\ua2a5\ua2a6\ua2a7\ua2a8\ua2a9\ua2aa\ua2ab\ua2ac\ua2ad\ua2ae\ua2af\ua2b0\ua2b1\ua2b2\ua2b3\ua2b4\ua2b5\ua2b6\ua2b7\ua2b8\ua2b9\ua2ba\ua2bb\ua2bc\ua2bd\ua2be\ua2bf\ua2c0\ua2c1\ua2c2\ua2c3\ua2c4\ua2c5\ua2c6\ua2c7\ua2c8\ua2c9\ua2ca\ua2cb\ua2cc\ua2cd\ua2ce\ua2cf\ua2d0\ua2d1\ua2d2\ua2d3\ua2d4\ua2d5\ua2d6\ua2d7\ua2d8\ua2d9\ua2da\ua2db\ua2dc\ua2dd\ua2de\ua2df\ua2e0\ua2e1\ua2e2\ua2e3\ua2e4\ua2e5\ua2e6\ua2e7\ua2e8\ua2e9\ua2ea\ua2eb\ua2ec\ua2ed\ua2ee\ua2ef\ua2f0\ua2f1\ua2f2\ua2f3\ua2f4\ua2f5\ua2f6\ua2f7\ua2f8\ua2f9\ua2fa\ua2fb\ua2fc\ua2fd\ua2fe\ua2ff\ua300\ua301\ua302\ua303\ua304\ua305\ua306\ua307\ua308\ua309\ua30a\ua30b\ua30c\ua30d\ua30e\ua30f\ua310\ua311\ua312\ua313\ua314\ua315\ua316\ua317\ua318\ua319\ua31a\ua31b\ua31c\ua31d\ua31e\ua31f\ua320\ua321\ua322\ua323\ua324\ua325\ua326\ua327\ua328\ua329\ua32a\ua32b\ua32c\ua32d\ua32e\ua32f\ua330\ua331\ua332\ua333\ua334\ua335\ua336\ua337\ua338\ua339\ua33a\ua33b\ua33c\ua33d\ua33e\ua33f\ua340\ua341\ua342\ua343\ua344\ua345\ua346\ua347\ua348\ua349\ua34a\ua34b\ua34c\ua34d\ua34e\ua34f\ua350\ua351\ua352\ua353\ua354\ua355\ua356\ua357\ua358\ua359\ua35a\ua35b\ua35c\ua35d\ua35e\ua35f\ua360\ua361\ua362\ua363\ua364\ua365\ua366\ua367\ua368\ua369\ua36a\ua36b\ua36c\ua36d\ua36e\ua36f\ua370\ua371\ua372\ua373\ua374\ua375\ua376\ua377\ua378\ua379\ua37a\ua37b\ua37c\ua37d\ua37e\ua37f\ua380\ua381\ua382\ua383\ua384\ua385\ua386\ua387\ua388\ua389\ua38a\ua38b\ua38c\ua38d\ua38e\ua38f\ua390\ua391\ua392\ua393\ua394\ua395\ua396\ua397\ua398\ua399\ua39a\ua39b\ua39c\ua39d\ua39e\ua39f\ua3a0\ua3a1\ua3a2\ua3a3\ua3a4\ua3a5\ua3a6\ua3a7\ua3a8\ua3a9\ua3aa\ua3ab\ua3ac\ua3ad\ua3ae\ua3af\ua3b0\ua3b1\ua3b2\ua3b3\ua3b4\ua3b5\ua3b6\ua3b7\ua3b8\ua3b9\ua3ba\ua3bb\ua3bc\ua3bd\ua3be\ua3bf\ua3c0\ua3c1\ua3c2\ua3c3\ua3c4\ua3c5\ua3c6\ua3c7\ua3c8\ua3c9\ua3ca\ua3cb\ua3cc\ua3cd\ua3ce\ua3cf\ua3d0\ua3d1\ua3d2\ua3d3\ua3d4\ua3d5\ua3d6\ua3d7\ua3d8\ua3d9\ua3da\ua3db\ua3dc\ua3dd\ua3de\ua3df\ua3e0\ua3e1\ua3e2\ua3e3\ua3e4\ua3e5\ua3e6\ua3e7\ua3e8\ua3e9\ua3ea\ua3eb\ua3ec\ua3ed\ua3ee\ua3ef\ua3f0\ua3f1\ua3f2\ua3f3\ua3f4\ua3f5\ua3f6\ua3f7\ua3f8\ua3f9\ua3fa\ua3fb\ua3fc\ua3fd\ua3fe\ua3ff\ua400\ua401\ua402\ua403\ua404\ua405\ua406\ua407\ua408\ua409\ua40a\ua40b\ua40c\ua40d\ua40e\ua40f\ua410\ua411\ua412\ua413\ua414\ua415\ua416\ua417\ua418\ua419\ua41a\ua41b\ua41c\ua41d\ua41e\ua41f\ua420\ua421\ua422\ua423\ua424\ua425\ua426\ua427\ua428\ua429\ua42a\ua42b\ua42c\ua42d\ua42e\ua42f\ua430\ua431\ua432\ua433\ua434\ua435\ua436\ua437\ua438\ua439\ua43a\ua43b\ua43c\ua43d\ua43e\ua43f\ua440\ua441\ua442\ua443\ua444\ua445\ua446\ua447\ua448\ua449\ua44a\ua44b\ua44c\ua44d\ua44e\ua44f\ua450\ua451\ua452\ua453\ua454\ua455\ua456\ua457\ua458\ua459\ua45a\ua45b\ua45c\ua45d\ua45e\ua45f\ua460\ua461\ua462\ua463\ua464\ua465\ua466\ua467\ua468\ua469\ua46a\ua46b\ua46c\ua46d\ua46e\ua46f\ua470\ua471\ua472\ua473\ua474\ua475\ua476\ua477\ua478\ua479\ua47a\ua47b\ua47c\ua47d\ua47e\ua47f\ua480\ua481\ua482\ua483\ua484\ua485\ua486\ua487\ua488\ua489\ua48a\ua48b\ua48c\ua4d0\ua4d1\ua4d2\ua4d3\ua4d4\ua4d5\ua4d6\ua4d7\ua4d8\ua4d9\ua4da\ua4db\ua4dc\ua4dd\ua4de\ua4df\ua4e0\ua4e1\ua4e2\ua4e3\ua4e4\ua4e5\ua4e6\ua4e7\ua4e8\ua4e9\ua4ea\ua4eb\ua4ec\ua4ed\ua4ee\ua4ef\ua4f0\ua4f1\ua4f2\ua4f3\ua4f4\ua4f5\ua4f6\ua4f7\ua500\ua501\ua502\ua503\ua504\ua505\ua506\ua507\ua508\ua509\ua50a\ua50b\ua50c\ua50d\ua50e\ua50f\ua510\ua511\ua512\ua513\ua514\ua515\ua516\ua517\ua518\ua519\ua51a\ua51b\ua51c\ua51d\ua51e\ua51f\ua520\ua521\ua522\ua523\ua524\ua525\ua526\ua527\ua528\ua529\ua52a\ua52b\ua52c\ua52d\ua52e\ua52f\ua530\ua531\ua532\ua533\ua534\ua535\ua536\ua537\ua538\ua539\ua53a\ua53b\ua53c\ua53d\ua53e\ua53f\ua540\ua541\ua542\ua543\ua544\ua545\ua546\ua547\ua548\ua549\ua54a\ua54b\ua54c\ua54d\ua54e\ua54f\ua550\ua551\ua552\ua553\ua554\ua555\ua556\ua557\ua558\ua559\ua55a\ua55b\ua55c\ua55d\ua55e\ua55f\ua560\ua561\ua562\ua563\ua564\ua565\ua566\ua567\ua568\ua569\ua56a\ua56b\ua56c\ua56d\ua56e\ua56f\ua570\ua571\ua572\ua573\ua574\ua575\ua576\ua577\ua578\ua579\ua57a\ua57b\ua57c\ua57d\ua57e\ua57f\ua580\ua581\ua582\ua583\ua584\ua585\ua586\ua587\ua588\ua589\ua58a\ua58b\ua58c\ua58d\ua58e\ua58f\ua590\ua591\ua592\ua593\ua594\ua595\ua596\ua597\ua598\ua599\ua59a\ua59b\ua59c\ua59d\ua59e\ua59f\ua5a0\ua5a1\ua5a2\ua5a3\ua5a4\ua5a5\ua5a6\ua5a7\ua5a8\ua5a9\ua5aa\ua5ab\ua5ac\ua5ad\ua5ae\ua5af\ua5b0\ua5b1\ua5b2\ua5b3\ua5b4\ua5b5\ua5b6\ua5b7\ua5b8\ua5b9\ua5ba\ua5bb\ua5bc\ua5bd\ua5be\ua5bf\ua5c0\ua5c1\ua5c2\ua5c3\ua5c4\ua5c5\ua5c6\ua5c7\ua5c8\ua5c9\ua5ca\ua5cb\ua5cc\ua5cd\ua5ce\ua5cf\ua5d0\ua5d1\ua5d2\ua5d3\ua5d4\ua5d5\ua5d6\ua5d7\ua5d8\ua5d9\ua5da\ua5db\ua5dc\ua5dd\ua5de\ua5df\ua5e0\ua5e1\ua5e2\ua5e3\ua5e4\ua5e5\ua5e6\ua5e7\ua5e8\ua5e9\ua5ea\ua5eb\ua5ec\ua5ed\ua5ee\ua5ef\ua5f0\ua5f1\ua5f2\ua5f3\ua5f4\ua5f5\ua5f6\ua5f7\ua5f8\ua5f9\ua5fa\ua5fb\ua5fc\ua5fd\ua5fe\ua5ff\ua600\ua601\ua602\ua603\ua604\ua605\ua606\ua607\ua608\ua609\ua60a\ua60b\ua610\ua611\ua612\ua613\ua614\ua615\ua616\ua617\ua618\ua619\ua61a\ua61b\ua61c\ua61d\ua61e\ua61f\ua62a\ua62b\ua66e\ua6a0\ua6a1\ua6a2\ua6a3\ua6a4\ua6a5\ua6a6\ua6a7\ua6a8\ua6a9\ua6aa\ua6ab\ua6ac\ua6ad\ua6ae\ua6af\ua6b0\ua6b1\ua6b2\ua6b3\ua6b4\ua6b5\ua6b6\ua6b7\ua6b8\ua6b9\ua6ba\ua6bb\ua6bc\ua6bd\ua6be\ua6bf\ua6c0\ua6c1\ua6c2\ua6c3\ua6c4\ua6c5\ua6c6\ua6c7\ua6c8\ua6c9\ua6ca\ua6cb\ua6cc\ua6cd\ua6ce\ua6cf\ua6d0\ua6d1\ua6d2\ua6d3\ua6d4\ua6d5\ua6d6\ua6d7\ua6d8\ua6d9\ua6da\ua6db\ua6dc\ua6dd\ua6de\ua6df\ua6e0\ua6e1\ua6e2\ua6e3\ua6e4\ua6e5\ua7fb\ua7fc\ua7fd\ua7fe\ua7ff\ua800\ua801\ua803\ua804\ua805\ua807\ua808\ua809\ua80a\ua80c\ua80d\ua80e\ua80f\ua810\ua811\ua812\ua813\ua814\ua815\ua816\ua817\ua818\ua819\ua81a\ua81b\ua81c\ua81d\ua81e\ua81f\ua820\ua821\ua822\ua840\ua841\ua842\ua843\ua844\ua845\ua846\ua847\ua848\ua849\ua84a\ua84b\ua84c\ua84d\ua84e\ua84f\ua850\ua851\ua852\ua853\ua854\ua855\ua856\ua857\ua858\ua859\ua85a\ua85b\ua85c\ua85d\ua85e\ua85f\ua860\ua861\ua862\ua863\ua864\ua865\ua866\ua867\ua868\ua869\ua86a\ua86b\ua86c\ua86d\ua86e\ua86f\ua870\ua871\ua872\ua873\ua882\ua883\ua884\ua885\ua886\ua887\ua888\ua889\ua88a\ua88b\ua88c\ua88d\ua88e\ua88f\ua890\ua891\ua892\ua893\ua894\ua895\ua896\ua897\ua898\ua899\ua89a\ua89b\ua89c\ua89d\ua89e\ua89f\ua8a0\ua8a1\ua8a2\ua8a3\ua8a4\ua8a5\ua8a6\ua8a7\ua8a8\ua8a9\ua8aa\ua8ab\ua8ac\ua8ad\ua8ae\ua8af\ua8b0\ua8b1\ua8b2\ua8b3\ua8f2\ua8f3\ua8f4\ua8f5\ua8f6\ua8f7\ua8fb\ua90a\ua90b\ua90c\ua90d\ua90e\ua90f\ua910\ua911\ua912\ua913\ua914\ua915\ua916\ua917\ua918\ua919\ua91a\ua91b\ua91c\ua91d\ua91e\ua91f\ua920\ua921\ua922\ua923\ua924\ua925\ua930\ua931\ua932\ua933\ua934\ua935\ua936\ua937\ua938\ua939\ua93a\ua93b\ua93c\ua93d\ua93e\ua93f\ua940\ua941\ua942\ua943\ua944\ua945\ua946\ua960\ua961\ua962\ua963\ua964\ua965\ua966\ua967\ua968\ua969\ua96a\ua96b\ua96c\ua96d\ua96e\ua96f\ua970\ua971\ua972\ua973\ua974\ua975\ua976\ua977\ua978\ua979\ua97a\ua97b\ua97c\ua984\ua985\ua986\ua987\ua988\ua989\ua98a\ua98b\ua98c\ua98d\ua98e\ua98f\ua990\ua991\ua992\ua993\ua994\ua995\ua996\ua997\ua998\ua999\ua99a\ua99b\ua99c\ua99d\ua99e\ua99f\ua9a0\ua9a1\ua9a2\ua9a3\ua9a4\ua9a5\ua9a6\ua9a7\ua9a8\ua9a9\ua9aa\ua9ab\ua9ac\ua9ad\ua9ae\ua9af\ua9b0\ua9b1\ua9b2\uaa00\uaa01\uaa02\uaa03\uaa04\uaa05\uaa06\uaa07\uaa08\uaa09\uaa0a\uaa0b\uaa0c\uaa0d\uaa0e\uaa0f\uaa10\uaa11\uaa12\uaa13\uaa14\uaa15\uaa16\uaa17\uaa18\uaa19\uaa1a\uaa1b\uaa1c\uaa1d\uaa1e\uaa1f\uaa20\uaa21\uaa22\uaa23\uaa24\uaa25\uaa26\uaa27\uaa28\uaa40\uaa41\uaa42\uaa44\uaa45\uaa46\uaa47\uaa48\uaa49\uaa4a\uaa4b\uaa60\uaa61\uaa62\uaa63\uaa64\uaa65\uaa66\uaa67\uaa68\uaa69\uaa6a\uaa6b\uaa6c\uaa6d\uaa6e\uaa6f\uaa71\uaa72\uaa73\uaa74\uaa75\uaa76\uaa7a\uaa80\uaa81\uaa82\uaa83\uaa84\uaa85\uaa86\uaa87\uaa88\uaa89\uaa8a\uaa8b\uaa8c\uaa8d\uaa8e\uaa8f\uaa90\uaa91\uaa92\uaa93\uaa94\uaa95\uaa96\uaa97\uaa98\uaa99\uaa9a\uaa9b\uaa9c\uaa9d\uaa9e\uaa9f\uaaa0\uaaa1\uaaa2\uaaa3\uaaa4\uaaa5\uaaa6\uaaa7\uaaa8\uaaa9\uaaaa\uaaab\uaaac\uaaad\uaaae\uaaaf\uaab1\uaab5\uaab6\uaab9\uaaba\uaabb\uaabc\uaabd\uaac0\uaac2\uaadb\uaadc\uab01\uab02\uab03\uab04\uab05\uab06\uab09\uab0a\uab0b\uab0c\uab0d\uab0e\uab11\uab12\uab13\uab14\uab15\uab16\uab20\uab21\uab22\uab23\uab24\uab25\uab26\uab28\uab29\uab2a\uab2b\uab2c\uab2d\uab2e\uabc0\uabc1\uabc2\uabc3\uabc4\uabc5\uabc6\uabc7\uabc8\uabc9\uabca\uabcb\uabcc\uabcd\uabce\uabcf\uabd0\uabd1\uabd2\uabd3\uabd4\uabd5\uabd6\uabd7\uabd8\uabd9\uabda\uabdb\uabdc\uabdd\uabde\uabdf\uabe0\uabe1\uabe2\uac00\ud7a3\ud7b0\ud7b1\ud7b2\ud7b3\ud7b4\ud7b5\ud7b6\ud7b7\ud7b8\ud7b9\ud7ba\ud7bb\ud7bc\ud7bd\ud7be\ud7bf\ud7c0\ud7c1\ud7c2\ud7c3\ud7c4\ud7c5\ud7c6\ud7cb\ud7cc\ud7cd\ud7ce\ud7cf\ud7d0\ud7d1\ud7d2\ud7d3\ud7d4\ud7d5\ud7d6\ud7d7\ud7d8\ud7d9\ud7da\ud7db\ud7dc\ud7dd\ud7de\ud7df\ud7e0\ud7e1\ud7e2\ud7e3\ud7e4\ud7e5\ud7e6\ud7e7\ud7e8\ud7e9\ud7ea\ud7eb\ud7ec\ud7ed\ud7ee\ud7ef\ud7f0\ud7f1\ud7f2\ud7f3\ud7f4\ud7f5\ud7f6\ud7f7\ud7f8\ud7f9\ud7fa\ud7fb\u8c48\u66f4\u8eca\u8cc8\u6ed1\u4e32\u53e5\u9f9c\u9f9c\u5951\u91d1\u5587\u5948\u61f6\u7669\u7f85\u863f\u87ba\u88f8\u908f\u6a02\u6d1b\u70d9\u73de\u843d\u916a\u99f1\u4e82\u5375\u6b04\u721b\u862d\u9e1e\u5d50\u6feb\u85cd\u8964\u62c9\u81d8\u881f\u5eca\u6717\u6d6a\u72fc\u90ce\u4f86\u51b7\u52de\u64c4\u6ad3\u7210\u76e7\u8001\u8606\u865c\u8def\u9732\u9b6f\u9dfa\u788c\u797f\u7da0\u83c9\u9304\u9e7f\u8ad6\u58df\u5f04\u7c60\u807e\u7262\u78ca\u8cc2\u96f7\u58d8\u5c62\u6a13\u6dda\u6f0f\u7d2f\u7e37\u964b\u52d2\u808b\u51dc\u51cc\u7a1c\u7dbe\u83f1\u9675\u8b80\u62cf\u6a02\u8afe\u4e39\u5be7\u6012\u7387\u7570\u5317\u78fb\u4fbf\u5fa9\u4e0d\u6ccc\u6578\u7d22\u53c3\u585e\u7701\u8449\u8aaa\u6bba\u8fb0\u6c88\u62fe\u82e5\u63a0\u7565\u4eae\u5169\u51c9\u6881\u7ce7\u826f\u8ad2\u91cf\u52f5\u5442\u5973\u5eec\u65c5\u6ffe\u792a\u95ad\u9a6a\u9e97\u9ece\u529b\u66c6\u6b77\u8f62\u5e74\u6190\u6200\u649a\u6f23\u7149\u7489\u79ca\u7df4\u806f\u8f26\u84ee\u9023\u934a\u5217\u52a3\u54bd\u70c8\u88c2\u8aaa\u5ec9\u5ff5\u637b\u6bae\u7c3e\u7375\u4ee4\u56f9\u5be7\u5dba\u601c\u73b2\u7469\u7f9a\u8046\u9234\u96f6\u9748\u9818\u4f8b\u79ae\u91b4\u96b8\u60e1\u4e86\u50da\u5bee\u5c3f\u6599\u6a02\u71ce\u7642\u84fc\u907c\u9f8d\u6688\u962e\u5289\u677b\u67f3\u6d41\u6e9c\u7409\u7559\u786b\u7d10\u985e\u516d\u622e\u9678\u502b\u5d19\u6dea\u8f2a\u5f8b\u6144\u6817\u7387\u9686\u5229\u540f\u5c65\u6613\u674e\u68a8\u6ce5\u7406\u75e2\u7f79\u88cf\u88e1\u91cc\u96e2\u533f\u6eba\u541d\u71d0\u7498\u85fa\u96a3\u9c57\u9e9f\u6797\u6dcb\u81e8\u7acb\u7b20\u7c92\u72c0\u7099\u8b58\u4ec0\u8336\u523a\u5207\u5ea6\u62d3\u7cd6\u5b85\u6d1e\u66b4\u8f3b\u884c\u964d\u898b\u5ed3\u5140\u55c0\ufa0e\ufa0f\u585a\ufa11\u6674\ufa13\ufa14\u51de\u732a\u76ca\u793c\u795e\u7965\u798f\u9756\u7cbe\u7fbd\ufa1f\u8612\ufa21\u8af8\ufa23\ufa24\u9038\u90fd\ufa27\ufa28\ufa29\u98ef\u98fc\u9928\u9db4\u4fae\u50e7\u514d\u52c9\u52e4\u5351\u559d\u5606\u5668\u5840\u58a8\u5c64\u5c6e\u6094\u6168\u618e\u61f2\u654f\u65e2\u6691\u6885\u6d77\u6e1a\u6f22\u716e\u722b\u7422\u7891\u793e\u7949\u7948\u7950\u7956\u795d\u798d\u798e\u7a40\u7a81\u7bc0\u7df4\u7e09\u7e41\u7f72\u8005\u81ed\u8279\u8279\u8457\u8910\u8996\u8b01\u8b39\u8cd3\u8d08\u8fb6\u9038\u96e3\u97ff\u983b\u6075\ud850\udeee\u8218\u4e26\u51b5\u5168\u4f80\u5145\u5180\u52c7\u52fa\u559d\u5555\u5599\u55e2\u585a\u58b3\u5944\u5954\u5a62\u5b28\u5ed2\u5ed9\u5f69\u5fad\u60d8\u614e\u6108\u618e\u6160\u61f2\u6234\u63c4\u641c\u6452\u6556\u6674\u6717\u671b\u6756\u6b79\u6bba\u6d41\u6edb\u6ecb\u6f22\u701e\u716e\u77a7\u7235\u72af\u732a\u7471\u7506\u753b\u761d\u761f\u76ca\u76db\u76f4\u774a\u7740\u78cc\u7ab1\u7bc0\u7c7b\u7d5b\u7df4\u7f3e\u8005\u8352\u83ef\u8779\u8941\u8986\u8996\u8abf\u8af8\u8acb\u8b01\u8afe\u8aed\u8b39\u8b8a\u8d08\u8f38\u9072\u9199\u9276\u967c\u96e3\u9756\u97db\u97ff\u980b\u983b\u9b12\u9f9c\ud84a\udc4a\ud84a\udc44\ud84c\udfd5\u3b9d\u4018\u4039\ud854\ude49\ud857\udcd0\ud85f\uded3\u9f43\u9f8e\u05d9\u05b4\u05f2\u05b7\ufb20\ufb21\ufb22\ufb23\ufb24\ufb25\ufb26\ufb27\ufb28\u05e9\u05c1\u05e9\u05c2\u05e9\u05bc\u05c1\u05e9\u05bc\u05c2\u05d0\u05b7\u05d0\u05b8\u05d0\u05bc\u05d1\u05bc\u05d2\u05bc\u05d3\u05bc\u05d4\u05bc\u05d5\u05bc\u05d6\u05bc\u05d8\u05bc\u05d9\u05bc\u05da\u05bc\u05db\u05bc\u05dc\u05bc\u05de\u05bc\u05e0\u05bc\u05e1\u05bc\u05e3\u05bc\u05e4\u05bc\u05e6\u05bc\u05e7\u05bc\u05e8\u05bc\u05e9\u05bc\u05ea\u05bc\u05d5\u05b9\u05d1\u05bf\u05db\u05bf\u05e4\u05bf\ufb4f\ufb50\ufb51\ufb52\ufb53\ufb54\ufb55\ufb56\ufb57\ufb58\ufb59\ufb5a\ufb5b\ufb5c\ufb5d\ufb5e\ufb5f\ufb60\ufb61\ufb62\ufb63\ufb64\ufb65\ufb66\ufb67\ufb68\ufb69\ufb6a\ufb6b\ufb6c\ufb6d\ufb6e\ufb6f\ufb70\ufb71\ufb72\ufb73\ufb74\ufb75\ufb76\ufb77\ufb78\ufb79\ufb7a\ufb7b\ufb7c\ufb7d\ufb7e\ufb7f\ufb80\ufb81\ufb82\ufb83\ufb84\ufb85\ufb86\ufb87\ufb88\ufb89\ufb8a\ufb8b\ufb8c\ufb8d\ufb8e\ufb8f\ufb90\ufb91\ufb92\ufb93\ufb94\ufb95\ufb96\ufb97\ufb98\ufb99\ufb9a\ufb9b\ufb9c\ufb9d\ufb9e\ufb9f\ufba0\ufba1\ufba2\ufba3\ufba4\ufba5\ufba6\ufba7\ufba8\ufba9\ufbaa\ufbab\ufbac\ufbad\ufbae\ufbaf\ufbb0\ufbb1\ufbd3\ufbd4\ufbd5\ufbd6\ufbd7\ufbd8\ufbd9\ufbda\ufbdb\ufbdc\ufbdd\ufbde\ufbdf\ufbe0\ufbe1\ufbe2\ufbe3\ufbe4\ufbe5\ufbe6\ufbe7\ufbe8\ufbe9\ufbea\ufbeb\ufbec\ufbed\ufbee\ufbef\ufbf0\ufbf1\ufbf2\ufbf3\ufbf4\ufbf5\ufbf6\ufbf7\ufbf8\ufbf9\ufbfa\ufbfb\ufbfc\ufbfd\ufbfe\ufbff\ufc00\ufc01\ufc02\ufc03\ufc04\ufc05\ufc06\ufc07\ufc08\ufc09\ufc0a\ufc0b\ufc0c\ufc0d\ufc0e\ufc0f\ufc10\ufc11\ufc12\ufc13\ufc14\ufc15\ufc16\ufc17\ufc18\ufc19\ufc1a\ufc1b\ufc1c\ufc1d\ufc1e\ufc1f\ufc20\ufc21\ufc22\ufc23\ufc24\ufc25\ufc26\ufc27\ufc28\ufc29\ufc2a\ufc2b\ufc2c\ufc2d\ufc2e\ufc2f\ufc30\ufc31\ufc32\ufc33\ufc34\ufc35\ufc36\ufc37\ufc38\ufc39\ufc3a\ufc3b\ufc3c\ufc3d\ufc3e\ufc3f\ufc40\ufc41\ufc42\ufc43\ufc44\ufc45\ufc46\ufc47\ufc48\ufc49\ufc4a\ufc4b\ufc4c\ufc4d\ufc4e\ufc4f\ufc50\ufc51\ufc52\ufc53\ufc54\ufc55\ufc56\ufc57\ufc58\ufc59\ufc5a\ufc5b\ufc5c\ufc5d\ufc5e\ufc5f\ufc60\ufc61\ufc62\ufc63\ufc64\ufc65\ufc66\ufc67\ufc68\ufc69\ufc6a\ufc6b\ufc6c\ufc6d\ufc6e\ufc6f\ufc70\ufc71\ufc72\ufc73\ufc74\ufc75\ufc76\ufc77\ufc78\ufc79\ufc7a\ufc7b\ufc7c\ufc7d\ufc7e\ufc7f\ufc80\ufc81\ufc82\ufc83\ufc84\ufc85\ufc86\ufc87\ufc88\ufc89\ufc8a\ufc8b\ufc8c\ufc8d\ufc8e\ufc8f\ufc90\ufc91\ufc92\ufc93\ufc94\ufc95\ufc96\ufc97\ufc98\ufc99\ufc9a\ufc9b\ufc9c\ufc9d\ufc9e\ufc9f\ufca0\ufca1\ufca2\ufca3\ufca4\ufca5\ufca6\ufca7\ufca8\ufca9\ufcaa\ufcab\ufcac\ufcad\ufcae\ufcaf\ufcb0\ufcb1\ufcb2\ufcb3\ufcb4\ufcb5\ufcb6\ufcb7\ufcb8\ufcb9\ufcba\ufcbb\ufcbc\ufcbd\ufcbe\ufcbf\ufcc0\ufcc1\ufcc2\ufcc3\ufcc4\ufcc5\ufcc6\ufcc7\ufcc8\ufcc9\ufcca\ufccb\ufccc\ufccd\ufcce\ufccf\ufcd0\ufcd1\ufcd2\ufcd3\ufcd4\ufcd5\ufcd6\ufcd7\ufcd8\ufcd9\ufcda\ufcdb\ufcdc\ufcdd\ufcde\ufcdf\ufce0\ufce1\ufce2\ufce3\ufce4\ufce5\ufce6\ufce7\ufce8\ufce9\ufcea\ufceb\ufcec\ufced\ufcee\ufcef\ufcf0\ufcf1\ufcf2\ufcf3\ufcf4\ufcf5\ufcf6\ufcf7\ufcf8\ufcf9\ufcfa\ufcfb\ufcfc\ufcfd\ufcfe\ufcff\ufd00\ufd01\ufd02\ufd03\ufd04\ufd05\ufd06\ufd07\ufd08\ufd09\ufd0a\ufd0b\ufd0c\ufd0d\ufd0e\ufd0f\ufd10\ufd11\ufd12\ufd13\ufd14\ufd15\ufd16\ufd17\ufd18\ufd19\ufd1a\ufd1b\ufd1c\ufd1d\ufd1e\ufd1f\ufd20\ufd21\ufd22\ufd23\ufd24\ufd25\ufd26\ufd27\ufd28\ufd29\ufd2a\ufd2b\ufd2c\ufd2d\ufd2e\ufd2f\ufd30\ufd31\ufd32\ufd33\ufd34\ufd35\ufd36\ufd37\ufd38\ufd39\ufd3a\ufd3b\ufd3c\ufd3d\ufd50\ufd51\ufd52\ufd53\ufd54\ufd55\ufd56\ufd57\ufd58\ufd59\ufd5a\ufd5b\ufd5c\ufd5d\ufd5e\ufd5f\ufd60\ufd61\ufd62\ufd63\ufd64\ufd65\ufd66\ufd67\ufd68\ufd69\ufd6a\ufd6b\ufd6c\ufd6d\ufd6e\ufd6f\ufd70\ufd71\ufd72\ufd73\ufd74\ufd75\ufd76\ufd77\ufd78\ufd79\ufd7a\ufd7b\ufd7c\ufd7d\ufd7e\ufd7f\ufd80\ufd81\ufd82\ufd83\ufd84\ufd85\ufd86\ufd87\ufd88\ufd89\ufd8a\ufd8b\ufd8c\ufd8d\ufd8e\ufd8f\ufd92\ufd93\ufd94\ufd95\ufd96\ufd97\ufd98\ufd99\ufd9a\ufd9b\ufd9c\ufd9d\ufd9e\ufd9f\ufda0\ufda1\ufda2\ufda3\ufda4\ufda5\ufda6\ufda7\ufda8\ufda9\ufdaa\ufdab\ufdac\ufdad\ufdae\ufdaf\ufdb0\ufdb1\ufdb2\ufdb3\ufdb4\ufdb5\ufdb6\ufdb7\ufdb8\ufdb9\ufdba\ufdbb\ufdbc\ufdbd\ufdbe\ufdbf\ufdc0\ufdc1\ufdc2\ufdc3\ufdc4\ufdc5\ufdc6\ufdc7\ufdf0\ufdf1\ufdf2\ufdf3\ufdf4\ufdf5\ufdf6\ufdf7\ufdf8\ufdf9\ufdfa\ufdfb\ufe70\ufe71\ufe72\ufe73\ufe74\ufe76\ufe77\ufe78\ufe79\ufe7a\ufe7b\ufe7c\ufe7d\ufe7e\ufe7f\ufe80\ufe81\ufe82\ufe83\ufe84\ufe85\ufe86\ufe87\ufe88\ufe89\ufe8a\ufe8b\ufe8c\ufe8d\ufe8e\ufe8f\ufe90\ufe91\ufe92\ufe93\ufe94\ufe95\ufe96\ufe97\ufe98\ufe99\ufe9a\ufe9b\ufe9c\ufe9d\ufe9e\ufe9f\ufea0\ufea1\ufea2\ufea3\ufea4\ufea5\ufea6\ufea7\ufea8\ufea9\ufeaa\ufeab\ufeac\ufead\ufeae\ufeaf\ufeb0\ufeb1\ufeb2\ufeb3\ufeb4\ufeb5\ufeb6\ufeb7\ufeb8\ufeb9\ufeba\ufebb\ufebc\ufebd\ufebe\ufebf\ufec0\ufec1\ufec2\ufec3\ufec4\ufec5\ufec6\ufec7\ufec8\ufec9\ufeca\ufecb\ufecc\ufecd\ufece\ufecf\ufed0\ufed1\ufed2\ufed3\ufed4\ufed5\ufed6\ufed7\ufed8\ufed9\ufeda\ufedb\ufedc\ufedd\ufede\ufedf\ufee0\ufee1\ufee2\ufee3\ufee4\ufee5\ufee6\ufee7\ufee8\ufee9\ufeea\ufeeb\ufeec\ufeed\ufeee\ufeef\ufef0\ufef1\ufef2\ufef3\ufef4\ufef5\ufef6\ufef7\ufef8\ufef9\ufefa\ufefb\ufefc\uff66\uff67\uff68\uff69\uff6a\uff6b\uff6c\uff6d\uff6e\uff6f\uff71\uff72\uff73\uff74\uff75\uff76\uff77\uff78\uff79\uff7a\uff7b\uff7c\uff7d\uff7e\uff7f\uff80\uff81\uff82\uff83\uff84\uff85\uff86\uff87\uff88\uff89\uff8a\uff8b\uff8c\uff8d\uff8e\uff8f\uff90\uff91\uff92\uff93\uff94\uff95\uff96\uff97\uff98\uff99\uff9a\uff9b\uff9c\uff9d\uffa0\uffa1\uffa2\uffa3\uffa4\uffa5\uffa6\uffa7\uffa8\uffa9\uffaa\uffab\uffac\uffad\uffae\uffaf\uffb0\uffb1\uffb2\uffb3\uffb4\uffb5\uffb6\uffb7\uffb8\uffb9\uffba\uffbb\uffbc\uffbd\uffbe\uffc2\uffc3\uffc4\uffc5\uffc6\uffc7\uffca\uffcb\uffcc\uffcd\uffce\uffcf\uffd2\uffd3\uffd4\uffd5\uffd6\uffd7\uffda\uffdb\uffdc\ud800\udc00\ud800\udc01\ud800\udc02\ud800\udc03\ud800\udc04\ud800\udc05\ud800\udc06\ud800\udc07\ud800\udc08\ud800\udc09\ud800\udc0a\ud800\udc0b\ud800\udc0d\ud800\udc0e\ud800\udc0f\ud800\udc10\ud800\udc11\ud800\udc12\ud800\udc13\ud800\udc14\ud800\udc15\ud800\udc16\ud800\udc17\ud800\udc18\ud800\udc19\ud800\udc1a\ud800\udc1b\ud800\udc1c\ud800\udc1d\ud800\udc1e\ud800\udc1f\ud800\udc20\ud800\udc21\ud800\udc22\ud800\udc23\ud800\udc24\ud800\udc25\ud800\udc26\ud800\udc28\ud800\udc29\ud800\udc2a\ud800\udc2b\ud800\udc2c\ud800\udc2d\ud800\udc2e\ud800\udc2f\ud800\udc30\ud800\udc31\ud800\udc32\ud800\udc33\ud800\udc34\ud800\udc35\ud800\udc36\ud800\udc37\ud800\udc38\ud800\udc39\ud800\udc3a\ud800\udc3c\ud800\udc3d\ud800\udc3f\ud800\udc40\ud800\udc41\ud800\udc42\ud800\udc43\ud800\udc44\ud800\udc45\ud800\udc46\ud800\udc47\ud800\udc48\ud800\udc49\ud800\udc4a\ud800\udc4b\ud800\udc4c\ud800\udc4d\ud800\udc50\ud800\udc51\ud800\udc52\ud800\udc53\ud800\udc54\ud800\udc55\ud800\udc56\ud800\udc57\ud800\udc58\ud800\udc59\ud800\udc5a\ud800\udc5b\ud800\udc5c\ud800\udc5d\ud800\udc80\ud800\udc81\ud800\udc82\ud800\udc83\ud800\udc84\ud800\udc85\ud800\udc86\ud800\udc87\ud800\udc88\ud800\udc89\ud800\udc8a\ud800\udc8b\ud800\udc8c\ud800\udc8d\ud800\udc8e\ud800\udc8f\ud800\udc90\ud800\udc91\ud800\udc92\ud800\udc93\ud800\udc94\ud800\udc95\ud800\udc96\ud800\udc97\ud800\udc98\ud800\udc99\ud800\udc9a\ud800\udc9b\ud800\udc9c\ud800\udc9d\ud800\udc9e\ud800\udc9f\ud800\udca0\ud800\udca1\ud800\udca2\ud800\udca3\ud800\udca4\ud800\udca5\ud800\udca6\ud800\udca7\ud800\udca8\ud800\udca9\ud800\udcaa\ud800\udcab\ud800\udcac\ud800\udcad\ud800\udcae\ud800\udcaf\ud800\udcb0\ud800\udcb1\ud800\udcb2\ud800\udcb3\ud800\udcb4\ud800\udcb5\ud800\udcb6\ud800\udcb7\ud800\udcb8\ud800\udcb9\ud800\udcba\ud800\udcbb\ud800\udcbc\ud800\udcbd\ud800\udcbe\ud800\udcbf\ud800\udcc0\ud800\udcc1\ud800\udcc2\ud800\udcc3\ud800\udcc4\ud800\udcc5\ud800\udcc6\ud800\udcc7\ud800\udcc8\ud800\udcc9\ud800\udcca\ud800\udccb\ud800\udccc\ud800\udccd\ud800\udcce\ud800\udccf\ud800\udcd0\ud800\udcd1\ud800\udcd2\ud800\udcd3\ud800\udcd4\ud800\udcd5\ud800\udcd6\ud800\udcd7\ud800\udcd8\ud800\udcd9\ud800\udcda\ud800\udcdb\ud800\udcdc\ud800\udcdd\ud800\udcde\ud800\udcdf\ud800\udce0\ud800\udce1\ud800\udce2\ud800\udce3\ud800\udce4\ud800\udce5\ud800\udce6\ud800\udce7\ud800\udce8\ud800\udce9\ud800\udcea\ud800\udceb\ud800\udcec\ud800\udced\ud800\udcee\ud800\udcef\ud800\udcf0\ud800\udcf1\ud800\udcf2\ud800\udcf3\ud800\udcf4\ud800\udcf5\ud800\udcf6\ud800\udcf7\ud800\udcf8\ud800\udcf9\ud800\udcfa\ud800\ude80\ud800\ude81\ud800\ude82\ud800\ude83\ud800\ude84\ud800\ude85\ud800\ude86\ud800\ude87\ud800\ude88\ud800\ude89\ud800\ude8a\ud800\ude8b\ud800\ude8c\ud800\ude8d\ud800\ude8e\ud800\ude8f\ud800\ude90\ud800\ude91\ud800\ude92\ud800\ude93\ud800\ude94\ud800\ude95\ud800\ude96\ud800\ude97\ud800\ude98\ud800\ude99\ud800\ude9a\ud800\ude9b\ud800\ude9c\ud800\udea0\ud800\udea1\ud800\udea2\ud800\udea3\ud800\udea4\ud800\udea5\ud800\udea6\ud800\udea7\ud800\udea8\ud800\udea9\ud800\udeaa\ud800\udeab\ud800\udeac\ud800\udead\ud800\udeae\ud800\udeaf\ud800\udeb0\ud800\udeb1\ud800\udeb2\ud800\udeb3\ud800\udeb4\ud800\udeb5\ud800\udeb6\ud800\udeb7\ud800\udeb8\ud800\udeb9\ud800\udeba\ud800\udebb\ud800\udebc\ud800\udebd\ud800\udebe\ud800\udebf\ud800\udec0\ud800\udec1\ud800\udec2\ud800\udec3\ud800\udec4\ud800\udec5\ud800\udec6\ud800\udec7\ud800\udec8\ud800\udec9\ud800\udeca\ud800\udecb\ud800\udecc\ud800\udecd\ud800\udece\ud800\udecf\ud800\uded0\ud800\udf00\ud800\udf01\ud800\udf02\ud800\udf03\ud800\udf04\ud800\udf05\ud800\udf06\ud800\udf07\ud800\udf08\ud800\udf09\ud800\udf0a\ud800\udf0b\ud800\udf0c\ud800\udf0d\ud800\udf0e\ud800\udf0f\ud800\udf10\ud800\udf11\ud800\udf12\ud800\udf13\ud800\udf14\ud800\udf15\ud800\udf16\ud800\udf17\ud800\udf18\ud800\udf19\ud800\udf1a\ud800\udf1b\ud800\udf1c\ud800\udf1d\ud800\udf1e\ud800\udf30\ud800\udf31\ud800\udf32\ud800\udf33\ud800\udf34\ud800\udf35\ud800\udf36\ud800\udf37\ud800\udf38\ud800\udf39\ud800\udf3a\ud800\udf3b\ud800\udf3c\ud800\udf3d\ud800\udf3e\ud800\udf3f\ud800\udf40\ud800\udf42\ud800\udf43\ud800\udf44\ud800\udf45\ud800\udf46\ud800\udf47\ud800\udf48\ud800\udf49\ud800\udf80\ud800\udf81\ud800\udf82\ud800\udf83\ud800\udf84\ud800\udf85\ud800\udf86\ud800\udf87\ud800\udf88\ud800\udf89\ud800\udf8a\ud800\udf8b\ud800\udf8c\ud800\udf8d\ud800\udf8e\ud800\udf8f\ud800\udf90\ud800\udf91\ud800\udf92\ud800\udf93\ud800\udf94\ud800\udf95\ud800\udf96\ud800\udf97\ud800\udf98\ud800\udf99\ud800\udf9a\ud800\udf9b\ud800\udf9c\ud800\udf9d\ud800\udfa0\ud800\udfa1\ud800\udfa2\ud800\udfa3\ud800\udfa4\ud800\udfa5\ud800\udfa6\ud800\udfa7\ud800\udfa8\ud800\udfa9\ud800\udfaa\ud800\udfab\ud800\udfac\ud800\udfad\ud800\udfae\ud800\udfaf\ud800\udfb0\ud800\udfb1\ud800\udfb2\ud800\udfb3\ud800\udfb4\ud800\udfb5\ud800\udfb6\ud800\udfb7\ud800\udfb8\ud800\udfb9\ud800\udfba\ud800\udfbb\ud800\udfbc\ud800\udfbd\ud800\udfbe\ud800\udfbf\ud800\udfc0\ud800\udfc1\ud800\udfc2\ud800\udfc3\ud800\udfc8\ud800\udfc9\ud800\udfca\ud800\udfcb\ud800\udfcc\ud800\udfcd\ud800\udfce\ud800\udfcf\ud801\udc50\ud801\udc51\ud801\udc52\ud801\udc53\ud801\udc54\ud801\udc55\ud801\udc56\ud801\udc57\ud801\udc58\ud801\udc59\ud801\udc5a\ud801\udc5b\ud801\udc5c\ud801\udc5d\ud801\udc5e\ud801\udc5f\ud801\udc60\ud801\udc61\ud801\udc62\ud801\udc63\ud801\udc64\ud801\udc65\ud801\udc66\ud801\udc67\ud801\udc68\ud801\udc69\ud801\udc6a\ud801\udc6b\ud801\udc6c\ud801\udc6d\ud801\udc6e\ud801\udc6f\ud801\udc70\ud801\udc71\ud801\udc72\ud801\udc73\ud801\udc74\ud801\udc75\ud801\udc76\ud801\udc77\ud801\udc78\ud801\udc79\ud801\udc7a\ud801\udc7b\ud801\udc7c\ud801\udc7d\ud801\udc7e\ud801\udc7f\ud801\udc80\ud801\udc81\ud801\udc82\ud801\udc83\ud801\udc84\ud801\udc85\ud801\udc86\ud801\udc87\ud801\udc88\ud801\udc89\ud801\udc8a\ud801\udc8b\ud801\udc8c\ud801\udc8d\ud801\udc8e\ud801\udc8f\ud801\udc90\ud801\udc91\ud801\udc92\ud801\udc93\ud801\udc94\ud801\udc95\ud801\udc96\ud801\udc97\ud801\udc98\ud801\udc99\ud801\udc9a\ud801\udc9b\ud801\udc9c\ud801\udc9d\ud802\udc00\ud802\udc01\ud802\udc02\ud802\udc03\ud802\udc04\ud802\udc05\ud802\udc08\ud802\udc0a\ud802\udc0b\ud802\udc0c\ud802\udc0d\ud802\udc0e\ud802\udc0f\ud802\udc10\ud802\udc11\ud802\udc12\ud802\udc13\ud802\udc14\ud802\udc15\ud802\udc16\ud802\udc17\ud802\udc18\ud802\udc19\ud802\udc1a\ud802\udc1b\ud802\udc1c\ud802\udc1d\ud802\udc1e\ud802\udc1f\ud802\udc20\ud802\udc21\ud802\udc22\ud802\udc23\ud802\udc24\ud802\udc25\ud802\udc26\ud802\udc27\ud802\udc28\ud802\udc29\ud802\udc2a\ud802\udc2b\ud802\udc2c\ud802\udc2d\ud802\udc2e\ud802\udc2f\ud802\udc30\ud802\udc31\ud802\udc32\ud802\udc33\ud802\udc34\ud802\udc35\ud802\udc37\ud802\udc38\ud802\udc3c\ud802\udc3f\ud802\udc40\ud802\udc41\ud802\udc42\ud802\udc43\ud802\udc44\ud802\udc45\ud802\udc46\ud802\udc47\ud802\udc48\ud802\udc49\ud802\udc4a\ud802\udc4b\ud802\udc4c\ud802\udc4d\ud802\udc4e\ud802\udc4f\ud802\udc50\ud802\udc51\ud802\udc52\ud802\udc53\ud802\udc54\ud802\udc55\ud802\udd00\ud802\udd01\ud802\udd02\ud802\udd03\ud802\udd04\ud802\udd05\ud802\udd06\ud802\udd07\ud802\udd08\ud802\udd09\ud802\udd0a\ud802\udd0b\ud802\udd0c\ud802\udd0d\ud802\udd0e\ud802\udd0f\ud802\udd10\ud802\udd11\ud802\udd12\ud802\udd13\ud802\udd14\ud802\udd15\ud802\udd20\ud802\udd21\ud802\udd22\ud802\udd23\ud802\udd24\ud802\udd25\ud802\udd26\ud802\udd27\ud802\udd28\ud802\udd29\ud802\udd2a\ud802\udd2b\ud802\udd2c\ud802\udd2d\ud802\udd2e\ud802\udd2f\ud802\udd30\ud802\udd31\ud802\udd32\ud802\udd33\ud802\udd34\ud802\udd35\ud802\udd36\ud802\udd37\ud802\udd38\ud802\udd39\ud802\ude00\ud802\ude10\ud802\ude11\ud802\ude12\ud802\ude13\ud802\ude15\ud802\ude16\ud802\ude17\ud802\ude19\ud802\ude1a\ud802\ude1b\ud802\ude1c\ud802\ude1d\ud802\ude1e\ud802\ude1f\ud802\ude20\ud802\ude21\ud802\ude22\ud802\ude23\ud802\ude24\ud802\ude25\ud802\ude26\ud802\ude27\ud802\ude28\ud802\ude29\ud802\ude2a\ud802\ude2b\ud802\ude2c\ud802\ude2d\ud802\ude2e\ud802\ude2f\ud802\ude30\ud802\ude31\ud802\ude32\ud802\ude33\ud802\ude60\ud802\ude61\ud802\ude62\ud802\ude63\ud802\ude64\ud802\ude65\ud802\ude66\ud802\ude67\ud802\ude68\ud802\ude69\ud802\ude6a\ud802\ude6b\ud802\ude6c\ud802\ude6d\ud802\ude6e\ud802\ude6f\ud802\ude70\ud802\ude71\ud802\ude72\ud802\ude73\ud802\ude74\ud802\ude75\ud802\ude76\ud802\ude77\ud802\ude78\ud802\ude79\ud802\ude7a\ud802\ude7b\ud802\ude7c\ud802\udf00\ud802\udf01\ud802\udf02\ud802\udf03\ud802\udf04\ud802\udf05\ud802\udf06\ud802\udf07\ud802\udf08\ud802\udf09\ud802\udf0a\ud802\udf0b\ud802\udf0c\ud802\udf0d\ud802\udf0e\ud802\udf0f\ud802\udf10\ud802\udf11\ud802\udf12\ud802\udf13\ud802\udf14\ud802\udf15\ud802\udf16\ud802\udf17\ud802\udf18\ud802\udf19\ud802\udf1a\ud802\udf1b\ud802\udf1c\ud802\udf1d\ud802\udf1e\ud802\udf1f\ud802\udf20\ud802\udf21\ud802\udf22\ud802\udf23\ud802\udf24\ud802\udf25\ud802\udf26\ud802\udf27\ud802\udf28\ud802\udf29\ud802\udf2a\ud802\udf2b\ud802\udf2c\ud802\udf2d\ud802\udf2e\ud802\udf2f\ud802\udf30\ud802\udf31\ud802\udf32\ud802\udf33\ud802\udf34\ud802\udf35\ud802\udf40\ud802\udf41\ud802\udf42\ud802\udf43\ud802\udf44\ud802\udf45\ud802\udf46\ud802\udf47\ud802\udf48\ud802\udf49\ud802\udf4a\ud802\udf4b\ud802\udf4c\ud802\udf4d\ud802\udf4e\ud802\udf4f\ud802\udf50\ud802\udf51\ud802\udf52\ud802\udf53\ud802\udf54\ud802\udf55\ud802\udf60\ud802\udf61\ud802\udf62\ud802\udf63\ud802\udf64\ud802\udf65\ud802\udf66\ud802\udf67\ud802\udf68\ud802\udf69\ud802\udf6a\ud802\udf6b\ud802\udf6c\ud802\udf6d\ud802\udf6e\ud802\udf6f\ud802\udf70\ud802\udf71\ud802\udf72\ud803\udc00\ud803\udc01\ud803\udc02\ud803\udc03\ud803\udc04\ud803\udc05\ud803\udc06\ud803\udc07\ud803\udc08\ud803\udc09\ud803\udc0a\ud803\udc0b\ud803\udc0c\ud803\udc0d\ud803\udc0e\ud803\udc0f\ud803\udc10\ud803\udc11\ud803\udc12\ud803\udc13\ud803\udc14\ud803\udc15\ud803\udc16\ud803\udc17\ud803\udc18\ud803\udc19\ud803\udc1a\ud803\udc1b\ud803\udc1c\ud803\udc1d\ud803\udc1e\ud803\udc1f\ud803\udc20\ud803\udc21\ud803\udc22\ud803\udc23\ud803\udc24\ud803\udc25\ud803\udc26\ud803\udc27\ud803\udc28\ud803\udc29\ud803\udc2a\ud803\udc2b\ud803\udc2c\ud803\udc2d\ud803\udc2e\ud803\udc2f\ud803\udc30\ud803\udc31\ud803\udc32\ud803\udc33\ud803\udc34\ud803\udc35\ud803\udc36\ud803\udc37\ud803\udc38\ud803\udc39\ud803\udc3a\ud803\udc3b\ud803\udc3c\ud803\udc3d\ud803\udc3e\ud803\udc3f\ud803\udc40\ud803\udc41\ud803\udc42\ud803\udc43\ud803\udc44\ud803\udc45\ud803\udc46\ud803\udc47\ud803\udc48\ud804\udc03\ud804\udc04\ud804\udc05\ud804\udc06\ud804\udc07\ud804\udc08\ud804\udc09\ud804\udc0a\ud804\udc0b\ud804\udc0c\ud804\udc0d\ud804\udc0e\ud804\udc0f\ud804\udc10\ud804\udc11\ud804\udc12\ud804\udc13\ud804\udc14\ud804\udc15\ud804\udc16\ud804\udc17\ud804\udc18\ud804\udc19\ud804\udc1a\ud804\udc1b\ud804\udc1c\ud804\udc1d\ud804\udc1e\ud804\udc1f\ud804\udc20\ud804\udc21\ud804\udc22\ud804\udc23\ud804\udc24\ud804\udc25\ud804\udc26\ud804\udc27\ud804\udc28\ud804\udc29\ud804\udc2a\ud804\udc2b\ud804\udc2c\ud804\udc2d\ud804\udc2e\ud804\udc2f\ud804\udc30\ud804\udc31\ud804\udc32\ud804\udc33\ud804\udc34\ud804\udc35\ud804\udc36\ud804\udc37\ud804\udc83\ud804\udc84\ud804\udc85\ud804\udc86\ud804\udc87\ud804\udc88\ud804\udc89\ud804\udc8a\ud804\udc8b\ud804\udc8c\ud804\udc8d\ud804\udc8e\ud804\udc8f\ud804\udc90\ud804\udc91\ud804\udc92\ud804\udc93\ud804\udc94\ud804\udc95\ud804\udc96\ud804\udc97\ud804\udc98\ud804\udc99\ud804\udc9a\ud804\udc9b\ud804\udc9c\ud804\udc9d\ud804\udc9e\ud804\udc9f\ud804\udca0\ud804\udca1\ud804\udca2\ud804\udca3\ud804\udca4\ud804\udca5\ud804\udca6\ud804\udca7\ud804\udca8\ud804\udca9\ud804\udcaa\ud804\udcab\ud804\udcac\ud804\udcad\ud804\udcae\ud804\udcaf\ud808\udc00\ud808\udc01\ud808\udc02\ud808\udc03\ud808\udc04\ud808\udc05\ud808\udc06\ud808\udc07\ud808\udc08\ud808\udc09\ud808\udc0a\ud808\udc0b\ud808\udc0c\ud808\udc0d\ud808\udc0e\ud808\udc0f\ud808\udc10\ud808\udc11\ud808\udc12\ud808\udc13\ud808\udc14\ud808\udc15\ud808\udc16\ud808\udc17\ud808\udc18\ud808\udc19\ud808\udc1a\ud808\udc1b\ud808\udc1c\ud808\udc1d\ud808\udc1e\ud808\udc1f\ud808\udc20\ud808\udc21\ud808\udc22\ud808\udc23\ud808\udc24\ud808\udc25\ud808\udc26\ud808\udc27\ud808\udc28\ud808\udc29\ud808\udc2a\ud808\udc2b\ud808\udc2c\ud808\udc2d\ud808\udc2e\ud808\udc2f\ud808\udc30\ud808\udc31\ud808\udc32\ud808\udc33\ud808\udc34\ud808\udc35\ud808\udc36\ud808\udc37\ud808\udc38\ud808\udc39\ud808\udc3a\ud808\udc3b\ud808\udc3c\ud808\udc3d\ud808\udc3e\ud808\udc3f\ud808\udc40\ud808\udc41\ud808\udc42\ud808\udc43\ud808\udc44\ud808\udc45\ud808\udc46\ud808\udc47\ud808\udc48\ud808\udc49\ud808\udc4a\ud808\udc4b\ud808\udc4c\ud808\udc4d\ud808\udc4e\ud808\udc4f\ud808\udc50\ud808\udc51\ud808\udc52\ud808\udc53\ud808\udc54\ud808\udc55\ud808\udc56\ud808\udc57\ud808\udc58\ud808\udc59\ud808\udc5a\ud808\udc5b\ud808\udc5c\ud808\udc5d\ud808\udc5e\ud808\udc5f\ud808\udc60\ud808\udc61\ud808\udc62\ud808\udc63\ud808\udc64\ud808\udc65\ud808\udc66\ud808\udc67\ud808\udc68\ud808\udc69\ud808\udc6a\ud808\udc6b\ud808\udc6c\ud808\udc6d\ud808\udc6e\ud808\udc6f\ud808\udc70\ud808\udc71\ud808\udc72\ud808\udc73\ud808\udc74\ud808\udc75\ud808\udc76\ud808\udc77\ud808\udc78\ud808\udc79\ud808\udc7a\ud808\udc7b\ud808\udc7c\ud808\udc7d\ud808\udc7e\ud808\udc7f\ud808\udc80\ud808\udc81\ud808\udc82\ud808\udc83\ud808\udc84\ud808\udc85\ud808\udc86\ud808\udc87\ud808\udc88\ud808\udc89\ud808\udc8a\ud808\udc8b\ud808\udc8c\ud808\udc8d\ud808\udc8e\ud808\udc8f\ud808\udc90\ud808\udc91\ud808\udc92\ud808\udc93\ud808\udc94\ud808\udc95\ud808\udc96\ud808\udc97\ud808\udc98\ud808\udc99\ud808\udc9a\ud808\udc9b\ud808\udc9c\ud808\udc9d\ud808\udc9e\ud808\udc9f\ud808\udca0\ud808\udca1\ud808\udca2\ud808\udca3\ud808\udca4\ud808\udca5\ud808\udca6\ud808\udca7\ud808\udca8\ud808\udca9\ud808\udcaa\ud808\udcab\ud808\udcac\ud808\udcad\ud808\udcae\ud808\udcaf\ud808\udcb0\ud808\udcb1\ud808\udcb2\ud808\udcb3\ud808\udcb4\ud808\udcb5\ud808\udcb6\ud808\udcb7\ud808\udcb8\ud808\udcb9\ud808\udcba\ud808\udcbb\ud808\udcbc\ud808\udcbd\ud808\udcbe\ud808\udcbf\ud808\udcc0\ud808\udcc1\ud808\udcc2\ud808\udcc3\ud808\udcc4\ud808\udcc5\ud808\udcc6\ud808\udcc7\ud808\udcc8\ud808\udcc9\ud808\udcca\ud808\udccb\ud808\udccc\ud808\udccd\ud808\udcce\ud808\udccf\ud808\udcd0\ud808\udcd1\ud808\udcd2\ud808\udcd3\ud808\udcd4\ud808\udcd5\ud808\udcd6\ud808\udcd7\ud808\udcd8\ud808\udcd9\ud808\udcda\ud808\udcdb\ud808\udcdc\ud808\udcdd\ud808\udcde\ud808\udcdf\ud808\udce0\ud808\udce1\ud808\udce2\ud808\udce3\ud808\udce4\ud808\udce5\ud808\udce6\ud808\udce7\ud808\udce8\ud808\udce9\ud808\udcea\ud808\udceb\ud808\udcec\ud808\udced\ud808\udcee\ud808\udcef\ud808\udcf0\ud808\udcf1\ud808\udcf2\ud808\udcf3\ud808\udcf4\ud808\udcf5\ud808\udcf6\ud808\udcf7\ud808\udcf8\ud808\udcf9\ud808\udcfa\ud808\udcfb\ud808\udcfc\ud808\udcfd\ud808\udcfe\ud808\udcff\ud808\udd00\ud808\udd01\ud808\udd02\ud808\udd03\ud808\udd04\ud808\udd05\ud808\udd06\ud808\udd07\ud808\udd08\ud808\udd09\ud808\udd0a\ud808\udd0b\ud808\udd0c\ud808\udd0d\ud808\udd0e\ud808\udd0f\ud808\udd10\ud808\udd11\ud808\udd12\ud808\udd13\ud808\udd14\ud808\udd15\ud808\udd16\ud808\udd17\ud808\udd18\ud808\udd19\ud808\udd1a\ud808\udd1b\ud808\udd1c\ud808\udd1d\ud808\udd1e\ud808\udd1f\ud808\udd20\ud808\udd21\ud808\udd22\ud808\udd23\ud808\udd24\ud808\udd25\ud808\udd26\ud808\udd27\ud808\udd28\ud808\udd29\ud808\udd2a\ud808\udd2b\ud808\udd2c\ud808\udd2d\ud808\udd2e\ud808\udd2f\ud808\udd30\ud808\udd31\ud808\udd32\ud808\udd33\ud808\udd34\ud808\udd35\ud808\udd36\ud808\udd37\ud808\udd38\ud808\udd39\ud808\udd3a\ud808\udd3b\ud808\udd3c\ud808\udd3d\ud808\udd3e\ud808\udd3f\ud808\udd40\ud808\udd41\ud808\udd42\ud808\udd43\ud808\udd44\ud808\udd45\ud808\udd46\ud808\udd47\ud808\udd48\ud808\udd49\ud808\udd4a\ud808\udd4b\ud808\udd4c\ud808\udd4d\ud808\udd4e\ud808\udd4f\ud808\udd50\ud808\udd51\ud808\udd52\ud808\udd53\ud808\udd54\ud808\udd55\ud808\udd56\ud808\udd57\ud808\udd58\ud808\udd59\ud808\udd5a\ud808\udd5b\ud808\udd5c\ud808\udd5d\ud808\udd5e\ud808\udd5f\ud808\udd60\ud808\udd61\ud808\udd62\ud808\udd63\ud808\udd64\ud808\udd65\ud808\udd66\ud808\udd67\ud808\udd68\ud808\udd69\ud808\udd6a\ud808\udd6b\ud808\udd6c\ud808\udd6d\ud808\udd6e\ud808\udd6f\ud808\udd70\ud808\udd71\ud808\udd72\ud808\udd73\ud808\udd74\ud808\udd75\ud808\udd76\ud808\udd77\ud808\udd78\ud808\udd79\ud808\udd7a\ud808\udd7b\ud808\udd7c\ud808\udd7d\ud808\udd7e\ud808\udd7f\ud808\udd80\ud808\udd81\ud808\udd82\ud808\udd83\ud808\udd84\ud808\udd85\ud808\udd86\ud808\udd87\ud808\udd88\ud808\udd89\ud808\udd8a\ud808\udd8b\ud808\udd8c\ud808\udd8d\ud808\udd8e\ud808\udd8f\ud808\udd90\ud808\udd91\ud808\udd92\ud808\udd93\ud808\udd94\ud808\udd95\ud808\udd96\ud808\udd97\ud808\udd98\ud808\udd99\ud808\udd9a\ud808\udd9b\ud808\udd9c\ud808\udd9d\ud808\udd9e\ud808\udd9f\ud808\udda0\ud808\udda1\ud808\udda2\ud808\udda3\ud808\udda4\ud808\udda5\ud808\udda6\ud808\udda7\ud808\udda8\ud808\udda9\ud808\uddaa\ud808\uddab\ud808\uddac\ud808\uddad\ud808\uddae\ud808\uddaf\ud808\uddb0\ud808\uddb1\ud808\uddb2\ud808\uddb3\ud808\uddb4\ud808\uddb5\ud808\uddb6\ud808\uddb7\ud808\uddb8\ud808\uddb9\ud808\uddba\ud808\uddbb\ud808\uddbc\ud808\uddbd\ud808\uddbe\ud808\uddbf\ud808\uddc0\ud808\uddc1\ud808\uddc2\ud808\uddc3\ud808\uddc4\ud808\uddc5\ud808\uddc6\ud808\uddc7\ud808\uddc8\ud808\uddc9\ud808\uddca\ud808\uddcb\ud808\uddcc\ud808\uddcd\ud808\uddce\ud808\uddcf\ud808\uddd0\ud808\uddd1\ud808\uddd2\ud808\uddd3\ud808\uddd4\ud808\uddd5\ud808\uddd6\ud808\uddd7\ud808\uddd8\ud808\uddd9\ud808\uddda\ud808\udddb\ud808\udddc\ud808\udddd\ud808\uddde\ud808\udddf\ud808\udde0\ud808\udde1\ud808\udde2\ud808\udde3\ud808\udde4\ud808\udde5\ud808\udde6\ud808\udde7\ud808\udde8\ud808\udde9\ud808\uddea\ud808\uddeb\ud808\uddec\ud808\udded\ud808\uddee\ud808\uddef\ud808\uddf0\ud808\uddf1\ud808\uddf2\ud808\uddf3\ud808\uddf4\ud808\uddf5\ud808\uddf6\ud808\uddf7\ud808\uddf8\ud808\uddf9\ud808\uddfa\ud808\uddfb\ud808\uddfc\ud808\uddfd\ud808\uddfe\ud808\uddff\ud808\ude00\ud808\ude01\ud808\ude02\ud808\ude03\ud808\ude04\ud808\ude05\ud808\ude06\ud808\ude07\ud808\ude08\ud808\ude09\ud808\ude0a\ud808\ude0b\ud808\ude0c\ud808\ude0d\ud808\ude0e\ud808\ude0f\ud808\ude10\ud808\ude11\ud808\ude12\ud808\ude13\ud808\ude14\ud808\ude15\ud808\ude16\ud808\ude17\ud808\ude18\ud808\ude19\ud808\ude1a\ud808\ude1b\ud808\ude1c\ud808\ude1d\ud808\ude1e\ud808\ude1f\ud808\ude20\ud808\ude21\ud808\ude22\ud808\ude23\ud808\ude24\ud808\ude25\ud808\ude26\ud808\ude27\ud808\ude28\ud808\ude29\ud808\ude2a\ud808\ude2b\ud808\ude2c\ud808\ude2d\ud808\ude2e\ud808\ude2f\ud808\ude30\ud808\ude31\ud808\ude32\ud808\ude33\ud808\ude34\ud808\ude35\ud808\ude36\ud808\ude37\ud808\ude38\ud808\ude39\ud808\ude3a\ud808\ude3b\ud808\ude3c\ud808\ude3d\ud808\ude3e\ud808\ude3f\ud808\ude40\ud808\ude41\ud808\ude42\ud808\ude43\ud808\ude44\ud808\ude45\ud808\ude46\ud808\ude47\ud808\ude48\ud808\ude49\ud808\ude4a\ud808\ude4b\ud808\ude4c\ud808\ude4d\ud808\ude4e\ud808\ude4f\ud808\ude50\ud808\ude51\ud808\ude52\ud808\ude53\ud808\ude54\ud808\ude55\ud808\ude56\ud808\ude57\ud808\ude58\ud808\ude59\ud808\ude5a\ud808\ude5b\ud808\ude5c\ud808\ude5d\ud808\ude5e\ud808\ude5f\ud808\ude60\ud808\ude61\ud808\ude62\ud808\ude63\ud808\ude64\ud808\ude65\ud808\ude66\ud808\ude67\ud808\ude68\ud808\ude69\ud808\ude6a\ud808\ude6b\ud808\ude6c\ud808\ude6d\ud808\ude6e\ud808\ude6f\ud808\ude70\ud808\ude71\ud808\ude72\ud808\ude73\ud808\ude74\ud808\ude75\ud808\ude76\ud808\ude77\ud808\ude78\ud808\ude79\ud808\ude7a\ud808\ude7b\ud808\ude7c\ud808\ude7d\ud808\ude7e\ud808\ude7f\ud808\ude80\ud808\ude81\ud808\ude82\ud808\ude83\ud808\ude84\ud808\ude85\ud808\ude86\ud808\ude87\ud808\ude88\ud808\ude89\ud808\ude8a\ud808\ude8b\ud808\ude8c\ud808\ude8d\ud808\ude8e\ud808\ude8f\ud808\ude90\ud808\ude91\ud808\ude92\ud808\ude93\ud808\ude94\ud808\ude95\ud808\ude96\ud808\ude97\ud808\ude98\ud808\ude99\ud808\ude9a\ud808\ude9b\ud808\ude9c\ud808\ude9d\ud808\ude9e\ud808\ude9f\ud808\udea0\ud808\udea1\ud808\udea2\ud808\udea3\ud808\udea4\ud808\udea5\ud808\udea6\ud808\udea7\ud808\udea8\ud808\udea9\ud808\udeaa\ud808\udeab\ud808\udeac\ud808\udead\ud808\udeae\ud808\udeaf\ud808\udeb0\ud808\udeb1\ud808\udeb2\ud808\udeb3\ud808\udeb4\ud808\udeb5\ud808\udeb6\ud808\udeb7\ud808\udeb8\ud808\udeb9\ud808\udeba\ud808\udebb\ud808\udebc\ud808\udebd\ud808\udebe\ud808\udebf\ud808\udec0\ud808\udec1\ud808\udec2\ud808\udec3\ud808\udec4\ud808\udec5\ud808\udec6\ud808\udec7\ud808\udec8\ud808\udec9\ud808\udeca\ud808\udecb\ud808\udecc\ud808\udecd\ud808\udece\ud808\udecf\ud808\uded0\ud808\uded1\ud808\uded2\ud808\uded3\ud808\uded4\ud808\uded5\ud808\uded6\ud808\uded7\ud808\uded8\ud808\uded9\ud808\udeda\ud808\udedb\ud808\udedc\ud808\udedd\ud808\udede\ud808\udedf\ud808\udee0\ud808\udee1\ud808\udee2\ud808\udee3\ud808\udee4\ud808\udee5\ud808\udee6\ud808\udee7\ud808\udee8\ud808\udee9\ud808\udeea\ud808\udeeb\ud808\udeec\ud808\udeed\ud808\udeee\ud808\udeef\ud808\udef0\ud808\udef1\ud808\udef2\ud808\udef3\ud808\udef4\ud808\udef5\ud808\udef6\ud808\udef7\ud808\udef8\ud808\udef9\ud808\udefa\ud808\udefb\ud808\udefc\ud808\udefd\ud808\udefe\ud808\udeff\ud808\udf00\ud808\udf01\ud808\udf02\ud808\udf03\ud808\udf04\ud808\udf05\ud808\udf06\ud808\udf07\ud808\udf08\ud808\udf09\ud808\udf0a\ud808\udf0b\ud808\udf0c\ud808\udf0d\ud808\udf0e\ud808\udf0f\ud808\udf10\ud808\udf11\ud808\udf12\ud808\udf13\ud808\udf14\ud808\udf15\ud808\udf16\ud808\udf17\ud808\udf18\ud808\udf19\ud808\udf1a\ud808\udf1b\ud808\udf1c\ud808\udf1d\ud808\udf1e\ud808\udf1f\ud808\udf20\ud808\udf21\ud808\udf22\ud808\udf23\ud808\udf24\ud808\udf25\ud808\udf26\ud808\udf27\ud808\udf28\ud808\udf29\ud808\udf2a\ud808\udf2b\ud808\udf2c\ud808\udf2d\ud808\udf2e\ud808\udf2f\ud808\udf30\ud808\udf31\ud808\udf32\ud808\udf33\ud808\udf34\ud808\udf35\ud808\udf36\ud808\udf37\ud808\udf38\ud808\udf39\ud808\udf3a\ud808\udf3b\ud808\udf3c\ud808\udf3d\ud808\udf3e\ud808\udf3f\ud808\udf40\ud808\udf41\ud808\udf42\ud808\udf43\ud808\udf44\ud808\udf45\ud808\udf46\ud808\udf47\ud808\udf48\ud808\udf49\ud808\udf4a\ud808\udf4b\ud808\udf4c\ud808\udf4d\ud808\udf4e\ud808\udf4f\ud808\udf50\ud808\udf51\ud808\udf52\ud808\udf53\ud808\udf54\ud808\udf55\ud808\udf56\ud808\udf57\ud808\udf58\ud808\udf59\ud808\udf5a\ud808\udf5b\ud808\udf5c\ud808\udf5d\ud808\udf5e\ud808\udf5f\ud808\udf60\ud808\udf61\ud808\udf62\ud808\udf63\ud808\udf64\ud808\udf65\ud808\udf66\ud808\udf67\ud808\udf68\ud808\udf69\ud808\udf6a\ud808\udf6b\ud808\udf6c\ud808\udf6d\ud808\udf6e\ud80c\udc00\ud80c\udc01\ud80c\udc02\ud80c\udc03\ud80c\udc04\ud80c\udc05\ud80c\udc06\ud80c\udc07\ud80c\udc08\ud80c\udc09\ud80c\udc0a\ud80c\udc0b\ud80c\udc0c\ud80c\udc0d\ud80c\udc0e\ud80c\udc0f\ud80c\udc10\ud80c\udc11\ud80c\udc12\ud80c\udc13\ud80c\udc14\ud80c\udc15\ud80c\udc16\ud80c\udc17\ud80c\udc18\ud80c\udc19\ud80c\udc1a\ud80c\udc1b\ud80c\udc1c\ud80c\udc1d\ud80c\udc1e\ud80c\udc1f\ud80c\udc20\ud80c\udc21\ud80c\udc22\ud80c\udc23\ud80c\udc24\ud80c\udc25\ud80c\udc26\ud80c\udc27\ud80c\udc28\ud80c\udc29\ud80c\udc2a\ud80c\udc2b\ud80c\udc2c\ud80c\udc2d\ud80c\udc2e\ud80c\udc2f\ud80c\udc30\ud80c\udc31\ud80c\udc32\ud80c\udc33\ud80c\udc34\ud80c\udc35\ud80c\udc36\ud80c\udc37\ud80c\udc38\ud80c\udc39\ud80c\udc3a\ud80c\udc3b\ud80c\udc3c\ud80c\udc3d\ud80c\udc3e\ud80c\udc3f\ud80c\udc40\ud80c\udc41\ud80c\udc42\ud80c\udc43\ud80c\udc44\ud80c\udc45\ud80c\udc46\ud80c\udc47\ud80c\udc48\ud80c\udc49\ud80c\udc4a\ud80c\udc4b\ud80c\udc4c\ud80c\udc4d\ud80c\udc4e\ud80c\udc4f\ud80c\udc50\ud80c\udc51\ud80c\udc52\ud80c\udc53\ud80c\udc54\ud80c\udc55\ud80c\udc56\ud80c\udc57\ud80c\udc58\ud80c\udc59\ud80c\udc5a\ud80c\udc5b\ud80c\udc5c\ud80c\udc5d\ud80c\udc5e\ud80c\udc5f\ud80c\udc60\ud80c\udc61\ud80c\udc62\ud80c\udc63\ud80c\udc64\ud80c\udc65\ud80c\udc66\ud80c\udc67\ud80c\udc68\ud80c\udc69\ud80c\udc6a\ud80c\udc6b\ud80c\udc6c\ud80c\udc6d\ud80c\udc6e\ud80c\udc6f\ud80c\udc70\ud80c\udc71\ud80c\udc72\ud80c\udc73\ud80c\udc74\ud80c\udc75\ud80c\udc76\ud80c\udc77\ud80c\udc78\ud80c\udc79\ud80c\udc7a\ud80c\udc7b\ud80c\udc7c\ud80c\udc7d\ud80c\udc7e\ud80c\udc7f\ud80c\udc80\ud80c\udc81\ud80c\udc82\ud80c\udc83\ud80c\udc84\ud80c\udc85\ud80c\udc86\ud80c\udc87\ud80c\udc88\ud80c\udc89\ud80c\udc8a\ud80c\udc8b\ud80c\udc8c\ud80c\udc8d\ud80c\udc8e\ud80c\udc8f\ud80c\udc90\ud80c\udc91\ud80c\udc92\ud80c\udc93\ud80c\udc94\ud80c\udc95\ud80c\udc96\ud80c\udc97\ud80c\udc98\ud80c\udc99\ud80c\udc9a\ud80c\udc9b\ud80c\udc9c\ud80c\udc9d\ud80c\udc9e\ud80c\udc9f\ud80c\udca0\ud80c\udca1\ud80c\udca2\ud80c\udca3\ud80c\udca4\ud80c\udca5\ud80c\udca6\ud80c\udca7\ud80c\udca8\ud80c\udca9\ud80c\udcaa\ud80c\udcab\ud80c\udcac\ud80c\udcad\ud80c\udcae\ud80c\udcaf\ud80c\udcb0\ud80c\udcb1\ud80c\udcb2\ud80c\udcb3\ud80c\udcb4\ud80c\udcb5\ud80c\udcb6\ud80c\udcb7\ud80c\udcb8\ud80c\udcb9\ud80c\udcba\ud80c\udcbb\ud80c\udcbc\ud80c\udcbd\ud80c\udcbe\ud80c\udcbf\ud80c\udcc0\ud80c\udcc1\ud80c\udcc2\ud80c\udcc3\ud80c\udcc4\ud80c\udcc5\ud80c\udcc6\ud80c\udcc7\ud80c\udcc8\ud80c\udcc9\ud80c\udcca\ud80c\udccb\ud80c\udccc\ud80c\udccd\ud80c\udcce\ud80c\udccf\ud80c\udcd0\ud80c\udcd1\ud80c\udcd2\ud80c\udcd3\ud80c\udcd4\ud80c\udcd5\ud80c\udcd6\ud80c\udcd7\ud80c\udcd8\ud80c\udcd9\ud80c\udcda\ud80c\udcdb\ud80c\udcdc\ud80c\udcdd\ud80c\udcde\ud80c\udcdf\ud80c\udce0\ud80c\udce1\ud80c\udce2\ud80c\udce3\ud80c\udce4\ud80c\udce5\ud80c\udce6\ud80c\udce7\ud80c\udce8\ud80c\udce9\ud80c\udcea\ud80c\udceb\ud80c\udcec\ud80c\udced\ud80c\udcee\ud80c\udcef\ud80c\udcf0\ud80c\udcf1\ud80c\udcf2\ud80c\udcf3\ud80c\udcf4\ud80c\udcf5\ud80c\udcf6\ud80c\udcf7\ud80c\udcf8\ud80c\udcf9\ud80c\udcfa\ud80c\udcfb\ud80c\udcfc\ud80c\udcfd\ud80c\udcfe\ud80c\udcff\ud80c\udd00\ud80c\udd01\ud80c\udd02\ud80c\udd03\ud80c\udd04\ud80c\udd05\ud80c\udd06\ud80c\udd07\ud80c\udd08\ud80c\udd09\ud80c\udd0a\ud80c\udd0b\ud80c\udd0c\ud80c\udd0d\ud80c\udd0e\ud80c\udd0f\ud80c\udd10\ud80c\udd11\ud80c\udd12\ud80c\udd13\ud80c\udd14\ud80c\udd15\ud80c\udd16\ud80c\udd17\ud80c\udd18\ud80c\udd19\ud80c\udd1a\ud80c\udd1b\ud80c\udd1c\ud80c\udd1d\ud80c\udd1e\ud80c\udd1f\ud80c\udd20\ud80c\udd21\ud80c\udd22\ud80c\udd23\ud80c\udd24\ud80c\udd25\ud80c\udd26\ud80c\udd27\ud80c\udd28\ud80c\udd29\ud80c\udd2a\ud80c\udd2b\ud80c\udd2c\ud80c\udd2d\ud80c\udd2e\ud80c\udd2f\ud80c\udd30\ud80c\udd31\ud80c\udd32\ud80c\udd33\ud80c\udd34\ud80c\udd35\ud80c\udd36\ud80c\udd37\ud80c\udd38\ud80c\udd39\ud80c\udd3a\ud80c\udd3b\ud80c\udd3c\ud80c\udd3d\ud80c\udd3e\ud80c\udd3f\ud80c\udd40\ud80c\udd41\ud80c\udd42\ud80c\udd43\ud80c\udd44\ud80c\udd45\ud80c\udd46\ud80c\udd47\ud80c\udd48\ud80c\udd49\ud80c\udd4a\ud80c\udd4b\ud80c\udd4c\ud80c\udd4d\ud80c\udd4e\ud80c\udd4f\ud80c\udd50\ud80c\udd51\ud80c\udd52\ud80c\udd53\ud80c\udd54\ud80c\udd55\ud80c\udd56\ud80c\udd57\ud80c\udd58\ud80c\udd59\ud80c\udd5a\ud80c\udd5b\ud80c\udd5c\ud80c\udd5d\ud80c\udd5e\ud80c\udd5f\ud80c\udd60\ud80c\udd61\ud80c\udd62\ud80c\udd63\ud80c\udd64\ud80c\udd65\ud80c\udd66\ud80c\udd67\ud80c\udd68\ud80c\udd69\ud80c\udd6a\ud80c\udd6b\ud80c\udd6c\ud80c\udd6d\ud80c\udd6e\ud80c\udd6f\ud80c\udd70\ud80c\udd71\ud80c\udd72\ud80c\udd73\ud80c\udd74\ud80c\udd75\ud80c\udd76\ud80c\udd77\ud80c\udd78\ud80c\udd79\ud80c\udd7a\ud80c\udd7b\ud80c\udd7c\ud80c\udd7d\ud80c\udd7e\ud80c\udd7f\ud80c\udd80\ud80c\udd81\ud80c\udd82\ud80c\udd83\ud80c\udd84\ud80c\udd85\ud80c\udd86\ud80c\udd87\ud80c\udd88\ud80c\udd89\ud80c\udd8a\ud80c\udd8b\ud80c\udd8c\ud80c\udd8d\ud80c\udd8e\ud80c\udd8f\ud80c\udd90\ud80c\udd91\ud80c\udd92\ud80c\udd93\ud80c\udd94\ud80c\udd95\ud80c\udd96\ud80c\udd97\ud80c\udd98\ud80c\udd99\ud80c\udd9a\ud80c\udd9b\ud80c\udd9c\ud80c\udd9d\ud80c\udd9e\ud80c\udd9f\ud80c\udda0\ud80c\udda1\ud80c\udda2\ud80c\udda3\ud80c\udda4\ud80c\udda5\ud80c\udda6\ud80c\udda7\ud80c\udda8\ud80c\udda9\ud80c\uddaa\ud80c\uddab\ud80c\uddac\ud80c\uddad\ud80c\uddae\ud80c\uddaf\ud80c\uddb0\ud80c\uddb1\ud80c\uddb2\ud80c\uddb3\ud80c\uddb4\ud80c\uddb5\ud80c\uddb6\ud80c\uddb7\ud80c\uddb8\ud80c\uddb9\ud80c\uddba\ud80c\uddbb\ud80c\uddbc\ud80c\uddbd\ud80c\uddbe\ud80c\uddbf\ud80c\uddc0\ud80c\uddc1\ud80c\uddc2\ud80c\uddc3\ud80c\uddc4\ud80c\uddc5\ud80c\uddc6\ud80c\uddc7\ud80c\uddc8\ud80c\uddc9\ud80c\uddca\ud80c\uddcb\ud80c\uddcc\ud80c\uddcd\ud80c\uddce\ud80c\uddcf\ud80c\uddd0\ud80c\uddd1\ud80c\uddd2\ud80c\uddd3\ud80c\uddd4\ud80c\uddd5\ud80c\uddd6\ud80c\uddd7\ud80c\uddd8\ud80c\uddd9\ud80c\uddda\ud80c\udddb\ud80c\udddc\ud80c\udddd\ud80c\uddde\ud80c\udddf\ud80c\udde0\ud80c\udde1\ud80c\udde2\ud80c\udde3\ud80c\udde4\ud80c\udde5\ud80c\udde6\ud80c\udde7\ud80c\udde8\ud80c\udde9\ud80c\uddea\ud80c\uddeb\ud80c\uddec\ud80c\udded\ud80c\uddee\ud80c\uddef\ud80c\uddf0\ud80c\uddf1\ud80c\uddf2\ud80c\uddf3\ud80c\uddf4\ud80c\uddf5\ud80c\uddf6\ud80c\uddf7\ud80c\uddf8\ud80c\uddf9\ud80c\uddfa\ud80c\uddfb\ud80c\uddfc\ud80c\uddfd\ud80c\uddfe\ud80c\uddff\ud80c\ude00\ud80c\ude01\ud80c\ude02\ud80c\ude03\ud80c\ude04\ud80c\ude05\ud80c\ude06\ud80c\ude07\ud80c\ude08\ud80c\ude09\ud80c\ude0a\ud80c\ude0b\ud80c\ude0c\ud80c\ude0d\ud80c\ude0e\ud80c\ude0f\ud80c\ude10\ud80c\ude11\ud80c\ude12\ud80c\ude13\ud80c\ude14\ud80c\ude15\ud80c\ude16\ud80c\ude17\ud80c\ude18\ud80c\ude19\ud80c\ude1a\ud80c\ude1b\ud80c\ude1c\ud80c\ude1d\ud80c\ude1e\ud80c\ude1f\ud80c\ude20\ud80c\ude21\ud80c\ude22\ud80c\ude23\ud80c\ude24\ud80c\ude25\ud80c\ude26\ud80c\ude27\ud80c\ude28\ud80c\ude29\ud80c\ude2a\ud80c\ude2b\ud80c\ude2c\ud80c\ude2d\ud80c\ude2e\ud80c\ude2f\ud80c\ude30\ud80c\ude31\ud80c\ude32\ud80c\ude33\ud80c\ude34\ud80c\ude35\ud80c\ude36\ud80c\ude37\ud80c\ude38\ud80c\ude39\ud80c\ude3a\ud80c\ude3b\ud80c\ude3c\ud80c\ude3d\ud80c\ude3e\ud80c\ude3f\ud80c\ude40\ud80c\ude41\ud80c\ude42\ud80c\ude43\ud80c\ude44\ud80c\ude45\ud80c\ude46\ud80c\ude47\ud80c\ude48\ud80c\ude49\ud80c\ude4a\ud80c\ude4b\ud80c\ude4c\ud80c\ude4d\ud80c\ude4e\ud80c\ude4f\ud80c\ude50\ud80c\ude51\ud80c\ude52\ud80c\ude53\ud80c\ude54\ud80c\ude55\ud80c\ude56\ud80c\ude57\ud80c\ude58\ud80c\ude59\ud80c\ude5a\ud80c\ude5b\ud80c\ude5c\ud80c\ude5d\ud80c\ude5e\ud80c\ude5f\ud80c\ude60\ud80c\ude61\ud80c\ude62\ud80c\ude63\ud80c\ude64\ud80c\ude65\ud80c\ude66\ud80c\ude67\ud80c\ude68\ud80c\ude69\ud80c\ude6a\ud80c\ude6b\ud80c\ude6c\ud80c\ude6d\ud80c\ude6e\ud80c\ude6f\ud80c\ude70\ud80c\ude71\ud80c\ude72\ud80c\ude73\ud80c\ude74\ud80c\ude75\ud80c\ude76\ud80c\ude77\ud80c\ude78\ud80c\ude79\ud80c\ude7a\ud80c\ude7b\ud80c\ude7c\ud80c\ude7d\ud80c\ude7e\ud80c\ude7f\ud80c\ude80\ud80c\ude81\ud80c\ude82\ud80c\ude83\ud80c\ude84\ud80c\ude85\ud80c\ude86\ud80c\ude87\ud80c\ude88\ud80c\ude89\ud80c\ude8a\ud80c\ude8b\ud80c\ude8c\ud80c\ude8d\ud80c\ude8e\ud80c\ude8f\ud80c\ude90\ud80c\ude91\ud80c\ude92\ud80c\ude93\ud80c\ude94\ud80c\ude95\ud80c\ude96\ud80c\ude97\ud80c\ude98\ud80c\ude99\ud80c\ude9a\ud80c\ude9b\ud80c\ude9c\ud80c\ude9d\ud80c\ude9e\ud80c\ude9f\ud80c\udea0\ud80c\udea1\ud80c\udea2\ud80c\udea3\ud80c\udea4\ud80c\udea5\ud80c\udea6\ud80c\udea7\ud80c\udea8\ud80c\udea9\ud80c\udeaa\ud80c\udeab\ud80c\udeac\ud80c\udead\ud80c\udeae\ud80c\udeaf\ud80c\udeb0\ud80c\udeb1\ud80c\udeb2\ud80c\udeb3\ud80c\udeb4\ud80c\udeb5\ud80c\udeb6\ud80c\udeb7\ud80c\udeb8\ud80c\udeb9\ud80c\udeba\ud80c\udebb\ud80c\udebc\ud80c\udebd\ud80c\udebe\ud80c\udebf\ud80c\udec0\ud80c\udec1\ud80c\udec2\ud80c\udec3\ud80c\udec4\ud80c\udec5\ud80c\udec6\ud80c\udec7\ud80c\udec8\ud80c\udec9\ud80c\udeca\ud80c\udecb\ud80c\udecc\ud80c\udecd\ud80c\udece\ud80c\udecf\ud80c\uded0\ud80c\uded1\ud80c\uded2\ud80c\uded3\ud80c\uded4\ud80c\uded5\ud80c\uded6\ud80c\uded7\ud80c\uded8\ud80c\uded9\ud80c\udeda\ud80c\udedb\ud80c\udedc\ud80c\udedd\ud80c\udede\ud80c\udedf\ud80c\udee0\ud80c\udee1\ud80c\udee2\ud80c\udee3\ud80c\udee4\ud80c\udee5\ud80c\udee6\ud80c\udee7\ud80c\udee8\ud80c\udee9\ud80c\udeea\ud80c\udeeb\ud80c\udeec\ud80c\udeed\ud80c\udeee\ud80c\udeef\ud80c\udef0\ud80c\udef1\ud80c\udef2\ud80c\udef3\ud80c\udef4\ud80c\udef5\ud80c\udef6\ud80c\udef7\ud80c\udef8\ud80c\udef9\ud80c\udefa\ud80c\udefb\ud80c\udefc\ud80c\udefd\ud80c\udefe\ud80c\udeff\ud80c\udf00\ud80c\udf01\ud80c\udf02\ud80c\udf03\ud80c\udf04\ud80c\udf05\ud80c\udf06\ud80c\udf07\ud80c\udf08\ud80c\udf09\ud80c\udf0a\ud80c\udf0b\ud80c\udf0c\ud80c\udf0d\ud80c\udf0e\ud80c\udf0f\ud80c\udf10\ud80c\udf11\ud80c\udf12\ud80c\udf13\ud80c\udf14\ud80c\udf15\ud80c\udf16\ud80c\udf17\ud80c\udf18\ud80c\udf19\ud80c\udf1a\ud80c\udf1b\ud80c\udf1c\ud80c\udf1d\ud80c\udf1e\ud80c\udf1f\ud80c\udf20\ud80c\udf21\ud80c\udf22\ud80c\udf23\ud80c\udf24\ud80c\udf25\ud80c\udf26\ud80c\udf27\ud80c\udf28\ud80c\udf29\ud80c\udf2a\ud80c\udf2b\ud80c\udf2c\ud80c\udf2d\ud80c\udf2e\ud80c\udf2f\ud80c\udf30\ud80c\udf31\ud80c\udf32\ud80c\udf33\ud80c\udf34\ud80c\udf35\ud80c\udf36\ud80c\udf37\ud80c\udf38\ud80c\udf39\ud80c\udf3a\ud80c\udf3b\ud80c\udf3c\ud80c\udf3d\ud80c\udf3e\ud80c\udf3f\ud80c\udf40\ud80c\udf41\ud80c\udf42\ud80c\udf43\ud80c\udf44\ud80c\udf45\ud80c\udf46\ud80c\udf47\ud80c\udf48\ud80c\udf49\ud80c\udf4a\ud80c\udf4b\ud80c\udf4c\ud80c\udf4d\ud80c\udf4e\ud80c\udf4f\ud80c\udf50\ud80c\udf51\ud80c\udf52\ud80c\udf53\ud80c\udf54\ud80c\udf55\ud80c\udf56\ud80c\udf57\ud80c\udf58\ud80c\udf59\ud80c\udf5a\ud80c\udf5b\ud80c\udf5c\ud80c\udf5d\ud80c\udf5e\ud80c\udf5f\ud80c\udf60\ud80c\udf61\ud80c\udf62\ud80c\udf63\ud80c\udf64\ud80c\udf65\ud80c\udf66\ud80c\udf67\ud80c\udf68\ud80c\udf69\ud80c\udf6a\ud80c\udf6b\ud80c\udf6c\ud80c\udf6d\ud80c\udf6e\ud80c\udf6f\ud80c\udf70\ud80c\udf71\ud80c\udf72\ud80c\udf73\ud80c\udf74\ud80c\udf75\ud80c\udf76\ud80c\udf77\ud80c\udf78\ud80c\udf79\ud80c\udf7a\ud80c\udf7b\ud80c\udf7c\ud80c\udf7d\ud80c\udf7e\ud80c\udf7f\ud80c\udf80\ud80c\udf81\ud80c\udf82\ud80c\udf83\ud80c\udf84\ud80c\udf85\ud80c\udf86\ud80c\udf87\ud80c\udf88\ud80c\udf89\ud80c\udf8a\ud80c\udf8b\ud80c\udf8c\ud80c\udf8d\ud80c\udf8e\ud80c\udf8f\ud80c\udf90\ud80c\udf91\ud80c\udf92\ud80c\udf93\ud80c\udf94\ud80c\udf95\ud80c\udf96\ud80c\udf97\ud80c\udf98\ud80c\udf99\ud80c\udf9a\ud80c\udf9b\ud80c\udf9c\ud80c\udf9d\ud80c\udf9e\ud80c\udf9f\ud80c\udfa0\ud80c\udfa1\ud80c\udfa2\ud80c\udfa3\ud80c\udfa4\ud80c\udfa5\ud80c\udfa6\ud80c\udfa7\ud80c\udfa8\ud80c\udfa9\ud80c\udfaa\ud80c\udfab\ud80c\udfac\ud80c\udfad\ud80c\udfae\ud80c\udfaf\ud80c\udfb0\ud80c\udfb1\ud80c\udfb2\ud80c\udfb3\ud80c\udfb4\ud80c\udfb5\ud80c\udfb6\ud80c\udfb7\ud80c\udfb8\ud80c\udfb9\ud80c\udfba\ud80c\udfbb\ud80c\udfbc\ud80c\udfbd\ud80c\udfbe\ud80c\udfbf\ud80c\udfc0\ud80c\udfc1\ud80c\udfc2\ud80c\udfc3\ud80c\udfc4\ud80c\udfc5\ud80c\udfc6\ud80c\udfc7\ud80c\udfc8\ud80c\udfc9\ud80c\udfca\ud80c\udfcb\ud80c\udfcc\ud80c\udfcd\ud80c\udfce\ud80c\udfcf\ud80c\udfd0\ud80c\udfd1\ud80c\udfd2\ud80c\udfd3\ud80c\udfd4\ud80c\udfd5\ud80c\udfd6\ud80c\udfd7\ud80c\udfd8\ud80c\udfd9\ud80c\udfda\ud80c\udfdb\ud80c\udfdc\ud80c\udfdd\ud80c\udfde\ud80c\udfdf\ud80c\udfe0\ud80c\udfe1\ud80c\udfe2\ud80c\udfe3\ud80c\udfe4\ud80c\udfe5\ud80c\udfe6\ud80c\udfe7\ud80c\udfe8\ud80c\udfe9\ud80c\udfea\ud80c\udfeb\ud80c\udfec\ud80c\udfed\ud80c\udfee\ud80c\udfef\ud80c\udff0\ud80c\udff1\ud80c\udff2\ud80c\udff3\ud80c\udff4\ud80c\udff5\ud80c\udff6\ud80c\udff7\ud80c\udff8\ud80c\udff9\ud80c\udffa\ud80c\udffb\ud80c\udffc\ud80c\udffd\ud80c\udffe\ud80c\udfff\ud80d\udc00\ud80d\udc01\ud80d\udc02\ud80d\udc03\ud80d\udc04\ud80d\udc05\ud80d\udc06\ud80d\udc07\ud80d\udc08\ud80d\udc09\ud80d\udc0a\ud80d\udc0b\ud80d\udc0c\ud80d\udc0d\ud80d\udc0e\ud80d\udc0f\ud80d\udc10\ud80d\udc11\ud80d\udc12\ud80d\udc13\ud80d\udc14\ud80d\udc15\ud80d\udc16\ud80d\udc17\ud80d\udc18\ud80d\udc19\ud80d\udc1a\ud80d\udc1b\ud80d\udc1c\ud80d\udc1d\ud80d\udc1e\ud80d\udc1f\ud80d\udc20\ud80d\udc21\ud80d\udc22\ud80d\udc23\ud80d\udc24\ud80d\udc25\ud80d\udc26\ud80d\udc27\ud80d\udc28\ud80d\udc29\ud80d\udc2a\ud80d\udc2b\ud80d\udc2c\ud80d\udc2d\ud80d\udc2e\ud81a\udc00\ud81a\udc01\ud81a\udc02\ud81a\udc03\ud81a\udc04\ud81a\udc05\ud81a\udc06\ud81a\udc07\ud81a\udc08\ud81a\udc09\ud81a\udc0a\ud81a\udc0b\ud81a\udc0c\ud81a\udc0d\ud81a\udc0e\ud81a\udc0f\ud81a\udc10\ud81a\udc11\ud81a\udc12\ud81a\udc13\ud81a\udc14\ud81a\udc15\ud81a\udc16\ud81a\udc17\ud81a\udc18\ud81a\udc19\ud81a\udc1a\ud81a\udc1b\ud81a\udc1c\ud81a\udc1d\ud81a\udc1e\ud81a\udc1f\ud81a\udc20\ud81a\udc21\ud81a\udc22\ud81a\udc23\ud81a\udc24\ud81a\udc25\ud81a\udc26\ud81a\udc27\ud81a\udc28\ud81a\udc29\ud81a\udc2a\ud81a\udc2b\ud81a\udc2c\ud81a\udc2d\ud81a\udc2e\ud81a\udc2f\ud81a\udc30\ud81a\udc31\ud81a\udc32\ud81a\udc33\ud81a\udc34\ud81a\udc35\ud81a\udc36\ud81a\udc37\ud81a\udc38\ud81a\udc39\ud81a\udc3a\ud81a\udc3b\ud81a\udc3c\ud81a\udc3d\ud81a\udc3e\ud81a\udc3f\ud81a\udc40\ud81a\udc41\ud81a\udc42\ud81a\udc43\ud81a\udc44\ud81a\udc45\ud81a\udc46\ud81a\udc47\ud81a\udc48\ud81a\udc49\ud81a\udc4a\ud81a\udc4b\ud81a\udc4c\ud81a\udc4d\ud81a\udc4e\ud81a\udc4f\ud81a\udc50\ud81a\udc51\ud81a\udc52\ud81a\udc53\ud81a\udc54\ud81a\udc55\ud81a\udc56\ud81a\udc57\ud81a\udc58\ud81a\udc59\ud81a\udc5a\ud81a\udc5b\ud81a\udc5c\ud81a\udc5d\ud81a\udc5e\ud81a\udc5f\ud81a\udc60\ud81a\udc61\ud81a\udc62\ud81a\udc63\ud81a\udc64\ud81a\udc65\ud81a\udc66\ud81a\udc67\ud81a\udc68\ud81a\udc69\ud81a\udc6a\ud81a\udc6b\ud81a\udc6c\ud81a\udc6d\ud81a\udc6e\ud81a\udc6f\ud81a\udc70\ud81a\udc71\ud81a\udc72\ud81a\udc73\ud81a\udc74\ud81a\udc75\ud81a\udc76\ud81a\udc77\ud81a\udc78\ud81a\udc79\ud81a\udc7a\ud81a\udc7b\ud81a\udc7c\ud81a\udc7d\ud81a\udc7e\ud81a\udc7f\ud81a\udc80\ud81a\udc81\ud81a\udc82\ud81a\udc83\ud81a\udc84\ud81a\udc85\ud81a\udc86\ud81a\udc87\ud81a\udc88\ud81a\udc89\ud81a\udc8a\ud81a\udc8b\ud81a\udc8c\ud81a\udc8d\ud81a\udc8e\ud81a\udc8f\ud81a\udc90\ud81a\udc91\ud81a\udc92\ud81a\udc93\ud81a\udc94\ud81a\udc95\ud81a\udc96\ud81a\udc97\ud81a\udc98\ud81a\udc99\ud81a\udc9a\ud81a\udc9b\ud81a\udc9c\ud81a\udc9d\ud81a\udc9e\ud81a\udc9f\ud81a\udca0\ud81a\udca1\ud81a\udca2\ud81a\udca3\ud81a\udca4\ud81a\udca5\ud81a\udca6\ud81a\udca7\ud81a\udca8\ud81a\udca9\ud81a\udcaa\ud81a\udcab\ud81a\udcac\ud81a\udcad\ud81a\udcae\ud81a\udcaf\ud81a\udcb0\ud81a\udcb1\ud81a\udcb2\ud81a\udcb3\ud81a\udcb4\ud81a\udcb5\ud81a\udcb6\ud81a\udcb7\ud81a\udcb8\ud81a\udcb9\ud81a\udcba\ud81a\udcbb\ud81a\udcbc\ud81a\udcbd\ud81a\udcbe\ud81a\udcbf\ud81a\udcc0\ud81a\udcc1\ud81a\udcc2\ud81a\udcc3\ud81a\udcc4\ud81a\udcc5\ud81a\udcc6\ud81a\udcc7\ud81a\udcc8\ud81a\udcc9\ud81a\udcca\ud81a\udccb\ud81a\udccc\ud81a\udccd\ud81a\udcce\ud81a\udccf\ud81a\udcd0\ud81a\udcd1\ud81a\udcd2\ud81a\udcd3\ud81a\udcd4\ud81a\udcd5\ud81a\udcd6\ud81a\udcd7\ud81a\udcd8\ud81a\udcd9\ud81a\udcda\ud81a\udcdb\ud81a\udcdc\ud81a\udcdd\ud81a\udcde\ud81a\udcdf\ud81a\udce0\ud81a\udce1\ud81a\udce2\ud81a\udce3\ud81a\udce4\ud81a\udce5\ud81a\udce6\ud81a\udce7\ud81a\udce8\ud81a\udce9\ud81a\udcea\ud81a\udceb\ud81a\udcec\ud81a\udced\ud81a\udcee\ud81a\udcef\ud81a\udcf0\ud81a\udcf1\ud81a\udcf2\ud81a\udcf3\ud81a\udcf4\ud81a\udcf5\ud81a\udcf6\ud81a\udcf7\ud81a\udcf8\ud81a\udcf9\ud81a\udcfa\ud81a\udcfb\ud81a\udcfc\ud81a\udcfd\ud81a\udcfe\ud81a\udcff\ud81a\udd00\ud81a\udd01\ud81a\udd02\ud81a\udd03\ud81a\udd04\ud81a\udd05\ud81a\udd06\ud81a\udd07\ud81a\udd08\ud81a\udd09\ud81a\udd0a\ud81a\udd0b\ud81a\udd0c\ud81a\udd0d\ud81a\udd0e\ud81a\udd0f\ud81a\udd10\ud81a\udd11\ud81a\udd12\ud81a\udd13\ud81a\udd14\ud81a\udd15\ud81a\udd16\ud81a\udd17\ud81a\udd18\ud81a\udd19\ud81a\udd1a\ud81a\udd1b\ud81a\udd1c\ud81a\udd1d\ud81a\udd1e\ud81a\udd1f\ud81a\udd20\ud81a\udd21\ud81a\udd22\ud81a\udd23\ud81a\udd24\ud81a\udd25\ud81a\udd26\ud81a\udd27\ud81a\udd28\ud81a\udd29\ud81a\udd2a\ud81a\udd2b\ud81a\udd2c\ud81a\udd2d\ud81a\udd2e\ud81a\udd2f\ud81a\udd30\ud81a\udd31\ud81a\udd32\ud81a\udd33\ud81a\udd34\ud81a\udd35\ud81a\udd36\ud81a\udd37\ud81a\udd38\ud81a\udd39\ud81a\udd3a\ud81a\udd3b\ud81a\udd3c\ud81a\udd3d\ud81a\udd3e\ud81a\udd3f\ud81a\udd40\ud81a\udd41\ud81a\udd42\ud81a\udd43\ud81a\udd44\ud81a\udd45\ud81a\udd46\ud81a\udd47\ud81a\udd48\ud81a\udd49\ud81a\udd4a\ud81a\udd4b\ud81a\udd4c\ud81a\udd4d\ud81a\udd4e\ud81a\udd4f\ud81a\udd50\ud81a\udd51\ud81a\udd52\ud81a\udd53\ud81a\udd54\ud81a\udd55\ud81a\udd56\ud81a\udd57\ud81a\udd58\ud81a\udd59\ud81a\udd5a\ud81a\udd5b\ud81a\udd5c\ud81a\udd5d\ud81a\udd5e\ud81a\udd5f\ud81a\udd60\ud81a\udd61\ud81a\udd62\ud81a\udd63\ud81a\udd64\ud81a\udd65\ud81a\udd66\ud81a\udd67\ud81a\udd68\ud81a\udd69\ud81a\udd6a\ud81a\udd6b\ud81a\udd6c\ud81a\udd6d\ud81a\udd6e\ud81a\udd6f\ud81a\udd70\ud81a\udd71\ud81a\udd72\ud81a\udd73\ud81a\udd74\ud81a\udd75\ud81a\udd76\ud81a\udd77\ud81a\udd78\ud81a\udd79\ud81a\udd7a\ud81a\udd7b\ud81a\udd7c\ud81a\udd7d\ud81a\udd7e\ud81a\udd7f\ud81a\udd80\ud81a\udd81\ud81a\udd82\ud81a\udd83\ud81a\udd84\ud81a\udd85\ud81a\udd86\ud81a\udd87\ud81a\udd88\ud81a\udd89\ud81a\udd8a\ud81a\udd8b\ud81a\udd8c\ud81a\udd8d\ud81a\udd8e\ud81a\udd8f\ud81a\udd90\ud81a\udd91\ud81a\udd92\ud81a\udd93\ud81a\udd94\ud81a\udd95\ud81a\udd96\ud81a\udd97\ud81a\udd98\ud81a\udd99\ud81a\udd9a\ud81a\udd9b\ud81a\udd9c\ud81a\udd9d\ud81a\udd9e\ud81a\udd9f\ud81a\udda0\ud81a\udda1\ud81a\udda2\ud81a\udda3\ud81a\udda4\ud81a\udda5\ud81a\udda6\ud81a\udda7\ud81a\udda8\ud81a\udda9\ud81a\uddaa\ud81a\uddab\ud81a\uddac\ud81a\uddad\ud81a\uddae\ud81a\uddaf\ud81a\uddb0\ud81a\uddb1\ud81a\uddb2\ud81a\uddb3\ud81a\uddb4\ud81a\uddb5\ud81a\uddb6\ud81a\uddb7\ud81a\uddb8\ud81a\uddb9\ud81a\uddba\ud81a\uddbb\ud81a\uddbc\ud81a\uddbd\ud81a\uddbe\ud81a\uddbf\ud81a\uddc0\ud81a\uddc1\ud81a\uddc2\ud81a\uddc3\ud81a\uddc4\ud81a\uddc5\ud81a\uddc6\ud81a\uddc7\ud81a\uddc8\ud81a\uddc9\ud81a\uddca\ud81a\uddcb\ud81a\uddcc\ud81a\uddcd\ud81a\uddce\ud81a\uddcf\ud81a\uddd0\ud81a\uddd1\ud81a\uddd2\ud81a\uddd3\ud81a\uddd4\ud81a\uddd5\ud81a\uddd6\ud81a\uddd7\ud81a\uddd8\ud81a\uddd9\ud81a\uddda\ud81a\udddb\ud81a\udddc\ud81a\udddd\ud81a\uddde\ud81a\udddf\ud81a\udde0\ud81a\udde1\ud81a\udde2\ud81a\udde3\ud81a\udde4\ud81a\udde5\ud81a\udde6\ud81a\udde7\ud81a\udde8\ud81a\udde9\ud81a\uddea\ud81a\uddeb\ud81a\uddec\ud81a\udded\ud81a\uddee\ud81a\uddef\ud81a\uddf0\ud81a\uddf1\ud81a\uddf2\ud81a\uddf3\ud81a\uddf4\ud81a\uddf5\ud81a\uddf6\ud81a\uddf7\ud81a\uddf8\ud81a\uddf9\ud81a\uddfa\ud81a\uddfb\ud81a\uddfc\ud81a\uddfd\ud81a\uddfe\ud81a\uddff\ud81a\ude00\ud81a\ude01\ud81a\ude02\ud81a\ude03\ud81a\ude04\ud81a\ude05\ud81a\ude06\ud81a\ude07\ud81a\ude08\ud81a\ude09\ud81a\ude0a\ud81a\ude0b\ud81a\ude0c\ud81a\ude0d\ud81a\ude0e\ud81a\ude0f\ud81a\ude10\ud81a\ude11\ud81a\ude12\ud81a\ude13\ud81a\ude14\ud81a\ude15\ud81a\ude16\ud81a\ude17\ud81a\ude18\ud81a\ude19\ud81a\ude1a\ud81a\ude1b\ud81a\ude1c\ud81a\ude1d\ud81a\ude1e\ud81a\ude1f\ud81a\ude20\ud81a\ude21\ud81a\ude22\ud81a\ude23\ud81a\ude24\ud81a\ude25\ud81a\ude26\ud81a\ude27\ud81a\ude28\ud81a\ude29\ud81a\ude2a\ud81a\ude2b\ud81a\ude2c\ud81a\ude2d\ud81a\ude2e\ud81a\ude2f\ud81a\ude30\ud81a\ude31\ud81a\ude32\ud81a\ude33\ud81a\ude34\ud81a\ude35\ud81a\ude36\ud81a\ude37\ud81a\ude38\ud82c\udc00\ud82c\udc01\ud840\udc00\ud869\uded6\ud869\udf00\ud86d\udf34\ud86d\udf40\ud86e\udc1d\u4e3d\u4e38\u4e41\ud840\udd22\u4f60\u4fae\u4fbb\u5002\u507a\u5099\u50e7\u50cf\u349e\ud841\ude3a\u514d\u5154\u5164\u5177\ud841\udd1c\u34b9\u5167\u518d\ud841\udd4b\u5197\u51a4\u4ecc\u51ac\u51b5\ud864\udddf\u51f5\u5203\u34df\u523b\u5246\u5272\u5277\u3515\u52c7\u52c9\u52e4\u52fa\u5305\u5306\u5317\u5349\u5351\u535a\u5373\u537d\u537f\u537f\u537f\ud842\ude2c\u7070\u53ca\u53df\ud842\udf63\u53eb\u53f1\u5406\u549e\u5438\u5448\u5468\u54a2\u54f6\u5510\u5553\u5563\u5584\u5584\u5599\u55ab\u55b3\u55c2\u5716\u5606\u5717\u5651\u5674\u5207\u58ee\u57ce\u57f4\u580d\u578b\u5832\u5831\u58ac\ud845\udce4\u58f2\u58f7\u5906\u591a\u5922\u5962\ud845\udea8\ud845\udeea\u59ec\u5a1b\u5a27\u59d8\u5a66\u36ee\u36fc\u5b08\u5b3e\u5b3e\ud846\uddc8\u5bc3\u5bd8\u5be7\u5bf3\ud846\udf18\u5bff\u5c06\u5f53\u5c22\u3781\u5c60\u5c6e\u5cc0\u5c8d\ud847\udde4\u5d43\ud847\udde6\u5d6e\u5d6b\u5d7c\u5de1\u5de2\u382f\u5dfd\u5e28\u5e3d\u5e69\u3862\ud848\udd83\u387c\u5eb0\u5eb3\u5eb6\u5eca\ud868\udf92\u5efe\ud848\udf31\ud848\udf31\u8201\u5f22\u5f22\u38c7\ud84c\udeb8\ud858\uddda\u5f62\u5f6b\u38e3\u5f9a\u5fcd\u5fd7\u5ff9\u6081\u393a\u391c\u6094\ud849\uded4\u60c7\u6148\u614c\u614e\u614c\u617a\u618e\u61b2\u61a4\u61af\u61de\u61f2\u61f6\u6210\u621b\u625d\u62b1\u62d4\u6350\ud84a\udf0c\u633d\u62fc\u6368\u6383\u63e4\ud84a\udff1\u6422\u63c5\u63a9\u3a2e\u6469\u647e\u649d\u6477\u3a6c\u654f\u656c\ud84c\udc0a\u65e3\u66f8\u6649\u3b19\u6691\u3b08\u3ae4\u5192\u5195\u6700\u669c\u80ad\u43d9\u6717\u671b\u6721\u675e\u6753\ud84c\udfc3\u3b49\u67fa\u6785\u6852\u6885\ud84d\udc6d\u688e\u681f\u6914\u3b9d\u6942\u69a3\u69ea\u6aa8\ud84d\udea3\u6adb\u3c18\u6b21\ud84e\udca7\u6b54\u3c4e\u6b72\u6b9f\u6bba\u6bbb\ud84e\ude8d\ud847\udd0b\ud84e\udefa\u6c4e\ud84f\udcbc\u6cbf\u6ccd\u6c67\u6d16\u6d3e\u6d77\u6d41\u6d69\u6d78\u6d85\ud84f\udd1e\u6d34\u6e2f\u6e6e\u3d33\u6ecb\u6ec7\ud84f\uded1\u6df9\u6f6e\ud84f\udf5e\ud84f\udf8e\u6fc6\u7039\u701e\u701b\u3d96\u704a\u707d\u7077\u70ad\ud841\udd25\u7145\ud850\ude63\u719c\ud850\udfab\u7228\u7235\u7250\ud851\ude08\u7280\u7295\ud851\udf35\ud852\udc14\u737a\u738b\u3eac\u73a5\u3eb8\u3eb8\u7447\u745c\u7471\u7485\u74ca\u3f1b\u7524\ud853\udc36\u753e\ud853\udc92\u7570\ud848\udd9f\u7610\ud853\udfa1\ud853\udfb8\ud854\udc44\u3ffc\u4008\u76f4\ud854\udcf3\ud854\udcf2\ud854\udd19\ud854\udd33\u771e\u771f\u771f\u774a\u4039\u778b\u4046\u4096\ud855\udc1d\u784e\u788c\u78cc\u40e3\ud855\ude26\u7956\ud855\ude9a\ud855\udec5\u798f\u79eb\u412f\u7a40\u7a4a\u7a4f\ud856\udd7c\ud856\udea7\ud856\udea7\u7aee\u4202\ud856\udfab\u7bc6\u7bc9\u4227\ud857\udc80\u7cd2\u42a0\u7ce8\u7ce3\u7d00\ud857\udf86\u7d63\u4301\u7dc7\u7e02\u7e45\u4334\ud858\ude28\ud858\ude47\u4359\ud858\uded9\u7f7a\ud858\udf3e\u7f95\u7ffa\u8005\ud859\udcda\ud859\udd23\u8060\ud859\udda8\u8070\ud84c\udf5f\u43d5\u80b2\u8103\u440b\u813e\u5ab5\ud859\udfa7\ud859\udfb5\ud84c\udf93\ud84c\udf9c\u8201\u8204\u8f9e\u446b\u8291\u828b\u829d\u52b3\u82b1\u82b3\u82bd\u82e6\ud85a\udf3c\u82e5\u831d\u8363\u83ad\u8323\u83bd\u83e7\u8457\u8353\u83ca\u83cc\u83dc\ud85b\udc36\ud85b\udd6b\ud85b\udcd5\u452b\u84f1\u84f3\u8516\ud85c\udfca\u8564\ud85b\udf2c\u455d\u4561\ud85b\udfb1\ud85c\udcd2\u456b\u8650\u865c\u8667\u8669\u86a9\u8688\u870e\u86e2\u8779\u8728\u876b\u8786\u45d7\u87e1\u8801\u45f9\u8860\u8863\ud85d\ude67\u88d7\u88de\u4635\u88fa\u34bb\ud85e\udcae\ud85e\udd66\u46be\u46c7\u8aa0\u8aed\u8b8a\u8c55\ud85f\udca8\u8cab\u8cc1\u8d1b\u8d77\ud85f\udf2f\ud842\udc04\u8dcb\u8dbc\u8df0\ud842\udcde\u8ed4\u8f38\ud861\uddd2\ud861\udded\u9094\u90f1\u9111\ud861\udf2e\u911b\u9238\u92d7\u92d8\u927c\u93f9\u9415\ud862\udffa\u958b\u4995\u95b7\ud863\udd77\u49e6\u96c3\u5db2\u9723\ud864\udd45\ud864\ude1a\u4a6e\u4a76\u97e0\ud865\udc0a\u4ab2\ud865\udc96\u980b\u980b\u9829\ud865\uddb6\u98e2\u4b33\u9929\u99a7\u99c2\u99fe\u4bce\ud866\udf30\u9b12\u9c40\u9cfd\u4cce\u4ced\u9d67\ud868\udcce\u4cf8\ud868\udd05\ud868\ude0e\ud868\ude91\u9ebb\u4d56\u9ef9\u9efe\u9f05\u9f0f\u9f16\u9f3b\ud869\ude00'
			},
			{
				'description': 'Unicode character category [Lt]: letter, titlecase',
				'decoded': [453, 456, 459, 498, 8072, 8073, 8074, 8075, 8076, 8077, 8078, 8079, 8088, 8089, 8090, 8091, 8092, 8093, 8094, 8095, 8104, 8105, 8106, 8107, 8108, 8109, 8110, 8111, 8124, 8140, 8188],
				'encoded': '\u01c5\u01c8\u01cb\u01f2\u1f88\u1f89\u1f8a\u1f8b\u1f8c\u1f8d\u1f8e\u1f8f\u1f98\u1f99\u1f9a\u1f9b\u1f9c\u1f9d\u1f9e\u1f9f\u1fa8\u1fa9\u1faa\u1fab\u1fac\u1fad\u1fae\u1faf\u1fbc\u1fcc\u1ffc'
			},
			{
				'description': 'Unicode character category [Lu]: letter, uppercase',
				'decoded': [65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 192, 193, 194, 195, 196, 197, 198, 199, 200, 201, 202, 203, 204, 205, 206, 207, 208, 209, 210, 211, 212, 213, 214, 216, 217, 218, 219, 220, 221, 222, 256, 258, 260, 262, 264, 266, 268, 270, 272, 274, 276, 278, 280, 282, 284, 286, 288, 290, 292, 294, 296, 298, 300, 302, 304, 306, 308, 310, 313, 315, 317, 319, 321, 323, 325, 327, 330, 332, 334, 336, 338, 340, 342, 344, 346, 348, 350, 352, 354, 356, 358, 360, 362, 364, 366, 368, 370, 372, 374, 376, 377, 379, 381, 385, 386, 388, 390, 391, 393, 394, 395, 398, 399, 400, 401, 403, 404, 406, 407, 408, 412, 413, 415, 416, 418, 420, 422, 423, 425, 428, 430, 431, 433, 434, 435, 437, 439, 440, 444, 452, 455, 458, 461, 463, 465, 467, 469, 471, 473, 475, 478, 480, 482, 484, 486, 488, 490, 492, 494, 497, 500, 502, 503, 504, 506, 508, 510, 512, 514, 516, 518, 520, 522, 524, 526, 528, 530, 532, 534, 536, 538, 540, 542, 544, 546, 548, 550, 552, 554, 556, 558, 560, 562, 570, 571, 573, 574, 577, 579, 580, 581, 582, 584, 586, 588, 590, 880, 882, 886, 902, 904, 905, 906, 908, 910, 911, 913, 914, 915, 916, 917, 918, 919, 920, 921, 922, 923, 924, 925, 926, 927, 928, 929, 931, 932, 933, 934, 935, 936, 937, 938, 939, 975, 978, 979, 980, 984, 986, 988, 990, 992, 994, 996, 998, 1000, 1002, 1004, 1006, 1012, 1015, 1017, 1018, 1021, 1022, 1023, 1024, 1025, 1026, 1027, 1028, 1029, 1030, 1031, 1032, 1033, 1034, 1035, 1036, 1037, 1038, 1039, 1040, 1041, 1042, 1043, 1044, 1045, 1046, 1047, 1048, 1049, 1050, 1051, 1052, 1053, 1054, 1055, 1056, 1057, 1058, 1059, 1060, 1061, 1062, 1063, 1064, 1065, 1066, 1067, 1068, 1069, 1070, 1071, 1120, 1122, 1124, 1126, 1128, 1130, 1132, 1134, 1136, 1138, 1140, 1142, 1144, 1146, 1148, 1150, 1152, 1162, 1164, 1166, 1168, 1170, 1172, 1174, 1176, 1178, 1180, 1182, 1184, 1186, 1188, 1190, 1192, 1194, 1196, 1198, 1200, 1202, 1204, 1206, 1208, 1210, 1212, 1214, 1216, 1217, 1219, 1221, 1223, 1225, 1227, 1229, 1232, 1234, 1236, 1238, 1240, 1242, 1244, 1246, 1248, 1250, 1252, 1254, 1256, 1258, 1260, 1262, 1264, 1266, 1268, 1270, 1272, 1274, 1276, 1278, 1280, 1282, 1284, 1286, 1288, 1290, 1292, 1294, 1296, 1298, 1300, 1302, 1304, 1306, 1308, 1310, 1312, 1314, 1316, 1318, 1329, 1330, 1331, 1332, 1333, 1334, 1335, 1336, 1337, 1338, 1339, 1340, 1341, 1342, 1343, 1344, 1345, 1346, 1347, 1348, 1349, 1350, 1351, 1352, 1353, 1354, 1355, 1356, 1357, 1358, 1359, 1360, 1361, 1362, 1363, 1364, 1365, 1366, 4256, 4257, 4258, 4259, 4260, 4261, 4262, 4263, 4264, 4265, 4266, 4267, 4268, 4269, 4270, 4271, 4272, 4273, 4274, 4275, 4276, 4277, 4278, 4279, 4280, 4281, 4282, 4283, 4284, 4285, 4286, 4287, 4288, 4289, 4290, 4291, 4292, 4293, 7680, 7682, 7684, 7686, 7688, 7690, 7692, 7694, 7696, 7698, 7700, 7702, 7704, 7706, 7708, 7710, 7712, 7714, 7716, 7718, 7720, 7722, 7724, 7726, 7728, 7730, 7732, 7734, 7736, 7738, 7740, 7742, 7744, 7746, 7748, 7750, 7752, 7754, 7756, 7758, 7760, 7762, 7764, 7766, 7768, 7770, 7772, 7774, 7776, 7778, 7780, 7782, 7784, 7786, 7788, 7790, 7792, 7794, 7796, 7798, 7800, 7802, 7804, 7806, 7808, 7810, 7812, 7814, 7816, 7818, 7820, 7822, 7824, 7826, 7828, 7838, 7840, 7842, 7844, 7846, 7848, 7850, 7852, 7854, 7856, 7858, 7860, 7862, 7864, 7866, 7868, 7870, 7872, 7874, 7876, 7878, 7880, 7882, 7884, 7886, 7888, 7890, 7892, 7894, 7896, 7898, 7900, 7902, 7904, 7906, 7908, 7910, 7912, 7914, 7916, 7918, 7920, 7922, 7924, 7926, 7928, 7930, 7932, 7934, 7944, 7945, 7946, 7947, 7948, 7949, 7950, 7951, 7960, 7961, 7962, 7963, 7964, 7965, 7976, 7977, 7978, 7979, 7980, 7981, 7982, 7983, 7992, 7993, 7994, 7995, 7996, 7997, 7998, 7999, 8008, 8009, 8010, 8011, 8012, 8013, 8025, 8027, 8029, 8031, 8040, 8041, 8042, 8043, 8044, 8045, 8046, 8047, 8120, 8121, 8122, 8123, 8136, 8137, 8138, 8139, 8152, 8153, 8154, 8155, 8168, 8169, 8170, 8171, 8172, 8184, 8185, 8186, 8187, 8450, 8455, 8459, 8460, 8461, 8464, 8465, 8466, 8469, 8473, 8474, 8475, 8476, 8477, 8484, 8486, 8488, 8490, 8491, 8492, 8493, 8496, 8497, 8498, 8499, 8510, 8511, 8517, 8579, 11264, 11265, 11266, 11267, 11268, 11269, 11270, 11271, 11272, 11273, 11274, 11275, 11276, 11277, 11278, 11279, 11280, 11281, 11282, 11283, 11284, 11285, 11286, 11287, 11288, 11289, 11290, 11291, 11292, 11293, 11294, 11295, 11296, 11297, 11298, 11299, 11300, 11301, 11302, 11303, 11304, 11305, 11306, 11307, 11308, 11309, 11310, 11360, 11362, 11363, 11364, 11367, 11369, 11371, 11373, 11374, 11375, 11376, 11378, 11381, 11390, 11391, 11392, 11394, 11396, 11398, 11400, 11402, 11404, 11406, 11408, 11410, 11412, 11414, 11416, 11418, 11420, 11422, 11424, 11426, 11428, 11430, 11432, 11434, 11436, 11438, 11440, 11442, 11444, 11446, 11448, 11450, 11452, 11454, 11456, 11458, 11460, 11462, 11464, 11466, 11468, 11470, 11472, 11474, 11476, 11478, 11480, 11482, 11484, 11486, 11488, 11490, 11499, 11501, 42560, 42562, 42564, 42566, 42568, 42570, 42572, 42574, 42576, 42578, 42580, 42582, 42584, 42586, 42588, 42590, 42592, 42594, 42596, 42598, 42600, 42602, 42604, 42624, 42626, 42628, 42630, 42632, 42634, 42636, 42638, 42640, 42642, 42644, 42646, 42786, 42788, 42790, 42792, 42794, 42796, 42798, 42802, 42804, 42806, 42808, 42810, 42812, 42814, 42816, 42818, 42820, 42822, 42824, 42826, 42828, 42830, 42832, 42834, 42836, 42838, 42840, 42842, 42844, 42846, 42848, 42850, 42852, 42854, 42856, 42858, 42860, 42862, 42873, 42875, 42877, 42878, 42880, 42882, 42884, 42886, 42891, 42893, 42896, 42912, 42914, 42916, 42918, 42920, 65313, 65314, 65315, 65316, 65317, 65318, 65319, 65320, 65321, 65322, 65323, 65324, 65325, 65326, 65327, 65328, 65329, 65330, 65331, 65332, 65333, 65334, 65335, 65336, 65337, 65338, 66560, 66561, 66562, 66563, 66564, 66565, 66566, 66567, 66568, 66569, 66570, 66571, 66572, 66573, 66574, 66575, 66576, 66577, 66578, 66579, 66580, 66581, 66582, 66583, 66584, 66585, 66586, 66587, 66588, 66589, 66590, 66591, 66592, 66593, 66594, 66595, 66596, 66597, 66598, 66599, 119808, 119809, 119810, 119811, 119812, 119813, 119814, 119815, 119816, 119817, 119818, 119819, 119820, 119821, 119822, 119823, 119824, 119825, 119826, 119827, 119828, 119829, 119830, 119831, 119832, 119833, 119860, 119861, 119862, 119863, 119864, 119865, 119866, 119867, 119868, 119869, 119870, 119871, 119872, 119873, 119874, 119875, 119876, 119877, 119878, 119879, 119880, 119881, 119882, 119883, 119884, 119885, 119912, 119913, 119914, 119915, 119916, 119917, 119918, 119919, 119920, 119921, 119922, 119923, 119924, 119925, 119926, 119927, 119928, 119929, 119930, 119931, 119932, 119933, 119934, 119935, 119936, 119937, 119964, 119966, 119967, 119970, 119973, 119974, 119977, 119978, 119979, 119980, 119982, 119983, 119984, 119985, 119986, 119987, 119988, 119989, 120016, 120017, 120018, 120019, 120020, 120021, 120022, 120023, 120024, 120025, 120026, 120027, 120028, 120029, 120030, 120031, 120032, 120033, 120034, 120035, 120036, 120037, 120038, 120039, 120040, 120041, 120068, 120069, 120071, 120072, 120073, 120074, 120077, 120078, 120079, 120080, 120081, 120082, 120083, 120084, 120086, 120087, 120088, 120089, 120090, 120091, 120092, 120120, 120121, 120123, 120124, 120125, 120126, 120128, 120129, 120130, 120131, 120132, 120134, 120138, 120139, 120140, 120141, 120142, 120143, 120144, 120172, 120173, 120174, 120175, 120176, 120177, 120178, 120179, 120180, 120181, 120182, 120183, 120184, 120185, 120186, 120187, 120188, 120189, 120190, 120191, 120192, 120193, 120194, 120195, 120196, 120197, 120224, 120225, 120226, 120227, 120228, 120229, 120230, 120231, 120232, 120233, 120234, 120235, 120236, 120237, 120238, 120239, 120240, 120241, 120242, 120243, 120244, 120245, 120246, 120247, 120248, 120249, 120276, 120277, 120278, 120279, 120280, 120281, 120282, 120283, 120284, 120285, 120286, 120287, 120288, 120289, 120290, 120291, 120292, 120293, 120294, 120295, 120296, 120297, 120298, 120299, 120300, 120301, 120328, 120329, 120330, 120331, 120332, 120333, 120334, 120335, 120336, 120337, 120338, 120339, 120340, 120341, 120342, 120343, 120344, 120345, 120346, 120347, 120348, 120349, 120350, 120351, 120352, 120353, 120380, 120381, 120382, 120383, 120384, 120385, 120386, 120387, 120388, 120389, 120390, 120391, 120392, 120393, 120394, 120395, 120396, 120397, 120398, 120399, 120400, 120401, 120402, 120403, 120404, 120405, 120432, 120433, 120434, 120435, 120436, 120437, 120438, 120439, 120440, 120441, 120442, 120443, 120444, 120445, 120446, 120447, 120448, 120449, 120450, 120451, 120452, 120453, 120454, 120455, 120456, 120457, 120488, 120489, 120490, 120491, 120492, 120493, 120494, 120495, 120496, 120497, 120498, 120499, 120500, 120501, 120502, 120503, 120504, 120505, 120506, 120507, 120508, 120509, 120510, 120511, 120512, 120546, 120547, 120548, 120549, 120550, 120551, 120552, 120553, 120554, 120555, 120556, 120557, 120558, 120559, 120560, 120561, 120562, 120563, 120564, 120565, 120566, 120567, 120568, 120569, 120570, 120604, 120605, 120606, 120607, 120608, 120609, 120610, 120611, 120612, 120613, 120614, 120615, 120616, 120617, 120618, 120619, 120620, 120621, 120622, 120623, 120624, 120625, 120626, 120627, 120628, 120662, 120663, 120664, 120665, 120666, 120667, 120668, 120669, 120670, 120671, 120672, 120673, 120674, 120675, 120676, 120677, 120678, 120679, 120680, 120681, 120682, 120683, 120684, 120685, 120686, 120720, 120721, 120722, 120723, 120724, 120725, 120726, 120727, 120728, 120729, 120730, 120731, 120732, 120733, 120734, 120735, 120736, 120737, 120738, 120739, 120740, 120741, 120742, 120743, 120744, 120778],
				'encoded': 'ABCDEFGHIJKLMNOPQRSTUVWXYZ\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd8\xd9\xda\xdb\xdc\xdd\xde\u0100\u0102\u0104\u0106\u0108\u010a\u010c\u010e\u0110\u0112\u0114\u0116\u0118\u011a\u011c\u011e\u0120\u0122\u0124\u0126\u0128\u012a\u012c\u012e\u0130\u0132\u0134\u0136\u0139\u013b\u013d\u013f\u0141\u0143\u0145\u0147\u014a\u014c\u014e\u0150\u0152\u0154\u0156\u0158\u015a\u015c\u015e\u0160\u0162\u0164\u0166\u0168\u016a\u016c\u016e\u0170\u0172\u0174\u0176\u0178\u0179\u017b\u017d\u0181\u0182\u0184\u0186\u0187\u0189\u018a\u018b\u018e\u018f\u0190\u0191\u0193\u0194\u0196\u0197\u0198\u019c\u019d\u019f\u01a0\u01a2\u01a4\u01a6\u01a7\u01a9\u01ac\u01ae\u01af\u01b1\u01b2\u01b3\u01b5\u01b7\u01b8\u01bc\u01c4\u01c7\u01ca\u01cd\u01cf\u01d1\u01d3\u01d5\u01d7\u01d9\u01db\u01de\u01e0\u01e2\u01e4\u01e6\u01e8\u01ea\u01ec\u01ee\u01f1\u01f4\u01f6\u01f7\u01f8\u01fa\u01fc\u01fe\u0200\u0202\u0204\u0206\u0208\u020a\u020c\u020e\u0210\u0212\u0214\u0216\u0218\u021a\u021c\u021e\u0220\u0222\u0224\u0226\u0228\u022a\u022c\u022e\u0230\u0232\u023a\u023b\u023d\u023e\u0241\u0243\u0244\u0245\u0246\u0248\u024a\u024c\u024e\u0370\u0372\u0376\u0386\u0388\u0389\u038a\u038c\u038e\u038f\u0391\u0392\u0393\u0394\u0395\u0396\u0397\u0398\u0399\u039a\u039b\u039c\u039d\u039e\u039f\u03a0\u03a1\u03a3\u03a4\u03a5\u03a6\u03a7\u03a8\u03a9\u03aa\u03ab\u03cf\u03d2\u03d3\u03d4\u03d8\u03da\u03dc\u03de\u03e0\u03e2\u03e4\u03e6\u03e8\u03ea\u03ec\u03ee\u03f4\u03f7\u03f9\u03fa\u03fd\u03fe\u03ff\u0400\u0401\u0402\u0403\u0404\u0405\u0406\u0407\u0408\u0409\u040a\u040b\u040c\u040d\u040e\u040f\u0410\u0411\u0412\u0413\u0414\u0415\u0416\u0417\u0418\u0419\u041a\u041b\u041c\u041d\u041e\u041f\u0420\u0421\u0422\u0423\u0424\u0425\u0426\u0427\u0428\u0429\u042a\u042b\u042c\u042d\u042e\u042f\u0460\u0462\u0464\u0466\u0468\u046a\u046c\u046e\u0470\u0472\u0474\u0476\u0478\u047a\u047c\u047e\u0480\u048a\u048c\u048e\u0490\u0492\u0494\u0496\u0498\u049a\u049c\u049e\u04a0\u04a2\u04a4\u04a6\u04a8\u04aa\u04ac\u04ae\u04b0\u04b2\u04b4\u04b6\u04b8\u04ba\u04bc\u04be\u04c0\u04c1\u04c3\u04c5\u04c7\u04c9\u04cb\u04cd\u04d0\u04d2\u04d4\u04d6\u04d8\u04da\u04dc\u04de\u04e0\u04e2\u04e4\u04e6\u04e8\u04ea\u04ec\u04ee\u04f0\u04f2\u04f4\u04f6\u04f8\u04fa\u04fc\u04fe\u0500\u0502\u0504\u0506\u0508\u050a\u050c\u050e\u0510\u0512\u0514\u0516\u0518\u051a\u051c\u051e\u0520\u0522\u0524\u0526\u0531\u0532\u0533\u0534\u0535\u0536\u0537\u0538\u0539\u053a\u053b\u053c\u053d\u053e\u053f\u0540\u0541\u0542\u0543\u0544\u0545\u0546\u0547\u0548\u0549\u054a\u054b\u054c\u054d\u054e\u054f\u0550\u0551\u0552\u0553\u0554\u0555\u0556\u10a0\u10a1\u10a2\u10a3\u10a4\u10a5\u10a6\u10a7\u10a8\u10a9\u10aa\u10ab\u10ac\u10ad\u10ae\u10af\u10b0\u10b1\u10b2\u10b3\u10b4\u10b5\u10b6\u10b7\u10b8\u10b9\u10ba\u10bb\u10bc\u10bd\u10be\u10bf\u10c0\u10c1\u10c2\u10c3\u10c4\u10c5\u1e00\u1e02\u1e04\u1e06\u1e08\u1e0a\u1e0c\u1e0e\u1e10\u1e12\u1e14\u1e16\u1e18\u1e1a\u1e1c\u1e1e\u1e20\u1e22\u1e24\u1e26\u1e28\u1e2a\u1e2c\u1e2e\u1e30\u1e32\u1e34\u1e36\u1e38\u1e3a\u1e3c\u1e3e\u1e40\u1e42\u1e44\u1e46\u1e48\u1e4a\u1e4c\u1e4e\u1e50\u1e52\u1e54\u1e56\u1e58\u1e5a\u1e5c\u1e5e\u1e60\u1e62\u1e64\u1e66\u1e68\u1e6a\u1e6c\u1e6e\u1e70\u1e72\u1e74\u1e76\u1e78\u1e7a\u1e7c\u1e7e\u1e80\u1e82\u1e84\u1e86\u1e88\u1e8a\u1e8c\u1e8e\u1e90\u1e92\u1e94\u1e9e\u1ea0\u1ea2\u1ea4\u1ea6\u1ea8\u1eaa\u1eac\u1eae\u1eb0\u1eb2\u1eb4\u1eb6\u1eb8\u1eba\u1ebc\u1ebe\u1ec0\u1ec2\u1ec4\u1ec6\u1ec8\u1eca\u1ecc\u1ece\u1ed0\u1ed2\u1ed4\u1ed6\u1ed8\u1eda\u1edc\u1ede\u1ee0\u1ee2\u1ee4\u1ee6\u1ee8\u1eea\u1eec\u1eee\u1ef0\u1ef2\u1ef4\u1ef6\u1ef8\u1efa\u1efc\u1efe\u1f08\u1f09\u1f0a\u1f0b\u1f0c\u1f0d\u1f0e\u1f0f\u1f18\u1f19\u1f1a\u1f1b\u1f1c\u1f1d\u1f28\u1f29\u1f2a\u1f2b\u1f2c\u1f2d\u1f2e\u1f2f\u1f38\u1f39\u1f3a\u1f3b\u1f3c\u1f3d\u1f3e\u1f3f\u1f48\u1f49\u1f4a\u1f4b\u1f4c\u1f4d\u1f59\u1f5b\u1f5d\u1f5f\u1f68\u1f69\u1f6a\u1f6b\u1f6c\u1f6d\u1f6e\u1f6f\u1fb8\u1fb9\u1fba\u1fbb\u1fc8\u1fc9\u1fca\u1fcb\u1fd8\u1fd9\u1fda\u1fdb\u1fe8\u1fe9\u1fea\u1feb\u1fec\u1ff8\u1ff9\u1ffa\u1ffb\u2102\u2107\u210b\u210c\u210d\u2110\u2111\u2112\u2115\u2119\u211a\u211b\u211c\u211d\u2124\u2126\u2128\u212a\u212b\u212c\u212d\u2130\u2131\u2132\u2133\u213e\u213f\u2145\u2183\u2c00\u2c01\u2c02\u2c03\u2c04\u2c05\u2c06\u2c07\u2c08\u2c09\u2c0a\u2c0b\u2c0c\u2c0d\u2c0e\u2c0f\u2c10\u2c11\u2c12\u2c13\u2c14\u2c15\u2c16\u2c17\u2c18\u2c19\u2c1a\u2c1b\u2c1c\u2c1d\u2c1e\u2c1f\u2c20\u2c21\u2c22\u2c23\u2c24\u2c25\u2c26\u2c27\u2c28\u2c29\u2c2a\u2c2b\u2c2c\u2c2d\u2c2e\u2c60\u2c62\u2c63\u2c64\u2c67\u2c69\u2c6b\u2c6d\u2c6e\u2c6f\u2c70\u2c72\u2c75\u2c7e\u2c7f\u2c80\u2c82\u2c84\u2c86\u2c88\u2c8a\u2c8c\u2c8e\u2c90\u2c92\u2c94\u2c96\u2c98\u2c9a\u2c9c\u2c9e\u2ca0\u2ca2\u2ca4\u2ca6\u2ca8\u2caa\u2cac\u2cae\u2cb0\u2cb2\u2cb4\u2cb6\u2cb8\u2cba\u2cbc\u2cbe\u2cc0\u2cc2\u2cc4\u2cc6\u2cc8\u2cca\u2ccc\u2cce\u2cd0\u2cd2\u2cd4\u2cd6\u2cd8\u2cda\u2cdc\u2cde\u2ce0\u2ce2\u2ceb\u2ced\ua640\ua642\ua644\ua646\ua648\ua64a\ua64c\ua64e\ua650\ua652\ua654\ua656\ua658\ua65a\ua65c\ua65e\ua660\ua662\ua664\ua666\ua668\ua66a\ua66c\ua680\ua682\ua684\ua686\ua688\ua68a\ua68c\ua68e\ua690\ua692\ua694\ua696\ua722\ua724\ua726\ua728\ua72a\ua72c\ua72e\ua732\ua734\ua736\ua738\ua73a\ua73c\ua73e\ua740\ua742\ua744\ua746\ua748\ua74a\ua74c\ua74e\ua750\ua752\ua754\ua756\ua758\ua75a\ua75c\ua75e\ua760\ua762\ua764\ua766\ua768\ua76a\ua76c\ua76e\ua779\ua77b\ua77d\ua77e\ua780\ua782\ua784\ua786\ua78b\ua78d\ua790\ua7a0\ua7a2\ua7a4\ua7a6\ua7a8\uff21\uff22\uff23\uff24\uff25\uff26\uff27\uff28\uff29\uff2a\uff2b\uff2c\uff2d\uff2e\uff2f\uff30\uff31\uff32\uff33\uff34\uff35\uff36\uff37\uff38\uff39\uff3a\ud801\udc00\ud801\udc01\ud801\udc02\ud801\udc03\ud801\udc04\ud801\udc05\ud801\udc06\ud801\udc07\ud801\udc08\ud801\udc09\ud801\udc0a\ud801\udc0b\ud801\udc0c\ud801\udc0d\ud801\udc0e\ud801\udc0f\ud801\udc10\ud801\udc11\ud801\udc12\ud801\udc13\ud801\udc14\ud801\udc15\ud801\udc16\ud801\udc17\ud801\udc18\ud801\udc19\ud801\udc1a\ud801\udc1b\ud801\udc1c\ud801\udc1d\ud801\udc1e\ud801\udc1f\ud801\udc20\ud801\udc21\ud801\udc22\ud801\udc23\ud801\udc24\ud801\udc25\ud801\udc26\ud801\udc27\ud835\udc00\ud835\udc01\ud835\udc02\ud835\udc03\ud835\udc04\ud835\udc05\ud835\udc06\ud835\udc07\ud835\udc08\ud835\udc09\ud835\udc0a\ud835\udc0b\ud835\udc0c\ud835\udc0d\ud835\udc0e\ud835\udc0f\ud835\udc10\ud835\udc11\ud835\udc12\ud835\udc13\ud835\udc14\ud835\udc15\ud835\udc16\ud835\udc17\ud835\udc18\ud835\udc19\ud835\udc34\ud835\udc35\ud835\udc36\ud835\udc37\ud835\udc38\ud835\udc39\ud835\udc3a\ud835\udc3b\ud835\udc3c\ud835\udc3d\ud835\udc3e\ud835\udc3f\ud835\udc40\ud835\udc41\ud835\udc42\ud835\udc43\ud835\udc44\ud835\udc45\ud835\udc46\ud835\udc47\ud835\udc48\ud835\udc49\ud835\udc4a\ud835\udc4b\ud835\udc4c\ud835\udc4d\ud835\udc68\ud835\udc69\ud835\udc6a\ud835\udc6b\ud835\udc6c\ud835\udc6d\ud835\udc6e\ud835\udc6f\ud835\udc70\ud835\udc71\ud835\udc72\ud835\udc73\ud835\udc74\ud835\udc75\ud835\udc76\ud835\udc77\ud835\udc78\ud835\udc79\ud835\udc7a\ud835\udc7b\ud835\udc7c\ud835\udc7d\ud835\udc7e\ud835\udc7f\ud835\udc80\ud835\udc81\ud835\udc9c\ud835\udc9e\ud835\udc9f\ud835\udca2\ud835\udca5\ud835\udca6\ud835\udca9\ud835\udcaa\ud835\udcab\ud835\udcac\ud835\udcae\ud835\udcaf\ud835\udcb0\ud835\udcb1\ud835\udcb2\ud835\udcb3\ud835\udcb4\ud835\udcb5\ud835\udcd0\ud835\udcd1\ud835\udcd2\ud835\udcd3\ud835\udcd4\ud835\udcd5\ud835\udcd6\ud835\udcd7\ud835\udcd8\ud835\udcd9\ud835\udcda\ud835\udcdb\ud835\udcdc\ud835\udcdd\ud835\udcde\ud835\udcdf\ud835\udce0\ud835\udce1\ud835\udce2\ud835\udce3\ud835\udce4\ud835\udce5\ud835\udce6\ud835\udce7\ud835\udce8\ud835\udce9\ud835\udd04\ud835\udd05\ud835\udd07\ud835\udd08\ud835\udd09\ud835\udd0a\ud835\udd0d\ud835\udd0e\ud835\udd0f\ud835\udd10\ud835\udd11\ud835\udd12\ud835\udd13\ud835\udd14\ud835\udd16\ud835\udd17\ud835\udd18\ud835\udd19\ud835\udd1a\ud835\udd1b\ud835\udd1c\ud835\udd38\ud835\udd39\ud835\udd3b\ud835\udd3c\ud835\udd3d\ud835\udd3e\ud835\udd40\ud835\udd41\ud835\udd42\ud835\udd43\ud835\udd44\ud835\udd46\ud835\udd4a\ud835\udd4b\ud835\udd4c\ud835\udd4d\ud835\udd4e\ud835\udd4f\ud835\udd50\ud835\udd6c\ud835\udd6d\ud835\udd6e\ud835\udd6f\ud835\udd70\ud835\udd71\ud835\udd72\ud835\udd73\ud835\udd74\ud835\udd75\ud835\udd76\ud835\udd77\ud835\udd78\ud835\udd79\ud835\udd7a\ud835\udd7b\ud835\udd7c\ud835\udd7d\ud835\udd7e\ud835\udd7f\ud835\udd80\ud835\udd81\ud835\udd82\ud835\udd83\ud835\udd84\ud835\udd85\ud835\udda0\ud835\udda1\ud835\udda2\ud835\udda3\ud835\udda4\ud835\udda5\ud835\udda6\ud835\udda7\ud835\udda8\ud835\udda9\ud835\uddaa\ud835\uddab\ud835\uddac\ud835\uddad\ud835\uddae\ud835\uddaf\ud835\uddb0\ud835\uddb1\ud835\uddb2\ud835\uddb3\ud835\uddb4\ud835\uddb5\ud835\uddb6\ud835\uddb7\ud835\uddb8\ud835\uddb9\ud835\uddd4\ud835\uddd5\ud835\uddd6\ud835\uddd7\ud835\uddd8\ud835\uddd9\ud835\uddda\ud835\udddb\ud835\udddc\ud835\udddd\ud835\uddde\ud835\udddf\ud835\udde0\ud835\udde1\ud835\udde2\ud835\udde3\ud835\udde4\ud835\udde5\ud835\udde6\ud835\udde7\ud835\udde8\ud835\udde9\ud835\uddea\ud835\uddeb\ud835\uddec\ud835\udded\ud835\ude08\ud835\ude09\ud835\ude0a\ud835\ude0b\ud835\ude0c\ud835\ude0d\ud835\ude0e\ud835\ude0f\ud835\ude10\ud835\ude11\ud835\ude12\ud835\ude13\ud835\ude14\ud835\ude15\ud835\ude16\ud835\ude17\ud835\ude18\ud835\ude19\ud835\ude1a\ud835\ude1b\ud835\ude1c\ud835\ude1d\ud835\ude1e\ud835\ude1f\ud835\ude20\ud835\ude21\ud835\ude3c\ud835\ude3d\ud835\ude3e\ud835\ude3f\ud835\ude40\ud835\ude41\ud835\ude42\ud835\ude43\ud835\ude44\ud835\ude45\ud835\ude46\ud835\ude47\ud835\ude48\ud835\ude49\ud835\ude4a\ud835\ude4b\ud835\ude4c\ud835\ude4d\ud835\ude4e\ud835\ude4f\ud835\ude50\ud835\ude51\ud835\ude52\ud835\ude53\ud835\ude54\ud835\ude55\ud835\ude70\ud835\ude71\ud835\ude72\ud835\ude73\ud835\ude74\ud835\ude75\ud835\ude76\ud835\ude77\ud835\ude78\ud835\ude79\ud835\ude7a\ud835\ude7b\ud835\ude7c\ud835\ude7d\ud835\ude7e\ud835\ude7f\ud835\ude80\ud835\ude81\ud835\ude82\ud835\ude83\ud835\ude84\ud835\ude85\ud835\ude86\ud835\ude87\ud835\ude88\ud835\ude89\ud835\udea8\ud835\udea9\ud835\udeaa\ud835\udeab\ud835\udeac\ud835\udead\ud835\udeae\ud835\udeaf\ud835\udeb0\ud835\udeb1\ud835\udeb2\ud835\udeb3\ud835\udeb4\ud835\udeb5\ud835\udeb6\ud835\udeb7\ud835\udeb8\ud835\udeb9\ud835\udeba\ud835\udebb\ud835\udebc\ud835\udebd\ud835\udebe\ud835\udebf\ud835\udec0\ud835\udee2\ud835\udee3\ud835\udee4\ud835\udee5\ud835\udee6\ud835\udee7\ud835\udee8\ud835\udee9\ud835\udeea\ud835\udeeb\ud835\udeec\ud835\udeed\ud835\udeee\ud835\udeef\ud835\udef0\ud835\udef1\ud835\udef2\ud835\udef3\ud835\udef4\ud835\udef5\ud835\udef6\ud835\udef7\ud835\udef8\ud835\udef9\ud835\udefa\ud835\udf1c\ud835\udf1d\ud835\udf1e\ud835\udf1f\ud835\udf20\ud835\udf21\ud835\udf22\ud835\udf23\ud835\udf24\ud835\udf25\ud835\udf26\ud835\udf27\ud835\udf28\ud835\udf29\ud835\udf2a\ud835\udf2b\ud835\udf2c\ud835\udf2d\ud835\udf2e\ud835\udf2f\ud835\udf30\ud835\udf31\ud835\udf32\ud835\udf33\ud835\udf34\ud835\udf56\ud835\udf57\ud835\udf58\ud835\udf59\ud835\udf5a\ud835\udf5b\ud835\udf5c\ud835\udf5d\ud835\udf5e\ud835\udf5f\ud835\udf60\ud835\udf61\ud835\udf62\ud835\udf63\ud835\udf64\ud835\udf65\ud835\udf66\ud835\udf67\ud835\udf68\ud835\udf69\ud835\udf6a\ud835\udf6b\ud835\udf6c\ud835\udf6d\ud835\udf6e\ud835\udf90\ud835\udf91\ud835\udf92\ud835\udf93\ud835\udf94\ud835\udf95\ud835\udf96\ud835\udf97\ud835\udf98\ud835\udf99\ud835\udf9a\ud835\udf9b\ud835\udf9c\ud835\udf9d\ud835\udf9e\ud835\udf9f\ud835\udfa0\ud835\udfa1\ud835\udfa2\ud835\udfa3\ud835\udfa4\ud835\udfa5\ud835\udfa6\ud835\udfa7\ud835\udfa8\ud835\udfca'
			},
			{
				'description': 'Unicode character category [Mc]: mark, spacing combining',
				'decoded': [2307, 2363, 2366, 2367, 2368, 2377, 2378, 2379, 2380, 2382, 2383, 2434, 2435, 2494, 2495, 2496, 2503, 2504, 2507, 2508, 2519, 2563, 2622, 2623, 2624, 2691, 2750, 2751, 2752, 2761, 2763, 2764, 2818, 2819, 2878, 2880, 2887, 2888, 2891, 2892, 2903, 3006, 3007, 3009, 3010, 3014, 3015, 3016, 3018, 3019, 3020, 3031, 3073, 3074, 3075, 3137, 3138, 3139, 3140, 3202, 3203, 3262, 3264, 3265, 3266, 3267, 3268, 3271, 3272, 3274, 3275, 3285, 3286, 3330, 3331, 3390, 3391, 3392, 3398, 3399, 3400, 3402, 3403, 3404, 3415, 3458, 3459, 3535, 3536, 3537, 3544, 3545, 3546, 3547, 3548, 3549, 3550, 3551, 3570, 3571, 3902, 3903, 3967, 4139, 4140, 4145, 4152, 4155, 4156, 4182, 4183, 4194, 4195, 4196, 4199, 4200, 4201, 4202, 4203, 4204, 4205, 4227, 4228, 4231, 4232, 4233, 4234, 4235, 4236, 4239, 4250, 4251, 4252, 6070, 6078, 6079, 6080, 6081, 6082, 6083, 6084, 6085, 6087, 6088, 6435, 6436, 6437, 6438, 6441, 6442, 6443, 6448, 6449, 6451, 6452, 6453, 6454, 6455, 6456, 6576, 6577, 6578, 6579, 6580, 6581, 6582, 6583, 6584, 6585, 6586, 6587, 6588, 6589, 6590, 6591, 6592, 6600, 6601, 6681, 6682, 6683, 6741, 6743, 6753, 6755, 6756, 6765, 6766, 6767, 6768, 6769, 6770, 6916, 6965, 6971, 6973, 6974, 6975, 6976, 6977, 6979, 6980, 7042, 7073, 7078, 7079, 7082, 7143, 7146, 7147, 7148, 7150, 7154, 7155, 7204, 7205, 7206, 7207, 7208, 7209, 7210, 7211, 7220, 7221, 7393, 7410, 43043, 43044, 43047, 43136, 43137, 43188, 43189, 43190, 43191, 43192, 43193, 43194, 43195, 43196, 43197, 43198, 43199, 43200, 43201, 43202, 43203, 43346, 43347, 43395, 43444, 43445, 43450, 43451, 43453, 43454, 43455, 43456, 43567, 43568, 43571, 43572, 43597, 43643, 44003, 44004, 44006, 44007, 44009, 44010, 44012, 69632, 69634, 69762, 69808, 69809, 69810, 69815, 69816, 119141, 119142, 119149, 119150, 119151, 119152, 119153, 119154],
				'encoded': '\u0903\u093b\u093e\u093f\u0940\u0949\u094a\u094b\u094c\u094e\u094f\u0982\u0983\u09be\u09bf\u09c0\u09c7\u09c8\u09cb\u09cc\u09d7\u0a03\u0a3e\u0a3f\u0a40\u0a83\u0abe\u0abf\u0ac0\u0ac9\u0acb\u0acc\u0b02\u0b03\u0b3e\u0b40\u0b47\u0b48\u0b4b\u0b4c\u0b57\u0bbe\u0bbf\u0bc1\u0bc2\u0bc6\u0bc7\u0bc8\u0bca\u0bcb\u0bcc\u0bd7\u0c01\u0c02\u0c03\u0c41\u0c42\u0c43\u0c44\u0c82\u0c83\u0cbe\u0cc0\u0cc1\u0cc2\u0cc3\u0cc4\u0cc7\u0cc8\u0cca\u0ccb\u0cd5\u0cd6\u0d02\u0d03\u0d3e\u0d3f\u0d40\u0d46\u0d47\u0d48\u0d4a\u0d4b\u0d4c\u0d57\u0d82\u0d83\u0dcf\u0dd0\u0dd1\u0dd8\u0dd9\u0dda\u0ddb\u0ddc\u0ddd\u0dde\u0ddf\u0df2\u0df3\u0f3e\u0f3f\u0f7f\u102b\u102c\u1031\u1038\u103b\u103c\u1056\u1057\u1062\u1063\u1064\u1067\u1068\u1069\u106a\u106b\u106c\u106d\u1083\u1084\u1087\u1088\u1089\u108a\u108b\u108c\u108f\u109a\u109b\u109c\u17b6\u17be\u17bf\u17c0\u17c1\u17c2\u17c3\u17c4\u17c5\u17c7\u17c8\u1923\u1924\u1925\u1926\u1929\u192a\u192b\u1930\u1931\u1933\u1934\u1935\u1936\u1937\u1938\u19b0\u19b1\u19b2\u19b3\u19b4\u19b5\u19b6\u19b7\u19b8\u19b9\u19ba\u19bb\u19bc\u19bd\u19be\u19bf\u19c0\u19c8\u19c9\u1a19\u1a1a\u1a1b\u1a55\u1a57\u1a61\u1a63\u1a64\u1a6d\u1a6e\u1a6f\u1a70\u1a71\u1a72\u1b04\u1b35\u1b3b\u1b3d\u1b3e\u1b3f\u1b40\u1b41\u1b43\u1b44\u1b82\u1ba1\u1ba6\u1ba7\u1baa\u1be7\u1bea\u1beb\u1bec\u1bee\u1bf2\u1bf3\u1c24\u1c25\u1c26\u1c27\u1c28\u1c29\u1c2a\u1c2b\u1c34\u1c35\u1ce1\u1cf2\ua823\ua824\ua827\ua880\ua881\ua8b4\ua8b5\ua8b6\ua8b7\ua8b8\ua8b9\ua8ba\ua8bb\ua8bc\ua8bd\ua8be\ua8bf\ua8c0\ua8c1\ua8c2\ua8c3\ua952\ua953\ua983\ua9b4\ua9b5\ua9ba\ua9bb\ua9bd\ua9be\ua9bf\ua9c0\uaa2f\uaa30\uaa33\uaa34\uaa4d\uaa7b\uabe3\uabe4\uabe6\uabe7\uabe9\uabea\uabec\ud804\udc00\ud804\udc02\ud804\udc82\ud804\udcb0\ud804\udcb1\ud804\udcb2\ud804\udcb7\ud804\udcb8\ud834\udd65\ud834\udd66\ud834\udd6d\ud834\udd6e\ud834\udd6f\ud834\udd70\ud834\udd71\ud834\udd72'
			},
			{
				'description': 'Unicode character category [Me]: mark, enclosing',
				'decoded': [1160, 1161, 8413, 8414, 8415, 8416, 8418, 8419, 8420, 42608, 42609, 42610],
				'encoded': '\u0488\u0489\u20dd\u20de\u20df\u20e0\u20e2\u20e3\u20e4\ua670\ua671\ua672'
			},
			{
				'description': 'Unicode character category [Mn]: mark, non-spacing',
				'decoded': [768, 769, 770, 771, 772, 773, 774, 775, 776, 777, 778, 779, 780, 781, 782, 783, 784, 785, 786, 787, 788, 789, 790, 791, 792, 793, 794, 795, 796, 797, 798, 799, 800, 801, 802, 803, 804, 805, 806, 807, 808, 809, 810, 811, 812, 813, 814, 815, 816, 817, 818, 819, 820, 821, 822, 823, 824, 825, 826, 827, 828, 829, 830, 831, 832, 833, 834, 835, 836, 837, 838, 839, 840, 841, 842, 843, 844, 845, 846, 847, 848, 849, 850, 851, 852, 853, 854, 855, 856, 857, 858, 859, 860, 861, 862, 863, 864, 865, 866, 867, 868, 869, 870, 871, 872, 873, 874, 875, 876, 877, 878, 879, 1155, 1156, 1157, 1158, 1159, 1425, 1426, 1427, 1428, 1429, 1430, 1431, 1432, 1433, 1434, 1435, 1436, 1437, 1438, 1439, 1440, 1441, 1442, 1443, 1444, 1445, 1446, 1447, 1448, 1449, 1450, 1451, 1452, 1453, 1454, 1455, 1456, 1457, 1458, 1459, 1460, 1461, 1462, 1463, 1464, 1465, 1466, 1467, 1468, 1469, 1471, 1473, 1474, 1476, 1477, 1479, 1552, 1553, 1554, 1555, 1556, 1557, 1558, 1559, 1560, 1561, 1562, 1611, 1612, 1613, 1614, 1615, 1616, 1617, 1618, 1619, 1620, 1621, 1622, 1623, 1624, 1625, 1626, 1627, 1628, 1629, 1630, 1631, 1648, 1750, 1751, 1752, 1753, 1754, 1755, 1756, 1759, 1760, 1761, 1762, 1763, 1764, 1767, 1768, 1770, 1771, 1772, 1773, 1809, 1840, 1841, 1842, 1843, 1844, 1845, 1846, 1847, 1848, 1849, 1850, 1851, 1852, 1853, 1854, 1855, 1856, 1857, 1858, 1859, 1860, 1861, 1862, 1863, 1864, 1865, 1866, 1958, 1959, 1960, 1961, 1962, 1963, 1964, 1965, 1966, 1967, 1968, 2027, 2028, 2029, 2030, 2031, 2032, 2033, 2034, 2035, 2070, 2071, 2072, 2073, 2075, 2076, 2077, 2078, 2079, 2080, 2081, 2082, 2083, 2085, 2086, 2087, 2089, 2090, 2091, 2092, 2093, 2137, 2138, 2139, 2304, 2305, 2306, 2362, 2364, 2369, 2370, 2371, 2372, 2373, 2374, 2375, 2376, 2381, 2385, 2386, 2387, 2388, 2389, 2390, 2391, 2402, 2403, 2433, 2492, 2497, 2498, 2499, 2500, 2509, 2530, 2531, 2561, 2562, 2620, 2625, 2626, 2631, 2632, 2635, 2636, 2637, 2641, 2672, 2673, 2677, 2689, 2690, 2748, 2753, 2754, 2755, 2756, 2757, 2759, 2760, 2765, 2786, 2787, 2817, 2876, 2879, 2881, 2882, 2883, 2884, 2893, 2902, 2914, 2915, 2946, 3008, 3021, 3134, 3135, 3136, 3142, 3143, 3144, 3146, 3147, 3148, 3149, 3157, 3158, 3170, 3171, 3260, 3263, 3270, 3276, 3277, 3298, 3299, 3393, 3394, 3395, 3396, 3405, 3426, 3427, 3530, 3538, 3539, 3540, 3542, 3633, 3636, 3637, 3638, 3639, 3640, 3641, 3642, 3655, 3656, 3657, 3658, 3659, 3660, 3661, 3662, 3761, 3764, 3765, 3766, 3767, 3768, 3769, 3771, 3772, 3784, 3785, 3786, 3787, 3788, 3789, 3864, 3865, 3893, 3895, 3897, 3953, 3954, 3955, 3956, 3957, 3958, 3959, 3960, 3961, 3962, 3963, 3964, 3965, 3966, 3968, 3969, 3970, 3971, 3972, 3974, 3975, 3981, 3982, 3983, 3984, 3985, 3986, 3987, 3988, 3989, 3990, 3991, 3993, 3994, 3995, 3996, 3997, 3998, 3999, 4000, 4001, 4002, 4003, 4004, 4005, 4006, 4007, 4008, 4009, 4010, 4011, 4012, 4013, 4014, 4015, 4016, 4017, 4018, 4019, 4020, 4021, 4022, 4023, 4024, 4025, 4026, 4027, 4028, 4038, 4141, 4142, 4143, 4144, 4146, 4147, 4148, 4149, 4150, 4151, 4153, 4154, 4157, 4158, 4184, 4185, 4190, 4191, 4192, 4209, 4210, 4211, 4212, 4226, 4229, 4230, 4237, 4253, 4957, 4958, 4959, 5906, 5907, 5908, 5938, 5939, 5940, 5970, 5971, 6002, 6003, 6071, 6072, 6073, 6074, 6075, 6076, 6077, 6086, 6089, 6090, 6091, 6092, 6093, 6094, 6095, 6096, 6097, 6098, 6099, 6109, 6155, 6156, 6157, 6313, 6432, 6433, 6434, 6439, 6440, 6450, 6457, 6458, 6459, 6679, 6680, 6742, 6744, 6745, 6746, 6747, 6748, 6749, 6750, 6752, 6754, 6757, 6758, 6759, 6760, 6761, 6762, 6763, 6764, 6771, 6772, 6773, 6774, 6775, 6776, 6777, 6778, 6779, 6780, 6783, 6912, 6913, 6914, 6915, 6964, 6966, 6967, 6968, 6969, 6970, 6972, 6978, 7019, 7020, 7021, 7022, 7023, 7024, 7025, 7026, 7027, 7040, 7041, 7074, 7075, 7076, 7077, 7080, 7081, 7142, 7144, 7145, 7149, 7151, 7152, 7153, 7212, 7213, 7214, 7215, 7216, 7217, 7218, 7219, 7222, 7223, 7376, 7377, 7378, 7380, 7381, 7382, 7383, 7384, 7385, 7386, 7387, 7388, 7389, 7390, 7391, 7392, 7394, 7395, 7396, 7397, 7398, 7399, 7400, 7405, 7616, 7617, 7618, 7619, 7620, 7621, 7622, 7623, 7624, 7625, 7626, 7627, 7628, 7629, 7630, 7631, 7632, 7633, 7634, 7635, 7636, 7637, 7638, 7639, 7640, 7641, 7642, 7643, 7644, 7645, 7646, 7647, 7648, 7649, 7650, 7651, 7652, 7653, 7654, 7676, 7677, 7678, 7679, 8400, 8401, 8402, 8403, 8404, 8405, 8406, 8407, 8408, 8409, 8410, 8411, 8412, 8417, 8421, 8422, 8423, 8424, 8425, 8426, 8427, 8428, 8429, 8430, 8431, 8432, 11503, 11504, 11505, 11647, 11744, 11745, 11746, 11747, 11748, 11749, 11750, 11751, 11752, 11753, 11754, 11755, 11756, 11757, 11758, 11759, 11760, 11761, 11762, 11763, 11764, 11765, 11766, 11767, 11768, 11769, 11770, 11771, 11772, 11773, 11774, 11775, 12330, 12331, 12332, 12333, 12334, 12335, 12441, 12442, 42607, 42620, 42621, 42736, 42737, 43010, 43014, 43019, 43045, 43046, 43204, 43232, 43233, 43234, 43235, 43236, 43237, 43238, 43239, 43240, 43241, 43242, 43243, 43244, 43245, 43246, 43247, 43248, 43249, 43302, 43303, 43304, 43305, 43306, 43307, 43308, 43309, 43335, 43336, 43337, 43338, 43339, 43340, 43341, 43342, 43343, 43344, 43345, 43392, 43393, 43394, 43443, 43446, 43447, 43448, 43449, 43452, 43561, 43562, 43563, 43564, 43565, 43566, 43569, 43570, 43573, 43574, 43587, 43596, 43696, 43698, 43699, 43700, 43703, 43704, 43710, 43711, 43713, 44005, 44008, 44013, 64286, 65024, 65025, 65026, 65027, 65028, 65029, 65030, 65031, 65032, 65033, 65034, 65035, 65036, 65037, 65038, 65039, 65056, 65057, 65058, 65059, 65060, 65061, 65062, 66045, 68097, 68098, 68099, 68101, 68102, 68108, 68109, 68110, 68111, 68152, 68153, 68154, 68159, 69633, 69688, 69689, 69690, 69691, 69692, 69693, 69694, 69695, 69696, 69697, 69698, 69699, 69700, 69701, 69702, 69760, 69761, 69811, 69812, 69813, 69814, 69817, 69818, 119143, 119144, 119145, 119163, 119164, 119165, 119166, 119167, 119168, 119169, 119170, 119173, 119174, 119175, 119176, 119177, 119178, 119179, 119210, 119211, 119212, 119213, 119362, 119363, 119364, 917760, 917761, 917762, 917763, 917764, 917765, 917766, 917767, 917768, 917769, 917770, 917771, 917772, 917773, 917774, 917775, 917776, 917777, 917778, 917779, 917780, 917781, 917782, 917783, 917784, 917785, 917786, 917787, 917788, 917789, 917790, 917791, 917792, 917793, 917794, 917795, 917796, 917797, 917798, 917799, 917800, 917801, 917802, 917803, 917804, 917805, 917806, 917807, 917808, 917809, 917810, 917811, 917812, 917813, 917814, 917815, 917816, 917817, 917818, 917819, 917820, 917821, 917822, 917823, 917824, 917825, 917826, 917827, 917828, 917829, 917830, 917831, 917832, 917833, 917834, 917835, 917836, 917837, 917838, 917839, 917840, 917841, 917842, 917843, 917844, 917845, 917846, 917847, 917848, 917849, 917850, 917851, 917852, 917853, 917854, 917855, 917856, 917857, 917858, 917859, 917860, 917861, 917862, 917863, 917864, 917865, 917866, 917867, 917868, 917869, 917870, 917871, 917872, 917873, 917874, 917875, 917876, 917877, 917878, 917879, 917880, 917881, 917882, 917883, 917884, 917885, 917886, 917887, 917888, 917889, 917890, 917891, 917892, 917893, 917894, 917895, 917896, 917897, 917898, 917899, 917900, 917901, 917902, 917903, 917904, 917905, 917906, 917907, 917908, 917909, 917910, 917911, 917912, 917913, 917914, 917915, 917916, 917917, 917918, 917919, 917920, 917921, 917922, 917923, 917924, 917925, 917926, 917927, 917928, 917929, 917930, 917931, 917932, 917933, 917934, 917935, 917936, 917937, 917938, 917939, 917940, 917941, 917942, 917943, 917944, 917945, 917946, 917947, 917948, 917949, 917950, 917951, 917952, 917953, 917954, 917955, 917956, 917957, 917958, 917959, 917960, 917961, 917962, 917963, 917964, 917965, 917966, 917967, 917968, 917969, 917970, 917971, 917972, 917973, 917974, 917975, 917976, 917977, 917978, 917979, 917980, 917981, 917982, 917983, 917984, 917985, 917986, 917987, 917988, 917989, 917990, 917991, 917992, 917993, 917994, 917995, 917996, 917997, 917998, 917999],
				'encoded': '\u0300\u0301\u0302\u0303\u0304\u0305\u0306\u0307\u0308\u0309\u030a\u030b\u030c\u030d\u030e\u030f\u0310\u0311\u0312\u0313\u0314\u0315\u0316\u0317\u0318\u0319\u031a\u031b\u031c\u031d\u031e\u031f\u0320\u0321\u0322\u0323\u0324\u0325\u0326\u0327\u0328\u0329\u032a\u032b\u032c\u032d\u032e\u032f\u0330\u0331\u0332\u0333\u0334\u0335\u0336\u0337\u0338\u0339\u033a\u033b\u033c\u033d\u033e\u033f\u0340\u0341\u0342\u0343\u0344\u0345\u0346\u0347\u0348\u0349\u034a\u034b\u034c\u034d\u034e\u034f\u0350\u0351\u0352\u0353\u0354\u0355\u0356\u0357\u0358\u0359\u035a\u035b\u035c\u035d\u035e\u035f\u0360\u0361\u0362\u0363\u0364\u0365\u0366\u0367\u0368\u0369\u036a\u036b\u036c\u036d\u036e\u036f\u0483\u0484\u0485\u0486\u0487\u0591\u0592\u0593\u0594\u0595\u0596\u0597\u0598\u0599\u059a\u059b\u059c\u059d\u059e\u059f\u05a0\u05a1\u05a2\u05a3\u05a4\u05a5\u05a6\u05a7\u05a8\u05a9\u05aa\u05ab\u05ac\u05ad\u05ae\u05af\u05b0\u05b1\u05b2\u05b3\u05b4\u05b5\u05b6\u05b7\u05b8\u05b9\u05ba\u05bb\u05bc\u05bd\u05bf\u05c1\u05c2\u05c4\u05c5\u05c7\u0610\u0611\u0612\u0613\u0614\u0615\u0616\u0617\u0618\u0619\u061a\u064b\u064c\u064d\u064e\u064f\u0650\u0651\u0652\u0653\u0654\u0655\u0656\u0657\u0658\u0659\u065a\u065b\u065c\u065d\u065e\u065f\u0670\u06d6\u06d7\u06d8\u06d9\u06da\u06db\u06dc\u06df\u06e0\u06e1\u06e2\u06e3\u06e4\u06e7\u06e8\u06ea\u06eb\u06ec\u06ed\u0711\u0730\u0731\u0732\u0733\u0734\u0735\u0736\u0737\u0738\u0739\u073a\u073b\u073c\u073d\u073e\u073f\u0740\u0741\u0742\u0743\u0744\u0745\u0746\u0747\u0748\u0749\u074a\u07a6\u07a7\u07a8\u07a9\u07aa\u07ab\u07ac\u07ad\u07ae\u07af\u07b0\u07eb\u07ec\u07ed\u07ee\u07ef\u07f0\u07f1\u07f2\u07f3\u0816\u0817\u0818\u0819\u081b\u081c\u081d\u081e\u081f\u0820\u0821\u0822\u0823\u0825\u0826\u0827\u0829\u082a\u082b\u082c\u082d\u0859\u085a\u085b\u0900\u0901\u0902\u093a\u093c\u0941\u0942\u0943\u0944\u0945\u0946\u0947\u0948\u094d\u0951\u0952\u0953\u0954\u0955\u0956\u0957\u0962\u0963\u0981\u09bc\u09c1\u09c2\u09c3\u09c4\u09cd\u09e2\u09e3\u0a01\u0a02\u0a3c\u0a41\u0a42\u0a47\u0a48\u0a4b\u0a4c\u0a4d\u0a51\u0a70\u0a71\u0a75\u0a81\u0a82\u0abc\u0ac1\u0ac2\u0ac3\u0ac4\u0ac5\u0ac7\u0ac8\u0acd\u0ae2\u0ae3\u0b01\u0b3c\u0b3f\u0b41\u0b42\u0b43\u0b44\u0b4d\u0b56\u0b62\u0b63\u0b82\u0bc0\u0bcd\u0c3e\u0c3f\u0c40\u0c46\u0c47\u0c48\u0c4a\u0c4b\u0c4c\u0c4d\u0c55\u0c56\u0c62\u0c63\u0cbc\u0cbf\u0cc6\u0ccc\u0ccd\u0ce2\u0ce3\u0d41\u0d42\u0d43\u0d44\u0d4d\u0d62\u0d63\u0dca\u0dd2\u0dd3\u0dd4\u0dd6\u0e31\u0e34\u0e35\u0e36\u0e37\u0e38\u0e39\u0e3a\u0e47\u0e48\u0e49\u0e4a\u0e4b\u0e4c\u0e4d\u0e4e\u0eb1\u0eb4\u0eb5\u0eb6\u0eb7\u0eb8\u0eb9\u0ebb\u0ebc\u0ec8\u0ec9\u0eca\u0ecb\u0ecc\u0ecd\u0f18\u0f19\u0f35\u0f37\u0f39\u0f71\u0f72\u0f73\u0f74\u0f75\u0f76\u0f77\u0f78\u0f79\u0f7a\u0f7b\u0f7c\u0f7d\u0f7e\u0f80\u0f81\u0f82\u0f83\u0f84\u0f86\u0f87\u0f8d\u0f8e\u0f8f\u0f90\u0f91\u0f92\u0f93\u0f94\u0f95\u0f96\u0f97\u0f99\u0f9a\u0f9b\u0f9c\u0f9d\u0f9e\u0f9f\u0fa0\u0fa1\u0fa2\u0fa3\u0fa4\u0fa5\u0fa6\u0fa7\u0fa8\u0fa9\u0faa\u0fab\u0fac\u0fad\u0fae\u0faf\u0fb0\u0fb1\u0fb2\u0fb3\u0fb4\u0fb5\u0fb6\u0fb7\u0fb8\u0fb9\u0fba\u0fbb\u0fbc\u0fc6\u102d\u102e\u102f\u1030\u1032\u1033\u1034\u1035\u1036\u1037\u1039\u103a\u103d\u103e\u1058\u1059\u105e\u105f\u1060\u1071\u1072\u1073\u1074\u1082\u1085\u1086\u108d\u109d\u135d\u135e\u135f\u1712\u1713\u1714\u1732\u1733\u1734\u1752\u1753\u1772\u1773\u17b7\u17b8\u17b9\u17ba\u17bb\u17bc\u17bd\u17c6\u17c9\u17ca\u17cb\u17cc\u17cd\u17ce\u17cf\u17d0\u17d1\u17d2\u17d3\u17dd\u180b\u180c\u180d\u18a9\u1920\u1921\u1922\u1927\u1928\u1932\u1939\u193a\u193b\u1a17\u1a18\u1a56\u1a58\u1a59\u1a5a\u1a5b\u1a5c\u1a5d\u1a5e\u1a60\u1a62\u1a65\u1a66\u1a67\u1a68\u1a69\u1a6a\u1a6b\u1a6c\u1a73\u1a74\u1a75\u1a76\u1a77\u1a78\u1a79\u1a7a\u1a7b\u1a7c\u1a7f\u1b00\u1b01\u1b02\u1b03\u1b34\u1b36\u1b37\u1b38\u1b39\u1b3a\u1b3c\u1b42\u1b6b\u1b6c\u1b6d\u1b6e\u1b6f\u1b70\u1b71\u1b72\u1b73\u1b80\u1b81\u1ba2\u1ba3\u1ba4\u1ba5\u1ba8\u1ba9\u1be6\u1be8\u1be9\u1bed\u1bef\u1bf0\u1bf1\u1c2c\u1c2d\u1c2e\u1c2f\u1c30\u1c31\u1c32\u1c33\u1c36\u1c37\u1cd0\u1cd1\u1cd2\u1cd4\u1cd5\u1cd6\u1cd7\u1cd8\u1cd9\u1cda\u1cdb\u1cdc\u1cdd\u1cde\u1cdf\u1ce0\u1ce2\u1ce3\u1ce4\u1ce5\u1ce6\u1ce7\u1ce8\u1ced\u1dc0\u1dc1\u1dc2\u1dc3\u1dc4\u1dc5\u1dc6\u1dc7\u1dc8\u1dc9\u1dca\u1dcb\u1dcc\u1dcd\u1dce\u1dcf\u1dd0\u1dd1\u1dd2\u1dd3\u1dd4\u1dd5\u1dd6\u1dd7\u1dd8\u1dd9\u1dda\u1ddb\u1ddc\u1ddd\u1dde\u1ddf\u1de0\u1de1\u1de2\u1de3\u1de4\u1de5\u1de6\u1dfc\u1dfd\u1dfe\u1dff\u20d0\u20d1\u20d2\u20d3\u20d4\u20d5\u20d6\u20d7\u20d8\u20d9\u20da\u20db\u20dc\u20e1\u20e5\u20e6\u20e7\u20e8\u20e9\u20ea\u20eb\u20ec\u20ed\u20ee\u20ef\u20f0\u2cef\u2cf0\u2cf1\u2d7f\u2de0\u2de1\u2de2\u2de3\u2de4\u2de5\u2de6\u2de7\u2de8\u2de9\u2dea\u2deb\u2dec\u2ded\u2dee\u2def\u2df0\u2df1\u2df2\u2df3\u2df4\u2df5\u2df6\u2df7\u2df8\u2df9\u2dfa\u2dfb\u2dfc\u2dfd\u2dfe\u2dff\u302a\u302b\u302c\u302d\u302e\u302f\u3099\u309a\ua66f\ua67c\ua67d\ua6f0\ua6f1\ua802\ua806\ua80b\ua825\ua826\ua8c4\ua8e0\ua8e1\ua8e2\ua8e3\ua8e4\ua8e5\ua8e6\ua8e7\ua8e8\ua8e9\ua8ea\ua8eb\ua8ec\ua8ed\ua8ee\ua8ef\ua8f0\ua8f1\ua926\ua927\ua928\ua929\ua92a\ua92b\ua92c\ua92d\ua947\ua948\ua949\ua94a\ua94b\ua94c\ua94d\ua94e\ua94f\ua950\ua951\ua980\ua981\ua982\ua9b3\ua9b6\ua9b7\ua9b8\ua9b9\ua9bc\uaa29\uaa2a\uaa2b\uaa2c\uaa2d\uaa2e\uaa31\uaa32\uaa35\uaa36\uaa43\uaa4c\uaab0\uaab2\uaab3\uaab4\uaab7\uaab8\uaabe\uaabf\uaac1\uabe5\uabe8\uabed\ufb1e\ufe00\ufe01\ufe02\ufe03\ufe04\ufe05\ufe06\ufe07\ufe08\ufe09\ufe0a\ufe0b\ufe0c\ufe0d\ufe0e\ufe0f\ufe20\ufe21\ufe22\ufe23\ufe24\ufe25\ufe26\ud800\uddfd\ud802\ude01\ud802\ude02\ud802\ude03\ud802\ude05\ud802\ude06\ud802\ude0c\ud802\ude0d\ud802\ude0e\ud802\ude0f\ud802\ude38\ud802\ude39\ud802\ude3a\ud802\ude3f\ud804\udc01\ud804\udc38\ud804\udc39\ud804\udc3a\ud804\udc3b\ud804\udc3c\ud804\udc3d\ud804\udc3e\ud804\udc3f\ud804\udc40\ud804\udc41\ud804\udc42\ud804\udc43\ud804\udc44\ud804\udc45\ud804\udc46\ud804\udc80\ud804\udc81\ud804\udcb3\ud804\udcb4\ud804\udcb5\ud804\udcb6\ud804\udcb9\ud804\udcba\ud834\udd67\ud834\udd68\ud834\udd69\ud834\udd7b\ud834\udd7c\ud834\udd7d\ud834\udd7e\ud834\udd7f\ud834\udd80\ud834\udd81\ud834\udd82\ud834\udd85\ud834\udd86\ud834\udd87\ud834\udd88\ud834\udd89\ud834\udd8a\ud834\udd8b\ud834\uddaa\ud834\uddab\ud834\uddac\ud834\uddad\ud834\ude42\ud834\ude43\ud834\ude44\udb40\udd00\udb40\udd01\udb40\udd02\udb40\udd03\udb40\udd04\udb40\udd05\udb40\udd06\udb40\udd07\udb40\udd08\udb40\udd09\udb40\udd0a\udb40\udd0b\udb40\udd0c\udb40\udd0d\udb40\udd0e\udb40\udd0f\udb40\udd10\udb40\udd11\udb40\udd12\udb40\udd13\udb40\udd14\udb40\udd15\udb40\udd16\udb40\udd17\udb40\udd18\udb40\udd19\udb40\udd1a\udb40\udd1b\udb40\udd1c\udb40\udd1d\udb40\udd1e\udb40\udd1f\udb40\udd20\udb40\udd21\udb40\udd22\udb40\udd23\udb40\udd24\udb40\udd25\udb40\udd26\udb40\udd27\udb40\udd28\udb40\udd29\udb40\udd2a\udb40\udd2b\udb40\udd2c\udb40\udd2d\udb40\udd2e\udb40\udd2f\udb40\udd30\udb40\udd31\udb40\udd32\udb40\udd33\udb40\udd34\udb40\udd35\udb40\udd36\udb40\udd37\udb40\udd38\udb40\udd39\udb40\udd3a\udb40\udd3b\udb40\udd3c\udb40\udd3d\udb40\udd3e\udb40\udd3f\udb40\udd40\udb40\udd41\udb40\udd42\udb40\udd43\udb40\udd44\udb40\udd45\udb40\udd46\udb40\udd47\udb40\udd48\udb40\udd49\udb40\udd4a\udb40\udd4b\udb40\udd4c\udb40\udd4d\udb40\udd4e\udb40\udd4f\udb40\udd50\udb40\udd51\udb40\udd52\udb40\udd53\udb40\udd54\udb40\udd55\udb40\udd56\udb40\udd57\udb40\udd58\udb40\udd59\udb40\udd5a\udb40\udd5b\udb40\udd5c\udb40\udd5d\udb40\udd5e\udb40\udd5f\udb40\udd60\udb40\udd61\udb40\udd62\udb40\udd63\udb40\udd64\udb40\udd65\udb40\udd66\udb40\udd67\udb40\udd68\udb40\udd69\udb40\udd6a\udb40\udd6b\udb40\udd6c\udb40\udd6d\udb40\udd6e\udb40\udd6f\udb40\udd70\udb40\udd71\udb40\udd72\udb40\udd73\udb40\udd74\udb40\udd75\udb40\udd76\udb40\udd77\udb40\udd78\udb40\udd79\udb40\udd7a\udb40\udd7b\udb40\udd7c\udb40\udd7d\udb40\udd7e\udb40\udd7f\udb40\udd80\udb40\udd81\udb40\udd82\udb40\udd83\udb40\udd84\udb40\udd85\udb40\udd86\udb40\udd87\udb40\udd88\udb40\udd89\udb40\udd8a\udb40\udd8b\udb40\udd8c\udb40\udd8d\udb40\udd8e\udb40\udd8f\udb40\udd90\udb40\udd91\udb40\udd92\udb40\udd93\udb40\udd94\udb40\udd95\udb40\udd96\udb40\udd97\udb40\udd98\udb40\udd99\udb40\udd9a\udb40\udd9b\udb40\udd9c\udb40\udd9d\udb40\udd9e\udb40\udd9f\udb40\udda0\udb40\udda1\udb40\udda2\udb40\udda3\udb40\udda4\udb40\udda5\udb40\udda6\udb40\udda7\udb40\udda8\udb40\udda9\udb40\uddaa\udb40\uddab\udb40\uddac\udb40\uddad\udb40\uddae\udb40\uddaf\udb40\uddb0\udb40\uddb1\udb40\uddb2\udb40\uddb3\udb40\uddb4\udb40\uddb5\udb40\uddb6\udb40\uddb7\udb40\uddb8\udb40\uddb9\udb40\uddba\udb40\uddbb\udb40\uddbc\udb40\uddbd\udb40\uddbe\udb40\uddbf\udb40\uddc0\udb40\uddc1\udb40\uddc2\udb40\uddc3\udb40\uddc4\udb40\uddc5\udb40\uddc6\udb40\uddc7\udb40\uddc8\udb40\uddc9\udb40\uddca\udb40\uddcb\udb40\uddcc\udb40\uddcd\udb40\uddce\udb40\uddcf\udb40\uddd0\udb40\uddd1\udb40\uddd2\udb40\uddd3\udb40\uddd4\udb40\uddd5\udb40\uddd6\udb40\uddd7\udb40\uddd8\udb40\uddd9\udb40\uddda\udb40\udddb\udb40\udddc\udb40\udddd\udb40\uddde\udb40\udddf\udb40\udde0\udb40\udde1\udb40\udde2\udb40\udde3\udb40\udde4\udb40\udde5\udb40\udde6\udb40\udde7\udb40\udde8\udb40\udde9\udb40\uddea\udb40\uddeb\udb40\uddec\udb40\udded\udb40\uddee\udb40\uddef'
			},
			{
				'description': 'Unicode character category [Nd]: number, decimal digit',
				'decoded': [48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 1632, 1633, 1634, 1635, 1636, 1637, 1638, 1639, 1640, 1641, 1776, 1777, 1778, 1779, 1780, 1781, 1782, 1783, 1784, 1785, 1984, 1985, 1986, 1987, 1988, 1989, 1990, 1991, 1992, 1993, 2406, 2407, 2408, 2409, 2410, 2411, 2412, 2413, 2414, 2415, 2534, 2535, 2536, 2537, 2538, 2539, 2540, 2541, 2542, 2543, 2662, 2663, 2664, 2665, 2666, 2667, 2668, 2669, 2670, 2671, 2790, 2791, 2792, 2793, 2794, 2795, 2796, 2797, 2798, 2799, 2918, 2919, 2920, 2921, 2922, 2923, 2924, 2925, 2926, 2927, 3046, 3047, 3048, 3049, 3050, 3051, 3052, 3053, 3054, 3055, 3174, 3175, 3176, 3177, 3178, 3179, 3180, 3181, 3182, 3183, 3302, 3303, 3304, 3305, 3306, 3307, 3308, 3309, 3310, 3311, 3430, 3431, 3432, 3433, 3434, 3435, 3436, 3437, 3438, 3439, 3664, 3665, 3666, 3667, 3668, 3669, 3670, 3671, 3672, 3673, 3792, 3793, 3794, 3795, 3796, 3797, 3798, 3799, 3800, 3801, 3872, 3873, 3874, 3875, 3876, 3877, 3878, 3879, 3880, 3881, 4160, 4161, 4162, 4163, 4164, 4165, 4166, 4167, 4168, 4169, 4240, 4241, 4242, 4243, 4244, 4245, 4246, 4247, 4248, 4249, 6112, 6113, 6114, 6115, 6116, 6117, 6118, 6119, 6120, 6121, 6160, 6161, 6162, 6163, 6164, 6165, 6166, 6167, 6168, 6169, 6470, 6471, 6472, 6473, 6474, 6475, 6476, 6477, 6478, 6479, 6608, 6609, 6610, 6611, 6612, 6613, 6614, 6615, 6616, 6617, 6784, 6785, 6786, 6787, 6788, 6789, 6790, 6791, 6792, 6793, 6800, 6801, 6802, 6803, 6804, 6805, 6806, 6807, 6808, 6809, 6992, 6993, 6994, 6995, 6996, 6997, 6998, 6999, 7000, 7001, 7088, 7089, 7090, 7091, 7092, 7093, 7094, 7095, 7096, 7097, 7232, 7233, 7234, 7235, 7236, 7237, 7238, 7239, 7240, 7241, 7248, 7249, 7250, 7251, 7252, 7253, 7254, 7255, 7256, 7257, 42528, 42529, 42530, 42531, 42532, 42533, 42534, 42535, 42536, 42537, 43216, 43217, 43218, 43219, 43220, 43221, 43222, 43223, 43224, 43225, 43264, 43265, 43266, 43267, 43268, 43269, 43270, 43271, 43272, 43273, 43472, 43473, 43474, 43475, 43476, 43477, 43478, 43479, 43480, 43481, 43600, 43601, 43602, 43603, 43604, 43605, 43606, 43607, 43608, 43609, 44016, 44017, 44018, 44019, 44020, 44021, 44022, 44023, 44024, 44025, 65296, 65297, 65298, 65299, 65300, 65301, 65302, 65303, 65304, 65305, 66720, 66721, 66722, 66723, 66724, 66725, 66726, 66727, 66728, 66729, 69734, 69735, 69736, 69737, 69738, 69739, 69740, 69741, 69742, 69743, 120782, 120783, 120784, 120785, 120786, 120787, 120788, 120789, 120790, 120791, 120792, 120793, 120794, 120795, 120796, 120797, 120798, 120799, 120800, 120801, 120802, 120803, 120804, 120805, 120806, 120807, 120808, 120809, 120810, 120811, 120812, 120813, 120814, 120815, 120816, 120817, 120818, 120819, 120820, 120821, 120822, 120823, 120824, 120825, 120826, 120827, 120828, 120829, 120830, 120831],
				'encoded': '0123456789\u0660\u0661\u0662\u0663\u0664\u0665\u0666\u0667\u0668\u0669\u06f0\u06f1\u06f2\u06f3\u06f4\u06f5\u06f6\u06f7\u06f8\u06f9\u07c0\u07c1\u07c2\u07c3\u07c4\u07c5\u07c6\u07c7\u07c8\u07c9\u0966\u0967\u0968\u0969\u096a\u096b\u096c\u096d\u096e\u096f\u09e6\u09e7\u09e8\u09e9\u09ea\u09eb\u09ec\u09ed\u09ee\u09ef\u0a66\u0a67\u0a68\u0a69\u0a6a\u0a6b\u0a6c\u0a6d\u0a6e\u0a6f\u0ae6\u0ae7\u0ae8\u0ae9\u0aea\u0aeb\u0aec\u0aed\u0aee\u0aef\u0b66\u0b67\u0b68\u0b69\u0b6a\u0b6b\u0b6c\u0b6d\u0b6e\u0b6f\u0be6\u0be7\u0be8\u0be9\u0bea\u0beb\u0bec\u0bed\u0bee\u0bef\u0c66\u0c67\u0c68\u0c69\u0c6a\u0c6b\u0c6c\u0c6d\u0c6e\u0c6f\u0ce6\u0ce7\u0ce8\u0ce9\u0cea\u0ceb\u0cec\u0ced\u0cee\u0cef\u0d66\u0d67\u0d68\u0d69\u0d6a\u0d6b\u0d6c\u0d6d\u0d6e\u0d6f\u0e50\u0e51\u0e52\u0e53\u0e54\u0e55\u0e56\u0e57\u0e58\u0e59\u0ed0\u0ed1\u0ed2\u0ed3\u0ed4\u0ed5\u0ed6\u0ed7\u0ed8\u0ed9\u0f20\u0f21\u0f22\u0f23\u0f24\u0f25\u0f26\u0f27\u0f28\u0f29\u1040\u1041\u1042\u1043\u1044\u1045\u1046\u1047\u1048\u1049\u1090\u1091\u1092\u1093\u1094\u1095\u1096\u1097\u1098\u1099\u17e0\u17e1\u17e2\u17e3\u17e4\u17e5\u17e6\u17e7\u17e8\u17e9\u1810\u1811\u1812\u1813\u1814\u1815\u1816\u1817\u1818\u1819\u1946\u1947\u1948\u1949\u194a\u194b\u194c\u194d\u194e\u194f\u19d0\u19d1\u19d2\u19d3\u19d4\u19d5\u19d6\u19d7\u19d8\u19d9\u1a80\u1a81\u1a82\u1a83\u1a84\u1a85\u1a86\u1a87\u1a88\u1a89\u1a90\u1a91\u1a92\u1a93\u1a94\u1a95\u1a96\u1a97\u1a98\u1a99\u1b50\u1b51\u1b52\u1b53\u1b54\u1b55\u1b56\u1b57\u1b58\u1b59\u1bb0\u1bb1\u1bb2\u1bb3\u1bb4\u1bb5\u1bb6\u1bb7\u1bb8\u1bb9\u1c40\u1c41\u1c42\u1c43\u1c44\u1c45\u1c46\u1c47\u1c48\u1c49\u1c50\u1c51\u1c52\u1c53\u1c54\u1c55\u1c56\u1c57\u1c58\u1c59\ua620\ua621\ua622\ua623\ua624\ua625\ua626\ua627\ua628\ua629\ua8d0\ua8d1\ua8d2\ua8d3\ua8d4\ua8d5\ua8d6\ua8d7\ua8d8\ua8d9\ua900\ua901\ua902\ua903\ua904\ua905\ua906\ua907\ua908\ua909\ua9d0\ua9d1\ua9d2\ua9d3\ua9d4\ua9d5\ua9d6\ua9d7\ua9d8\ua9d9\uaa50\uaa51\uaa52\uaa53\uaa54\uaa55\uaa56\uaa57\uaa58\uaa59\uabf0\uabf1\uabf2\uabf3\uabf4\uabf5\uabf6\uabf7\uabf8\uabf9\uff10\uff11\uff12\uff13\uff14\uff15\uff16\uff17\uff18\uff19\ud801\udca0\ud801\udca1\ud801\udca2\ud801\udca3\ud801\udca4\ud801\udca5\ud801\udca6\ud801\udca7\ud801\udca8\ud801\udca9\ud804\udc66\ud804\udc67\ud804\udc68\ud804\udc69\ud804\udc6a\ud804\udc6b\ud804\udc6c\ud804\udc6d\ud804\udc6e\ud804\udc6f\ud835\udfce\ud835\udfcf\ud835\udfd0\ud835\udfd1\ud835\udfd2\ud835\udfd3\ud835\udfd4\ud835\udfd5\ud835\udfd6\ud835\udfd7\ud835\udfd8\ud835\udfd9\ud835\udfda\ud835\udfdb\ud835\udfdc\ud835\udfdd\ud835\udfde\ud835\udfdf\ud835\udfe0\ud835\udfe1\ud835\udfe2\ud835\udfe3\ud835\udfe4\ud835\udfe5\ud835\udfe6\ud835\udfe7\ud835\udfe8\ud835\udfe9\ud835\udfea\ud835\udfeb\ud835\udfec\ud835\udfed\ud835\udfee\ud835\udfef\ud835\udff0\ud835\udff1\ud835\udff2\ud835\udff3\ud835\udff4\ud835\udff5\ud835\udff6\ud835\udff7\ud835\udff8\ud835\udff9\ud835\udffa\ud835\udffb\ud835\udffc\ud835\udffd\ud835\udffe\ud835\udfff'
			},
			{
				'description': 'Unicode character category [Nl]: number, letter',
				'decoded': [5870, 5871, 5872, 8544, 8545, 8546, 8547, 8548, 8549, 8550, 8551, 8552, 8553, 8554, 8555, 8556, 8557, 8558, 8559, 8560, 8561, 8562, 8563, 8564, 8565, 8566, 8567, 8568, 8569, 8570, 8571, 8572, 8573, 8574, 8575, 8576, 8577, 8578, 8581, 8582, 8583, 8584, 12295, 12321, 12322, 12323, 12324, 12325, 12326, 12327, 12328, 12329, 12344, 12345, 12346, 42726, 42727, 42728, 42729, 42730, 42731, 42732, 42733, 42734, 42735, 65856, 65857, 65858, 65859, 65860, 65861, 65862, 65863, 65864, 65865, 65866, 65867, 65868, 65869, 65870, 65871, 65872, 65873, 65874, 65875, 65876, 65877, 65878, 65879, 65880, 65881, 65882, 65883, 65884, 65885, 65886, 65887, 65888, 65889, 65890, 65891, 65892, 65893, 65894, 65895, 65896, 65897, 65898, 65899, 65900, 65901, 65902, 65903, 65904, 65905, 65906, 65907, 65908, 66369, 66378, 66513, 66514, 66515, 66516, 66517, 74752, 74753, 74754, 74755, 74756, 74757, 74758, 74759, 74760, 74761, 74762, 74763, 74764, 74765, 74766, 74767, 74768, 74769, 74770, 74771, 74772, 74773, 74774, 74775, 74776, 74777, 74778, 74779, 74780, 74781, 74782, 74783, 74784, 74785, 74786, 74787, 74788, 74789, 74790, 74791, 74792, 74793, 74794, 74795, 74796, 74797, 74798, 74799, 74800, 74801, 74802, 74803, 74804, 74805, 74806, 74807, 74808, 74809, 74810, 74811, 74812, 74813, 74814, 74815, 74816, 74817, 74818, 74819, 74820, 74821, 74822, 74823, 74824, 74825, 74826, 74827, 74828, 74829, 74830, 74831, 74832, 74833, 74834, 74835, 74836, 74837, 74838, 74839, 74840, 74841, 74842, 74843, 74844, 74845, 74846, 74847, 74848, 74849, 74850],
				'encoded': '\u16ee\u16ef\u16f0\u2160\u2161\u2162\u2163\u2164\u2165\u2166\u2167\u2168\u2169\u216a\u216b\u216c\u216d\u216e\u216f\u2170\u2171\u2172\u2173\u2174\u2175\u2176\u2177\u2178\u2179\u217a\u217b\u217c\u217d\u217e\u217f\u2180\u2181\u2182\u2185\u2186\u2187\u2188\u3007\u3021\u3022\u3023\u3024\u3025\u3026\u3027\u3028\u3029\u3038\u3039\u303a\ua6e6\ua6e7\ua6e8\ua6e9\ua6ea\ua6eb\ua6ec\ua6ed\ua6ee\ua6ef\ud800\udd40\ud800\udd41\ud800\udd42\ud800\udd43\ud800\udd44\ud800\udd45\ud800\udd46\ud800\udd47\ud800\udd48\ud800\udd49\ud800\udd4a\ud800\udd4b\ud800\udd4c\ud800\udd4d\ud800\udd4e\ud800\udd4f\ud800\udd50\ud800\udd51\ud800\udd52\ud800\udd53\ud800\udd54\ud800\udd55\ud800\udd56\ud800\udd57\ud800\udd58\ud800\udd59\ud800\udd5a\ud800\udd5b\ud800\udd5c\ud800\udd5d\ud800\udd5e\ud800\udd5f\ud800\udd60\ud800\udd61\ud800\udd62\ud800\udd63\ud800\udd64\ud800\udd65\ud800\udd66\ud800\udd67\ud800\udd68\ud800\udd69\ud800\udd6a\ud800\udd6b\ud800\udd6c\ud800\udd6d\ud800\udd6e\ud800\udd6f\ud800\udd70\ud800\udd71\ud800\udd72\ud800\udd73\ud800\udd74\ud800\udf41\ud800\udf4a\ud800\udfd1\ud800\udfd2\ud800\udfd3\ud800\udfd4\ud800\udfd5\ud809\udc00\ud809\udc01\ud809\udc02\ud809\udc03\ud809\udc04\ud809\udc05\ud809\udc06\ud809\udc07\ud809\udc08\ud809\udc09\ud809\udc0a\ud809\udc0b\ud809\udc0c\ud809\udc0d\ud809\udc0e\ud809\udc0f\ud809\udc10\ud809\udc11\ud809\udc12\ud809\udc13\ud809\udc14\ud809\udc15\ud809\udc16\ud809\udc17\ud809\udc18\ud809\udc19\ud809\udc1a\ud809\udc1b\ud809\udc1c\ud809\udc1d\ud809\udc1e\ud809\udc1f\ud809\udc20\ud809\udc21\ud809\udc22\ud809\udc23\ud809\udc24\ud809\udc25\ud809\udc26\ud809\udc27\ud809\udc28\ud809\udc29\ud809\udc2a\ud809\udc2b\ud809\udc2c\ud809\udc2d\ud809\udc2e\ud809\udc2f\ud809\udc30\ud809\udc31\ud809\udc32\ud809\udc33\ud809\udc34\ud809\udc35\ud809\udc36\ud809\udc37\ud809\udc38\ud809\udc39\ud809\udc3a\ud809\udc3b\ud809\udc3c\ud809\udc3d\ud809\udc3e\ud809\udc3f\ud809\udc40\ud809\udc41\ud809\udc42\ud809\udc43\ud809\udc44\ud809\udc45\ud809\udc46\ud809\udc47\ud809\udc48\ud809\udc49\ud809\udc4a\ud809\udc4b\ud809\udc4c\ud809\udc4d\ud809\udc4e\ud809\udc4f\ud809\udc50\ud809\udc51\ud809\udc52\ud809\udc53\ud809\udc54\ud809\udc55\ud809\udc56\ud809\udc57\ud809\udc58\ud809\udc59\ud809\udc5a\ud809\udc5b\ud809\udc5c\ud809\udc5d\ud809\udc5e\ud809\udc5f\ud809\udc60\ud809\udc61\ud809\udc62'
			},
			{
				'description': 'Unicode character category [No]: number, other',
				'decoded': [178, 179, 185, 188, 189, 190, 2548, 2549, 2550, 2551, 2552, 2553, 2930, 2931, 2932, 2933, 2934, 2935, 3056, 3057, 3058, 3192, 3193, 3194, 3195, 3196, 3197, 3198, 3440, 3441, 3442, 3443, 3444, 3445, 3882, 3883, 3884, 3885, 3886, 3887, 3888, 3889, 3890, 3891, 4969, 4970, 4971, 4972, 4973, 4974, 4975, 4976, 4977, 4978, 4979, 4980, 4981, 4982, 4983, 4984, 4985, 4986, 4987, 4988, 6128, 6129, 6130, 6131, 6132, 6133, 6134, 6135, 6136, 6137, 6618, 8304, 8308, 8309, 8310, 8311, 8312, 8313, 8320, 8321, 8322, 8323, 8324, 8325, 8326, 8327, 8328, 8329, 8528, 8529, 8530, 8531, 8532, 8533, 8534, 8535, 8536, 8537, 8538, 8539, 8540, 8541, 8542, 8543, 8585, 9312, 9313, 9314, 9315, 9316, 9317, 9318, 9319, 9320, 9321, 9322, 9323, 9324, 9325, 9326, 9327, 9328, 9329, 9330, 9331, 9332, 9333, 9334, 9335, 9336, 9337, 9338, 9339, 9340, 9341, 9342, 9343, 9344, 9345, 9346, 9347, 9348, 9349, 9350, 9351, 9352, 9353, 9354, 9355, 9356, 9357, 9358, 9359, 9360, 9361, 9362, 9363, 9364, 9365, 9366, 9367, 9368, 9369, 9370, 9371, 9450, 9451, 9452, 9453, 9454, 9455, 9456, 9457, 9458, 9459, 9460, 9461, 9462, 9463, 9464, 9465, 9466, 9467, 9468, 9469, 9470, 9471, 10102, 10103, 10104, 10105, 10106, 10107, 10108, 10109, 10110, 10111, 10112, 10113, 10114, 10115, 10116, 10117, 10118, 10119, 10120, 10121, 10122, 10123, 10124, 10125, 10126, 10127, 10128, 10129, 10130, 10131, 11517, 12690, 12691, 12692, 12693, 12832, 12833, 12834, 12835, 12836, 12837, 12838, 12839, 12840, 12841, 12881, 12882, 12883, 12884, 12885, 12886, 12887, 12888, 12889, 12890, 12891, 12892, 12893, 12894, 12895, 12928, 12929, 12930, 12931, 12932, 12933, 12934, 12935, 12936, 12937, 12977, 12978, 12979, 12980, 12981, 12982, 12983, 12984, 12985, 12986, 12987, 12988, 12989, 12990, 12991, 43056, 43057, 43058, 43059, 43060, 43061, 65799, 65800, 65801, 65802, 65803, 65804, 65805, 65806, 65807, 65808, 65809, 65810, 65811, 65812, 65813, 65814, 65815, 65816, 65817, 65818, 65819, 65820, 65821, 65822, 65823, 65824, 65825, 65826, 65827, 65828, 65829, 65830, 65831, 65832, 65833, 65834, 65835, 65836, 65837, 65838, 65839, 65840, 65841, 65842, 65843, 65909, 65910, 65911, 65912, 65930, 66336, 66337, 66338, 66339, 67672, 67673, 67674, 67675, 67676, 67677, 67678, 67679, 67862, 67863, 67864, 67865, 67866, 67867, 68160, 68161, 68162, 68163, 68164, 68165, 68166, 68167, 68221, 68222, 68440, 68441, 68442, 68443, 68444, 68445, 68446, 68447, 68472, 68473, 68474, 68475, 68476, 68477, 68478, 68479, 69216, 69217, 69218, 69219, 69220, 69221, 69222, 69223, 69224, 69225, 69226, 69227, 69228, 69229, 69230, 69231, 69232, 69233, 69234, 69235, 69236, 69237, 69238, 69239, 69240, 69241, 69242, 69243, 69244, 69245, 69246, 69714, 69715, 69716, 69717, 69718, 69719, 69720, 69721, 69722, 69723, 69724, 69725, 69726, 69727, 69728, 69729, 69730, 69731, 69732, 69733, 119648, 119649, 119650, 119651, 119652, 119653, 119654, 119655, 119656, 119657, 119658, 119659, 119660, 119661, 119662, 119663, 119664, 119665, 127232, 127233, 127234, 127235, 127236, 127237, 127238, 127239, 127240, 127241, 127242],
				'encoded': '\xb2\xb3\xb9\xbc\xbd\xbe\u09f4\u09f5\u09f6\u09f7\u09f8\u09f9\u0b72\u0b73\u0b74\u0b75\u0b76\u0b77\u0bf0\u0bf1\u0bf2\u0c78\u0c79\u0c7a\u0c7b\u0c7c\u0c7d\u0c7e\u0d70\u0d71\u0d72\u0d73\u0d74\u0d75\u0f2a\u0f2b\u0f2c\u0f2d\u0f2e\u0f2f\u0f30\u0f31\u0f32\u0f33\u1369\u136a\u136b\u136c\u136d\u136e\u136f\u1370\u1371\u1372\u1373\u1374\u1375\u1376\u1377\u1378\u1379\u137a\u137b\u137c\u17f0\u17f1\u17f2\u17f3\u17f4\u17f5\u17f6\u17f7\u17f8\u17f9\u19da\u2070\u2074\u2075\u2076\u2077\u2078\u2079\u2080\u2081\u2082\u2083\u2084\u2085\u2086\u2087\u2088\u2089\u2150\u2151\u2152\u2153\u2154\u2155\u2156\u2157\u2158\u2159\u215a\u215b\u215c\u215d\u215e\u215f\u2189\u2460\u2461\u2462\u2463\u2464\u2465\u2466\u2467\u2468\u2469\u246a\u246b\u246c\u246d\u246e\u246f\u2470\u2471\u2472\u2473\u2474\u2475\u2476\u2477\u2478\u2479\u247a\u247b\u247c\u247d\u247e\u247f\u2480\u2481\u2482\u2483\u2484\u2485\u2486\u2487\u2488\u2489\u248a\u248b\u248c\u248d\u248e\u248f\u2490\u2491\u2492\u2493\u2494\u2495\u2496\u2497\u2498\u2499\u249a\u249b\u24ea\u24eb\u24ec\u24ed\u24ee\u24ef\u24f0\u24f1\u24f2\u24f3\u24f4\u24f5\u24f6\u24f7\u24f8\u24f9\u24fa\u24fb\u24fc\u24fd\u24fe\u24ff\u2776\u2777\u2778\u2779\u277a\u277b\u277c\u277d\u277e\u277f\u2780\u2781\u2782\u2783\u2784\u2785\u2786\u2787\u2788\u2789\u278a\u278b\u278c\u278d\u278e\u278f\u2790\u2791\u2792\u2793\u2cfd\u3192\u3193\u3194\u3195\u3220\u3221\u3222\u3223\u3224\u3225\u3226\u3227\u3228\u3229\u3251\u3252\u3253\u3254\u3255\u3256\u3257\u3258\u3259\u325a\u325b\u325c\u325d\u325e\u325f\u3280\u3281\u3282\u3283\u3284\u3285\u3286\u3287\u3288\u3289\u32b1\u32b2\u32b3\u32b4\u32b5\u32b6\u32b7\u32b8\u32b9\u32ba\u32bb\u32bc\u32bd\u32be\u32bf\ua830\ua831\ua832\ua833\ua834\ua835\ud800\udd07\ud800\udd08\ud800\udd09\ud800\udd0a\ud800\udd0b\ud800\udd0c\ud800\udd0d\ud800\udd0e\ud800\udd0f\ud800\udd10\ud800\udd11\ud800\udd12\ud800\udd13\ud800\udd14\ud800\udd15\ud800\udd16\ud800\udd17\ud800\udd18\ud800\udd19\ud800\udd1a\ud800\udd1b\ud800\udd1c\ud800\udd1d\ud800\udd1e\ud800\udd1f\ud800\udd20\ud800\udd21\ud800\udd22\ud800\udd23\ud800\udd24\ud800\udd25\ud800\udd26\ud800\udd27\ud800\udd28\ud800\udd29\ud800\udd2a\ud800\udd2b\ud800\udd2c\ud800\udd2d\ud800\udd2e\ud800\udd2f\ud800\udd30\ud800\udd31\ud800\udd32\ud800\udd33\ud800\udd75\ud800\udd76\ud800\udd77\ud800\udd78\ud800\udd8a\ud800\udf20\ud800\udf21\ud800\udf22\ud800\udf23\ud802\udc58\ud802\udc59\ud802\udc5a\ud802\udc5b\ud802\udc5c\ud802\udc5d\ud802\udc5e\ud802\udc5f\ud802\udd16\ud802\udd17\ud802\udd18\ud802\udd19\ud802\udd1a\ud802\udd1b\ud802\ude40\ud802\ude41\ud802\ude42\ud802\ude43\ud802\ude44\ud802\ude45\ud802\ude46\ud802\ude47\ud802\ude7d\ud802\ude7e\ud802\udf58\ud802\udf59\ud802\udf5a\ud802\udf5b\ud802\udf5c\ud802\udf5d\ud802\udf5e\ud802\udf5f\ud802\udf78\ud802\udf79\ud802\udf7a\ud802\udf7b\ud802\udf7c\ud802\udf7d\ud802\udf7e\ud802\udf7f\ud803\ude60\ud803\ude61\ud803\ude62\ud803\ude63\ud803\ude64\ud803\ude65\ud803\ude66\ud803\ude67\ud803\ude68\ud803\ude69\ud803\ude6a\ud803\ude6b\ud803\ude6c\ud803\ude6d\ud803\ude6e\ud803\ude6f\ud803\ude70\ud803\ude71\ud803\ude72\ud803\ude73\ud803\ude74\ud803\ude75\ud803\ude76\ud803\ude77\ud803\ude78\ud803\ude79\ud803\ude7a\ud803\ude7b\ud803\ude7c\ud803\ude7d\ud803\ude7e\ud804\udc52\ud804\udc53\ud804\udc54\ud804\udc55\ud804\udc56\ud804\udc57\ud804\udc58\ud804\udc59\ud804\udc5a\ud804\udc5b\ud804\udc5c\ud804\udc5d\ud804\udc5e\ud804\udc5f\ud804\udc60\ud804\udc61\ud804\udc62\ud804\udc63\ud804\udc64\ud804\udc65\ud834\udf60\ud834\udf61\ud834\udf62\ud834\udf63\ud834\udf64\ud834\udf65\ud834\udf66\ud834\udf67\ud834\udf68\ud834\udf69\ud834\udf6a\ud834\udf6b\ud834\udf6c\ud834\udf6d\ud834\udf6e\ud834\udf6f\ud834\udf70\ud834\udf71\ud83c\udd00\ud83c\udd01\ud83c\udd02\ud83c\udd03\ud83c\udd04\ud83c\udd05\ud83c\udd06\ud83c\udd07\ud83c\udd08\ud83c\udd09\ud83c\udd0a'
			},
			{
				'description': 'Unicode character category [Pc]: punctuation, connector',
				'decoded': [95, 8255, 8256, 8276, 65075, 65076, 65101, 65102, 65103, 65343],
				'encoded': '_\u203f\u2040\u2054\ufe33\ufe34\ufe4d\ufe4e\ufe4f\uff3f'
			},
			{
				'description': 'Unicode character category [Pd]: punctuation, dash',
				'decoded': [45, 1418, 1470, 5120, 6150, 8208, 8209, 8210, 8211, 8212, 8213, 11799, 11802, 12316, 12336, 12448, 65073, 65074, 65112, 65123, 65293],
				'encoded': '-\u058a\u05be\u1400\u1806\u2010\u2011\u2012\u2013\u2014\u2015\u2e17\u2e1a\u301c\u3030\u30a0\ufe31\ufe32\ufe58\ufe63\uff0d'
			},
			{
				'description': 'Unicode character category [Pe]: punctuation, close',
				'decoded': [41, 93, 125, 3899, 3901, 5788, 8262, 8318, 8334, 9002, 10089, 10091, 10093, 10095, 10097, 10099, 10101, 10182, 10215, 10217, 10219, 10221, 10223, 10628, 10630, 10632, 10634, 10636, 10638, 10640, 10642, 10644, 10646, 10648, 10713, 10715, 10749, 11811, 11813, 11815, 11817, 12297, 12299, 12301, 12303, 12305, 12309, 12311, 12313, 12315, 12318, 12319, 64831, 65048, 65078, 65080, 65082, 65084, 65086, 65088, 65090, 65092, 65096, 65114, 65116, 65118, 65289, 65341, 65373, 65376, 65379],
				'encoded': ')]}\u0f3b\u0f3d\u169c\u2046\u207e\u208e\u232a\u2769\u276b\u276d\u276f\u2771\u2773\u2775\u27c6\u27e7\u27e9\u27eb\u27ed\u27ef\u2984\u2986\u2988\u298a\u298c\u298e\u2990\u2992\u2994\u2996\u2998\u29d9\u29db\u29fd\u2e23\u2e25\u2e27\u2e29\u3009\u300b\u300d\u300f\u3011\u3015\u3017\u3019\u301b\u301e\u301f\ufd3f\ufe18\ufe36\ufe38\ufe3a\ufe3c\ufe3e\ufe40\ufe42\ufe44\ufe48\ufe5a\ufe5c\ufe5e\uff09\uff3d\uff5d\uff60\uff63'
			},
			{
				'description': 'Unicode character category [Pf]: punctuation, final quote',
				'decoded': [187, 8217, 8221, 8250, 11779, 11781, 11786, 11789, 11805, 11809],
				'encoded': '\xbb\u2019\u201d\u203a\u2e03\u2e05\u2e0a\u2e0d\u2e1d\u2e21'
			},
			{
				'description': 'Unicode character category [Pi]: punctuation, initial quote',
				'decoded': [171, 8216, 8219, 8220, 8223, 8249, 11778, 11780, 11785, 11788, 11804, 11808],
				'encoded': '\xab\u2018\u201b\u201c\u201f\u2039\u2e02\u2e04\u2e09\u2e0c\u2e1c\u2e20'
			},
			{
				'description': 'Unicode character category [Po]: punctuation, other',
				'decoded': [33, 34, 35, 37, 38, 39, 42, 44, 46, 47, 58, 59, 63, 64, 161, 183, 191, 894, 903, 1370, 1371, 1372, 1373, 1374, 1375, 1417, 1472, 1475, 1478, 1523, 1524, 1545, 1546, 1548, 1549, 1563, 1566, 1567, 1642, 1643, 1644, 1645, 1748, 1792, 1793, 1794, 1795, 1796, 1797, 1798, 1799, 1800, 1801, 1802, 1803, 1804, 1805, 2039, 2040, 2041, 2096, 2097, 2098, 2099, 2100, 2101, 2102, 2103, 2104, 2105, 2106, 2107, 2108, 2109, 2110, 2142, 2404, 2405, 2416, 3572, 3663, 3674, 3675, 3844, 3845, 3846, 3847, 3848, 3849, 3850, 3851, 3852, 3853, 3854, 3855, 3856, 3857, 3858, 3973, 4048, 4049, 4050, 4051, 4052, 4057, 4058, 4170, 4171, 4172, 4173, 4174, 4175, 4347, 4961, 4962, 4963, 4964, 4965, 4966, 4967, 4968, 5741, 5742, 5867, 5868, 5869, 5941, 5942, 6100, 6101, 6102, 6104, 6105, 6106, 6144, 6145, 6146, 6147, 6148, 6149, 6151, 6152, 6153, 6154, 6468, 6469, 6686, 6687, 6816, 6817, 6818, 6819, 6820, 6821, 6822, 6824, 6825, 6826, 6827, 6828, 6829, 7002, 7003, 7004, 7005, 7006, 7007, 7008, 7164, 7165, 7166, 7167, 7227, 7228, 7229, 7230, 7231, 7294, 7295, 7379, 8214, 8215, 8224, 8225, 8226, 8227, 8228, 8229, 8230, 8231, 8240, 8241, 8242, 8243, 8244, 8245, 8246, 8247, 8248, 8251, 8252, 8253, 8254, 8257, 8258, 8259, 8263, 8264, 8265, 8266, 8267, 8268, 8269, 8270, 8271, 8272, 8273, 8275, 8277, 8278, 8279, 8280, 8281, 8282, 8283, 8284, 8285, 8286, 11513, 11514, 11515, 11516, 11518, 11519, 11632, 11776, 11777, 11782, 11783, 11784, 11787, 11790, 11791, 11792, 11793, 11794, 11795, 11796, 11797, 11798, 11800, 11801, 11803, 11806, 11807, 11818, 11819, 11820, 11821, 11822, 11824, 11825, 12289, 12290, 12291, 12349, 12539, 42238, 42239, 42509, 42510, 42511, 42611, 42622, 42738, 42739, 42740, 42741, 42742, 42743, 43124, 43125, 43126, 43127, 43214, 43215, 43256, 43257, 43258, 43310, 43311, 43359, 43457, 43458, 43459, 43460, 43461, 43462, 43463, 43464, 43465, 43466, 43467, 43468, 43469, 43486, 43487, 43612, 43613, 43614, 43615, 43742, 43743, 44011, 65040, 65041, 65042, 65043, 65044, 65045, 65046, 65049, 65072, 65093, 65094, 65097, 65098, 65099, 65100, 65104, 65105, 65106, 65108, 65109, 65110, 65111, 65119, 65120, 65121, 65128, 65130, 65131, 65281, 65282, 65283, 65285, 65286, 65287, 65290, 65292, 65294, 65295, 65306, 65307, 65311, 65312, 65340, 65377, 65380, 65381, 65792, 65793, 66463, 66512, 67671, 67871, 67903, 68176, 68177, 68178, 68179, 68180, 68181, 68182, 68183, 68184, 68223, 68409, 68410, 68411, 68412, 68413, 68414, 68415, 69703, 69704, 69705, 69706, 69707, 69708, 69709, 69819, 69820, 69822, 69823, 69824, 69825, 74864, 74865, 74866, 74867],
				'encoded': '!"#%&\'*,./:;?@\xa1\xb7\xbf\u037e\u0387\u055a\u055b\u055c\u055d\u055e\u055f\u0589\u05c0\u05c3\u05c6\u05f3\u05f4\u0609\u060a\u060c\u060d\u061b\u061e\u061f\u066a\u066b\u066c\u066d\u06d4\u0700\u0701\u0702\u0703\u0704\u0705\u0706\u0707\u0708\u0709\u070a\u070b\u070c\u070d\u07f7\u07f8\u07f9\u0830\u0831\u0832\u0833\u0834\u0835\u0836\u0837\u0838\u0839\u083a\u083b\u083c\u083d\u083e\u085e\u0964\u0965\u0970\u0df4\u0e4f\u0e5a\u0e5b\u0f04\u0f05\u0f06\u0f07\u0f08\u0f09\u0f0a\u0f0b\u0f0c\u0f0d\u0f0e\u0f0f\u0f10\u0f11\u0f12\u0f85\u0fd0\u0fd1\u0fd2\u0fd3\u0fd4\u0fd9\u0fda\u104a\u104b\u104c\u104d\u104e\u104f\u10fb\u1361\u1362\u1363\u1364\u1365\u1366\u1367\u1368\u166d\u166e\u16eb\u16ec\u16ed\u1735\u1736\u17d4\u17d5\u17d6\u17d8\u17d9\u17da\u1800\u1801\u1802\u1803\u1804\u1805\u1807\u1808\u1809\u180a\u1944\u1945\u1a1e\u1a1f\u1aa0\u1aa1\u1aa2\u1aa3\u1aa4\u1aa5\u1aa6\u1aa8\u1aa9\u1aaa\u1aab\u1aac\u1aad\u1b5a\u1b5b\u1b5c\u1b5d\u1b5e\u1b5f\u1b60\u1bfc\u1bfd\u1bfe\u1bff\u1c3b\u1c3c\u1c3d\u1c3e\u1c3f\u1c7e\u1c7f\u1cd3\u2016\u2017\u2020\u2021\u2022\u2023\u2024\u2025\u2026\u2027\u2030\u2031\u2032\u2033\u2034\u2035\u2036\u2037\u2038\u203b\u203c\u203d\u203e\u2041\u2042\u2043\u2047\u2048\u2049\u204a\u204b\u204c\u204d\u204e\u204f\u2050\u2051\u2053\u2055\u2056\u2057\u2058\u2059\u205a\u205b\u205c\u205d\u205e\u2cf9\u2cfa\u2cfb\u2cfc\u2cfe\u2cff\u2d70\u2e00\u2e01\u2e06\u2e07\u2e08\u2e0b\u2e0e\u2e0f\u2e10\u2e11\u2e12\u2e13\u2e14\u2e15\u2e16\u2e18\u2e19\u2e1b\u2e1e\u2e1f\u2e2a\u2e2b\u2e2c\u2e2d\u2e2e\u2e30\u2e31\u3001\u3002\u3003\u303d\u30fb\ua4fe\ua4ff\ua60d\ua60e\ua60f\ua673\ua67e\ua6f2\ua6f3\ua6f4\ua6f5\ua6f6\ua6f7\ua874\ua875\ua876\ua877\ua8ce\ua8cf\ua8f8\ua8f9\ua8fa\ua92e\ua92f\ua95f\ua9c1\ua9c2\ua9c3\ua9c4\ua9c5\ua9c6\ua9c7\ua9c8\ua9c9\ua9ca\ua9cb\ua9cc\ua9cd\ua9de\ua9df\uaa5c\uaa5d\uaa5e\uaa5f\uaade\uaadf\uabeb\ufe10\ufe11\ufe12\ufe13\ufe14\ufe15\ufe16\ufe19\ufe30\ufe45\ufe46\ufe49\ufe4a\ufe4b\ufe4c\ufe50\ufe51\ufe52\ufe54\ufe55\ufe56\ufe57\ufe5f\ufe60\ufe61\ufe68\ufe6a\ufe6b\uff01\uff02\uff03\uff05\uff06\uff07\uff0a\uff0c\uff0e\uff0f\uff1a\uff1b\uff1f\uff20\uff3c\uff61\uff64\uff65\ud800\udd00\ud800\udd01\ud800\udf9f\ud800\udfd0\ud802\udc57\ud802\udd1f\ud802\udd3f\ud802\ude50\ud802\ude51\ud802\ude52\ud802\ude53\ud802\ude54\ud802\ude55\ud802\ude56\ud802\ude57\ud802\ude58\ud802\ude7f\ud802\udf39\ud802\udf3a\ud802\udf3b\ud802\udf3c\ud802\udf3d\ud802\udf3e\ud802\udf3f\ud804\udc47\ud804\udc48\ud804\udc49\ud804\udc4a\ud804\udc4b\ud804\udc4c\ud804\udc4d\ud804\udcbb\ud804\udcbc\ud804\udcbe\ud804\udcbf\ud804\udcc0\ud804\udcc1\ud809\udc70\ud809\udc71\ud809\udc72\ud809\udc73'
			},
			{
				'description': 'Unicode character category [Ps]: punctuation, open',
				'decoded': [40, 91, 123, 3898, 3900, 5787, 8218, 8222, 8261, 8317, 8333, 9001, 10088, 10090, 10092, 10094, 10096, 10098, 10100, 10181, 10214, 10216, 10218, 10220, 10222, 10627, 10629, 10631, 10633, 10635, 10637, 10639, 10641, 10643, 10645, 10647, 10712, 10714, 10748, 11810, 11812, 11814, 11816, 12296, 12298, 12300, 12302, 12304, 12308, 12310, 12312, 12314, 12317, 64830, 65047, 65077, 65079, 65081, 65083, 65085, 65087, 65089, 65091, 65095, 65113, 65115, 65117, 65288, 65339, 65371, 65375, 65378],
				'encoded': '([{\u0f3a\u0f3c\u169b\u201a\u201e\u2045\u207d\u208d\u2329\u2768\u276a\u276c\u276e\u2770\u2772\u2774\u27c5\u27e6\u27e8\u27ea\u27ec\u27ee\u2983\u2985\u2987\u2989\u298b\u298d\u298f\u2991\u2993\u2995\u2997\u29d8\u29da\u29fc\u2e22\u2e24\u2e26\u2e28\u3008\u300a\u300c\u300e\u3010\u3014\u3016\u3018\u301a\u301d\ufd3e\ufe17\ufe35\ufe37\ufe39\ufe3b\ufe3d\ufe3f\ufe41\ufe43\ufe47\ufe59\ufe5b\ufe5d\uff08\uff3b\uff5b\uff5f\uff62'
			},
			{
				'description': 'Unicode character category [Sc]: symbol, currency',
				'decoded': [36, 162, 163, 164, 165, 1547, 2546, 2547, 2555, 2801, 3065, 3647, 6107, 8352, 8353, 8354, 8355, 8356, 8357, 8358, 8359, 8360, 8361, 8362, 8363, 8364, 8365, 8366, 8367, 8368, 8369, 8370, 8371, 8372, 8373, 8374, 8375, 8376, 8377, 43064, 65020, 65129, 65284, 65504, 65505, 65509, 65510],
				'encoded': '$\xa2\xa3\xa4\xa5\u060b\u09f2\u09f3\u09fb\u0af1\u0bf9\u0e3f\u17db\u20a0\u20a1\u20a2\u20a3\u20a4\u20a5\u20a6\u20a7\u20a8\u20a9\u20aa\u20ab\u20ac\u20ad\u20ae\u20af\u20b0\u20b1\u20b2\u20b3\u20b4\u20b5\u20b6\u20b7\u20b8\u20b9\ua838\ufdfc\ufe69\uff04\uffe0\uffe1\uffe5\uffe6'
			},
			{
				'description': 'Unicode character category [Sk]: symbol, modifier',
				'decoded': [94, 96, 168, 175, 180, 184, 706, 707, 708, 709, 722, 723, 724, 725, 726, 727, 728, 729, 730, 731, 732, 733, 734, 735, 741, 742, 743, 744, 745, 746, 747, 749, 751, 752, 753, 754, 755, 756, 757, 758, 759, 760, 761, 762, 763, 764, 765, 766, 767, 885, 900, 901, 8125, 8127, 8128, 8129, 8141, 8142, 8143, 8157, 8158, 8159, 8173, 8174, 8175, 8189, 8190, 12443, 12444, 42752, 42753, 42754, 42755, 42756, 42757, 42758, 42759, 42760, 42761, 42762, 42763, 42764, 42765, 42766, 42767, 42768, 42769, 42770, 42771, 42772, 42773, 42774, 42784, 42785, 42889, 42890, 64434, 64435, 64436, 64437, 64438, 64439, 64440, 64441, 64442, 64443, 64444, 64445, 64446, 64447, 64448, 64449, 65342, 65344, 65507],
				'encoded': '^`\xa8\xaf\xb4\xb8\u02c2\u02c3\u02c4\u02c5\u02d2\u02d3\u02d4\u02d5\u02d6\u02d7\u02d8\u02d9\u02da\u02db\u02dc\u02dd\u02de\u02df\u02e5\u02e6\u02e7\u02e8\u02e9\u02ea\u02eb\u02ed\u02ef\u02f0\u02f1\u02f2\u02f3\u02f4\u02f5\u02f6\u02f7\u02f8\u02f9\u02fa\u02fb\u02fc\u02fd\u02fe\u02ff\u0375\u0384\u0385\u1fbd\u1fbf\u1fc0\u1fc1\u1fcd\u1fce\u1fcf\u1fdd\u1fde\u1fdf\u1fed\u1fee\u1fef\u1ffd\u1ffe\u309b\u309c\ua700\ua701\ua702\ua703\ua704\ua705\ua706\ua707\ua708\ua709\ua70a\ua70b\ua70c\ua70d\ua70e\ua70f\ua710\ua711\ua712\ua713\ua714\ua715\ua716\ua720\ua721\ua789\ua78a\ufbb2\ufbb3\ufbb4\ufbb5\ufbb6\ufbb7\ufbb8\ufbb9\ufbba\ufbbb\ufbbc\ufbbd\ufbbe\ufbbf\ufbc0\ufbc1\uff3e\uff40\uffe3'
			},
			{
				'description': 'Unicode character category [Sm]: symbol, math',
				'decoded': [43, 60, 61, 62, 124, 126, 172, 177, 215, 247, 1014, 1542, 1543, 1544, 8260, 8274, 8314, 8315, 8316, 8330, 8331, 8332, 8472, 8512, 8513, 8514, 8515, 8516, 8523, 8592, 8593, 8594, 8595, 8596, 8602, 8603, 8608, 8611, 8614, 8622, 8654, 8655, 8658, 8660, 8692, 8693, 8694, 8695, 8696, 8697, 8698, 8699, 8700, 8701, 8702, 8703, 8704, 8705, 8706, 8707, 8708, 8709, 8710, 8711, 8712, 8713, 8714, 8715, 8716, 8717, 8718, 8719, 8720, 8721, 8722, 8723, 8724, 8725, 8726, 8727, 8728, 8729, 8730, 8731, 8732, 8733, 8734, 8735, 8736, 8737, 8738, 8739, 8740, 8741, 8742, 8743, 8744, 8745, 8746, 8747, 8748, 8749, 8750, 8751, 8752, 8753, 8754, 8755, 8756, 8757, 8758, 8759, 8760, 8761, 8762, 8763, 8764, 8765, 8766, 8767, 8768, 8769, 8770, 8771, 8772, 8773, 8774, 8775, 8776, 8777, 8778, 8779, 8780, 8781, 8782, 8783, 8784, 8785, 8786, 8787, 8788, 8789, 8790, 8791, 8792, 8793, 8794, 8795, 8796, 8797, 8798, 8799, 8800, 8801, 8802, 8803, 8804, 8805, 8806, 8807, 8808, 8809, 8810, 8811, 8812, 8813, 8814, 8815, 8816, 8817, 8818, 8819, 8820, 8821, 8822, 8823, 8824, 8825, 8826, 8827, 8828, 8829, 8830, 8831, 8832, 8833, 8834, 8835, 8836, 8837, 8838, 8839, 8840, 8841, 8842, 8843, 8844, 8845, 8846, 8847, 8848, 8849, 8850, 8851, 8852, 8853, 8854, 8855, 8856, 8857, 8858, 8859, 8860, 8861, 8862, 8863, 8864, 8865, 8866, 8867, 8868, 8869, 8870, 8871, 8872, 8873, 8874, 8875, 8876, 8877, 8878, 8879, 8880, 8881, 8882, 8883, 8884, 8885, 8886, 8887, 8888, 8889, 8890, 8891, 8892, 8893, 8894, 8895, 8896, 8897, 8898, 8899, 8900, 8901, 8902, 8903, 8904, 8905, 8906, 8907, 8908, 8909, 8910, 8911, 8912, 8913, 8914, 8915, 8916, 8917, 8918, 8919, 8920, 8921, 8922, 8923, 8924, 8925, 8926, 8927, 8928, 8929, 8930, 8931, 8932, 8933, 8934, 8935, 8936, 8937, 8938, 8939, 8940, 8941, 8942, 8943, 8944, 8945, 8946, 8947, 8948, 8949, 8950, 8951, 8952, 8953, 8954, 8955, 8956, 8957, 8958, 8959, 8968, 8969, 8970, 8971, 8992, 8993, 9084, 9115, 9116, 9117, 9118, 9119, 9120, 9121, 9122, 9123, 9124, 9125, 9126, 9127, 9128, 9129, 9130, 9131, 9132, 9133, 9134, 9135, 9136, 9137, 9138, 9139, 9180, 9181, 9182, 9183, 9184, 9185, 9655, 9665, 9720, 9721, 9722, 9723, 9724, 9725, 9726, 9727, 9839, 10176, 10177, 10178, 10179, 10180, 10183, 10184, 10185, 10186, 10188, 10190, 10191, 10192, 10193, 10194, 10195, 10196, 10197, 10198, 10199, 10200, 10201, 10202, 10203, 10204, 10205, 10206, 10207, 10208, 10209, 10210, 10211, 10212, 10213, 10224, 10225, 10226, 10227, 10228, 10229, 10230, 10231, 10232, 10233, 10234, 10235, 10236, 10237, 10238, 10239, 10496, 10497, 10498, 10499, 10500, 10501, 10502, 10503, 10504, 10505, 10506, 10507, 10508, 10509, 10510, 10511, 10512, 10513, 10514, 10515, 10516, 10517, 10518, 10519, 10520, 10521, 10522, 10523, 10524, 10525, 10526, 10527, 10528, 10529, 10530, 10531, 10532, 10533, 10534, 10535, 10536, 10537, 10538, 10539, 10540, 10541, 10542, 10543, 10544, 10545, 10546, 10547, 10548, 10549, 10550, 10551, 10552, 10553, 10554, 10555, 10556, 10557, 10558, 10559, 10560, 10561, 10562, 10563, 10564, 10565, 10566, 10567, 10568, 10569, 10570, 10571, 10572, 10573, 10574, 10575, 10576, 10577, 10578, 10579, 10580, 10581, 10582, 10583, 10584, 10585, 10586, 10587, 10588, 10589, 10590, 10591, 10592, 10593, 10594, 10595, 10596, 10597, 10598, 10599, 10600, 10601, 10602, 10603, 10604, 10605, 10606, 10607, 10608, 10609, 10610, 10611, 10612, 10613, 10614, 10615, 10616, 10617, 10618, 10619, 10620, 10621, 10622, 10623, 10624, 10625, 10626, 10649, 10650, 10651, 10652, 10653, 10654, 10655, 10656, 10657, 10658, 10659, 10660, 10661, 10662, 10663, 10664, 10665, 10666, 10667, 10668, 10669, 10670, 10671, 10672, 10673, 10674, 10675, 10676, 10677, 10678, 10679, 10680, 10681, 10682, 10683, 10684, 10685, 10686, 10687, 10688, 10689, 10690, 10691, 10692, 10693, 10694, 10695, 10696, 10697, 10698, 10699, 10700, 10701, 10702, 10703, 10704, 10705, 10706, 10707, 10708, 10709, 10710, 10711, 10716, 10717, 10718, 10719, 10720, 10721, 10722, 10723, 10724, 10725, 10726, 10727, 10728, 10729, 10730, 10731, 10732, 10733, 10734, 10735, 10736, 10737, 10738, 10739, 10740, 10741, 10742, 10743, 10744, 10745, 10746, 10747, 10750, 10751, 10752, 10753, 10754, 10755, 10756, 10757, 10758, 10759, 10760, 10761, 10762, 10763, 10764, 10765, 10766, 10767, 10768, 10769, 10770, 10771, 10772, 10773, 10774, 10775, 10776, 10777, 10778, 10779, 10780, 10781, 10782, 10783, 10784, 10785, 10786, 10787, 10788, 10789, 10790, 10791, 10792, 10793, 10794, 10795, 10796, 10797, 10798, 10799, 10800, 10801, 10802, 10803, 10804, 10805, 10806, 10807, 10808, 10809, 10810, 10811, 10812, 10813, 10814, 10815, 10816, 10817, 10818, 10819, 10820, 10821, 10822, 10823, 10824, 10825, 10826, 10827, 10828, 10829, 10830, 10831, 10832, 10833, 10834, 10835, 10836, 10837, 10838, 10839, 10840, 10841, 10842, 10843, 10844, 10845, 10846, 10847, 10848, 10849, 10850, 10851, 10852, 10853, 10854, 10855, 10856, 10857, 10858, 10859, 10860, 10861, 10862, 10863, 10864, 10865, 10866, 10867, 10868, 10869, 10870, 10871, 10872, 10873, 10874, 10875, 10876, 10877, 10878, 10879, 10880, 10881, 10882, 10883, 10884, 10885, 10886, 10887, 10888, 10889, 10890, 10891, 10892, 10893, 10894, 10895, 10896, 10897, 10898, 10899, 10900, 10901, 10902, 10903, 10904, 10905, 10906, 10907, 10908, 10909, 10910, 10911, 10912, 10913, 10914, 10915, 10916, 10917, 10918, 10919, 10920, 10921, 10922, 10923, 10924, 10925, 10926, 10927, 10928, 10929, 10930, 10931, 10932, 10933, 10934, 10935, 10936, 10937, 10938, 10939, 10940, 10941, 10942, 10943, 10944, 10945, 10946, 10947, 10948, 10949, 10950, 10951, 10952, 10953, 10954, 10955, 10956, 10957, 10958, 10959, 10960, 10961, 10962, 10963, 10964, 10965, 10966, 10967, 10968, 10969, 10970, 10971, 10972, 10973, 10974, 10975, 10976, 10977, 10978, 10979, 10980, 10981, 10982, 10983, 10984, 10985, 10986, 10987, 10988, 10989, 10990, 10991, 10992, 10993, 10994, 10995, 10996, 10997, 10998, 10999, 11000, 11001, 11002, 11003, 11004, 11005, 11006, 11007, 11056, 11057, 11058, 11059, 11060, 11061, 11062, 11063, 11064, 11065, 11066, 11067, 11068, 11069, 11070, 11071, 11072, 11073, 11074, 11075, 11076, 11079, 11080, 11081, 11082, 11083, 11084, 64297, 65122, 65124, 65125, 65126, 65291, 65308, 65309, 65310, 65372, 65374, 65506, 65513, 65514, 65515, 65516, 120513, 120539, 120571, 120597, 120629, 120655, 120687, 120713, 120745, 120771],
				'encoded': '+<=>|~\xac\xb1\xd7\xf7\u03f6\u0606\u0607\u0608\u2044\u2052\u207a\u207b\u207c\u208a\u208b\u208c\u2118\u2140\u2141\u2142\u2143\u2144\u214b\u2190\u2191\u2192\u2193\u2194\u219a\u219b\u21a0\u21a3\u21a6\u21ae\u21ce\u21cf\u21d2\u21d4\u21f4\u21f5\u21f6\u21f7\u21f8\u21f9\u21fa\u21fb\u21fc\u21fd\u21fe\u21ff\u2200\u2201\u2202\u2203\u2204\u2205\u2206\u2207\u2208\u2209\u220a\u220b\u220c\u220d\u220e\u220f\u2210\u2211\u2212\u2213\u2214\u2215\u2216\u2217\u2218\u2219\u221a\u221b\u221c\u221d\u221e\u221f\u2220\u2221\u2222\u2223\u2224\u2225\u2226\u2227\u2228\u2229\u222a\u222b\u222c\u222d\u222e\u222f\u2230\u2231\u2232\u2233\u2234\u2235\u2236\u2237\u2238\u2239\u223a\u223b\u223c\u223d\u223e\u223f\u2240\u2241\u2242\u2243\u2244\u2245\u2246\u2247\u2248\u2249\u224a\u224b\u224c\u224d\u224e\u224f\u2250\u2251\u2252\u2253\u2254\u2255\u2256\u2257\u2258\u2259\u225a\u225b\u225c\u225d\u225e\u225f\u2260\u2261\u2262\u2263\u2264\u2265\u2266\u2267\u2268\u2269\u226a\u226b\u226c\u226d\u226e\u226f\u2270\u2271\u2272\u2273\u2274\u2275\u2276\u2277\u2278\u2279\u227a\u227b\u227c\u227d\u227e\u227f\u2280\u2281\u2282\u2283\u2284\u2285\u2286\u2287\u2288\u2289\u228a\u228b\u228c\u228d\u228e\u228f\u2290\u2291\u2292\u2293\u2294\u2295\u2296\u2297\u2298\u2299\u229a\u229b\u229c\u229d\u229e\u229f\u22a0\u22a1\u22a2\u22a3\u22a4\u22a5\u22a6\u22a7\u22a8\u22a9\u22aa\u22ab\u22ac\u22ad\u22ae\u22af\u22b0\u22b1\u22b2\u22b3\u22b4\u22b5\u22b6\u22b7\u22b8\u22b9\u22ba\u22bb\u22bc\u22bd\u22be\u22bf\u22c0\u22c1\u22c2\u22c3\u22c4\u22c5\u22c6\u22c7\u22c8\u22c9\u22ca\u22cb\u22cc\u22cd\u22ce\u22cf\u22d0\u22d1\u22d2\u22d3\u22d4\u22d5\u22d6\u22d7\u22d8\u22d9\u22da\u22db\u22dc\u22dd\u22de\u22df\u22e0\u22e1\u22e2\u22e3\u22e4\u22e5\u22e6\u22e7\u22e8\u22e9\u22ea\u22eb\u22ec\u22ed\u22ee\u22ef\u22f0\u22f1\u22f2\u22f3\u22f4\u22f5\u22f6\u22f7\u22f8\u22f9\u22fa\u22fb\u22fc\u22fd\u22fe\u22ff\u2308\u2309\u230a\u230b\u2320\u2321\u237c\u239b\u239c\u239d\u239e\u239f\u23a0\u23a1\u23a2\u23a3\u23a4\u23a5\u23a6\u23a7\u23a8\u23a9\u23aa\u23ab\u23ac\u23ad\u23ae\u23af\u23b0\u23b1\u23b2\u23b3\u23dc\u23dd\u23de\u23df\u23e0\u23e1\u25b7\u25c1\u25f8\u25f9\u25fa\u25fb\u25fc\u25fd\u25fe\u25ff\u266f\u27c0\u27c1\u27c2\u27c3\u27c4\u27c7\u27c8\u27c9\u27ca\u27cc\u27ce\u27cf\u27d0\u27d1\u27d2\u27d3\u27d4\u27d5\u27d6\u27d7\u27d8\u27d9\u27da\u27db\u27dc\u27dd\u27de\u27df\u27e0\u27e1\u27e2\u27e3\u27e4\u27e5\u27f0\u27f1\u27f2\u27f3\u27f4\u27f5\u27f6\u27f7\u27f8\u27f9\u27fa\u27fb\u27fc\u27fd\u27fe\u27ff\u2900\u2901\u2902\u2903\u2904\u2905\u2906\u2907\u2908\u2909\u290a\u290b\u290c\u290d\u290e\u290f\u2910\u2911\u2912\u2913\u2914\u2915\u2916\u2917\u2918\u2919\u291a\u291b\u291c\u291d\u291e\u291f\u2920\u2921\u2922\u2923\u2924\u2925\u2926\u2927\u2928\u2929\u292a\u292b\u292c\u292d\u292e\u292f\u2930\u2931\u2932\u2933\u2934\u2935\u2936\u2937\u2938\u2939\u293a\u293b\u293c\u293d\u293e\u293f\u2940\u2941\u2942\u2943\u2944\u2945\u2946\u2947\u2948\u2949\u294a\u294b\u294c\u294d\u294e\u294f\u2950\u2951\u2952\u2953\u2954\u2955\u2956\u2957\u2958\u2959\u295a\u295b\u295c\u295d\u295e\u295f\u2960\u2961\u2962\u2963\u2964\u2965\u2966\u2967\u2968\u2969\u296a\u296b\u296c\u296d\u296e\u296f\u2970\u2971\u2972\u2973\u2974\u2975\u2976\u2977\u2978\u2979\u297a\u297b\u297c\u297d\u297e\u297f\u2980\u2981\u2982\u2999\u299a\u299b\u299c\u299d\u299e\u299f\u29a0\u29a1\u29a2\u29a3\u29a4\u29a5\u29a6\u29a7\u29a8\u29a9\u29aa\u29ab\u29ac\u29ad\u29ae\u29af\u29b0\u29b1\u29b2\u29b3\u29b4\u29b5\u29b6\u29b7\u29b8\u29b9\u29ba\u29bb\u29bc\u29bd\u29be\u29bf\u29c0\u29c1\u29c2\u29c3\u29c4\u29c5\u29c6\u29c7\u29c8\u29c9\u29ca\u29cb\u29cc\u29cd\u29ce\u29cf\u29d0\u29d1\u29d2\u29d3\u29d4\u29d5\u29d6\u29d7\u29dc\u29dd\u29de\u29df\u29e0\u29e1\u29e2\u29e3\u29e4\u29e5\u29e6\u29e7\u29e8\u29e9\u29ea\u29eb\u29ec\u29ed\u29ee\u29ef\u29f0\u29f1\u29f2\u29f3\u29f4\u29f5\u29f6\u29f7\u29f8\u29f9\u29fa\u29fb\u29fe\u29ff\u2a00\u2a01\u2a02\u2a03\u2a04\u2a05\u2a06\u2a07\u2a08\u2a09\u2a0a\u2a0b\u2a0c\u2a0d\u2a0e\u2a0f\u2a10\u2a11\u2a12\u2a13\u2a14\u2a15\u2a16\u2a17\u2a18\u2a19\u2a1a\u2a1b\u2a1c\u2a1d\u2a1e\u2a1f\u2a20\u2a21\u2a22\u2a23\u2a24\u2a25\u2a26\u2a27\u2a28\u2a29\u2a2a\u2a2b\u2a2c\u2a2d\u2a2e\u2a2f\u2a30\u2a31\u2a32\u2a33\u2a34\u2a35\u2a36\u2a37\u2a38\u2a39\u2a3a\u2a3b\u2a3c\u2a3d\u2a3e\u2a3f\u2a40\u2a41\u2a42\u2a43\u2a44\u2a45\u2a46\u2a47\u2a48\u2a49\u2a4a\u2a4b\u2a4c\u2a4d\u2a4e\u2a4f\u2a50\u2a51\u2a52\u2a53\u2a54\u2a55\u2a56\u2a57\u2a58\u2a59\u2a5a\u2a5b\u2a5c\u2a5d\u2a5e\u2a5f\u2a60\u2a61\u2a62\u2a63\u2a64\u2a65\u2a66\u2a67\u2a68\u2a69\u2a6a\u2a6b\u2a6c\u2a6d\u2a6e\u2a6f\u2a70\u2a71\u2a72\u2a73\u2a74\u2a75\u2a76\u2a77\u2a78\u2a79\u2a7a\u2a7b\u2a7c\u2a7d\u2a7e\u2a7f\u2a80\u2a81\u2a82\u2a83\u2a84\u2a85\u2a86\u2a87\u2a88\u2a89\u2a8a\u2a8b\u2a8c\u2a8d\u2a8e\u2a8f\u2a90\u2a91\u2a92\u2a93\u2a94\u2a95\u2a96\u2a97\u2a98\u2a99\u2a9a\u2a9b\u2a9c\u2a9d\u2a9e\u2a9f\u2aa0\u2aa1\u2aa2\u2aa3\u2aa4\u2aa5\u2aa6\u2aa7\u2aa8\u2aa9\u2aaa\u2aab\u2aac\u2aad\u2aae\u2aaf\u2ab0\u2ab1\u2ab2\u2ab3\u2ab4\u2ab5\u2ab6\u2ab7\u2ab8\u2ab9\u2aba\u2abb\u2abc\u2abd\u2abe\u2abf\u2ac0\u2ac1\u2ac2\u2ac3\u2ac4\u2ac5\u2ac6\u2ac7\u2ac8\u2ac9\u2aca\u2acb\u2acc\u2acd\u2ace\u2acf\u2ad0\u2ad1\u2ad2\u2ad3\u2ad4\u2ad5\u2ad6\u2ad7\u2ad8\u2ad9\u2ada\u2adb\u2adc\u2add\u2ade\u2adf\u2ae0\u2ae1\u2ae2\u2ae3\u2ae4\u2ae5\u2ae6\u2ae7\u2ae8\u2ae9\u2aea\u2aeb\u2aec\u2aed\u2aee\u2aef\u2af0\u2af1\u2af2\u2af3\u2af4\u2af5\u2af6\u2af7\u2af8\u2af9\u2afa\u2afb\u2afc\u2afd\u2afe\u2aff\u2b30\u2b31\u2b32\u2b33\u2b34\u2b35\u2b36\u2b37\u2b38\u2b39\u2b3a\u2b3b\u2b3c\u2b3d\u2b3e\u2b3f\u2b40\u2b41\u2b42\u2b43\u2b44\u2b47\u2b48\u2b49\u2b4a\u2b4b\u2b4c\ufb29\ufe62\ufe64\ufe65\ufe66\uff0b\uff1c\uff1d\uff1e\uff5c\uff5e\uffe2\uffe9\uffea\uffeb\uffec\ud835\udec1\ud835\udedb\ud835\udefb\ud835\udf15\ud835\udf35\ud835\udf4f\ud835\udf6f\ud835\udf89\ud835\udfa9\ud835\udfc3'
			},
			{
				'description': 'Unicode character category [So]: symbol, other',
				'decoded': [166, 167, 169, 174, 176, 182, 1154, 1550, 1551, 1758, 1769, 1789, 1790, 2038, 2554, 2928, 3059, 3060, 3061, 3062, 3063, 3064, 3066, 3199, 3449, 3841, 3842, 3843, 3859, 3860, 3861, 3862, 3863, 3866, 3867, 3868, 3869, 3870, 3871, 3892, 3894, 3896, 4030, 4031, 4032, 4033, 4034, 4035, 4036, 4037, 4039, 4040, 4041, 4042, 4043, 4044, 4046, 4047, 4053, 4054, 4055, 4056, 4254, 4255, 4960, 5008, 5009, 5010, 5011, 5012, 5013, 5014, 5015, 5016, 5017, 6464, 6622, 6623, 6624, 6625, 6626, 6627, 6628, 6629, 6630, 6631, 6632, 6633, 6634, 6635, 6636, 6637, 6638, 6639, 6640, 6641, 6642, 6643, 6644, 6645, 6646, 6647, 6648, 6649, 6650, 6651, 6652, 6653, 6654, 6655, 7009, 7010, 7011, 7012, 7013, 7014, 7015, 7016, 7017, 7018, 7028, 7029, 7030, 7031, 7032, 7033, 7034, 7035, 7036, 8448, 8449, 8451, 8452, 8453, 8454, 8456, 8457, 8468, 8470, 8471, 8478, 8479, 8480, 8481, 8482, 8483, 8485, 8487, 8489, 8494, 8506, 8507, 8522, 8524, 8525, 8527, 8597, 8598, 8599, 8600, 8601, 8604, 8605, 8606, 8607, 8609, 8610, 8612, 8613, 8615, 8616, 8617, 8618, 8619, 8620, 8621, 8623, 8624, 8625, 8626, 8627, 8628, 8629, 8630, 8631, 8632, 8633, 8634, 8635, 8636, 8637, 8638, 8639, 8640, 8641, 8642, 8643, 8644, 8645, 8646, 8647, 8648, 8649, 8650, 8651, 8652, 8653, 8656, 8657, 8659, 8661, 8662, 8663, 8664, 8665, 8666, 8667, 8668, 8669, 8670, 8671, 8672, 8673, 8674, 8675, 8676, 8677, 8678, 8679, 8680, 8681, 8682, 8683, 8684, 8685, 8686, 8687, 8688, 8689, 8690, 8691, 8960, 8961, 8962, 8963, 8964, 8965, 8966, 8967, 8972, 8973, 8974, 8975, 8976, 8977, 8978, 8979, 8980, 8981, 8982, 8983, 8984, 8985, 8986, 8987, 8988, 8989, 8990, 8991, 8994, 8995, 8996, 8997, 8998, 8999, 9000, 9003, 9004, 9005, 9006, 9007, 9008, 9009, 9010, 9011, 9012, 9013, 9014, 9015, 9016, 9017, 9018, 9019, 9020, 9021, 9022, 9023, 9024, 9025, 9026, 9027, 9028, 9029, 9030, 9031, 9032, 9033, 9034, 9035, 9036, 9037, 9038, 9039, 9040, 9041, 9042, 9043, 9044, 9045, 9046, 9047, 9048, 9049, 9050, 9051, 9052, 9053, 9054, 9055, 9056, 9057, 9058, 9059, 9060, 9061, 9062, 9063, 9064, 9065, 9066, 9067, 9068, 9069, 9070, 9071, 9072, 9073, 9074, 9075, 9076, 9077, 9078, 9079, 9080, 9081, 9082, 9083, 9085, 9086, 9087, 9088, 9089, 9090, 9091, 9092, 9093, 9094, 9095, 9096, 9097, 9098, 9099, 9100, 9101, 9102, 9103, 9104, 9105, 9106, 9107, 9108, 9109, 9110, 9111, 9112, 9113, 9114, 9140, 9141, 9142, 9143, 9144, 9145, 9146, 9147, 9148, 9149, 9150, 9151, 9152, 9153, 9154, 9155, 9156, 9157, 9158, 9159, 9160, 9161, 9162, 9163, 9164, 9165, 9166, 9167, 9168, 9169, 9170, 9171, 9172, 9173, 9174, 9175, 9176, 9177, 9178, 9179, 9186, 9187, 9188, 9189, 9190, 9191, 9192, 9193, 9194, 9195, 9196, 9197, 9198, 9199, 9200, 9201, 9202, 9203, 9216, 9217, 9218, 9219, 9220, 9221, 9222, 9223, 9224, 9225, 9226, 9227, 9228, 9229, 9230, 9231, 9232, 9233, 9234, 9235, 9236, 9237, 9238, 9239, 9240, 9241, 9242, 9243, 9244, 9245, 9246, 9247, 9248, 9249, 9250, 9251, 9252, 9253, 9254, 9280, 9281, 9282, 9283, 9284, 9285, 9286, 9287, 9288, 9289, 9290, 9372, 9373, 9374, 9375, 9376, 9377, 9378, 9379, 9380, 9381, 9382, 9383, 9384, 9385, 9386, 9387, 9388, 9389, 9390, 9391, 9392, 9393, 9394, 9395, 9396, 9397, 9398, 9399, 9400, 9401, 9402, 9403, 9404, 9405, 9406, 9407, 9408, 9409, 9410, 9411, 9412, 9413, 9414, 9415, 9416, 9417, 9418, 9419, 9420, 9421, 9422, 9423, 9424, 9425, 9426, 9427, 9428, 9429, 9430, 9431, 9432, 9433, 9434, 9435, 9436, 9437, 9438, 9439, 9440, 9441, 9442, 9443, 9444, 9445, 9446, 9447, 9448, 9449, 9472, 9473, 9474, 9475, 9476, 9477, 9478, 9479, 9480, 9481, 9482, 9483, 9484, 9485, 9486, 9487, 9488, 9489, 9490, 9491, 9492, 9493, 9494, 9495, 9496, 9497, 9498, 9499, 9500, 9501, 9502, 9503, 9504, 9505, 9506, 9507, 9508, 9509, 9510, 9511, 9512, 9513, 9514, 9515, 9516, 9517, 9518, 9519, 9520, 9521, 9522, 9523, 9524, 9525, 9526, 9527, 9528, 9529, 9530, 9531, 9532, 9533, 9534, 9535, 9536, 9537, 9538, 9539, 9540, 9541, 9542, 9543, 9544, 9545, 9546, 9547, 9548, 9549, 9550, 9551, 9552, 9553, 9554, 9555, 9556, 9557, 9558, 9559, 9560, 9561, 9562, 9563, 9564, 9565, 9566, 9567, 9568, 9569, 9570, 9571, 9572, 9573, 9574, 9575, 9576, 9577, 9578, 9579, 9580, 9581, 9582, 9583, 9584, 9585, 9586, 9587, 9588, 9589, 9590, 9591, 9592, 9593, 9594, 9595, 9596, 9597, 9598, 9599, 9600, 9601, 9602, 9603, 9604, 9605, 9606, 9607, 9608, 9609, 9610, 9611, 9612, 9613, 9614, 9615, 9616, 9617, 9618, 9619, 9620, 9621, 9622, 9623, 9624, 9625, 9626, 9627, 9628, 9629, 9630, 9631, 9632, 9633, 9634, 9635, 9636, 9637, 9638, 9639, 9640, 9641, 9642, 9643, 9644, 9645, 9646, 9647, 9648, 9649, 9650, 9651, 9652, 9653, 9654, 9656, 9657, 9658, 9659, 9660, 9661, 9662, 9663, 9664, 9666, 9667, 9668, 9669, 9670, 9671, 9672, 9673, 9674, 9675, 9676, 9677, 9678, 9679, 9680, 9681, 9682, 9683, 9684, 9685, 9686, 9687, 9688, 9689, 9690, 9691, 9692, 9693, 9694, 9695, 9696, 9697, 9698, 9699, 9700, 9701, 9702, 9703, 9704, 9705, 9706, 9707, 9708, 9709, 9710, 9711, 9712, 9713, 9714, 9715, 9716, 9717, 9718, 9719, 9728, 9729, 9730, 9731, 9732, 9733, 9734, 9735, 9736, 9737, 9738, 9739, 9740, 9741, 9742, 9743, 9744, 9745, 9746, 9747, 9748, 9749, 9750, 9751, 9752, 9753, 9754, 9755, 9756, 9757, 9758, 9759, 9760, 9761, 9762, 9763, 9764, 9765, 9766, 9767, 9768, 9769, 9770, 9771, 9772, 9773, 9774, 9775, 9776, 9777, 9778, 9779, 9780, 9781, 9782, 9783, 9784, 9785, 9786, 9787, 9788, 9789, 9790, 9791, 9792, 9793, 9794, 9795, 9796, 9797, 9798, 9799, 9800, 9801, 9802, 9803, 9804, 9805, 9806, 9807, 9808, 9809, 9810, 9811, 9812, 9813, 9814, 9815, 9816, 9817, 9818, 9819, 9820, 9821, 9822, 9823, 9824, 9825, 9826, 9827, 9828, 9829, 9830, 9831, 9832, 9833, 9834, 9835, 9836, 9837, 9838, 9840, 9841, 9842, 9843, 9844, 9845, 9846, 9847, 9848, 9849, 9850, 9851, 9852, 9853, 9854, 9855, 9856, 9857, 9858, 9859, 9860, 9861, 9862, 9863, 9864, 9865, 9866, 9867, 9868, 9869, 9870, 9871, 9872, 9873, 9874, 9875, 9876, 9877, 9878, 9879, 9880, 9881, 9882, 9883, 9884, 9885, 9886, 9887, 9888, 9889, 9890, 9891, 9892, 9893, 9894, 9895, 9896, 9897, 9898, 9899, 9900, 9901, 9902, 9903, 9904, 9905, 9906, 9907, 9908, 9909, 9910, 9911, 9912, 9913, 9914, 9915, 9916, 9917, 9918, 9919, 9920, 9921, 9922, 9923, 9924, 9925, 9926, 9927, 9928, 9929, 9930, 9931, 9932, 9933, 9934, 9935, 9936, 9937, 9938, 9939, 9940, 9941, 9942, 9943, 9944, 9945, 9946, 9947, 9948, 9949, 9950, 9951, 9952, 9953, 9954, 9955, 9956, 9957, 9958, 9959, 9960, 9961, 9962, 9963, 9964, 9965, 9966, 9967, 9968, 9969, 9970, 9971, 9972, 9973, 9974, 9975, 9976, 9977, 9978, 9979, 9980, 9981, 9982, 9983, 9985, 9986, 9987, 9988, 9989, 9990, 9991, 9992, 9993, 9994, 9995, 9996, 9997, 9998, 9999, 10000, 10001, 10002, 10003, 10004, 10005, 10006, 10007, 10008, 10009, 10010, 10011, 10012, 10013, 10014, 10015, 10016, 10017, 10018, 10019, 10020, 10021, 10022, 10023, 10024, 10025, 10026, 10027, 10028, 10029, 10030, 10031, 10032, 10033, 10034, 10035, 10036, 10037, 10038, 10039, 10040, 10041, 10042, 10043, 10044, 10045, 10046, 10047, 10048, 10049, 10050, 10051, 10052, 10053, 10054, 10055, 10056, 10057, 10058, 10059, 10060, 10061, 10062, 10063, 10064, 10065, 10066, 10067, 10068, 10069, 10070, 10071, 10072, 10073, 10074, 10075, 10076, 10077, 10078, 10079, 10080, 10081, 10082, 10083, 10084, 10085, 10086, 10087, 10132, 10133, 10134, 10135, 10136, 10137, 10138, 10139, 10140, 10141, 10142, 10143, 10144, 10145, 10146, 10147, 10148, 10149, 10150, 10151, 10152, 10153, 10154, 10155, 10156, 10157, 10158, 10159, 10160, 10161, 10162, 10163, 10164, 10165, 10166, 10167, 10168, 10169, 10170, 10171, 10172, 10173, 10174, 10175, 10240, 10241, 10242, 10243, 10244, 10245, 10246, 10247, 10248, 10249, 10250, 10251, 10252, 10253, 10254, 10255, 10256, 10257, 10258, 10259, 10260, 10261, 10262, 10263, 10264, 10265, 10266, 10267, 10268, 10269, 10270, 10271, 10272, 10273, 10274, 10275, 10276, 10277, 10278, 10279, 10280, 10281, 10282, 10283, 10284, 10285, 10286, 10287, 10288, 10289, 10290, 10291, 10292, 10293, 10294, 10295, 10296, 10297, 10298, 10299, 10300, 10301, 10302, 10303, 10304, 10305, 10306, 10307, 10308, 10309, 10310, 10311, 10312, 10313, 10314, 10315, 10316, 10317, 10318, 10319, 10320, 10321, 10322, 10323, 10324, 10325, 10326, 10327, 10328, 10329, 10330, 10331, 10332, 10333, 10334, 10335, 10336, 10337, 10338, 10339, 10340, 10341, 10342, 10343, 10344, 10345, 10346, 10347, 10348, 10349, 10350, 10351, 10352, 10353, 10354, 10355, 10356, 10357, 10358, 10359, 10360, 10361, 10362, 10363, 10364, 10365, 10366, 10367, 10368, 10369, 10370, 10371, 10372, 10373, 10374, 10375, 10376, 10377, 10378, 10379, 10380, 10381, 10382, 10383, 10384, 10385, 10386, 10387, 10388, 10389, 10390, 10391, 10392, 10393, 10394, 10395, 10396, 10397, 10398, 10399, 10400, 10401, 10402, 10403, 10404, 10405, 10406, 10407, 10408, 10409, 10410, 10411, 10412, 10413, 10414, 10415, 10416, 10417, 10418, 10419, 10420, 10421, 10422, 10423, 10424, 10425, 10426, 10427, 10428, 10429, 10430, 10431, 10432, 10433, 10434, 10435, 10436, 10437, 10438, 10439, 10440, 10441, 10442, 10443, 10444, 10445, 10446, 10447, 10448, 10449, 10450, 10451, 10452, 10453, 10454, 10455, 10456, 10457, 10458, 10459, 10460, 10461, 10462, 10463, 10464, 10465, 10466, 10467, 10468, 10469, 10470, 10471, 10472, 10473, 10474, 10475, 10476, 10477, 10478, 10479, 10480, 10481, 10482, 10483, 10484, 10485, 10486, 10487, 10488, 10489, 10490, 10491, 10492, 10493, 10494, 10495, 11008, 11009, 11010, 11011, 11012, 11013, 11014, 11015, 11016, 11017, 11018, 11019, 11020, 11021, 11022, 11023, 11024, 11025, 11026, 11027, 11028, 11029, 11030, 11031, 11032, 11033, 11034, 11035, 11036, 11037, 11038, 11039, 11040, 11041, 11042, 11043, 11044, 11045, 11046, 11047, 11048, 11049, 11050, 11051, 11052, 11053, 11054, 11055, 11077, 11078, 11088, 11089, 11090, 11091, 11092, 11093, 11094, 11095, 11096, 11097, 11493, 11494, 11495, 11496, 11497, 11498, 11904, 11905, 11906, 11907, 11908, 11909, 11910, 11911, 11912, 11913, 11914, 11915, 11916, 11917, 11918, 11919, 11920, 11921, 11922, 11923, 11924, 11925, 11926, 11927, 11928, 11929, 11931, 11932, 11933, 11934, 11935, 11936, 11937, 11938, 11939, 11940, 11941, 11942, 11943, 11944, 11945, 11946, 11947, 11948, 11949, 11950, 11951, 11952, 11953, 11954, 11955, 11956, 11957, 11958, 11959, 11960, 11961, 11962, 11963, 11964, 11965, 11966, 11967, 11968, 11969, 11970, 11971, 11972, 11973, 11974, 11975, 11976, 11977, 11978, 11979, 11980, 11981, 11982, 11983, 11984, 11985, 11986, 11987, 11988, 11989, 11990, 11991, 11992, 11993, 11994, 11995, 11996, 11997, 11998, 11999, 12000, 12001, 12002, 12003, 12004, 12005, 12006, 12007, 12008, 12009, 12010, 12011, 12012, 12013, 12014, 12015, 12016, 12017, 12018, 12019, 12032, 12033, 12034, 12035, 12036, 12037, 12038, 12039, 12040, 12041, 12042, 12043, 12044, 12045, 12046, 12047, 12048, 12049, 12050, 12051, 12052, 12053, 12054, 12055, 12056, 12057, 12058, 12059, 12060, 12061, 12062, 12063, 12064, 12065, 12066, 12067, 12068, 12069, 12070, 12071, 12072, 12073, 12074, 12075, 12076, 12077, 12078, 12079, 12080, 12081, 12082, 12083, 12084, 12085, 12086, 12087, 12088, 12089, 12090, 12091, 12092, 12093, 12094, 12095, 12096, 12097, 12098, 12099, 12100, 12101, 12102, 12103, 12104, 12105, 12106, 12107, 12108, 12109, 12110, 12111, 12112, 12113, 12114, 12115, 12116, 12117, 12118, 12119, 12120, 12121, 12122, 12123, 12124, 12125, 12126, 12127, 12128, 12129, 12130, 12131, 12132, 12133, 12134, 12135, 12136, 12137, 12138, 12139, 12140, 12141, 12142, 12143, 12144, 12145, 12146, 12147, 12148, 12149, 12150, 12151, 12152, 12153, 12154, 12155, 12156, 12157, 12158, 12159, 12160, 12161, 12162, 12163, 12164, 12165, 12166, 12167, 12168, 12169, 12170, 12171, 12172, 12173, 12174, 12175, 12176, 12177, 12178, 12179, 12180, 12181, 12182, 12183, 12184, 12185, 12186, 12187, 12188, 12189, 12190, 12191, 12192, 12193, 12194, 12195, 12196, 12197, 12198, 12199, 12200, 12201, 12202, 12203, 12204, 12205, 12206, 12207, 12208, 12209, 12210, 12211, 12212, 12213, 12214, 12215, 12216, 12217, 12218, 12219, 12220, 12221, 12222, 12223, 12224, 12225, 12226, 12227, 12228, 12229, 12230, 12231, 12232, 12233, 12234, 12235, 12236, 12237, 12238, 12239, 12240, 12241, 12242, 12243, 12244, 12245, 12272, 12273, 12274, 12275, 12276, 12277, 12278, 12279, 12280, 12281, 12282, 12283, 12292, 12306, 12307, 12320, 12342, 12343, 12350, 12351, 12688, 12689, 12694, 12695, 12696, 12697, 12698, 12699, 12700, 12701, 12702, 12703, 12736, 12737, 12738, 12739, 12740, 12741, 12742, 12743, 12744, 12745, 12746, 12747, 12748, 12749, 12750, 12751, 12752, 12753, 12754, 12755, 12756, 12757, 12758, 12759, 12760, 12761, 12762, 12763, 12764, 12765, 12766, 12767, 12768, 12769, 12770, 12771, 12800, 12801, 12802, 12803, 12804, 12805, 12806, 12807, 12808, 12809, 12810, 12811, 12812, 12813, 12814, 12815, 12816, 12817, 12818, 12819, 12820, 12821, 12822, 12823, 12824, 12825, 12826, 12827, 12828, 12829, 12830, 12842, 12843, 12844, 12845, 12846, 12847, 12848, 12849, 12850, 12851, 12852, 12853, 12854, 12855, 12856, 12857, 12858, 12859, 12860, 12861, 12862, 12863, 12864, 12865, 12866, 12867, 12868, 12869, 12870, 12871, 12872, 12873, 12874, 12875, 12876, 12877, 12878, 12879, 12880, 12896, 12897, 12898, 12899, 12900, 12901, 12902, 12903, 12904, 12905, 12906, 12907, 12908, 12909, 12910, 12911, 12912, 12913, 12914, 12915, 12916, 12917, 12918, 12919, 12920, 12921, 12922, 12923, 12924, 12925, 12926, 12927, 12938, 12939, 12940, 12941, 12942, 12943, 12944, 12945, 12946, 12947, 12948, 12949, 12950, 12951, 12952, 12953, 12954, 12955, 12956, 12957, 12958, 12959, 12960, 12961, 12962, 12963, 12964, 12965, 12966, 12967, 12968, 12969, 12970, 12971, 12972, 12973, 12974, 12975, 12976, 12992, 12993, 12994, 12995, 12996, 12997, 12998, 12999, 13000, 13001, 13002, 13003, 13004, 13005, 13006, 13007, 13008, 13009, 13010, 13011, 13012, 13013, 13014, 13015, 13016, 13017, 13018, 13019, 13020, 13021, 13022, 13023, 13024, 13025, 13026, 13027, 13028, 13029, 13030, 13031, 13032, 13033, 13034, 13035, 13036, 13037, 13038, 13039, 13040, 13041, 13042, 13043, 13044, 13045, 13046, 13047, 13048, 13049, 13050, 13051, 13052, 13053, 13054, 13056, 13057, 13058, 13059, 13060, 13061, 13062, 13063, 13064, 13065, 13066, 13067, 13068, 13069, 13070, 13071, 13072, 13073, 13074, 13075, 13076, 13077, 13078, 13079, 13080, 13081, 13082, 13083, 13084, 13085, 13086, 13087, 13088, 13089, 13090, 13091, 13092, 13093, 13094, 13095, 13096, 13097, 13098, 13099, 13100, 13101, 13102, 13103, 13104, 13105, 13106, 13107, 13108, 13109, 13110, 13111, 13112, 13113, 13114, 13115, 13116, 13117, 13118, 13119, 13120, 13121, 13122, 13123, 13124, 13125, 13126, 13127, 13128, 13129, 13130, 13131, 13132, 13133, 13134, 13135, 13136, 13137, 13138, 13139, 13140, 13141, 13142, 13143, 13144, 13145, 13146, 13147, 13148, 13149, 13150, 13151, 13152, 13153, 13154, 13155, 13156, 13157, 13158, 13159, 13160, 13161, 13162, 13163, 13164, 13165, 13166, 13167, 13168, 13169, 13170, 13171, 13172, 13173, 13174, 13175, 13176, 13177, 13178, 13179, 13180, 13181, 13182, 13183, 13184, 13185, 13186, 13187, 13188, 13189, 13190, 13191, 13192, 13193, 13194, 13195, 13196, 13197, 13198, 13199, 13200, 13201, 13202, 13203, 13204, 13205, 13206, 13207, 13208, 13209, 13210, 13211, 13212, 13213, 13214, 13215, 13216, 13217, 13218, 13219, 13220, 13221, 13222, 13223, 13224, 13225, 13226, 13227, 13228, 13229, 13230, 13231, 13232, 13233, 13234, 13235, 13236, 13237, 13238, 13239, 13240, 13241, 13242, 13243, 13244, 13245, 13246, 13247, 13248, 13249, 13250, 13251, 13252, 13253, 13254, 13255, 13256, 13257, 13258, 13259, 13260, 13261, 13262, 13263, 13264, 13265, 13266, 13267, 13268, 13269, 13270, 13271, 13272, 13273, 13274, 13275, 13276, 13277, 13278, 13279, 13280, 13281, 13282, 13283, 13284, 13285, 13286, 13287, 13288, 13289, 13290, 13291, 13292, 13293, 13294, 13295, 13296, 13297, 13298, 13299, 13300, 13301, 13302, 13303, 13304, 13305, 13306, 13307, 13308, 13309, 13310, 13311, 19904, 19905, 19906, 19907, 19908, 19909, 19910, 19911, 19912, 19913, 19914, 19915, 19916, 19917, 19918, 19919, 19920, 19921, 19922, 19923, 19924, 19925, 19926, 19927, 19928, 19929, 19930, 19931, 19932, 19933, 19934, 19935, 19936, 19937, 19938, 19939, 19940, 19941, 19942, 19943, 19944, 19945, 19946, 19947, 19948, 19949, 19950, 19951, 19952, 19953, 19954, 19955, 19956, 19957, 19958, 19959, 19960, 19961, 19962, 19963, 19964, 19965, 19966, 19967, 42128, 42129, 42130, 42131, 42132, 42133, 42134, 42135, 42136, 42137, 42138, 42139, 42140, 42141, 42142, 42143, 42144, 42145, 42146, 42147, 42148, 42149, 42150, 42151, 42152, 42153, 42154, 42155, 42156, 42157, 42158, 42159, 42160, 42161, 42162, 42163, 42164, 42165, 42166, 42167, 42168, 42169, 42170, 42171, 42172, 42173, 42174, 42175, 42176, 42177, 42178, 42179, 42180, 42181, 42182, 43048, 43049, 43050, 43051, 43062, 43063, 43065, 43639, 43640, 43641, 65021, 65508, 65512, 65517, 65518, 65532, 65533, 65794, 65847, 65848, 65849, 65850, 65851, 65852, 65853, 65854, 65855, 65913, 65914, 65915, 65916, 65917, 65918, 65919, 65920, 65921, 65922, 65923, 65924, 65925, 65926, 65927, 65928, 65929, 65936, 65937, 65938, 65939, 65940, 65941, 65942, 65943, 65944, 65945, 65946, 65947, 66000, 66001, 66002, 66003, 66004, 66005, 66006, 66007, 66008, 66009, 66010, 66011, 66012, 66013, 66014, 66015, 66016, 66017, 66018, 66019, 66020, 66021, 66022, 66023, 66024, 66025, 66026, 66027, 66028, 66029, 66030, 66031, 66032, 66033, 66034, 66035, 66036, 66037, 66038, 66039, 66040, 66041, 66042, 66043, 66044, 118784, 118785, 118786, 118787, 118788, 118789, 118790, 118791, 118792, 118793, 118794, 118795, 118796, 118797, 118798, 118799, 118800, 118801, 118802, 118803, 118804, 118805, 118806, 118807, 118808, 118809, 118810, 118811, 118812, 118813, 118814, 118815, 118816, 118817, 118818, 118819, 118820, 118821, 118822, 118823, 118824, 118825, 118826, 118827, 118828, 118829, 118830, 118831, 118832, 118833, 118834, 118835, 118836, 118837, 118838, 118839, 118840, 118841, 118842, 118843, 118844, 118845, 118846, 118847, 118848, 118849, 118850, 118851, 118852, 118853, 118854, 118855, 118856, 118857, 118858, 118859, 118860, 118861, 118862, 118863, 118864, 118865, 118866, 118867, 118868, 118869, 118870, 118871, 118872, 118873, 118874, 118875, 118876, 118877, 118878, 118879, 118880, 118881, 118882, 118883, 118884, 118885, 118886, 118887, 118888, 118889, 118890, 118891, 118892, 118893, 118894, 118895, 118896, 118897, 118898, 118899, 118900, 118901, 118902, 118903, 118904, 118905, 118906, 118907, 118908, 118909, 118910, 118911, 118912, 118913, 118914, 118915, 118916, 118917, 118918, 118919, 118920, 118921, 118922, 118923, 118924, 118925, 118926, 118927, 118928, 118929, 118930, 118931, 118932, 118933, 118934, 118935, 118936, 118937, 118938, 118939, 118940, 118941, 118942, 118943, 118944, 118945, 118946, 118947, 118948, 118949, 118950, 118951, 118952, 118953, 118954, 118955, 118956, 118957, 118958, 118959, 118960, 118961, 118962, 118963, 118964, 118965, 118966, 118967, 118968, 118969, 118970, 118971, 118972, 118973, 118974, 118975, 118976, 118977, 118978, 118979, 118980, 118981, 118982, 118983, 118984, 118985, 118986, 118987, 118988, 118989, 118990, 118991, 118992, 118993, 118994, 118995, 118996, 118997, 118998, 118999, 119000, 119001, 119002, 119003, 119004, 119005, 119006, 119007, 119008, 119009, 119010, 119011, 119012, 119013, 119014, 119015, 119016, 119017, 119018, 119019, 119020, 119021, 119022, 119023, 119024, 119025, 119026, 119027, 119028, 119029, 119040, 119041, 119042, 119043, 119044, 119045, 119046, 119047, 119048, 119049, 119050, 119051, 119052, 119053, 119054, 119055, 119056, 119057, 119058, 119059, 119060, 119061, 119062, 119063, 119064, 119065, 119066, 119067, 119068, 119069, 119070, 119071, 119072, 119073, 119074, 119075, 119076, 119077, 119078, 119081, 119082, 119083, 119084, 119085, 119086, 119087, 119088, 119089, 119090, 119091, 119092, 119093, 119094, 119095, 119096, 119097, 119098, 119099, 119100, 119101, 119102, 119103, 119104, 119105, 119106, 119107, 119108, 119109, 119110, 119111, 119112, 119113, 119114, 119115, 119116, 119117, 119118, 119119, 119120, 119121, 119122, 119123, 119124, 119125, 119126, 119127, 119128, 119129, 119130, 119131, 119132, 119133, 119134, 119135, 119136, 119137, 119138, 119139, 119140, 119146, 119147, 119148, 119171, 119172, 119180, 119181, 119182, 119183, 119184, 119185, 119186, 119187, 119188, 119189, 119190, 119191, 119192, 119193, 119194, 119195, 119196, 119197, 119198, 119199, 119200, 119201, 119202, 119203, 119204, 119205, 119206, 119207, 119208, 119209, 119214, 119215, 119216, 119217, 119218, 119219, 119220, 119221, 119222, 119223, 119224, 119225, 119226, 119227, 119228, 119229, 119230, 119231, 119232, 119233, 119234, 119235, 119236, 119237, 119238, 119239, 119240, 119241, 119242, 119243, 119244, 119245, 119246, 119247, 119248, 119249, 119250, 119251, 119252, 119253, 119254, 119255, 119256, 119257, 119258, 119259, 119260, 119261, 119296, 119297, 119298, 119299, 119300, 119301, 119302, 119303, 119304, 119305, 119306, 119307, 119308, 119309, 119310, 119311, 119312, 119313, 119314, 119315, 119316, 119317, 119318, 119319, 119320, 119321, 119322, 119323, 119324, 119325, 119326, 119327, 119328, 119329, 119330, 119331, 119332, 119333, 119334, 119335, 119336, 119337, 119338, 119339, 119340, 119341, 119342, 119343, 119344, 119345, 119346, 119347, 119348, 119349, 119350, 119351, 119352, 119353, 119354, 119355, 119356, 119357, 119358, 119359, 119360, 119361, 119365, 119552, 119553, 119554, 119555, 119556, 119557, 119558, 119559, 119560, 119561, 119562, 119563, 119564, 119565, 119566, 119567, 119568, 119569, 119570, 119571, 119572, 119573, 119574, 119575, 119576, 119577, 119578, 119579, 119580, 119581, 119582, 119583, 119584, 119585, 119586, 119587, 119588, 119589, 119590, 119591, 119592, 119593, 119594, 119595, 119596, 119597, 119598, 119599, 119600, 119601, 119602, 119603, 119604, 119605, 119606, 119607, 119608, 119609, 119610, 119611, 119612, 119613, 119614, 119615, 119616, 119617, 119618, 119619, 119620, 119621, 119622, 119623, 119624, 119625, 119626, 119627, 119628, 119629, 119630, 119631, 119632, 119633, 119634, 119635, 119636, 119637, 119638, 126976, 126977, 126978, 126979, 126980, 126981, 126982, 126983, 126984, 126985, 126986, 126987, 126988, 126989, 126990, 126991, 126992, 126993, 126994, 126995, 126996, 126997, 126998, 126999, 127000, 127001, 127002, 127003, 127004, 127005, 127006, 127007, 127008, 127009, 127010, 127011, 127012, 127013, 127014, 127015, 127016, 127017, 127018, 127019, 127024, 127025, 127026, 127027, 127028, 127029, 127030, 127031, 127032, 127033, 127034, 127035, 127036, 127037, 127038, 127039, 127040, 127041, 127042, 127043, 127044, 127045, 127046, 127047, 127048, 127049, 127050, 127051, 127052, 127053, 127054, 127055, 127056, 127057, 127058, 127059, 127060, 127061, 127062, 127063, 127064, 127065, 127066, 127067, 127068, 127069, 127070, 127071, 127072, 127073, 127074, 127075, 127076, 127077, 127078, 127079, 127080, 127081, 127082, 127083, 127084, 127085, 127086, 127087, 127088, 127089, 127090, 127091, 127092, 127093, 127094, 127095, 127096, 127097, 127098, 127099, 127100, 127101, 127102, 127103, 127104, 127105, 127106, 127107, 127108, 127109, 127110, 127111, 127112, 127113, 127114, 127115, 127116, 127117, 127118, 127119, 127120, 127121, 127122, 127123, 127136, 127137, 127138, 127139, 127140, 127141, 127142, 127143, 127144, 127145, 127146, 127147, 127148, 127149, 127150, 127153, 127154, 127155, 127156, 127157, 127158, 127159, 127160, 127161, 127162, 127163, 127164, 127165, 127166, 127169, 127170, 127171, 127172, 127173, 127174, 127175, 127176, 127177, 127178, 127179, 127180, 127181, 127182, 127183, 127185, 127186, 127187, 127188, 127189, 127190, 127191, 127192, 127193, 127194, 127195, 127196, 127197, 127198, 127199, 127248, 127249, 127250, 127251, 127252, 127253, 127254, 127255, 127256, 127257, 127258, 127259, 127260, 127261, 127262, 127263, 127264, 127265, 127266, 127267, 127268, 127269, 127270, 127271, 127272, 127273, 127274, 127275, 127276, 127277, 127278, 127280, 127281, 127282, 127283, 127284, 127285, 127286, 127287, 127288, 127289, 127290, 127291, 127292, 127293, 127294, 127295, 127296, 127297, 127298, 127299, 127300, 127301, 127302, 127303, 127304, 127305, 127306, 127307, 127308, 127309, 127310, 127311, 127312, 127313, 127314, 127315, 127316, 127317, 127318, 127319, 127320, 127321, 127322, 127323, 127324, 127325, 127326, 127327, 127328, 127329, 127330, 127331, 127332, 127333, 127334, 127335, 127336, 127337, 127344, 127345, 127346, 127347, 127348, 127349, 127350, 127351, 127352, 127353, 127354, 127355, 127356, 127357, 127358, 127359, 127360, 127361, 127362, 127363, 127364, 127365, 127366, 127367, 127368, 127369, 127370, 127371, 127372, 127373, 127374, 127375, 127376, 127377, 127378, 127379, 127380, 127381, 127382, 127383, 127384, 127385, 127386, 127462, 127463, 127464, 127465, 127466, 127467, 127468, 127469, 127470, 127471, 127472, 127473, 127474, 127475, 127476, 127477, 127478, 127479, 127480, 127481, 127482, 127483, 127484, 127485, 127486, 127487, 127488, 127489, 127490, 127504, 127505, 127506, 127507, 127508, 127509, 127510, 127511, 127512, 127513, 127514, 127515, 127516, 127517, 127518, 127519, 127520, 127521, 127522, 127523, 127524, 127525, 127526, 127527, 127528, 127529, 127530, 127531, 127532, 127533, 127534, 127535, 127536, 127537, 127538, 127539, 127540, 127541, 127542, 127543, 127544, 127545, 127546, 127552, 127553, 127554, 127555, 127556, 127557, 127558, 127559, 127560, 127568, 127569, 127744, 127745, 127746, 127747, 127748, 127749, 127750, 127751, 127752, 127753, 127754, 127755, 127756, 127757, 127758, 127759, 127760, 127761, 127762, 127763, 127764, 127765, 127766, 127767, 127768, 127769, 127770, 127771, 127772, 127773, 127774, 127775, 127776, 127792, 127793, 127794, 127795, 127796, 127797, 127799, 127800, 127801, 127802, 127803, 127804, 127805, 127806, 127807, 127808, 127809, 127810, 127811, 127812, 127813, 127814, 127815, 127816, 127817, 127818, 127819, 127820, 127821, 127822, 127823, 127824, 127825, 127826, 127827, 127828, 127829, 127830, 127831, 127832, 127833, 127834, 127835, 127836, 127837, 127838, 127839, 127840, 127841, 127842, 127843, 127844, 127845, 127846, 127847, 127848, 127849, 127850, 127851, 127852, 127853, 127854, 127855, 127856, 127857, 127858, 127859, 127860, 127861, 127862, 127863, 127864, 127865, 127866, 127867, 127868, 127872, 127873, 127874, 127875, 127876, 127877, 127878, 127879, 127880, 127881, 127882, 127883, 127884, 127885, 127886, 127887, 127888, 127889, 127890, 127891, 127904, 127905, 127906, 127907, 127908, 127909, 127910, 127911, 127912, 127913, 127914, 127915, 127916, 127917, 127918, 127919, 127920, 127921, 127922, 127923, 127924, 127925, 127926, 127927, 127928, 127929, 127930, 127931, 127932, 127933, 127934, 127935, 127936, 127937, 127938, 127939, 127940, 127942, 127943, 127944, 127945, 127946, 127968, 127969, 127970, 127971, 127972, 127973, 127974, 127975, 127976, 127977, 127978, 127979, 127980, 127981, 127982, 127983, 127984, 128000, 128001, 128002, 128003, 128004, 128005, 128006, 128007, 128008, 128009, 128010, 128011, 128012, 128013, 128014, 128015, 128016, 128017, 128018, 128019, 128020, 128021, 128022, 128023, 128024, 128025, 128026, 128027, 128028, 128029, 128030, 128031, 128032, 128033, 128034, 128035, 128036, 128037, 128038, 128039, 128040, 128041, 128042, 128043, 128044, 128045, 128046, 128047, 128048, 128049, 128050, 128051, 128052, 128053, 128054, 128055, 128056, 128057, 128058, 128059, 128060, 128061, 128062, 128064, 128066, 128067, 128068, 128069, 128070, 128071, 128072, 128073, 128074, 128075, 128076, 128077, 128078, 128079, 128080, 128081, 128082, 128083, 128084, 128085, 128086, 128087, 128088, 128089, 128090, 128091, 128092, 128093, 128094, 128095, 128096, 128097, 128098, 128099, 128100, 128101, 128102, 128103, 128104, 128105, 128106, 128107, 128108, 128109, 128110, 128111, 128112, 128113, 128114, 128115, 128116, 128117, 128118, 128119, 128120, 128121, 128122, 128123, 128124, 128125, 128126, 128127, 128128, 128129, 128130, 128131, 128132, 128133, 128134, 128135, 128136, 128137, 128138, 128139, 128140, 128141, 128142, 128143, 128144, 128145, 128146, 128147, 128148, 128149, 128150, 128151, 128152, 128153, 128154, 128155, 128156, 128157, 128158, 128159, 128160, 128161, 128162, 128163, 128164, 128165, 128166, 128167, 128168, 128169, 128170, 128171, 128172, 128173, 128174, 128175, 128176, 128177, 128178, 128179, 128180, 128181, 128182, 128183, 128184, 128185, 128186, 128187, 128188, 128189, 128190, 128191, 128192, 128193, 128194, 128195, 128196, 128197, 128198, 128199, 128200, 128201, 128202, 128203, 128204, 128205, 128206, 128207, 128208, 128209, 128210, 128211, 128212, 128213, 128214, 128215, 128216, 128217, 128218, 128219, 128220, 128221, 128222, 128223, 128224, 128225, 128226, 128227, 128228, 128229, 128230, 128231, 128232, 128233, 128234, 128235, 128236, 128237, 128238, 128239, 128240, 128241, 128242, 128243, 128244, 128245, 128246, 128247, 128249, 128250, 128251, 128252, 128256, 128257, 128258, 128259, 128260, 128261, 128262, 128263, 128264, 128265, 128266, 128267, 128268, 128269, 128270, 128271, 128272, 128273, 128274, 128275, 128276, 128277, 128278, 128279, 128280, 128281, 128282, 128283, 128284, 128285, 128286, 128287, 128288, 128289, 128290, 128291, 128292, 128293, 128294, 128295, 128296, 128297, 128298, 128299, 128300, 128301, 128302, 128303, 128304, 128305, 128306, 128307, 128308, 128309, 128310, 128311, 128312, 128313, 128314, 128315, 128316, 128317, 128336, 128337, 128338, 128339, 128340, 128341, 128342, 128343, 128344, 128345, 128346, 128347, 128348, 128349, 128350, 128351, 128352, 128353, 128354, 128355, 128356, 128357, 128358, 128359, 128507, 128508, 128509, 128510, 128511, 128513, 128514, 128515, 128516, 128517, 128518, 128519, 128520, 128521, 128522, 128523, 128524, 128525, 128526, 128527, 128528, 128530, 128531, 128532, 128534, 128536, 128538, 128540, 128541, 128542, 128544, 128545, 128546, 128547, 128548, 128549, 128552, 128553, 128554, 128555, 128557, 128560, 128561, 128562, 128563, 128565, 128566, 128567, 128568, 128569, 128570, 128571, 128572, 128573, 128574, 128575, 128576, 128581, 128582, 128583, 128584, 128585, 128586, 128587, 128588, 128589, 128590, 128591, 128640, 128641, 128642, 128643, 128644, 128645, 128646, 128647, 128648, 128649, 128650, 128651, 128652, 128653, 128654, 128655, 128656, 128657, 128658, 128659, 128660, 128661, 128662, 128663, 128664, 128665, 128666, 128667, 128668, 128669, 128670, 128671, 128672, 128673, 128674, 128675, 128676, 128677, 128678, 128679, 128680, 128681, 128682, 128683, 128684, 128685, 128686, 128687, 128688, 128689, 128690, 128691, 128692, 128693, 128694, 128695, 128696, 128697, 128698, 128699, 128700, 128701, 128702, 128703, 128704, 128705, 128706, 128707, 128708, 128709, 128768, 128769, 128770, 128771, 128772, 128773, 128774, 128775, 128776, 128777, 128778, 128779, 128780, 128781, 128782, 128783, 128784, 128785, 128786, 128787, 128788, 128789, 128790, 128791, 128792, 128793, 128794, 128795, 128796, 128797, 128798, 128799, 128800, 128801, 128802, 128803, 128804, 128805, 128806, 128807, 128808, 128809, 128810, 128811, 128812, 128813, 128814, 128815, 128816, 128817, 128818, 128819, 128820, 128821, 128822, 128823, 128824, 128825, 128826, 128827, 128828, 128829, 128830, 128831, 128832, 128833, 128834, 128835, 128836, 128837, 128838, 128839, 128840, 128841, 128842, 128843, 128844, 128845, 128846, 128847, 128848, 128849, 128850, 128851, 128852, 128853, 128854, 128855, 128856, 128857, 128858, 128859, 128860, 128861, 128862, 128863, 128864, 128865, 128866, 128867, 128868, 128869, 128870, 128871, 128872, 128873, 128874, 128875, 128876, 128877, 128878, 128879, 128880, 128881, 128882, 128883],
				'encoded': '\xa6\xa7\xa9\xae\xb0\xb6\u0482\u060e\u060f\u06de\u06e9\u06fd\u06fe\u07f6\u09fa\u0b70\u0bf3\u0bf4\u0bf5\u0bf6\u0bf7\u0bf8\u0bfa\u0c7f\u0d79\u0f01\u0f02\u0f03\u0f13\u0f14\u0f15\u0f16\u0f17\u0f1a\u0f1b\u0f1c\u0f1d\u0f1e\u0f1f\u0f34\u0f36\u0f38\u0fbe\u0fbf\u0fc0\u0fc1\u0fc2\u0fc3\u0fc4\u0fc5\u0fc7\u0fc8\u0fc9\u0fca\u0fcb\u0fcc\u0fce\u0fcf\u0fd5\u0fd6\u0fd7\u0fd8\u109e\u109f\u1360\u1390\u1391\u1392\u1393\u1394\u1395\u1396\u1397\u1398\u1399\u1940\u19de\u19df\u19e0\u19e1\u19e2\u19e3\u19e4\u19e5\u19e6\u19e7\u19e8\u19e9\u19ea\u19eb\u19ec\u19ed\u19ee\u19ef\u19f0\u19f1\u19f2\u19f3\u19f4\u19f5\u19f6\u19f7\u19f8\u19f9\u19fa\u19fb\u19fc\u19fd\u19fe\u19ff\u1b61\u1b62\u1b63\u1b64\u1b65\u1b66\u1b67\u1b68\u1b69\u1b6a\u1b74\u1b75\u1b76\u1b77\u1b78\u1b79\u1b7a\u1b7b\u1b7c\u2100\u2101\u2103\u2104\u2105\u2106\u2108\u2109\u2114\u2116\u2117\u211e\u211f\u2120\u2121\u2122\u2123\u2125\u2127\u2129\u212e\u213a\u213b\u214a\u214c\u214d\u214f\u2195\u2196\u2197\u2198\u2199\u219c\u219d\u219e\u219f\u21a1\u21a2\u21a4\u21a5\u21a7\u21a8\u21a9\u21aa\u21ab\u21ac\u21ad\u21af\u21b0\u21b1\u21b2\u21b3\u21b4\u21b5\u21b6\u21b7\u21b8\u21b9\u21ba\u21bb\u21bc\u21bd\u21be\u21bf\u21c0\u21c1\u21c2\u21c3\u21c4\u21c5\u21c6\u21c7\u21c8\u21c9\u21ca\u21cb\u21cc\u21cd\u21d0\u21d1\u21d3\u21d5\u21d6\u21d7\u21d8\u21d9\u21da\u21db\u21dc\u21dd\u21de\u21df\u21e0\u21e1\u21e2\u21e3\u21e4\u21e5\u21e6\u21e7\u21e8\u21e9\u21ea\u21eb\u21ec\u21ed\u21ee\u21ef\u21f0\u21f1\u21f2\u21f3\u2300\u2301\u2302\u2303\u2304\u2305\u2306\u2307\u230c\u230d\u230e\u230f\u2310\u2311\u2312\u2313\u2314\u2315\u2316\u2317\u2318\u2319\u231a\u231b\u231c\u231d\u231e\u231f\u2322\u2323\u2324\u2325\u2326\u2327\u2328\u232b\u232c\u232d\u232e\u232f\u2330\u2331\u2332\u2333\u2334\u2335\u2336\u2337\u2338\u2339\u233a\u233b\u233c\u233d\u233e\u233f\u2340\u2341\u2342\u2343\u2344\u2345\u2346\u2347\u2348\u2349\u234a\u234b\u234c\u234d\u234e\u234f\u2350\u2351\u2352\u2353\u2354\u2355\u2356\u2357\u2358\u2359\u235a\u235b\u235c\u235d\u235e\u235f\u2360\u2361\u2362\u2363\u2364\u2365\u2366\u2367\u2368\u2369\u236a\u236b\u236c\u236d\u236e\u236f\u2370\u2371\u2372\u2373\u2374\u2375\u2376\u2377\u2378\u2379\u237a\u237b\u237d\u237e\u237f\u2380\u2381\u2382\u2383\u2384\u2385\u2386\u2387\u2388\u2389\u238a\u238b\u238c\u238d\u238e\u238f\u2390\u2391\u2392\u2393\u2394\u2395\u2396\u2397\u2398\u2399\u239a\u23b4\u23b5\u23b6\u23b7\u23b8\u23b9\u23ba\u23bb\u23bc\u23bd\u23be\u23bf\u23c0\u23c1\u23c2\u23c3\u23c4\u23c5\u23c6\u23c7\u23c8\u23c9\u23ca\u23cb\u23cc\u23cd\u23ce\u23cf\u23d0\u23d1\u23d2\u23d3\u23d4\u23d5\u23d6\u23d7\u23d8\u23d9\u23da\u23db\u23e2\u23e3\u23e4\u23e5\u23e6\u23e7\u23e8\u23e9\u23ea\u23eb\u23ec\u23ed\u23ee\u23ef\u23f0\u23f1\u23f2\u23f3\u2400\u2401\u2402\u2403\u2404\u2405\u2406\u2407\u2408\u2409\u240a\u240b\u240c\u240d\u240e\u240f\u2410\u2411\u2412\u2413\u2414\u2415\u2416\u2417\u2418\u2419\u241a\u241b\u241c\u241d\u241e\u241f\u2420\u2421\u2422\u2423\u2424\u2425\u2426\u2440\u2441\u2442\u2443\u2444\u2445\u2446\u2447\u2448\u2449\u244a\u249c\u249d\u249e\u249f\u24a0\u24a1\u24a2\u24a3\u24a4\u24a5\u24a6\u24a7\u24a8\u24a9\u24aa\u24ab\u24ac\u24ad\u24ae\u24af\u24b0\u24b1\u24b2\u24b3\u24b4\u24b5\u24b6\u24b7\u24b8\u24b9\u24ba\u24bb\u24bc\u24bd\u24be\u24bf\u24c0\u24c1\u24c2\u24c3\u24c4\u24c5\u24c6\u24c7\u24c8\u24c9\u24ca\u24cb\u24cc\u24cd\u24ce\u24cf\u24d0\u24d1\u24d2\u24d3\u24d4\u24d5\u24d6\u24d7\u24d8\u24d9\u24da\u24db\u24dc\u24dd\u24de\u24df\u24e0\u24e1\u24e2\u24e3\u24e4\u24e5\u24e6\u24e7\u24e8\u24e9\u2500\u2501\u2502\u2503\u2504\u2505\u2506\u2507\u2508\u2509\u250a\u250b\u250c\u250d\u250e\u250f\u2510\u2511\u2512\u2513\u2514\u2515\u2516\u2517\u2518\u2519\u251a\u251b\u251c\u251d\u251e\u251f\u2520\u2521\u2522\u2523\u2524\u2525\u2526\u2527\u2528\u2529\u252a\u252b\u252c\u252d\u252e\u252f\u2530\u2531\u2532\u2533\u2534\u2535\u2536\u2537\u2538\u2539\u253a\u253b\u253c\u253d\u253e\u253f\u2540\u2541\u2542\u2543\u2544\u2545\u2546\u2547\u2548\u2549\u254a\u254b\u254c\u254d\u254e\u254f\u2550\u2551\u2552\u2553\u2554\u2555\u2556\u2557\u2558\u2559\u255a\u255b\u255c\u255d\u255e\u255f\u2560\u2561\u2562\u2563\u2564\u2565\u2566\u2567\u2568\u2569\u256a\u256b\u256c\u256d\u256e\u256f\u2570\u2571\u2572\u2573\u2574\u2575\u2576\u2577\u2578\u2579\u257a\u257b\u257c\u257d\u257e\u257f\u2580\u2581\u2582\u2583\u2584\u2585\u2586\u2587\u2588\u2589\u258a\u258b\u258c\u258d\u258e\u258f\u2590\u2591\u2592\u2593\u2594\u2595\u2596\u2597\u2598\u2599\u259a\u259b\u259c\u259d\u259e\u259f\u25a0\u25a1\u25a2\u25a3\u25a4\u25a5\u25a6\u25a7\u25a8\u25a9\u25aa\u25ab\u25ac\u25ad\u25ae\u25af\u25b0\u25b1\u25b2\u25b3\u25b4\u25b5\u25b6\u25b8\u25b9\u25ba\u25bb\u25bc\u25bd\u25be\u25bf\u25c0\u25c2\u25c3\u25c4\u25c5\u25c6\u25c7\u25c8\u25c9\u25ca\u25cb\u25cc\u25cd\u25ce\u25cf\u25d0\u25d1\u25d2\u25d3\u25d4\u25d5\u25d6\u25d7\u25d8\u25d9\u25da\u25db\u25dc\u25dd\u25de\u25df\u25e0\u25e1\u25e2\u25e3\u25e4\u25e5\u25e6\u25e7\u25e8\u25e9\u25ea\u25eb\u25ec\u25ed\u25ee\u25ef\u25f0\u25f1\u25f2\u25f3\u25f4\u25f5\u25f6\u25f7\u2600\u2601\u2602\u2603\u2604\u2605\u2606\u2607\u2608\u2609\u260a\u260b\u260c\u260d\u260e\u260f\u2610\u2611\u2612\u2613\u2614\u2615\u2616\u2617\u2618\u2619\u261a\u261b\u261c\u261d\u261e\u261f\u2620\u2621\u2622\u2623\u2624\u2625\u2626\u2627\u2628\u2629\u262a\u262b\u262c\u262d\u262e\u262f\u2630\u2631\u2632\u2633\u2634\u2635\u2636\u2637\u2638\u2639\u263a\u263b\u263c\u263d\u263e\u263f\u2640\u2641\u2642\u2643\u2644\u2645\u2646\u2647\u2648\u2649\u264a\u264b\u264c\u264d\u264e\u264f\u2650\u2651\u2652\u2653\u2654\u2655\u2656\u2657\u2658\u2659\u265a\u265b\u265c\u265d\u265e\u265f\u2660\u2661\u2662\u2663\u2664\u2665\u2666\u2667\u2668\u2669\u266a\u266b\u266c\u266d\u266e\u2670\u2671\u2672\u2673\u2674\u2675\u2676\u2677\u2678\u2679\u267a\u267b\u267c\u267d\u267e\u267f\u2680\u2681\u2682\u2683\u2684\u2685\u2686\u2687\u2688\u2689\u268a\u268b\u268c\u268d\u268e\u268f\u2690\u2691\u2692\u2693\u2694\u2695\u2696\u2697\u2698\u2699\u269a\u269b\u269c\u269d\u269e\u269f\u26a0\u26a1\u26a2\u26a3\u26a4\u26a5\u26a6\u26a7\u26a8\u26a9\u26aa\u26ab\u26ac\u26ad\u26ae\u26af\u26b0\u26b1\u26b2\u26b3\u26b4\u26b5\u26b6\u26b7\u26b8\u26b9\u26ba\u26bb\u26bc\u26bd\u26be\u26bf\u26c0\u26c1\u26c2\u26c3\u26c4\u26c5\u26c6\u26c7\u26c8\u26c9\u26ca\u26cb\u26cc\u26cd\u26ce\u26cf\u26d0\u26d1\u26d2\u26d3\u26d4\u26d5\u26d6\u26d7\u26d8\u26d9\u26da\u26db\u26dc\u26dd\u26de\u26df\u26e0\u26e1\u26e2\u26e3\u26e4\u26e5\u26e6\u26e7\u26e8\u26e9\u26ea\u26eb\u26ec\u26ed\u26ee\u26ef\u26f0\u26f1\u26f2\u26f3\u26f4\u26f5\u26f6\u26f7\u26f8\u26f9\u26fa\u26fb\u26fc\u26fd\u26fe\u26ff\u2701\u2702\u2703\u2704\u2705\u2706\u2707\u2708\u2709\u270a\u270b\u270c\u270d\u270e\u270f\u2710\u2711\u2712\u2713\u2714\u2715\u2716\u2717\u2718\u2719\u271a\u271b\u271c\u271d\u271e\u271f\u2720\u2721\u2722\u2723\u2724\u2725\u2726\u2727\u2728\u2729\u272a\u272b\u272c\u272d\u272e\u272f\u2730\u2731\u2732\u2733\u2734\u2735\u2736\u2737\u2738\u2739\u273a\u273b\u273c\u273d\u273e\u273f\u2740\u2741\u2742\u2743\u2744\u2745\u2746\u2747\u2748\u2749\u274a\u274b\u274c\u274d\u274e\u274f\u2750\u2751\u2752\u2753\u2754\u2755\u2756\u2757\u2758\u2759\u275a\u275b\u275c\u275d\u275e\u275f\u2760\u2761\u2762\u2763\u2764\u2765\u2766\u2767\u2794\u2795\u2796\u2797\u2798\u2799\u279a\u279b\u279c\u279d\u279e\u279f\u27a0\u27a1\u27a2\u27a3\u27a4\u27a5\u27a6\u27a7\u27a8\u27a9\u27aa\u27ab\u27ac\u27ad\u27ae\u27af\u27b0\u27b1\u27b2\u27b3\u27b4\u27b5\u27b6\u27b7\u27b8\u27b9\u27ba\u27bb\u27bc\u27bd\u27be\u27bf\u2800\u2801\u2802\u2803\u2804\u2805\u2806\u2807\u2808\u2809\u280a\u280b\u280c\u280d\u280e\u280f\u2810\u2811\u2812\u2813\u2814\u2815\u2816\u2817\u2818\u2819\u281a\u281b\u281c\u281d\u281e\u281f\u2820\u2821\u2822\u2823\u2824\u2825\u2826\u2827\u2828\u2829\u282a\u282b\u282c\u282d\u282e\u282f\u2830\u2831\u2832\u2833\u2834\u2835\u2836\u2837\u2838\u2839\u283a\u283b\u283c\u283d\u283e\u283f\u2840\u2841\u2842\u2843\u2844\u2845\u2846\u2847\u2848\u2849\u284a\u284b\u284c\u284d\u284e\u284f\u2850\u2851\u2852\u2853\u2854\u2855\u2856\u2857\u2858\u2859\u285a\u285b\u285c\u285d\u285e\u285f\u2860\u2861\u2862\u2863\u2864\u2865\u2866\u2867\u2868\u2869\u286a\u286b\u286c\u286d\u286e\u286f\u2870\u2871\u2872\u2873\u2874\u2875\u2876\u2877\u2878\u2879\u287a\u287b\u287c\u287d\u287e\u287f\u2880\u2881\u2882\u2883\u2884\u2885\u2886\u2887\u2888\u2889\u288a\u288b\u288c\u288d\u288e\u288f\u2890\u2891\u2892\u2893\u2894\u2895\u2896\u2897\u2898\u2899\u289a\u289b\u289c\u289d\u289e\u289f\u28a0\u28a1\u28a2\u28a3\u28a4\u28a5\u28a6\u28a7\u28a8\u28a9\u28aa\u28ab\u28ac\u28ad\u28ae\u28af\u28b0\u28b1\u28b2\u28b3\u28b4\u28b5\u28b6\u28b7\u28b8\u28b9\u28ba\u28bb\u28bc\u28bd\u28be\u28bf\u28c0\u28c1\u28c2\u28c3\u28c4\u28c5\u28c6\u28c7\u28c8\u28c9\u28ca\u28cb\u28cc\u28cd\u28ce\u28cf\u28d0\u28d1\u28d2\u28d3\u28d4\u28d5\u28d6\u28d7\u28d8\u28d9\u28da\u28db\u28dc\u28dd\u28de\u28df\u28e0\u28e1\u28e2\u28e3\u28e4\u28e5\u28e6\u28e7\u28e8\u28e9\u28ea\u28eb\u28ec\u28ed\u28ee\u28ef\u28f0\u28f1\u28f2\u28f3\u28f4\u28f5\u28f6\u28f7\u28f8\u28f9\u28fa\u28fb\u28fc\u28fd\u28fe\u28ff\u2b00\u2b01\u2b02\u2b03\u2b04\u2b05\u2b06\u2b07\u2b08\u2b09\u2b0a\u2b0b\u2b0c\u2b0d\u2b0e\u2b0f\u2b10\u2b11\u2b12\u2b13\u2b14\u2b15\u2b16\u2b17\u2b18\u2b19\u2b1a\u2b1b\u2b1c\u2b1d\u2b1e\u2b1f\u2b20\u2b21\u2b22\u2b23\u2b24\u2b25\u2b26\u2b27\u2b28\u2b29\u2b2a\u2b2b\u2b2c\u2b2d\u2b2e\u2b2f\u2b45\u2b46\u2b50\u2b51\u2b52\u2b53\u2b54\u2b55\u2b56\u2b57\u2b58\u2b59\u2ce5\u2ce6\u2ce7\u2ce8\u2ce9\u2cea\u2e80\u2e81\u2e82\u2e83\u2e84\u2e85\u2e86\u2e87\u2e88\u2e89\u2e8a\u2e8b\u2e8c\u2e8d\u2e8e\u2e8f\u2e90\u2e91\u2e92\u2e93\u2e94\u2e95\u2e96\u2e97\u2e98\u2e99\u2e9b\u2e9c\u2e9d\u2e9e\u2e9f\u2ea0\u2ea1\u2ea2\u2ea3\u2ea4\u2ea5\u2ea6\u2ea7\u2ea8\u2ea9\u2eaa\u2eab\u2eac\u2ead\u2eae\u2eaf\u2eb0\u2eb1\u2eb2\u2eb3\u2eb4\u2eb5\u2eb6\u2eb7\u2eb8\u2eb9\u2eba\u2ebb\u2ebc\u2ebd\u2ebe\u2ebf\u2ec0\u2ec1\u2ec2\u2ec3\u2ec4\u2ec5\u2ec6\u2ec7\u2ec8\u2ec9\u2eca\u2ecb\u2ecc\u2ecd\u2ece\u2ecf\u2ed0\u2ed1\u2ed2\u2ed3\u2ed4\u2ed5\u2ed6\u2ed7\u2ed8\u2ed9\u2eda\u2edb\u2edc\u2edd\u2ede\u2edf\u2ee0\u2ee1\u2ee2\u2ee3\u2ee4\u2ee5\u2ee6\u2ee7\u2ee8\u2ee9\u2eea\u2eeb\u2eec\u2eed\u2eee\u2eef\u2ef0\u2ef1\u2ef2\u2ef3\u2f00\u2f01\u2f02\u2f03\u2f04\u2f05\u2f06\u2f07\u2f08\u2f09\u2f0a\u2f0b\u2f0c\u2f0d\u2f0e\u2f0f\u2f10\u2f11\u2f12\u2f13\u2f14\u2f15\u2f16\u2f17\u2f18\u2f19\u2f1a\u2f1b\u2f1c\u2f1d\u2f1e\u2f1f\u2f20\u2f21\u2f22\u2f23\u2f24\u2f25\u2f26\u2f27\u2f28\u2f29\u2f2a\u2f2b\u2f2c\u2f2d\u2f2e\u2f2f\u2f30\u2f31\u2f32\u2f33\u2f34\u2f35\u2f36\u2f37\u2f38\u2f39\u2f3a\u2f3b\u2f3c\u2f3d\u2f3e\u2f3f\u2f40\u2f41\u2f42\u2f43\u2f44\u2f45\u2f46\u2f47\u2f48\u2f49\u2f4a\u2f4b\u2f4c\u2f4d\u2f4e\u2f4f\u2f50\u2f51\u2f52\u2f53\u2f54\u2f55\u2f56\u2f57\u2f58\u2f59\u2f5a\u2f5b\u2f5c\u2f5d\u2f5e\u2f5f\u2f60\u2f61\u2f62\u2f63\u2f64\u2f65\u2f66\u2f67\u2f68\u2f69\u2f6a\u2f6b\u2f6c\u2f6d\u2f6e\u2f6f\u2f70\u2f71\u2f72\u2f73\u2f74\u2f75\u2f76\u2f77\u2f78\u2f79\u2f7a\u2f7b\u2f7c\u2f7d\u2f7e\u2f7f\u2f80\u2f81\u2f82\u2f83\u2f84\u2f85\u2f86\u2f87\u2f88\u2f89\u2f8a\u2f8b\u2f8c\u2f8d\u2f8e\u2f8f\u2f90\u2f91\u2f92\u2f93\u2f94\u2f95\u2f96\u2f97\u2f98\u2f99\u2f9a\u2f9b\u2f9c\u2f9d\u2f9e\u2f9f\u2fa0\u2fa1\u2fa2\u2fa3\u2fa4\u2fa5\u2fa6\u2fa7\u2fa8\u2fa9\u2faa\u2fab\u2fac\u2fad\u2fae\u2faf\u2fb0\u2fb1\u2fb2\u2fb3\u2fb4\u2fb5\u2fb6\u2fb7\u2fb8\u2fb9\u2fba\u2fbb\u2fbc\u2fbd\u2fbe\u2fbf\u2fc0\u2fc1\u2fc2\u2fc3\u2fc4\u2fc5\u2fc6\u2fc7\u2fc8\u2fc9\u2fca\u2fcb\u2fcc\u2fcd\u2fce\u2fcf\u2fd0\u2fd1\u2fd2\u2fd3\u2fd4\u2fd5\u2ff0\u2ff1\u2ff2\u2ff3\u2ff4\u2ff5\u2ff6\u2ff7\u2ff8\u2ff9\u2ffa\u2ffb\u3004\u3012\u3013\u3020\u3036\u3037\u303e\u303f\u3190\u3191\u3196\u3197\u3198\u3199\u319a\u319b\u319c\u319d\u319e\u319f\u31c0\u31c1\u31c2\u31c3\u31c4\u31c5\u31c6\u31c7\u31c8\u31c9\u31ca\u31cb\u31cc\u31cd\u31ce\u31cf\u31d0\u31d1\u31d2\u31d3\u31d4\u31d5\u31d6\u31d7\u31d8\u31d9\u31da\u31db\u31dc\u31dd\u31de\u31df\u31e0\u31e1\u31e2\u31e3\u3200\u3201\u3202\u3203\u3204\u3205\u3206\u3207\u3208\u3209\u320a\u320b\u320c\u320d\u320e\u320f\u3210\u3211\u3212\u3213\u3214\u3215\u3216\u3217\u3218\u3219\u321a\u321b\u321c\u321d\u321e\u322a\u322b\u322c\u322d\u322e\u322f\u3230\u3231\u3232\u3233\u3234\u3235\u3236\u3237\u3238\u3239\u323a\u323b\u323c\u323d\u323e\u323f\u3240\u3241\u3242\u3243\u3244\u3245\u3246\u3247\u3248\u3249\u324a\u324b\u324c\u324d\u324e\u324f\u3250\u3260\u3261\u3262\u3263\u3264\u3265\u3266\u3267\u3268\u3269\u326a\u326b\u326c\u326d\u326e\u326f\u3270\u3271\u3272\u3273\u3274\u3275\u3276\u3277\u3278\u3279\u327a\u327b\u327c\u327d\u327e\u327f\u328a\u328b\u328c\u328d\u328e\u328f\u3290\u3291\u3292\u3293\u3294\u3295\u3296\u3297\u3298\u3299\u329a\u329b\u329c\u329d\u329e\u329f\u32a0\u32a1\u32a2\u32a3\u32a4\u32a5\u32a6\u32a7\u32a8\u32a9\u32aa\u32ab\u32ac\u32ad\u32ae\u32af\u32b0\u32c0\u32c1\u32c2\u32c3\u32c4\u32c5\u32c6\u32c7\u32c8\u32c9\u32ca\u32cb\u32cc\u32cd\u32ce\u32cf\u32d0\u32d1\u32d2\u32d3\u32d4\u32d5\u32d6\u32d7\u32d8\u32d9\u32da\u32db\u32dc\u32dd\u32de\u32df\u32e0\u32e1\u32e2\u32e3\u32e4\u32e5\u32e6\u32e7\u32e8\u32e9\u32ea\u32eb\u32ec\u32ed\u32ee\u32ef\u32f0\u32f1\u32f2\u32f3\u32f4\u32f5\u32f6\u32f7\u32f8\u32f9\u32fa\u32fb\u32fc\u32fd\u32fe\u3300\u3301\u3302\u3303\u3304\u3305\u3306\u3307\u3308\u3309\u330a\u330b\u330c\u330d\u330e\u330f\u3310\u3311\u3312\u3313\u3314\u3315\u3316\u3317\u3318\u3319\u331a\u331b\u331c\u331d\u331e\u331f\u3320\u3321\u3322\u3323\u3324\u3325\u3326\u3327\u3328\u3329\u332a\u332b\u332c\u332d\u332e\u332f\u3330\u3331\u3332\u3333\u3334\u3335\u3336\u3337\u3338\u3339\u333a\u333b\u333c\u333d\u333e\u333f\u3340\u3341\u3342\u3343\u3344\u3345\u3346\u3347\u3348\u3349\u334a\u334b\u334c\u334d\u334e\u334f\u3350\u3351\u3352\u3353\u3354\u3355\u3356\u3357\u3358\u3359\u335a\u335b\u335c\u335d\u335e\u335f\u3360\u3361\u3362\u3363\u3364\u3365\u3366\u3367\u3368\u3369\u336a\u336b\u336c\u336d\u336e\u336f\u3370\u3371\u3372\u3373\u3374\u3375\u3376\u3377\u3378\u3379\u337a\u337b\u337c\u337d\u337e\u337f\u3380\u3381\u3382\u3383\u3384\u3385\u3386\u3387\u3388\u3389\u338a\u338b\u338c\u338d\u338e\u338f\u3390\u3391\u3392\u3393\u3394\u3395\u3396\u3397\u3398\u3399\u339a\u339b\u339c\u339d\u339e\u339f\u33a0\u33a1\u33a2\u33a3\u33a4\u33a5\u33a6\u33a7\u33a8\u33a9\u33aa\u33ab\u33ac\u33ad\u33ae\u33af\u33b0\u33b1\u33b2\u33b3\u33b4\u33b5\u33b6\u33b7\u33b8\u33b9\u33ba\u33bb\u33bc\u33bd\u33be\u33bf\u33c0\u33c1\u33c2\u33c3\u33c4\u33c5\u33c6\u33c7\u33c8\u33c9\u33ca\u33cb\u33cc\u33cd\u33ce\u33cf\u33d0\u33d1\u33d2\u33d3\u33d4\u33d5\u33d6\u33d7\u33d8\u33d9\u33da\u33db\u33dc\u33dd\u33de\u33df\u33e0\u33e1\u33e2\u33e3\u33e4\u33e5\u33e6\u33e7\u33e8\u33e9\u33ea\u33eb\u33ec\u33ed\u33ee\u33ef\u33f0\u33f1\u33f2\u33f3\u33f4\u33f5\u33f6\u33f7\u33f8\u33f9\u33fa\u33fb\u33fc\u33fd\u33fe\u33ff\u4dc0\u4dc1\u4dc2\u4dc3\u4dc4\u4dc5\u4dc6\u4dc7\u4dc8\u4dc9\u4dca\u4dcb\u4dcc\u4dcd\u4dce\u4dcf\u4dd0\u4dd1\u4dd2\u4dd3\u4dd4\u4dd5\u4dd6\u4dd7\u4dd8\u4dd9\u4dda\u4ddb\u4ddc\u4ddd\u4dde\u4ddf\u4de0\u4de1\u4de2\u4de3\u4de4\u4de5\u4de6\u4de7\u4de8\u4de9\u4dea\u4deb\u4dec\u4ded\u4dee\u4def\u4df0\u4df1\u4df2\u4df3\u4df4\u4df5\u4df6\u4df7\u4df8\u4df9\u4dfa\u4dfb\u4dfc\u4dfd\u4dfe\u4dff\ua490\ua491\ua492\ua493\ua494\ua495\ua496\ua497\ua498\ua499\ua49a\ua49b\ua49c\ua49d\ua49e\ua49f\ua4a0\ua4a1\ua4a2\ua4a3\ua4a4\ua4a5\ua4a6\ua4a7\ua4a8\ua4a9\ua4aa\ua4ab\ua4ac\ua4ad\ua4ae\ua4af\ua4b0\ua4b1\ua4b2\ua4b3\ua4b4\ua4b5\ua4b6\ua4b7\ua4b8\ua4b9\ua4ba\ua4bb\ua4bc\ua4bd\ua4be\ua4bf\ua4c0\ua4c1\ua4c2\ua4c3\ua4c4\ua4c5\ua4c6\ua828\ua829\ua82a\ua82b\ua836\ua837\ua839\uaa77\uaa78\uaa79\ufdfd\uffe4\uffe8\uffed\uffee\ufffc\ufffd\ud800\udd02\ud800\udd37\ud800\udd38\ud800\udd39\ud800\udd3a\ud800\udd3b\ud800\udd3c\ud800\udd3d\ud800\udd3e\ud800\udd3f\ud800\udd79\ud800\udd7a\ud800\udd7b\ud800\udd7c\ud800\udd7d\ud800\udd7e\ud800\udd7f\ud800\udd80\ud800\udd81\ud800\udd82\ud800\udd83\ud800\udd84\ud800\udd85\ud800\udd86\ud800\udd87\ud800\udd88\ud800\udd89\ud800\udd90\ud800\udd91\ud800\udd92\ud800\udd93\ud800\udd94\ud800\udd95\ud800\udd96\ud800\udd97\ud800\udd98\ud800\udd99\ud800\udd9a\ud800\udd9b\ud800\uddd0\ud800\uddd1\ud800\uddd2\ud800\uddd3\ud800\uddd4\ud800\uddd5\ud800\uddd6\ud800\uddd7\ud800\uddd8\ud800\uddd9\ud800\uddda\ud800\udddb\ud800\udddc\ud800\udddd\ud800\uddde\ud800\udddf\ud800\udde0\ud800\udde1\ud800\udde2\ud800\udde3\ud800\udde4\ud800\udde5\ud800\udde6\ud800\udde7\ud800\udde8\ud800\udde9\ud800\uddea\ud800\uddeb\ud800\uddec\ud800\udded\ud800\uddee\ud800\uddef\ud800\uddf0\ud800\uddf1\ud800\uddf2\ud800\uddf3\ud800\uddf4\ud800\uddf5\ud800\uddf6\ud800\uddf7\ud800\uddf8\ud800\uddf9\ud800\uddfa\ud800\uddfb\ud800\uddfc\ud834\udc00\ud834\udc01\ud834\udc02\ud834\udc03\ud834\udc04\ud834\udc05\ud834\udc06\ud834\udc07\ud834\udc08\ud834\udc09\ud834\udc0a\ud834\udc0b\ud834\udc0c\ud834\udc0d\ud834\udc0e\ud834\udc0f\ud834\udc10\ud834\udc11\ud834\udc12\ud834\udc13\ud834\udc14\ud834\udc15\ud834\udc16\ud834\udc17\ud834\udc18\ud834\udc19\ud834\udc1a\ud834\udc1b\ud834\udc1c\ud834\udc1d\ud834\udc1e\ud834\udc1f\ud834\udc20\ud834\udc21\ud834\udc22\ud834\udc23\ud834\udc24\ud834\udc25\ud834\udc26\ud834\udc27\ud834\udc28\ud834\udc29\ud834\udc2a\ud834\udc2b\ud834\udc2c\ud834\udc2d\ud834\udc2e\ud834\udc2f\ud834\udc30\ud834\udc31\ud834\udc32\ud834\udc33\ud834\udc34\ud834\udc35\ud834\udc36\ud834\udc37\ud834\udc38\ud834\udc39\ud834\udc3a\ud834\udc3b\ud834\udc3c\ud834\udc3d\ud834\udc3e\ud834\udc3f\ud834\udc40\ud834\udc41\ud834\udc42\ud834\udc43\ud834\udc44\ud834\udc45\ud834\udc46\ud834\udc47\ud834\udc48\ud834\udc49\ud834\udc4a\ud834\udc4b\ud834\udc4c\ud834\udc4d\ud834\udc4e\ud834\udc4f\ud834\udc50\ud834\udc51\ud834\udc52\ud834\udc53\ud834\udc54\ud834\udc55\ud834\udc56\ud834\udc57\ud834\udc58\ud834\udc59\ud834\udc5a\ud834\udc5b\ud834\udc5c\ud834\udc5d\ud834\udc5e\ud834\udc5f\ud834\udc60\ud834\udc61\ud834\udc62\ud834\udc63\ud834\udc64\ud834\udc65\ud834\udc66\ud834\udc67\ud834\udc68\ud834\udc69\ud834\udc6a\ud834\udc6b\ud834\udc6c\ud834\udc6d\ud834\udc6e\ud834\udc6f\ud834\udc70\ud834\udc71\ud834\udc72\ud834\udc73\ud834\udc74\ud834\udc75\ud834\udc76\ud834\udc77\ud834\udc78\ud834\udc79\ud834\udc7a\ud834\udc7b\ud834\udc7c\ud834\udc7d\ud834\udc7e\ud834\udc7f\ud834\udc80\ud834\udc81\ud834\udc82\ud834\udc83\ud834\udc84\ud834\udc85\ud834\udc86\ud834\udc87\ud834\udc88\ud834\udc89\ud834\udc8a\ud834\udc8b\ud834\udc8c\ud834\udc8d\ud834\udc8e\ud834\udc8f\ud834\udc90\ud834\udc91\ud834\udc92\ud834\udc93\ud834\udc94\ud834\udc95\ud834\udc96\ud834\udc97\ud834\udc98\ud834\udc99\ud834\udc9a\ud834\udc9b\ud834\udc9c\ud834\udc9d\ud834\udc9e\ud834\udc9f\ud834\udca0\ud834\udca1\ud834\udca2\ud834\udca3\ud834\udca4\ud834\udca5\ud834\udca6\ud834\udca7\ud834\udca8\ud834\udca9\ud834\udcaa\ud834\udcab\ud834\udcac\ud834\udcad\ud834\udcae\ud834\udcaf\ud834\udcb0\ud834\udcb1\ud834\udcb2\ud834\udcb3\ud834\udcb4\ud834\udcb5\ud834\udcb6\ud834\udcb7\ud834\udcb8\ud834\udcb9\ud834\udcba\ud834\udcbb\ud834\udcbc\ud834\udcbd\ud834\udcbe\ud834\udcbf\ud834\udcc0\ud834\udcc1\ud834\udcc2\ud834\udcc3\ud834\udcc4\ud834\udcc5\ud834\udcc6\ud834\udcc7\ud834\udcc8\ud834\udcc9\ud834\udcca\ud834\udccb\ud834\udccc\ud834\udccd\ud834\udcce\ud834\udccf\ud834\udcd0\ud834\udcd1\ud834\udcd2\ud834\udcd3\ud834\udcd4\ud834\udcd5\ud834\udcd6\ud834\udcd7\ud834\udcd8\ud834\udcd9\ud834\udcda\ud834\udcdb\ud834\udcdc\ud834\udcdd\ud834\udcde\ud834\udcdf\ud834\udce0\ud834\udce1\ud834\udce2\ud834\udce3\ud834\udce4\ud834\udce5\ud834\udce6\ud834\udce7\ud834\udce8\ud834\udce9\ud834\udcea\ud834\udceb\ud834\udcec\ud834\udced\ud834\udcee\ud834\udcef\ud834\udcf0\ud834\udcf1\ud834\udcf2\ud834\udcf3\ud834\udcf4\ud834\udcf5\ud834\udd00\ud834\udd01\ud834\udd02\ud834\udd03\ud834\udd04\ud834\udd05\ud834\udd06\ud834\udd07\ud834\udd08\ud834\udd09\ud834\udd0a\ud834\udd0b\ud834\udd0c\ud834\udd0d\ud834\udd0e\ud834\udd0f\ud834\udd10\ud834\udd11\ud834\udd12\ud834\udd13\ud834\udd14\ud834\udd15\ud834\udd16\ud834\udd17\ud834\udd18\ud834\udd19\ud834\udd1a\ud834\udd1b\ud834\udd1c\ud834\udd1d\ud834\udd1e\ud834\udd1f\ud834\udd20\ud834\udd21\ud834\udd22\ud834\udd23\ud834\udd24\ud834\udd25\ud834\udd26\ud834\udd29\ud834\udd2a\ud834\udd2b\ud834\udd2c\ud834\udd2d\ud834\udd2e\ud834\udd2f\ud834\udd30\ud834\udd31\ud834\udd32\ud834\udd33\ud834\udd34\ud834\udd35\ud834\udd36\ud834\udd37\ud834\udd38\ud834\udd39\ud834\udd3a\ud834\udd3b\ud834\udd3c\ud834\udd3d\ud834\udd3e\ud834\udd3f\ud834\udd40\ud834\udd41\ud834\udd42\ud834\udd43\ud834\udd44\ud834\udd45\ud834\udd46\ud834\udd47\ud834\udd48\ud834\udd49\ud834\udd4a\ud834\udd4b\ud834\udd4c\ud834\udd4d\ud834\udd4e\ud834\udd4f\ud834\udd50\ud834\udd51\ud834\udd52\ud834\udd53\ud834\udd54\ud834\udd55\ud834\udd56\ud834\udd57\ud834\udd58\ud834\udd59\ud834\udd5a\ud834\udd5b\ud834\udd5c\ud834\udd5d\ud834\udd5e\ud834\udd5f\ud834\udd60\ud834\udd61\ud834\udd62\ud834\udd63\ud834\udd64\ud834\udd6a\ud834\udd6b\ud834\udd6c\ud834\udd83\ud834\udd84\ud834\udd8c\ud834\udd8d\ud834\udd8e\ud834\udd8f\ud834\udd90\ud834\udd91\ud834\udd92\ud834\udd93\ud834\udd94\ud834\udd95\ud834\udd96\ud834\udd97\ud834\udd98\ud834\udd99\ud834\udd9a\ud834\udd9b\ud834\udd9c\ud834\udd9d\ud834\udd9e\ud834\udd9f\ud834\udda0\ud834\udda1\ud834\udda2\ud834\udda3\ud834\udda4\ud834\udda5\ud834\udda6\ud834\udda7\ud834\udda8\ud834\udda9\ud834\uddae\ud834\uddaf\ud834\uddb0\ud834\uddb1\ud834\uddb2\ud834\uddb3\ud834\uddb4\ud834\uddb5\ud834\uddb6\ud834\uddb7\ud834\uddb8\ud834\uddb9\ud834\uddba\ud834\uddbb\ud834\uddbc\ud834\uddbd\ud834\uddbe\ud834\uddbf\ud834\uddc0\ud834\uddc1\ud834\uddc2\ud834\uddc3\ud834\uddc4\ud834\uddc5\ud834\uddc6\ud834\uddc7\ud834\uddc8\ud834\uddc9\ud834\uddca\ud834\uddcb\ud834\uddcc\ud834\uddcd\ud834\uddce\ud834\uddcf\ud834\uddd0\ud834\uddd1\ud834\uddd2\ud834\uddd3\ud834\uddd4\ud834\uddd5\ud834\uddd6\ud834\uddd7\ud834\uddd8\ud834\uddd9\ud834\uddda\ud834\udddb\ud834\udddc\ud834\udddd\ud834\ude00\ud834\ude01\ud834\ude02\ud834\ude03\ud834\ude04\ud834\ude05\ud834\ude06\ud834\ude07\ud834\ude08\ud834\ude09\ud834\ude0a\ud834\ude0b\ud834\ude0c\ud834\ude0d\ud834\ude0e\ud834\ude0f\ud834\ude10\ud834\ude11\ud834\ude12\ud834\ude13\ud834\ude14\ud834\ude15\ud834\ude16\ud834\ude17\ud834\ude18\ud834\ude19\ud834\ude1a\ud834\ude1b\ud834\ude1c\ud834\ude1d\ud834\ude1e\ud834\ude1f\ud834\ude20\ud834\ude21\ud834\ude22\ud834\ude23\ud834\ude24\ud834\ude25\ud834\ude26\ud834\ude27\ud834\ude28\ud834\ude29\ud834\ude2a\ud834\ude2b\ud834\ude2c\ud834\ude2d\ud834\ude2e\ud834\ude2f\ud834\ude30\ud834\ude31\ud834\ude32\ud834\ude33\ud834\ude34\ud834\ude35\ud834\ude36\ud834\ude37\ud834\ude38\ud834\ude39\ud834\ude3a\ud834\ude3b\ud834\ude3c\ud834\ude3d\ud834\ude3e\ud834\ude3f\ud834\ude40\ud834\ude41\ud834\ude45\ud834\udf00\ud834\udf01\ud834\udf02\ud834\udf03\ud834\udf04\ud834\udf05\ud834\udf06\ud834\udf07\ud834\udf08\ud834\udf09\ud834\udf0a\ud834\udf0b\ud834\udf0c\ud834\udf0d\ud834\udf0e\ud834\udf0f\ud834\udf10\ud834\udf11\ud834\udf12\ud834\udf13\ud834\udf14\ud834\udf15\ud834\udf16\ud834\udf17\ud834\udf18\ud834\udf19\ud834\udf1a\ud834\udf1b\ud834\udf1c\ud834\udf1d\ud834\udf1e\ud834\udf1f\ud834\udf20\ud834\udf21\ud834\udf22\ud834\udf23\ud834\udf24\ud834\udf25\ud834\udf26\ud834\udf27\ud834\udf28\ud834\udf29\ud834\udf2a\ud834\udf2b\ud834\udf2c\ud834\udf2d\ud834\udf2e\ud834\udf2f\ud834\udf30\ud834\udf31\ud834\udf32\ud834\udf33\ud834\udf34\ud834\udf35\ud834\udf36\ud834\udf37\ud834\udf38\ud834\udf39\ud834\udf3a\ud834\udf3b\ud834\udf3c\ud834\udf3d\ud834\udf3e\ud834\udf3f\ud834\udf40\ud834\udf41\ud834\udf42\ud834\udf43\ud834\udf44\ud834\udf45\ud834\udf46\ud834\udf47\ud834\udf48\ud834\udf49\ud834\udf4a\ud834\udf4b\ud834\udf4c\ud834\udf4d\ud834\udf4e\ud834\udf4f\ud834\udf50\ud834\udf51\ud834\udf52\ud834\udf53\ud834\udf54\ud834\udf55\ud834\udf56\ud83c\udc00\ud83c\udc01\ud83c\udc02\ud83c\udc03\ud83c\udc04\ud83c\udc05\ud83c\udc06\ud83c\udc07\ud83c\udc08\ud83c\udc09\ud83c\udc0a\ud83c\udc0b\ud83c\udc0c\ud83c\udc0d\ud83c\udc0e\ud83c\udc0f\ud83c\udc10\ud83c\udc11\ud83c\udc12\ud83c\udc13\ud83c\udc14\ud83c\udc15\ud83c\udc16\ud83c\udc17\ud83c\udc18\ud83c\udc19\ud83c\udc1a\ud83c\udc1b\ud83c\udc1c\ud83c\udc1d\ud83c\udc1e\ud83c\udc1f\ud83c\udc20\ud83c\udc21\ud83c\udc22\ud83c\udc23\ud83c\udc24\ud83c\udc25\ud83c\udc26\ud83c\udc27\ud83c\udc28\ud83c\udc29\ud83c\udc2a\ud83c\udc2b\ud83c\udc30\ud83c\udc31\ud83c\udc32\ud83c\udc33\ud83c\udc34\ud83c\udc35\ud83c\udc36\ud83c\udc37\ud83c\udc38\ud83c\udc39\ud83c\udc3a\ud83c\udc3b\ud83c\udc3c\ud83c\udc3d\ud83c\udc3e\ud83c\udc3f\ud83c\udc40\ud83c\udc41\ud83c\udc42\ud83c\udc43\ud83c\udc44\ud83c\udc45\ud83c\udc46\ud83c\udc47\ud83c\udc48\ud83c\udc49\ud83c\udc4a\ud83c\udc4b\ud83c\udc4c\ud83c\udc4d\ud83c\udc4e\ud83c\udc4f\ud83c\udc50\ud83c\udc51\ud83c\udc52\ud83c\udc53\ud83c\udc54\ud83c\udc55\ud83c\udc56\ud83c\udc57\ud83c\udc58\ud83c\udc59\ud83c\udc5a\ud83c\udc5b\ud83c\udc5c\ud83c\udc5d\ud83c\udc5e\ud83c\udc5f\ud83c\udc60\ud83c\udc61\ud83c\udc62\ud83c\udc63\ud83c\udc64\ud83c\udc65\ud83c\udc66\ud83c\udc67\ud83c\udc68\ud83c\udc69\ud83c\udc6a\ud83c\udc6b\ud83c\udc6c\ud83c\udc6d\ud83c\udc6e\ud83c\udc6f\ud83c\udc70\ud83c\udc71\ud83c\udc72\ud83c\udc73\ud83c\udc74\ud83c\udc75\ud83c\udc76\ud83c\udc77\ud83c\udc78\ud83c\udc79\ud83c\udc7a\ud83c\udc7b\ud83c\udc7c\ud83c\udc7d\ud83c\udc7e\ud83c\udc7f\ud83c\udc80\ud83c\udc81\ud83c\udc82\ud83c\udc83\ud83c\udc84\ud83c\udc85\ud83c\udc86\ud83c\udc87\ud83c\udc88\ud83c\udc89\ud83c\udc8a\ud83c\udc8b\ud83c\udc8c\ud83c\udc8d\ud83c\udc8e\ud83c\udc8f\ud83c\udc90\ud83c\udc91\ud83c\udc92\ud83c\udc93\ud83c\udca0\ud83c\udca1\ud83c\udca2\ud83c\udca3\ud83c\udca4\ud83c\udca5\ud83c\udca6\ud83c\udca7\ud83c\udca8\ud83c\udca9\ud83c\udcaa\ud83c\udcab\ud83c\udcac\ud83c\udcad\ud83c\udcae\ud83c\udcb1\ud83c\udcb2\ud83c\udcb3\ud83c\udcb4\ud83c\udcb5\ud83c\udcb6\ud83c\udcb7\ud83c\udcb8\ud83c\udcb9\ud83c\udcba\ud83c\udcbb\ud83c\udcbc\ud83c\udcbd\ud83c\udcbe\ud83c\udcc1\ud83c\udcc2\ud83c\udcc3\ud83c\udcc4\ud83c\udcc5\ud83c\udcc6\ud83c\udcc7\ud83c\udcc8\ud83c\udcc9\ud83c\udcca\ud83c\udccb\ud83c\udccc\ud83c\udccd\ud83c\udcce\ud83c\udccf\ud83c\udcd1\ud83c\udcd2\ud83c\udcd3\ud83c\udcd4\ud83c\udcd5\ud83c\udcd6\ud83c\udcd7\ud83c\udcd8\ud83c\udcd9\ud83c\udcda\ud83c\udcdb\ud83c\udcdc\ud83c\udcdd\ud83c\udcde\ud83c\udcdf\ud83c\udd10\ud83c\udd11\ud83c\udd12\ud83c\udd13\ud83c\udd14\ud83c\udd15\ud83c\udd16\ud83c\udd17\ud83c\udd18\ud83c\udd19\ud83c\udd1a\ud83c\udd1b\ud83c\udd1c\ud83c\udd1d\ud83c\udd1e\ud83c\udd1f\ud83c\udd20\ud83c\udd21\ud83c\udd22\ud83c\udd23\ud83c\udd24\ud83c\udd25\ud83c\udd26\ud83c\udd27\ud83c\udd28\ud83c\udd29\ud83c\udd2a\ud83c\udd2b\ud83c\udd2c\ud83c\udd2d\ud83c\udd2e\ud83c\udd30\ud83c\udd31\ud83c\udd32\ud83c\udd33\ud83c\udd34\ud83c\udd35\ud83c\udd36\ud83c\udd37\ud83c\udd38\ud83c\udd39\ud83c\udd3a\ud83c\udd3b\ud83c\udd3c\ud83c\udd3d\ud83c\udd3e\ud83c\udd3f\ud83c\udd40\ud83c\udd41\ud83c\udd42\ud83c\udd43\ud83c\udd44\ud83c\udd45\ud83c\udd46\ud83c\udd47\ud83c\udd48\ud83c\udd49\ud83c\udd4a\ud83c\udd4b\ud83c\udd4c\ud83c\udd4d\ud83c\udd4e\ud83c\udd4f\ud83c\udd50\ud83c\udd51\ud83c\udd52\ud83c\udd53\ud83c\udd54\ud83c\udd55\ud83c\udd56\ud83c\udd57\ud83c\udd58\ud83c\udd59\ud83c\udd5a\ud83c\udd5b\ud83c\udd5c\ud83c\udd5d\ud83c\udd5e\ud83c\udd5f\ud83c\udd60\ud83c\udd61\ud83c\udd62\ud83c\udd63\ud83c\udd64\ud83c\udd65\ud83c\udd66\ud83c\udd67\ud83c\udd68\ud83c\udd69\ud83c\udd70\ud83c\udd71\ud83c\udd72\ud83c\udd73\ud83c\udd74\ud83c\udd75\ud83c\udd76\ud83c\udd77\ud83c\udd78\ud83c\udd79\ud83c\udd7a\ud83c\udd7b\ud83c\udd7c\ud83c\udd7d\ud83c\udd7e\ud83c\udd7f\ud83c\udd80\ud83c\udd81\ud83c\udd82\ud83c\udd83\ud83c\udd84\ud83c\udd85\ud83c\udd86\ud83c\udd87\ud83c\udd88\ud83c\udd89\ud83c\udd8a\ud83c\udd8b\ud83c\udd8c\ud83c\udd8d\ud83c\udd8e\ud83c\udd8f\ud83c\udd90\ud83c\udd91\ud83c\udd92\ud83c\udd93\ud83c\udd94\ud83c\udd95\ud83c\udd96\ud83c\udd97\ud83c\udd98\ud83c\udd99\ud83c\udd9a\ud83c\udde6\ud83c\udde7\ud83c\udde8\ud83c\udde9\ud83c\uddea\ud83c\uddeb\ud83c\uddec\ud83c\udded\ud83c\uddee\ud83c\uddef\ud83c\uddf0\ud83c\uddf1\ud83c\uddf2\ud83c\uddf3\ud83c\uddf4\ud83c\uddf5\ud83c\uddf6\ud83c\uddf7\ud83c\uddf8\ud83c\uddf9\ud83c\uddfa\ud83c\uddfb\ud83c\uddfc\ud83c\uddfd\ud83c\uddfe\ud83c\uddff\ud83c\ude00\ud83c\ude01\ud83c\ude02\ud83c\ude10\ud83c\ude11\ud83c\ude12\ud83c\ude13\ud83c\ude14\ud83c\ude15\ud83c\ude16\ud83c\ude17\ud83c\ude18\ud83c\ude19\ud83c\ude1a\ud83c\ude1b\ud83c\ude1c\ud83c\ude1d\ud83c\ude1e\ud83c\ude1f\ud83c\ude20\ud83c\ude21\ud83c\ude22\ud83c\ude23\ud83c\ude24\ud83c\ude25\ud83c\ude26\ud83c\ude27\ud83c\ude28\ud83c\ude29\ud83c\ude2a\ud83c\ude2b\ud83c\ude2c\ud83c\ude2d\ud83c\ude2e\ud83c\ude2f\ud83c\ude30\ud83c\ude31\ud83c\ude32\ud83c\ude33\ud83c\ude34\ud83c\ude35\ud83c\ude36\ud83c\ude37\ud83c\ude38\ud83c\ude39\ud83c\ude3a\ud83c\ude40\ud83c\ude41\ud83c\ude42\ud83c\ude43\ud83c\ude44\ud83c\ude45\ud83c\ude46\ud83c\ude47\ud83c\ude48\ud83c\ude50\ud83c\ude51\ud83c\udf00\ud83c\udf01\ud83c\udf02\ud83c\udf03\ud83c\udf04\ud83c\udf05\ud83c\udf06\ud83c\udf07\ud83c\udf08\ud83c\udf09\ud83c\udf0a\ud83c\udf0b\ud83c\udf0c\ud83c\udf0d\ud83c\udf0e\ud83c\udf0f\ud83c\udf10\ud83c\udf11\ud83c\udf12\ud83c\udf13\ud83c\udf14\ud83c\udf15\ud83c\udf16\ud83c\udf17\ud83c\udf18\ud83c\udf19\ud83c\udf1a\ud83c\udf1b\ud83c\udf1c\ud83c\udf1d\ud83c\udf1e\ud83c\udf1f\ud83c\udf20\ud83c\udf30\ud83c\udf31\ud83c\udf32\ud83c\udf33\ud83c\udf34\ud83c\udf35\ud83c\udf37\ud83c\udf38\ud83c\udf39\ud83c\udf3a\ud83c\udf3b\ud83c\udf3c\ud83c\udf3d\ud83c\udf3e\ud83c\udf3f\ud83c\udf40\ud83c\udf41\ud83c\udf42\ud83c\udf43\ud83c\udf44\ud83c\udf45\ud83c\udf46\ud83c\udf47\ud83c\udf48\ud83c\udf49\ud83c\udf4a\ud83c\udf4b\ud83c\udf4c\ud83c\udf4d\ud83c\udf4e\ud83c\udf4f\ud83c\udf50\ud83c\udf51\ud83c\udf52\ud83c\udf53\ud83c\udf54\ud83c\udf55\ud83c\udf56\ud83c\udf57\ud83c\udf58\ud83c\udf59\ud83c\udf5a\ud83c\udf5b\ud83c\udf5c\ud83c\udf5d\ud83c\udf5e\ud83c\udf5f\ud83c\udf60\ud83c\udf61\ud83c\udf62\ud83c\udf63\ud83c\udf64\ud83c\udf65\ud83c\udf66\ud83c\udf67\ud83c\udf68\ud83c\udf69\ud83c\udf6a\ud83c\udf6b\ud83c\udf6c\ud83c\udf6d\ud83c\udf6e\ud83c\udf6f\ud83c\udf70\ud83c\udf71\ud83c\udf72\ud83c\udf73\ud83c\udf74\ud83c\udf75\ud83c\udf76\ud83c\udf77\ud83c\udf78\ud83c\udf79\ud83c\udf7a\ud83c\udf7b\ud83c\udf7c\ud83c\udf80\ud83c\udf81\ud83c\udf82\ud83c\udf83\ud83c\udf84\ud83c\udf85\ud83c\udf86\ud83c\udf87\ud83c\udf88\ud83c\udf89\ud83c\udf8a\ud83c\udf8b\ud83c\udf8c\ud83c\udf8d\ud83c\udf8e\ud83c\udf8f\ud83c\udf90\ud83c\udf91\ud83c\udf92\ud83c\udf93\ud83c\udfa0\ud83c\udfa1\ud83c\udfa2\ud83c\udfa3\ud83c\udfa4\ud83c\udfa5\ud83c\udfa6\ud83c\udfa7\ud83c\udfa8\ud83c\udfa9\ud83c\udfaa\ud83c\udfab\ud83c\udfac\ud83c\udfad\ud83c\udfae\ud83c\udfaf\ud83c\udfb0\ud83c\udfb1\ud83c\udfb2\ud83c\udfb3\ud83c\udfb4\ud83c\udfb5\ud83c\udfb6\ud83c\udfb7\ud83c\udfb8\ud83c\udfb9\ud83c\udfba\ud83c\udfbb\ud83c\udfbc\ud83c\udfbd\ud83c\udfbe\ud83c\udfbf\ud83c\udfc0\ud83c\udfc1\ud83c\udfc2\ud83c\udfc3\ud83c\udfc4\ud83c\udfc6\ud83c\udfc7\ud83c\udfc8\ud83c\udfc9\ud83c\udfca\ud83c\udfe0\ud83c\udfe1\ud83c\udfe2\ud83c\udfe3\ud83c\udfe4\ud83c\udfe5\ud83c\udfe6\ud83c\udfe7\ud83c\udfe8\ud83c\udfe9\ud83c\udfea\ud83c\udfeb\ud83c\udfec\ud83c\udfed\ud83c\udfee\ud83c\udfef\ud83c\udff0\ud83d\udc00\ud83d\udc01\ud83d\udc02\ud83d\udc03\ud83d\udc04\ud83d\udc05\ud83d\udc06\ud83d\udc07\ud83d\udc08\ud83d\udc09\ud83d\udc0a\ud83d\udc0b\ud83d\udc0c\ud83d\udc0d\ud83d\udc0e\ud83d\udc0f\ud83d\udc10\ud83d\udc11\ud83d\udc12\ud83d\udc13\ud83d\udc14\ud83d\udc15\ud83d\udc16\ud83d\udc17\ud83d\udc18\ud83d\udc19\ud83d\udc1a\ud83d\udc1b\ud83d\udc1c\ud83d\udc1d\ud83d\udc1e\ud83d\udc1f\ud83d\udc20\ud83d\udc21\ud83d\udc22\ud83d\udc23\ud83d\udc24\ud83d\udc25\ud83d\udc26\ud83d\udc27\ud83d\udc28\ud83d\udc29\ud83d\udc2a\ud83d\udc2b\ud83d\udc2c\ud83d\udc2d\ud83d\udc2e\ud83d\udc2f\ud83d\udc30\ud83d\udc31\ud83d\udc32\ud83d\udc33\ud83d\udc34\ud83d\udc35\ud83d\udc36\ud83d\udc37\ud83d\udc38\ud83d\udc39\ud83d\udc3a\ud83d\udc3b\ud83d\udc3c\ud83d\udc3d\ud83d\udc3e\ud83d\udc40\ud83d\udc42\ud83d\udc43\ud83d\udc44\ud83d\udc45\ud83d\udc46\ud83d\udc47\ud83d\udc48\ud83d\udc49\ud83d\udc4a\ud83d\udc4b\ud83d\udc4c\ud83d\udc4d\ud83d\udc4e\ud83d\udc4f\ud83d\udc50\ud83d\udc51\ud83d\udc52\ud83d\udc53\ud83d\udc54\ud83d\udc55\ud83d\udc56\ud83d\udc57\ud83d\udc58\ud83d\udc59\ud83d\udc5a\ud83d\udc5b\ud83d\udc5c\ud83d\udc5d\ud83d\udc5e\ud83d\udc5f\ud83d\udc60\ud83d\udc61\ud83d\udc62\ud83d\udc63\ud83d\udc64\ud83d\udc65\ud83d\udc66\ud83d\udc67\ud83d\udc68\ud83d\udc69\ud83d\udc6a\ud83d\udc6b\ud83d\udc6c\ud83d\udc6d\ud83d\udc6e\ud83d\udc6f\ud83d\udc70\ud83d\udc71\ud83d\udc72\ud83d\udc73\ud83d\udc74\ud83d\udc75\ud83d\udc76\ud83d\udc77\ud83d\udc78\ud83d\udc79\ud83d\udc7a\ud83d\udc7b\ud83d\udc7c\ud83d\udc7d\ud83d\udc7e\ud83d\udc7f\ud83d\udc80\ud83d\udc81\ud83d\udc82\ud83d\udc83\ud83d\udc84\ud83d\udc85\ud83d\udc86\ud83d\udc87\ud83d\udc88\ud83d\udc89\ud83d\udc8a\ud83d\udc8b\ud83d\udc8c\ud83d\udc8d\ud83d\udc8e\ud83d\udc8f\ud83d\udc90\ud83d\udc91\ud83d\udc92\ud83d\udc93\ud83d\udc94\ud83d\udc95\ud83d\udc96\ud83d\udc97\ud83d\udc98\ud83d\udc99\ud83d\udc9a\ud83d\udc9b\ud83d\udc9c\ud83d\udc9d\ud83d\udc9e\ud83d\udc9f\ud83d\udca0\ud83d\udca1\ud83d\udca2\ud83d\udca3\ud83d\udca4\ud83d\udca5\ud83d\udca6\ud83d\udca7\ud83d\udca8\ud83d\udca9\ud83d\udcaa\ud83d\udcab\ud83d\udcac\ud83d\udcad\ud83d\udcae\ud83d\udcaf\ud83d\udcb0\ud83d\udcb1\ud83d\udcb2\ud83d\udcb3\ud83d\udcb4\ud83d\udcb5\ud83d\udcb6\ud83d\udcb7\ud83d\udcb8\ud83d\udcb9\ud83d\udcba\ud83d\udcbb\ud83d\udcbc\ud83d\udcbd\ud83d\udcbe\ud83d\udcbf\ud83d\udcc0\ud83d\udcc1\ud83d\udcc2\ud83d\udcc3\ud83d\udcc4\ud83d\udcc5\ud83d\udcc6\ud83d\udcc7\ud83d\udcc8\ud83d\udcc9\ud83d\udcca\ud83d\udccb\ud83d\udccc\ud83d\udccd\ud83d\udcce\ud83d\udccf\ud83d\udcd0\ud83d\udcd1\ud83d\udcd2\ud83d\udcd3\ud83d\udcd4\ud83d\udcd5\ud83d\udcd6\ud83d\udcd7\ud83d\udcd8\ud83d\udcd9\ud83d\udcda\ud83d\udcdb\ud83d\udcdc\ud83d\udcdd\ud83d\udcde\ud83d\udcdf\ud83d\udce0\ud83d\udce1\ud83d\udce2\ud83d\udce3\ud83d\udce4\ud83d\udce5\ud83d\udce6\ud83d\udce7\ud83d\udce8\ud83d\udce9\ud83d\udcea\ud83d\udceb\ud83d\udcec\ud83d\udced\ud83d\udcee\ud83d\udcef\ud83d\udcf0\ud83d\udcf1\ud83d\udcf2\ud83d\udcf3\ud83d\udcf4\ud83d\udcf5\ud83d\udcf6\ud83d\udcf7\ud83d\udcf9\ud83d\udcfa\ud83d\udcfb\ud83d\udcfc\ud83d\udd00\ud83d\udd01\ud83d\udd02\ud83d\udd03\ud83d\udd04\ud83d\udd05\ud83d\udd06\ud83d\udd07\ud83d\udd08\ud83d\udd09\ud83d\udd0a\ud83d\udd0b\ud83d\udd0c\ud83d\udd0d\ud83d\udd0e\ud83d\udd0f\ud83d\udd10\ud83d\udd11\ud83d\udd12\ud83d\udd13\ud83d\udd14\ud83d\udd15\ud83d\udd16\ud83d\udd17\ud83d\udd18\ud83d\udd19\ud83d\udd1a\ud83d\udd1b\ud83d\udd1c\ud83d\udd1d\ud83d\udd1e\ud83d\udd1f\ud83d\udd20\ud83d\udd21\ud83d\udd22\ud83d\udd23\ud83d\udd24\ud83d\udd25\ud83d\udd26\ud83d\udd27\ud83d\udd28\ud83d\udd29\ud83d\udd2a\ud83d\udd2b\ud83d\udd2c\ud83d\udd2d\ud83d\udd2e\ud83d\udd2f\ud83d\udd30\ud83d\udd31\ud83d\udd32\ud83d\udd33\ud83d\udd34\ud83d\udd35\ud83d\udd36\ud83d\udd37\ud83d\udd38\ud83d\udd39\ud83d\udd3a\ud83d\udd3b\ud83d\udd3c\ud83d\udd3d\ud83d\udd50\ud83d\udd51\ud83d\udd52\ud83d\udd53\ud83d\udd54\ud83d\udd55\ud83d\udd56\ud83d\udd57\ud83d\udd58\ud83d\udd59\ud83d\udd5a\ud83d\udd5b\ud83d\udd5c\ud83d\udd5d\ud83d\udd5e\ud83d\udd5f\ud83d\udd60\ud83d\udd61\ud83d\udd62\ud83d\udd63\ud83d\udd64\ud83d\udd65\ud83d\udd66\ud83d\udd67\ud83d\uddfb\ud83d\uddfc\ud83d\uddfd\ud83d\uddfe\ud83d\uddff\ud83d\ude01\ud83d\ude02\ud83d\ude03\ud83d\ude04\ud83d\ude05\ud83d\ude06\ud83d\ude07\ud83d\ude08\ud83d\ude09\ud83d\ude0a\ud83d\ude0b\ud83d\ude0c\ud83d\ude0d\ud83d\ude0e\ud83d\ude0f\ud83d\ude10\ud83d\ude12\ud83d\ude13\ud83d\ude14\ud83d\ude16\ud83d\ude18\ud83d\ude1a\ud83d\ude1c\ud83d\ude1d\ud83d\ude1e\ud83d\ude20\ud83d\ude21\ud83d\ude22\ud83d\ude23\ud83d\ude24\ud83d\ude25\ud83d\ude28\ud83d\ude29\ud83d\ude2a\ud83d\ude2b\ud83d\ude2d\ud83d\ude30\ud83d\ude31\ud83d\ude32\ud83d\ude33\ud83d\ude35\ud83d\ude36\ud83d\ude37\ud83d\ude38\ud83d\ude39\ud83d\ude3a\ud83d\ude3b\ud83d\ude3c\ud83d\ude3d\ud83d\ude3e\ud83d\ude3f\ud83d\ude40\ud83d\ude45\ud83d\ude46\ud83d\ude47\ud83d\ude48\ud83d\ude49\ud83d\ude4a\ud83d\ude4b\ud83d\ude4c\ud83d\ude4d\ud83d\ude4e\ud83d\ude4f\ud83d\ude80\ud83d\ude81\ud83d\ude82\ud83d\ude83\ud83d\ude84\ud83d\ude85\ud83d\ude86\ud83d\ude87\ud83d\ude88\ud83d\ude89\ud83d\ude8a\ud83d\ude8b\ud83d\ude8c\ud83d\ude8d\ud83d\ude8e\ud83d\ude8f\ud83d\ude90\ud83d\ude91\ud83d\ude92\ud83d\ude93\ud83d\ude94\ud83d\ude95\ud83d\ude96\ud83d\ude97\ud83d\ude98\ud83d\ude99\ud83d\ude9a\ud83d\ude9b\ud83d\ude9c\ud83d\ude9d\ud83d\ude9e\ud83d\ude9f\ud83d\udea0\ud83d\udea1\ud83d\udea2\ud83d\udea3\ud83d\udea4\ud83d\udea5\ud83d\udea6\ud83d\udea7\ud83d\udea8\ud83d\udea9\ud83d\udeaa\ud83d\udeab\ud83d\udeac\ud83d\udead\ud83d\udeae\ud83d\udeaf\ud83d\udeb0\ud83d\udeb1\ud83d\udeb2\ud83d\udeb3\ud83d\udeb4\ud83d\udeb5\ud83d\udeb6\ud83d\udeb7\ud83d\udeb8\ud83d\udeb9\ud83d\udeba\ud83d\udebb\ud83d\udebc\ud83d\udebd\ud83d\udebe\ud83d\udebf\ud83d\udec0\ud83d\udec1\ud83d\udec2\ud83d\udec3\ud83d\udec4\ud83d\udec5\ud83d\udf00\ud83d\udf01\ud83d\udf02\ud83d\udf03\ud83d\udf04\ud83d\udf05\ud83d\udf06\ud83d\udf07\ud83d\udf08\ud83d\udf09\ud83d\udf0a\ud83d\udf0b\ud83d\udf0c\ud83d\udf0d\ud83d\udf0e\ud83d\udf0f\ud83d\udf10\ud83d\udf11\ud83d\udf12\ud83d\udf13\ud83d\udf14\ud83d\udf15\ud83d\udf16\ud83d\udf17\ud83d\udf18\ud83d\udf19\ud83d\udf1a\ud83d\udf1b\ud83d\udf1c\ud83d\udf1d\ud83d\udf1e\ud83d\udf1f\ud83d\udf20\ud83d\udf21\ud83d\udf22\ud83d\udf23\ud83d\udf24\ud83d\udf25\ud83d\udf26\ud83d\udf27\ud83d\udf28\ud83d\udf29\ud83d\udf2a\ud83d\udf2b\ud83d\udf2c\ud83d\udf2d\ud83d\udf2e\ud83d\udf2f\ud83d\udf30\ud83d\udf31\ud83d\udf32\ud83d\udf33\ud83d\udf34\ud83d\udf35\ud83d\udf36\ud83d\udf37\ud83d\udf38\ud83d\udf39\ud83d\udf3a\ud83d\udf3b\ud83d\udf3c\ud83d\udf3d\ud83d\udf3e\ud83d\udf3f\ud83d\udf40\ud83d\udf41\ud83d\udf42\ud83d\udf43\ud83d\udf44\ud83d\udf45\ud83d\udf46\ud83d\udf47\ud83d\udf48\ud83d\udf49\ud83d\udf4a\ud83d\udf4b\ud83d\udf4c\ud83d\udf4d\ud83d\udf4e\ud83d\udf4f\ud83d\udf50\ud83d\udf51\ud83d\udf52\ud83d\udf53\ud83d\udf54\ud83d\udf55\ud83d\udf56\ud83d\udf57\ud83d\udf58\ud83d\udf59\ud83d\udf5a\ud83d\udf5b\ud83d\udf5c\ud83d\udf5d\ud83d\udf5e\ud83d\udf5f\ud83d\udf60\ud83d\udf61\ud83d\udf62\ud83d\udf63\ud83d\udf64\ud83d\udf65\ud83d\udf66\ud83d\udf67\ud83d\udf68\ud83d\udf69\ud83d\udf6a\ud83d\udf6b\ud83d\udf6c\ud83d\udf6d\ud83d\udf6e\ud83d\udf6f\ud83d\udf70\ud83d\udf71\ud83d\udf72\ud83d\udf73'
			},
			{
				'description': 'Unicode character category [Zl]: separator, line',
				'decoded': [8232],
				'encoded': '\u2028'
			},
			{
				'description': 'Unicode character category [Zp]: separator, paragraph',
				'decoded': [8233],
				'encoded': '\u2029'
			},
			{
				'description': 'Unicode character category [Zs]: separator, space',
				'decoded': [32, 160, 5760, 6158, 8192, 8193, 8194, 8195, 8196, 8197, 8198, 8199, 8200, 8201, 8202, 8239, 8287, 12288],
				'encoded': ' \xa0\u1680\u180e\u2000\u2001\u2002\u2003\u2004\u2005\u2006\u2007\u2008\u2009\u200a\u202f\u205f\u3000'
			}
		],
		'domains': [
			{
				'decoded': 'ma\xf1ana.com', // 'mañana.com'
				'encoded': 'xn--maana-pta.com'
			},
			{
				'decoded': 'b\xfccher.com', // 'bücher.com'
				'encoded': 'xn--bcher-kva.com'
			},
			{
				'decoded': 'caf\xe9.com', // 'café.com'
				'encoded': 'xn--caf-dma.com'
			},
			{
				'decoded': '\u2603-\u2318.com', // '☃-⌘.com'
				'encoded': 'xn----dqo34k.com'
			},
			{
				'decoded': '\ud400\u2603-\u2318.com', // '퐀☃-⌘.com'
				'encoded': 'xn----dqo34kn65z.com'
			},
			{
				'description': 'Emoji',
				'decoded': '\ud83d\udca9.la', // '.la'
				'encoded': 'xn--ls8h.la'
			}
		]
	};

	/*--------------------------------------------------------------------------*/

	// simple `Array#forEach`-like helper function
	function each(array, fn) {
		var length = array.length;
		while (length--) {
			fn(array[length]);
		}
	}

	/*--------------------------------------------------------------------------*/

	test('punycode.ucs2.decode', function() {
		each(testData.ucs2, function(object) {
			deepEqual(punycode.ucs2.decode(object.encoded), object.decoded, object.description);
		});
	});

	test('punycode.ucs2.encode', function() {
		each(testData.ucs2, function(object) {
			equal(punycode.ucs2.encode(object.decoded), object.encoded, object.description);
		});
	});

	test('punycode.decode', function() {
		each(testData.strings, function(object) {
			equal(punycode.decode(object.encoded), object.decoded, object.description);
		});
	});

	test('punycode.encode', function() {
		each(testData.strings, function(object) {
			equal(punycode.encode(object.decoded), object.encoded, object.description);
		});
	});

	test('punycode.toUnicode', function() {
		each(testData.domains, function(object) {
			equal(punycode.toUnicode(object.encoded), object.decoded, object.description);
		});
		/**
		 * Domain names (or other strings) that don’t start with `xn--` may not be
		 * converted.
		 */
		each(testData.strings, function(object) {
			var message = 'Domain names (or other strings) that don’t start with `xn--` may not be converted';
			equal(punycode.toUnicode(object.encoded), object.encoded, message);
			equal(punycode.toUnicode(object.decoded), object.decoded, message);
		});
	});

	test('punycode.toASCII', function() {
		each(testData.domains, function(object) {
			equal(punycode.toASCII(object.decoded), object.encoded, object.description);
		});
		/**
		 * Domain names (or other strings) that are already in ASCII may not be
		 * converted.
		 */
		each(testData.strings, function(object) {
			equal(punycode.toASCII(object.encoded), object.encoded, 'Domain names (or other strings) that are already in ASCII may not be converted');
		});
	});

	if (root.document && root.require) {
		test('require(\'punycode\')', function() {
			equal((punycode2 || {}).version, punycode.version, 'require(\'punycode\')');
		});
	}

	/*--------------------------------------------------------------------------*/

	// explicitly call `QUnit.start()` in a CLI environment
	if (!root.document) {
		QUnit.start();
	}

}(typeof global == 'object' && global || this));