/**
* @license
* Copyright Google Inc. All Rights Reserved.
*
* Use of this source code is governed by an MIT-style license that can be
* found in the LICENSE file at https://angular.io/license
*/
import { SecurityContext } from '../../sanitization/security';
import { SANITIZER } from '../interfaces/view';
import { getProp as getOldProp, getSinglePropIndexValue as getOldSinglePropIndexValue } from '../styling/class_and_style_bindings';
import { getCurrentStyleSanitizer, setCurrentStyleSanitizer } from './state';
var MAP_BASED_ENTRY_PROP_NAME = '--MAP--';
/**
 * Creates a new instance of the `TStylingContext`.
 *
 * This function will also pre-fill the context with data
 * for map-based bindings.
 */
export function allocTStylingContext() {
    // because map-based bindings deal with a dynamic set of values, there
    // is no way to know ahead of time whether or not sanitization is required.
    // For this reason the configuration will always mark sanitization as active
    // (this means that when map-based values are applied then sanitization will
    // be checked against each property).
    var mapBasedConfig = 1 /* SanitizationRequired */;
    return [0 /* Initial */, 0, mapBasedConfig, 0, MAP_BASED_ENTRY_PROP_NAME];
}
/**
 * Temporary function that allows for a string-based property name to be
 * obtained from an index-based property identifier.
 *
 * This function will be removed once the new styling refactor code (which
 * lives inside of `render3/styling_next/`) replaces the existing styling
 * implementation.
 */
export function getBindingNameFromIndex(stylingContext, offset, directiveIndex, isClassBased) {
    var singleIndex = getOldSinglePropIndexValue(stylingContext, directiveIndex, offset, isClassBased);
    return getOldProp(stylingContext, singleIndex);
}
export function updateContextDirectiveIndex(context, index) {
    context[1 /* MaxDirectiveIndexPosition */] = index;
}
function getConfig(context) {
    return context[0 /* ConfigPosition */];
}
export function setConfig(context, value) {
    context[0 /* ConfigPosition */] = value;
}
export function getProp(context, index) {
    return context[index + 2 /* PropOffset */];
}
function getPropConfig(context, index) {
    return context[index + 0 /* ConfigAndGuardOffset */] &
        1 /* Mask */;
}
export function isSanitizationRequired(context, index) {
    return (getPropConfig(context, index) & 1 /* SanitizationRequired */) > 0;
}
export function getGuardMask(context, index) {
    var configGuardValue = context[index + 0 /* ConfigAndGuardOffset */];
    return configGuardValue >> 1 /* TotalBits */;
}
export function setGuardMask(context, index, maskValue) {
    var config = getPropConfig(context, index);
    var guardMask = maskValue << 1 /* TotalBits */;
    context[index + 0 /* ConfigAndGuardOffset */] = config | guardMask;
}
export function getValuesCount(context, index) {
    return context[index + 1 /* ValuesCountOffset */];
}
export function getBindingValue(context, index, offset) {
    return context[index + 3 /* BindingsStartOffset */ + offset];
}
export function getDefaultValue(context, index) {
    var valuesCount = getValuesCount(context, index);
    return context[index + 3 /* BindingsStartOffset */ + valuesCount - 1];
}
/**
 * Temporary function which determines whether or not a context is
 * allowed to be flushed based on the provided directive index.
 */
export function allowStylingFlush(context, index) {
    return index === context[1 /* MaxDirectiveIndexPosition */];
}
export function lockContext(context) {
    setConfig(context, getConfig(context) | 1 /* Locked */);
}
export function isContextLocked(context) {
    return (getConfig(context) & 1 /* Locked */) > 0;
}
export function getPropValuesStartPosition(context) {
    return 5 /* MapBindingsBindingsStartPosition */ +
        context[3 /* MapBindingsValuesCountPosition */];
}
export function isMapBased(prop) {
    return prop === MAP_BASED_ENTRY_PROP_NAME;
}
export function hasValueChanged(a, b) {
    var compareValueA = Array.isArray(a) ? a[0 /* RawValuePosition */] : a;
    var compareValueB = Array.isArray(b) ? b[0 /* RawValuePosition */] : b;
    return compareValueA !== compareValueB;
}
/**
 * Determines whether the provided styling value is truthy or falsy.
 */
export function isStylingValueDefined(value) {
    // the reason why null is compared against is because
    // a CSS class value that is set to `false` must be
    // respected (otherwise it would be treated as falsy).
    // Empty string values are because developers usually
    // set a value to an empty string to remove it.
    return value != null && value !== '';
}
/**
 * Returns the current style sanitizer function for the given view.
 *
 * The default style sanitizer (which lives inside of `LView`) will
 * be returned depending on whether the `styleSanitizer` instruction
 * was called or not prior to any styling instructions running.
 */
export function getCurrentOrLViewSanitizer(lView) {
    var sanitizer = (getCurrentStyleSanitizer() || lView[SANITIZER]);
    if (sanitizer && typeof sanitizer !== 'function') {
        setCurrentStyleSanitizer(sanitizer);
        return sanitizeUsingSanitizerObject;
    }
    return sanitizer;
}
/**
 * Style sanitization function that internally uses a `Sanitizer` instance to handle style
 * sanitization.
 */
var sanitizeUsingSanitizerObject = function (prop, value, mode) {
    var sanitizer = getCurrentStyleSanitizer();
    if (sanitizer) {
        if (mode & 2 /* SanitizeOnly */) {
            return sanitizer.sanitize(SecurityContext.STYLE, value);
        }
        else {
            return true;
        }
    }
    return value;
};
var ɵ0 = sanitizeUsingSanitizerObject;
export { ɵ0 };
//# sourceMappingURL=data:application/json;base64,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