/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import '../util/ng_dev_mode';
import { ChangeDetectionStrategy } from '../change_detection/constants';
import { NG_INJECTABLE_DEF, ɵɵdefineInjectable } from '../di/interface/defs';
import { ViewEncapsulation } from '../metadata/view';
import { noSideEffects } from '../util/closure';
import { stringify } from '../util/stringify';
import { EMPTY_ARRAY, EMPTY_OBJ } from './empty';
import { NG_BASE_DEF, NG_COMPONENT_DEF, NG_DIRECTIVE_DEF, NG_LOCALE_ID_DEF, NG_MODULE_DEF, NG_PIPE_DEF } from './fields';
var _renderCompCount = 0;
/**
 * Create a component definition object.
 *
 *
 * # Example
 * ```
 * class MyDirective {
 *   // Generated by Angular Template Compiler
 *   // [Symbol] syntax will not be supported by TypeScript until v2.7
 *   static ngComponentDef = defineComponent({
 *     ...
 *   });
 * }
 * ```
 * @codeGenApi
 */
export function ɵɵdefineComponent(componentDefinition) {
    var type = componentDefinition.type;
    var typePrototype = type.prototype;
    var declaredInputs = {};
    var def = {
        type: type,
        providersResolver: null,
        consts: componentDefinition.consts,
        vars: componentDefinition.vars,
        factory: componentDefinition.factory,
        template: componentDefinition.template || null,
        ngContentSelectors: componentDefinition.ngContentSelectors,
        hostBindings: componentDefinition.hostBindings || null,
        contentQueries: componentDefinition.contentQueries || null,
        declaredInputs: declaredInputs,
        inputs: null,
        outputs: null,
        exportAs: componentDefinition.exportAs || null,
        onChanges: null,
        onInit: typePrototype.ngOnInit || null,
        doCheck: typePrototype.ngDoCheck || null,
        afterContentInit: typePrototype.ngAfterContentInit || null,
        afterContentChecked: typePrototype.ngAfterContentChecked || null,
        afterViewInit: typePrototype.ngAfterViewInit || null,
        afterViewChecked: typePrototype.ngAfterViewChecked || null,
        onDestroy: typePrototype.ngOnDestroy || null,
        onPush: componentDefinition.changeDetection === ChangeDetectionStrategy.OnPush,
        directiveDefs: null,
        pipeDefs: null,
        selectors: componentDefinition.selectors,
        viewQuery: componentDefinition.viewQuery || null,
        features: componentDefinition.features || null,
        data: componentDefinition.data || {},
        // TODO(misko): convert ViewEncapsulation into const enum so that it can be used directly in the
        // next line. Also `None` should be 0 not 2.
        encapsulation: componentDefinition.encapsulation || ViewEncapsulation.Emulated,
        id: 'c',
        styles: componentDefinition.styles || EMPTY_ARRAY,
        _: null,
        setInput: null,
        schemas: componentDefinition.schemas || null,
        tView: null,
    };
    def._ = noSideEffects(function () {
        var directiveTypes = componentDefinition.directives;
        var feature = componentDefinition.features;
        var pipeTypes = componentDefinition.pipes;
        def.id += _renderCompCount++;
        def.inputs = invertObject(componentDefinition.inputs, declaredInputs),
            def.outputs = invertObject(componentDefinition.outputs),
            feature && feature.forEach(function (fn) { return fn(def); });
        def.directiveDefs = directiveTypes ?
            function () { return (typeof directiveTypes === 'function' ? directiveTypes() : directiveTypes)
                .map(extractDirectiveDef); } :
            null;
        def.pipeDefs = pipeTypes ?
            function () { return (typeof pipeTypes === 'function' ? pipeTypes() : pipeTypes).map(extractPipeDef); } :
            null;
        // Add ngInjectableDef so components are reachable through the module injector by default
        // (unless it has already been set by the @Injectable decorator). This is mostly to
        // support injecting components in tests. In real application code, components should
        // be retrieved through the node injector, so this isn't a problem.
        if (!type.hasOwnProperty(NG_INJECTABLE_DEF)) {
            type[NG_INJECTABLE_DEF] =
                ɵɵdefineInjectable({ token: type, factory: componentDefinition.factory });
        }
    });
    return def;
}
/**
 * @codeGenApi
 */
export function ɵɵsetComponentScope(type, directives, pipes) {
    var def = type.ngComponentDef;
    def.directiveDefs = function () { return directives.map(extractDirectiveDef); };
    def.pipeDefs = function () { return pipes.map(extractPipeDef); };
}
export function extractDirectiveDef(type) {
    var def = getComponentDef(type) || getDirectiveDef(type);
    if (ngDevMode && !def) {
        throw new Error("'" + type.name + "' is neither 'ComponentType' or 'DirectiveType'.");
    }
    return def;
}
export function extractPipeDef(type) {
    var def = getPipeDef(type);
    if (ngDevMode && !def) {
        throw new Error("'" + type.name + "' is not a 'PipeType'.");
    }
    return def;
}
/**
 * @codeGenApi
 */
export function ɵɵdefineNgModule(def) {
    var res = {
        type: def.type,
        bootstrap: def.bootstrap || EMPTY_ARRAY,
        declarations: def.declarations || EMPTY_ARRAY,
        imports: def.imports || EMPTY_ARRAY,
        exports: def.exports || EMPTY_ARRAY,
        transitiveCompileScopes: null,
        schemas: def.schemas || null,
        id: def.id || null,
    };
    return res;
}
/**
 * Adds the module metadata that is necessary to compute the module's transitive scope to an
 * existing module definition.
 *
 * Scope metadata of modules is not used in production builds, so calls to this function can be
 * marked pure to tree-shake it from the bundle, allowing for all referenced declarations
 * to become eligible for tree-shaking as well.
 *
 * @codeGenApi
 */
export function ɵɵsetNgModuleScope(type, scope) {
    return noSideEffects(function () {
        var ngModuleDef = getNgModuleDef(type, true);
        ngModuleDef.declarations = scope.declarations || EMPTY_ARRAY;
        ngModuleDef.imports = scope.imports || EMPTY_ARRAY;
        ngModuleDef.exports = scope.exports || EMPTY_ARRAY;
    });
}
/**
 * Inverts an inputs or outputs lookup such that the keys, which were the
 * minified keys, are part of the values, and the values are parsed so that
 * the publicName of the property is the new key
 *
 * e.g. for
 *
 * ```
 * class Comp {
 *   @Input()
 *   propName1: string;
 *
 *   @Input('publicName2')
 *   declaredPropName2: number;
 * }
 * ```
 *
 * will be serialized as
 *
 * ```
 * {
 *   propName1: 'propName1',
 *   declaredPropName2: ['publicName2', 'declaredPropName2'],
 * }
 * ```
 *
 * which is than translated by the minifier as:
 *
 * ```
 * {
 *   minifiedPropName1: 'propName1',
 *   minifiedPropName2: ['publicName2', 'declaredPropName2'],
 * }
 * ```
 *
 * becomes: (public name => minifiedName)
 *
 * ```
 * {
 *  'propName1': 'minifiedPropName1',
 *  'publicName2': 'minifiedPropName2',
 * }
 * ```
 *
 * Optionally the function can take `secondary` which will result in: (public name => declared name)
 *
 * ```
 * {
 *  'propName1': 'propName1',
 *  'publicName2': 'declaredPropName2',
 * }
 * ```
 *

 */
function invertObject(obj, secondary) {
    if (obj == null)
        return EMPTY_OBJ;
    var newLookup = {};
    for (var minifiedKey in obj) {
        if (obj.hasOwnProperty(minifiedKey)) {
            var publicName = obj[minifiedKey];
            var declaredName = publicName;
            if (Array.isArray(publicName)) {
                declaredName = publicName[1];
                publicName = publicName[0];
            }
            newLookup[publicName] = minifiedKey;
            if (secondary) {
                (secondary[publicName] = declaredName);
            }
        }
    }
    return newLookup;
}
/**
 * Create a base definition
 *
 * # Example
 * ```ts
 * class ShouldBeInherited {
 *   static ngBaseDef = ɵɵdefineBase({
 *      ...
 *   })
 * }
 * ```
 *
 * @param baseDefinition The base definition parameters
 *
 * @codeGenApi
 */
export function ɵɵdefineBase(baseDefinition) {
    var declaredInputs = {};
    return {
        inputs: invertObject(baseDefinition.inputs, declaredInputs),
        declaredInputs: declaredInputs,
        outputs: invertObject(baseDefinition.outputs),
        viewQuery: baseDefinition.viewQuery || null,
        contentQueries: baseDefinition.contentQueries || null,
        hostBindings: baseDefinition.hostBindings || null
    };
}
/**
 * Create a directive definition object.
 *
 * # Example
 * ```ts
 * class MyDirective {
 *   // Generated by Angular Template Compiler
 *   // [Symbol] syntax will not be supported by TypeScript until v2.7
 *   static ngDirectiveDef = ɵɵdefineDirective({
 *     ...
 *   });
 * }
 * ```
 *
 * @codeGenApi
 */
export var ɵɵdefineDirective = ɵɵdefineComponent;
/**
 * Create a pipe definition object.
 *
 * # Example
 * ```
 * class MyPipe implements PipeTransform {
 *   // Generated by Angular Template Compiler
 *   static ngPipeDef = definePipe({
 *     ...
 *   });
 * }
 * ```
 * @param pipeDef Pipe definition generated by the compiler
 *
 * @codeGenApi
 */
export function ɵɵdefinePipe(pipeDef) {
    return {
        name: pipeDef.name,
        factory: pipeDef.factory,
        pure: pipeDef.pure !== false,
        onDestroy: pipeDef.type.prototype.ngOnDestroy || null
    };
}
/**
 * The following getter methods retrieve the definition form the type. Currently the retrieval
 * honors inheritance, but in the future we may change the rule to require that definitions are
 * explicit. This would require some sort of migration strategy.
 */
export function getComponentDef(type) {
    return type[NG_COMPONENT_DEF] || null;
}
export function getDirectiveDef(type) {
    return type[NG_DIRECTIVE_DEF] || null;
}
export function getPipeDef(type) {
    return type[NG_PIPE_DEF] || null;
}
export function getBaseDef(type) {
    return type[NG_BASE_DEF] || null;
}
export function getNgModuleDef(type, throwNotFound) {
    var ngModuleDef = type[NG_MODULE_DEF] || null;
    if (!ngModuleDef && throwNotFound === true) {
        throw new Error("Type " + stringify(type) + " does not have 'ngModuleDef' property.");
    }
    return ngModuleDef;
}
export function getNgLocaleIdDef(type) {
    return type[NG_LOCALE_ID_DEF] || null;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZGVmaW5pdGlvbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uLy4uLy4uLy4uLy4uLy4uLy4uLy4uL3BhY2thZ2VzL2NvcmUvc3JjL3JlbmRlcjMvZGVmaW5pdGlvbi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBQTs7Ozs7O0dBTUc7QUFFSCxPQUFPLHFCQUFxQixDQUFDO0FBQzdCLE9BQU8sRUFBQyx1QkFBdUIsRUFBQyxNQUFNLCtCQUErQixDQUFDO0FBQ3RFLE9BQU8sRUFBQyxpQkFBaUIsRUFBRSxrQkFBa0IsRUFBQyxNQUFNLHNCQUFzQixDQUFDO0FBSTNFLE9BQU8sRUFBQyxpQkFBaUIsRUFBQyxNQUFNLGtCQUFrQixDQUFDO0FBQ25ELE9BQU8sRUFBQyxhQUFhLEVBQUMsTUFBTSxpQkFBaUIsQ0FBQztBQUM5QyxPQUFPLEVBQUMsU0FBUyxFQUFDLE1BQU0sbUJBQW1CLENBQUM7QUFDNUMsT0FBTyxFQUFDLFdBQVcsRUFBRSxTQUFTLEVBQUMsTUFBTSxTQUFTLENBQUM7QUFDL0MsT0FBTyxFQUFDLFdBQVcsRUFBRSxnQkFBZ0IsRUFBRSxnQkFBZ0IsRUFBRSxnQkFBZ0IsRUFBRSxhQUFhLEVBQUUsV0FBVyxFQUFDLE1BQU0sVUFBVSxDQUFDO0FBTXZILElBQUksZ0JBQWdCLEdBQUcsQ0FBQyxDQUFDO0FBRXpCOzs7Ozs7Ozs7Ozs7Ozs7R0FlRztBQUNILE1BQU0sVUFBVSxpQkFBaUIsQ0FBSSxtQkEwTXBDO0lBQ0MsSUFBTSxJQUFJLEdBQUcsbUJBQW1CLENBQUMsSUFBSSxDQUFDO0lBQ3RDLElBQU0sYUFBYSxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUM7SUFDckMsSUFBTSxjQUFjLEdBQTRCLEVBQVMsQ0FBQztJQUMxRCxJQUFNLEdBQUcsR0FBd0Q7UUFDL0QsSUFBSSxFQUFFLElBQUk7UUFDVixpQkFBaUIsRUFBRSxJQUFJO1FBQ3ZCLE1BQU0sRUFBRSxtQkFBbUIsQ0FBQyxNQUFNO1FBQ2xDLElBQUksRUFBRSxtQkFBbUIsQ0FBQyxJQUFJO1FBQzlCLE9BQU8sRUFBRSxtQkFBbUIsQ0FBQyxPQUFPO1FBQ3BDLFFBQVEsRUFBRSxtQkFBbUIsQ0FBQyxRQUFRLElBQUksSUFBTTtRQUNoRCxrQkFBa0IsRUFBRSxtQkFBbUIsQ0FBQyxrQkFBa0I7UUFDMUQsWUFBWSxFQUFFLG1CQUFtQixDQUFDLFlBQVksSUFBSSxJQUFJO1FBQ3RELGNBQWMsRUFBRSxtQkFBbUIsQ0FBQyxjQUFjLElBQUksSUFBSTtRQUMxRCxjQUFjLEVBQUUsY0FBYztRQUM5QixNQUFNLEVBQUUsSUFBTTtRQUNkLE9BQU8sRUFBRSxJQUFNO1FBQ2YsUUFBUSxFQUFFLG1CQUFtQixDQUFDLFFBQVEsSUFBSSxJQUFJO1FBQzlDLFNBQVMsRUFBRSxJQUFJO1FBQ2YsTUFBTSxFQUFFLGFBQWEsQ0FBQyxRQUFRLElBQUksSUFBSTtRQUN0QyxPQUFPLEVBQUUsYUFBYSxDQUFDLFNBQVMsSUFBSSxJQUFJO1FBQ3hDLGdCQUFnQixFQUFFLGFBQWEsQ0FBQyxrQkFBa0IsSUFBSSxJQUFJO1FBQzFELG1CQUFtQixFQUFFLGFBQWEsQ0FBQyxxQkFBcUIsSUFBSSxJQUFJO1FBQ2hFLGFBQWEsRUFBRSxhQUFhLENBQUMsZUFBZSxJQUFJLElBQUk7UUFDcEQsZ0JBQWdCLEVBQUUsYUFBYSxDQUFDLGtCQUFrQixJQUFJLElBQUk7UUFDMUQsU0FBUyxFQUFFLGFBQWEsQ0FBQyxXQUFXLElBQUksSUFBSTtRQUM1QyxNQUFNLEVBQUUsbUJBQW1CLENBQUMsZUFBZSxLQUFLLHVCQUF1QixDQUFDLE1BQU07UUFDOUUsYUFBYSxFQUFFLElBQU07UUFDckIsUUFBUSxFQUFFLElBQU07UUFDaEIsU0FBUyxFQUFFLG1CQUFtQixDQUFDLFNBQVM7UUFDeEMsU0FBUyxFQUFFLG1CQUFtQixDQUFDLFNBQVMsSUFBSSxJQUFJO1FBQ2hELFFBQVEsRUFBRSxtQkFBbUIsQ0FBQyxRQUFRLElBQUksSUFBSTtRQUM5QyxJQUFJLEVBQUUsbUJBQW1CLENBQUMsSUFBSSxJQUFJLEVBQUU7UUFDcEMsZ0dBQWdHO1FBQ2hHLDRDQUE0QztRQUM1QyxhQUFhLEVBQUUsbUJBQW1CLENBQUMsYUFBYSxJQUFJLGlCQUFpQixDQUFDLFFBQVE7UUFDOUUsRUFBRSxFQUFFLEdBQUc7UUFDUCxNQUFNLEVBQUUsbUJBQW1CLENBQUMsTUFBTSxJQUFJLFdBQVc7UUFDakQsQ0FBQyxFQUFFLElBQWE7UUFDaEIsUUFBUSxFQUFFLElBQUk7UUFDZCxPQUFPLEVBQUUsbUJBQW1CLENBQUMsT0FBTyxJQUFJLElBQUk7UUFDNUMsS0FBSyxFQUFFLElBQUk7S0FDWixDQUFDO0lBQ0YsR0FBRyxDQUFDLENBQUMsR0FBRyxhQUFhLENBQUM7UUFDcEIsSUFBTSxjQUFjLEdBQUcsbUJBQW1CLENBQUMsVUFBWSxDQUFDO1FBQ3hELElBQU0sT0FBTyxHQUFHLG1CQUFtQixDQUFDLFFBQVEsQ0FBQztRQUM3QyxJQUFNLFNBQVMsR0FBRyxtQkFBbUIsQ0FBQyxLQUFPLENBQUM7UUFDOUMsR0FBRyxDQUFDLEVBQUUsSUFBSSxnQkFBZ0IsRUFBRSxDQUFDO1FBQzdCLEdBQUcsQ0FBQyxNQUFNLEdBQUcsWUFBWSxDQUFDLG1CQUFtQixDQUFDLE1BQU0sRUFBRSxjQUFjLENBQUM7WUFDckUsR0FBRyxDQUFDLE9BQU8sR0FBRyxZQUFZLENBQUMsbUJBQW1CLENBQUMsT0FBTyxDQUFDO1lBQ3ZELE9BQU8sSUFBSSxPQUFPLENBQUMsT0FBTyxDQUFDLFVBQUMsRUFBRSxJQUFLLE9BQUEsRUFBRSxDQUFDLEdBQUcsQ0FBQyxFQUFQLENBQU8sQ0FBQyxDQUFDO1FBQzVDLEdBQUcsQ0FBQyxhQUFhLEdBQUcsY0FBYyxDQUFDLENBQUM7WUFDaEMsY0FBTSxPQUFBLENBQUMsT0FBTyxjQUFjLEtBQUssVUFBVSxDQUFDLENBQUMsQ0FBQyxjQUFjLEVBQUUsQ0FBQyxDQUFDLENBQUMsY0FBYyxDQUFDO2lCQUNyRSxHQUFHLENBQUMsbUJBQW1CLENBQUMsRUFEN0IsQ0FDNkIsQ0FBQyxDQUFDO1lBQ3JDLElBQUksQ0FBQztRQUNULEdBQUcsQ0FBQyxRQUFRLEdBQUcsU0FBUyxDQUFDLENBQUM7WUFDdEIsY0FBTSxPQUFBLENBQUMsT0FBTyxTQUFTLEtBQUssVUFBVSxDQUFDLENBQUMsQ0FBQyxTQUFTLEVBQUUsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsR0FBRyxDQUFDLGNBQWMsQ0FBQyxFQUEvRSxDQUErRSxDQUFDLENBQUM7WUFDdkYsSUFBSSxDQUFDO1FBRVQseUZBQXlGO1FBQ3pGLG1GQUFtRjtRQUNuRixxRkFBcUY7UUFDckYsbUVBQW1FO1FBQ25FLElBQUksQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLGlCQUFpQixDQUFDLEVBQUU7WUFDMUMsSUFBWSxDQUFDLGlCQUFpQixDQUFDO2dCQUM1QixrQkFBa0IsQ0FBSSxFQUFDLEtBQUssRUFBRSxJQUFJLEVBQUUsT0FBTyxFQUFFLG1CQUFtQixDQUFDLE9BQWlCLEVBQUMsQ0FBQyxDQUFDO1NBQzFGO0lBQ0gsQ0FBQyxDQUFVLENBQUM7SUFFWixPQUFPLEdBQVksQ0FBQztBQUN0QixDQUFDO0FBRUQ7O0dBRUc7QUFDSCxNQUFNLFVBQVUsbUJBQW1CLENBQy9CLElBQXdCLEVBQUUsVUFBdUIsRUFBRSxLQUFrQjtJQUN2RSxJQUFNLEdBQUcsR0FBSSxJQUFJLENBQUMsY0FBb0MsQ0FBQztJQUN2RCxHQUFHLENBQUMsYUFBYSxHQUFHLGNBQU0sT0FBQSxVQUFVLENBQUMsR0FBRyxDQUFDLG1CQUFtQixDQUFDLEVBQW5DLENBQW1DLENBQUM7SUFDOUQsR0FBRyxDQUFDLFFBQVEsR0FBRyxjQUFNLE9BQUEsS0FBSyxDQUFDLEdBQUcsQ0FBQyxjQUFjLENBQUMsRUFBekIsQ0FBeUIsQ0FBQztBQUNqRCxDQUFDO0FBRUQsTUFBTSxVQUFVLG1CQUFtQixDQUFDLElBQTRDO0lBRTlFLElBQU0sR0FBRyxHQUFHLGVBQWUsQ0FBQyxJQUFJLENBQUMsSUFBSSxlQUFlLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDM0QsSUFBSSxTQUFTLElBQUksQ0FBQyxHQUFHLEVBQUU7UUFDckIsTUFBTSxJQUFJLEtBQUssQ0FBQyxNQUFJLElBQUksQ0FBQyxJQUFJLHFEQUFrRCxDQUFDLENBQUM7S0FDbEY7SUFDRCxPQUFPLEdBQUssQ0FBQztBQUNmLENBQUM7QUFFRCxNQUFNLFVBQVUsY0FBYyxDQUFDLElBQW1CO0lBQ2hELElBQU0sR0FBRyxHQUFHLFVBQVUsQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUM3QixJQUFJLFNBQVMsSUFBSSxDQUFDLEdBQUcsRUFBRTtRQUNyQixNQUFNLElBQUksS0FBSyxDQUFDLE1BQUksSUFBSSxDQUFDLElBQUksMkJBQXdCLENBQUMsQ0FBQztLQUN4RDtJQUNELE9BQU8sR0FBSyxDQUFDO0FBQ2YsQ0FBQztBQUVEOztHQUVHO0FBQ0gsTUFBTSxVQUFVLGdCQUFnQixDQUFJLEdBd0JuQztJQUNDLElBQU0sR0FBRyxHQUFtQjtRQUMxQixJQUFJLEVBQUUsR0FBRyxDQUFDLElBQUk7UUFDZCxTQUFTLEVBQUUsR0FBRyxDQUFDLFNBQVMsSUFBSSxXQUFXO1FBQ3ZDLFlBQVksRUFBRSxHQUFHLENBQUMsWUFBWSxJQUFJLFdBQVc7UUFDN0MsT0FBTyxFQUFFLEdBQUcsQ0FBQyxPQUFPLElBQUksV0FBVztRQUNuQyxPQUFPLEVBQUUsR0FBRyxDQUFDLE9BQU8sSUFBSSxXQUFXO1FBQ25DLHVCQUF1QixFQUFFLElBQUk7UUFDN0IsT0FBTyxFQUFFLEdBQUcsQ0FBQyxPQUFPLElBQUksSUFBSTtRQUM1QixFQUFFLEVBQUUsR0FBRyxDQUFDLEVBQUUsSUFBSSxJQUFJO0tBQ25CLENBQUM7SUFDRixPQUFPLEdBQVksQ0FBQztBQUN0QixDQUFDO0FBRUQ7Ozs7Ozs7OztHQVNHO0FBQ0gsTUFBTSxVQUFVLGtCQUFrQixDQUFDLElBQVMsRUFBRSxLQVk3QztJQUNDLE9BQU8sYUFBYSxDQUFDO1FBQ25CLElBQU0sV0FBVyxHQUFHLGNBQWMsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLENBQUM7UUFDL0MsV0FBVyxDQUFDLFlBQVksR0FBRyxLQUFLLENBQUMsWUFBWSxJQUFJLFdBQVcsQ0FBQztRQUM3RCxXQUFXLENBQUMsT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLElBQUksV0FBVyxDQUFDO1FBQ25ELFdBQVcsQ0FBQyxPQUFPLEdBQUcsS0FBSyxDQUFDLE9BQU8sSUFBSSxXQUFXLENBQUM7SUFDckQsQ0FBQyxDQUFVLENBQUM7QUFDZCxDQUFDO0FBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztHQXNERztBQUNILFNBQVMsWUFBWSxDQUNqQixHQUFrRCxFQUNsRCxTQUFtQztJQUNyQyxJQUFJLEdBQUcsSUFBSSxJQUFJO1FBQUUsT0FBTyxTQUFnQixDQUFDO0lBQ3pDLElBQU0sU0FBUyxHQUFRLEVBQUUsQ0FBQztJQUMxQixLQUFLLElBQU0sV0FBVyxJQUFJLEdBQUcsRUFBRTtRQUM3QixJQUFJLEdBQUcsQ0FBQyxjQUFjLENBQUMsV0FBVyxDQUFDLEVBQUU7WUFDbkMsSUFBSSxVQUFVLEdBQTRCLEdBQUcsQ0FBQyxXQUFXLENBQUcsQ0FBQztZQUM3RCxJQUFJLFlBQVksR0FBRyxVQUFVLENBQUM7WUFDOUIsSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLFVBQVUsQ0FBQyxFQUFFO2dCQUM3QixZQUFZLEdBQUcsVUFBVSxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUM3QixVQUFVLEdBQUcsVUFBVSxDQUFDLENBQUMsQ0FBQyxDQUFDO2FBQzVCO1lBQ0QsU0FBUyxDQUFDLFVBQVUsQ0FBQyxHQUFHLFdBQVcsQ0FBQztZQUNwQyxJQUFJLFNBQVMsRUFBRTtnQkFDYixDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQUMsR0FBRyxZQUFzQixDQUFDLENBQUM7YUFDbEQ7U0FDRjtLQUNGO0lBQ0QsT0FBTyxTQUFTLENBQUM7QUFDbkIsQ0FBQztBQUVEOzs7Ozs7Ozs7Ozs7Ozs7R0FlRztBQUNILE1BQU0sVUFBVSxZQUFZLENBQUksY0F5RS9CO0lBQ0MsSUFBTSxjQUFjLEdBQTZCLEVBQVMsQ0FBQztJQUMzRCxPQUFPO1FBQ0wsTUFBTSxFQUFFLFlBQVksQ0FBSSxjQUFjLENBQUMsTUFBYSxFQUFFLGNBQWMsQ0FBQztRQUNyRSxjQUFjLEVBQUUsY0FBYztRQUM5QixPQUFPLEVBQUUsWUFBWSxDQUFJLGNBQWMsQ0FBQyxPQUFjLENBQUM7UUFDdkQsU0FBUyxFQUFFLGNBQWMsQ0FBQyxTQUFTLElBQUksSUFBSTtRQUMzQyxjQUFjLEVBQUUsY0FBYyxDQUFDLGNBQWMsSUFBSSxJQUFJO1FBQ3JELFlBQVksRUFBRSxjQUFjLENBQUMsWUFBWSxJQUFJLElBQUk7S0FDbEQsQ0FBQztBQUNKLENBQUM7QUFFRDs7Ozs7Ozs7Ozs7Ozs7O0dBZUc7QUFDSCxNQUFNLENBQUMsSUFBTSxpQkFBaUIsR0FBRyxpQkFvR3RCLENBQUM7QUFFWjs7Ozs7Ozs7Ozs7Ozs7O0dBZUc7QUFDSCxNQUFNLFVBQVUsWUFBWSxDQUFJLE9BWS9CO0lBQ0MsT0FBb0I7UUFDbEIsSUFBSSxFQUFFLE9BQU8sQ0FBQyxJQUFJO1FBQ2xCLE9BQU8sRUFBRSxPQUFPLENBQUMsT0FBTztRQUN4QixJQUFJLEVBQUUsT0FBTyxDQUFDLElBQUksS0FBSyxLQUFLO1FBQzVCLFNBQVMsRUFBRSxPQUFPLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxXQUFXLElBQUksSUFBSTtLQUM1QyxDQUFDO0FBQ2QsQ0FBQztBQUVEOzs7O0dBSUc7QUFFSCxNQUFNLFVBQVUsZUFBZSxDQUFJLElBQVM7SUFDMUMsT0FBUSxJQUFZLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxJQUFJLENBQUM7QUFDakQsQ0FBQztBQUVELE1BQU0sVUFBVSxlQUFlLENBQUksSUFBUztJQUMxQyxPQUFRLElBQVksQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLElBQUksQ0FBQztBQUNqRCxDQUFDO0FBRUQsTUFBTSxVQUFVLFVBQVUsQ0FBSSxJQUFTO0lBQ3JDLE9BQVEsSUFBWSxDQUFDLFdBQVcsQ0FBQyxJQUFJLElBQUksQ0FBQztBQUM1QyxDQUFDO0FBRUQsTUFBTSxVQUFVLFVBQVUsQ0FBSSxJQUFTO0lBQ3JDLE9BQVEsSUFBWSxDQUFDLFdBQVcsQ0FBQyxJQUFJLElBQUksQ0FBQztBQUM1QyxDQUFDO0FBSUQsTUFBTSxVQUFVLGNBQWMsQ0FBSSxJQUFTLEVBQUUsYUFBdUI7SUFDbEUsSUFBTSxXQUFXLEdBQUksSUFBWSxDQUFDLGFBQWEsQ0FBQyxJQUFJLElBQUksQ0FBQztJQUN6RCxJQUFJLENBQUMsV0FBVyxJQUFJLGFBQWEsS0FBSyxJQUFJLEVBQUU7UUFDMUMsTUFBTSxJQUFJLEtBQUssQ0FBQyxVQUFRLFNBQVMsQ0FBQyxJQUFJLENBQUMsMkNBQXdDLENBQUMsQ0FBQztLQUNsRjtJQUNELE9BQU8sV0FBVyxDQUFDO0FBQ3JCLENBQUM7QUFFRCxNQUFNLFVBQVUsZ0JBQWdCLENBQUMsSUFBUztJQUN4QyxPQUFRLElBQVksQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLElBQUksQ0FBQztBQUNqRCxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBAbGljZW5zZVxuICogQ29weXJpZ2h0IEdvb2dsZSBJbmMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogVXNlIG9mIHRoaXMgc291cmNlIGNvZGUgaXMgZ292ZXJuZWQgYnkgYW4gTUlULXN0eWxlIGxpY2Vuc2UgdGhhdCBjYW4gYmVcbiAqIGZvdW5kIGluIHRoZSBMSUNFTlNFIGZpbGUgYXQgaHR0cHM6Ly9hbmd1bGFyLmlvL2xpY2Vuc2VcbiAqL1xuXG5pbXBvcnQgJy4uL3V0aWwvbmdfZGV2X21vZGUnO1xuaW1wb3J0IHtDaGFuZ2VEZXRlY3Rpb25TdHJhdGVneX0gZnJvbSAnLi4vY2hhbmdlX2RldGVjdGlvbi9jb25zdGFudHMnO1xuaW1wb3J0IHtOR19JTkpFQ1RBQkxFX0RFRiwgybXJtWRlZmluZUluamVjdGFibGV9IGZyb20gJy4uL2RpL2ludGVyZmFjZS9kZWZzJztcbmltcG9ydCB7TXV0YWJsZSwgVHlwZX0gZnJvbSAnLi4vaW50ZXJmYWNlL3R5cGUnO1xuaW1wb3J0IHtOZ01vZHVsZURlZn0gZnJvbSAnLi4vbWV0YWRhdGEvbmdfbW9kdWxlJztcbmltcG9ydCB7U2NoZW1hTWV0YWRhdGF9IGZyb20gJy4uL21ldGFkYXRhL3NjaGVtYSc7XG5pbXBvcnQge1ZpZXdFbmNhcHN1bGF0aW9ufSBmcm9tICcuLi9tZXRhZGF0YS92aWV3JztcbmltcG9ydCB7bm9TaWRlRWZmZWN0c30gZnJvbSAnLi4vdXRpbC9jbG9zdXJlJztcbmltcG9ydCB7c3RyaW5naWZ5fSBmcm9tICcuLi91dGlsL3N0cmluZ2lmeSc7XG5pbXBvcnQge0VNUFRZX0FSUkFZLCBFTVBUWV9PQkp9IGZyb20gJy4vZW1wdHknO1xuaW1wb3J0IHtOR19CQVNFX0RFRiwgTkdfQ09NUE9ORU5UX0RFRiwgTkdfRElSRUNUSVZFX0RFRiwgTkdfTE9DQUxFX0lEX0RFRiwgTkdfTU9EVUxFX0RFRiwgTkdfUElQRV9ERUZ9IGZyb20gJy4vZmllbGRzJztcbmltcG9ydCB7Q29tcG9uZW50RGVmLCBDb21wb25lbnREZWZGZWF0dXJlLCBDb21wb25lbnRUZW1wbGF0ZSwgQ29tcG9uZW50VHlwZSwgQ29udGVudFF1ZXJpZXNGdW5jdGlvbiwgRGlyZWN0aXZlRGVmLCBEaXJlY3RpdmVEZWZGZWF0dXJlLCBEaXJlY3RpdmVUeXBlLCBEaXJlY3RpdmVUeXBlc09yRmFjdG9yeSwgRmFjdG9yeUZuLCBIb3N0QmluZGluZ3NGdW5jdGlvbiwgUGlwZURlZiwgUGlwZVR5cGUsIFBpcGVUeXBlc09yRmFjdG9yeSwgVmlld1F1ZXJpZXNGdW5jdGlvbiwgybXJtUJhc2VEZWZ9IGZyb20gJy4vaW50ZXJmYWNlcy9kZWZpbml0aW9uJztcbi8vIHdoaWxlIFNlbGVjdG9yRmxhZ3MgaXMgdW51c2VkIGhlcmUsIGl0J3MgcmVxdWlyZWQgc28gdGhhdCB0eXBlcyBkb24ndCBnZXQgcmVzb2x2ZWQgbGF6aWx5XG4vLyBzZWU6IGh0dHBzOi8vZ2l0aHViLmNvbS9NaWNyb3NvZnQvd2ViLWJ1aWxkLXRvb2xzL2lzc3Vlcy8xMDUwXG5pbXBvcnQge0Nzc1NlbGVjdG9yTGlzdCwgU2VsZWN0b3JGbGFnc30gZnJvbSAnLi9pbnRlcmZhY2VzL3Byb2plY3Rpb24nO1xuXG5sZXQgX3JlbmRlckNvbXBDb3VudCA9IDA7XG5cbi8qKlxuICogQ3JlYXRlIGEgY29tcG9uZW50IGRlZmluaXRpb24gb2JqZWN0LlxuICpcbiAqXG4gKiAjIEV4YW1wbGVcbiAqIGBgYFxuICogY2xhc3MgTXlEaXJlY3RpdmUge1xuICogICAvLyBHZW5lcmF0ZWQgYnkgQW5ndWxhciBUZW1wbGF0ZSBDb21waWxlclxuICogICAvLyBbU3ltYm9sXSBzeW50YXggd2lsbCBub3QgYmUgc3VwcG9ydGVkIGJ5IFR5cGVTY3JpcHQgdW50aWwgdjIuN1xuICogICBzdGF0aWMgbmdDb21wb25lbnREZWYgPSBkZWZpbmVDb21wb25lbnQoe1xuICogICAgIC4uLlxuICogICB9KTtcbiAqIH1cbiAqIGBgYFxuICogQGNvZGVHZW5BcGlcbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIMm1ybVkZWZpbmVDb21wb25lbnQ8VD4oY29tcG9uZW50RGVmaW5pdGlvbjoge1xuICAvKipcbiAgICogRGlyZWN0aXZlIHR5cGUsIG5lZWRlZCB0byBjb25maWd1cmUgdGhlIGluamVjdG9yLlxuICAgKi9cbiAgdHlwZTogVHlwZTxUPjtcblxuICAvKiogVGhlIHNlbGVjdG9ycyB0aGF0IHdpbGwgYmUgdXNlZCB0byBtYXRjaCBub2RlcyB0byB0aGlzIGNvbXBvbmVudC4gKi9cbiAgc2VsZWN0b3JzOiBDc3NTZWxlY3Rvckxpc3Q7XG5cbiAgLyoqXG4gICAqIEZhY3RvcnkgbWV0aG9kIHVzZWQgdG8gY3JlYXRlIGFuIGluc3RhbmNlIG9mIGRpcmVjdGl2ZS5cbiAgICovXG4gIGZhY3Rvcnk6IEZhY3RvcnlGbjxUPjtcblxuICAvKipcbiAgICogVGhlIG51bWJlciBvZiBub2RlcywgbG9jYWwgcmVmcywgYW5kIHBpcGVzIGluIHRoaXMgY29tcG9uZW50IHRlbXBsYXRlLlxuICAgKlxuICAgKiBVc2VkIHRvIGNhbGN1bGF0ZSB0aGUgbGVuZ3RoIG9mIHRoaXMgY29tcG9uZW50J3MgTFZpZXcgYXJyYXksIHNvIHdlXG4gICAqIGNhbiBwcmUtZmlsbCB0aGUgYXJyYXkgYW5kIHNldCB0aGUgYmluZGluZyBzdGFydCBpbmRleC5cbiAgICovXG4gIC8vIFRPRE8oa2FyYSk6IHJlbW92ZSBxdWVyaWVzIGZyb20gdGhpcyBjb3VudFxuICBjb25zdHM6IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIG51bWJlciBvZiBiaW5kaW5ncyBpbiB0aGlzIGNvbXBvbmVudCB0ZW1wbGF0ZSAoaW5jbHVkaW5nIHB1cmUgZm4gYmluZGluZ3MpLlxuICAgKlxuICAgKiBVc2VkIHRvIGNhbGN1bGF0ZSB0aGUgbGVuZ3RoIG9mIHRoaXMgY29tcG9uZW50J3MgTFZpZXcgYXJyYXksIHNvIHdlXG4gICAqIGNhbiBwcmUtZmlsbCB0aGUgYXJyYXkgYW5kIHNldCB0aGUgaG9zdCBiaW5kaW5nIHN0YXJ0IGluZGV4LlxuICAgKi9cbiAgdmFyczogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBBIG1hcCBvZiBpbnB1dCBuYW1lcy5cbiAgICpcbiAgICogVGhlIGZvcm1hdCBpcyBpbjogYHtbYWN0dWFsUHJvcGVydHlOYW1lOiBzdHJpbmddOihzdHJpbmd8W3N0cmluZywgc3RyaW5nXSl9YC5cbiAgICpcbiAgICogR2l2ZW46XG4gICAqIGBgYFxuICAgKiBjbGFzcyBNeUNvbXBvbmVudCB7XG4gICAqICAgQElucHV0KClcbiAgICogICBwdWJsaWNJbnB1dDE6IHN0cmluZztcbiAgICpcbiAgICogICBASW5wdXQoJ3B1YmxpY0lucHV0MicpXG4gICAqICAgZGVjbGFyZWRJbnB1dDI6IHN0cmluZztcbiAgICogfVxuICAgKiBgYGBcbiAgICpcbiAgICogaXMgZGVzY3JpYmVkIGFzOlxuICAgKiBgYGBcbiAgICoge1xuICAgKiAgIHB1YmxpY0lucHV0MTogJ3B1YmxpY0lucHV0MScsXG4gICAqICAgZGVjbGFyZWRJbnB1dDI6IFsncHVibGljSW5wdXQyJywgJ2RlY2xhcmVkSW5wdXQyJ10sXG4gICAqIH1cbiAgICogYGBgXG4gICAqXG4gICAqIFdoaWNoIHRoZSBtaW5pZmllciBtYXkgdHJhbnNsYXRlIHRvOlxuICAgKiBgYGBcbiAgICoge1xuICAgKiAgIG1pbmlmaWVkUHVibGljSW5wdXQxOiAncHVibGljSW5wdXQxJyxcbiAgICogICBtaW5pZmllZERlY2xhcmVkSW5wdXQyOiBbJ3B1YmxpY0lucHV0MicsICdkZWNsYXJlZElucHV0MiddLFxuICAgKiB9XG4gICAqIGBgYFxuICAgKlxuICAgKiBUaGlzIGFsbG93cyB0aGUgcmVuZGVyIHRvIHJlLWNvbnN0cnVjdCB0aGUgbWluaWZpZWQsIHB1YmxpYywgYW5kIGRlY2xhcmVkIG5hbWVzXG4gICAqIG9mIHByb3BlcnRpZXMuXG4gICAqXG4gICAqIE5PVEU6XG4gICAqICAtIEJlY2F1c2UgZGVjbGFyZWQgYW5kIHB1YmxpYyBuYW1lIGFyZSB1c3VhbGx5IHNhbWUgd2Ugb25seSBnZW5lcmF0ZSB0aGUgYXJyYXlcbiAgICogICAgYFsncHVibGljJywgJ2RlY2xhcmVkJ11gIGZvcm1hdCB3aGVuIHRoZXkgZGlmZmVyLlxuICAgKiAgLSBUaGUgcmVhc29uIHdoeSB0aGlzIEFQSSBhbmQgYG91dHB1dHNgIEFQSSBpcyBub3QgdGhlIHNhbWUgaXMgdGhhdCBgTmdPbkNoYW5nZXNgIGhhc1xuICAgKiAgICBpbmNvbnNpc3RlbnQgYmVoYXZpb3IgaW4gdGhhdCBpdCB1c2VzIGRlY2xhcmVkIG5hbWVzIHJhdGhlciB0aGFuIG1pbmlmaWVkIG9yIHB1YmxpYy4gRm9yXG4gICAqICAgIHRoaXMgcmVhc29uIGBOZ09uQ2hhbmdlc2Agd2lsbCBiZSBkZXByZWNhdGVkIGFuZCByZW1vdmVkIGluIGZ1dHVyZSB2ZXJzaW9uIGFuZCB0aGlzXG4gICAqICAgIEFQSSB3aWxsIGJlIHNpbXBsaWZpZWQgdG8gYmUgY29uc2lzdGVudCB3aXRoIGBvdXRwdXRgLlxuICAgKi9cbiAgaW5wdXRzPzoge1tQIGluIGtleW9mIFRdPzogc3RyaW5nIHwgW3N0cmluZywgc3RyaW5nXX07XG5cbiAgLyoqXG4gICAqIEEgbWFwIG9mIG91dHB1dCBuYW1lcy5cbiAgICpcbiAgICogVGhlIGZvcm1hdCBpcyBpbjogYHtbYWN0dWFsUHJvcGVydHlOYW1lOiBzdHJpbmddOnN0cmluZ31gLlxuICAgKlxuICAgKiBXaGljaCB0aGUgbWluaWZpZXIgbWF5IHRyYW5zbGF0ZSB0bzogYHtbbWluaWZpZWRQcm9wZXJ0eU5hbWU6IHN0cmluZ106c3RyaW5nfWAuXG4gICAqXG4gICAqIFRoaXMgYWxsb3dzIHRoZSByZW5kZXIgdG8gcmUtY29uc3RydWN0IHRoZSBtaW5pZmllZCBhbmQgbm9uLW1pbmlmaWVkIG5hbWVzXG4gICAqIG9mIHByb3BlcnRpZXMuXG4gICAqL1xuICBvdXRwdXRzPzoge1tQIGluIGtleW9mIFRdPzogc3RyaW5nfTtcblxuICAvKipcbiAgICogRnVuY3Rpb24gZXhlY3V0ZWQgYnkgdGhlIHBhcmVudCB0ZW1wbGF0ZSB0byBhbGxvdyBjaGlsZCBkaXJlY3RpdmUgdG8gYXBwbHkgaG9zdCBiaW5kaW5ncy5cbiAgICovXG4gIGhvc3RCaW5kaW5ncz86IEhvc3RCaW5kaW5nc0Z1bmN0aW9uPFQ+O1xuXG4gIC8qKlxuICAgKiBGdW5jdGlvbiB0byBjcmVhdGUgaW5zdGFuY2VzIG9mIGNvbnRlbnQgcXVlcmllcyBhc3NvY2lhdGVkIHdpdGggYSBnaXZlbiBkaXJlY3RpdmUuXG4gICAqL1xuICBjb250ZW50UXVlcmllcz86IENvbnRlbnRRdWVyaWVzRnVuY3Rpb248VD47XG5cbiAgLyoqXG4gICAqIERlZmluZXMgdGhlIG5hbWUgdGhhdCBjYW4gYmUgdXNlZCBpbiB0aGUgdGVtcGxhdGUgdG8gYXNzaWduIHRoaXMgZGlyZWN0aXZlIHRvIGEgdmFyaWFibGUuXG4gICAqXG4gICAqIFNlZToge0BsaW5rIERpcmVjdGl2ZS5leHBvcnRBc31cbiAgICovXG4gIGV4cG9ydEFzPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIFRlbXBsYXRlIGZ1bmN0aW9uIHVzZSBmb3IgcmVuZGVyaW5nIERPTS5cbiAgICpcbiAgICogVGhpcyBmdW5jdGlvbiBoYXMgZm9sbG93aW5nIHN0cnVjdHVyZS5cbiAgICpcbiAgICogYGBgXG4gICAqIGZ1bmN0aW9uIFRlbXBsYXRlPFQ+KGN0eDpULCBjcmVhdGlvbk1vZGU6IGJvb2xlYW4pIHtcbiAgICogICBpZiAoY3JlYXRpb25Nb2RlKSB7XG4gICAqICAgICAvLyBDb250YWlucyBjcmVhdGlvbiBtb2RlIGluc3RydWN0aW9ucy5cbiAgICogICB9XG4gICAqICAgLy8gQ29udGFpbnMgYmluZGluZyB1cGRhdGUgaW5zdHJ1Y3Rpb25zXG4gICAqIH1cbiAgICogYGBgXG4gICAqXG4gICAqIENvbW1vbiBpbnN0cnVjdGlvbnMgYXJlOlxuICAgKiBDcmVhdGlvbiBtb2RlIGluc3RydWN0aW9uczpcbiAgICogIC0gYGVsZW1lbnRTdGFydGAsIGBlbGVtZW50RW5kYFxuICAgKiAgLSBgdGV4dGBcbiAgICogIC0gYGNvbnRhaW5lcmBcbiAgICogIC0gYGxpc3RlbmVyYFxuICAgKlxuICAgKiBCaW5kaW5nIHVwZGF0ZSBpbnN0cnVjdGlvbnM6XG4gICAqIC0gYGJpbmRgXG4gICAqIC0gYGVsZW1lbnRBdHRyaWJ1dGVgXG4gICAqIC0gYGVsZW1lbnRQcm9wZXJ0eWBcbiAgICogLSBgZWxlbWVudENsYXNzYFxuICAgKiAtIGBlbGVtZW50U3R5bGVgXG4gICAqXG4gICAqL1xuICB0ZW1wbGF0ZTogQ29tcG9uZW50VGVtcGxhdGU8VD47XG5cbiAgLyoqXG4gICAqIEFuIGFycmF5IG9mIGBuZ0NvbnRlbnRbc2VsZWN0b3JdYCB2YWx1ZXMgdGhhdCB3ZXJlIGZvdW5kIGluIHRoZSB0ZW1wbGF0ZS5cbiAgICovXG4gIG5nQ29udGVudFNlbGVjdG9ycz86IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBBZGRpdGlvbmFsIHNldCBvZiBpbnN0cnVjdGlvbnMgc3BlY2lmaWMgdG8gdmlldyBxdWVyeSBwcm9jZXNzaW5nLiBUaGlzIGNvdWxkIGJlIHNlZW4gYXMgYVxuICAgKiBzZXQgb2YgaW5zdHJ1Y3Rpb24gdG8gYmUgaW5zZXJ0ZWQgaW50byB0aGUgdGVtcGxhdGUgZnVuY3Rpb24uXG4gICAqXG4gICAqIFF1ZXJ5LXJlbGF0ZWQgaW5zdHJ1Y3Rpb25zIG5lZWQgdG8gYmUgcHVsbGVkIG91dCB0byBhIHNwZWNpZmljIGZ1bmN0aW9uIGFzIGEgdGltaW5nIG9mXG4gICAqIGV4ZWN1dGlvbiBpcyBkaWZmZXJlbnQgYXMgY29tcGFyZWQgdG8gYWxsIG90aGVyIGluc3RydWN0aW9ucyAoYWZ0ZXIgY2hhbmdlIGRldGVjdGlvbiBob29rcyBidXRcbiAgICogYmVmb3JlIHZpZXcgaG9va3MpLlxuICAgKi9cbiAgdmlld1F1ZXJ5PzogVmlld1F1ZXJpZXNGdW5jdGlvbjxUPnwgbnVsbDtcblxuICAvKipcbiAgICogQSBsaXN0IG9mIG9wdGlvbmFsIGZlYXR1cmVzIHRvIGFwcGx5LlxuICAgKlxuICAgKiBTZWU6IHtAbGluayBOZ09uQ2hhbmdlc0ZlYXR1cmV9LCB7QGxpbmsgUHJvdmlkZXJzRmVhdHVyZX1cbiAgICovXG4gIGZlYXR1cmVzPzogQ29tcG9uZW50RGVmRmVhdHVyZVtdO1xuXG4gIC8qKlxuICAgKiBEZWZpbmVzIHRlbXBsYXRlIGFuZCBzdHlsZSBlbmNhcHN1bGF0aW9uIG9wdGlvbnMgYXZhaWxhYmxlIGZvciBDb21wb25lbnQncyB7QGxpbmsgQ29tcG9uZW50fS5cbiAgICovXG4gIGVuY2Fwc3VsYXRpb24/OiBWaWV3RW5jYXBzdWxhdGlvbjtcblxuICAvKipcbiAgICogRGVmaW5lcyBhcmJpdHJhcnkgZGV2ZWxvcGVyLWRlZmluZWQgZGF0YSB0byBiZSBzdG9yZWQgb24gYSByZW5kZXJlciBpbnN0YW5jZS5cbiAgICogVGhpcyBpcyB1c2VmdWwgZm9yIHJlbmRlcmVycyB0aGF0IGRlbGVnYXRlIHRvIG90aGVyIHJlbmRlcmVycy5cbiAgICpcbiAgICogc2VlOiBhbmltYXRpb25cbiAgICovXG4gIGRhdGE/OiB7W2tpbmQ6IHN0cmluZ106IGFueX07XG5cbiAgLyoqXG4gICAqIEEgc2V0IG9mIHN0eWxlcyB0aGF0IHRoZSBjb21wb25lbnQgbmVlZHMgdG8gYmUgcHJlc2VudCBmb3IgY29tcG9uZW50IHRvIHJlbmRlciBjb3JyZWN0bHkuXG4gICAqL1xuICBzdHlsZXM/OiBzdHJpbmdbXTtcblxuICAvKipcbiAgICogVGhlIHN0cmF0ZWd5IHRoYXQgdGhlIGRlZmF1bHQgY2hhbmdlIGRldGVjdG9yIHVzZXMgdG8gZGV0ZWN0IGNoYW5nZXMuXG4gICAqIFdoZW4gc2V0LCB0YWtlcyBlZmZlY3QgdGhlIG5leHQgdGltZSBjaGFuZ2UgZGV0ZWN0aW9uIGlzIHRyaWdnZXJlZC5cbiAgICovXG4gIGNoYW5nZURldGVjdGlvbj86IENoYW5nZURldGVjdGlvblN0cmF0ZWd5O1xuXG4gIC8qKlxuICAgKiBSZWdpc3RyeSBvZiBkaXJlY3RpdmVzIGFuZCBjb21wb25lbnRzIHRoYXQgbWF5IGJlIGZvdW5kIGluIHRoaXMgY29tcG9uZW50J3Mgdmlldy5cbiAgICpcbiAgICogVGhlIHByb3BlcnR5IGlzIGVpdGhlciBhbiBhcnJheSBvZiBgRGlyZWN0aXZlRGVmYHMgb3IgYSBmdW5jdGlvbiB3aGljaCByZXR1cm5zIHRoZSBhcnJheSBvZlxuICAgKiBgRGlyZWN0aXZlRGVmYHMuIFRoZSBmdW5jdGlvbiBpcyBuZWNlc3NhcnkgdG8gYmUgYWJsZSB0byBzdXBwb3J0IGZvcndhcmQgZGVjbGFyYXRpb25zLlxuICAgKi9cbiAgZGlyZWN0aXZlcz86IERpcmVjdGl2ZVR5cGVzT3JGYWN0b3J5IHwgbnVsbDtcblxuICAvKipcbiAgICogUmVnaXN0cnkgb2YgcGlwZXMgdGhhdCBtYXkgYmUgZm91bmQgaW4gdGhpcyBjb21wb25lbnQncyB2aWV3LlxuICAgKlxuICAgKiBUaGUgcHJvcGVydHkgaXMgZWl0aGVyIGFuIGFycmF5IG9mIGBQaXBlRGVmc2BzIG9yIGEgZnVuY3Rpb24gd2hpY2ggcmV0dXJucyB0aGUgYXJyYXkgb2ZcbiAgICogYFBpcGVEZWZzYHMuIFRoZSBmdW5jdGlvbiBpcyBuZWNlc3NhcnkgdG8gYmUgYWJsZSB0byBzdXBwb3J0IGZvcndhcmQgZGVjbGFyYXRpb25zLlxuICAgKi9cbiAgcGlwZXM/OiBQaXBlVHlwZXNPckZhY3RvcnkgfCBudWxsO1xuXG4gIC8qKlxuICAgKiBUaGUgc2V0IG9mIHNjaGVtYXMgdGhhdCBkZWNsYXJlIGVsZW1lbnRzIHRvIGJlIGFsbG93ZWQgaW4gdGhlIGNvbXBvbmVudCdzIHRlbXBsYXRlLlxuICAgKi9cbiAgc2NoZW1hcz86IFNjaGVtYU1ldGFkYXRhW10gfCBudWxsO1xufSk6IG5ldmVyIHtcbiAgY29uc3QgdHlwZSA9IGNvbXBvbmVudERlZmluaXRpb24udHlwZTtcbiAgY29uc3QgdHlwZVByb3RvdHlwZSA9IHR5cGUucHJvdG90eXBlO1xuICBjb25zdCBkZWNsYXJlZElucHV0czoge1trZXk6IHN0cmluZ106IHN0cmluZ30gPSB7fSBhcyBhbnk7XG4gIGNvbnN0IGRlZjogTXV0YWJsZTxDb21wb25lbnREZWY8YW55Piwga2V5b2YgQ29tcG9uZW50RGVmPGFueT4+ID0ge1xuICAgIHR5cGU6IHR5cGUsXG4gICAgcHJvdmlkZXJzUmVzb2x2ZXI6IG51bGwsXG4gICAgY29uc3RzOiBjb21wb25lbnREZWZpbml0aW9uLmNvbnN0cyxcbiAgICB2YXJzOiBjb21wb25lbnREZWZpbml0aW9uLnZhcnMsXG4gICAgZmFjdG9yeTogY29tcG9uZW50RGVmaW5pdGlvbi5mYWN0b3J5LFxuICAgIHRlbXBsYXRlOiBjb21wb25lbnREZWZpbml0aW9uLnRlbXBsYXRlIHx8IG51bGwgISxcbiAgICBuZ0NvbnRlbnRTZWxlY3RvcnM6IGNvbXBvbmVudERlZmluaXRpb24ubmdDb250ZW50U2VsZWN0b3JzLFxuICAgIGhvc3RCaW5kaW5nczogY29tcG9uZW50RGVmaW5pdGlvbi5ob3N0QmluZGluZ3MgfHwgbnVsbCxcbiAgICBjb250ZW50UXVlcmllczogY29tcG9uZW50RGVmaW5pdGlvbi5jb250ZW50UXVlcmllcyB8fCBudWxsLFxuICAgIGRlY2xhcmVkSW5wdXRzOiBkZWNsYXJlZElucHV0cyxcbiAgICBpbnB1dHM6IG51bGwgISwgICAvLyBhc3NpZ25lZCBpbiBub1NpZGVFZmZlY3RzXG4gICAgb3V0cHV0czogbnVsbCAhLCAgLy8gYXNzaWduZWQgaW4gbm9TaWRlRWZmZWN0c1xuICAgIGV4cG9ydEFzOiBjb21wb25lbnREZWZpbml0aW9uLmV4cG9ydEFzIHx8IG51bGwsXG4gICAgb25DaGFuZ2VzOiBudWxsLFxuICAgIG9uSW5pdDogdHlwZVByb3RvdHlwZS5uZ09uSW5pdCB8fCBudWxsLFxuICAgIGRvQ2hlY2s6IHR5cGVQcm90b3R5cGUubmdEb0NoZWNrIHx8IG51bGwsXG4gICAgYWZ0ZXJDb250ZW50SW5pdDogdHlwZVByb3RvdHlwZS5uZ0FmdGVyQ29udGVudEluaXQgfHwgbnVsbCxcbiAgICBhZnRlckNvbnRlbnRDaGVja2VkOiB0eXBlUHJvdG90eXBlLm5nQWZ0ZXJDb250ZW50Q2hlY2tlZCB8fCBudWxsLFxuICAgIGFmdGVyVmlld0luaXQ6IHR5cGVQcm90b3R5cGUubmdBZnRlclZpZXdJbml0IHx8IG51bGwsXG4gICAgYWZ0ZXJWaWV3Q2hlY2tlZDogdHlwZVByb3RvdHlwZS5uZ0FmdGVyVmlld0NoZWNrZWQgfHwgbnVsbCxcbiAgICBvbkRlc3Ryb3k6IHR5cGVQcm90b3R5cGUubmdPbkRlc3Ryb3kgfHwgbnVsbCxcbiAgICBvblB1c2g6IGNvbXBvbmVudERlZmluaXRpb24uY2hhbmdlRGV0ZWN0aW9uID09PSBDaGFuZ2VEZXRlY3Rpb25TdHJhdGVneS5PblB1c2gsXG4gICAgZGlyZWN0aXZlRGVmczogbnVsbCAhLCAgLy8gYXNzaWduZWQgaW4gbm9TaWRlRWZmZWN0c1xuICAgIHBpcGVEZWZzOiBudWxsICEsICAgICAgIC8vIGFzc2lnbmVkIGluIG5vU2lkZUVmZmVjdHNcbiAgICBzZWxlY3RvcnM6IGNvbXBvbmVudERlZmluaXRpb24uc2VsZWN0b3JzLFxuICAgIHZpZXdRdWVyeTogY29tcG9uZW50RGVmaW5pdGlvbi52aWV3UXVlcnkgfHwgbnVsbCxcbiAgICBmZWF0dXJlczogY29tcG9uZW50RGVmaW5pdGlvbi5mZWF0dXJlcyB8fCBudWxsLFxuICAgIGRhdGE6IGNvbXBvbmVudERlZmluaXRpb24uZGF0YSB8fCB7fSxcbiAgICAvLyBUT0RPKG1pc2tvKTogY29udmVydCBWaWV3RW5jYXBzdWxhdGlvbiBpbnRvIGNvbnN0IGVudW0gc28gdGhhdCBpdCBjYW4gYmUgdXNlZCBkaXJlY3RseSBpbiB0aGVcbiAgICAvLyBuZXh0IGxpbmUuIEFsc28gYE5vbmVgIHNob3VsZCBiZSAwIG5vdCAyLlxuICAgIGVuY2Fwc3VsYXRpb246IGNvbXBvbmVudERlZmluaXRpb24uZW5jYXBzdWxhdGlvbiB8fCBWaWV3RW5jYXBzdWxhdGlvbi5FbXVsYXRlZCxcbiAgICBpZDogJ2MnLFxuICAgIHN0eWxlczogY29tcG9uZW50RGVmaW5pdGlvbi5zdHlsZXMgfHwgRU1QVFlfQVJSQVksXG4gICAgXzogbnVsbCBhcyBuZXZlcixcbiAgICBzZXRJbnB1dDogbnVsbCxcbiAgICBzY2hlbWFzOiBjb21wb25lbnREZWZpbml0aW9uLnNjaGVtYXMgfHwgbnVsbCxcbiAgICB0VmlldzogbnVsbCxcbiAgfTtcbiAgZGVmLl8gPSBub1NpZGVFZmZlY3RzKCgpID0+IHtcbiAgICBjb25zdCBkaXJlY3RpdmVUeXBlcyA9IGNvbXBvbmVudERlZmluaXRpb24uZGlyZWN0aXZlcyAhO1xuICAgIGNvbnN0IGZlYXR1cmUgPSBjb21wb25lbnREZWZpbml0aW9uLmZlYXR1cmVzO1xuICAgIGNvbnN0IHBpcGVUeXBlcyA9IGNvbXBvbmVudERlZmluaXRpb24ucGlwZXMgITtcbiAgICBkZWYuaWQgKz0gX3JlbmRlckNvbXBDb3VudCsrO1xuICAgIGRlZi5pbnB1dHMgPSBpbnZlcnRPYmplY3QoY29tcG9uZW50RGVmaW5pdGlvbi5pbnB1dHMsIGRlY2xhcmVkSW5wdXRzKSxcbiAgICBkZWYub3V0cHV0cyA9IGludmVydE9iamVjdChjb21wb25lbnREZWZpbml0aW9uLm91dHB1dHMpLFxuICAgIGZlYXR1cmUgJiYgZmVhdHVyZS5mb3JFYWNoKChmbikgPT4gZm4oZGVmKSk7XG4gICAgZGVmLmRpcmVjdGl2ZURlZnMgPSBkaXJlY3RpdmVUeXBlcyA/XG4gICAgICAgICgpID0+ICh0eXBlb2YgZGlyZWN0aXZlVHlwZXMgPT09ICdmdW5jdGlvbicgPyBkaXJlY3RpdmVUeXBlcygpIDogZGlyZWN0aXZlVHlwZXMpXG4gICAgICAgICAgICAgICAgICAubWFwKGV4dHJhY3REaXJlY3RpdmVEZWYpIDpcbiAgICAgICAgbnVsbDtcbiAgICBkZWYucGlwZURlZnMgPSBwaXBlVHlwZXMgP1xuICAgICAgICAoKSA9PiAodHlwZW9mIHBpcGVUeXBlcyA9PT0gJ2Z1bmN0aW9uJyA/IHBpcGVUeXBlcygpIDogcGlwZVR5cGVzKS5tYXAoZXh0cmFjdFBpcGVEZWYpIDpcbiAgICAgICAgbnVsbDtcblxuICAgIC8vIEFkZCBuZ0luamVjdGFibGVEZWYgc28gY29tcG9uZW50cyBhcmUgcmVhY2hhYmxlIHRocm91Z2ggdGhlIG1vZHVsZSBpbmplY3RvciBieSBkZWZhdWx0XG4gICAgLy8gKHVubGVzcyBpdCBoYXMgYWxyZWFkeSBiZWVuIHNldCBieSB0aGUgQEluamVjdGFibGUgZGVjb3JhdG9yKS4gVGhpcyBpcyBtb3N0bHkgdG9cbiAgICAvLyBzdXBwb3J0IGluamVjdGluZyBjb21wb25lbnRzIGluIHRlc3RzLiBJbiByZWFsIGFwcGxpY2F0aW9uIGNvZGUsIGNvbXBvbmVudHMgc2hvdWxkXG4gICAgLy8gYmUgcmV0cmlldmVkIHRocm91Z2ggdGhlIG5vZGUgaW5qZWN0b3IsIHNvIHRoaXMgaXNuJ3QgYSBwcm9ibGVtLlxuICAgIGlmICghdHlwZS5oYXNPd25Qcm9wZXJ0eShOR19JTkpFQ1RBQkxFX0RFRikpIHtcbiAgICAgICh0eXBlIGFzIGFueSlbTkdfSU5KRUNUQUJMRV9ERUZdID1cbiAgICAgICAgICDJtcm1ZGVmaW5lSW5qZWN0YWJsZTxUPih7dG9rZW46IHR5cGUsIGZhY3Rvcnk6IGNvbXBvbmVudERlZmluaXRpb24uZmFjdG9yeSBhcygpID0+IFR9KTtcbiAgICB9XG4gIH0pIGFzIG5ldmVyO1xuXG4gIHJldHVybiBkZWYgYXMgbmV2ZXI7XG59XG5cbi8qKlxuICogQGNvZGVHZW5BcGlcbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIMm1ybVzZXRDb21wb25lbnRTY29wZShcbiAgICB0eXBlOiBDb21wb25lbnRUeXBlPGFueT4sIGRpcmVjdGl2ZXM6IFR5cGU8YW55PltdLCBwaXBlczogVHlwZTxhbnk+W10pOiB2b2lkIHtcbiAgY29uc3QgZGVmID0gKHR5cGUubmdDb21wb25lbnREZWYgYXMgQ29tcG9uZW50RGVmPGFueT4pO1xuICBkZWYuZGlyZWN0aXZlRGVmcyA9ICgpID0+IGRpcmVjdGl2ZXMubWFwKGV4dHJhY3REaXJlY3RpdmVEZWYpO1xuICBkZWYucGlwZURlZnMgPSAoKSA9PiBwaXBlcy5tYXAoZXh0cmFjdFBpcGVEZWYpO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gZXh0cmFjdERpcmVjdGl2ZURlZih0eXBlOiBEaXJlY3RpdmVUeXBlPGFueT4mIENvbXBvbmVudFR5cGU8YW55Pik6XG4gICAgRGlyZWN0aXZlRGVmPGFueT58Q29tcG9uZW50RGVmPGFueT4ge1xuICBjb25zdCBkZWYgPSBnZXRDb21wb25lbnREZWYodHlwZSkgfHwgZ2V0RGlyZWN0aXZlRGVmKHR5cGUpO1xuICBpZiAobmdEZXZNb2RlICYmICFkZWYpIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoYCcke3R5cGUubmFtZX0nIGlzIG5laXRoZXIgJ0NvbXBvbmVudFR5cGUnIG9yICdEaXJlY3RpdmVUeXBlJy5gKTtcbiAgfVxuICByZXR1cm4gZGVmICE7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBleHRyYWN0UGlwZURlZih0eXBlOiBQaXBlVHlwZTxhbnk+KTogUGlwZURlZjxhbnk+IHtcbiAgY29uc3QgZGVmID0gZ2V0UGlwZURlZih0eXBlKTtcbiAgaWYgKG5nRGV2TW9kZSAmJiAhZGVmKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKGAnJHt0eXBlLm5hbWV9JyBpcyBub3QgYSAnUGlwZVR5cGUnLmApO1xuICB9XG4gIHJldHVybiBkZWYgITtcbn1cblxuLyoqXG4gKiBAY29kZUdlbkFwaVxuICovXG5leHBvcnQgZnVuY3Rpb24gybXJtWRlZmluZU5nTW9kdWxlPFQ+KGRlZjoge1xuICAvKiogVG9rZW4gcmVwcmVzZW50aW5nIHRoZSBtb2R1bGUuIFVzZWQgYnkgREkuICovXG4gIHR5cGU6IFQ7XG5cbiAgLyoqIExpc3Qgb2YgY29tcG9uZW50cyB0byBib290c3RyYXAuICovXG4gIGJvb3RzdHJhcD86IFR5cGU8YW55PltdIHwgKCgpID0+IFR5cGU8YW55PltdKTtcblxuICAvKiogTGlzdCBvZiBjb21wb25lbnRzLCBkaXJlY3RpdmVzLCBhbmQgcGlwZXMgZGVjbGFyZWQgYnkgdGhpcyBtb2R1bGUuICovXG4gIGRlY2xhcmF0aW9ucz86IFR5cGU8YW55PltdIHwgKCgpID0+IFR5cGU8YW55PltdKTtcblxuICAvKiogTGlzdCBvZiBtb2R1bGVzIG9yIGBNb2R1bGVXaXRoUHJvdmlkZXJzYCBpbXBvcnRlZCBieSB0aGlzIG1vZHVsZS4gKi9cbiAgaW1wb3J0cz86IFR5cGU8YW55PltdIHwgKCgpID0+IFR5cGU8YW55PltdKTtcblxuICAvKipcbiAgICogTGlzdCBvZiBtb2R1bGVzLCBgTW9kdWxlV2l0aFByb3ZpZGVyc2AsIGNvbXBvbmVudHMsIGRpcmVjdGl2ZXMsIG9yIHBpcGVzIGV4cG9ydGVkIGJ5IHRoaXNcbiAgICogbW9kdWxlLlxuICAgKi9cbiAgZXhwb3J0cz86IFR5cGU8YW55PltdIHwgKCgpID0+IFR5cGU8YW55PltdKTtcblxuICAvKiogVGhlIHNldCBvZiBzY2hlbWFzIHRoYXQgZGVjbGFyZSBlbGVtZW50cyB0byBiZSBhbGxvd2VkIGluIHRoZSBOZ01vZHVsZS4gKi9cbiAgc2NoZW1hcz86IFNjaGVtYU1ldGFkYXRhW10gfCBudWxsO1xuXG4gIC8qKiBVbmlxdWUgSUQgZm9yIHRoZSBtb2R1bGUgdGhhdCBpcyB1c2VkIHdpdGggYGdldE1vZHVsZUZhY3RvcnlgLiAqL1xuICBpZD86IHN0cmluZyB8IG51bGw7XG59KTogbmV2ZXIge1xuICBjb25zdCByZXM6IE5nTW9kdWxlRGVmPFQ+ID0ge1xuICAgIHR5cGU6IGRlZi50eXBlLFxuICAgIGJvb3RzdHJhcDogZGVmLmJvb3RzdHJhcCB8fCBFTVBUWV9BUlJBWSxcbiAgICBkZWNsYXJhdGlvbnM6IGRlZi5kZWNsYXJhdGlvbnMgfHwgRU1QVFlfQVJSQVksXG4gICAgaW1wb3J0czogZGVmLmltcG9ydHMgfHwgRU1QVFlfQVJSQVksXG4gICAgZXhwb3J0czogZGVmLmV4cG9ydHMgfHwgRU1QVFlfQVJSQVksXG4gICAgdHJhbnNpdGl2ZUNvbXBpbGVTY29wZXM6IG51bGwsXG4gICAgc2NoZW1hczogZGVmLnNjaGVtYXMgfHwgbnVsbCxcbiAgICBpZDogZGVmLmlkIHx8IG51bGwsXG4gIH07XG4gIHJldHVybiByZXMgYXMgbmV2ZXI7XG59XG5cbi8qKlxuICogQWRkcyB0aGUgbW9kdWxlIG1ldGFkYXRhIHRoYXQgaXMgbmVjZXNzYXJ5IHRvIGNvbXB1dGUgdGhlIG1vZHVsZSdzIHRyYW5zaXRpdmUgc2NvcGUgdG8gYW5cbiAqIGV4aXN0aW5nIG1vZHVsZSBkZWZpbml0aW9uLlxuICpcbiAqIFNjb3BlIG1ldGFkYXRhIG9mIG1vZHVsZXMgaXMgbm90IHVzZWQgaW4gcHJvZHVjdGlvbiBidWlsZHMsIHNvIGNhbGxzIHRvIHRoaXMgZnVuY3Rpb24gY2FuIGJlXG4gKiBtYXJrZWQgcHVyZSB0byB0cmVlLXNoYWtlIGl0IGZyb20gdGhlIGJ1bmRsZSwgYWxsb3dpbmcgZm9yIGFsbCByZWZlcmVuY2VkIGRlY2xhcmF0aW9uc1xuICogdG8gYmVjb21lIGVsaWdpYmxlIGZvciB0cmVlLXNoYWtpbmcgYXMgd2VsbC5cbiAqXG4gKiBAY29kZUdlbkFwaVxuICovXG5leHBvcnQgZnVuY3Rpb24gybXJtXNldE5nTW9kdWxlU2NvcGUodHlwZTogYW55LCBzY29wZToge1xuICAvKiogTGlzdCBvZiBjb21wb25lbnRzLCBkaXJlY3RpdmVzLCBhbmQgcGlwZXMgZGVjbGFyZWQgYnkgdGhpcyBtb2R1bGUuICovXG4gIGRlY2xhcmF0aW9ucz86IFR5cGU8YW55PltdIHwgKCgpID0+IFR5cGU8YW55PltdKTtcblxuICAvKiogTGlzdCBvZiBtb2R1bGVzIG9yIGBNb2R1bGVXaXRoUHJvdmlkZXJzYCBpbXBvcnRlZCBieSB0aGlzIG1vZHVsZS4gKi9cbiAgaW1wb3J0cz86IFR5cGU8YW55PltdIHwgKCgpID0+IFR5cGU8YW55PltdKTtcblxuICAvKipcbiAgICogTGlzdCBvZiBtb2R1bGVzLCBgTW9kdWxlV2l0aFByb3ZpZGVyc2AsIGNvbXBvbmVudHMsIGRpcmVjdGl2ZXMsIG9yIHBpcGVzIGV4cG9ydGVkIGJ5IHRoaXNcbiAgICogbW9kdWxlLlxuICAgKi9cbiAgZXhwb3J0cz86IFR5cGU8YW55PltdIHwgKCgpID0+IFR5cGU8YW55PltdKTtcbn0pOiB2b2lkIHtcbiAgcmV0dXJuIG5vU2lkZUVmZmVjdHMoKCkgPT4ge1xuICAgIGNvbnN0IG5nTW9kdWxlRGVmID0gZ2V0TmdNb2R1bGVEZWYodHlwZSwgdHJ1ZSk7XG4gICAgbmdNb2R1bGVEZWYuZGVjbGFyYXRpb25zID0gc2NvcGUuZGVjbGFyYXRpb25zIHx8IEVNUFRZX0FSUkFZO1xuICAgIG5nTW9kdWxlRGVmLmltcG9ydHMgPSBzY29wZS5pbXBvcnRzIHx8IEVNUFRZX0FSUkFZO1xuICAgIG5nTW9kdWxlRGVmLmV4cG9ydHMgPSBzY29wZS5leHBvcnRzIHx8IEVNUFRZX0FSUkFZO1xuICB9KSBhcyBuZXZlcjtcbn1cblxuLyoqXG4gKiBJbnZlcnRzIGFuIGlucHV0cyBvciBvdXRwdXRzIGxvb2t1cCBzdWNoIHRoYXQgdGhlIGtleXMsIHdoaWNoIHdlcmUgdGhlXG4gKiBtaW5pZmllZCBrZXlzLCBhcmUgcGFydCBvZiB0aGUgdmFsdWVzLCBhbmQgdGhlIHZhbHVlcyBhcmUgcGFyc2VkIHNvIHRoYXRcbiAqIHRoZSBwdWJsaWNOYW1lIG9mIHRoZSBwcm9wZXJ0eSBpcyB0aGUgbmV3IGtleVxuICpcbiAqIGUuZy4gZm9yXG4gKlxuICogYGBgXG4gKiBjbGFzcyBDb21wIHtcbiAqICAgQElucHV0KClcbiAqICAgcHJvcE5hbWUxOiBzdHJpbmc7XG4gKlxuICogICBASW5wdXQoJ3B1YmxpY05hbWUyJylcbiAqICAgZGVjbGFyZWRQcm9wTmFtZTI6IG51bWJlcjtcbiAqIH1cbiAqIGBgYFxuICpcbiAqIHdpbGwgYmUgc2VyaWFsaXplZCBhc1xuICpcbiAqIGBgYFxuICoge1xuICogICBwcm9wTmFtZTE6ICdwcm9wTmFtZTEnLFxuICogICBkZWNsYXJlZFByb3BOYW1lMjogWydwdWJsaWNOYW1lMicsICdkZWNsYXJlZFByb3BOYW1lMiddLFxuICogfVxuICogYGBgXG4gKlxuICogd2hpY2ggaXMgdGhhbiB0cmFuc2xhdGVkIGJ5IHRoZSBtaW5pZmllciBhczpcbiAqXG4gKiBgYGBcbiAqIHtcbiAqICAgbWluaWZpZWRQcm9wTmFtZTE6ICdwcm9wTmFtZTEnLFxuICogICBtaW5pZmllZFByb3BOYW1lMjogWydwdWJsaWNOYW1lMicsICdkZWNsYXJlZFByb3BOYW1lMiddLFxuICogfVxuICogYGBgXG4gKlxuICogYmVjb21lczogKHB1YmxpYyBuYW1lID0+IG1pbmlmaWVkTmFtZSlcbiAqXG4gKiBgYGBcbiAqIHtcbiAqICAncHJvcE5hbWUxJzogJ21pbmlmaWVkUHJvcE5hbWUxJyxcbiAqICAncHVibGljTmFtZTInOiAnbWluaWZpZWRQcm9wTmFtZTInLFxuICogfVxuICogYGBgXG4gKlxuICogT3B0aW9uYWxseSB0aGUgZnVuY3Rpb24gY2FuIHRha2UgYHNlY29uZGFyeWAgd2hpY2ggd2lsbCByZXN1bHQgaW46IChwdWJsaWMgbmFtZSA9PiBkZWNsYXJlZCBuYW1lKVxuICpcbiAqIGBgYFxuICoge1xuICogICdwcm9wTmFtZTEnOiAncHJvcE5hbWUxJyxcbiAqICAncHVibGljTmFtZTInOiAnZGVjbGFyZWRQcm9wTmFtZTInLFxuICogfVxuICogYGBgXG4gKlxuXG4gKi9cbmZ1bmN0aW9uIGludmVydE9iamVjdDxUPihcbiAgICBvYmo/OiB7W1AgaW4ga2V5b2YgVF0/OiBzdHJpbmcgfCBbc3RyaW5nLCBzdHJpbmddfSxcbiAgICBzZWNvbmRhcnk/OiB7W2tleTogc3RyaW5nXTogc3RyaW5nfSk6IHtbUCBpbiBrZXlvZiBUXTogc3RyaW5nfSB7XG4gIGlmIChvYmogPT0gbnVsbCkgcmV0dXJuIEVNUFRZX09CSiBhcyBhbnk7XG4gIGNvbnN0IG5ld0xvb2t1cDogYW55ID0ge307XG4gIGZvciAoY29uc3QgbWluaWZpZWRLZXkgaW4gb2JqKSB7XG4gICAgaWYgKG9iai5oYXNPd25Qcm9wZXJ0eShtaW5pZmllZEtleSkpIHtcbiAgICAgIGxldCBwdWJsaWNOYW1lOiBzdHJpbmd8W3N0cmluZywgc3RyaW5nXSA9IG9ialttaW5pZmllZEtleV0gITtcbiAgICAgIGxldCBkZWNsYXJlZE5hbWUgPSBwdWJsaWNOYW1lO1xuICAgICAgaWYgKEFycmF5LmlzQXJyYXkocHVibGljTmFtZSkpIHtcbiAgICAgICAgZGVjbGFyZWROYW1lID0gcHVibGljTmFtZVsxXTtcbiAgICAgICAgcHVibGljTmFtZSA9IHB1YmxpY05hbWVbMF07XG4gICAgICB9XG4gICAgICBuZXdMb29rdXBbcHVibGljTmFtZV0gPSBtaW5pZmllZEtleTtcbiAgICAgIGlmIChzZWNvbmRhcnkpIHtcbiAgICAgICAgKHNlY29uZGFyeVtwdWJsaWNOYW1lXSA9IGRlY2xhcmVkTmFtZSBhcyBzdHJpbmcpO1xuICAgICAgfVxuICAgIH1cbiAgfVxuICByZXR1cm4gbmV3TG9va3VwO1xufVxuXG4vKipcbiAqIENyZWF0ZSBhIGJhc2UgZGVmaW5pdGlvblxuICpcbiAqICMgRXhhbXBsZVxuICogYGBgdHNcbiAqIGNsYXNzIFNob3VsZEJlSW5oZXJpdGVkIHtcbiAqICAgc3RhdGljIG5nQmFzZURlZiA9IMm1ybVkZWZpbmVCYXNlKHtcbiAqICAgICAgLi4uXG4gKiAgIH0pXG4gKiB9XG4gKiBgYGBcbiAqXG4gKiBAcGFyYW0gYmFzZURlZmluaXRpb24gVGhlIGJhc2UgZGVmaW5pdGlvbiBwYXJhbWV0ZXJzXG4gKlxuICogQGNvZGVHZW5BcGlcbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIMm1ybVkZWZpbmVCYXNlPFQ+KGJhc2VEZWZpbml0aW9uOiB7XG4gIC8qKlxuICAgKiBBIG1hcCBvZiBpbnB1dCBuYW1lcy5cbiAgICpcbiAgICogVGhlIGZvcm1hdCBpcyBpbjogYHtbYWN0dWFsUHJvcGVydHlOYW1lOiBzdHJpbmddOihzdHJpbmd8W3N0cmluZywgc3RyaW5nXSl9YC5cbiAgICpcbiAgICogR2l2ZW46XG4gICAqIGBgYFxuICAgKiBjbGFzcyBNeUNvbXBvbmVudCB7XG4gICAqICAgQElucHV0KClcbiAgICogICBwdWJsaWNJbnB1dDE6IHN0cmluZztcbiAgICpcbiAgICogICBASW5wdXQoJ3B1YmxpY0lucHV0MicpXG4gICAqICAgZGVjbGFyZWRJbnB1dDI6IHN0cmluZztcbiAgICogfVxuICAgKiBgYGBcbiAgICpcbiAgICogaXMgZGVzY3JpYmVkIGFzOlxuICAgKiBgYGBcbiAgICoge1xuICAgKiAgIHB1YmxpY0lucHV0MTogJ3B1YmxpY0lucHV0MScsXG4gICAqICAgZGVjbGFyZWRJbnB1dDI6IFsnZGVjbGFyZWRJbnB1dDInLCAncHVibGljSW5wdXQyJ10sXG4gICAqIH1cbiAgICogYGBgXG4gICAqXG4gICAqIFdoaWNoIHRoZSBtaW5pZmllciBtYXkgdHJhbnNsYXRlIHRvOlxuICAgKiBgYGBcbiAgICoge1xuICAgKiAgIG1pbmlmaWVkUHVibGljSW5wdXQxOiAncHVibGljSW5wdXQxJyxcbiAgICogICBtaW5pZmllZERlY2xhcmVkSW5wdXQyOiBbICdkZWNsYXJlZElucHV0MicsICdwdWJsaWNJbnB1dDInXSxcbiAgICogfVxuICAgKiBgYGBcbiAgICpcbiAgICogVGhpcyBhbGxvd3MgdGhlIHJlbmRlciB0byByZS1jb25zdHJ1Y3QgdGhlIG1pbmlmaWVkLCBwdWJsaWMsIGFuZCBkZWNsYXJlZCBuYW1lc1xuICAgKiBvZiBwcm9wZXJ0aWVzLlxuICAgKlxuICAgKiBOT1RFOlxuICAgKiAgLSBCZWNhdXNlIGRlY2xhcmVkIGFuZCBwdWJsaWMgbmFtZSBhcmUgdXN1YWxseSBzYW1lIHdlIG9ubHkgZ2VuZXJhdGUgdGhlIGFycmF5XG4gICAqICAgIGBbJ2RlY2xhcmVkJywgJ3B1YmxpYyddYCBmb3JtYXQgd2hlbiB0aGV5IGRpZmZlci5cbiAgICogIC0gVGhlIHJlYXNvbiB3aHkgdGhpcyBBUEkgYW5kIGBvdXRwdXRzYCBBUEkgaXMgbm90IHRoZSBzYW1lIGlzIHRoYXQgYE5nT25DaGFuZ2VzYCBoYXNcbiAgICogICAgaW5jb25zaXN0ZW50IGJlaGF2aW9yIGluIHRoYXQgaXQgdXNlcyBkZWNsYXJlZCBuYW1lcyByYXRoZXIgdGhhbiBtaW5pZmllZCBvciBwdWJsaWMuIEZvclxuICAgKiAgICB0aGlzIHJlYXNvbiBgTmdPbkNoYW5nZXNgIHdpbGwgYmUgZGVwcmVjYXRlZCBhbmQgcmVtb3ZlZCBpbiBmdXR1cmUgdmVyc2lvbiBhbmQgdGhpc1xuICAgKiAgICBBUEkgd2lsbCBiZSBzaW1wbGlmaWVkIHRvIGJlIGNvbnNpc3RlbnQgd2l0aCBgb3V0cHV0c2AuXG4gICAqL1xuICBpbnB1dHM/OiB7W1AgaW4ga2V5b2YgVF0/OiBzdHJpbmcgfCBbc3RyaW5nLCBzdHJpbmddfTtcblxuICAvKipcbiAgICogQSBtYXAgb2Ygb3V0cHV0IG5hbWVzLlxuICAgKlxuICAgKiBUaGUgZm9ybWF0IGlzIGluOiBge1thY3R1YWxQcm9wZXJ0eU5hbWU6IHN0cmluZ106c3RyaW5nfWAuXG4gICAqXG4gICAqIFdoaWNoIHRoZSBtaW5pZmllciBtYXkgdHJhbnNsYXRlIHRvOiBge1ttaW5pZmllZFByb3BlcnR5TmFtZTogc3RyaW5nXTpzdHJpbmd9YC5cbiAgICpcbiAgICogVGhpcyBhbGxvd3MgdGhlIHJlbmRlciB0byByZS1jb25zdHJ1Y3QgdGhlIG1pbmlmaWVkIGFuZCBub24tbWluaWZpZWQgbmFtZXNcbiAgICogb2YgcHJvcGVydGllcy5cbiAgICovXG4gIG91dHB1dHM/OiB7W1AgaW4ga2V5b2YgVF0/OiBzdHJpbmd9O1xuXG4gIC8qKlxuICAgKiBGdW5jdGlvbiB0byBjcmVhdGUgaW5zdGFuY2VzIG9mIGNvbnRlbnQgcXVlcmllcyBhc3NvY2lhdGVkIHdpdGggYSBnaXZlbiBkaXJlY3RpdmUuXG4gICAqL1xuICBjb250ZW50UXVlcmllcz86IENvbnRlbnRRdWVyaWVzRnVuY3Rpb248VD58IG51bGw7XG5cbiAgLyoqXG4gICAqIEFkZGl0aW9uYWwgc2V0IG9mIGluc3RydWN0aW9ucyBzcGVjaWZpYyB0byB2aWV3IHF1ZXJ5IHByb2Nlc3NpbmcuIFRoaXMgY291bGQgYmUgc2VlbiBhcyBhXG4gICAqIHNldCBvZiBpbnN0cnVjdGlvbnMgdG8gYmUgaW5zZXJ0ZWQgaW50byB0aGUgdGVtcGxhdGUgZnVuY3Rpb24uXG4gICAqL1xuICB2aWV3UXVlcnk/OiBWaWV3UXVlcmllc0Z1bmN0aW9uPFQ+fCBudWxsO1xuXG4gIC8qKlxuICAgKiBGdW5jdGlvbiBleGVjdXRlZCBieSB0aGUgcGFyZW50IHRlbXBsYXRlIHRvIGFsbG93IGNoaWxkcmVuIHRvIGFwcGx5IGhvc3QgYmluZGluZ3MuXG4gICAqL1xuICBob3N0QmluZGluZ3M/OiBIb3N0QmluZGluZ3NGdW5jdGlvbjxUPjtcbn0pOiDJtcm1QmFzZURlZjxUPiB7XG4gIGNvbnN0IGRlY2xhcmVkSW5wdXRzOiB7W1AgaW4ga2V5b2YgVF06IHN0cmluZ30gPSB7fSBhcyBhbnk7XG4gIHJldHVybiB7XG4gICAgaW5wdXRzOiBpbnZlcnRPYmplY3Q8VD4oYmFzZURlZmluaXRpb24uaW5wdXRzIGFzIGFueSwgZGVjbGFyZWRJbnB1dHMpLFxuICAgIGRlY2xhcmVkSW5wdXRzOiBkZWNsYXJlZElucHV0cyxcbiAgICBvdXRwdXRzOiBpbnZlcnRPYmplY3Q8VD4oYmFzZURlZmluaXRpb24ub3V0cHV0cyBhcyBhbnkpLFxuICAgIHZpZXdRdWVyeTogYmFzZURlZmluaXRpb24udmlld1F1ZXJ5IHx8IG51bGwsXG4gICAgY29udGVudFF1ZXJpZXM6IGJhc2VEZWZpbml0aW9uLmNvbnRlbnRRdWVyaWVzIHx8IG51bGwsXG4gICAgaG9zdEJpbmRpbmdzOiBiYXNlRGVmaW5pdGlvbi5ob3N0QmluZGluZ3MgfHwgbnVsbFxuICB9O1xufVxuXG4vKipcbiAqIENyZWF0ZSBhIGRpcmVjdGl2ZSBkZWZpbml0aW9uIG9iamVjdC5cbiAqXG4gKiAjIEV4YW1wbGVcbiAqIGBgYHRzXG4gKiBjbGFzcyBNeURpcmVjdGl2ZSB7XG4gKiAgIC8vIEdlbmVyYXRlZCBieSBBbmd1bGFyIFRlbXBsYXRlIENvbXBpbGVyXG4gKiAgIC8vIFtTeW1ib2xdIHN5bnRheCB3aWxsIG5vdCBiZSBzdXBwb3J0ZWQgYnkgVHlwZVNjcmlwdCB1bnRpbCB2Mi43XG4gKiAgIHN0YXRpYyBuZ0RpcmVjdGl2ZURlZiA9IMm1ybVkZWZpbmVEaXJlY3RpdmUoe1xuICogICAgIC4uLlxuICogICB9KTtcbiAqIH1cbiAqIGBgYFxuICpcbiAqIEBjb2RlR2VuQXBpXG4gKi9cbmV4cG9ydCBjb25zdCDJtcm1ZGVmaW5lRGlyZWN0aXZlID0gybXJtWRlZmluZUNvbXBvbmVudCBhcyBhbnkgYXM8VD4oZGlyZWN0aXZlRGVmaW5pdGlvbjoge1xuICAvKipcbiAgICogRGlyZWN0aXZlIHR5cGUsIG5lZWRlZCB0byBjb25maWd1cmUgdGhlIGluamVjdG9yLlxuICAgKi9cbiAgdHlwZTogVHlwZTxUPjtcblxuICAvKiogVGhlIHNlbGVjdG9ycyB0aGF0IHdpbGwgYmUgdXNlZCB0byBtYXRjaCBub2RlcyB0byB0aGlzIGRpcmVjdGl2ZS4gKi9cbiAgc2VsZWN0b3JzOiBDc3NTZWxlY3Rvckxpc3Q7XG5cbiAgLyoqXG4gICAqIEZhY3RvcnkgbWV0aG9kIHVzZWQgdG8gY3JlYXRlIGFuIGluc3RhbmNlIG9mIGRpcmVjdGl2ZS5cbiAgICovXG4gIGZhY3Rvcnk6IEZhY3RvcnlGbjxUPjtcblxuICAvKipcbiAgICogQSBtYXAgb2YgaW5wdXQgbmFtZXMuXG4gICAqXG4gICAqIFRoZSBmb3JtYXQgaXMgaW46IGB7W2FjdHVhbFByb3BlcnR5TmFtZTogc3RyaW5nXTooc3RyaW5nfFtzdHJpbmcsIHN0cmluZ10pfWAuXG4gICAqXG4gICAqIEdpdmVuOlxuICAgKiBgYGBcbiAgICogY2xhc3MgTXlDb21wb25lbnQge1xuICAgKiAgIEBJbnB1dCgpXG4gICAqICAgcHVibGljSW5wdXQxOiBzdHJpbmc7XG4gICAqXG4gICAqICAgQElucHV0KCdwdWJsaWNJbnB1dDInKVxuICAgKiAgIGRlY2xhcmVkSW5wdXQyOiBzdHJpbmc7XG4gICAqIH1cbiAgICogYGBgXG4gICAqXG4gICAqIGlzIGRlc2NyaWJlZCBhczpcbiAgICogYGBgXG4gICAqIHtcbiAgICogICBwdWJsaWNJbnB1dDE6ICdwdWJsaWNJbnB1dDEnLFxuICAgKiAgIGRlY2xhcmVkSW5wdXQyOiBbJ2RlY2xhcmVkSW5wdXQyJywgJ3B1YmxpY0lucHV0MiddLFxuICAgKiB9XG4gICAqIGBgYFxuICAgKlxuICAgKiBXaGljaCB0aGUgbWluaWZpZXIgbWF5IHRyYW5zbGF0ZSB0bzpcbiAgICogYGBgXG4gICAqIHtcbiAgICogICBtaW5pZmllZFB1YmxpY0lucHV0MTogJ3B1YmxpY0lucHV0MScsXG4gICAqICAgbWluaWZpZWREZWNsYXJlZElucHV0MjogWyAncHVibGljSW5wdXQyJywgJ2RlY2xhcmVkSW5wdXQyJ10sXG4gICAqIH1cbiAgICogYGBgXG4gICAqXG4gICAqIFRoaXMgYWxsb3dzIHRoZSByZW5kZXIgdG8gcmUtY29uc3RydWN0IHRoZSBtaW5pZmllZCwgcHVibGljLCBhbmQgZGVjbGFyZWQgbmFtZXNcbiAgICogb2YgcHJvcGVydGllcy5cbiAgICpcbiAgICogTk9URTpcbiAgICogIC0gQmVjYXVzZSBkZWNsYXJlZCBhbmQgcHVibGljIG5hbWUgYXJlIHVzdWFsbHkgc2FtZSB3ZSBvbmx5IGdlbmVyYXRlIHRoZSBhcnJheVxuICAgKiAgICBgWydkZWNsYXJlZCcsICdwdWJsaWMnXWAgZm9ybWF0IHdoZW4gdGhleSBkaWZmZXIuXG4gICAqICAtIFRoZSByZWFzb24gd2h5IHRoaXMgQVBJIGFuZCBgb3V0cHV0c2AgQVBJIGlzIG5vdCB0aGUgc2FtZSBpcyB0aGF0IGBOZ09uQ2hhbmdlc2AgaGFzXG4gICAqICAgIGluY29uc2lzdGVudCBiZWhhdmlvciBpbiB0aGF0IGl0IHVzZXMgZGVjbGFyZWQgbmFtZXMgcmF0aGVyIHRoYW4gbWluaWZpZWQgb3IgcHVibGljLiBGb3JcbiAgICogICAgdGhpcyByZWFzb24gYE5nT25DaGFuZ2VzYCB3aWxsIGJlIGRlcHJlY2F0ZWQgYW5kIHJlbW92ZWQgaW4gZnV0dXJlIHZlcnNpb24gYW5kIHRoaXNcbiAgICogICAgQVBJIHdpbGwgYmUgc2ltcGxpZmllZCB0byBiZSBjb25zaXN0ZW50IHdpdGggYG91dHB1dGAuXG4gICAqL1xuICBpbnB1dHM/OiB7W1AgaW4ga2V5b2YgVF0/OiBzdHJpbmcgfCBbc3RyaW5nLCBzdHJpbmddfTtcblxuICAvKipcbiAgICogQSBtYXAgb2Ygb3V0cHV0IG5hbWVzLlxuICAgKlxuICAgKiBUaGUgZm9ybWF0IGlzIGluOiBge1thY3R1YWxQcm9wZXJ0eU5hbWU6IHN0cmluZ106c3RyaW5nfWAuXG4gICAqXG4gICAqIFdoaWNoIHRoZSBtaW5pZmllciBtYXkgdHJhbnNsYXRlIHRvOiBge1ttaW5pZmllZFByb3BlcnR5TmFtZTogc3RyaW5nXTpzdHJpbmd9YC5cbiAgICpcbiAgICogVGhpcyBhbGxvd3MgdGhlIHJlbmRlciB0byByZS1jb25zdHJ1Y3QgdGhlIG1pbmlmaWVkIGFuZCBub24tbWluaWZpZWQgbmFtZXNcbiAgICogb2YgcHJvcGVydGllcy5cbiAgICovXG4gIG91dHB1dHM/OiB7W1AgaW4ga2V5b2YgVF0/OiBzdHJpbmd9O1xuXG4gIC8qKlxuICAgKiBBIGxpc3Qgb2Ygb3B0aW9uYWwgZmVhdHVyZXMgdG8gYXBwbHkuXG4gICAqXG4gICAqIFNlZToge0BsaW5rIE5nT25DaGFuZ2VzRmVhdHVyZX0sIHtAbGluayBQcm92aWRlcnNGZWF0dXJlfSwge0BsaW5rIEluaGVyaXREZWZpbml0aW9uRmVhdHVyZX1cbiAgICovXG4gIGZlYXR1cmVzPzogRGlyZWN0aXZlRGVmRmVhdHVyZVtdO1xuXG4gIC8qKlxuICAgKiBGdW5jdGlvbiBleGVjdXRlZCBieSB0aGUgcGFyZW50IHRlbXBsYXRlIHRvIGFsbG93IGNoaWxkIGRpcmVjdGl2ZSB0byBhcHBseSBob3N0IGJpbmRpbmdzLlxuICAgKi9cbiAgaG9zdEJpbmRpbmdzPzogSG9zdEJpbmRpbmdzRnVuY3Rpb248VD47XG5cbiAgLyoqXG4gICAqIEZ1bmN0aW9uIHRvIGNyZWF0ZSBpbnN0YW5jZXMgb2YgY29udGVudCBxdWVyaWVzIGFzc29jaWF0ZWQgd2l0aCBhIGdpdmVuIGRpcmVjdGl2ZS5cbiAgICovXG4gIGNvbnRlbnRRdWVyaWVzPzogQ29udGVudFF1ZXJpZXNGdW5jdGlvbjxUPjtcblxuICAvKipcbiAgICogQWRkaXRpb25hbCBzZXQgb2YgaW5zdHJ1Y3Rpb25zIHNwZWNpZmljIHRvIHZpZXcgcXVlcnkgcHJvY2Vzc2luZy4gVGhpcyBjb3VsZCBiZSBzZWVuIGFzIGFcbiAgICogc2V0IG9mIGluc3RydWN0aW9ucyB0byBiZSBpbnNlcnRlZCBpbnRvIHRoZSB0ZW1wbGF0ZSBmdW5jdGlvbi5cbiAgICovXG4gIHZpZXdRdWVyeT86IFZpZXdRdWVyaWVzRnVuY3Rpb248VD58IG51bGw7XG5cbiAgLyoqXG4gICAqIERlZmluZXMgdGhlIG5hbWUgdGhhdCBjYW4gYmUgdXNlZCBpbiB0aGUgdGVtcGxhdGUgdG8gYXNzaWduIHRoaXMgZGlyZWN0aXZlIHRvIGEgdmFyaWFibGUuXG4gICAqXG4gICAqIFNlZToge0BsaW5rIERpcmVjdGl2ZS5leHBvcnRBc31cbiAgICovXG4gIGV4cG9ydEFzPzogc3RyaW5nW107XG59KSA9PiBuZXZlcjtcblxuLyoqXG4gKiBDcmVhdGUgYSBwaXBlIGRlZmluaXRpb24gb2JqZWN0LlxuICpcbiAqICMgRXhhbXBsZVxuICogYGBgXG4gKiBjbGFzcyBNeVBpcGUgaW1wbGVtZW50cyBQaXBlVHJhbnNmb3JtIHtcbiAqICAgLy8gR2VuZXJhdGVkIGJ5IEFuZ3VsYXIgVGVtcGxhdGUgQ29tcGlsZXJcbiAqICAgc3RhdGljIG5nUGlwZURlZiA9IGRlZmluZVBpcGUoe1xuICogICAgIC4uLlxuICogICB9KTtcbiAqIH1cbiAqIGBgYFxuICogQHBhcmFtIHBpcGVEZWYgUGlwZSBkZWZpbml0aW9uIGdlbmVyYXRlZCBieSB0aGUgY29tcGlsZXJcbiAqXG4gKiBAY29kZUdlbkFwaVxuICovXG5leHBvcnQgZnVuY3Rpb24gybXJtWRlZmluZVBpcGU8VD4ocGlwZURlZjoge1xuICAvKiogTmFtZSBvZiB0aGUgcGlwZS4gVXNlZCBmb3IgbWF0Y2hpbmcgcGlwZXMgaW4gdGVtcGxhdGUgdG8gcGlwZSBkZWZzLiAqL1xuICBuYW1lOiBzdHJpbmcsXG5cbiAgLyoqIFBpcGUgY2xhc3MgcmVmZXJlbmNlLiBOZWVkZWQgdG8gZXh0cmFjdCBwaXBlIGxpZmVjeWNsZSBob29rcy4gKi9cbiAgdHlwZTogVHlwZTxUPixcblxuICAvKiogQSBmYWN0b3J5IGZvciBjcmVhdGluZyBhIHBpcGUgaW5zdGFuY2UuICovXG4gIGZhY3Rvcnk6IEZhY3RvcnlGbjxUPixcblxuICAvKiogV2hldGhlciB0aGUgcGlwZSBpcyBwdXJlLiAqL1xuICBwdXJlPzogYm9vbGVhblxufSk6IG5ldmVyIHtcbiAgcmV0dXJuICg8UGlwZURlZjxUPj57XG4gICAgbmFtZTogcGlwZURlZi5uYW1lLFxuICAgIGZhY3Rvcnk6IHBpcGVEZWYuZmFjdG9yeSxcbiAgICBwdXJlOiBwaXBlRGVmLnB1cmUgIT09IGZhbHNlLFxuICAgIG9uRGVzdHJveTogcGlwZURlZi50eXBlLnByb3RvdHlwZS5uZ09uRGVzdHJveSB8fCBudWxsXG4gIH0pIGFzIG5ldmVyO1xufVxuXG4vKipcbiAqIFRoZSBmb2xsb3dpbmcgZ2V0dGVyIG1ldGhvZHMgcmV0cmlldmUgdGhlIGRlZmluaXRpb24gZm9ybSB0aGUgdHlwZS4gQ3VycmVudGx5IHRoZSByZXRyaWV2YWxcbiAqIGhvbm9ycyBpbmhlcml0YW5jZSwgYnV0IGluIHRoZSBmdXR1cmUgd2UgbWF5IGNoYW5nZSB0aGUgcnVsZSB0byByZXF1aXJlIHRoYXQgZGVmaW5pdGlvbnMgYXJlXG4gKiBleHBsaWNpdC4gVGhpcyB3b3VsZCByZXF1aXJlIHNvbWUgc29ydCBvZiBtaWdyYXRpb24gc3RyYXRlZ3kuXG4gKi9cblxuZXhwb3J0IGZ1bmN0aW9uIGdldENvbXBvbmVudERlZjxUPih0eXBlOiBhbnkpOiBDb21wb25lbnREZWY8VD58bnVsbCB7XG4gIHJldHVybiAodHlwZSBhcyBhbnkpW05HX0NPTVBPTkVOVF9ERUZdIHx8IG51bGw7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBnZXREaXJlY3RpdmVEZWY8VD4odHlwZTogYW55KTogRGlyZWN0aXZlRGVmPFQ+fG51bGwge1xuICByZXR1cm4gKHR5cGUgYXMgYW55KVtOR19ESVJFQ1RJVkVfREVGXSB8fCBudWxsO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gZ2V0UGlwZURlZjxUPih0eXBlOiBhbnkpOiBQaXBlRGVmPFQ+fG51bGwge1xuICByZXR1cm4gKHR5cGUgYXMgYW55KVtOR19QSVBFX0RFRl0gfHwgbnVsbDtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGdldEJhc2VEZWY8VD4odHlwZTogYW55KTogybXJtUJhc2VEZWY8VD58bnVsbCB7XG4gIHJldHVybiAodHlwZSBhcyBhbnkpW05HX0JBU0VfREVGXSB8fCBudWxsO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gZ2V0TmdNb2R1bGVEZWY8VD4odHlwZTogYW55LCB0aHJvd05vdEZvdW5kOiB0cnVlKTogTmdNb2R1bGVEZWY8VD47XG5leHBvcnQgZnVuY3Rpb24gZ2V0TmdNb2R1bGVEZWY8VD4odHlwZTogYW55KTogTmdNb2R1bGVEZWY8VD58bnVsbDtcbmV4cG9ydCBmdW5jdGlvbiBnZXROZ01vZHVsZURlZjxUPih0eXBlOiBhbnksIHRocm93Tm90Rm91bmQ/OiBib29sZWFuKTogTmdNb2R1bGVEZWY8VD58bnVsbCB7XG4gIGNvbnN0IG5nTW9kdWxlRGVmID0gKHR5cGUgYXMgYW55KVtOR19NT0RVTEVfREVGXSB8fCBudWxsO1xuICBpZiAoIW5nTW9kdWxlRGVmICYmIHRocm93Tm90Rm91bmQgPT09IHRydWUpIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoYFR5cGUgJHtzdHJpbmdpZnkodHlwZSl9IGRvZXMgbm90IGhhdmUgJ25nTW9kdWxlRGVmJyBwcm9wZXJ0eS5gKTtcbiAgfVxuICByZXR1cm4gbmdNb2R1bGVEZWY7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBnZXROZ0xvY2FsZUlkRGVmKHR5cGU6IGFueSk6IHN0cmluZ3xudWxsIHtcbiAgcmV0dXJuICh0eXBlIGFzIGFueSlbTkdfTE9DQUxFX0lEX0RFRl0gfHwgbnVsbDtcbn1cbiJdfQ==