#!/usr/bin/perl
#@ _TITLE_
# db_browse.pl - Umożliwienie przeglądania bazy danych sampdb za pomocą internetu.
#@ _TITLE_

# Pamiętaj, że instalacja tego skryptu stanowi także potencjalną
# lukę w zabezpieczeniach. Upewnij się o zrozumieniu informacji
# na ten temat przedstawionych w rozdziale 8.

#@ _PREAMBLE_
use strict;
use warnings;
use DBI;
use CGI qw (:standard escapeHTML escape);
#@ _PREAMBLE_

use Cwd;
# Plik opcji, który w systemie UNIX powinien zawierać parametry połączenia.
my $option_file = "/usr/local/apache/conf/sampdb.cnf";
my $option_drive_root;
# Nadpisanie położenia pliku w przypadku systemu Windows.
if ($^O =~ /^MSWin/i || $^O =~ /^dos/)
{
  $option_drive_root = "C:/";
  $option_file = "/Apache/conf/sampdb.cnf";
}

# Przygotowanie źródła danych i nawiązanie połączenia z serwerem (w systemie Windows trzeba
# najpierw zachować nazwę bieżącego katalogu roboczego, przejść do dysku zawierającego plik
# opcji, nawiązać połączenie, a następnie powrócić do zapisanego katalogu bieżącego).
my $orig_dir;
if (defined ($option_drive_root))
{
  $orig_dir = cwd ();
  chdir ($option_drive_root)
    or die "Nie można przejść do dysku $option_drive_root: $!\n";
}
my $dsn = "DBI:mysql:sampdb;mysql_read_default_file=$option_file";
my %conn_attrs = (RaiseError => 1, PrintError => 0, AutoCommit => 1);
my $dbh = DBI->connect ($dsn, undef, undef, \%conn_attrs);
if (defined ($option_drive_root))
{
  chdir ($orig_dir)
    or die "Nie można przejść do katalogu $orig_dir: $!\n";
}

#@ _MAIN_BODY_
my $db_name = "sampdb";

# Wygenerowanie początkowej części strony internetowej.
my $title = "Przeglądarka bazy danych $db_name";
print header ();
print start_html (-title => $title, -bgcolor => "white");
print h1 ($title);

# Parametry, które mogą znajdować się w adresie URL.
my $tbl_name = param ("tbl_name");
my $sort_col = param ("sort_col");

# Jeżeli zmienna $tbl_name nie ma wartości, wtedy zostanie wyświetlona lista nazw tabel, które można kliknąć.
# W przeciwnym razie wyświetlana jest zawartość wskazanej tabeli. Zmienna  $sort_col, o ile jest ustawiona,
# wskazuje kolumnę, względem której ma być posortowana tabela.

if (!defined ($tbl_name))
{
  display_table_names ($dbh, $db_name)
}
else
{
  display_table_contents ($dbh, $db_name, $tbl_name, $sort_col);
}

print end_html ();
#@ _MAIN_BODY_

$dbh->disconnect ();

#@ _DISPLAY_TABLE_NAMES_
sub display_table_names
{
my ($dbh, $db_name) = @_;

  print p ("Wybierz tabelę klikając jej nazwę:");

  # Pobranie odniesienia do składającej się z pojedynczej kolumny tablicy zawierającej nazwy tabel.
  my $sth = $dbh->prepare (qq{
              SELECT TABLE_NAME FROM INFORMATION_SCHEMA.TABLES
              WHERE TABLE_SCHEMA = ? ORDER BY TABLE_NAME
            });
  $sth->execute ($db_name);

  # Utworzenie nieporządkowanej listy za pomocą wywołania funkcji
  # ul() i li(). Każdy element jest łączem, którego kliknięcie ponownie
  # wywołuje skrypt w celu wyświetlenia zawartości wskazanej tabeli.

  my @item;
  while (my ($tbl_name) = $sth->fetchrow_array ())
  {
    my $url = sprintf ("%s?tbl_name=%s", url (), escape ($tbl_name));
    my $link = a ({-href => $url}, escapeHTML ($tbl_name));
    push (@item, li ($link));
  }
  print ul (@item);
}
#@ _DISPLAY_TABLE_NAMES_

#@ _DISPLAY_TABLE_CONTENTS_
sub display_table_contents
{
my ($dbh, $db_name, $tbl_name, $sort_col) = @_;
my $sort_clause = "";
my @rows;
my @cells;

  # Jeżeli podana została kolumna sortowania, należy jej użyć do posortowania wyników.
  if (defined ($sort_col))
  {
    $sort_clause = " ORDER BY " . $dbh->quote_identifier ($sort_col);
  }

  # Wyświetlenie łącza pozwalającego na powrót do listy tabel.
  print p (a ({-href => url ()}, "Wyświetl listę tabel"));

  print p (strong ("Zawartość tabeli $tbl_name:"));

  my $sth = $dbh->prepare (
              "SELECT * FROM "
            . $dbh->quote_identifier ($db_name, $tbl_name)
            . "$sort_clause LIMIT 200"
            );
  $sth->execute ();

  # Użycie nazw kolumn w tabeli bazy danych jako nagłówków
  # w tabeli HTML. Każda nazwa jest łączem powodującym
  # ponowne wywołanie skryptu w celu wyświetlenia zawartości
  # tabeli posortowanej względem wybranej kolumny.

  foreach my $col_name (@{$sth->{NAME}})
  {
    my $url = sprintf ("%s?tbl_name=%s;sort_col=%s",
                       url (),
                       escape ($tbl_name),
                       escape ($col_name));
    my $link = a ({-href => $url}, escapeHTML ($col_name));
    push (@cells, th ($link));
  }
  push (@rows, Tr (@cells));

  # Wyświetlenie rekordów tabeli.
  while (my @ary = $sth->fetchrow_array ())
  {
    @cells = ();
    foreach my $val (@ary)
    {
      # Wyświetlenie niepustych wartości, w przeciwnym razie wyświetlana jest spacja niełamiąca.
      if (defined ($val) && $val ne "")
      {
        $val = escapeHTML ($val);
      }
      else
      {
        $val = "&nbsp;";
      }
      push (@cells, td ($val));
    }
    push (@rows, Tr (@cells));
  }

  # Wyświetlenie tabeli wraz z obramowaniem.
  print table ({-border => "1"}, @rows);
}
#@ _DISPLAY_TABLE_CONTENTS_
