#!/usr/bin/perl
# need_renewal.pl - Wyszukanie członków Ligi Historycznej, którzy wkrótce muszą
# odnowić członkostwo. Domyślnie, słowo "wkrótce" oznacza 30 dni. Inną wartość
# można podać w wierszu poleceń.

# Podanie wartość 0 wyszukuje osoby, których członkostwo wygasło.

use strict;
use warnings;
use DBI;

# Przetworzenie parametrów z wiersza poleceń, o ile zostały podane.

use Getopt::Long;
$Getopt::Long::ignorecase = 0; # Wielkość liter w opcjach ma znaczenie.
$Getopt::Long::bundling = 1;   # -uname = -u name, a nie -u -n -a -m -e

# Parametry domyślne, wszystkie początkowo niezdefiniowane.
my ($host_name, $password, $port_num, $socket_name, $user_name);

GetOptions (
  # =i oznacza, że po opcji trzeba podać wartość w postaci liczby całkowitej.
  # =s oznacza, że po opcji trzeba podać wartość w postaci ciągu tekstowego.
  "host|h=s"      => \$host_name,
  "password|p=s"  => \$password,
  "port|P=i"      => \$port_num,
  "socket|S=s"    => \$socket_name,
  "user|u=s"      => \$user_name
) or exit (1);

# Utworzenie źródła danych.
my $dsn = "DBI:mysql:sampdb";
$dsn .= ";host=$host_name" if $host_name;
$dsn .= ";port=$port_num" if $port_num;
$dsn .= ";mysql_socket=$socket_name" if $socket_name;
$dsn .= ";mysql_read_default_group=client";

# Nawiązanie połączenia z serwerem.
my %conn_attrs = (RaiseError => 1, PrintError => 0, AutoCommit => 1);
my $dbh = DBI->connect ($dsn, $user_name, $password, \%conn_attrs);

#@ _MAIN_BODY_
# Domyślnie, osoba jest informowana na 30 dni przed wygaśnięciem członkostwa...
my $cutoff = 30;
# ...ale w wierszu poleceń można podać inną liczbę.
$cutoff = shift (@ARGV) if @ARGV && $ARGV[0] =~ /^\d+$/;

# Poinformowanie użytkownika o liczbie dni używanej przez skrypt.
warn "Liczba dni do upłynięcia członkostwa: $cutoff\n";

my $sth = $dbh->prepare (qq{
            SELECT
              member_id, email, last_name, first_name, expiration,
              TO_DAYS(expiration) - TO_DAYS(CURDATE()) AS days
            FROM member
            WHERE expiration < DATE_ADD(CURDATE(), INTERVAL ? DAY)
            ORDER BY expiration, last_name, first_name
          });
$sth->execute ($cutoff);  # Przekazanie liczby dni jako wartości miejsca zarezerwowanego.

while (my $entry_ref = $sth->fetchrow_hashref ())
{
  # Konwersja wartości undef na puste ciągi tekstowe w celu wyświetlenia na ekranie.
  foreach my $key (keys (%{$entry_ref}))
  {
    $entry_ref->{$key} = "" if !defined ($entry_ref->{$key});
  }
  print join ("\t",
              $entry_ref->{member_id},
              $entry_ref->{email},
              $entry_ref->{last_name},
              $entry_ref->{first_name},
              $entry_ref->{expiration},
              $entry_ref->{days} . " dni"),
        "\n";
}
#@ _MAIN_BODY_

$dbh->disconnect ();
