#!/usr/bin/perl
# edit_member.pl - Edycja z poziomu wiersza poleceń informacji dotyczących członków Ligi Historycznej.

# Aktualnie program ignoruje weryfikację danych.

use strict;
use warnings;
use DBI;

# Przetworzenie parametrów z wiersza poleceń, o ile zostały podane.

use Getopt::Long;
$Getopt::Long::ignorecase = 0; # Wielkość liter w opcjach ma znaczenie.
$Getopt::Long::bundling = 1;   # -uname = -u name, nie -u -n -a -m -e

# Parametry domyślne, wszystkie początkowo niezdefiniowane.
my ($host_name, $password, $port_num, $socket_name, $user_name);

GetOptions (
  # =i oznacza, że po opcji trzeba podać wartość w postaci liczby całkowitej.
  # =s oznacza, że po opcji trzeba podać wartość w postaci ciągu tekstowego.
  "host|h=s"      => \$host_name,
  "password|p=s"  => \$password,
  "port|P=i"      => \$port_num,
  "socket|S=s"    => \$socket_name,
  "user|u=s"      => \$user_name
) or exit (1);

# Utworzenie źródła danych.
my $dsn = "DBI:mysql:sampdb";
$dsn .= ";host=$host_name" if $host_name;
$dsn .= ";port=$port_num" if $port_num;
$dsn .= ";mysql_socket=$socket_name" if $socket_name;
$dsn .= ";mysql_read_default_group=client";

# Nawiązanie połączenia z serwerem.
my %conn_attrs = (RaiseError => 1, PrintError => 0, AutoCommit => 1);
my $dbh = DBI->connect ($dsn, $user_name, $password, \%conn_attrs);

#@ _GET_COL_INFO_
my @col_name = ();       # Tablica nazw kolumn.
my %nullable = ();       # Czy kolumna akceptuje wartości NULL?
# Pobranie nazw kolumn tabeli member.
my $sth = $dbh->prepare (qq{
            SELECT COLUMN_NAME, UPPER(IS_NULLABLE)
            FROM INFORMATION_SCHEMA.COLUMNS
            WHERE TABLE_SCHEMA = ? AND TABLE_NAME = ?
          });
$sth->execute ("sampdb", "member");
while (my ($col_name, $is_nullable) = $sth->fetchrow_array ())
{
  push (@col_name, $col_name);
  $nullable{$col_name} = ($is_nullable eq "YES");
}
#@ _GET_COL_INFO_

#@ _MAIN_LOOP_
if (@ARGV == 0) # Jeżeli nie podano argumentów, wtedy należy utworzyć nowy rekord.
{
  # Przekazanie odniesienia do tablicy nazw kolumn.
  new_member (\@col_name);
}
else            # W przeciwnym razie przeprowadzana jest edycja rekordów wskazanych przez argumenty.
{
   # Zachowanie tablicy @ARGV, a następnie jej opróżnienie, aby kiedy skrypt odczytuje dane wejściowe
   # z STDIN, nie interpretował zawartości @ARGV jako nazw plików danych wejściowych.
  my @id = @ARGV;
  @ARGV = ();
  # Dla każdej wartości ID należy wyszukać odpowiedni rekord, a następnie przeprowadzić jego edycję.
  while (my $id = shift (@id))
  {
    $sth = $dbh->prepare (qq{
             SELECT * FROM member WHERE member_id = ?
           });
    $sth->execute ($id);
    my $entry_ref = $sth->fetchrow_hashref ();
    $sth->finish ();
    if (!$entry_ref)
    {
      warn "Nie znaleziono osoby o identyfikatorze = $id\n";
      next;
    }
    # Przekazanie odniesień do tablicy nazw kolumn i rekordu.
    edit_member (\@col_name, $entry_ref);
  }
}
#@ _MAIN_LOOP_

$dbh->disconnect ();

# Wyświetlenie pytania, oczekiwanie na odpowiedź.

#@ _X_PROMPT_
sub prompt
{
my $str = shift;

  print STDERR $str;
  chomp ($str = <STDIN>);
  return ($str);
}
#@ _X_PROMPT_

# Mając nazwę kolumny i odniesienie do wartości hash, można wyświetlić pytanie o nową 
# wartość kolumny. Jeżeli wartością hash nie jest undef, w pytaniu będzie wyświetlona 
# bieżąca wartość kolumny, aby użytkownik mógł ją zobaczyć.

# UWAGA:
# Ta wersja została początkowo przedstawiona w książce. Metoda col_prompt() w poniższej
# postaci jest zmodyfikowaną wersją sprawdzając datę wygaśnięcia członkostwa. Aby użyć
# tej wersji, zmień nazwę col_prompt na col_prompt1, a następnie zmień nazwę
# col_prompt0 na col_prompt.

sub col_prompt0
#@ _COL_PROMPT0_
{
my ($col_name, $entry_ref) = @_;

  my $prompt = $col_name;
  if (defined ($entry_ref))
  {
    my $cur_val = $entry_ref->{$col_name};
    $cur_val = "NULL" if !defined ($cur_val);
    $prompt .= " [$cur_val]";
  }
  $prompt .= ": ";
  print STDERR $prompt;
  my $str = <STDIN>;
  chomp ($str);
  return ($str);
}
#@ _COL_PROMPT0_

# Wyświetlenie pytania o nową wartość kolumny; wartość bieżąca będzie wyświetlona,
# jeśli $show_current przyjmuje wartość true.

sub col_prompt
#@ _COL_PROMPT1_
{
my ($col_name, $entry_ref) = @_;

loop:
  my $prompt = $col_name;
  if (defined ($entry_ref))
  {
    my $cur_val = $entry_ref->{$col_name};
    $cur_val = "NULL" if !defined ($cur_val);
    $prompt .= " [$cur_val]";
  }
  $prompt .= ": ";
  print STDERR $prompt;
  my $str = <STDIN>;
  chomp ($str);
  # Przeprowadzenie prostej operacji sprawdzenia daty wygaśnięcia członkostwa.
  if ($str && $col_name eq "expiration")  # Sprawdzenie formatu daty dla kolumny expiration.
  {
    if ($str !~ /^\d+\D\d+\D\d+$/)
    {
      warn "$str is not a legal date, try again\n";
      goto loop;
    }
  }
  return ($str);
}
#@ _COL_PROMPT1_

# Wyświetlenie zawartości rekordu.

sub show_member
{
# Odniesienie do tablicy nazw kolumn i wartości hash.
my ($col_name_ref, $entry_ref) = @_;

  print "\n";
  foreach my $col_name (@{$col_name_ref})
  {
    my $col_val = $entry_ref->{$col_name};
    $col_val = "NULL" if !defined ($col_val);
    printf "%s: %s\n", $col_name, $col_val;
  }
}

# Utworzenie rekordu nowego członka.

#@ _NEW_MEMBER_
sub new_member
{
my $col_name_ref = shift; # Odniesienie do tablicy nazw kolumn.
my $entry_ref = { };    # Utworzenie nowego rekordu jako tablicy.

  return unless prompt ("Create new entry (y/n)? ") =~ /^y/i;
  # Wyświetlenie pytań o nowe wartości; użytkownik podaje nową wartość lub naciska klawisz Enter,
  # pozostawiając wartość niezmienioną. Słowo "BRAK" powoduje usunięcie wartości, natomiast
  # "KONIEC" kończy działanie skryptu bez utworzenia rekordu.

  foreach my $col_name (@{$col_name_ref})
  {
    next if $col_name eq "member_id";   # Pominięcie pola klucza.
    my $col_val = col_prompt ($col_name, undef);
    next if $col_val eq "";             # Użytkownik nacisnął klawisz Enter.
    return if uc ($col_val) eq "KONIEC";  # Zakończenie działania skryptu.
    if (uc ($col_val) eq "BRAK")
    {
      # Wstawienie wartości NULL, jeśli kolumna je akceptuje, w przeciwnym razie wstawiany jest pusty ciąg tekstowy.
      $col_val = ($nullable{$col_name} ? undef : "");
    }
    $entry_ref->{$col_name} = $col_val;
  }
  # Wyświetlenie wartości i pytania o potwierdzenie danych, zanim zostaną wstawione.
  show_member ($col_name_ref, $entry_ref);
  return unless prompt ("\nWstawić rekord (t/n)? ") =~ /^t/i;

  # Przygotowanie zapytania INSERT, a następnie jego wykonanie.
  my $stmt = "INSERT INTO member";
  my $delim = " SET "; # Umieszczenie "SET" przed pierwszą kolumną, natomiast "," przed pozostałymi.
  foreach my $col_name (@{$col_name_ref})
  {
    # Użyte będą tylko te kolumny, dla których podano wartości.
    next if !defined ($entry_ref->{$col_name});
    # Metoda quote() wstawia undef jako słowo NULL (bez znaków cytowania),
    # co jest żądanym zachowaniem. Następuje przypisanie Kkolumnom typu NOT NULL następuje
    # przypisanie ich wartości domyślnych.
    $stmt .= sprintf ("%s %s=%s", $delim, $col_name,
                      $dbh->quote ($entry_ref->{$col_name}));
    $delim = ",";
  }
  $dbh->do ($stmt) or warn "Ostrzeżenie: nie utworzono nowego rekordu!\n"
}
#@ _NEW_MEMBER_

# Edycja treści istniejącego rekordu.

#@ _EDIT_MEMBER_
sub edit_member
{
# Odniesienie do tablicy nazw kolumn i tablicy rekordu.
my ($col_name_ref, $entry_ref) = @_;

  # Wyświetlenie wartości początkowych i pytania, czy można przejść dalej do edycji rekordu.
  show_member ($col_name_ref, $entry_ref);
  return unless prompt ("\nChcesz edytować ten rekord (t/n)? ") =~ /^t/i;
  # Wyświetlenie pytań o nowe wartości; użytkownik podaje nową wartość lub naciska klawisz Enter,
  # pozostawiając wartość niezmienioną. Słowo "BRAK" powoduje usunięcie wartości, natomiast
  # "KONIEC" kończy działanie skryptu bez utworzenia rekordu.
  foreach my $col_name (@{$col_name_ref})
  {
    next if $col_name eq "member_id";   # Pominięcie pola klucza.
    my $col_val = col_prompt ($col_name, $entry_ref);
    next if $col_val eq "";             # Użytkownik nacisnął klawisz Enter.
    return if uc ($col_val) eq "KONIEC";  # Zakończenie działania skryptu.
    if (uc ($col_val) eq "BRAK")
    {
      # Wstawienie wartości NULL, jeśli kolumna je akceptuje, w przeciwnym razie wstawiany jest pusty ciąg tekstowy.
      $col_val = ($nullable{$col_name} ? undef : "");
    }

    $entry_ref->{$col_name} = $col_val;
  }
  # Wyświetlenie nowych wartości i pytania o potwierdzenie danych, zanim zostaną wstawione.
  show_member ($col_name_ref, $entry_ref);
  return unless prompt ("\nUaktualnić ten rekord (t/n?") =~ /^t/i;

  # Przygotowanie zapytania UPDATE, a następnie jego wykonanie.
  my $stmt = "UPDATE member";
  my $delim = " SET "; # Umieszczenie "SET" przed pierwszą kolumną, natomiast "," przed pozostałymi.
  foreach my $col_name (@{$col_name_ref})
  {
    next if $col_name eq "member_id"; # Pominięcie pola klucza.
    # Metoda quote() wstawia undef jako słowo NULL (bez znaków cytowania),
    # co jest żądanym zachowaniem.
    $stmt .= sprintf ("%s %s=%s", $delim, $col_name,
                      $dbh->quote ($entry_ref->{$col_name}));
    $delim = ",";
  }
  $stmt .= " WHERE member_id = " . $dbh->quote ($entry_ref->{member_id});
  $dbh->do ($stmt) or warn "Ostrzeżenie: nie uaktualniono rekordu!\n"
}
#@ _EDIT_MEMBER_
