{*************************************************************}
{                     Modul RORONLD                           }
{       PROCEDURY ROZWIAZYWANIA ROWNAN ROZNICZKOWYCH          }
{                 NIELINIOWYCH RZEDU N                        }
{         zapis rozwiazania do macierzy dynamicznej           }
{                 Turbo Pascal  wersja 7.0                    }
{                   autor Bernard Baron                       }
{-------------------------------------------------------------}
UNIT RORONLD;
  {$IFDEF CPU87} {$N+}
         {$ELSE} {$N-}
  {$ENDIF}
  {$F+}

INTERFACE

  USES  CRT,TFLOAT,ALGELIN;

  CONST MAXP   =1025;

  TYPE  MACD3  =ARRAY[0..MAXP]OF ^WEK;
        PROCXT =PROCEDURE(VAR F:WEK;
                              X:WEK;
                              t:FLOAT;
                              N:BYTE);
        WEK8   =ARRAY[0..8] OF WEK;


    PROCEDURE NEWWIERD3(VAR A    :MACD3;
                            N    :WORD;
                        VAR BLAD :BYTE);
    { Utworzenie N-tego wiersza tablicy dynamicznej A }
    { do zapisu rozwiazania rownania rozniczkowego    }
    { BLAD - nr bledu; 0 - brak bledu                 }

    PROCEDURE DISMACD3(VAR A :MACD3;
                           N :WORD);
    { Zwolnienie tablicy dynamicznej A o N wierszach }


    PROCEDURE RUNGE_KUTTY(    K,N    :BYTE;
                              NIELIN :PROCXT;
                              h,t0   :FLOAT;
                              X0     :WEK;
                          VAR X      :WEK;
                          VAR BLAD   :BYTE);
{-------------------------------------------------------------}
{ Calkowanie ukladu N rownan nieliniowych dX(t)/dt=F[X(t),t]  }
{  o warunku poczatkowym X(t0)=X0 z krokiem h tj. wyznaczanie }
{  X(t0+h) wg metody Rungego-Kutty rzedu K=1,2,3,4,5,6        }
{  wzory (5.10)(5.11)(5.12)(5.13)(5.14)(5.15)                 }
{  NIELIN  - procedura typu PROCXT okreslajaca odwzorowanie   }
{            wektora X:WEK oraz t:FLOAT w wektor Y:WEK        }
{            tj. generacja funkcji wektorowej prawej strony   }
{            rownania rozniczkowego   Y = F ( X , t )         }
{               dX/dt = F ( X , t );                          }
{     h  - krok calkowania;                                   }
{     t0 - chwila poczatkowa;                                 }
{     X0 - wektor wartosci poczatkowych;                      }
{     N  - ilosc rownan nieliniowych;                         }
{     X  - rozwiazanie ukladu rownan w punkcie t0+h           }
{   BLAD - nr bledu; 0 - brak bledu                           }
{-------------------------------------------------------------}

  PROCEDURE MET_RUN_KUT(    ST,K,N            :BYTE;
                            NIELIN            :PROCXT;
                            h0,t0,t1,
                            epsw,epswmin,epsa :FLOAT;
                            X0                :WEK;
                        VAR M                 :WORD;
                        VAR XX                :MACD3;
                        VAR BLAD              :BYTE;
                            LICZNIK           :WORD);
{-------------------------------------------------------------}
{   Metoda Rungego Kutty rzedu K rozwiazywania nieliniowego   }
{ ukladu N rownan rozniczkowych z automatycznym doborem kroku }
{ calkowania do z gory zalozonej dokladnosci wzglednej epsw   }
{ oraz absolutnej epsa                                        }
{ NIELIN  - procedura typu PROCXT okreslajaca odwzorowanie    }
{           wektora X:WEK oraz t:FLOAT w wektor Y:WEK         }
{           tj. generacja funkcji wektorowej prawej strony    }
{           rownania rozniczkowego   Y = F ( X , t )          }
{               dX/dt = F ( X , t );                          }
{ ST - parametr pomocniczy                                    }
{ ST=1 staly krok calkowania przy automatycznym doborze rzedu }
{ ST=2 automatyczny dobor kroku                               }
{ t0,t1   - poczatek i koniec przedzialu calkowania <t0,t1>   }
{ epsw    - tolerancja bledu wzglednego                       }
{ epswmin - najmniejsza tolerancja bledu wzglednego jakiego   }
{           dopuszczamy w obliczeniach (np. 1E-12)            }
{ epsa - tolerancja bledu absolutnego                         }
{ X0   - wektor wartosci poczatkowych;                        }
{ M    - ilosc iteracji obliczona i zapisana w macierzy XX    }
{ XX   - macierz dynamiczna do zapisania rozwiazania;         }
{ BLAD - nr bledu; 0 - brak bledu                             }
{ LICZNIK - parametr wyswietlania numeru iteracji:            }
{           0       - brak wyswietlania,                      }
{           X*256+Y - wyswietlenie piecio-znakowe             }
{                     w wierszu X i kolumnie Y.               }
{-------------------------------------------------------------}

   PROCEDURE FEHLBERG(    K,N    :BYTE;
                          NIELIN :PROCXT;
                          h,t0   :FLOAT;
                          X0     :WEK;
                      VAR X,E    :WEK;
                      VAR BLAD   :BYTE);
{-------------------------------------------------------------}
{ Calkowanie ukladu N rownan nieliniowych dX(t)/dt=F[X(t),t]  }
{  o warunku poczatkowy X(t0)=X0 z krokiem h tj. wyznaczanie  }
{  X(t0+h) wg metod wlozonych Fehlberga rzedu K=1,2,3,4,5,6   }
{  wzory (5.36)(5.37)(5.38)(5.39)(5.40)                       }
{  NIELIN  - procedura typu PROCXT okreslajaca odwzorowanie   }
{            wektora X:WEK oraz t:FLOAT w wektor Y:WEK        }
{            tj. generacja funkcji wektorowej prawej strony   }
{            rownania rozniczkowego   Y = F ( X , t )         }
{               dX/dt = F ( X , t );                          }
{     h  - krok calkowania;                                   }
{     t0 - chwila poczatkowa;                                 }
{     X0 - wektor wartosci poczatkowych;                      }
{     N  - ilosc rownan nieliniowych;                         }
{     X  - wektor rozwiazanie ukladu rownan w punkcie t0+h    }
{     E  - wektor bledu aproksymacji wg wzoru (5.34)          }
{   BLAD - nr bledu; 0 - brak bledu                           }
{-------------------------------------------------------------}

  PROCEDURE MET_FEHLB(    ST,K,N            :BYTE;
                          NIELIN            :PROCXT;
                          h0,t0,t1,
                          epsw,epswmin,epsa :FLOAT;
                      VAR X0                :WEK;
                      VAR M                 :WORD;
                      VAR XX                :MACD3;
                      VAR BLAD              :BYTE;
                          LICZNIK           :WORD);
{-------------------------------------------------------------}
{   Metoda Fehlberga tworzona z par wlozonych rzedu K i K+1   }
{ rozwiazywania nieliniowego ukladu N rownan rozniczkowych    }
{ z automatycznym doborem kroku calkowania do z gory zalozonej}
{ dokladnosci wzglednej epsw oraz absolutnej epsa             }
{ NIELIN  - procedura typu PROCXT okreslajaca odwzorowanie    }
{           wektora X:WEK oraz t:FLOAT w wektor Y:WEK         }
{           tj. generacja funkcji wektorowej prawej strony    }
{           rownania rozniczkowego   Y = F ( X , t )          }
{               dX/dt = F ( X , t );                          }
{ ST - parametr pomocniczy                                    }
{ ST=1 staly krok calkowania przy automatycznym doborze rzedu }
{ ST=2 automatyczny dobor kroku i rzedu                       }
{ t0,t1   - poczatek i koniec przedzialu calkowania <t0,t1>   }
{ epsw    - tolerancja bledu wzglednego                       }
{ epswmin - najmniejsza tolerancja bledu wzglednego jakiego   }
{           dopuszczamy w obliczeniach (np. 1E-12)            }
{ epsa - tolerancja bledu absolutnego                         }
{ X0   - wektor wartosci poczatkowych;                        }
{ M    - ilosc iteracji obliczona i zapisana w macierzy XX    }
{ XX   - macierz dynamiczna do zapisania rozwiazania;         }
{ BLAD - nr bledu; 0 - brak bledu                             }
{ LICZNIK - parametr wyswietlania numeru iteracji:            }
{           0       - brak wyswietlania,                      }
{           X*256+Y - wyswietlenie piecio-znakowe             }
{                     w wierszu X i kolumnie Y.               }
{-------------------------------------------------------------}

  PROCEDURE MET_ADAMS_MUL(    ST,N              :BYTE;
                              NIELIN            :PROCXT;
                              h0,t0,t1,
                              epsw,epswmin,epsa :FLOAT;
                              X0                :WEK;
                          VAR M                 :WORD;
                          VAR XX                :MACD3;
                          VAR BLAD              :BYTE;
                              LICZNIK           :WORD);
{-------------------------------------------------------------}
{  Algorytm wielokrokowy z czlonem przewidywania Adamsa-Bash- }
{ forta rzedu K i czlonem korekcji Adamsa-Multona rzedu K+1   }
{ rozwiazywania nieliniowego ukladu N rownan rozniczkowych    }
{ z automatyczna zmiana rzedu i kroku calkowania              }
{ NIELIN  - procedura typu PROCXT okreslajaca odwzorowanie    }
{           wektora X:WEK oraz t:FLOAT w wektor Y:WEK         }
{           tj. generacja funkcji wektorowej prawej strony    }
{           rownania rozniczkowego   Y = F ( X , t )          }
{               dX/dt = F ( X , t );                          }
{ ST - parametr pomocniczy                                    }
{ ST=1 staly krok calkowania przy automatycznym doborze rzedu }
{ ST=2 automatyczny dobor kroku i rzedu                       }
{ h0- ustalony krok calkowania dla parametru ST=1             }
{ t0,t1   - poczatek i koniec przedzialu calkowania <t0,t1>   }
{ epsw    - tolerancja bledu wzglednego                       }
{ epswmin - najmniejsza tolerancja bledu wzglednego jakiego   }
{           dopuszczamy w obliczeniach (np. 1E-12)            }
{ epsa - tolerancja bledu absolutnego                         }
{ X0   - wektor wartosci poczatkowych;                        }
{ M    - ilosc iteracji obliczona i zapisana w macierzy XX    }
{ XX   - macierz dynamiczna do zapisania rozwiazania;         }
{ BLAD - nr bledu; 0 - brak bledu                             }
{ LICZNIK - parametr wyswietlania numeru iteracji:            }
{           0       - brak wyswietlania,                      }
{           X*256+Y - wyswietlenie piecio-znakowe             }
{                     w wierszu X i kolumnie Y.               }
{-------------------------------------------------------------}

  PROCEDURE MET_GEAR(    ST,N              :BYTE;
                         NIELIN            :PROCXT;
                         h0,t0,t1,
                         epsw,epswmin,epsa :FLOAT;
                         X0                :WEK;
                     VAR M                 :WORD;
                     VAR XX                :MACD3;
                     VAR BLAD              :BYTE;
                         LICZNIK           :WORD);
{-------------------------------------------------------------}
{        Algorytm wielokrokowy Geara rzedu K                  }
{ rozwiazywania nieliniowego ukladu N rownan rozniczkowych    }
{ z automatyczna zmiana rzedu i kroku calkowania              }
{ NIELIN  - procedura typu PROCXT okreslajaca odwzorowanie    }
{           wektora X:WEK oraz t:FLOAT w wektor Y:WEK         }
{           tj. generacja funkcji wektorowej prawej strony    }
{           rownania rozniczkowego   Y = F ( X , t )          }
{               dX/dt = F ( X , t );                          }
{ ST - parametr pomocniczy                                    }
{ ST=1 staly krok calkowania przy automatycznym doborze rzedu }
{ ST=2 automatyczny dobor kroku i rzedu                       }
{ h0- ustalony krok calkowania dla parametru ST=1             }
{ t0,t1   - poczatek i koniec przedzialu calkowania <t0,t1>   }
{ epsw    - tolerancja bledu wzglednego                       }
{ epswmin - najmniejsza tolerancja bledu wzglednego jakiego   }
{           dopuszczamy w obliczeniach (np. 1E-12)            }
{ epsa - tolerancja bledu absolutnego                         }
{ X0   - wektor wartosci poczatkowych;                        }
{ M    - ilosc iteracji obliczona i zapisana w macierzy XX    }
{ XX   - macierz dynamiczna do zapisania rozwiazania;         }
{ BLAD - nr bledu; 0 - brak bledu                             }
{ LICZNIK - parametr wyswietlania numeru iteracji:            }
{           0       - brak wyswietlania,                      }
{           X*256+Y - wyswietlenie piecio-znakowe             }
{                     w wierszu X i kolumnie Y.               }
{-------------------------------------------------------------}


IMPLEMENTATION


   PROCEDURE NEWWIERD3;
     VAR PAO :LONGINT;
     BEGIN
       BLAD:=0;
       IF N>MAXP
         THEN BLAD:=114
         ELSE BEGIN
                PAO:=SizeOf(WEK);
                IF PAO < MaxAvail
                  THEN NEW(A[N])
                  ELSE BLAD:=20
              END
     END { NEWWIERD3 };

   PROCEDURE DISMACD3;
     VAR i:WORD;
     BEGIN
       FOR i:=0 TO N DO
         DISPOSE(A[i])
     END { DISMACD3 };


  FUNCTION POTEGA(N:INTEGER;
                  X:FLOAT):FLOAT;
  { Calkowita potega N liczby rzeczywistej X }
    VAR W :FLOAT;
        i :INTEGER;
    BEGIN
      IF N=0
        THEN W:=1.0
        ELSE BEGIN
               W:=X;
               FOR i:=2 TO N DO
                 W:=W*X
             END;
      POTEGA:=W
    END { POTEGA };

  PROCEDURE MULRW(VAR Y :WEK;
                      a :FLOAT;
                      X :WEK;
                      N :BYTE);
{  Iloczyn wektora przez liczbe: Y[i] = a*X[i] ; i = 1,2,...,N }
    VAR i:BYTE;
    BEGIN
      FOR i:=1 TO N DO
        Y[i]:=a*X[i]
    END { MULRW };

  PROCEDURE ADDWEK(VAR X     :WEK;
                       X1,X2 :WEK;
                       N     :BYTE);
  { Suma wektorow X = X1 + X2 }
    VAR i:BYTE;
    BEGIN
      FOR i:=1 TO N DO
        X[i]:=X1[i]+X2[i]
    END { ADDWEK };

  PROCEDURE SUBWEK(VAR X     :WEK;
                       X1,X2 :WEK;
                       N     :BYTE);
  { Roznica wektorow X = X1 - X2 }
    VAR i:BYTE;
    BEGIN
      FOR i:=1 TO N DO
        X[i]:=X1[i]-X2[i]
    END { SUBWEK };

  FUNCTION NORMAX(N :BYTE;
                  E :WEK):FLOAT;
  { Norma wektora E, ||E|| = max E[i] }
  {                           i       }
    VAR i   :BYTE;
        S,W :FLOAT;
    BEGIN
      S:=0;
      FOR i:=1 TO N DO
      BEGIN
        W:=ABS(E[i]);
        IF W>S THEN
          S:=W
      END;
      NORMAX:=S
    END { NORMAX };

  PROCEDURE RUNGE_KUTTY;
    VAR i,j,l                :INTEGER;
        t,dX                 :FLOAT;
        K1,K2,K3,K4,K5,K6,X1 :WEK;
    BEGIN
      BLAD:=0;
      IF K in [1..4,6]
        THEN CASE K OF
               1: BEGIN { wzor (5.10) }
                    NIELIN(K1,X0,t0,N);
                    FOR j:=1 TO N DO
                      X[j]:=X0[j]+h*K1[j]
                  END {1};
               2: BEGIN { wzor (5.11) }
                    NIELIN(K1,X0,t0,N);  MULRW(K1,h,K1,N);
                    t:=t0+h;
                    FOR l:=1 TO N DO
                      X1[l]:=X0[l]+K1[l];
                    NIELIN(K2,X1,t,N);   MULRW(K2,h,K2,N);
                    FOR j:=1 TO N DO
                    BEGIN
                      dX:=(K1[j]+K2[j])/2;
                      X[j]:=X0[j]+dX
                    END
                  END {2};
               3: BEGIN { wzor (5.12) }
                    NIELIN(K1,X0,t0,N);  MULRW(K1,h,K1,N);
                    t:=t0+h/2;
                    FOR l:=1 TO N DO
                      X1[l]:=X0[l]+K1[l]/2;
                    NIELIN(K2,X1,t,N);   MULRW(K2,h,K2,N);
                    t:=t0+h;
                    FOR l:=1 TO N DO
                      X1[l]:=X0[l]-K1[l]+2*K2[l];
                    NIELIN(K3,X1,t,N);   MULRW(K3,h,K3,N);
                    FOR j:=1 TO N DO
                    BEGIN
                      dX:=(K1[j]+4*K2[j]+K3[j])/6;
                      X[j]:=X0[j]+dX
                    END
                  END {3};
               4: BEGIN { wzor (5.13) }
                    NIELIN(K1,X0,t0,N);  MULRW(K1,h,K1,N);
                    t:=t0+h/2;
                    FOR l:=1 TO N DO
                      X1[l]:=X0[l]+K1[l]/2;
                    NIELIN(K2,X1,t,N);   MULRW(K2,h,K2,N);
                    FOR l:=1 TO N DO
                      X1[l]:=X0[l]+K2[l]/2;
                    NIELIN(K3,X1,t,N);   MULRW(K3,h,K3,N);
                    t:=t0+h;
                    FOR l:=1 TO N DO
                      X1[l]:=X0[l]+K3[l];
                    NIELIN(K4,X1,t,N);   MULRW(K4,h,K4,N);
                    FOR j:=1 TO N DO
                    BEGIN
                      dX:=(K1[j]+2*K2[j]+2*K3[j]+K4[j])/6;
                      X[j]:=X0[j]+dX
                    END
                  END {4};

               6: BEGIN { wzor (5.15) }
                    NIELIN(K1,X0,t0,N);  MULRW(K1,h,K1,N);
                    t:=t0+h/3;
                    FOR l:=1 TO N DO
                      X1[l]:=X0[l]+K1[l]/3;
                    NIELIN(K2,X1,t,N);   MULRW(K2,h,K2,N);
                    t:=t0+2*h/5;
                    FOR l:=1 TO N DO
                      X1[l]:=X0[l]+(6*K2[l]+4*K1[l])/25;
                    NIELIN(K3,X1,t,N);   MULRW(K3,h,K3,N);
                    t:=t0+h;
                    FOR l:=1 TO N DO
                      X1[l]:=X0[l]+(15*K3[l]-12*K2[l]+K1[l])/4;
                    NIELIN(K4,X1,t,N);   MULRW(K4,h,K4,N);
                    t:=t0+2*h/3;
                    FOR l:=1 TO N DO
                      X1[l]:=X0[l]+(8*K4[l]-50*K3[l]+90*K2[l]+6*K1[l])/81;
                    NIELIN(K5,X1,t,N);   MULRW(K5,h,K5,N);
                    t:=t0+4*h/5;
                    FOR l:=1 TO N DO
                      X1[l]:=X0[l]+(8*K4[l]+10*K3[l]+36*K2[l]+6*K1[l])/75;
                    NIELIN(K6,X1,t,N);   MULRW(K6,h,K6,N);
                    FOR j:=1 TO N DO
                    BEGIN
                      dX:=(23*K1[j]+125*K3[j]-81*K5[j]+125*K6[j])/192;
                      X[j]:=X0[j]+dX
                    END
                  END {6}
             END
        ELSE BLAD:=100
  END { RUNGE_KUTTY };

  PROCEDURE MET_RUN_KUT;
    VAR j,l              :BYTE;
        t,e,s1,
        Hmin,Hh,Hh1,h1,h2,ee,Hph,
        epsm,epsm1,epsw1,
        A26,xt,mxt,eps   :FLOAT;
        X1,X2,Y          :WEK;
    BEGIN
      BLAD:=0;
      IF ST<>1
        THEN Hh:=(t1-t0)/100
        ELSE Hh:=h0;
      { Wyznaczanie wzglednej dokladnosci maszynowej epsm }
      epsm:=1.0;
      REPEAT
        epsm:=epsm/2; epsm1:=epsm+1.0
      UNTIL epsm1<=1;
      { Sprawdzenie dopuszczalnej tolerancji wzglednej (5.27) }
      IF epsw>=2*epsm+epswmin
        THEN epsw1:=epsw
        ELSE epsw1:=2*epsm+epswmin;
      A26:=26*epsm;
      IF ABS(t1-t0)<A26 THEN
      BEGIN
        BLAD:=101;
        EXIT
      END;
      t:=t0;  Y:=X0; Y[0]:=t0; M:=0;
      NEWWIERD3(XX,M,BLAD);
      IF BLAD<>0 THEN
        EXIT;
      XX[M]^:=Y;
      REPEAT
        RUNGE_KUTTY(K,N,NIELIN,Hh,t,Y,X1,BLAD);
        h2:=Hh/2;
        RUNGE_KUTTY(K,N,NIELIN,h2,t,Y,X2,BLAD);
        RUNGE_KUTTY(K,N,NIELIN,h2,t,X2,X2,BLAD);
        e:=0; mxt:=0;
        FOR j:=1 TO N DO
        BEGIN
          s1:=ABS(X2[j]-X1[j]);
          IF e < s1 THEN
            e:=s1;
          xt:=(ABS(Y[j])+ABS(X2[j]))/2;
          IF mxt < xt THEN
            mxt:=xt
        END;
        s1:=POTEGA(K,2.0);
        eps:=mxt*epsw1+epsa;    { wzor (5.18) }
        ee:=s1*e/((s1-1)*eps);
        Hph:=EXP(LN(ee)/(K+1)); { wzor (5.25) }
        h1:=Hh/Hph;
        Hmin:=A26*ABS(t);       { wzor (5.26) }
        IF (ABS(h1)<Hmin) AND (ST<>1)
          THEN BLAD:=102
          ELSE BEGIN
                 IF Hph >=3
                   THEN IF ST<>1
                          THEN Hh:=2*h1
                          ELSE BLAD:=103
                   ELSE BEGIN
                          Y:=X2; t:=t+Hh ; Y[0]:=t; INC(M);
                          NEWWIERD3(XX,M,BLAD);
                          IF BLAD=0 THEN
                          BEGIN
                            XX[M]^:=Y;
                            IF ST<>1 THEN
                              Hh:=2*h1
                          END
                        END;
                 IF BLAD=0 THEN
                 BEGIN
                   Hh1:=t1-t;
                   IF (Hh1<Hh) AND (Hh>0) AND (ST<>1)
                     THEN Hh:=Hh1
                     ELSE IF (Hh1>Hh) AND (Hh<0) AND (ST<>1) THEN
                            Hh:=Hh1;
                   IF LICZNIK<>0 THEN
                   BEGIN
                     GOTOXY(HI(LICZNIK),LO(LICZNIK));
                     WRITE(M:5)
                   END
                 END
               END
      UNTIL ((t>=t1) AND (t1>t0)) OR ((t<=t1) AND (t0>t1)) OR (BLAD<>0)
  END { MET_RUN_KUT };

  PROCEDURE FEHLBERG;
    VAR i,j,l                :INTEGER;
        t,dX                 :FLOAT;
        K1,K2,K3,K4,K5,K6,X1 :WEK;
    BEGIN
      BLAD:=0;
      IF K in [1,2,4]
        THEN CASE K OF
               1: BEGIN
                 { para metod wlozonych 1 i 2 rzedu wzory (5.37) }
                    NIELIN(K1,X0,t0,N);   MULRW(K1,h,K1,N);
                    t:=t0+h/2;
                    FOR l:=1 TO N DO
                      X1[l]:=X0[l]+K1[l]/2;
                    NIELIN(K2,X1,t,N);    MULRW(K2,h,K2,N);
                    t:=t0+h;
                    FOR l:=1 TO N DO
                      X1[l]:=X0[l]+K1[l]/256+255*K2[l]/256;
                    NIELIN(K3,X1,t,N);    MULRW(K3,h,K3,N);
                    FOR j:=1 TO N DO
                    BEGIN
                      dX:=(K1[j]+510*K2[j]+K3[j])/512;
                      X[j]:=X0[j]+dX;
                      E[j]:=(-K1[j]+K3[j])/512
                    END
                  END {1};
               2: BEGIN
                 { para metod wlozonych 2 i 3 rzedu wzory (5.38) }
                    NIELIN(K1,X0,t0,N);   MULRW(K1,h,K1,N);
                    t:=t0+h/4;
                    FOR l:=1 TO N DO
                      X1[l]:=X0[l]+K1[l]/4;
                    NIELIN(K2,X1,t,N);    MULRW(K2,h,K2,N);
                    t:=t0+27*h/40;
                    FOR l:=1 TO N DO
                       X1[l]:=X0[l]+(-189*K1[l]+729*K2[l])/800;
                    NIELIN(K3,X1,t,N);    MULRW(K3,h,K3,N);
                    t:=t0+h;
                    FOR l:=1 TO N DO
                      X1[l]:=X0[l]+(214*K1[l]+27*K2[l]+650*K3[l])/891;
                    NIELIN(K4,X1,t,N);    MULRW(K4,h,K4,N);
                    FOR j:=1 TO N DO
                    BEGIN
                      dX:=(533*K1[j]+1600*K3[j]-27*K4[j])/2106;
                      X[j]:=X0[j]+dX;
                      E[j]:=23*K1[j]/1782-K2[j]/33+350*K3[j]/11583-K4[j]/78
                    END
                  END {2};
               3: BEGIN

                  END {3};
               4: BEGIN
                 { para metod wlozonych 4 i 5 rzedu wzory (5.40) }
                    NIELIN(K1,X0,t0,N);   MULRW(K1,h,K1,N);
                    t:=t0+h/4;
                    FOR l:=1 TO N DO
                      X1[l]:=X0[l]+K1[l]/4;
                    NIELIN(K2,X1,t,N);    MULRW(K2,h,K2,N);
                    t:=t0+3*h/8;
                    FOR l:=1 TO N DO
                      X1[l]:=X0[l]+(3*K1[l]+9*K2[l])/32;
                    NIELIN(K3,X1,t,N);    MULRW(K3,h,K3,N);
                    t:=t0+12*h/13;
                    FOR l:=1 TO N DO
                      X1[l]:=X0[l]+(1932*K1[l]-7200*K2[l]+7296*K3[l])/2197;
                    NIELIN(K4,X1,t,N);    MULRW(K4,h,K4,N);
                    t:=t0+h;
                    FOR l:=1 TO N DO
                      X1[l]:=X0[l]+(8341*K1[l]-32832*K2[l]+29440*K3[l]-845*K4[l])/4104;
                    NIELIN(K5,X1,t,N);    MULRW(K5,h,K5,N);
                    t:=t0+h/2;
                    FOR l:=1 TO N DO
                      X1[l]:=X0[l]+(-6080*K1[l]+41040*K2[l]-28352*K3[l]+9295*K4[l]-5643*K5[l])/20520;
                    NIELIN(K6,X1,t,N);    MULRW(K6,h,K6,N);
                    FOR j:=1 TO N DO
                    BEGIN
                      dX:=16*K1[j]/135+6656*K3[j]/12825+28561*K4[j]/56430-9*K5[j]/50+2*K6[j]/55;
                      X[j]:=X0[j]+dX;
                      E[j]:=K1[j]/360-128*K3[j]/4275-2197*K4[j]/75240+K5[j]/50+2*K6[j]/55
                    END
                  END {4}
             END
        ELSE BLAD:=104
  END { FEHLBERG };

  PROCEDURE MET_FEHLB;
    VAR KK,j,l              :BYTE;
        t,ee,s1,
        Hh,Hh1,h1,Hph,Hmin,
        epsm,epsm1,epsw1,
        ALFA,A26,xt,mxt     :FLOAT;
        E,X1,Y              :WEK;
    BEGIN
      BLAD:=0;
      IF ST<>1
        THEN Hh:=(t1-t0)/100
        ELSE Hh:=h0;
     { Wyznaczanie wzglednej dokladnosci maszynowej epsm }
      epsm:=1.0;
      REPEAT
        epsm:=epsm/2;
        epsm1:=epsm+1.0
      UNTIL epsm1<=1;
     { Sprawdzenie dopuszczalnej tolerancji wzglednej wg (5.52)}
      IF epsw>=2*epsm+epswmin
        THEN epsw1:=epsw
        ELSE epsw1:=2*epsm+epswmin;
      A26:=26*epsm;
      IF ABS(t1-t0)<A26 THEN
      BEGIN
        BLAD:=101;
        EXIT
      END;
      t:=t0; Y:=X0;  Y[0]:=t0; M:=0; KK:=0;
      NEWWIERD3(XX,M,BLAD);
      IF BLAD<>0 THEN
        EXIT;
      XX[M]^:=Y;
      REPEAT
        FEHLBERG(K,N,NIELIN,Hh,t,Y,X1,E,BLAD); { pkt 5.4 }
        { Wyznaczanie normy ee wektora bledu E (5.42) oraz }
        { normy mxt wektora stanu o wsp. (5.46) wg (5.47)  }
        ee:=0; mxt:=0;
        FOR j:=1 TO N DO
        BEGIN
          s1:=ABS(E[j]);
          IF ee < s1 THEN
            ee:=s1;
          xt:=(ABS(Y[j])+ABS(X1[j]))/2;
          IF mxt < xt THEN
            mxt:=xt
        END;
        ee:=ee/(mxt*epsw1+epsa);
        Hph:=EXP(LN(ee)/(K+1)); { wzor (5.49) }
        h1:=Hh/Hph;
        ALFA:=0.9/Hph;
        Hmin:=A26*ABS(t) ; { wzor (5.51) }
        IF (ABS(h1) < Hmin) AND (ST<>1) THEN
        BEGIN
          BLAD:=102;
          EXIT
        END;
        IF Hph<=1
          THEN BEGIN { wzory (5.50) }
                 Y:=X1; t:=t+Hh ; X1[0]:=t;
                 INC(M);
                 NEWWIERD3(XX,M,BLAD);
                 IF BLAD<>0 THEN
                   EXIT;
                 XX[M]^:=X1; { zapis rozwiazania }
                 IF (KK=0) AND (ST<>1)
                   THEN BEGIN
                          IF (ALFA<5) AND (ALFA>=0.9)
                            THEN Hh:=ALFA*Hh
                            ELSE IF ALFA>=5 THEN
                                   Hh:=5*Hh
                        END
                   ELSE IF (KK>0) AND (ALFA<1) AND (ST<>1) THEN
                          Hh:=ALFA*Hh;
                 KK:=0
               END
          ELSE BEGIN
                  INC(KK);
                  IF ST<>1
                    THEN Hh:=ALFA*Hh
                    ELSE BEGIN
                           BLAD:=103;
                           EXIT
                         END
               END;
        Hh1:=t1-t;
        IF (Hh1<Hh) AND (Hh>0) AND (ST<>1)
          THEN Hh:=Hh1
          ELSE IF (Hh1>Hh) AND (Hh<0) AND (ST<>1) THEN
                 Hh:=Hh1;
        IF LICZNIK<>0 THEN
        BEGIN
          GOTOXY(HI(LICZNIK),LO(LICZNIK));
          WRITE(M:5)
        END
      UNTIL ((t>=t1) AND (t1>t0)) OR ((t<=t1) AND (t0>t1))
  END { MET_FEHLB } ;

  PROCEDURE GENMACT(VAR T    :MAC;
                        K    :BYTE;
                    VAR BLAD :BYTE);
  { Generacja macierzy Tk (5.92) transformacji Zn=Tk*Yn (5.91)}
    VAR i,j :BYTE;
        S   :FLOAT;
        D   :MAC;
    BEGIN
      { Formowanie macierzy T ukladu rownan (5.86) }
      FOR i:=1 TO K-1 DO
        FOR j:=1 TO K-1 DO
          D[i,j]:=(j+1)*POTEGA(j,-i);
      { Odwracanie macierzy T }
      ODWMAC1(D,K-1,1E-10,BLAD);
      IF BLAD<>0
        THEN BLAD:=113
        ELSE BEGIN
               MACJEDEN(T,K+1);
               FOR i:=1 TO K-1 DO
               BEGIN { Formowanie macierzy Tk (5.92) (5.89) }
                 S:=0;
                 FOR j:=1 TO K-1 DO
                   S:=S-D[i,j];
                 T[i+2,2]:=S;
                 FOR j:=1 TO K-1 DO
                   T[i+2,j+2]:=D[i,j]
               END
             END
    END { GENMACT };

  PROCEDURE GENMACTG(VAR T    :MAC;
                         K    :BYTE;
                     VAR BLAD :BYTE);
  { Generacja macierzy Tk (5.124) transformacji Zn=Tk*Yn (5.118)}
    VAR i,j :BYTE;
        S   :FLOAT;
        C,D :MAC;
    BEGIN
      { Formowanie macierzy T ukladu rownan (5.121) }
      FOR i:=1 TO K-1 DO
        FOR j:=1 TO K-1 DO
          D[i,j]:=POTEGA(j+1,-i);
      { Odwracanie macierzy T }
      ODWMAC1(D,K-1,1E-10,BLAD);
      IF BLAD<>0
        THEN BLAD:=113
        ELSE BEGIN
               MACZERO(T,K+1,K+1);
               T[1,1]:=1; T[2,K+1]:=1;
               FOR i:=1 TO K-1 DO
               BEGIN  { Formowanie macierzy Tk (5.124) }
                 S:=0;
                 FOR j:=1 TO K-1 DO
                   S:=S-D[i,j];
                 T[i+2,1]:=S;
                 S:=0;
                 FOR j:=1 TO K-1 DO
                   S:=S+j*D[i,j];
                 T[i+2,K+1]:=S;
                 FOR j:=1 TO K-1 DO
                   T[i+2,j+1]:=D[i,j]
               END
             END
    END { GENMACTG };

  PROCEDURE DOBORKROKUIRZEDU(    N,Kmax :BYTE;
                             VAR K      :BYTE;
                             VAR ALFA   :FLOAT;
                                 C      :WEK;
                             VAR F      :WEK8;
                                 eps    :FLOAT);
  { Dobor kroku alfa*h poprzez wyznaczenie parametru alfa }
  { oraz rzedu K wg wzorow (5.115) (5.116) (5.117)        }
  { N - ilosc rownan                                      }
  { K - wstepnie zadany rzad metody wielokrokowej         }
  { C - wektor wspolczynnikow bledu obciecia metody wielo-}
  {     krokowej np. (5.65) Adamsa-Multona                }
  { F - macierz Norsiecka typu (5.78)                     }
  { eps - tolerancja bledu (5.114)                        }
    VAR A1,A2,A3:FLOAT;
    BEGIN
      A1:=ALFA; { ALFA obliczone w procedurze DOBORKROKU  }
      IF K<Kmax
        THEN BEGIN
               A2:=eps/(C[K-1]*NORMAX(N,F[K]));
               A3:=eps/(C[K+1]*NORMAX(N,F[K+2]));
               A2:=EXP(LN(A2)/K)/1.3;    { wzor (5.116) }
               A3:=EXP(LN(A3)/(K+2))/1.4;  { wzor (5.117) }
             END
        ELSE BEGIN
               A2:=eps/(C[K-1]*NORMAX(N,F[K]));
               A2:=EXP(LN(A1)/(K))/1.3;
               A3:=0
             END;
      { Dobor rzedu K=1..6 i kroku ALFA*h poprzez wybor }
      { maksimum z posrod A1,A2,A3                      }
      IF (A1>=A2) AND (A1>=A3)
        THEN ALFA:=A1
        ELSE IF (A2>A1) AND (A2>=A3)
                THEN IF K>1
                       THEN BEGIN
                              ALFA:=A2;
                              DEC(K)
                            END
                       ELSE ALFA:=A1
                ELSE IF (A3>A1) AND (A3>A2) THEN
                       IF K<Kmax
                         THEN BEGIN
                                ALFA:=A3;
                                INC(K)
                              END
                         ELSE ALFA:=A1
    END { DOBORKROKUIRZEDU };

  PROCEDURE DOBORKROKU(    N     :BYTE;
                       VAR K     :BYTE;
                       VAR ALFA  :FLOAT;
                           C     :WEK;
                       VAR F     :WEK8;
                           eps   :FLOAT);
  { Dobor kroku alfa*h poprzez wyznaczenie parametru alfa }
  { wg wzoru (5.115)                                      }
  { N - ilosc rownan                                      }
  { K - zadany rzad metody wielokrokowej                  }
  { C - wektor wspolczynnikow bledu obciecia metody wielo-}
  {     krokowej np. (5.65) Adamsa-Multona                }
  { F - macierz Norsiecka typu (5.78)                     }
  { eps - tolerancja bledu (5.114)                        }
    VAR A1:FLOAT;
    BEGIN
      A1:=eps/(C[K]*NORMAX(N,F[K+1]));
      A1:=EXP(LN(A1)/(K+1))/1.2; { wzor (5.115) }
      ALFA:=A1
    END { DOBORKROKU };

  PROCEDURE ZMIANAMACNOR(    N,K  :BYTE;
                         VAR F    :WEK8;
                             ALFA :FLOAT);
  { Transformacja macierzy Nordsiecka wg (5.82) przy zmianie }
  { kroku danym parametrem ALFA                              }
  { N - ilosc rownan                                         }
  { K - zadany rzad metody wielokrokowej                     }
  { F - macierz Norsiecka typu (5.78)                        }
    VAR i :BYTE;
        A :FLOAT;
    BEGIN
      A:=1;
      FOR i:=1 TO K+2 DO
      BEGIN
        A:=A*ALFA;
        MULRW(F[i],A,F[i],N)
      END
    END { ZMIANAMACNOR };

  PROCEDURE PREDYKTOR_MW(    N,K :BYTE;
                         VAR F   :WEK8);
  { Zastosowanie macierzy trojkatnej Pascala (5.98) w czlonie  }
  { przewidywania (5.93) dla macierzy Nordsiecka F typu (5.78) }
    VAR i,j,l:BYTE;
    BEGIN
      FOR i:=1 TO K DO
        FOR j:=i TO K DO
        BEGIN
          l:=K-j+i-1;
          ADDWEK(F[l],F[l],F[l+1],N)
        END
    END { PREDYKTOR_MW };

    PROCEDURE START_MW(    ST,N,K            :BYTE;
                           NIELIN            :PROCXT;
                           X0                :WEK;
                       VAR F                 :WEK8;
                       VAR h,t,A26           :FLOAT;
                           t0,t1             :FLOAT;
                       VAR epsw,epswmin,epsa :FLOAT;
                       VAR M                 :WORD;
                       VAR XX                :MACD3;
                       VAR BLAD              :BYTE);
{-------------------------------------------------------------}
{ Faza wstepna obliczen metody wielokrokowej z zastosowaniem  }
{ transformacji (5.91) do wstepnego sformulowania macierzy    }
{ Nordsiecka                                                  }
{ ST - parametr pomocniczy                                    }
{ ST=1 staly krok calkowania przy automatycznym doborze rzedu }
{ ST=2 automatyczny dobor kroku i rzedu                       }
{ N  - ilosc rownan                                           }
{ K  - rzad metody jednokrokowej Fehlberga przyjety do fazy   }
{      poczatkowej metody wielokrokowj                        }
{ NIELIN  - procedura typu PROCXT okreslajaca odwzorowanie    }
{           wektora X:WEK oraz t:FLOAT w wektor Y:WEK         }
{           tj.generacja funkcji wektorowej prawej strony     }
{           rownania rozniczkowego   Y = F ( X , t )          }
{               dX/dt = F ( X , t );                          }
{ X0 - wektor wartosci poczatkowych;                          }
{ F  - macierz Norsiecka typu (5.78) przygotowana na starcie  }
{      metody wielokrokowej wg transformacji (5.91)           }
{ h  - krok calkowania-wymaga wstepnego ustalenia dla ST=1    }
{ t  - przyjmuje wartosci kolejnych chwil calkowania t:=t+h   }
{      i wymaga wstepnego przyjecia t=t0                      }
{ A26- parametr pomocniczy we wzorze (5.26)                   }
{ t0,t1   - poczatek i koniec przedzialu calkowania <t0,t1>   }
{ epsw    - tolerancja bledu wzglednego                       }
{ epswmin - najmniejsza tolerancja bledu wzglednego jakiego   }
{           dopuszczamy w obliczeniach (np. 1E-10)            }
{ epsa    - tolerancja bledu absolutnego                      }
{ M       - ilosc iteracji obliczona i zapisana w macierzy XX }
{ XX      - macierz dynamiczna do zapisania rozwiazania;      }
{ BLAD    - nr bledu; 0 - brak bledu                          }
{-------------------------------------------------------------}
      VAR i,j                :BYTE;
          NE,epsm,epsm1,eps1 :FLOAT;
          Y,X1,WK,WK1,E,FK   :WEK;
          F0                 :WEK8;
          TK                 :MAC;

      PROCEDURE TRANSFORMACJA; { wg wzoru (5.91) }
         VAR i,j:BYTE;
         BEGIN
          FOR i:=1 TO K-1 DO
          BEGIN
            FOR j:=1 TO N DO
              Y[j]:=0;
            FOR j:=1 TO K DO
            BEGIN
              MULRW(X1,TK[i+2,j+1],F0[j],N);
              ADDWEK(Y,Y,X1,N)
            END;
            F[i+1]:=Y
          END
         END { TRANSFORMACJA };

      BEGIN
        BLAD:=0;
       { Wyznaczanie wzglednej dokladnosci maszynowej epsm }
        epsm:=1.0;
        REPEAT
          epsm:=epsm/2;
          epsm1:=epsm+1.0
        UNTIL epsm1<=1;
       {Sprawdzenie dopuszczalnej tolerancji wzglednej wg (5.52)}
        IF epsw<2*epsm+epswmin THEN
          epsw:=2*epsm+epswmin;
        A26:=26*epsm;
        IF ABS(t1-t0)<A26 THEN
        BEGIN
          BLAD:=101;
          EXIT
        END;
        { Generacja macierzy (5.92) pkt 5.7.3 }
        GENMACT(TK,K,BLAD);
        IF BLAD<>0 THEN
          EXIT;
        i:=0;
        NIELIN(FK,X0,t0,N);
        F0[K]:=FK;
        IF ST<>1 THEN
        BEGIN
          FEHLBERG(4,N,NIELIN,h,t0,X0,Y,E,BLAD);
          NE:=NORMAX(N,E)/(epsw*NORMAX(N,Y)+epsa);
          h:=0.9*h/EXP(LN(NE)/5)
        END;
        Y:=X0; t:=t0;
        REPEAT
          INC(i);
          FEHLBERG(4,N,NIELIN,h,t,Y,X1,E,BLAD);  { pkt 5.5 }
          NE:=NORMAX(N,E);
          eps1:=epsw*NORMAX(N,Y)+epsa;
          IF NE<eps1
            THEN BEGIN
                   Y:=X1;
                   t:=t+h;
                   NIELIN(F0[K-i],X1,t,N);
                   X1[0]:=t;
                   INC(M);
                   NEWWIERD3(XX,M,BLAD);
                   IF BLAD=0 THEN
                     XX[M]^:=X1  { zapis rozwiazania }
                 END
            ELSE IF ST<>1
                   THEN BEGIN
                          h:=h/2; i:=0; t:=t0; Y:=X0;
                          DISMACD3(XX,M);
                          M:=0;  X1:=X0;  X1[0]:=t0;
                          { zapis rozwiazania od poczatku }
                          NEWWIERD3(XX,M,BLAD);
                          IF BLAD=0 THEN
                            XX[M]^:=X1;
                        END
                   ELSE BLAD:=105;
           IF ABS(h) < A26*ABS(t) THEN
             BLAD:=106
        UNTIL (i=K-1) OR (BLAD<>0);
        IF BLAD=0 THEN
        BEGIN
          FOR i:=1 TO K DO
            MULRW(F0[i],h,F0[i],N);
          F0[0]:=X1; F[1]:=F0[1]; F[0]:=X1;
          TRANSFORMACJA
        END
      END { START_MW };

    PROCEDURE START_MWG(    ST,N,K            :BYTE;
                            NIELIN            :PROCXT;
                            X0                :WEK;
                        VAR F                 :WEK8;
                        VAR h,t,A26           :FLOAT;
                            t0,t1             :FLOAT;
                        VAR epsw,epswmin,epsa :FLOAT;
                        VAR M                 :WORD;
                        VAR XX                :MACD3;
                        VAR BLAD              :BYTE);
{-------------------------------------------------------------}
{ Faza wstepna obliczen metody wielokrokowej z zastosowaniem  }
{ transformacji (5.118) do wstepnego sformulowania macierzy   }
{ Nordsiecka                                                  }
{ Opis parametrow jak w procedurze START_MW                   }
{-------------------------------------------------------------}
      VAR i,j                :BYTE;
          NE,epsm,epsm1,eps1 :FLOAT;
          Y,X1,WK,WK1,E      :WEK;
          F0                 :WEK8;
          TK                 :MAC;

       PROCEDURE TRANSFORMACJA; { wg wzoru (5.118) }
         VAR i,j:BYTE;
         BEGIN
           FOR i:=2 TO K DO
           BEGIN
             FOR j:=1 TO N DO
               Y[j]:=0;
             FOR j:=0 TO K DO
             BEGIN
               MULRW(X1,TK[i+1,j+1],F0[j],N);
               ADDWEK(Y,Y,X1,N)
             END;
             F[i]:=Y
           END
         END { TRANSFORMACJA };

      BEGIN
        BLAD:=0;
       { Wyznaczanie wzglednej dokladnosci maszynowej epsm }
        epsm:=1.0;
        REPEAT
          epsm:=epsm/2;
          epsm1:=epsm+1.0
        UNTIL epsm1<=1;
       {Sprawdzenie dopuszczalnej tolerancji wzglednej wg (5.52)}
        IF epsw<2*epsm+epswmin THEN
          epsw:=2*epsm+epswmin;
        A26:=26*epsm;
        IF ABS(t1-t0)<A26 THEN
        BEGIN
          BLAD:=101;
          EXIT
        END;
        { Generacja macierzy (5.124) pkt 5.7.4 }
        GENMACTG(TK,K,BLAD);
        IF BLAD<>0 THEN
          EXIT;
        i:=1;
        F0[K-1]:=X0;
        IF ST<>1 THEN
        BEGIN
          FEHLBERG(4,N,NIELIN,h,t0,X0,Y,E,BLAD);
          NE:=NORMAX(N,E)/(epsw*NORMAX(N,Y)+epsa);
          h:=0.9*h/EXP(LN(NE)/5)
        END;
        Y:=X0; t:=t0;
        REPEAT
          INC(i);
          FEHLBERG(4,N,NIELIN,h,t,Y,X1,E,BLAD); { pkt 5.5 }
          NE:=NORMAX(N,E); eps1:=epsw*NORMAX(N,Y)+epsa;
          IF NE<eps1
            THEN BEGIN
                   Y:=X1; t:=t+h; F0[K-i]:=X1;
                   X1[0]:=t; INC(M);
                   NEWWIERD3(XX,M,BLAD);
                   IF BLAD=0 THEN
                     XX[M]^:=X1;  { zapis rozwiazania }
                 END
            ELSE IF ST<>1
                   THEN BEGIN
                          h:=h/2; i:=0; t:=t0; Y:=X0;
                          DISMACD3(XX,M);
                          M:=0;
                          X1:=X0;  X1[0]:=t0;
                          NEWWIERD3(XX,M,BLAD);
                          IF BLAD=0 THEN
                            XX[M]^:=X1   { zapis rozwiazania od  poczatku}
                        END
                   ELSE BLAD:=105;
          IF ABS(h) < A26*ABS(t) THEN
            BLAD:=106
        UNTIL (i=K) OR (BLAD<>0);
        IF BLAD=0 THEN
        BEGIN
          F[0]:=F0[0];
          NIELIN(F0[K],X1,t,N);
          MULRW(F0[K],h,F0[K],N);
          F[1]:=F0[K];
          TRANSFORMACJA
        END
      END { START_MWG };


  PROCEDURE MET_ADAMS_MUL;
    VAR K,K1,K2,KKR,j,l         :BYTE;
        Q,t,ee,s1,h1,h2,h,Hmin,
        epsm,epsm1,eps,
        ALFA,A26,xt,mxt         :FLOAT;
        C,X1                    :WEK;
        i                       :WORD;
        F,FP                    :WEK8;
        PIERWSZY                :BOOLEAN;

    PROCEDURE PODSTAW(VAR C:WEK);
    { Podstawienie wspolczynnikow wystepujacych przy bledzie }
    { obciecia dla algorytmu Adamsa-Multona wg wzoru (5.66)  }
    { pod wektor C wg.  C[k]:=C[k+1]*(k+2)!                  }
      BEGIN
       C[1]:=0.5; C[2]:=1; C[3]:=19/6;
       C[4]:=27/2; C[5]:=863/12; C[6]:=1375/3
      END { PODSTAW };

    PROCEDURE KOREKTOR_AM(    ST,N,K      :BYTE;
                          VAR t,h,Hmin,
                              epsw1,epsa1 :FLOAT;
                          VAR F           :WEK8;
                          VAR BLAD        :BYTE);
    { FLOATizacja czlonu korekcji Adamsa-Multona w postaci    }
    { macierzowej Nordsiecka F typu (5.78) wg iteracji (5.94)}
    { K - rzad czlonu korekcji                               }
    { t - chwila czasu w ktorej dokonuje sie korekcji        }
    {     jezeli zachodzi zbieznosc korektora to t:=t+h      }
    { h - krok calkowania - jezeli nie zachodzi zbieznosc    }
    {     korektora po trzech probach  to h:=h/4             }
    { Hmin - minimalny krok calkowania przy ktorym nalezy    }
    {        przerwac obliczanie                             }
    { epsw1,epsa1 - tolerancja wzgledna i absolutna obliczen }
    { F - macierz Norsiecka typu (5.78)                      }
    { BLAD - nr bledu; 0 - brak bledu                        }
      VAR KZ,i,j                  :BYTE;
          NFz,r,eps               :FLOAT;
          Y,Cz,Fz,FK,FK1,Fm,F0,F1 :WEK;

          PROCEDURE POWTORZ;
          { Realizuje powtorzenie predykcji (5.93) jezeli po   }
          { trzech probach nie osiaga sie zbieznosci korektora }
          { z zadana tolerancja przyjmujac h:=h/4              }
            VAR i:BYTE;
            BEGIN
              h:=0.25*h; F:=FP;
              ZMIANAMACNOR(N,K,F,0.25);
              FK:=F[K]; FK1:=F[K+1]; F0:=F[0]; F1:=F[1];
              PREDYKTOR_MW(N,K,F);
              FOR i:=1 TO N DO
                Fm[i]:=0;
            END { POWTORZ };

      BEGIN
        BLAD:=0;
        IF (K>=1) AND (K<=6)
          THEN BEGIN
               { Wybor rzedu metody Adamsa-Multona wraz z podstawie- }
               { niem za wektor Cz wektora obliczonego wg (5.96) dla }
               { rzedu K rozpatrywanej metody                        }
                 CASE K OF
                   1: BEGIN
                        Cz[1]:=0.5;     Cz[2]:=1;
                      END {1};
                   2: BEGIN
                        Cz[1]:=5/12;    Cz[2]:=1;  Cz[3]:=0.5;
                      END {2};
                   3: BEGIN
                        Cz[1]:=3/8;     Cz[2]:=1;  Cz[3]:=3/4; Cz[4]:=1/6;
                      END {3};
                   4: BEGIN
                        Cz[1]:=251/720; Cz[2]:=1;  Cz[3]:=11/12;
                        Cz[4]:=1/3; Cz[5]:=1/24;
                      END {4};
                   5: BEGIN
                        Cz[1]:=475/1440; Cz[2]:=1;  Cz[3]:=25/24;
                        Cz[4]:=35/72; Cz[5]:=5/48; Cz[6]:=1/120;
                      END {5};
                   6: BEGIN
                        Cz[1]:=0.315591931215295; Cz[2]:=1;
                        Cz[3]:=137/120; Cz[4]:=5/8; Cz[5]:=17/96;
                        Cz[6]:=1/40; Cz[7]:=1/720
                      END {6};
                 END;
                 FK:=F[K]; FK1:=F[K+1]; F0:=F[0]; F1:=F[1];
                 eps:=epsw1*NORMAX(N,F[0])+epsa1;
                 FOR i:=1 TO N DO
                   Fm[i]:=0;
                 KZ:=0;
                 { Iteracja pierwszych dwoch wierszy F[0] i F[1]  }
                 { macierzy Nordsiecka wg (5.94)                  }
                 REPEAT
                   INC(KZ);
                   IF (KZ>3) AND (ST<>1) THEN
                   BEGIN
                     POWTORZ;
                     KZ:=1
                   END;
                   NIELIN(Fz,F0,t+h,N); MULRW(Fz,h,Fz,N);
                   SUBWEK(Fz,Fz,F1,N); { Roznica wektorowa (5.102) }
                   MULRW(Y,Cz[1],Fz,N); ADDWEK(F0,F0,Y,N);
                   MULRW(Y,Cz[2],Fz,N); ADDWEK(F1,F1,Y,N);
                   ADDWEK(Fm,Fm,Fz,N); { Suma wektorowa (5.103) }
                   NFz:=NORMAX(N,Fz);
                   IF (ABS(h) < Hmin) AND (ST<>1) THEN
                   BEGIN
                     BLAD:=107;
                     EXIT
                   END
                 UNTIL ((NFz<eps) AND (KZ<=3) AND (ST<>1)) OR
                       ((NFz<eps) AND (ST=1)) OR (KZ>25);
                 IF KZ>25 THEN
                 BEGIN
                   BLAD:=108;
                   EXIT
                 END;
                 F[0]:=F0; F[1]:=F1;
                 { Wyznaczanie pozostalych wierszy macierzy }
                 { Nordsiecka  wg wzoru (5.104)             }
                 FOR i:=2 TO K DO
                 BEGIN
                   MULRW(Y,Cz[i+1],Fm,N);
                   ADDWEK(F[i],F[i],Y,N)
                 END;
                 { Oszacowanie pochodnych rzedu K+1 i K+2 tj F[K+1] }
                 { F[K+2] jako wektorow roznicy wstecznej wg wzorow }
                 { (5.80)i (5.81)                                   }
                 SUBWEK(F[K+1],F[K],FK,N); r:=1/(K+1);
                 MULRW(F[K+1],r,F[K+1],N);
                 SUBWEK(F[K+2],F[K+1],FK1,N); r:=1/(K+2);
                 MULRW(F[K+2],r,F[K+2],N);
                 {t:=t+h}
               END
          ELSE BLAD:=109
      END { KOREKTOR_AM };

  BEGIN { Czesc operacyjna procedury MET_ADAMS_MUL }
    BLAD:=0;
    IF ST<>1
      THEN h:=(t1-t0)/100
      ELSE h:=h0;
    t:=t0;  X1:=X0;  X1[0]:=t0;  M:=0;
    NEWWIERD3(XX,M,BLAD);
    IF BLAD<>0 THEN
      EXIT;
    XX[M]^:=X1;  { zapis rozwiazania }
    PODSTAW(C);{ Podstawienie wspolczynnikow bledu (5.66) }
               { pod wektor C                             }
    START_MWG(ST,N,4,NIELIN,X0,F,h,t,A26,t0,t1,epsw,epswmin,epsa,M,XX,BLAD);
    IF BLAD<>0 THEN
      EXIT;
   { Faza wstepna obliczen metody wielokrokowej z zastoso-  }
   { waniem transformacji (5.118) do wstepnego sformulowania}
   { macierzy Nordsiecka F  pkt 5.7.4                       }
    KKR:=0; K:=3;
    PIERWSZY:=TRUE; FP:=F; h2:=h; K2:=K;
    eps:=epsw*NORMAX(N,F[0])+epsa;
    { prawa strona nierownosci (5.114) }
    DOBORKROKU(N,K,ALFA,C,F,eps);
    REPEAT
      IF (ST=1) AND (ALFA<1) THEN
      BEGIN
        BLAD:=103;
        EXIT
      END;
      IF PIERWSZY
        THEN BEGIN
               IF ST<>1 THEN
               BEGIN
                 h:=ALFA*h;
                 ZMIANAMACNOR(N,K,F,ALFA)
               END;
               PIERWSZY:=FALSE
             END
        ELSE IF (ALFA<1) OR ((ALFA>1) AND (KKR=K+1))
               THEN BEGIN
                      K1:=K;
                      DOBORKROKUIRZEDU(N,6,K,ALFA,C,F,eps);
                      IF ST<>1 THEN
                      BEGIN
                        h:=ALFA*h;
                        ZMIANAMACNOR(N,K1,F,ALFA);
                      END;
                      KKR:=0
                    END
               ELSE IF (ALFA>=2) AND (ST<>1) THEN
                    BEGIN
                      h:=2*h;
                      ZMIANAMACNOR(N,K,F,2.0)
                    END;
      { Dobor ostatniego kroku calkowania h1 celem }
      { trafienia do punktu koncowego  t1          }
      h1:=t1-t;
      IF (h1<h) AND (h>0)
        THEN BEGIN
               ALFA:=ABS(h1/h);
               h:=h1;
               ZMIANAMACNOR(N,K,F,ALFA)
             END
        ELSE IF (h1>h) AND (h<0) THEN
             BEGIN
               ALFA:=ABS(h1/h);
               h:=h1;
               ZMIANAMACNOR(N,K,F,ALFA)
             END;
      Hmin:=A26*ABS(t) ; { wzor (5.51) }
      IF (ABS(h) < Hmin) AND (ST<>1) THEN
      BEGIN
        BLAD:=107;
        EXIT
      END;
      REPEAT
        PREDYKTOR_MW(N,K,F);
        KOREKTOR_AM(ST,N,K,t,h,Hmin,epsw,epsa,F,BLAD);
        IF BLAD<>0 THEN
          EXIT;
        { Sprawdzenie kryterium bledu (5.114) rownowazne }
        { warunkowi ALFA>=1 po korekcie Adamsa-Multona   }
        eps:=epsw*NORMAX(N,F[0])+epsa;
        DOBORKROKU(N,K,ALFA,C,F,eps);
        ee:=1.2*ALFA;
        IF ee<1 THEN
        BEGIN
          h:=h2/2; K:=K2; F:=FP;
          ZMIANAMACNOR(N,K,F,0.5);
          FP:=F; h2:=h
        END
      UNTIL ee>=1; { Warunek dopuszczalnego bledu }
      INC(M); t:=t+h;  X1:=F[0]; X1[0]:=t;
      NEWWIERD3(XX,M,BLAD);
      IF BLAD<>0 THEN
        EXIT;
      XX[M]^:=X1;  { zapis rozwiazania }
      INC(KKR); FP:=F; h2:=h; K2:=K;
      IF LICZNIK<>0 THEN
      BEGIN
        GOTOXY(HI(LICZNIK),LO(LICZNIK));
        WRITE(M:5)
      END
    UNTIL ((t>=t1)AND(t1>t0)) OR ((t<=t1)AND(t0>t1))
  END { MET_ADAMS_MUL };

  PROCEDURE MACJAK(    NIELIN :PROCXT;
                       X      :WEK;
                       N      :BYTE;
                   VAR A      :MAC;
                       t,eps  :FLOAT);
  { Wyznaczanie macierzy Jacobiego A=dF/dX wg wzoru (5.133) }
  { NIELIN - procedura generujaca funkcje wektorowa prawej  }
  {          strony rownania rozniczkowego Y=F(X,t)         }
  {          dX/dt = F(X,t);                                }
      VAR  i,j     :BYTE;
           h,r     :FLOAT;
           X1,Y,Y1 :WEK;
      BEGIN
        NIELIN(Y,X,t,N);
        FOR i:=1 TO N DO
        BEGIN
          r:=ABS(X[i]);
          IF r>eps
            THEN h:=eps*r
            ELSE h:=eps;
          X1:=X; X1[i]:=X[i]+h;
          NIELIN(Y1,X1,t,N);
          FOR j:=1 TO N DO
            A[j,i]:=(Y1[j]-Y[j])/h
        END
      END { MACJAK };

  PROCEDURE MET_GEAR;
    VAR K,K1,K2,P2,KKR,j,l :BYTE;
        Q,t,ee,s1,h1,h2,h,Hmin,
        epsm,epsm1,eps,
        ALFA,A26,xt,mxt    :FLOAT;
        C,X1               :WEK;
        i                  :WORD;
        F,FP               :WEK8;
        PIERWSZY           :BOOLEAN;

    PROCEDURE PODSTAW(VAR C:WEK);
    { Podstawienie wspolczynnikow wystepujacych przy bledzie }
    { obciecia dla algorytmu Geara wg wzoru (5.130)          }
    { pod wektor C wg.  C[k]:=C[k]*(k+1)!                    }
      BEGIN
       C[1]:=1.0; C[2]:=4/3; C[3]:=36/11;
       C[4]:=288/25; C[5]:=7200/137; C[6]:=14400/49
      END {PODSTAW};

    PROCEDURE KOREKTOR_GEARA(    ST,N,K      :BYTE;
                             VAR t,h,Hmin,
                                 epsw1,epsa1 :FLOAT;
                             VAR F           :WEK8);
    { FLOATizacja czlonu korekcji GEARA w postaci macierzowej }
    { Nordsiecka F typu (5.78) wg iteracji (5.147)           }
    { K - rzad czlonu korekcji                               }
    { t - chwila czasu w ktorej dokonuje sie korekcji        }
    {     jezeli zachodzi zbieznosc korektora to t:=t+h      }
    { h - krok calkowania - jezeli nie zachodzi zbieznosc    }
    {     korektora po trzech probach  to h:=h/4             }
    { Hmin - minimalny krok calkowania przy ktorym nalezy    }
    {        przerwac obliczanie                             }
    { epsw1,epsa1 - tolerancja wzgledna i absolutna obliczen }
    { F - macierz Norsiecka typu (5.78)                      }
      VAR KZ,i,j                     :BYTE;
          NFz,r,eps                  :FLOAT;
          Y,Cz,Fa,Fz,FK,FK1,Fm,F0,F1 :WEK;
          A                          :MAC;

      PROCEDURE POWTORZ;
      { Realizuje powtorzenie predykcji (5.93) jezeli po   }
      { trzech probach nie osiaga sie zbieznosci korektora }
      { Geara z zadana tolerancja przyjmujac h:=h/4        }
        VAR i:BYTE;
        BEGIN
          h:=0.25*h; F:=FP;
          ZMIANAMACNOR(N,K,F,0.25);
          FK:=F[K]; FK1:=F[K+1]; F0:=F[0]; F1:=F[1];
          PREDYKTOR_MW(N,K,F);
          FOR i:=1 TO N DO
            Fm[i]:=0;
        END { POWTORZ };

      BEGIN
        BLAD:=0;
        IF (K>=1) AND (K<=6)
          THEN BEGIN
               { Wybor rzedu metody Geara wraz z podstawieniem za }
               { wektor Cz wektora obliczonego wg (5.149) (5.150) }
               { dla rzedu K rozpatrywanej metody                 }
                 CASE K OF
                   1: BEGIN
                        Cz[1]:=1.0; Cz[2]:=1.0;
                      END {1};
                   2: BEGIN
                        Cz[1]:=2/3; Cz[2]:=1.0;  Cz[3]:=1/3;
                      END {2};
                   3: BEGIN
                        Cz[1]:=6/11; Cz[2]:=1;  Cz[3]:=6/11; Cz[4]:=1/11;
                      END {3};
                   4: BEGIN
                        Cz[1]:=12/25; Cz[2]:=1;  Cz[3]:=0.7;
                        Cz[4]:=0.2; Cz[5]:=0.02;
                      END {4};
                   5: BEGIN
                        Cz[1]:=60/137; Cz[2]:=1;  Cz[3]:=225/274;
                        Cz[4]:=85/274; Cz[5]:=15/274; Cz[6]:=1/274;
                      END {5};
                   6: BEGIN
                        Cz[1]:=20/49; Cz[2]:=1;  Cz[3]:=58/63;
                        Cz[4]:=5/12; Cz[5]:=25/252; Cz[6]:=1/84;
                        Cz[7]:=1/1764
                      END {6}
                 END;
                 FK:=F[K];   FK1:=F[K+1];  F0:=F[0]; F1:=F[1];
                 eps:=epsw1*NORMAX(N,F[0])+epsa1;
                 FOR i:=1 TO N DO
                   Fm[i]:=0;
                 KZ:=0;
                 { Iteracja pierwszych dwoch wierszy F[0] i F[1]  }
                 { macierzy Nordsiecka wg (5.147)                 }
                 REPEAT
                   INC(KZ);
                   IF (KZ>3) AND (ST<>1) THEN
                   BEGIN
                     POWTORZ;
                     KZ:=1
                   END;
                   { Formowanie wektora Fz wg wzoru (5.148) }
                   NIELIN(Fa,F0,t+h,N); MULRW(Fa,h,Fa,N);
                   SUBWEK(Fa,Fa,F1,N);
                   { Macierz Jacobiego A=dF/dX wg (5.133) }
                   MACJAK(NIELIN,F0,N,A,t,1E-4); r:=h*Cz[1];
                   FOR i:=1 TO N DO
                     FOR j:=1 TO N DO
                       IF i=j
                         THEN A[i,i]:=1-r*A[i,i]
                         ELSE A[i,j]:=-r*A[i,j];
                   { Macierz odwrotna A                }
                   { procedura ODWMAC z modulu ALGELIN }
                   ODWMAC1(A,N,1E-12,BLAD);
                   IF BLAD=0
                     THEN BEGIN
                            FOR i:=1 TO N DO
                            BEGIN
                              r:=0;
                              FOR j:=1 TO N DO
                                r:=r+A[i,j]*Fa[j];
                              Fz[i]:=r
                            END {i};
                            MULRW(Y,Cz[1],Fz,N); ADDWEK(F0,F0,Y,N);
                            MULRW(Y,Cz[2],Fz,N); ADDWEK(F1,F1,Y,N);
                            ADDWEK(Fm,Fm,Fz,N); { Suma wektorowa (5.103) }
                            NFz:=NORMAX(N,Fz)
                          END
                     ELSE IF ST<>1
                            THEN BEGIN
                                   POWTORZ;
                                   KZ:=0;
                                   BLAD:=0;
                                 END
                            ELSE BEGIN
                                   BLAD:=110;
                                   EXIT
                                 END;
                   IF ABS(h) < Hmin THEN
                   BEGIN
                     BLAD:=107;
                     EXIT
                   END
                 UNTIL ((NFz<eps) AND (KZ<=3) AND (ST<>1)) OR
                       ((NFz<eps) AND (ST=1)) OR (KZ>25);
                 IF KZ>25 THEN
                 BEGIN
                   BLAD:=111;
                   EXIT
                 END;
                 F[0]:=F0; F[1]:=F1;
                 { Wyznaczanie pozostalych wierszy macierzy }
                 { Nordsiecka  wg wzoru (5.104)             }
                 FOR i:=2 TO K DO
                 BEGIN
                   MULRW(Y,Cz[i+1],Fm,N);
                   ADDWEK(F[i],F[i],Y,N)
                 END;
                 { Oszacowanie pochodnych rzedu K+1 i K+2 tj F[K+1] }
                 { F[K+2] jako wektorow roznicy wstecznej wg wzorow }
                 { (5.80)i (5.81)                                   }
                 SUBWEK(F[K+1],F[K],FK,N); r:=1/(K+1);
                 MULRW(F[K+1],r,F[K+1],N);
                 SUBWEK(F[K+2],F[K+1],FK1,N); r:=1/(K+2);
                 MULRW(F[K+2],r,F[K+2],N);
                 {t:=t+h}
               END
          ELSE BLAD:=112
      END { KOREKTOR_GEARA };

    BEGIN
      BLAD:=0;
      IF ST<>1
        THEN h:=(t1-t0)/100
        ELSE h:=h0;
      t:=t0;  X1:=X0;  X1[0]:=t0;  M:=0;
      NEWWIERD3(XX,M,BLAD);
      IF BLAD<>0 THEN
        EXIT;
      XX[M]^:=X1;  { zapis rozwiazania }
      PODSTAW(C);{ Podstawienie wspolczynnikow bledu (5.130) }
                 { pod wektor C                              }
      START_MWG(ST,N,4,NIELIN,X0,F,h,t,A26,t0,t1,epsw,epswmin,epsa,M,XX,BLAD);
      IF BLAD<>0 THEN
        EXIT;
     { Faza wstepna obliczen metody wielokrokowej z zastoso-  }
     { waniem transformacji (5.118) do wstepnego sformulowania}
     { macierzy Nordsiecka F  pkt 5.7.4                       }
      KKR:=0; K:=3;
      PIERWSZY:=TRUE; FP:=F; h2:=h; K2:=K;
      eps:=epsw*NORMAX(N,F[0])+epsa;
      DOBORKROKU(N,K,ALFA,C,F,eps);
      REPEAT
        IF (ST=1) AND (ALFA<1) THEN
        BEGIN
          BLAD:=103;
          EXIT
        END;
        IF PIERWSZY
          THEN BEGIN
                 IF ST<>1 THEN
                 BEGIN
                   h:=ALFA*h; ZMIANAMACNOR(N,K,F,ALFA)
                 END;
                 PIERWSZY:=FALSE
               END
          ELSE IF (ALFA<1) OR ((ALFA>1) AND (KKR=K+1))
                 THEN BEGIN
                        K1:=K;
                        DOBORKROKUIRZEDU(N,6,K,ALFA,C,F,eps);
                        IF ST<>1 THEN
                        BEGIN
                          h:=ALFA*h;
                          ZMIANAMACNOR(N,K1,F,ALFA);
                        END;
                        KKR:=0
                      END
                 ELSE IF (ALFA>=2) AND (ST<>1) THEN
                      BEGIN
                        h:=ALFA*h;
                        ZMIANAMACNOR(N,K,F,ALFA)
                      END;
        { Dobor ostatniego kroku calkowania h1 celem }
        { trafienia do punktu koncowego  t1          }
        h1:=t1-t;
        IF (h1<h) AND (h>0) AND (ST<>1)
          THEN BEGIN
                 ALFA:=ABS(h1/h); h:=h1;
                 ZMIANAMACNOR(N,K,F,ALFA)
               END
          ELSE IF (h1>h) AND (h<0) AND (ST<>1) THEN
               BEGIN
                 ALFA:=ABS(h1/h); h:=h1; ZMIANAMACNOR(N,K,F,ALFA)
               END;
        Hmin:=A26*ABS(t);
        IF (ABS(h) < Hmin) AND (ST<>1) THEN
        BEGIN
          BLAD:=107;
          EXIT
        END;
        REPEAT
          PREDYKTOR_MW(N,K,F);
          KOREKTOR_GEARA(ST,N,K,t,h,Hmin,epsw,epsa,F);
          { Sprawdzenie kryterium bledu (5.114) rownowazne }
          { warunkowi ALFA>=1 po korekcie Geara            }
          eps:=epsw*NORMAX(N,F[0])+epsa;
          DOBORKROKU(N,K,ALFA,C,F,eps);
          ee:=1.2*ALFA;
          IF ee<1 THEN
          BEGIN
            h:=h2/2; K:=K2; F:=FP;
            ZMIANAMACNOR(N,K,F,0.5);
            FP:=F; h2:=h
          END
        UNTIL ee>=1; { Warunek dopuszczalnego bledu }
        INC(M); t:=t+h;  X1:=F[0]; X1[0]:=t;
        NEWWIERD3(XX,M,BLAD);
        IF BLAD<>0 THEN
          EXIT;
        XX[M]^:=X1;  { zapis rozwiazania }
        KKR:=KKR+1; FP:=F; h2:=h; K2:=K;
        IF LICZNIK<>0 THEN
        BEGIN
          GOTOXY(HI(LICZNIK),LO(LICZNIK));
          WRITE(M:5)
        END
      UNTIL ((t>=t1)AND(t1>t0)) OR ((t<=t1)AND(t0>t1))
  END { MET_GEAR };


END.


