{***************************************************************************}
{                          Modul RONIELIN                                   }
{                       ROWNANIA NIELINIOWE                                 }
{                     Turbo Pascal  wersja 7.0                              }
{                       autor Bernard Baron                                 }
{***************************************************************************}
UNIT RONIELIN;

{$IFDEF CPU87} {$N+}
       {$ELSE} {$N-}
{$ENDIF}
{$F+}

INTERFACE

USES TFLOAT,ALGELIN,ALGEZES,ALGMZES,MINFUNBO;

TYPE PROCX =PROCEDURE(VAR F:WEK;
                          X:WEK;
                          N:BYTE);


     FUNCTION NORMAWEK( X:WEK;
                        N:BYTE):FLOAT;
{ Norma wektora X zdefiniowana   max ( |X[i]| ) }
{                                 i             }

    PROCEDURE MACJAK(    NIELIN :PROCX;
                         X      :WEK;
                         N      :BYTE;
                     VAR A      :MAC;
                         eps    :FLOAT);
{ Macierz Jacobiego ukladu funkcji F (X) , i = 1,2,...,N  }
{ w punkcie X                       i                     }
{ eps - wzgledny krok rozniczkowania.                     }

    PROCEDURE METNEW(    NIELIN    :PROCX;
                     VAR X         :WEK;
                         N         :BYTE;
                         eps1,eps2 :FLOAT;
                         maxit     :WORD;
                     VAR blad      :BYTE);
{ Metoda Newtona rozwiazywania ukladu rownan nieliniowych F (X) = 0      }
{ i = 1,2,...,N                                            i             }
{ NIELIN - procedura FLOATizujaca nieliniowa funkcje wektorowa F(X)       }
{          zmiennej wektorowej X;                                        }
{ X   - wektor rozwiazania wymagajacy zadania poczatkowego przyblizenia  }
{ N    - ilosc rownan nieliniowych;                                      }
{ eps1 - zadana dokladnosc iteracji;                                     }
{ eps2 - wzgledny krok rozniczkowania;                                   }
{ maxit- ustalona maksymalna ilosc iteracji konczaca oblicznia;          }
{ BLAD - nr bledu; 0 - brak bledu.                                       }

    PROCEDURE METGRAD(    NIELIN    :PROCX;
                      VAR X         :WEK;
                          N         :BYTE;
                          eps1,eps2 :FLOAT;
                          maxit     :WORD;
                      VAR BLAD      :BYTE);
{ Metoda gradientowa rozwiazywania ukladu rownan nieliniowych F (X) = 0  }
{ i = 1,2,...,N                                            i             }
{ NIELIN - procedura FLOATizujaca nieliniowa funkcje wektorowa F(X)       }
{          zmiennej wektorowej X;                                        }
{ X   - wektor rozwiazania wymagajacy zadania poczatkowego przyblizenia  }
{ N   - ilosc rownan nieliniowych;                                       }
{ eps1 - zadana dokladnosc iteracji;                                     }
{ eps2 - wzgledny krok rozniczkowania;                                   }
{ maxit- ustalona maksymalna ilosc iteracji konczaca oblicznia;          }
{ BLAD - nr bledu; 0 - brak bledu.                                       }

    PROCEDURE ZMETNEW(    NIELIN    :PROCX;
                      VAR X         :WEK;
                          N         :BYTE;
                          eps1,eps2 :FLOAT;
                          maxit     :WORD;
                      VAR BLAD      :BYTE);
{ Zmodyfikowana metoda Newtona rozwiazywania ukladu rownan nieliniowych  }
{ F (X) = 0 ;       i = 1,2,...,N ;                                      }
{  i                                                                     }
{ NIELIN - procedura FLOATizujaca nieliniowa funkcje wektorowa F(X)       }
{          zmiennej wektorowej X;                                        }
{ X   - wektor rozwiazania wymagajacy zadania poczatkowego przyblizenia  }
{ N   - ilosc rownan nieliniowych;                                       }
{ eps1 - zadana dokladnosc iteracji;                                     }
{ eps2 - wzgledny krok rozniczkowania;    h=eps2*NORMAWEK(X,N);          }
{ maxit- ustalona maksymalna ilosc iteracji konczaca oblicznia;          }
{ BLAD - nr bledu; 0 - brak bledu.                                       }

    PROCEDURE METITE(    NIELIN    :PROCX;
                     VAR X         :WEK;
                         N         :BYTE;
                         eps1,eps2 :FLOAT;
                         maxit     :WORD;
                     VAR BLAD      :BYTE);
{ Metoda iteracyjna rozwiazywania ukladu rownan nieliniowych F (X) = 0   }
{ i = 1,2,...,N                                            i             }
{ NIELIN - procedura FLOATizujaca nieliniowa funkcje wektorowa F(X)       }
{          zmiennej wektorowej X;                                        }
{ X   - wektor rozwiazania wymagajacy zadania poczatkowego przyblizenia  }
{ N   - ilosc rownan nieliniowych;                                       }
{ eps1 - zadana dokladnosc iteracji;                                     }
{ eps2 - wzgledny krok rozniczkowania;                                   }
{ maxit- ustalona maksymalna ilosc iteracji konczaca oblicznia;          }
{ BLAD - nr bledu; 0 - brak bledu.                                       }

    PROCEDURE PSEROZNLNEW(    NIELIN    :PROCX;
                          VAR X         :WEK;
                              M,N       :BYTE;
                              eps1,eps2 :FLOAT;
                          VAR minU      :FLOAT;
                              maxit     :WORD;
                          VAR BLAD      :BYTE);
{ Zmodyfikowana metoda  Newtona  pseudorozwiazania nadokreslonego        }
{ ukladu rownan nieliniowych  F (X) = 0 ;       i = 1,2,...,M ;          }
{                              i                                         }
{ NIELIN - procedura FLOATizujaca nieliniowa funkcje wektorowa F(X)       }
{          zmiennej wektorowej X;                                        }
{ X    - wektor rozwiazania wymagajacy zadania poczatkowego przyblizenia }
{ M    - ilosc rownan nieliniowych;                                      }
{ N    - ilosc zmiennych niezaleznych;      M>=N                         }
{ eps1 - zadana dokladnosc iteracji;                                     }
{ eps2 - wzgledny krok rozniczkowania;    h=eps2*NORMAWEK(X,N)           }
{ minU - dokladnosc bezwzgledna rozwiazania                              }
{ maxit- ustalona maksymalna ilosc iteracji konczaca oblicznia           }
{ BLAD - nr bledu; 0 - brak bledu.                                       }

    PROCEDURE PSEROZNLHJ(    NIELIN  :PROCX;
                         VAR X       :WEK;
                             M,N     :BYTE;
                             tau,eps :FLOAT;
                         VAR minU    :FLOAT;
                             maxit   :WORD;
                         VAR BLAD    :BYTE);
{ Zmodyfikowana metoda  Hooke'a - Jeevsa  pseudorozwiazania nadokreslonego }
{ ukladu rownan nieliniowych  F (X) = 0 ;       i = 1,2,...,M ;            }
{                              i                                           }
{ NIELIN - procedura FLOATizujaca nieliniowa funkcje wektorowa F(X)         }
{          zmiennej wektorowej X;                                          }
{ X    - wektor rozwiazania wymagajacy zadania poczatkowego przyblizenia   }
{ M    - ilosc rownan nieliniowych;                                        }
{ N    - ilosc zmiennych niezaleznych;      M>=N                           }
{ tau  - wstepny krok iteracji;                                            }
{ eps  - dokladnosc iteracji                                               }
{ minU - dokladnosc bezwzgledna rozwiazania                                }
{ maxit- ustalona maksymalna ilosc iteracji konczaca oblicznia             }
{ BLAD - nr bledu; 0 - brak bledu.                                         }

    PROCEDURE DIV1(VAR N    :BYTE;
                   VAR A    :WEK;
                       X0   :FLOAT;
                   VAR BLAD :BYTE);
{         Procedura DIV1 wykonuje dzielenie wielomianu                   }
{   A[0]*X^N+A[1]*X^(N-1)+...+A[N]  przez czynnik liniowy X-X0           }
{  dla rzeczywistego X0 wg. sklejanego algorytmu Hornera (3.49)(3.50)(3.51) }
{  i pozostawia obliczone wspolczynniki ilorazu A[0],A[1],...,A[N-1]     }
{  rowniez w tablicy A, natomiast obnizony o jeden stopien wielomianu    }
{  pozostawia takze pod zmienna N                                        }
{LAD - nr bledu; 0 - brak bledu.                                         }

   PROCEDURE DIV2(VAR N   :BYTE;
                  VAR A   :WEK;
                      P,Q :FLOAT);
{       Procedura DIV2 wykonuje dzielenie wielomianu                    }
{ A[0]*X^N+A[1]*X^(N-1)+...+A[N]  przez trojmian kwadratowy X^2-P*X+Q   }
{ i pozostawia obliczone wspolczynniki ilorazu A[0],A[1],...,A[N-2]     }
{ rowniez w tablicy A, natomiast obnizony o dwa stopien wielomianu      }
{ pozostawia takze pod zmienna N                                        }

   PROCEDURE BAIRSTOW(    N:BYTE;
                      VAR A:WEK;
                      VAR P,Q:FLOAT;
                          EPS,MEGA:FLOAT;
                      VAR BLAD:BYTE);
{ Poprawia czynnik kwadratowy X^2-P*x+Q bedacy dzielnikiem wielomianu   }
{ a[0]*X^N+a[1]*X^(N-1)+...+a[N] pozostawiajac poprawione wartosci      }
{ jako zmienne P,Q                                                      }
{ N   - stopien wielomianu                                              }
{ A   - wektor wspolczynnikow                                           }
{ EPS - dokladnosc bezwzgledna iteracji  np. EPS=1E-16                  }
{ MEGA - prog poszukiwamia dzielnika np.MEGA:=1E+3                      }
{ BLAD - nr bledu; 0 - brak bledu.                                      }

   PROCEDURE ZWBAIRSTOW(    N        :BYTE;
                        VAR A        :WEK;
                            P0,Q0    :FLOAT;
                            EPS,MEGA :FLOAT;
                        VAR BLAD     :BYTE;
                        VAR Z        :WEKZ);
{ Procedura ZWBAIRSTOW oblicza pierwiastki wielomianu (3.57) stopnia N  }
{ o wspolczynnikach zapisanych w wektorze A wg.metody BAIRSTOWA,pkt.4.11}
{ P0,Q0 - poczatkowe przyblizenie dla  X^2-P*X+Q  ( przy odwolaniu      }
{         mozna przyjac P0=Q0=1)                                        }
{ Z     - zespolony wektor pierwiastkow o czesci rzeczywistej Z[k].RE   }
{         i urojonej Z[k].IM dla k - tego pierwiastka k=1..N            }
{ EPS   - dokladnosc bezwzgledna iteracji  np. EPS=1E-16                }
{ MEGA  - prog poszukiwamia dzielnika np.MEGA:=1E+3                     }
{ BLAD   - nr bledu; 0 - brak bledu.                                     }

   PROCEDURE LAGUERRE(    N     :BYTE;
                      VAR A     :WEK;
                      VAR Z     :ZESPOL;
                          EPS   :FLOAT;
                          maxit :WORD;
                      VAR BLAD  :BYTE);
{ Wyznacza pierwiastek Z wielomianu stopna N o wspolczynnikach zapisanych   }
{ w wektorze A                                                              }
{ EPS  - dokladnosc bezwzgledna iteracji  np. EPS=1E-8                      }
{ maxit- ustalona maksymalna ilosc iteracji konczaca oblicznia              }
{ BLAD - nr bledu; 0 - brak bledu.                                          }

    PROCEDURE ZWLAGUERRE(    N    :BYTE;
                         VAR A    :WEK;
                             EPS  :FLOAT;
                             maxit:WORD;
                         VAR BLAD :BYTE;
                         VAR Z    :WEKZ);
{  Procedura ZWLAGUERRE oblicza pierwiastki wielomianu (3.57) stopnia N  }
{  o wspolczynnikach zapisanych w wektorze A wg.metody LAGUERRE,pkt.4.13 }
{  Z - zespolony wektor pierwiastkow o czesci rzeczywistej Z[k].RE       }
{      i urojonej Z[k].IM dla k - tego pierwiastka k=1..N                }
{  EPS  - dokladnosc bezwzgledna iteracji  np. EPS=1E-8                  }
{  maxit- ustalona maksymalna ilosc iteracji konczaca oblicznia          }
{  BLAD - nr bledu; 0 - brak bledu.                                      }

   PROCEDURE WSPWCHAR(    N    :BYTE;
                      VAR A    :MAC;
                      VAR Y0,P :WEK);
{  Obliczanie wspolczynnikow wielomianu charakterystycznego macierzy D   }
{  A  - dana macierz kwadratowa rzedu N                                  }
{  Y0 - dowolny niezerowy wektor o wymiarze 1..N                         }
{  P  - wektor ktorego skladowymi sa wspolczynniki wielomianu charakter. }
{       la^N+P[1]*la^(N-1)+P[2]*la^(N-2)+...+P[N-1]*la+P[N] = 0          }

   PROCEDURE WARWLMAC(    N        :BYTE;
                      VAR A        :MAC;
                          EPS,MEGA :FLOAT;
                      VAR BLAD     :BYTE;
                      VAR Z        :WEKZ);
{  Oblicza wartosci wlasne macierzy D rzedu N metoda Bairstowa           }
{  Z    - zespolony wektor pierwiastkow o czesci rzeczywistej Z[K].RE    }
{         i urojonej Z[K].IM dla K - tego pierwiastka K=1..N             }
{  Z[K] - K - ta wartosc wlasna wielomianu charakterystycznego           }
{         la^N +P(N-1)*la^(N-1) +P(N-2)*la^(N-2) + ... +P(1)*la +P(0) =0 }
{  EPS  - dokladnosc bezwzgledna iteracji  np. EPS=1E-16                 }
{  MEGA - prog poszukiwamia dzielnika np.MEGA:=1E+3                      }
{  BLAD - nr bledu; 0 - brak bledu.                                      }

   PROCEDURE WARWLMAL(    N    :BYTE;
                      VAR A    :MAC;
                          EPS  :FLOAT;
                          maxit:WORD;
                      VAR BLAD :BYTE;
                      VAR Z    :WEKZ);
{  Oblicza wartosci wlasne macierzy D rzedu N metoda Laguerre'a          }
{  Z    - zespolony wektor pierwiastkow o czesci rzeczywistej Z[K].RE    }
{         i urojonej Z[K].IM dla K - tego pierwiastka K=1..N             }
{  Z[K] - K - ta wartosc wlasna wielomianu charakterystycznego           }
{         la^N +P(N-1)*la^(N-1) +P(N-2)*la^(N-2) + ... +P(1)*la +P(0) =0 }
{  maxit- ustalona maksymalna ilosc iteracji konczaca oblicznia          }
{  EPS  - dokladnosc bezwzgledna iteracji  np. EPS=1E-8                  }
{  BLAD - nr bledu; 0 - brak bledu.                                      }

   PROCEDURE ZERAFUN(    F                    :FUNX;
                         XDOLNE,DX,XGORNE,EPS :FLOAT;
                     VAR A                    :WEK;
                     VAR IL                   :BYTE);
{ Wyznacza zera funkcji F(x) w przedziale < XDOLNE , XGORNE > z krokiem DX  }
{ metoda polowienia przedzialu ograniczonego dlugoscia EPS  np. EPS=1E-6    }
{ A  - wektor kolejnych pierwiastkow funkcji F(x)                           }
{ IL - pierwiastkow rzeczywistych funfcji F(x) w przedziale < XDOLNE,XGORNE }


 IMPLEMENTATION

FUNCTION NORMAWEK;
  VAR i   :BYTE;
      r,s :FLOAT;
  BEGIN
    s:=0;
    FOR i:=1 TO N DO
    BEGIN
      r:=ABS(X[i]);
      IF r > s THEN
        s:=r
    END;
    NORMAWEK:=s
  END { NORMAWEK };

FUNCTION MINNORMAWEK(X:WEK; N:BYTE):FLOAT;
  VAR i   :BYTE;
      r,s :FLOAT;
  BEGIN
    s:=ABS(X[1]);
    FOR i:=2 TO N DO
    BEGIN
      r:=ABS(X[i]);
      IF r < s THEN
        s:=r
    END;
    MINNORMAWEK:=s
  END { MINNORMAWEK };

PROCEDURE MACJAK;
  VAR i,j     :BYTE;
      h,r     :FLOAT;
      X1,Y,Y1 :WEK;
  BEGIN
    NIELIN(Y,X,N);
    FOR i:=1 TO N DO
    BEGIN
      r:=ABS(X[i]);
      IF r>eps
        THEN h:=eps*r
        ELSE h:=eps;
      X1:=X; X1[i]:=X[i]+h;
      NIELIN(Y1,X1,N);
      FOR j:=1 TO N DO
        A[j,i]:=(Y1[j]-Y[j])/h
    END {i}
  END {MACJAK};

PROCEDURE METNEW;
  VAR i,j  :BYTE;
      k    :WORD;
      s,ni :FLOAT;
      Y    :WEK;
      A    :MAC;
  BEGIN
    k:=0;
    REPEAT { Konstrukcja procesu iteracyjnego (3.10) }
      ni:=0;
      INC(k);
      { Konstrukcja macierzy Jacobiego A wzor (3.8)  }
      MACJAK(NIELIN,X,N,A,eps2) ;
      { Macierz odwrotna B macierzy Jacobiego A  }
      ODWMAC1(A,N,1E-16,BLAD);
      IF BLAD=0
        THEN BEGIN
               NIELIN(Y,X,N);
               FOR i:=1 TO N DO
               BEGIN
                 ni:=ni+ABS(Y[i]);
                 s:=0;
                 FOR j:=1 TO N DO
                   s:=s+A[i,j]*Y[j];  { wzor (3.9) }
                 X[i]:=X[i]-s
               END
             END
        ELSE BLAD:=80
    UNTIL (ni<eps1) OR (k>maxit) OR (BLAD<>0);
    IF (BLAD=0) AND (k>maxit) THEN
      BLAD:=81
  END { METNEW };

PROCEDURE METGRAD;
  VAR i,j        :BYTE;
      k          :WORD;
      s,s1,mi,ni :FLOAT;
      Y,dX,G     :WEK;
      A          :MAC;
  BEGIN
    k:=0; BLAD:=0;
    REPEAT
      ni:=0;
      INC(k);
      MACJAK(NIELIN,X,N,A,eps2) ;
      NIELIN(Y,X,N);
      FOR i:=1 TO N DO
      BEGIN
        s:=0;
        FOR j:=1 TO N DO
          s:=s+A[j,i]*Y[j];
        dX[i]:=s
      END;
      FOR i:=1 TO N DO
      BEGIN
        s:=0;
        FOR j:=1 TO N DO
          s:=s+A[i,j]*dX[j];
        G[i]:=s
      END;
      s:=0; s1:=0;
      FOR i:=1 TO N DO
      BEGIN
        s:=s+Y[i]*G[i];
        s1:=s1+SQR(G[i])
      END;
      mi:=s/s1;
      FOR i:=1 TO N DO
      BEGIN
        ni:=ni+ABS(Y[i]);
        s:=0;
        X[i]:=X[i]-mi*dX[i]
      END
    UNTIL (ni<eps1) OR (k>maxit);
    IF k>maxit THEN
      BLAD:=82
  END { METGRAD };

PROCEDURE ZMETNEW;
  VAR i,j  :BYTE;
      k    :WORD;
      s,ni :FLOAT;
      G    :WEK;
      A    :MAC;

  FUNCTION U(NIELIN:PROCX; X:WEK; N:BYTE):FLOAT;
    {  Iloczyn skalarny funkcji wektorowych  }
    {                    N                   }
    {                  ____                  }
    {                  \              2      }
    {          U   =    >  ( F[i](X) )       }
    {                  /                     }
    {                  -----                 }
    {                   i=1                  }
    VAR i:BYTE;
        s:FLOAT;
        Y:WEK;
    BEGIN
      NIELIN(Y,X,N);
      s:=0;
      FOR i:=1 TO N DO
        s:=s+SQR(Y[i]);
      U:=s
    END { U };

  PROCEDURE GRADU(NIELIN:PROCX; X:WEK; VAR GU:WEK; N:BYTE; eps:FLOAT);
    { Wyznaczanie gradientu funfcji U(X)  tj. GU = grad U(X)  w punkcie X       }
    VAR i      :BYTE;
        UX,h,g :FLOAT;
        X1     :WEK;
    BEGIN
      UX:=U(NIELIN,X,N);
      FOR i:=1 TO N DO
      BEGIN
        g:=ABS(X[i]);
        IF g>eps
          THEN h:=eps*g
          ELSE h:=eps;
        X1:=X; X1[i]:=X[i]+h;
        GU[i]:=(U(NIELIN,X1,N)-UX)/h
      END
    END {GRADU};

  PROCEDURE HESJAN(NIELIN:PROCX; X:WEK; N:BYTE; VAR A:MAC;eps:FLOAT);
    {---------------------------------------------------------------------------}
    { Generacja macierzy -hesjanu- funkcji  U(X) = <F(X),F(X)>                  }
    {                                                                           }
    {                          2                                                }
    {                         d U(X)                                            }
    {              A[i,j] = ----------    ;   i,j = 1,2,...,N                   }
    {                       dX[j]dX[i]                                          }
    VAR i,j            :BYTE;
        g,h,
        U0,U1,U2,U3,U4 :FLOAT;
        X1             :WEK;

    BEGIN
      U0:=U(NIELIN,X,N);
      FOR i:=1 TO N DO
      BEGIN
        g:=ABS(X[i]);
        IF g>eps
          THEN h:=eps*g
          ELSE h:=eps;
        X1:=X; X1[i]:=X[i]+h;
        U1:=U(NIELIN,X1,N);
        X1:=X; X1[i]:=X[i]-h;
        U2:=U(NIELIN,X1,N);
        A[i,i]:=( U1-2*U0+U2)/SQR(h);
        FOR j:=1 TO N DO
          IF i<>j THEN
          BEGIN
            X1:=X; X1[i]:=X[i]+h; X1[j]:=X[j]+h;
            U1:=U(NIELIN,X1,N);
            X1:=X; X1[i]:=X[i]-h; X1[j]:=X[j]-h;
            U2:=U(NIELIN,X1,N);
            X1:=X; X1[i]:=X[i]+h; X1[j]:=X[j]-h;
            U3:=U(NIELIN,X1,N);
            X1:=X; X1[i]:=X[i]-h; X1[j]:=X[j]+h;
            U4:=U(NIELIN,X1,N);
            A[i,j]:=(U1+U2-U3-U4)/(4*SQR(h))
          END
      END
    END { HESJAN };

  BEGIN
    { Czesc operacyjna procedury ZMMETNEW }
    k:=0;
    REPEAT
      ni:=0; k:=k+1;
      HESJAN(NIELIN,X,N,A,eps2);
      ODWMAC1(A,N,1E-16,BLAD);
      IF BLAD=0
        THEN BEGIN
               GRADU(NIELIN,X,G,N,eps2);
               FOR i:=1 TO N DO
               BEGIN
                 ni:=ni+ABS(G[i]);
                 s:=0;
                 FOR j:=1 TO N DO
                   s:=s+A[i,j]*G[j];
                 X[i]:=X[i]-s
               END
             END
        ELSE BLAD:=83
     UNTIL (ni<eps1) OR (k>maxit) OR (BLAD<>0);
     IF (BLAD=0) AND (k>maxit) THEN
       BLAD:=84
  END { ZMMETNEW };


PROCEDURE METITE;
  VAR i,j  :BYTE;
      k    :WORD;
      s,ni :FLOAT;
      Y    :WEK;
      A    :MAC;
  BEGIN
    { Generacja macierzy Jacobiego A w punkcie poczatkowym}
    MACJAK(NIELIN,X,N,A,eps2) ; { wzor (3.8) }
    { Macierz odwrotna B macierzy Jacobiego A w punkcie poczatkowym}
    ODWMAC1(A,N,1E-16,BLAD); { wzor (3.39) }
    IF BLAD<>0 THEN
    BEGIN
      BLAD:=85;
      EXIT
    END;
    k:=0;
    REPEAT  { Iteracja wg wzoru (3.40) }
      ni:=0;
      INC(k);
      NIELIN(Y,X,N);
      FOR i:=1 TO N DO
      BEGIN
        ni:=ni+ABS(Y[i]);
        s:=0;
        FOR j:=1 TO N DO
          s:=s+A[i,j]*Y[j];
        X[i]:=X[i]-s
      END
    UNTIL (ni<eps1) OR (k>maxit);
    IF k>maxit THEN
      BLAD:=86
  END { METITE };

PROCEDURE PSEROZNLNEW;
  VAR i,j  :BYTE;
      k    :WORD;
      s,ni :FLOAT;
      G    :WEK;
      A    :MAC;

       FUNCTION U(NIELIN:PROCX; X:WEK):FLOAT;
       { Iloczyn skalarny funkcji wektorowych }
       {                 M                    }
       {                ____                  }
       {                \              2      }
       {        U   =    >  ( F[i](X) )       }
       {                /                     }
       {                -----                 }
       {                 i=1                  }
        VAR i:BYTE;
            s:FLOAT;
            Y:WEK;
        BEGIN
          NIELIN(Y,X,N);
          s:=0;
          FOR i:=1 TO M DO
            s:=s+SQR(Y[i]);
          U:=s
        END { U };

       PROCEDURE GRADU(NIELIN:PROCX; X:WEK; VAR GU:WEK; N:BYTE; eps:FLOAT);
       { Wyznaczanie gradientu funfcji U(X)  tj. GU = grad U(X)  w punkcie X }
        VAR i      :BYTE;
            UX,h,g :FLOAT;
            X1     :WEK;
        BEGIN
          UX:=U(NIELIN,X);
          FOR i:=1 TO N DO
          BEGIN
            g:=ABS(X[i]);
            IF g>eps
              THEN h:=eps*g
              ELSE h:=eps;
            X1:=X; X1[i]:=X[i]+h;
            GU[i]:=(U(NIELIN,X1)-UX)/h
          END
        END { GRADU };

         PROCEDURE HESJAN(NIELIN:PROCX; X:WEK; N:BYTE; VAR A:MAC;eps:FLOAT);
       {---------------------------------------------------------------------------}
       { Generacja macierzy -hesjanu- funkcji  U(X) = <F(X),F(X)>                  }
       {                                                                           }
       {                          2                                                }
       {                         d U(X)                                            }
       {              A[i,j] = ----------    ;   i,j = 1,2,...,N                   }
       {                       dX[j]dX[i]                                          }

          VAR
             i,j:BYTE;
             g,h,U0,U1,U2,U3,U4:FLOAT;
             X1:WEK;

          BEGIN
            U0:=U(NIELIN,X);
            FOR i:=1 TO N DO
              BEGIN
                g:=ABS(X[i]);
                IF g>eps THEN h:=eps*g ELSE h:=eps;
                X1:=X; X1[i]:=X[i]+h;
                U1:=U(NIELIN,X1);
                X1:=X; X1[i]:=X[i]-h;
                U2:=U(NIELIN,X1);
                A[i,i]:=( U1-2*U0+U2)/SQR(h);
                FOR j:=1 TO N DO
                    IF i<>j THEN
                      BEGIN
                        X1:=X; X1[i]:=X[i]+h; X1[j]:=X[j]+h;
                        U1:=U(NIELIN,X1);
                        X1:=X; X1[i]:=X[i]-h; X1[j]:=X[j]-h;
                        U2:=U(NIELIN,X1);
                        X1:=X; X1[i]:=X[i]+h; X1[j]:=X[j]-h;
                        U3:=U(NIELIN,X1);
                        X1:=X; X1[i]:=X[i]-h; X1[j]:=X[j]+h;
                        U4:=U(NIELIN,X1);
                        A[i,j]:=(U1+U2-U3-U4)/(4*SQR(h))
                      END
              END {i}
          END { HESJAN };

  BEGIN
    { Czesc operacyjna procedury PSEROZNLNEW }
    BLAD:=0;
    k:=0;
    REPEAT
      ni:=0;
      INC(k);
      HESJAN(NIELIN,X,N,A,eps2);
      ODWMAC1(A,N,1E-16,BLAD);
      IF BLAD=0
        THEN BEGIN
               GRADU(NIELIN,X,G,N,eps2);
               FOR i:=1 TO N DO
               BEGIN
                 ni:=ni+ABS(G[i]);
                 s:=0;
                   FOR j:=1 TO N DO
                     s:=s+A[i,j]*G[j];
                 X[i]:=X[i]-s
               END
             END
        ELSE BLAD:=91
     UNTIL (ni<eps1) OR (k>maxit) OR (BLAD<>0);
     IF BLAD=0 THEN
     BEGIN
       minU:=U(NIELIN,X);
       IF k>maxit THEN
         BLAD:=92
     END
  END { PSEROZNLNEW };

PROCEDURE PSEROZNLHJ;

  FUNCTION V(NIELIN:PROCX; X:WEK):FLOAT;
    {---------------------------------}
    {  Norma wektora funkcyjnego      }
    {   V   =  max | F[i](X) |        }
    {           i                     }
    {---------------------------------}
   VAR i :BYTE;
       s :FLOAT;
       Y :WEK;
   BEGIN
     NIELIN(Y,X,N);
     s:=0;
     FOR i:=1 TO M DO
       IF s<ABS(Y[i]) THEN
         s:=ABS(Y[i]);
     V:=s
   END { V };

  PROCEDURE ZMINFUNHJ(    NIELIN  :PROCX;
                      VAR X       :WEK;
                          N       :BYTE;
                          tau,eps :FLOAT;
                      VAR minU    :FLOAT;
                          maxit   :WORD;
                      VAR BLAD    :BYTE);
    VAR F2,F1,F0,ni,tau1 :FLOAT;
        k,l,m            :WORD;
        X1,X0            :WEK;
        CZY              :BOOLEAN;

    PROCEDURE KROK(j:INTEGER);
      VAR i:INTEGER;
      BEGIN
        X1:=X;
        X1[k]:=X[k]+j*ni;
        F1:=V(NIELIN,X1)
      END { KROK };

  BEGIN {Czesc operacyjna procedury ZMINFUNHJ}
    F2:=V(NIELIN,X);
    X0:=X; F0:=F2;
    CZY:=FALSE;
    BLAD:=0;
    m:=0;  ni:=tau;
    REPEAT
      INC(m);
      tau1:=ni;
      FOR k:=1 TO N DO
      BEGIN
        KROK(1);
        IF F1<F2
          THEN BEGIN
                 X[k]:=X1[k];
                 F2:=F1
               END
          ELSE BEGIN
                 KROK(-1);
                 IF F1<F2 THEN
                 BEGIN
                   X[k]:=X1[k];
                   F2:=F1
                 END
               END
        END;
      IF F2<F0
        THEN BEGIN
               FOR k:=1 TO N DO
               BEGIN
                 X1[k]:=X0[k]; X0[k]:=X[k];
                 X[k]:=X0[k]+1.4*(X[k]-X1[k])
               END;
               F0:=F2;
               F2:=V(NIELIN,X);
               ni:=1.2*ni; CZY:=TRUE
             END
        ELSE IF CZY
               THEN BEGIN
                      X:=X0;
                      F2:=F0;
                      CZY:=FALSE
                    END
               ELSE IF tau1>eps THEN
                      ni:=0.2*ni
    UNTIL (tau1<=eps) OR (m>maxit);
    minU:=V(NIELIN,X);
    IF m>maxit THEN
      BLAD:=93
  END { ZMINFUNHJ };

  BEGIN
    { Czesc operacyjna procedury PSEROZNLHJ }
   ZMINFUNHJ(NIELIN,X,N,tau,eps,minU,maxit,BLAD);
  END { PSEROZNLHJ };




PROCEDURE DIV1;
  VAR i,k      :BYTE;
      X,DX,M,L :FLOAT;
      QP,QB    :WEK;
  BEGIN
    BLAD:=0;
    IF X0=0 THEN
    BEGIN
      BLAD:=87;
      EXIT
    END;
    QP[-1]:=0;
    FOR i:=0 TO N DO
      QP[i]:=A[i]+X0*QP[i-1]; { wzor (3.49) }
    QB[N]:=0;
    FOR i:=N DOWNTO 1 DO
      QB[i-1]:=(QB[i]-A[i])/X0;  { wzor (3.50) }
    X:=1E+30;
    FOR i:=1 TO N-1 DO
    BEGIN
      L:=ABS(QP[i]-QB[i]);
      M:=ABS(A[i])+ABS(QP[i]);
      IF M>0 THEN
      BEGIN
        DX:=L/M;  { wzor (3.52) }
        IF DX<X THEN
        BEGIN
          X:=DX; k:=i
        END
      END
    END;
    FOR i:=0 TO k DO
      A[i]:=QP[i];
    FOR i:=k+1 TO N DO
      A[i]:=QB[i];
    DEC(N)
  END {DIV1};

PROCEDURE DIV2;
 VAR K:BYTE;
 BEGIN
   A[-1]:=0;
   FOR K:=1 TO N-2 DO
     A[K]:=A[K]+P*A[K-1]-Q*A[K-2];
   DEC(N,2)
 END { DIV2 };

PROCEDURE BAIRSTOW;
  VAR R,R1,R2,R3,R4,P1,Q1,S :FLOAT;
      j                     :BYTE;
      B,C                   :WEK;
  BEGIN
    BLAD:=0;
    IF A[0]=0 THEN
    BEGIN
      BLAD:=88;
      EXIT
    END;
    IF (P=0) AND (Q=0) THEN
      P:=0.1;
    { Formula rekurencyjna (3.61) }
    REPEAT
      b[-1]:=0; b[0]:=a[0];
      FOR j:=1 TO N DO
        b[j]:=a[j]+P*b[j-1]-Q*b[j-2];
      { Formula rekurencyjna (3.67) }
      c[-1]:=0; c[0]:=b[0];
      FOR j:=1 TO N-2 DO
        c[j]:=b[j]+P*c[j-1]-Q*c[j-2];
      R1:=c[N-2]; R2:=c[N-3]; R3:=b[N-1]; R4:=b[N];
      P1:=P; Q1:=Q;  { Pamietanie stanu poprzedniego iteracji P1,Q1 }
      R:=P*R1-Q*R2;           { R=C[N-1] wzor (3.67) }
      S:=R1*R1-R*R2;          { S=delta wzor (3.71a) }
      P:=P-(R3*R1-R4*R2)/S;   { P=P+dp  wzory (3.64) (3.71b) }
      Q:=Q-(R3*R-R1*R4)/S;    { Q=Q+dq  wzory (3.64) (3.71c) }
      R:=(ABS(P-P1)+ABS(Q-Q1))/(ABS(P1)+ABS(Q1)); { Norma iteracji }
      IF R>MEGA THEN
        BLAD:=89;
    UNTIL (R<EPS) OR (BLAD<>0)
 END { BAIRSTOW };

PROCEDURE ZWBAIRSTOW;
  VAR P,Q,X1,X,Y :FLOAT;
      ZZ         :ZESPOL;
      i,j,k      :BYTE;
      RE         :BOOLEAN;

 PROCEDURE RKW(P,Q:FLOAT; VAR X,Y:FLOAT; VAR RE:BOOLEAN);
   { Procedura RKW rozwiazuje rownanie kwadratowe X^2-P*X+Q = 0          }
   { pozostawiajac jako pierwiastki zmienne X,Y (rzeczywiste) oraz       }
   { zmienna boolowska RE , przy czym X,Y sa rzeczywiste jesli RE=TRUE   }
   { lub X jest czescia rzeczywista,a Y-urojona zespolonych pierwiastkow,}
   { jesli RE=FALSE                                                      }
   VAR D:FLOAT;
   BEGIN
     X:=P/2; D:=X*X-Q; RE:=D>=0;
     D:=SQRT(ABS(D));
     IF RE
       THEN BEGIN
              Y:=X+D;
              X:=X-D
            END
       ELSE Y:=D
   END { RKW };

 PROCEDURE PRW2;
   { Blok proceduralny w ktorym dokonuje sie podstawienia kolejnych    }
   { pierwiastkow wielomianu pod wektor zespolony Z                    }
   { Z[k] -k-ty pierwiastek wielomianu                                 }
   BEGIN
     RKW(P,Q,X,Y,RE);
     IF RE
       THEN BEGIN
              INC(k);
              Z[k].RE:=X; Z[k].IM:=0;
              INC(k);
              Z[k].RE:=Y; Z[k].IM:=0
            END
       ELSE BEGIN
              INC(k);
              Z[k].RE:=X; Z[k].IM:=Y;
              INC(k);
              Z[k].RE:=X; Z[k].IM:=-Y
            END
   END { PRW2 };

  BEGIN { Czesc operacyjna procedury ZWBAIRSTOW }
    BLAD:=0;
    IF A[0]=0 THEN
    BEGIN
      BLAD:=88;
      EXIT
    END;
    k:=0; P:=P0; Q:=Q0;
    WHILE N>2 DO
    BEGIN
      BAIRSTOW(N,A,P,Q,EPS,MEGA,BLAD);
      IF BLAD<>0 THEN
        EXIT;
      PRW2;
      DIV2(N,A,P,Q)
    END;
    IF N=2
      THEN BEGIN
             P:=-A[1]/A[0]; Q:=A[2]/A[0];
             PRW2
           END
      ELSE BEGIN {N=1}
             INC(k);
             Z[k].RE:=-A[1]/A[0];  Z[k].IM:=0
           END;
    { Przestawianie pierwiastkow wg. wzrastajacych modulow }
    FOR i:=1 TO N DO
    BEGIN
      X:=MODUL(Z[i]); k:=i;
      FOR j:=i+1 TO N DO
      BEGIN
        X1:=MODUL(Z[j]);
        IF X1<X THEN
        BEGIN
          X:=X1; k:=j
        END
      END;
      ZZ:=Z[i]; Z[i]:=Z[k]; Z[k]:=ZZ
    END
  END { ZWBAIRSTOW };

PROCEDURE LAGUERRE;
  VAR B,C,D,S,H,M1,M2     :ZESPOL;
      MM1,MM2,PMH,FH,ALFA :FLOAT;
      N1,i,j              :BYTE;
      k                   :WORD;
  BEGIN
    BLAD:=0;
    IF A[0]=0 THEN
    BEGIN
      BLAD:=88;
      EXIT
    END;
    IF A[N]=0 THEN
    BEGIN
      Z:=Z0;
      EXIT
    END;
    N1:=N-1;
    k:=0;
    REPEAT
      { Wyznaczanie wartosci wielomianu i ich pochodnych pierwszego }
      { i drugiego rodzaju wg. algorytmu Hornera (3.76) }
      B:=Z0; C:=Z0; D:=Z0; B.RE:=A[0]; C.RE:=A[0]; D.RE:=A[0];
      FOR i:=1 TO N DO
        IF i<=N-2
          THEN BEGIN
                 MUL(S,B,Z); S.RE:=S.RE+A[i]; B:=S;
                 MUL(S,C,Z); ADD(C,S,B);  MUL(S,D,Z); ADD(D,S,C)
               END
          ELSE IF i<=N-1
                 THEN BEGIN
                        MUL(S,B,Z); S.RE:=S.RE+A[i]; B:=S;
                        MUL(S,C,Z); ADD(C,S,B)
                      END
                 ELSE BEGIN
                        MUL(S,B,Z); S.RE:=S.RE+A[i]; B:=S
                      END;
      MULRZ(D,D,2);
      { Konstukcja wzoru rekurencyjnego (3.74) }
      MUL(S,C,C); MULRZ(H,S,N1);  MUL(S,B,D); MULRZ(S,S,N);
      SUB(H,H,S); MULRZ(H,H,N1); { H - wg. wzoru (3.75) }
      PMH:=SQRT(MODUL(H)); FH:=ARG(H); FH:=FH/2;
      H.RE:=PMH*COS(FH); H.IM:=PMH*SIN(FH);
      ADD(M1,C,H); SUB(M2,C,H);
      DIW(M1,B,M1); DIW(M2,B,M2); MM1:=MODUL(M1); MM2:=MODUL(M2);
      IF MM1>MM2
        THEN MULRZ(H,M2,N)
        ELSE MULRZ(H,M1,N);
      SUB(Z,Z,H);
      INC(k);
      ALFA:=(ABS(H.RE)+ABS(H.IM))/(ABS(Z.RE)+ABS(Z.IM))
    UNTIL (ALFA<EPS) OR (k>maxit);
    IF k>maxit THEN
      BLAD:=90
  END { LAGUERRE };


PROCEDURE ZWLAGUERRE;
  VAR i,j,k,M           :BYTE;
      X1,X2,X,P,Q,DELTA :FLOAT;
      ZZ                :ZESPOL;
      TAK               :BOOLEAN;
  BEGIN
    BLAD:=0;
    IF A[0]=0 THEN
    BEGIN
      BLAD:=88;
      EXIT
    END;
    M:=N;  k:=0;
    WHILE M>2 DO
    BEGIN
      ZZ:=Z0;
      INC(k);
      IF A[M]=0
        THEN BEGIN
               Z[M]:=Z0;
               DEC(M)
             END
        ELSE BEGIN
               LAGUERRE(M,A,ZZ,EPS,maxit,BLAD);
               IF BLAD<>0 THEN
                 EXIT;
               IF ABS(ZZ.IM)<EPS*ABS(ZZ.RE) THEN
               BEGIN
                 Z[M].RE:=ZZ.RE; Z[M].IM:=0;
                 DIV1(M,A,ZZ.RE,BLAD);
                 IF BLAD<>0 THEN
                   EXIT;
               END;
               IF ABS(ZZ.IM)>=EPS*ABS(ZZ.RE) THEN
               BEGIN
                 Z[M].RE:=ZZ.RE; Z[M].IM:=ZZ.IM;
                 Z[M-1].RE:=ZZ.RE; Z[M-1].IM:=-ZZ.IM;
                 P:=2*ZZ.RE; Q:=KWMODUL(ZZ);
                 DIV2(M,A,P,Q);
               END
             END
    END;
    IF M=2
      THEN BEGIN
             IF A[M]=0
               THEN BEGIN
                      Z[M]:=Z0; M:=M-1
                    END
               ELSE BEGIN
                      DELTA:=SQR(A[1])-4*A[0]*A[2];
                      IF DELTA<0
                        THEN BEGIN
                               Z[M].RE  :=-0.5*A[1]/A[0];
                               Z[M-1].RE:=Z[M].RE;
                               Z[M].IM  :=0.5*SQRT(-DELTA)/A[0];
                               Z[M-1].IM:=-Z[M].IM
                             END
                        ELSE BEGIN
                               X1:=-0.5*A[1]/A[0];
                               X2:=0.5*SQRT(DELTA)/A[0];
                               Z[M].RE:=X1+X2; Z[M-1].RE:=X1-X2;
                               Z[M].IM:=0; Z[M-1].IM:=0
                             END
                       END
           END
      ELSE BEGIN
             Z[1].RE:=-A[1]/A[0];  Z[1].IM:=0
           END;
    { Przestawianie pierwiastkow wg. wzrastajacych moduluw }
    FOR i:=1 TO N DO
    BEGIN
      X:=MODUL(Z[i]); k:=i;
      FOR j:=i+1 TO N DO
      BEGIN
        X1:=MODUL(Z[j]);
        IF X1<X THEN
        BEGIN
          X:=X1; k:=j
        END
      END;
      ZZ:=Z[i]; Z[i]:=Z[k]; Z[k]:=ZZ
    END
  END { ZWLAGUERRE };

PROCEDURE WSPWCHAR;
  VAR i,j,k,l :BYTE;
      S,T     :FLOAT;
      Y       :MAC;
      B       :WEK;
      BLAD    :BYTE;
BEGIN
  FOR i:=1 TO N DO
    Y[i,N]:=Y0[i];
  FOR i:=1 TO N DO
  BEGIN
    FOR j:=1 TO N DO
    BEGIN
      T:=0;
      { Tworzenie ciagu wektorow wg wzoru (3.78) }
      FOR k:=1 TO N DO
        T:=T+A[j,k]*Y0[k];
      P[j]:=T
    END;
    l:=N-i;
    FOR j:=1 TO N DO
    BEGIN
      { Generacja macierzy Y ukladu rownan (3.80) }
      Y0[j]:=P[j] ; Y[j,l]:=Y0[j]
    END
  END;
  { Generacja wektora wyrazow wolnych ukladu (3.80) }
  FOR i:=1 TO N DO
    B[i]:=-Y[i,l];
  RRAL(Y,B,P,N,1E-16,BLAD); { Rozwiazanie ukladu (3.80) }
  P[0]:=1
END { WSPWCHAR };

PROCEDURE WARWLMAC;
  VAR C,Y0      :WEK;
      i         :BYTE;
      C1,D1,P,Q :FLOAT;

      PROCEDURE FROBENIUS(N:BYTE; VAR A:MAC; VAR C,D:FLOAT);
        { Wyznacza przedzial <C,D> w ktorym znajduja sie wszystkie wartosci }
        { wlasne macierzy A stopnia N                                       }
        VAR I,J           :BYTE;
            S,T,U,V,Z,X,Y :FLOAT;
        BEGIN
          C:=1E+38; U:=1E+38; D:=-1E+38; V:=-1E+38;
          FOR I:=1 TO N DO
          BEGIN
            Z:=0; S:=0;
            FOR J:=I-1 DOWNTO 1 DO
            BEGIN
              Z:=Z+ABS(A[I,J]); S:=S+ABS(A[J,I])
            END;
            FOR J:=I+1 TO N DO
            BEGIN
              Z:=Z+ABS(A[I,J]); S:=S+ABS(A[J,I])
            END;
            T:=ABS(A[I,I]); X:=Z+T; Y:=S+T;
            IF D<X THEN
              D:=X;
            IF V<Y THEN
              V:=Y;
            X:=T-Z; Y:=T-S;
            IF C>X THEN
              C:=X;
            IF U>Y THEN
              U:=Y
          END;
          IF D>V THEN
            D:=V;
          IF C<U THEN
            C:=U
        END { FROBENIUS };

      PROCEDURE PQ(C,D:FLOAT; VAR P,Q:FLOAT);
        { Wyznacza poczatkowe przyblizenie wartosci P,Q dla trojmianu  }
        { kwadratowego X^2 - P*X + Q  w oparciu o przedzial <C,D>      }
        { wyznaczony przez procedure FROBENIUS                                }
        VAR C2,D2,MAX :FLOAT;
            K1        :INTEGER;
        BEGIN
          P:=C+D; C2:=C*C; D2:=D*D;
          IF (C>=0) OR (D<0)
            THEN Q:=(C2+D2)/2
            ELSE BEGIN
                   IF D2>=C2
                     THEN BEGIN
                            MAX:=D2; K1:=1
                          END
                     ELSE BEGIN
                            MAX:=C2; K1:=-1
                          END;
                   Q:=K1*(C*D+MAX)/2
                 END;
          IF (P=0) AND (Q=0) THEN
          BEGIN
            P:=0 ; Q:=1
          END
        END { PQ };

  BEGIN { Blok glowny procedure WARWLMAC }
    FOR i:=1 TO N DO
      Y0[i]:=i;
    WSPWCHAR(N,A,Y0,C);
    FROBENIUS(N,A,C1,D1);
    PQ(C1,D1,P,Q);
    ZWBAIRSTOW(N,C,P,Q,EPS,MEGA,BLAD,Z)
  END { WARWLMAC };

PROCEDURE WARWLMAL;
  VAR C,Y0 :WEK;
      i    :BYTE;
  BEGIN
    FOR i:=1 TO N DO
      Y0[i]:=i;
    WSPWCHAR(N,A,Y0,C);
    ZWLAGUERRE(N,C,EPS,maxit,BLAD,Z);
  END { WARWLMAC };

PROCEDURE ZERAFUN;
  VAR ZNAK1,ZNAK2 :SHORTINT;
      X,Y         :FLOAT;

      FUNCTION SIGN(X:FLOAT):SHORTINT;
      { Funkcja znaku }
        BEGIN
          IF X>0
            THEN SIGN:=1
            ELSE IF X=0
                   THEN SIGN:=0
                   ELSE SIGN:=-1
        END { SIGN };

      FUNCTION ZERA(F:FUNX; C,D,EPS:FLOAT):FLOAT;
        { Znajdywanie miejsca zerowego funkcji F(x) w przedziale <C,D> }
        { metoda polowienia przedzialu ograniczonego dlugoscia EPS     }
        VAR FC,FD,R,V :FLOAT;
            SFC       :SHORTINT;
        BEGIN
          FC:=F(C); SFC:=SIGN(FC);
          FD:=F(D);
          IF SFC*FD<=0 THEN
          BEGIN
            WHILE (D-C)>EPS DO
            BEGIN
              R:=(D+C)*0.5;
              V:=F(R);
              IF SFC=SIGN(V)
                THEN C:=R
                ELSE D:=R
            END;
            ZERA:=(C+D)*0.5
          END
        END { ZERA };

  BEGIN { Blok glowny procedure ZERAFUN }
    ZNAK1:=SIGN(F(XDOLNE)); X:=XDOLNE+DX; IL:=0;
    REPEAT
      ZNAK2:=SIGN(F(X));
      IF ZNAK1<>ZNAK2 THEN
      BEGIN
        INC(IL);
        A[IL]:=ZERA(F,X-DX,X,EPS);
        ZNAK1:=ZNAK2;
      END;
      X:=X+DX
    UNTIL X>XGORNE
  END { ZERAFUN };


END.


