{*************************************************************}
{                        Modul ALGELIND                       }
{ ALGEBRA MACIERZY I METODY ROZWIAZYWANIA ROWNAN MACIERZOWYCH }
{                   dla macierzy dynamicznych                 }
{                   Turbo Pascal  wersja 7.0                  }
{                   autor Bernard Baron                       }
{*************************************************************}

UNIT ALGELIND;

{$IFDEF CPU87} {$N+}
       {$ELSE} {$N-}
{$ENDIF}

INTERFACE

   USES  TFLOAT;

   CONST MAXD = 100;                   { Maksymalna liczba niewiadomych i rownan }

   TYPE  PMAXD= -1..MAXD;              { Dopuszczalny przedzial ilosci niewiadomych i rownan }
         WEKD = ARRAY[PMAXD] OF FLOAT; { Typ wektorowy  }
         MACD = ARRAY[PMAXD] OF ^WEKD; { Typ macierzowy }


   PROCEDURE NEWMACD(VAR A    :MACD;
                         N    :BYTE;
                     VAR BLAD :BYTE);
     { Utworzenie tablicy dynamicznej A o N wierszach         }
     { BLAD - nr bledu; 0 - brak bledu                        }

   PROCEDURE DISMACD(VAR A :MACD;
                         N :BYTE);
     { Zwolnienie tablicy dynamicznej A o N wierszach         }

   PROCEDURE TRANMACD(VAR A,AT :MACD;
                          M,N  :BYTE);
     { Wyznaczanie macierzy transponowanej At wzgledem A;     }
     { M,N - odpowiednio liczba wierszy i kolumn macierzy A   }

   PROCEDURE ADDMACD(VAR C,A,B :MACD;
                         M,N   :BYTE);
     { Wyznaczanie sumy macierzy C=A+B                        }
     { o M - wierszach i N - kolumnach                        }

   PROCEDURE SUBMACD(VAR C,A,B :MACD;
                         M,N   :BYTE);
     { Wyznaczanie roznicy macierzy C=A-B                     }
     { o M - wierszach i N - kolumnach                        }

   PROCEDURE MULMACRD(VAR A,B :MACD;
                          t   :FLOAT;
                          M,N :BYTE);
     { Wyznaczanie iloczynu macierzy B przez liczbe t;  A=B*t }

   PROCEDURE MULMACD(VAR D,A,B :MACD;
                         M,P,N :BYTE);
     { Wyznaczanie iloczynu macierzy D=A*B                    }
     {  macierz D - posiada M - wierszy i P - kolumn;         }
     {  macierz A - posiada M - wierszy i N - kolumn;         }
     {  macierz B - posiada N - wierszy i P - kolumn          }

   PROCEDURE MACZEROD(VAR A0  :MACD;
                          M,N :BYTE);
     { Wyznaczanie macierzy A0 o elementach zerowych;         }
     { M - liczba wierszy, N - liczba kolumn                  }

   PROCEDURE MACJEDEND(VAR A1 :MACD;
                           N  :BYTE);
     { Wyznaczanie macierzy jednostkowej A1 rzedu N           }

   FUNCTION NORMACD(VAR A :MACD;
                        N :BYTE):FLOAT;
     { Norma macierzy kwadratowej rzedu N                     }
     {                           __                           }
     {          NORMAC(A) = max  >_ |A[i,j]|   ;  i,j=1..N    }
     {                       j    i                           }

   PROCEDURE EXPMACD(VAR A,B  :MACD;
                         N    :BYTE;
                         EPS  :FLOAT;
                         BLAD :BYTE);
     {                                           B            }
     { Generacja macierzy exponencjalnej   A = e              }
     {                ___                                     }
     {                \     1        n                        }
     {         A   =   >   ---  ( B )     ;   n=0,1,2,....    }
     {                /__   n!                                }
     {                 n                                      }
     {        N - rzad macierzy A i B                         }
     {      EPS - gorna granica bledu macierzy A              }


   PROCEDURE SKALROWD(VAR A :MACD;
                      VAR b :WEKD;
                          N :BYTE);
     { Skalowanie rownania A*x=b wg wzoru (1.30)              }
     {  A - macierz kwadratowa rzedu N                        }
     {  B - wektor wyrazow wolnych o N elementach             }

   PROCEDURE RRALD(VAR A    :MACD;
                   VAR B,X  :WEKD;
                       N    :BYTE;
                       EPS  :FLOAT;
                   VAR BLAD :BYTE);
     { Rozwiazywanie liniowego ukladu rownan A*X=B            }
     { metoda eliminacji Gaussa                               }
     {  A - macierz kwadratowa rzedu N                        }
     {  B - wektor wyrazow wolnych o N elementach             }
     {  X - wektor rozwiazan                                  }
     {  EPS - dokladnosc rozroznienia zerowego wiersza        }
     {        (np. EPS=1E-36)                                 }
     { BLAD - nr bledu; 0 - brak bledu                        }

   PROCEDURE SKALROWMACD(VAR A,B :MACD;
                             N,M :BYTE);
     { Skalowanie rownania macierzowego wg wzoru (1.30)       }
     { A - macierz kwadratowa rzedu N                         }
     { B - macierz wyrazow wolnych o N-wierszach i M-kolumnach}

   PROCEDURE RRMAD(VAR A,B,X :MACD;
                       N,M   :BYTE;
                       EPS   :FLOAT;
                   VAR BLAD  :BYTE);
     { Rozwiazywanie rownania macierzowego A*X=B dla duzych   }
     { macierzy z zastosowaniem tablic dynamicznych typu MACD }
     { A - macierz kwadratowa rzedu N                         }
     { B - macierz wyrazow wolnych o N-wierszach i M-kolumnach}
     { X - macierz rozwiazania o N -wierszach i M -kolumnach  }
     { EPS - dokladnosc rozroznienia zerowego wiersza         }
     {       (np. EPS=1E-36)                                  }
     { BLAD - nr bledu; 0 - brak bledu                        }

   PROCEDURE ODWMACD(VAR A,B  :MACD;
                         N    :BYTE;
                         EPS  :FLOAT;
                     VAR BLAD :BYTE);
     { Odwracanie dynamicznej macierzy kwadratowej A rzedu N  }
     { B - macierz odwrotna wzgledem macierzy A; tj. A*B=A1   }
     { A1 - macierz jednostkowa                               }
     { EPS - dokladnosc rozroznienia zerowego wiersza         }
     {       (np. EPS=1E-36)                                  }
     { BLAD - nr bledu; 0 - brak bledu                        }

  PROCEDURE ODWMAC1D(VAR A    :MACD;
                         N    :BYTE;
                         EPS  :FLOAT;
                     VAR BLAD :BYTE);
     { Odwrocenie macierzy A i zapisanie jej na miejscu oryginalnej }
     { BLAD - nr bledu; 0 - brak bledu                              }

   FUNCTION DETD(A   :MACD;
                 N   :BYTE;
                 EPS :FLOAT):FLOAT;
     { Obliczanie wyznacznika macierzy A rzedu N              }
     { EPS - dokladnosc rozroznienia zerowego wyznacznika     }
     {       (np. EPS=1E-36)                                  }

   FUNCTION WUMACD(VAR A:MACD;
                       N:BYTE;
                       EPS:FLOAT;
                   VAR BLAD:BYTE):FLOAT;
     { Wspolczynnik uwarunkowania macierzy A rzedu N          }
     { EPS - dokladnosc rozroznienia zerowego wiersza         }
     {       (np. EPS=1E-36)                                  }
     { BLAD - nr bledu; 0 - brak bledu                        }

   FUNCTION MWWMD(VAR A     :MACD;
                      N     :BYTE;
                      EPS   :FLOAT;
                      maxit :WORD;
                  VAR BLAD  :BYTE):FLOAT;
     { Wartosc wlasna macierzy A o najwiekszym module         }
     { A - macierz dynamiczna kwadratowa rzedu N              }
     { EPS - zadana dokladnosc bezwzgledna wyznaczania        }
     {       wartosci wlasnej                                 }
     { maxit - maksymalna zadana liczba iteracji              }
     { BLAD - nr bledu; 0 - brak bledu                        }

   FUNCTION MWWMAD(VAR A        :MACD;
                       N        :BYTE;
                       EPS,ALFA :FLOAT;
                       maxit    :WORD;
                   VAR BLAD     :BYTE):FLOAT;
     { Wartosc wlasna macierzy A1-ALFA*A o najwiekszym module }
     { A - macierz dynamiczna kwadratowa rzedu N              }
     { A1 - macierz jednostkowa                               }
     { ALFA - parametr wiekszy od zera                        }
     { EPS - zadana dokladnosc bezwzgledna wyznaczania        }
     {       wartosci wlasnej                                 }
     { maxit- maksymalna zadana liczba iteracji               }
     { BLAD - nr bledu; 0 - brak bledu                        }


   PROCEDURE RRALIJRD(VAR A        :MACD;
                      VAR b,x      :WEKD;
                          N        :BYTE;
                          EPS,ALFA :FLOAT;
                          maxit    :WORD;
                      VAR BLAD     :BYTE);
     { Rozwiazywanie ukladu rownan liniowych  A*x=b  metoda   }
     { iteracji prostej Jacobiego ALFA=1 wzory (1.59), (1.60) }
     { Richardsona  ALFA<>1 wzor (1.62)                       }
     { A - macierz kwadratowa rzedu N                         }
     { b - wektor wyrazow wolnych o N elementach              }
     { x - wektor rozwiazan - wymaga sie pewnych wartosci     }
     {     poczatkowych                                       }
     { EPS - dokladnosc bezwzgledna iteracji (np. EPS=1E-8)   }
     { ALFA - parametr regularyzacji Richardsona              }
     { maxit - zadana maksymalna liczba iteracji              }
     { BLAD - nr bledu; 0 - brak bledu                        }

   PROCEDURE RRALIGSD(VAR A         :MACD;
                      VAR b,x       :WEKD;
                          N         :BYTE;
                          EPS,OMEGA :FLOAT;
                          maxit     :WORD;
                      VAR BLAD      :BYTE);
     { Rozwiazywanie ukladu rownan liniowych  A*x=b  metoda  }
     { iteracji Gaussa-Seidela OMEGA=1 wzory (1.64), (1.65)  }
     { nadrelaksacji  0<OMEGA<>1 wzory (1.66), (1.67)        }
     { A - macierz kwadratowa rzedu N                        }
     { b - wektor wyrazow wolnych o N elementach             }
     { x - wektor rozwiazan - wymaga sie pewnych wartosci    }
     {     poczatkowych                                      }
     { EPS - dokladnosc bezwzgledna iteracji (np. EPS=1E-8)  }
     { OMEGA - parametr relaksacji                           }
     { maxit - zadana maksymalna liczba iteracji             }
     { BLAD - nr bledu; 0 - brak bledu                       }

   PROCEDURE RRMAIGSD(VAR A,B,X     :MACD;
                          N,M       :BYTE;
                          EPS,OMEGA :FLOAT;
                          maxit     :WORD;
                      VAR BLAD      :BYTE);
     { Rozwiazywanie rownania macierzowego  A*X=B  metoda     }
     { iteracji Gaussa-Seidela OMEGA=1 oraz                   }
     { nadrelaksacji  0<OMEGA<>1 wzory (1.68), (1.69)         }
     { A - macierz kwadratowa rzedu N                         }
     { B - macierz wyrazow wolnych o N-wierszach i M-kolumnach}
     { X - macierz rozwiazania o N-wierszach i M-kolumnach    }
     { EPS - dokladnosc bezwzgledna iteracji (np. EPS=1E-8)   }
     { OMEGA - parametr relaksacji                            }
     { maxit - zadana maksymalna liczba iteracji              }
     { BLAD - nr bledu; 0 - brak bledu                        }

   PROCEDURE RRALWTPD(VAR a,b,c,f,x :WEKD;
                          N         :BYTE;
                          EPS       :FLOAT;
                      VAR BLAD      :BYTE);
   { Rozwiazywanie ukladu rownan liniowych A*x=f o macierzy   }
   { trojprzekatniowej metoda jej rozkladu na dwie macierze   }
   { trojkatne wzor (1.70)                                    }
   { a,b,c - wektory wstegowe macierzy A o N elementach       }
   { f - wektor wyrazow wolnych o N elementach                }
   { x - rozwiazanie ukladu rownan jako wektor o N elementach }
   { EPS- wspolczynnik zwiazany z istnieniem macierzy         }
   {      trojkatnej dolnej D rozkladu A=D*G;                 }
   {      jezeli |alfa[i]-beta[i]*c[i-1]|<EPS (wzor (1.71)),  }
   {      to nie istnieje taki rozklad                        }
   { BLAD - nr bledu; 0 - brak bledu                          }

  PROCEDURE PSEROZD(VAR A,B,X :MACD;
                        M,N,R :BYTE;
                    VAR BLAD  :BYTE);
{ Pseudorozwiazanie rownania macierzowego nadokreslonego A*X=B}
{ z zastosowaniem tablic dynamicznych typu MACD (wzory (1.98))}
{  A - macierz glowna o wymiarach M*N gdzie  M>=N             }
{  B - macierz wyrazow wolnych o wymiarach M*R                }
{  X - macierz pseudorozwiazania o wymiarach M*R              }
{  M - liczba wierszy macierzy glownej A                      }
{  N - liczba kolumn macierzy glownej A                       }
{  R - liczba kolumn macierzy wyrazow wolnych B oraz          }
{      pseudorozwiazania X                                    }

   PROCEDURE ODWMAC2D(VAR A,B  :MACD;
                          N    :BYTE;
                      VAR BLAD :BYTE);
{ Odwracanie dynamicznej macierzy kwadratowej A rzedu N  }
{ metoda pseudorozwiazania rownania macierzowego A*B=A1  }
{ B - macierz odwrotna wzgledem macierzy A               }
{ A1 - macierz jednostkowa                               }
{ BLAD - nr bledu; 0 - brak bledu                        }


PROCEDURE PSEROZNKD(VAR A,B,X :MACD;
                        M,N,R :BYTE;
                        EPS   :FLOAT;
                    VAR BLAD  :BYTE);
{ Pseudorozwiazanie rownania macierzowego nadokreslonego A*X=B       }
{ z zastosowaniem tablic dynamicznych typu MACD (wzory (1.98))       }
{ metoda najmniejszych kwadratow                                     }
{ A - macierz glowna o wymiarach M*N gdzie  M>=N                     }
{ B - macierz wyrazow wolnych o wymiarach M*R                        }
{ X - macierz pseudorozwiazania o wymiarach M*R                      }
{ M - liczba wierszy macierzy glownej A                              }
{ N - liczba kolumn macierzy glownej A                               }
{ R - liczba kolumn macierzy wyrazow wolnych B oraz pseudorozwiaz. X }
{ EPS - dokladnosc rozroznienia zerowego wiersza np EPS=1E-36        }

IMPLEMENTATION

PROCEDURE NEWMACD;
VAR i   :INTEGER;
    PAO :LONGINT;
BEGIN
  BLAD:=0;
  PAO:=SizeOf(WEKD)*(N+2);
  IF PAO<MaxAvail
    THEN FOR i:=-1 TO N DO
           NEW(A[i])
    ELSE BLAD:=20
END { NEWMACD };

PROCEDURE DISMACD;
VAR i:INTEGER;
BEGIN
  FOR i:=-1 TO N DO
    DISPOSE(A[i])
END { DISMACD };

PROCEDURE TRANMACD;
VAR i,j:BYTE;
BEGIN
  FOR i:=1 TO M DO
    FOR j:=1 TO N DO
      At[j]^[i]:=A[i]^[j]
END { TRANMACD };

PROCEDURE ADDMACD;
VAR i,j:BYTE;
BEGIN
  FOR i:=1 TO M DO
    FOR j:=1 TO N DO
      C[i]^[j]:=A[i]^[j]+B[i]^[j]
END { ADDMACD };

PROCEDURE SUBMACD;
VAR i,j:BYTE;
BEGIN
  FOR i:=1 TO M DO
    FOR j:=1 TO N DO
      C[i]^[j]:=A[i]^[j]-B[i]^[j]
END; { SUBMACD }

PROCEDURE MULMACRD;
VAR i,j:BYTE;
BEGIN
  FOR i:=1 TO M DO
    FOR j:=1 TO N DO
      A[i]^[j]:=t*B[i]^[j]
END { MULMACRD };

PROCEDURE MULMACD;
VAR i,j,k:BYTE;
    S    :FLOAT;
    DP   :WEKD;
BEGIN
  FOR i:=1 TO M DO
  BEGIN
    FOR k:=1 TO P DO
    BEGIN
      S:=0;
      FOR j:=1 TO N DO
        S:=S+A[i]^[j]*B[j]^[k];
      DP[k]:=S
    END;
    D[i]^:=DP
  END
END { MULMACD };

PROCEDURE MACZEROD;
VAR I:BYTE;
BEGIN
  FOR I:=1 TO N DO
    FillChar(A0[I]^,SizeOf(A0[I]^),0)
END { MACZEROD };

PROCEDURE MACJEDEND;
VAR i:BYTE;
BEGIN
  MACZEROD(A1,N,N);
  FOR i:=1 TO N DO
    A1[i]^[i]:=1
END { MACJEDEND };

FUNCTION NORMACD(VAR A:MACD; N:BYTE):FLOAT;
VAR i,j :BYTE;
    RR,S:FLOAT;
BEGIN
  RR:=0;
  FOR j:=1 TO N DO
  BEGIN
    S:=0;
    FOR i:=1 TO N DO
      S:=S+ABS(A[i]^[j]);
    IF RR<S THEN
      RR:=S
  END;
  NORMACD:=RR
END { NORMACD };

PROCEDURE EXPMACD;
VAR k,i               :WORD;
    S,TETA,NB,NBk,EXPB:FLOAT;
    AX,AY             :MACD;
BEGIN
  NEWMACD(AX,N,BLAD);
  IF BLAD=0 THEN
  BEGIN
    NEWMACD(AY,N,BLAD);
    IF BLAD=0 THEN
    BEGIN
      MACJEDEND(AX,N);               {pkt (1.8)}
      k:=0; S:=1;
      NB:=NORMACD(B,N);              {pkt (1.9)}
      EXPB:=EXP(NB);
      FOR i:=1 TO N DO
        A[i]^:=AX[i]^;
      NBk:=NB;
      REPEAT
        k:=k+1;
        MULMACD(AY,AX,B,N,N,N);       {pkt (1.6)}
        S:=S/k;  NBk:=NBk*NB;
        MULMACRD(AX,AY,S,N,N);        {pkt (1.5)}
        ADDMACD(A,A,AX,N,N);          {pkt (1.3)}
        FOR i:=1 TO N DO
          AX[i]^:=AY[i]^;
        TETA:=NBk*S*EXPB/(k+1)        {wzor (1.18)}
      UNTIL TETA<EPS;
      DISMACD(AY,N)
    END;
    DISMACD(AX,N)
  END
END { EXPMACD };

PROCEDURE SKALROWD(VAR A:MACD; VAR b:WEKD; N:BYTE);
VAR i,j     :BYTE;
    LN2,S,S1:FLOAT;
BEGIN
  LN2:=LN(2);
  { Skalowanie macierzy A oraz wektora wyrazow wolnych B }
  FOR i:=1 TO N DO
  BEGIN
    S:=0;
    FOR j:=1 TO N DO
      S:=S+ABS(A[i]^[j]); { Norma wiersza wg (1.28) }
    { Wspolczynnik skalowania i-tego wiersza wg wzoru (1.30 ) }
    S1:=LN(S)/LN2+1;
    IF (S<0.5) OR (S>1.0) THEN
    BEGIN
      IF S1>=0
        THEN S:=EXP(-TRUNC(S1)*LN2)
        ELSE IF S1-TRUNC(S1)=0.0
               THEN S:=EXP(-TRUNC(S1)*LN2)
               ELSE S:=EXP(-TRUNC(S1-1)*LN2);
      FOR j:=1 TO N DO
        A[i]^[j]:=S*A[i]^[j];
      b[i]:=S*b[i]
    END
  END
END { SKALROWD };

PROCEDURE RRALD;
VAR i,j,k :BYTE;
    S1,S,T:FLOAT;
BEGIN
  BLAD:=0;
  SKALROWD(A,B,N);
  FOR i:=1 TO N DO
  BEGIN
    T:=ABS(A[i]^[i]); k:=i;
    FOR j:=i+1 TO N DO
      IF ABS(A[j]^[i])>T THEN
      BEGIN
        T:=ABS(A[j]^[i]); k:=j
      END;
    IF T<EPS THEN
    BEGIN
      BLAD:=21;
      EXIT
    END;
    IF i=k
      THEN T:=A[i]^[i]
      ELSE BEGIN
             T:=b[k]; b[k]:=b[i]; b[i]:=T;
             FOR j:=N DOWNTO i DO
             BEGIN
               T:=A[k]^[j]; A[k]^[j]:=A[i]^[j]; A[i]^[j]:=T
             END
           END;
    T:=1/T; A[i]^[i]:=T;
    FOR j:=i+1 TO N DO
    BEGIN
      S:=-A[j]^[i]*T;
      b[j]:=b[j]+S*b[i];
      FOR k:=i+1 TO N DO
        A[j]^[k]:=A[j]^[k]+A[i]^[k]*S
    END
  END;
  FOR i:=N DOWNTO 1 DO
  BEGIN
    T:=b[i];
    FOR j:=i+1 TO N DO
      T:=T-A[i]^[j]*X[j];
    X[i]:=T*A[i]^[i]
  END
END { RRALD };

PROCEDURE SKALROWMACD(VAR A,B:MACD; N,M:BYTE);
VAR i,j     :BYTE;
    LN2,S,S1:FLOAT;
BEGIN
  LN2:=LN(2);
  { Skalowanie macierzy A oraz macierzy wyrazow wolnych B }
  FOR i:=1 TO N DO
  BEGIN
    S:=0;
    FOR j:=1 TO N DO
      S:=S+ABS(A[i]^[j]);
    { Wspolczynnik skalowania i-tego wiersza wg wzoru (1.30) }
    S1:=LN(S)/LN2+1;
    IF (S<0.5) OR (S>1.0) THEN
    BEGIN
      IF S1>=0
        THEN S:=EXP(-TRUNC(S1)*LN2)
        ELSE IF S1-TRUNC(S1)=0.0
               THEN S:=EXP(-TRUNC(S1)*LN2)
               ELSE S:=EXP(-TRUNC(S1-1)*LN2);
      FOR j:=1 TO N DO
        A[i]^[j]:=S*A[i]^[j];
      FOR j:=1 TO M DO
        B[j]^[i]:=S*B[j]^[i]
    END
  END
END { SKALROWMACD };


PROCEDURE RRMAD;
VAR i,j,k :BYTE;
    S1,S,T:FLOAT ;
BEGIN
  BLAD:=0;
  SKALROWMACD(A,B,N,M);
  FOR i:=1 TO N DO
  BEGIN
    T:=ABS(A[i]^[i]); k:=i;
    FOR j:=i+1 TO N DO
      IF ABS(A[j]^[i])>T THEN
      BEGIN
        T:=ABS(A[j]^[i]); k:=j
      END;
    IF T<EPS THEN
    BEGIN
      BLAD:=22;
      EXIT
    END;
    IF i=k
      THEN T:=A[i]^[i]
      ELSE BEGIN
             FOR j:=1 TO M DO
             BEGIN
               T:=B[j]^[k]; B[j]^[k]:=B[j]^[i]; B[j]^[i]:=T
             END;
             FOR j:=N DOWNTO i DO
             BEGIN
               T:=A[k]^[j]; A[k]^[j]:=A[i]^[j]; A[i]^[j]:=T
             END
           END;
    T:=1/T; A[i]^[i]:=T;
    FOR j:=i+1 TO N DO
    BEGIN
      S:=-A[j]^[i]*T;
      FOR k:=1 TO M DO
        B[k]^[j]:=B[k]^[j]+B[k]^[i]*S;
      FOR k:=i+1 TO N DO
        A[j]^[k]:=A[j]^[k]+A[i]^[k]*S
    END
  END;
  FOR k:=1 TO M DO
    FOR i:=N DOWNTO 1 DO
    BEGIN
      T:=B[k]^[i];
      FOR j:=i+1 TO N DO
        T:=T-A[i]^[j]*X[k]^[j];
      X[k]^[i]:=T*A[i]^[i]
    END
END { RRMAD };

PROCEDURE ODWMACD;
VAR i,j:BYTE;
    A1 :MACD;
BEGIN
  NEWMACD(A1,N,BLAD); { Utworzenie tablicy dynamicznej }
  IF BLAD=0 THEN
  BEGIN
    { Generacja macierzy jednostkowej A1 (pkt 1.8) }
    FOR i:=1 TO N DO
      FOR j:=1 TO N DO
        IF i=j
          THEN A1[i]^[j]:=1
          ELSE A1[i]^[j]:=0;
    RRMAD(A,A1,B,N,N,EPS,BLAD); { pkt 1.21) }
    { Rozwiazanie rownania macierzowego (1.36) A*B=A1 }
    DISMACD(A1,N)  { Zwolnienie tablicy dynamicznej }
  END
END { ODWMACD };

PROCEDURE ODWMAC1D;
VAR i,j,l,k  :BYTE;
    maxA,d,e :FLOAT;
    M        :ARRAY[PMAXD] OF BYTE;
BEGIN
  BLAD:=0;
  FOR i:=1 TO N DO
  BEGIN
    { Czesciowy wybor elementu glownego wg wzoru (1.49)}
    maxA:=0;
    FOR j:=i TO N DO
    BEGIN
      d:=A[j]^[i];
      IF ABS(maxA)<ABS(d) THEN
      BEGIN
        maxA:=d; k:=j
      END
    END;
    IF ABS(maxA)<EPS THEN
    BEGIN
      BLAD:=31;
      EXIT
    END;
    { Zpisywanie wskaznikow wierszy wystepowania elementu
      ekstremalnego w i-tej iteracji w postaci wektora M[i] }
    M[i]:=k;
    A[k]^[i]:=1;
    FOR j:=1 TO N DO
    BEGIN
      { Przestawienie i-tego wiersz z k-tym }
      d:=A[k]^[j]/maxA;  A[k]^[j]:=A[i]^[j];  A[i]^[j]:=d;
    END;
    { Generacja ciagu macierzy (1.42) wg wzoru rekurencyjnego (1.41) }
    FOR j:=1 TO N DO
      IF j<>i THEN
      BEGIN
        d:=A[j]^[i];  A[j]^[i]:=0;
        FOR l:=1 TO N DO
        BEGIN
          e:=d*A[i]^[l];
          A[j]^[l]:=A[j]^[l]-e
        END
      END
  END;
  { Przestawianie kolumn macierzy zgodnie z wektorem wskaznikow M[i] (1.46) }
  FOR i:=N DOWNTO 1 DO
  BEGIN
    k:=M[i];
    IF k<>i THEN
      FOR j:=1 TO N DO
      BEGIN
        d:=A[j]^[i];
        A[j]^[i]:=A[j]^[k];
        A[j]^[k]:=d
      END
  END
END; { ODWMAC1D }

FUNCTION DETD;
VAR i,j,k,M:BYTE;
    S1,S,T :FLOAT;
BEGIN
  M:=0;
  { Konstrukcja ciagu macierzy A(i) wg wzorow
    (1.20), (1.21), (1.23) }
  FOR i:=1 TO N DO
  BEGIN
    T:=ABS(A[i]^[i]); k:=i;
    FOR j:=i+1 TO N DO
      IF ABS(A[j]^[i])>T THEN
      BEGIN
        { Wybor elementu glownego }
        T:=ABS(A[j]^[i]); k:=j
      END;
    IF T<EPS THEN
    BEGIN
    { Warunek przerwania obliczen ze wzgledu na }
    { zerowanie wyznacznika -  wzor (1.27)      }
      DETD:=0; EXIT
    END;
    IF i=k
      THEN T:=A[i]^[i]
      ELSE BEGIN
             { Zliczanie ilosci przestawien wierszy }
             INC(M);
             FOR j:=N DOWNTO i DO
             BEGIN
               { Zamiana wiersza k-tego z i-tym }
               T:=A[k]^[j]; A[k]^[j]:=A[i]^[j]; A[i]^[j]:=T
             END
           END;
    T:=1/T;
    FOR j:=i+1 TO N DO
    BEGIN
      S:=-A[j]^[i]*T;                  { wzor (1.23) }
      FOR k:=i+1 TO N DO
        A[j]^[k]:=A[j]^[k]+A[i]^[k]*S; { wzor (1.21) }
    END
  END;
  S:=1;
  { Obliczanie wyznacznika wg wzoru (1.48) }
  FOR i:=1 TO N DO
    S:=S*A[i]^[i];
  IF FRAC(M/2)=0.0
    THEN DETD:=S
    ELSE DETD:=-S
END { DETD };

FUNCTION WUMACD;
VAR i,j,k   :BYTE;
    KM1,KM2 :FLOAT;
    B       :MACD;
BEGIN
  KM1:=NORMACD(A,N);  { Norma macierzy dynamicznej A pkt 1.23 }
  NEWMACD(B,N,BLAD);
  IF BLAD=0 THEN
  BEGIN
    ODWMACD(A,B,N,EPS,BLAD);
    IF BLAD=0 THEN
    BEGIN
      KM2:=NORMACD(B,N); {Norma macierzy dynamicznej pkt 1.23 }
      WUMACD:=KM1*KM2;
      DISMACD(B,N)
    END
  END
END { WUMACD };

FUNCTION MWWMD;
VAR i,j            :BYTE;
    k              :WORD;
    P,R,S,T,WW1,WW2:FLOAT;
    W,U            :WEKD;
    CH             :CHAR;
BEGIN
  FOR i:=1 TO N DO
    W[i]:=0.01*i; { wektor poczatkowy (1.50) }
  P:=1.0; WW2:=100; k:=0;
  BLAD:=0;
  REPEAT
    R:=0.0;
    { wzor (1.50) }
    FOR i:=1 TO N DO
    BEGIN
      S:=0.0;
      FOR j:=1 TO N DO
        S:=S+A[i]^[j]*W[j];
      U[i]:=S;  S:=ABS(S);
      IF S>R THEN
        R:=S { wyznaczanie normy wektora wzor (1.51) }
    END;
    WW1:=R/P;  { przyblizanie granicy (1.49) }
    P:=R; W:=U; T:=ABS(WW1-WW2);
    WW2:=WW1; k:=k+1
  UNTIL (T<EPS) OR (k>maxit);
  IF k>maxit THEN
  BEGIN
    BLAD:=23;
    EXIT
  END;
  MWWMD:=WW1
END { MWWMD };

FUNCTION MWWMAD;
VAR i,j            :BYTE;
    k              :WORD;
    P,R,S,T,WW1,WW2:FLOAT;
    W,U            :WEKD;
    CH             :CHAR;
BEGIN
  BLAD:=0;
  FOR i:=1 TO N DO
    W[i]:=0.01*i; { wektor poczatkowy (1.50) }
  P:=1.0; WW2:=100.0; k:=0;
  REPEAT
    R:=0.0;
    { wzor (1.50a) }
    FOR i:=1 TO N DO
    BEGIN
      S:=0.0;
      FOR j:=1 TO N DO
        S:=S+A[i]^[j]*W[j];
      U[i]:=W[i]-ALFA*S;  S:=ABS(U[i]);
      IF S>R THEN
        R:=S { wyznaczanie normy wektora wzor (1.51) }
    END;
    WW1:=R/P;  { przyblizanie granicy (1.49) }
    P:=R; W:=U;  T:=ABS(WW1-WW2);  WW2:=WW1;  k:=k+1
  UNTIL (T<EPS) OR (k>maxit);
  IF k>maxit THEN
  BEGIN
    BLAD:=24;
    EXIT
  END;
  MWWMAD:=WW1
END { MWWMAD };

PROCEDURE RRALIJRD;
VAR i,j  :BYTE;
    k    :WORD;
    R,S,T:FLOAT;
    y    :WEKD;
    CH   :CHAR;
BEGIN
  BLAD:=0;
  SKALROWD(A,b,N);       { Skalowanie rownania A*x=b }
  { Przeksztalcenie macierzy A do postaci z mozliwie }
  { maksymalnymi co do modulu elementami na glownej  }
  { przekatnej metoda przestawiania wierszy          }
  FOR i:=1 TO N DO
  BEGIN
    T:=ABS(A[i]^[i]); k:=i;
    FOR j:=i+1 TO N DO
      IF ABS(A[j]^[i])>T THEN
      BEGIN
        T:=ABS(A[j]^[i]); k:=j
      END;
    IF i<>k THEN
    BEGIN
      T:=b[k]; b[k]:=b[i]; b[i]:=T;
      FOR j:=1 TO N DO
      BEGIN
        T:=A[k]^[j]; A[k]^[j]:=A[i]^[j]; A[i]^[j]:=T
      END
    END
  END;
  R:=MWWMAD(A,N,1E-4,ALFA,maxit,BLAD);
  IF BLAD=0 THEN
  BEGIN
    k:=0;
    { Najwieksza co do modulu wartosc wlasna macierzy A1-ALFA*A }
    IF R>=1 THEN
    BEGIN
      BLAD:=25;
      EXIT
    END;
    REPEAT
      { Iteracja prosta Jacobiego }
      y:=x; R:=0.0; k:=k+1;
      FOR i:=1 TO N DO
      BEGIN
        S:=b[i];
        FOR j:=1 TO N DO
          S:=S-A[i]^[j]*y[j];
        x[i]:=y[i]+ALFA*S;  S:=ABS(S);
        IF S>R THEN
          R:=S
      END
    UNTIL (R<EPS) OR (k>maxit);
    IF k>maxit THEN
      BLAD:=26
  END
END { RRALIJRD };

PROCEDURE RRALIGSD;
VAR i,j  :BYTE;
    k    :WORD;
    R,S,T:FLOAT;
    y    :WEKD;
    CH   :CHAR;
BEGIN
  BLAD:=0;
  SKALROWD(A,b,N);       { Skalowanie rownania A*x=b }
  { Przeksztalcenie macierzy A do postaci z mozliwie }
  { maksymalnymi co do modulu elementami na glownej  }
  { przekatnej metoda przestawiania wierszy          }
  FOR i:=1 TO N DO
  BEGIN
    T:=ABS(A[i]^[i]); k:=i;
    FOR j:=i+1 TO N DO
      IF ABS(A[j]^[i])>T THEN
      BEGIN
        T:=ABS(A[j]^[i]); k:=j
      END;
    IF i<>k THEN
    BEGIN
      T:=b[k]; b[k]:=b[i]; b[i]:=T;
      FOR j:=1 TO N DO
      BEGIN
        T:=A[k]^[j]; A[k]^[j]:=A[i]^[j]; A[i]^[j]:=T
      END
    END
  END;
  R:=MWWMAD(A,N,1E-4,OMEGA,maxit,BLAD);
  IF BLAD=0 THEN
  BEGIN
    k:=0;
    { Najwieksza co do modulu wartosc wlasna macierzy A1-OMEGA*A}
    IF R>=1 THEN
    BEGIN
      BLAD:=27;
      EXIT
    END;
    REPEAT
      y:=x; R:=0.0; k:=k+1;
      FOR i:=1 TO N DO
      BEGIN
        S:=b[i];
        FOR j:=1 TO i-1 DO
          S:=S-A[i]^[j]*x[j];
        FOR j:=i TO N DO
          S:=S-A[i]^[j]*y[j];
        x[i]:=y[i]+OMEGA*S;  S:=ABS(S);
        IF S>R THEN
          R:=S
      END
    UNTIL (R<EPS) OR (k>maxit);
    IF k>maxit THEN
      BLAD:=28
  END
END { RRALIGSD };

PROCEDURE RRMAIGSD;
VAR i,j,l,k :BYTE;
    p       :WORD;
    R,R1,S,T:FLOAT;
    y       :WEKD;
    CH      :CHAR;
BEGIN
  BLAD:=0;
  SKALROWMACD(A,B,N,M);  { Skalowanie rownania A*X=B }
  { Przeksztalcenie macierzy A do postaci z mozliwie }
  { maksymalnymi co do modulu elementami na glownej  }
  { przekatnej metoda przestawiania wierszy          }
  FOR i:=1 TO N DO
  BEGIN
    T:=ABS(A[i]^[i]); k:=i;
    FOR j:=i+1 TO N DO
      IF ABS(A[j]^[i])>T THEN
      BEGIN
        T:=ABS(A[j]^[i]);k:=j
      END;
    IF i<>k THEN
    BEGIN
      FOR j:=1 TO M DO
      BEGIN
        T:=B[j]^[k]; B[j]^[k]:=B[j]^[i]; B[j]^[i]:=T
      END;
      FOR j:=1 TO N DO
      BEGIN
        T:=A[k]^[j]; A[k]^[j]:=A[i]^[j]; A[i]^[j]:=T
      END
    END
  END;
  R:=MWWMAD(A,N,1E-4,OMEGA,maxit,BLAD);
  IF BLAD=0 THEN
  BEGIN
    p:=0;
    { Najwieksza co do modulu wartosc wlasna macierzy A1-A }
    IF R>=1 THEN
    BEGIN
      BLAD:=29;
      EXIT
    END;
    REPEAT
      R1:=0; p:=p+1;
      FOR l:=1 TO M DO
      BEGIN
        y:=X[l]^ ; R:=0.0;
        FOR i:=1 TO N DO
        BEGIN
          S:=B[l]^[i];
          FOR j:=1 TO i-1 DO
            S:=S-A[i]^[j]*X[l]^[j];
          FOR j:=i TO N DO
            S:=S-A[i]^[j]*y[j];
          X[l]^[i]:=y[i]+OMEGA*S;  S:=ABS(S);
          IF S>R THEN
            R:=S
        END;
        IF R>R1 THEN
          R1:=R
      END
    UNTIL (R1<EPS) OR (p>maxit);
    IF p>maxit THEN
      BLAD:=30;
  END
END { RRMAIGSD };

PROCEDURE RRALWTPD;
VAR i:BYTE;
    y:WEKD;
BEGIN
  BLAD:=0;
  y[1]:=f[1];
  { Rozklad trojkatny macierzy o wstegach a,b,c }
  { wzory (1.61) i (1.62)                       }
  FOR i:=2 TO N DO
  BEGIN
    IF ABS(a[i-1])<EPS THEN
    BEGIN
      BLAD:=11;
      EXIT
    END;
    b[i]:=b[i]/a[i-1]; a[i]:=a[i]-b[i]*c[i-1];
    { Rozwiazywanie ukladu rownan D*y=f o macierzy     }
    { trojkatnej dolnej D (wzor (1.61)) w przod (1.63) }
    y[i]:=f[i]-b[i]*y[i-1]
  END;
  { Rozwiazywanie ukladu rownan G*x=y o macierzy       }
  { trojkatnej gornej G (wzor (1.61)) wstecz (1.64)    }
  x[N]:=y[N]/a[N];
  FOR i:=N-1 DOWNTO 1 DO
    x[i]:=(y[i]-c[i]*x[i+1])/a[i]
END { RRALWTPD };

PROCEDURE PSEROZD;
VAR AR,BR,P           :MACD;
    W                 :WEKD;
    SIG,EPS,W1,BETA,R1:FLOAT;
    i,k,j,N1          :BYTE;

  PROCEDURE BWWT(L:BYTE;BETA:FLOAT);
    VAR i,j:BYTE;
    BEGIN
      FOR i:=1 TO L DO
        FOR j:=i TO L DO
          P[i]^[j]:=W[i]*W[j]*BETA;
      FOR i:=1 TO L DO
        FOR j:=i TO L DO
          P[j]^[i]:=P[i]^[j]
    END { BWWT };

  PROCEDURE Q(L:BYTE);
    VAR i,j:BYTE;
    BEGIN
      FOR i:=1 TO L DO
        FOR j:=1 TO L DO
          P[i]^[j]:=-P[i]^[j];
      FOR i:=1 TO L DO
        P[i]^[i]:=1+P[i]^[i]
    END { Q };

  PROCEDURE MULTM(k:BYTE);
    VAR i,j,R:BYTE;
        S    :FLOAT;
    BEGIN
      FOR i:=1 TO M DO
        FOR j:=1 TO N DO
          AR[i]^[j]:=A[i]^[j];
      FOR i:=k TO M DO
        FOR j:=k+1 TO N DO
        BEGIN
          S:=0;
          FOR R:=k TO M DO
            S:=S+P[i]^[R]*AR[R]^[j];
          A[i]^[j]:=S
        END;
      i:=k; j:=k; S:=0;
      FOR R:=k TO M DO
        S:=S+P[i]^[R]*AR[R]^[j];
      A[i]^[j]:=S;
      FOR i:=k+1 TO M DO
        A[i]^[k]:=0
    END { MULTM };

  PROCEDURE MULTW(k:BYTE);
    VAR i,j,l:BYTE;
        S    :FLOAT;
    BEGIN
      FOR i:=1 TO M DO
        FOR j:=1 TO R DO
          BR[j]^[i]:=B[j]^[i];
      FOR i:=k TO M DO
        FOR j:=1 TO R DO
        BEGIN
          S:=0;
          FOR L:=k TO M DO
            S:=S+P[i]^[L]*BR[j]^[L];
          B[j]^[i]:=S
        END
    END { MULTW };

  PROCEDURE PK(L:BYTE);
    VAR i,j:BYTE;
    BEGIN
      FOR i:=0 TO L-1 DO
        FOR j:=1 TO L DO
          P[M-i]^[j]:=P[L-i]^[j];
      FOR j:=0 TO L-1 DO
        FOR i:=M-L+1 TO M DO
          P[i]^[M-j]:=P[i]^[L-j];
      FOR i:=1 TO M DO
        FOR j:=1 TO M-L DO
          P[i]^[j]:=0;
      FOR i:=1 TO M-L DO
        FOR j:=M-L+1 TO M DO
          P[i]^[j]:=0;
      FOR i:=1 TO k-1 DO
        P[i]^[i]:=1
    END { PK };

BEGIN   { Blok glowny procedury PSEROZD }
  NEWMACD(AR,M,BLAD);
  IF BLAD=0 THEN
  BEGIN
    NEWMACD(P,M,BLAD);
    IF BLAD=0 THEN
    BEGIN
      NEWMACD(Br,R,BLAD);
      IF BLAD=0 THEN
      BEGIN
        SIG:=0;
        FOR i:=1 TO M DO
          SiG:=SIG+SQR(A[i]^[1]);
        SIG:=SQRT(SIG);
        IF A[1]^[1]>0
          THEN EPS:=1
          ELSE EPS:=-1;
        R1:=SIG+ABS(A[1]^[1]); W1:=EPS*R1; BETA:=1/(SIG*R1);
        W[1]:=W1;
        FOR i:=2 TO M DO
          W[i]:=A[i]^[1];
        BWWT(M,BETA); Q(M); MULTM(1); MULTW(1);
        IF M=N
          THEN N1:=N-1
          ELSE N1:=N;
        FOR k:=2 TO N1 DO
        BEGIN
          SIG:=0;
          FOR i:=k TO M DO
            SiG:=SIG+SQR(A[i]^[k]);
          SIG:=SQRT(SIG);
          IF A[k]^[k]>0
            THEN EPS:=1
            ELSE EPS:=-1;
          R1:=SIG+ABS(A[k]^[k]); W1:=EPS*R1;
          BETA:=1/(SIG*R1); W[1]:=W1;
          FOR i:=k+1 TO M DO
          W[i-k+1]:=A[i]^[k];
          BWWT(M-k+1,BETA); Q(M-k+1); PK(M-k+1);
          MULTM(k); MULTW(k)
        END;
        FOR k:=1 TO R DO
          FOR i:=N DOWNTO 1 DO
          BEGIN
            R1:=B[k]^[i];
            FOR j:=i+1 TO N DO
              R1:=R1-A[i]^[j]*X[k]^[j];
            X[k]^[i]:=R1/A[i]^[i]
          END;
        DISMACD(BR,R)
      END;
      DISMACD(P,M)
    END;
    DISMACD(AR,M)
  END;
END { PSEROZD };

PROCEDURE ODWMAC2D;
VAR i,j:BYTE;
    A1 :MACD;
BEGIN
  NEWMACD(A1,N,BLAD); { Utworzenie tablicy dynamicznej pkt (1.27) }
  IF BLAD=0 THEN
  BEGIN
    { Generacja macierzy jednostkowej A1 (pkt 1.8) }
    MACJEDEND(A1,N);
    { Rozwiazanie rownania macierzowego (1.36) A*B=A1 }
    PSEROZD(A,A1,B,N,N,N,BLAD);  { pkt 1.24 }
    DISMACD(A1,N)  { Zwolnienie tablicy dynamicznej pkt (1.28)}
  END
END { ODWMAC2D };

PROCEDURE PSEROZNKD;
  VAR i,j,k :BYTE;
      Y     :WEKD;
      S     :FLOAT;
  BEGIN
    { Obliczanie iloczynu macierzy transponowanej A tj. At z macierza B }
    { i umieszczenie jej w tej samej tablicy B bez dodatkowej deklaracji  }
    {    B:=At*B  }
    FOR i:=1 TO R DO
    BEGIN
      { Zapamietanie i-tej kolumny macierzy B jako wektora Y }
      FOR k:=1 TO M DO
        Y[k]:=B[i]^[k];
      FOR j:=1 TO N DO
      BEGIN
        S:=0;
        FOR k:=1 TO M DO
          S:=S+A[k]^[j]*Y[k];
        { Podstawienie iloczynu At*Y w i-tej kolumnie tablicy B }
        B[i]^[j]:=S
      END {j}
    END {i};
    { Obliczanie iloczynu macierzy transponowanej A tj. At z macierza A }
    { i umieszczenie jej w tej samej tablicy A bez dodatkowej deklaracji  }
    {    A:=At*A  }
    FOR i:=1 TO N DO
    BEGIN
      { Zapamietanie i-tej kolumny macierzy A jako wektora Y }
      FOR k:=1 TO M DO
        Y[k]:=A[k]^[i];
      FOR j:=i TO N DO
      BEGIN
        S:=0;
        FOR k:=1 TO M DO
          S:=S+A[k]^[j]*Y[k];
        { Podstawienie iloczynu podmacierzy At,wycietej poczawszy }
        { od i-tego wierszsz,z wektorem Y oraz umieszcanie wyniku }
        { tego iloczynu w i-tej kolumnie tablicy A od i-tego      }
        { wiersza poczawszy tj. dla j>=i                          }
        A[j]^[i]:=S
      END {j}
    END {i};
    FOR i:=1 TO N DO
      FOR j:=i+1 TO N DO
        A[i]^[j]:=A[j]^[i];
    RRMAD(A,B,X,N,R,EPS,BLAD);
    IF BLAD<>0 THEN
      BLAD:=32
  END { PSEROZNKD };


END.
