{*************************************************************}
{                        Modul ALGELIN                        }
{ ALGEBRA MACIERZY I METODY ROZWIAZYWANIA ROWNAN MACIERZOWYCH }
{                       dla macierzy statycznych              }
{                       Turbo Pascal  wersja 7.0              }
{                       autor Bernard Baron                   }
{*************************************************************}

UNIT ALGELIN;

{$IFDEF CPU87} {$N+}
       {$ELSE} {$N-}
{$ENDIF}

INTERFACE

   USES  TFLOAT;

   CONST MAX  = 25;                    { Maksymalna liczba niewiadomych i rownan }

   TYPE  PMAX = -1..MAX;               { Dopuszczalny przedzial ilosci niewiadomych i rownan }
         WEK  = ARRAY[PMAX] OF FLOAT;  { Typ wektorowy  }
         MAC  = ARRAY[PMAX] OF WEK;    { Typ macierzowy }


   PROCEDURE TRANMAC(VAR A,AT :MAC;
                         M,N  :BYTE);
     { Wyznaczanie macierzy transponowanej At wzgledem A;     }
     { M,N - odpowiednio liczba wierszy i kolumn macierzy A   }

   PROCEDURE ADDMAC(VAR C,A,B :MAC;
                        M,N   :BYTE);
     { Wyznaczanie sumy macierzy C=A+B                        }
     { o M - wierszach i N - kolumnach                        }

   PROCEDURE SUBMAC(VAR C,A,B :MAC;
                        M,N   :BYTE);
     { Wyznaczanie roznicy macierzy C=A-B                     }
     { o M - wierszach i N - kolumnach                        }

   PROCEDURE MULMACR(VAR A,B :MAC;
                         t   :FLOAT;
                         M,N :BYTE);
     { Wyznaczanie iloczynu macierzy B przez liczbe t;  A=B*t }

   PROCEDURE MULMAC(VAR D,A,B :MAC;
                        M,P,N :BYTE);
     { Wyznaczanie iloczynu macierzy D=A*B                    }
     {  macierz D - posiada M - wierszy i P - kolumn;         }
     {  macierz A - posiada M - wierszy i N - kolumn;         }
     {  macierz B - posiada N - wierszy i P - kolumn          }

   PROCEDURE MACZERO(VAR A0  :MAC;
                         M,N :BYTE);
     { Wyznaczanie macierzy A0 o elementach zerowych;         }
     { M - liczba wierszy, N - liczba kolumn                  }

   PROCEDURE MACJEDEN(VAR A1 :MAC;
                          N  :BYTE);
     { Wyznaczanie macierzy jednostkowej A1 rzedu N           }

   FUNCTION NORMAC(VAR A :MAC;
                       N :BYTE):FLOAT;
     { Norma macierzy kwadratowej rzedu N                     }
     {                           __                           }
     {          NORMAC(A) = max  >_ |A[i,j]|   ;  i,j=1..N    }
     {                       j    i                           }

   PROCEDURE EXPMAC(VAR A,B :MAC;
                        N   :BYTE;
                        EPS :FLOAT);
     {                                           B            }
     { Generacja macierzy exponencjalnej   A = e              }
     {                ___                                     }
     {                \     1        n                        }
     {         A   =   >   ---  ( B )     ;   n=0,1,2,....    }
     {                /__   n!                                }
     {                 n                                      }
     {        N - rzad macierzy A i B                         }
     {      EPS - gorna granica bledu macierzy A              }

   PROCEDURE SKALROW(VAR A :MAC;
                     VAR b :WEK;
                         N :BYTE);
     { Skalowanie rownania A*x=b wg wzoru (1.30)              }
     {  A - macierz kwadratowa rzedu N                        }
     {  B - wektor wyrazow wolnych o N elementach             }

   PROCEDURE RRAL(VAR A    :MAC;
                  VAR B,X  :WEK;
                      N    :BYTE;
                      EPS  :FLOAT;
                  VAR BLAD :BYTE);
     { Rozwiazywanie liniowego ukladu rownan A*X=B            }
     { metoda eliminacji Gaussa                               }
     { A - macierz kwadratowa rzedu N                         }
     { B - wektor wyrazow wolnych o N - elementach            }
     { X - wektor rozwiazania o N - rozwiazaniach             }
     { EPS - dokladnosc rozroznienia zerowego wiersza         }
     {       (np. EPS=1E-36)                                  }
     { BLAD - nr bledu; 0 - brak bledu                        }

   PROCEDURE SKALROWMAC(VAR A,B :MAC;
                            N,M :BYTE);
     { Skalowanie rownania macierzowego wg wzoru (1.30)       }
     { A - macierz kwadratowa rzedu N                         }
     { B - macierz wyrazow wolnych o N-wierszach i M-kolumnach}

   PROCEDURE RRMA(VAR A,B,X :MAC;
                      N,M   :BYTE;
                      EPS   :FLOAT;
                  VAR BLAD  :BYTE);
     { Rozwiazywanie rownania macierzowego A*X=B;             }
     { A - macierz kwadratowa rzedu N                         }
     { B - macierz wyrazow wolnych o N-wierszach i M-kolumnach}
     { X - macierz rozwiazania o N - wierszach i M - kolumnach}
     { EPS - dokladnosc rozroznienia zerowego wiersza         }
     {       (np. EPS=1E-25)                                  }
     { BLAD - nr bledu; 0 - brak bledu                        }

   PROCEDURE ODWMAC(VAR A,B  :MAC;
                        N    :BYTE;
                        EPS  :FLOAT;
                    VAR BLAD :BYTE);
     { Odwracanie macierzy kwadratowej A rzedu N              }
     { B - macierz odwrotna wzgledem macierzy A; tj. A*B=A1   }
     { A1 - macierz jednostkowa                               }
     { EPS - dokladnosc rozroznienia zerowego wiersza         }
     {       (np. EPS=1E-36)                                  }
     { BLAD - nr bledu; 0 - brak bledu                        }

  PROCEDURE ODWMAC1(VAR A    :MAC;
                        N    :BYTE;
                        EPS  :FLOAT;
                    VAR BLAD :BYTE);
     { Odwrocenie macierzy A i zapisanie jej na miejscu oryginalnej }
     { BLAD - nr bledu; 0 - brak bledu                              }

   FUNCTION DET(A:MAC; N:BYTE; EPS:FLOAT):FLOAT;
     { Obliczanie wyznacznika macierzy A rzedu N              }
     { EPS - dokladnosc rozroznienia zerowego wyznacznika     }
     {       (np. EPS=1E-36)                                  }

   FUNCTION WUMAC(VAR A    :MAC;
                      N    :BYTE;
                      EPS  :FLOAT;
                  VAR BLAD :BYTE):FLOAT;
     { Wspolczynnik uwarunkowania macierzy A rzedu N          }
     { EPS - dokladnosc rozroznienia zerowego wiersza         }
     {       (np. EPS=1E-36)                                  }
     { BLAD - nr bledu; 0 - brak bledu                        }

   FUNCTION MWWM(VAR A     :MAC;
                     N     :BYTE;
                     EPS   :FLOAT;
                     maxit :WORD;
                 VAR BLAD  :BYTE):FLOAT;
     { Wartosc wlasna macierzy A o najwiekszym module         }
     { A     - macierz kwadratowa rzedu N                     }
     { EPS   - zadana dokladnosc bezwzgledna wyznaczania      }
     {         wartosci wlasnej                               }
     { maxit - maksymalna zadana liczba iteracji              }
     { BLAD  - nr bledu; 0 - brak bledu                       }

   FUNCTION MWWMA(VAR A        :MAC;
                      N        :BYTE;
                      EPS,ALFA :FLOAT;
                      maxit    :WORD;
                  VAR BLAD     :BYTE):FLOAT;
     { Wartosc wlasna macierzy A1-ALFA*A o najwiekszym module }
     { A    - macierz kwadratowa rzedu N                      }
     { A1   - macierz jednostkowa                             }
     { ALFA - parametr wiekszy od zera                        }
     { EPS  - zadana dokladnosc bezwzgledna wyznaczania       }
     {        wartosci wlasnej                                }
     { maxit- maksymalna zadana liczba iteracji               }
     { BLAD - nr bledu; 0 - brak bledu                        }


   PROCEDURE RRALIJR(VAR A        :MAC;
                     VAR b,x      :WEK;
                         N        :BYTE;
                         EPS,ALFA :FLOAT;
                         maxit    :WORD;
                     VAR BLAD     :BYTE);
     { Rozwiazywanie ukladu rownan liniowych  A*x=b  metoda   }
     { iteracji prostej Jacobiego ALFA=1 wzory (1.59), (1.60) }
     { Richardsona  ALFA<>1 wzor (1.62)                       }
     { A - macierz kwadratowa rzedu N                         }
     { b - wektor wyrazow wolnych o N elementach              }
     { x - wektor rozwiazan - wymaga sie pewnych wartosci     }
     {     poczatkowych                                       }
     { EPS - dokladnosc bezwzgledna iteracji (np. EPS=1E-8)   }
     { ALFA - parametr regularyzacji Richardsona              }
     { maxit - zadana maksymalna liczba iteracji              }
     { BLAD - nr bledu; 0 - brak bledu                        }

   PROCEDURE RRALIGS(VAR A         :MAC;
                     VAR b,x       :WEK;
                         N         :BYTE;
                         EPS,OMEGA :FLOAT;
                         maxit     :WORD;
                     VAR BLAD      :BYTE);
     { Rozwiazywanie ukladu rownan liniowych  A*x=b  metoda  }
     { iteracji Gaussa-Seidela OMEGA=1 wzory (1.64), (1.65)  }
     { nadrelaksacji  0<OMEGA<>1 wzory (1.66), (1.67)        }
     { A - macierz kwadratowa rzedu N                        }
     { b - wektor wyrazow wolnych o N elementach             }
     { x - wektor rozwiazan - wymaga sie pewnych wartosci    }
     {     poczatkowych                                      }
     { EPS - dokladnosc bezwzgledna iteracji (np. EPS=1E-8)  }
     { OMEGA - parametr relaksacji                           }
     { maxit - zadana maksymalna liczba iteracji             }
     { BLAD - nr bledu; 0 - brak bledu                       }

   PROCEDURE RRMAIGS(VAR A,B,X     :MAC;
                         N,M       :BYTE;
                         EPS,OMEGA :FLOAT;
                         maxit     :WORD;
                     VAR BLAD      :BYTE);
     { Rozwiazywanie rownania macierzowego  A*X=B  metoda     }
     { iteracji Gaussa-Seidela OMEGA=1 oraz                   }
     { nadrelaksacji  0<OMEGA<>1 wzory (1.68), (1.69)         }
     { A - macierz kwadratowa rzedu N                         }
     { B - macierz wyrazow wolnych o N-wierszach i M-kolumnach}
     { X - macierz rozwiazania o N-wierszach i M-kolumnach    }
     { EPS - dokladnosc bezwzgledna iteracji (np. EPS=1E-8)   }
     { OMEGA - parametr relaksacji                            }
     { maxit - zadana maksymalna liczba iteracji              }
     { BLAD - nr bledu; 0 - brak bledu                        }

   PROCEDURE RRALWTP(VAR a,b,c,f,x :WEK;
                         N         :BYTE;
                         EPS       :FLOAT;
                     VAR BLAD      :BYTE);
   { Rozwiazywanie ukladu rownan liniowych A*x=f o macierzy   }
   { trojprzekatniowej metoda jej rozkladu na dwie macierze   }
   { trojkatne wzor (1.70)                                    }
   { a,b,c - wektory wstegowe macierzy A o N elementach       }
   { f - wektor wyrazow wolnych o N elementach                }
   { x - rozwiazanie ukladu rownan jako wektor o N elementach }
   { EPS- wspolczynnik zwiazany z istnieniem macierzy         }
   {      trojkatnej dolnej D rozkladu A=D*G;                 }
   {      jezeli |alfa[i]-beta[i]*c[i-1]|<EPS (wzor (1.71)),  }
   {      to nie istnieje taki rozklad                        }
   { BLAD - nr bledu; 0 - brak bledu                          }

   PROCEDURE PSEROZ(VAR A,B,X :MAC;
                        M,N,R :BYTE);
   { Pseudorozwiazanie rownania macierzowego nadokreslonego A*X=B}
   {  A - macierz glowna o wymiarach M*N gdzie  M>=N             }
   {  B - macierz wyrazow wolnych o wymiarach M*R                }
   {  X - macierz pseudorozwiazania o wymiarach M*R              }
   {  M - liczba wierszy macierzy glownej A                      }
   {  N - liczba kolumn macierzy glownej A                       }
   {  R - liczba kolumn macierzy wyrazow wolnych B oraz          }
   {      pseudorozwiazania X                                    }

   PROCEDURE ODWMAC2(VAR A,B  :MAC;
                         N    :BYTE);
     { Odwracanie macierzy kwadratowej A rzedu N              }
     { metoda psudorozwiazania rownania macierzowego A*B=A1   }
     { B - macierz odwrotna wzgledem macierzy A               }
     { A1 - macierz jednostkowa                               }

   PROCEDURE PSEROZNK(VAR A,B,X :MAC;
                          M,N,R :BYTE;
                          EPS   :FLOAT;
                      VAR BLAD  :BYTE);
    { Pseudorozwiazanie rownania macierzowego nadokreslonego A*X=B       }
    {     metoda najmniejszych kwadratow                                 }
    { A - macierz glowna o wymiarach M*N gdzie  M>=N                     }
    { B - macierz wyrazow wolnych o wymiarach M*R                        }
    { X - macierz pseudorozwiazania o wymiarach M*R                      }
    { M - liczba wierszy macierzy glownej A                              }
    { N - liczba kolumn macierzy glownej A                               }
    { R - liczba kolumn macierzy wyrazow wolnych B oraz pseudorozwiaz. X }
    { EPS - dokladnosc rozroznienia zerowego wiersza np EPS=1E-36        }
    { BLAD - nr bledu; 0 - brak bledu                                    }


IMPLEMENTATION

PROCEDURE TRANMAC;
VAR i,j:BYTE;
BEGIN
  FOR i:=1 TO M DO
    FOR j:=1 TO N DO
      At[j,i]:=A[i,j]
END { TRANMAC };

PROCEDURE ADDMAC;
VAR i,j:BYTE;
BEGIN
  FOR i:=1 TO M DO
    FOR j:=1 TO N DO
      C[i,j]:=A[i,j]+B[i,j]
END { ADDMAC };

PROCEDURE SUBMAC;
VAR i,j:BYTE;
BEGIN
  FOR i:=1 TO M DO
    FOR j:=1 TO N DO
      C[i,j]:=A[i,j]-B[i,j]
END; { SUBMAC }

PROCEDURE MULMACR;
VAR i,j:BYTE;
BEGIN
  FOR i:=1 TO M DO
    FOR j:=1 TO N DO
      A[i,j]:=t*B[i,j]
END { MULMACR };

PROCEDURE MULMAC;
VAR i,j,k:BYTE;
    S    :FLOAT;
    DP   :WEK;
BEGIN
  FOR i:=1 TO M DO
  BEGIN
    FOR k:=1 TO P DO
    BEGIN
      S:=0;
      FOR j:=1 TO N DO
        S:=S+A[i,j]*B[j,k];
      DP[k]:=S
    END;
    D[i]:=DP
  END
END { MULMAC };

PROCEDURE MACZERO;
BEGIN
  FillChar(A0,SizeOf(A0),0)
END { MACZERO };

PROCEDURE MACJEDEN;
VAR i:BYTE;
BEGIN
  MACZERO(A1,N,N);
  FOR i:=1 TO N DO
    A1[i,i]:=1
END { MACJEDEN };

FUNCTION NORMAC;
VAR i,j :BYTE;
    RR,S:FLOAT;
BEGIN
  RR:=0;
  FOR j:=1 TO N DO
  BEGIN
    S:=0;
    FOR i:=1 TO N DO
      S:=S+ABS(A[i,j]);
    IF RR<S THEN
      RR:=S
  END;
  NORMAC:=RR
END { NORMAC };

PROCEDURE EXPMAC;
VAR k                  :WORD;
    S,TETA,NB,NBk,EXPB :FLOAT;
    AX,AY              :MAC;
BEGIN
  MACJEDEN(AX,N);                {pkt (1.8)}
  k:=0; S:=1;
  NB:=NORMAC(B,N);       {pkt (1.9)}
  EXPB:=EXP(NB);
  A:=AX; NBk:=NB;
  REPEAT
    k:=k+1;
    MULMAC(AY,AX,B,N,N,N);       {pkt (1.6)}
    S:=S/k;  NBk:=NBk*NB;
    MULMACR(AX,AY,S,N,N);        {pkt (1.5)}
    ADDMAC(A,A,AX,N,N);          {pkt (1.3)}
    AX:=AY;
    TETA:=NBk*S*EXPB/(k+1)  {wzor (1.18)}
  UNTIL TETA<EPS
END { EXPMAC };

PROCEDURE SKALROW;
VAR i,j     :BYTE;
    LN2,S,S1:FLOAT;
BEGIN
  LN2:=LN(2);
  { Skalowanie macierzy A oraz wektora wyrazow wolnych B }
  FOR i:=1 TO N DO
  BEGIN
    S:=0;
    FOR j:=1 TO N DO
      S:=S+ABS(A[i,j]); { Norma wiersza wg (1.28) }
    { Wspolczynnik skalowania i-tego wiersza wg wzoru (1.30 ) }
    S1:=LN(S)/LN2+1;
    IF (S<0.5) OR (S>1.0) THEN
    BEGIN
      IF S1>=0
        THEN S:=EXP(-TRUNC(S1)*LN2)
        ELSE IF S1-TRUNC(S1)=0.0
               THEN S:=EXP(-TRUNC(S1)*LN2)
               ELSE S:=EXP(-TRUNC(S1-1)*LN2);
      FOR j:=1 TO N DO
        A[i,j]:=S*A[i,j];
      b[i]:=S*b[i]
    END
  END
END { SKALROW };


PROCEDURE RRAL;
VAR  i,j,k :BYTE;
     T,S   :FLOAT;
BEGIN
  BLAD:=0;
  { Skalowanie macierzy A i wektora wyrazow wolnych B }
  SKALROW(A,B,N);
  { Konstrukcja ciagu macierzy A(i) wzory (1.20),(1.21) i (1.23)
    oraz ciagu wektorow B(i) wzory (1.22) }
  FOR i:=1 TO N DO
  BEGIN
    { Wybor elementu glownego wg wzoru (1.24) }
    T:=ABS(A[i,i]); k:=i;
    FOR j:=i+1 TO N DO
      IF ABS(A[j,i])>T THEN
      BEGIN
        T:=ABS(A[j,i]); k:=j { Zamiana wiersza k-tego z i-tym }
      END;
    IF T<EPS THEN
    { Warunek przerwania poszukiwania elementu glownego (1.27)}
    { Warunek nie istnienia rozwiazania rownania              }
    BEGIN
      BLAD:=1;
      EXIT
    END;
    IF i=k
      THEN T:=A[i,i]
      ELSE BEGIN
             { Zamiana elementu k-tego z i-tym wektora B }
             T:=B[k]; B[k]:=B[i]; B[i]:=T;
             { Zamiana wiersza k-tego z i-tym macierzy A }
             FOR j:=N DOWNTO i DO
             BEGIN
               T:=A[k,j]; A[k,j]:=A[i,j]; A[i,j]:=T
             END
           END;
    T:=1/T; A[i,i]:=T;
    FOR j:=i+1 TO N DO
    BEGIN
      S:=-A[j,i]*T;         { wzor (1.23) }
      B[j]:=B[j]+B[i]*S;    { wzor (1.22) }
      FOR k:=i+1 TO N DO
        A[j,k]:=A[j,k]+A[i,k]*S; { wzor (1.21) }
    END
  END;
  { Rozwiazanie ukladu trojkatnego metoda postepowania
    wstecz wzor (1.26) }
  FOR i:=N DOWNTO 1 DO
  BEGIN
    T:=B[i];
    FOR j:=i+1 TO N DO
    T:=T-A[i,j]*x[j];
    X[i]:=T*A[i,i]
  END
END { RRAL };

PROCEDURE SKALROWMAC;
VAR i,j     :BYTE;
    LN2,S,S1:FLOAT;
BEGIN
  LN2:=LN(2);
  { Skalowanie macierzy A oraz macierzy wyrazow wolnych B }
  FOR i:=1 TO N DO
  BEGIN
    S:=0;
    FOR j:=1 TO N DO
      S:=S+ABS(A[i,j]);
    { Wspolczynnik skalowania i-tego wiersza wg wzoru (1.30) }
    S1:=LN(S)/LN2+1;
    IF (S<0.5) OR (S>1.0) THEN
    BEGIN
      IF S1>=0
        THEN S:=EXP(-TRUNC(S1)*LN2)
        ELSE IF S1-TRUNC(S1)=0.0
               THEN S:=EXP(-TRUNC(S1)*LN2)
               ELSE S:=EXP(-TRUNC(S1-1)*LN2);
      FOR j:=1 TO N DO
        A[i,j]:=S*A[i,j];
      FOR j:=1 TO M DO
        B[i,j]:=S*B[i,j]
    END
  END
END { SKALROWMAC };


PROCEDURE RRMA;
VAR i,j,k :BYTE;
    T,S   :FLOAT;
BEGIN
  BLAD:=0;
  { Skalowanie macierzy glownej A oraz macierzy wyrazow wolnych B }
  SKALROWMAC(A,B,N,M);
  { Konstrukcja ciagu macierzy A(i) wzory (1.20), (1.21), }
  { (1.23) oraz ciagu macierzy B(i) wg wzoru (1.32)       }
  FOR i:=1 TO N DO
  BEGIN
    { Wybor elementu glownego wg wzoru (1.24) }
    T:=ABS(A[i,i]); k:=i;
    FOR j:=i+1 TO N DO
      IF ABS(A[j,i])>T THEN
      BEGIN
        T:=ABS(A[j,i]); k:=j { Zamiana wiersza k-tego z i-tym}
      END;
    IF T<EPS THEN
    { Warunek przerwania poszukiwania elementu glownego (1.27)
      warunek nie istnienia rozwiazania rownania macierzowego}
    BEGIN
      BLAD:=2;
      EXIT
    END;
    IF i=k
      THEN T:=A[i,i]
      ELSE BEGIN
             FOR j:=1 TO M DO
             BEGIN
               { Zamiana wiersza k-tego z i-tym macierzy B }
               T:=B[k,j]; B[k,j]:=B[k,j]; B[k,j]:=T
             END;
             FOR j:=N DOWNTO i DO
             BEGIN
               { Zamiana wiersza k-tego z i-tym macierzy A }
               T:=A[k,j]; A[k,j]:=A[i,j]; A[i,j]:=T
             END
           END;
    T:=1/T; A[i,i]:=T;
    FOR j:=i+1 TO N DO
    BEGIN
      S:=-A[j,i]*T;                { wzor (1.23) }
      FOR k:=1 TO M DO
        B[j,k]:=B[j,k]+B[i,k]*S;   { wzor (1.32) }
      FOR k:=i+1 TO N DO
        A[j,k]:=A[j,k]+A[i,k]*S    { wzor (1.21) }
    END
  END;
{ Rozwiazanie ukladu trojkatnego (1.33) metoda postepowania
  wstecz wzor (1.34) }
  FOR k:=1 TO M DO
    FOR i:=N DOWNTO 1 DO
    BEGIN
      T:=B[i,k];
      FOR j:=i+1 TO N DO
        T:=T-A[i,j]*X[j,k];
      X[i,k]:=T*A[i,i]
    END
END { RRMA };


PROCEDURE ODWMAC;
VAR A1:MAC;
BEGIN
  MACJEDEN(A1,N); {Generacja macierzy jednostkowej (pkt 1.8)  }
  {                                                        -1 }
  { Rozwiazanie rownania macierzowego (1.36) A*B=A1; tj. B=A  }
  RRMA(A,A1,B,N,N,EPS,BLAD) { pkt 1.14 }
END { ODWMAC };

PROCEDURE ODWMAC1;
VAR i,j,l,k  :BYTE;
    maxA,d,e :FLOAT;
    M        :ARRAY[PMAX] OF BYTE;
BEGIN
  BLAD:=0;
  FOR i:=1 TO N DO
  BEGIN
    { Czesciowy wybor elementu glownego wg wzoru (1.49)}
    maxA:=0;
    FOR j:=i TO N DO
    BEGIN
      d:=A[j,i];
      IF ABS(maxA)<ABS(d) THEN
      BEGIN
        maxA:=d; k:=j
      END
    END;
    IF ABS(maxA)<EPS THEN
    BEGIN
      BLAD:=12;
      EXIT
    END;
    { Zapisywanie wskaznikow wierszy wystepowania elementu
      ekstremalnego w i-tej iteracji w postaci wektora M[i] }
    M[i]:=k;
    A[k,i]:=1;
    FOR j:=1 TO N DO
    BEGIN
      { Przestawienie i-tego wiersza z k-tym }
      d:=A[k,j]/maxA;  A[k,j]:=A[i,j];  A[i,j]:=d;
    END;
    { Generacja ciagu macierzy (1.42) wg wzoru rekurencyjnego (1.41) }
    FOR j:=1 TO N DO
      IF j<>i THEN
      BEGIN
        d:=A[j,i];  A[j,i]:=0;
        FOR l:=1 TO N DO
        BEGIN
          e:=d*A[i,l];
          A[j,l]:=A[j,l]-e
        END
      END
  END;
  { Przestawianie kolumn macierzy zgodnie z wektorem wskaznikow M[i] (1.46) }
  FOR i:=N DOWNTO 1 DO
  BEGIN
    k:=M[i];
    IF k<>i THEN
      FOR j:=1 TO N DO
      BEGIN
        d:=A[j,i];
        A[j,i]:=A[j,k];
        A[j,k]:=d
      END
  END
END; { ODWMAC1 }

FUNCTION DET;
VAR i,j,k,M:BYTE;
    S1,S,T :FLOAT;
BEGIN
  M:=0;
  { Konstrukcja ciagu macierzy A(i) wg wzorow
    (1.20), (1.21), (1.23) }
  FOR i:=1 TO N DO
  BEGIN
    T:=ABS(A[i,i]); k:=i;
    FOR j:=i+1 TO N DO
      IF ABS(A[j,i])>T THEN
      BEGIN
        { Wybor elementu glownego }
        T:=ABS(A[j,i]); k:=j
      END;
    IF T<EPS THEN
    BEGIN
    { Warunek przerwania obliczen ze wzgledu na }
    { zerowanie wyznacznika -  wzor (1.27)      }
      DET:=0; EXIT
    END;
    IF i=k
      THEN T:=A[i,i]
      ELSE BEGIN
             { Zliczanie ilosci przestawien wierszy }
             INC(M);
             FOR j:=N DOWNTO i DO
             BEGIN
               { Zamiana wiersza k-tego z i-tym }
               T:=A[k,j]; A[k,j]:=A[i,j]; A[i,j]:=T
             END
           END;
    T:=1/T;
    FOR j:=i+1 TO N DO
    BEGIN
      S:=-A[j,i]*T;              { wzor (1.23) }
      FOR k:=i+1 TO N DO
        A[j,k]:=A[j,k]+A[i,k]*S; { wzor (1.21) }
    END
  END;
  S:=1;
  { Obliczanie wyznacznika wg wzoru (1.48) }
  FOR i:=1 TO N DO
    S:=S*A[i,i];
  IF FRAC(M/2)=0.0
    THEN DET:=S
    ELSE DET:=-S
END { DET };


FUNCTION WUMAC;
VAR i,j,k   :BYTE;
    KM1,KM2 :FLOAT;
    B       :MAC;
BEGIN
  KM1:=NORMAC(A,N);  { Norma macierzy A pkt 1.9 }
  ODWMAC(A,B,N,EPS,BLAD);
  IF BLAD=0 THEN
  BEGIN
    KM2:=NORMAC(B,N); {Norma macierzy pkt 1.9 }
    WUMAC:=KM1*KM2;
  END
END { WUMAC };

FUNCTION MWWM;
VAR i,j            :BYTE;
    k              :WORD;
    P,R,S,T,WW1,WW2:FLOAT;
    W,U            :WEK;
    CH             :CHAR;
BEGIN
  FOR i:=1 TO N DO
    W[i]:=0.01*i; { wektor poczatkowy (1.56) }
  P:=1.0; WW2:=100; k:=0;
  BLAD:=0;
  REPEAT
    R:=0.0;
    { wzor (1.56) }
    FOR i:=1 TO N DO
    BEGIN
      S:=0.0;
      FOR j:=1 TO N DO
        S:=S+A[i,j]*W[j];
      U[i]:=S;  S:=ABS(S);
      IF S>R THEN
        R:=S { wyznaczanie normy wektora wzor (1.57) }
    END;
    WW1:=R/P;  { przyblizanie granicy (1.55) }
    P:=R; W:=U; T:=ABS(WW1-WW2);
    WW2:=WW1; k:=k+1
  UNTIL (T<EPS) OR (k>maxit);
  IF k>maxit THEN
  BEGIN
    BLAD:=3;
    EXIT
  END;
  MWWM:=WW1
END { MWWM };

FUNCTION MWWMA;
VAR i,j            :BYTE;
    k              :WORD;
    P,R,S,T,WW1,WW2:FLOAT;
    W,U            :WEK;
    CH             :CHAR;
BEGIN
  BLAD:=0;
  FOR i:=1 TO N DO
    W[i]:=0.01*i; { wektor poczatkowy (1.56) }
  P:=1.0; WW2:=100.0; k:=0;
  REPEAT
    R:=0.0;
    { wzor (1.58) }
    FOR i:=1 TO N DO
    BEGIN
      S:=0.0;
      FOR j:=1 TO N DO
        S:=S+A[i,j]*W[j];
      U[i]:=W[i]-ALFA*S;  S:=ABS(U[i]);
      IF S>R THEN
        R:=S { wyznaczanie normy wektora wzor (1.57) }
    END;
    WW1:=R/P;  { przyblizanie granicy (1.49) }
    P:=R; W:=U;  T:=ABS(WW1-WW2);  WW2:=WW1;  k:=k+1
  UNTIL (T<EPS) OR (k>maxit);
  IF k>maxit THEN
  BEGIN
    BLAD:=4;
    EXIT
  END;
  MWWMA:=WW1
END { MWWMA };

PROCEDURE RRALIJR;
VAR i,j  :BYTE;
    k    :WORD;
    R,S,T:FLOAT;
    y    :WEK;
    CH   :CHAR;
BEGIN
  SKALROW(A,b,N);       { Skalowanie rownania A*x=b }
  { Przeksztalcenie macierzy A do postaci z mozliwie }
  { maksymalnymi co do modulu elementami na glownej  }
  { przekatnej metoda przestawiania wierszy          }
  FOR i:=1 TO N DO
  BEGIN
    T:=ABS(A[i,i]); k:=i;
    FOR j:=i+1 TO N DO
      IF ABS(A[j,i])>T THEN
      BEGIN
        T:=ABS(A[j,i]);
        k:=j
      END;
    IF i<>k THEN
    BEGIN
      T:=b[k]; b[k]:=b[i]; b[i]:=T;
      FOR j:=1 TO N DO
      BEGIN
        T:=A[k,j]; A[k,j]:=A[i,j]; A[i,j]:=T
      END
    END
  END;
  R:=MWWMA(A,N,1E-4,ALFA,maxit,BLAD); k:=0;
  { Najwieksza co do modulu wartosc wlasna macierzy A1-ALFA*A }
  IF R>=1
    THEN BEGIN
          BLAD:=6;
          EXIT
         END
    ELSE BLAD:=0;
  REPEAT
    { Iteracja prosta Jacobiego }
    y:=x; R:=0.0; k:=k+1;
    FOR i:=1 TO N DO
    BEGIN
      S:=b[i];
      FOR j:=1 TO N DO
        S:=S-A[i,j]*y[j];
      x[i]:=y[i]+ALFA*S;  S:=ABS(S);
      IF S>R THEN
        R:=S
    END
  UNTIL (R<EPS) OR (k>maxit);
  IF k>maxit THEN
    BLAD:=6
END { RRALIJR };

PROCEDURE RRALIGS;
VAR i,j  :BYTE;
    k    :WORD;
    R,S,T:FLOAT;
    y    :WEK;
    CH   :CHAR;
BEGIN
  SKALROW(A,b,N);       { Skalowanie rownania A*x=b }
  { Przeksztalcenie macierzy A do postaci z mozliwie }
  { maksymalnymi co do modulu elementami na glownej  }
  { przekatnej metoda przestawiania wierszy          }
  FOR i:=1 TO N DO
  BEGIN
    T:=ABS(A[i,i]); k:=i;
    FOR j:=i+1 TO N DO
      IF ABS(A[j,i])>T THEN
      BEGIN
        T:=ABS(A[j,i]); k:=j
      END;
    IF i<>k THEN
    BEGIN
      T:=b[k]; b[k]:=b[i]; b[i]:=T;
      FOR j:=1 TO N DO
      BEGIN
        T:=A[k,j]; A[k,j]:=A[i,j]; A[i,j]:=T
      END
    END
  END;
  R:=MWWMA(A,N,1E-4,OMEGA,maxit,BLAD); k:=0;
  { Najwieksza co do modulu wartosc wlasna macierzy A1-OMEGA*A}
  IF R>=1
    THEN BEGIN
          BLAD:=7;
          EXIT
         END
    ELSE BLAD:=0;
  REPEAT
    y:=x; R:=0.0; k:=k+1;
    FOR i:=1 TO N DO
    BEGIN
      S:=b[i];
      FOR j:=1 TO i-1 DO
        S:=S-A[i,j]*x[j];
      FOR j:=i TO N DO
        S:=S-A[i,j]*y[j];
      x[i]:=y[i]+OMEGA*S;  S:=ABS(S);
      IF S>R THEN
        R:=S
    END
  UNTIL (R<EPS) OR (k>maxit);
  IF k>maxit THEN
    BLAD:=8
END { RRALIGS };

PROCEDURE RRMAIGS;
VAR i,j,l,k :BYTE;
    p       :WORD;
    R,R1,S,T:FLOAT;
    y       :WEK;
    CH      :CHAR;
BEGIN
  SKALROWMAC(A,B,N,M);  { Skalowanie rownania A*X=B }
  { Przeksztalcenie macierzy A do postaci z mozliwie }
  { maksymalnymi co do modulu elementami na glownej  }
  { przekatnej metoda przestawiania wierszy          }
  FOR i:=1 TO N DO
  BEGIN
    T:=ABS(A[i,i]); k:=i;
    FOR j:=i+1 TO N DO
      IF ABS(A[j,i])>T THEN
      BEGIN
        T:=ABS(A[j,i]);k:=j
      END;
    IF i<>k THEN
    BEGIN
      FOR j:=1 TO M DO
      BEGIN
        T:=B[k,j]; B[k,j]:=B[i,j]; B[i,j]:=T
      END;
      FOR j:=1 TO N DO
      BEGIN
        T:=A[k,j]; A[k,j]:=A[i,j]; A[i,j]:=T
      END
    END
  END;
  R:=MWWMA(A,N,1E-4,OMEGA,maxit,BLAD);
  p:=0;
  { Najwieksza co do modulu wartosc wlasna macierzy A1-A }
  IF R>=1
    THEN BEGIN
           BLAD:=9;
           EXIT
         END
    ELSE BLAD:=0;
  REPEAT
    R1:=0; p:=p+1;
    FOR l:=1 TO M DO
    BEGIN
      FOR i:=1 TO N DO
        y[i]:=X[i,l] ;
      R:=0.0;
      FOR i:=1 TO N DO
      BEGIN
        S:=B[i,l];
        FOR j:=1 TO i-1 DO
          S:=S-A[i,j]*X[j,l];
        FOR j:=i TO N DO
          S:=S-A[i,j]*y[j];
        X[l,i]:=y[i]+OMEGA*S;
        S:=ABS(S);
        IF S>R THEN
          R:=S
      END;
      IF R>R1 THEN
        R1:=R
    END
  UNTIL (R1<EPS) OR (p>maxit);
  IF p>maxit THEN
    BLAD:=10
END { RRMAIGS };

PROCEDURE RRALWTP;
VAR i:BYTE;
    y:WEK;
BEGIN
  BLAD:=0; y[1]:=f[1];
  { Rozklad trojkatny macierzy o wstegach a,b,c }
  { wzory (1.70) i (1.71)                       }
  FOR i:=2 TO N DO
  BEGIN
    IF ABS(a[i-1])<EPS THEN
    BEGIN
      BLAD:=11;
      EXIT
    END;
    b[i]:=b[i]/a[i-1]; a[i]:=a[i]-b[i]*c[i-1];
    { Rozwiazywanie ukladu rownan D*y=f o macierzy     }
    { trojkatnej dolnej D (wzor (1.70)) w przod (1.72) }
    y[i]:=f[i]-b[i]*y[i-1]
  END;
  { Rozwiazywanie ukladu rownan G*x=y o macierzy       }
  { trojkatnej gornej G (wzor (1.70)) wstecz (1.73)    }
  x[N]:=y[N]/a[N];
  FOR i:=N-1 DOWNTO 1 DO
    x[i]:=(y[i]-c[i]*x[i+1])/a[i]
END { RRALWTP };

PROCEDURE PSEROZ;
  VAR AR,BR,P           :MAC;
      W                 :WEK;
      SIG,EPS,W1,BETA,R1:FLOAT;
      i,k,j,N1          :BYTE;

  PROCEDURE BWWT(L:BYTE;BETA:FLOAT);
 {                         T                           }
 { Oblicza macierz beta*W*W  stopnia L*L (wzor (1.78)) }
    VAR i,j:BYTE;
    BEGIN
      FOR i:=1 TO L DO
        FOR j:=i TO L DO
          P[i,j]:=W[i]*W[j]*BETA;
      FOR i:=1 TO L DO
        FOR j:=i TO L DO
          P[j,i]:=P[i,j]
    END { BWWT };

  PROCEDURE Q(L:BYTE);
 {                             T                           }
 { Oblicza macierz Q=I-beta*W*W  stopnia L*L (wzor (1.78)) }
    VAR i,j:BYTE;
    BEGIN
      FOR i:=1 TO L DO
        FOR j:=1 TO L DO
          P[i,j]:=-P[i,j];
      FOR i:=1 TO L DO
        P[i,i]:=1+P[i,i]
    END { Q };

  PROCEDURE MULTM(k:BYTE);
 { Oblicza iloczyn macierzy P(k)*A(k) w k-tym kroku iteracyjnym
   (wzor (1.84)) z uwzglednieniem oszczednosci operacji
   wynikajacych z rozrzedzenia macierzy P(k) oraz A(k) }
    VAR i,j,r:BYTE; S:FLOAT;
    BEGIN
      FOR i:=1 TO M DO
        FOR j:=1 TO N DO
          Ar[i,j]:=A[i,j];
      FOR i:=k TO M DO
        FOR j:=k+1 TO N DO
        BEGIN
          S:=0;
          FOR r:=k TO M DO
            S:=S+P[i,r]*Ar[r,j];
          A[i,j]:=S
        END;
      i:=k; j:=k; S:=0;
      FOR r:=k TO M DO
        S:=S+P[i,r]*Ar[r,j];
      A[i,j]:=S;
      FOR i:=k+1 TO M DO
        A[i,k]:=0
    END { MULTM };

 PROCEDURE MULTW(k:BYTE);
 { Oblicza iloczyn macierzy P(k)*A(k) w k-tym kroku iteracyjnym
   (wzor (1.87)) z uwzglednieniem oszczednosci operacji
   wynikajacych z rozrzedzenia macierzy P(k) }
   VAR i,j,l:BYTE;
       S    :FLOAT;
   BEGIN
     FOR i:=1 TO M DO
       FOR j:=1 TO R DO
         Br[i,j]:=B[i,j];
     FOR i:=k TO M DO
       FOR j:=1 TO R DO
       BEGIN
         S:=0;
         FOR l:=k TO M DO
           S:=S+P[i,l]*Br[l,j];
         B[i,j]:=S
       END
   END { MULTW };

 PROCEDURE PK(L:BYTE);
 { Umieszcza macierz Q(k) stopnia L*L w macierzy P zachowujac
   strukture (1.83) }
   VAR i,j:BYTE;
   BEGIN
     FOR i:=0 TO L-1 DO
       FOR j:=1 TO L DO
         P[M-i,j]:=P[L-i,j];
     FOR j:=0 TO L-1 DO
       FOR i:=M-L+1 TO M DO
         P[i,M-j]:=P[i,L-j];
     FOR i:=1 TO M DO
       FOR j:=1 TO M-L DO
         P[i,j]:=0;
     FOR i:=1 TO M-L DO
       FOR j:=M-L+1 TO M DO
         P[i,j]:=0;
     FOR i:=1 TO k-1 DO
       P[i,i]:=1
    END { PK };

BEGIN   { Blok glowny procedury PSEROZ }
  { Generacja macierzy Q(1) wg wzorow
    (1.78), (1.79), (1.80), (1.81) }
  SIG:=0;
  FOR i:=1 TO M DO
    SIG:=SIG+SQR(A[i,1]);
  SIG:=SQRT(SIG);
  IF A[1,1]>0
    THEN EPS:=1
    ELSE EPS:=-1;
  R1:=SIG+ABS(A[1,1]); W1:=EPS*R1; BETA:=1/(SIG*R1);
  W[1]:=W1;
  FOR i:=2 TO M DO
    W[i]:=A[i,1];
  BWWT(M,BETA); Q(M); MULTM(1); MULTW(1);
  { Petla przemnozen ortogonalnych wg (1.87), przy czym }
  { N1=N-1 gdy A jest macierza kwadratowa oraz          }
  { N1=N gdy A jest macierza prostokatna M*N            }
  IF M=N
    THEN N1:=N-1
    ELSE N1:=N;
  FOR k:=2 TO N1 DO
  BEGIN
    SIG:=0;
    FOR i:=k TO M DO
      SIG:=SIG+SQR(A[i,k]);
    SIG:=SQRT(SIG);
    IF A[k,k]>0
      THEN EPS:=1
      ELSE EPS:=-1;
    R1:=SIG+ABS(A[k,k]); W1:=EPS*R1;
    BETA:=1/(SIG*R1); W[1]:=W1;
    FOR i:=k+1 TO M DO
      W[i-k+1]:=A[i,k];
    BWWT(M-k+1,BETA); Q(M-k+1); PK(M-k+1); MULTM(k); MULTW(k)
  END;
  { Rozwiazanie rownania macierzowego o strukturze macierzy   }
  { trojkatnej (1.89) rownowaznego rownaniu wejsciowemu (1.86)}
  { metoda postepowania wstecz                                }
  FOR k:=1 TO R DO
    FOR i:=N DOWNTO 1 DO
    BEGIN
      R1:=B[i,k];
      FOR j:=i+1 TO N DO
        R1:=R1-A[i,j]*X[j,k];
      X[i,k]:=R1/A[i,i]
    END
END { PSEROZ };

PROCEDURE ODWMAC2;
VAR A1:MAC;
BEGIN
  MACJEDEN(A1,N); {Generacja macierzy jednostkowej (pkt 1.8)  }
  {                                                        -1 }
  { Rozwiazanie rownania macierzowego (1.36) A*B=A1; tj. B=A  }
  { metoda pseudorozwiazania (pkt 1.24)                       }
  PSEROZ(A,A1,B,N,N,N);
END { ODWMAC2 };

PROCEDURE PSEROZNK;
  VAR i,j,k :BYTE;
      Y     :WEK;
      S     :FLOAT;
  BEGIN
    { Obliczanie iloczynu macierzy transponowanej A tj. At z macierza B  }
    { i umieszczenie jej w tej samej tablicy B bez dodatkowej deklaracji }
    {    B:=At*B  }
    FOR i:=1 TO R DO
    BEGIN
      { Zapamietanie i-tej kolumny macierzy B jako wektora Y }
      FOR k:=1 TO M DO
        Y[k]:=B[k,i];
      FOR j:=1 TO N DO
      BEGIN
        S:=0;
        FOR k:=1 TO M DO
          S:=S+A[k,j]*Y[k];
        { Podstawienie iloczynu At*Y w i-tej kolumnie tablicy B }
        B[j,i]:=S
      END {j}
    END {i};
    { Obliczanie iloczynu macierzy transponowanej A tj. At z macierza A  }
    { i umieszczenie jej w tej samej tablicy A bez dodatkowej deklaracji }
    {    A:=At*A  }
    FOR i:=1 TO N DO
    BEGIN
      { Zapamietanie i-tej kolumny macierzy A jako wektora Y }
      FOR k:=1 TO M DO
        Y[k]:=A[k,i];
      FOR j:=i TO N DO
      BEGIN
        S:=0;
        FOR k:=1 TO M DO
          S:=S+A[k,j]*Y[k];
        { Podstawienie iloczynu podmacierzy At,wycietej poczawszy }
        { od i-tego wierszsz,z wektorem Y oraz umieszcanie wyniku }
        { tego iloczynu w i-tej kolumnie tablicy A od i-tego      }
        { wiersza poczawszy tj. dla j>=i                          }
        A[j,i]:=S
      END {j}
    END {i};
    FOR i:=1 TO N DO
      FOR j:=i+1 TO N DO
        A[i,j]:=A[j,i];
    RRMA(A,B,X,N,R,EPS,BLAD);
    IF BLAD<>0 THEN
      BLAD:=13;
  END { PSEROZNK };

END.
