var cv = document.getElementById('canvas');
var ctx = cv.getContext('2d');
var w = cv.width;
var h = cv.height;
var x0 = w / 2.0;
var y0 = h / 2.0;
var lw = 0.5;

/*
 * Wektor swobodny. Jeśli nie podano żadnych argumentów tworzony jest wektor
 * zerowy. Jeżeli podano 2 argumenty muszą to być wspołrzędne x i y punktu
 */
var Vector2f = function (x, y) {
    switch (arguments.length) {
        case 0:
            this.x = 0.0;
            this.y = 0.0;
            break;
        case 2:
            this.x = arguments[0];
            this.y = arguments[1];
            break;
    }
};
/*
 * Oblicza odległość punktu podanego jako Vector2f lub w postaci (x,y) od tego
 * wektora
 */
Vector2f.prototype.distance = function () {
    var distance = 0.0;
    switch (arguments.length) {
        case 1:
            if (arguments[0] instanceof Vector2f) {
                var x = this.x - arguments[0].x;
                var y = this.y - arguments[0].y;
                distance = Math.sqrt(x * x + y * y);
            }
            break;
        case 2:
            var x1 = this.x - arguments[0];
            var y1 = this.y - arguments[0];
            distance = Math.sqrt(x1 * x1 + y1 * y1);
            break;
    }

    return distance;
};
/*
 * Vektor pomocniczy przyjmujący parametry rownania prostej Ax + By + C = 0 jako
 * a,b,c
 */
var Vector3f = function () {
    switch (arguments.length) {
        case 0:
            this.a = 0.0;
            this.b = 0.0;
            this.c = 0.0;
            break;
        case 3:
            this.a = arguments[0];
            this.b = arguments[1];
            this.c = arguments[2];
            break;
    }
};
Vector2f.prototype.toString = function () {
    return "V=[" + this.x + ", " + this.y + "]";
};

var atan2Deg = function (yy, xx) {
    return Math.atan2(yy, xx) * 180.0 / Math.PI;
};
function drawVectorf(x1, y1, x2, y2, axes, fillStyle) {
    ctx.save();
    ctx.beginPath();
    ctx.fillStyle = fillStyle;
    var radius = 0;
    var angle = 0;
    var vect = new Vector2f(x2 - x1, y2 - y1);
    radius = Math.sqrt(Math.pow(vect.y, 2) + Math.pow(vect.x, 2));
    angle = atan2Deg(vect.y, vect.x);
    if (axes === "complex" || axes === "cartesian") {
        drawArrow(x0 + x1 * 30, y0 - y1 * 30, radius * 30, lw, angle, bw, bh,
                cl, fillStyle);
    }
    if (axes === "js") {
        drawArrow(x1 + 15, y1 + 15, radius * 30, lw, -angle, bw, bh, cl,
                fillStyle);
    }
    if (axes === "norm") {
        drawArrow(x1, y1, arrowLength(x1, y1, x2, y2), lw, -arrowAngle(x1, y1,
                x2, y2), bw, bh, cl, fillStyle);
    }
    ctx.restore();
};

/*
 * Tworzymy linię używając dwóch wektorów lub 4 współrzędnych x1,y1,x2,y2,
 * punktów przez które przechodzi linia
 */

var Line = function () {
    switch (arguments.length) {
        case 2:
            if ((arguments[0] instanceof Vector2f)
                    && (arguments[1] instanceof Vector2f)) {
                this.start = arguments[0];
                this.end = arguments[1];
                this.length = this.start.distance(this.end);
                this.cosinus = (this.end.x - this.start.x) / this.length;
                this.sinus = (this.end.y - this.start.y) / this.length;
            }
            break;
        case 4:
            this.start = new Vector2f(arguments[0], arguments[1]);
            this.end = new Vector2f(arguments[2], arguments[3]);
            this.length = this.start.distance(this.end);
            this.cosinus = (arguments[0] - arguments[2]) / this.length;
            this.sinus = (arguments[1] - arguments[3]) / this.length;
            break;
    }
};
// mozna podać:
// 1. axes, fillStyle, Line
// 2. axes, fillStyle, Vector2f, Vector2f
// 3. axes, fillStyle, x1,y1,x2, y2
function drawLine() {
    ctx.save();
    ctx.beginPath();
    var x1 = 0.0;
    var y1 = 0.0;
    var x2 = 0.0;
    var y2 = 0.0;
    var axes = arguments[0];
    ctx.strokeStyle = arguments[1];
    switch (arguments.length) {
        case 3:
            if (arguments[2] instanceof Line) {
                x1 = arguments[2].start.x;
                y1 = arguments[2].start.y;
                x2 = arguments[2].end.x;
                y2 = arguments[2].end.y;
            }
            break;
        case 4:
            if (arguments[2] instanceof Vector2f
                    && arguments[3] instanceof Vector2f) {
                x1 = arguments[2].x;
                y1 = arguments[2].y;
                x2 = arguments[3].x;
                y2 = arguments[3].y;
            }
            break;
        case 6:
            x1 = arguments[2];
            y1 = arguments[3];
            x2 = arguments[4];
            y2 = arguments[5];
            break;
    }
    if (axes === "complex" || axes === "cartesian") {
        ctx.moveTo(x0 + x1 * 30, y0 - y1 * 30);
        ctx.lineTo(x0 + x2 * 30, y0 - y2 * 30);
        ctx.stroke();

    }
    if (axes === "js") {
        ctx.moveTo(15 + x1 * 30, 15 + y1 * 30);
        ctx.lineTo(15 + x2 * 30, 15 + y2 * 30);
        ctx.stroke();

    }
    if (axes === "norm") {
        ctx.moveTo(x1, y1);
        ctx.lineTo(x2, y2);
        ctx.stroke();
    }
    ctx.restore();
}
;
/**
 * Oblicza współczynnik nachylenia tej linii * 
 * @return zwraca współczynnik nachylenia linii
 */
Line.prototype.slope = function () {
    var xx1 = this.start.x - this.end.x;
    var yy1 = this.start.y - this.end.y;
    if (xx1 !== 0.0) {
        return yy1 / xx1;
    } else {
        throw new Error("Nie moge dzielic przez 0");
    }
};

/**
 * Oblicza wskazaną liczbę punktów na tej linii
 *  * @param liczbaPunktow
 * int - liczba obliczanych punktów point1,point 2, wektory 2f
 *      zwraca tablicę Vector2f;
 */
var pointsOnLine = function (nPoints, point1, point2) {
    var tabl = new Array(nPoints);
    var delta = 1.0 / (nPoints - 1.0);
    var point = null;
    var ax = point1.x;
    var ay = point1.y;
    var bx = point2.x;
    var by = point2.y;
    for (var i = 0; i < nPoints; i++) {
        var t = i * delta;
        var tt = 1.0 - t;
        var x = t * bx + tt * ax;
        var y = t * by + tt * ay;
        point = new Vector2f(x, y);
        tabl[i] = point;
    }
    return tabl;
};
var pointsOnLine = function (nPoints, line) {
    var tabl = new Array(nPoints);
    var delta = 1.0 / (nPoints - 1.0);
    var point = null;
    var ax = line.start.x;
    var ay = line.start.y;
    var bx = line.end.x;
    var by = line.end.y;
    for (var i = 0; i < nPoints; i++) {
        var t = i * delta;
        var tt = 1.0 - t;
        var x = t * bx + tt * ax;
        var y = t * by + tt * ay;
        point = new Vector2f(x, y);
        tabl[i] = point;
    }
    return tabl;
};

Line.prototype.toString = function () {
    return "[" + "(" + this.start.x + ", " + this.start.y + ")" + ";" + "("
            + this.end.x + ", " + this.end.y + ")" + "]";
};

/**
 * Oblicza współczynnik nachylenia linii przechodzącej przez dwa dane punkty
 * @param point1 Vector2d - pierwszy punkt
 * @param point2 Vector2d - drugi punkt
 * @return double - zwraca współczynnik nachylenia linii
 */
var slope = function (point1, point2) {
    var xx2 = 0.0;
    var yy2 = 0.0;
    if (point1 instanceof Vector2f && point2 instanceof Vector2f) {
        xx2 = point1.x - point2.x;
        yy2 = point1.y - point2.y;
    }
    if (xx2 !== 0.0) {
        return yy2 / xx2;
    } else {
        throw new Error("linia nie może być równoległa do osi Y!");

    }
};

/**
 * Sprawdza czy dwie wskazane linie przecinają się
 * @param line1 Line - pierwsza linia
 * @param line2 Line - druga linia
 * @return boolean - zwraca <code>true</code> jeśli linie przecinają się albo
 *         <code>false</code> jeśli są równoległe
 */
var areIntersect = function (line1, line2) {
    var m1 = line1.slope();
    var m2 = line2.slope();
    return (m1 !== m2);
};
/*
 * Sprawdza czy linie są prostopadłe
 */
var areNormal = function (line1, line2) {
    var m1 = line1.slope();
    var m2 = line2.slope();
    return m1 * m2 === -1;
};
/**
 * Zwraca współczynniki równania linii w postaci Vectora3d
 * @param line Line - badana linia
 * @return Vector3D - zawierający współczynniki równania liniowego
 *         Vector3d.getX() = a, Vector3d.getY() = b, Vector3d.getZ() = c
 *         otrzymujemy równanie linii AX + BY + C = 0;
 */
var findEquation = function (line) {
    var m = line.slope();
    var A = -m;
    var B = 1;
    var C = m * line.start.x - line.start.y;
    return (new Vector3f(A, B, C));
};
var findb = function (line) {
    var m = line.slope();
    var C = m * line.start.x - line.start.y;
    return -C;
};
/**
 * Oblicza i zwraca punkt przecięcia dwóch linii podanych jako obiekty typu Line
 * @param line1 Line - linia 1
 * @param line2 Line - linia 2
 * @return Vector2d - zwraca punkt przecięcia wskazanych linii lub
 *         <code>null</code> jeśli punkt przecięcia nie istnieje
 */
var interceptPointL = function (line1, line2) {
    var xx = 0;
    var yy = 0;
    if (areIntersect(line1, line2)) {
        var v1 = findEquation(line1);
        var v2 = findEquation(line2);
        xx = (v2.b * v1.c - v1.b * v2.c) / (v2.a * v1.b - v1.a * v2.b);
        yy = (-v2.c - v2.a * xx) / v2.b;
    } else {
        return null;
    }
    return (new Vector2f(xx, yy));
};

/**
 * Tworzy obiekt typu Line z równania linii, jeśli równanie linii wyrażone jest
 * w postaci aX + bY + c;
 * @param a double - wspólczynnik a równania linii;
 * @param b double - współczynnik b równania linii
 * @param c double - współczynnik c równania linii
 * @return Line - obiekt linii wyrażonej równaniem o podanych wspólczynnikach
 */
var findLine = function (a, b, c) {
    var x1 = 0;
    var x2 = 3;
    var y1 = -c / b - a / b * x1;
    var y2 = -c / b - a / b * x2;
    return new Line(new Vector2f(x1, y1), new Vector2f(x2, y2));
};

/**
 * Oblicza i zwraca punkt przecięcia się dwóch linii podanych w postaci Vectora.
 * Jeżeli równanie linii jest wyrażone jako aX + bY + c = 0, to parametrem
 * równania będzie new Vector(a,b,c)
 * @param v1 Vector3d - wektor 1
 * @param v2 Vector3d - wektor 2
 * @return Vector2d - zwraca punkt przecięcia, jeśli istnieje lub
 * <code>null</code> jeśli nie istnieje
 */
var interceptPointE = function (v1, v2) {
    var xx = 0;
    var yy = 0;
    var line1 = findLine(v1.a, v1.b, v1.c);
    var line2 = findLine(v2.a, v2.b, v2.c);
    if (areIntersect(line1, line2)) {
        xx = (v2.b * v1.c - v1.b * v2.c) / (v2.a * v1.b - v1.a * v2.b);
        yy = (-v2.c - v2.a * xx) / v2.b;
    } else {
        return null;
    }
    return (new Vector2f(xx, yy));
};
/*
 * Zwraca odleglość pomiedzy punktami określonymi albo jako wektory swobodne
 * albo wspolrzedne x1,x2,y1,y2
 */
var distance = function () {
    var distance = 0.0;
    switch (arguments.length) {
        case 2:
            if (arguments[0] instanceof Vector2f && arguments[1] instanceof Vector2f) {
                var x = arguments[0].x - arguments[1].x;
                var y = arguments[0].y - arguments[1].y;
                distance = Math.sqrt(x * x + y * y);
            }
            break;
        case 4:
            var x1 = arguments[0] - arguments[2];
            var y1 = arguments[1] - arguments[3];
            distance = Math.sqrt(x1 * x1 + y1 * y1);
            break;
    }
    return distance;
};
var distance2 = function (line1, line2) {
    var v1 = findEquation(line1);
    var v2 = findEquation(line2);
    var dist = Math.abs(v2.c - v1.c) / Math.sqrt(v1.a * v1.a + v1.b * v1.b);
    return dist;
};
var distance3 = function (line, point) {
    var v = null;
    var d = 0.0;
    if ((line instanceof Line) && (point instanceof Vector2f)) {
        v = findEquation(line);
        d = (v.a * point.x + v.b * point.y + v.c) / (Math.sqrt(v.a * v.a + v.b * v.b));
    }
    return d;
};
// Zwraca kąt w stopniach
var angleBetween = function (line1, line2) {
    var m1 = line1.slope();
    var m2 = line2.slope();
    var angle = (m2 - m1) / (1 + m1 * m2);
    return atanDeg(angle);
};
var radToDeg = function (rad) {
    return rad * 180.0 / Math.PI;
};
var atanDeg = function (ratio) {
    return Math.atan(ratio) * 180.0 / Math.PI;

};
var lineThruPoint = function (line, point) {
    var line1 = null;
    if ((line instanceof Line) && (point instanceof Vector2f)) {
        var v3f = findEquation(line);
        line1 = findLine(v3f.b, -v3f.a, v3f.a * point.y - v3f.b * point.x);
    }
    return line1;
};
var lineParallelThruPoint = function (line, point) {
    var line1 = null;
    if ((line instanceof Line) && (point instanceof Vector2f)) {
        var v3f = findEquation(line);
        line1 = findLine(v3f.b, v3f.a, v3f.a * point.y, v3f.b * point.x);
    }
    return line1;
};
