<?php
/**
 * class.mailing.php
 * 
 * <p>Includes {@link mailing} class.</p>
 * 
 * @package MaMML
 * @subpackage classes
 * @copyright  2004 Schmalls / Joshua Thompson / All Rights Reserved
 * @license http://www.gnu.org/copyleft/gpl.html GNU/GPL
 * @author Schmalls / Joshua Thompson <schmalls@gmail.com>
 * @version 0.5.0
 * @since 0.4.3
 * @link http://www.schmalls.com
 */
 
 /**
  *  Ensures this file is being included by a parent file
  */
 defined('_VALID_MOS') or die('Direct Access to this location is not allowed.');
 
 /**
  * Require the sef component
  */
 if (!function_exists('sefRelToAbs')) {
	 if (file_exists($mosConfig_absolute_path . '/components/com_sef/sef.php')) {
		 require_once($mosConfig_absolute_path . '/components/com_sef/sef.php');
	 } else {
		 // set sef to off temporarily to fix problems with sef
		 $sef_temp = $mosConfig_sef;
		 $mosConfig_sef = false;
		 require_once($mosConfig_absolute_path . '/includes/sef.php');
		 // restore sef
		 $mosConfig_sef = $sef_temp;
	 } // end if
 } // end if
 
 /**
  * Class with mailing functions
  * 
  * <p>Class for both frontend and backend mailing functions.</p>
  * 
  * @package MaMML
  * @author Schmalls / Joshua Thompson <schmalls@gmail.com>
  * @version 0.5.0
  * @since 0.4.3
  */
 class mailing {
	 
	 /**
	  * Gets layout and html information for the newsletter
	  * 
	  * <p>Gets the layout and html information for a newsletter from the 
	  * database and returns as an object.</p>
	  * 
	  * @version 0.4.3
	  * @since 0.4.3
	  * @param int $listid the list to get information for
	  * @global class access to the database
	  * @return object data for newlsetter layout and html options
	  */
	 function getListData($listid) {
		global $database;
		
	 	$query = 'SELECT layout, html FROM #__mamml_letters WHERE id = ' . $listid;
		$database->setQuery($query);
		$database->loadObject($list_data);
		
		return $list_data;
	 } // end function
	 
	 /**
	  * Gets letter data
	  * 
	  * <p>Gets the html content, subject, attachments, and text content for 
	  * a mailing from the database and returns as an object.</p>
	  * 
	  * @version 0.4.7
	  * @since 0.4.3
	  * @param int $listid the list to get information for
	  * @global class access to the database
	  * @return object information for unsent mailing for the newsletter
	  */
	 function getLetterData($listid) {
		global $database;
		
		$list = array();
		$query = 'SELECT list_content, list_subject, attachments, list_textonly FROM #__mamml_mailing WHERE list_id = ' . $listid . ' AND published = 0';
		$database->setQuery($query);
		$database->loadObject($list);
		
		if (!empty($list->attachments)) {
			$list->attachments = explode("\n", $list->attachments);
			array_pop($list->attachments);
		} // end if
		
		return $list;
	 } // end function
	 
	 /**
	  * Gets content items
	  * 
	  * <p>Gets all of the content items from the database with id, title, 
	  * section, and category information.</p>
	  * 
	  * @version 0.4.3
	  * @since 0.4.3
	  * @global class access to the database
	  * @returns array information about content items from the site
	  */
 	 function getContentItems() {
		global $database;
		
		$query = "SELECT a.id as id, a.title as title, b.title as section, c.title as category FROM #__content as a, #__sections as b, #__categories as c WHERE a.sectionid = b.id AND a.catid = c.id";
		$database->setQuery($query);
		$contentitems = $database->loadObjectList();
		
		return $contentitems;
	 } // end function
	 
	 /**
	  * Gets a specific content item
	  * 
	  * <p>Gets the content item's title, intro, and fulltext from the 
	  * database.</p>
	  * 
	  * @version 0.4.9
	  * @since 0.4.3
	  * @param int $id the content item's id
	  * @global class access to the database
	  * @return array information about the content item
	  */
	 function getContentItem($id) {
		global $database;
		
		$query = "SELECT a.title as title, a.introtext as intro, a.fulltext as full, a.images as images FROM #__content as a WHERE id = $id";
		$database->setQuery($query);
		$contentitem = $database->loadRow();
		
		return $contentitem;
	 }
	 
	 /**
	  * Gets the image array for a content item
	  * 
	  * <p>Creates the image array for a content item with image location 
	  * and attribute information.</p>
	  * 
	  * @version 0.4.3
	  * @since 0.4.3
	  * @param string $images string from image part of content item
	  * @return array image location and property information
	  */
	 function getImageArray($images) {
		 
		// split around newline
		$first = explode("\n",$images);
		
		// then split each line around |'s to get image properties
		for($i=0,$n=count($first);$i < $n; $i++) {
			$second = explode('|',$first[$i]);
			$third[$i]['image'] = $second[0];
			$third[$i]['align'] = $second[1];
			$third[$i]['alttext'] = $second[2];
			$third[$i]['border'] = $second[3];			
		} // end for
		
		return $third;
	 } // end function
	 
	 /**
	  * Replaces tags in the mailing
	  * 
	  * <p>Replaces the following:
	  * <ul>
	  *	<li>[CONFIRM] = This creates a clickable link where the subscriber can confirm their subscription. This is required to make MaMML work properly.</li>
	  *	<li>[NAME] = This will be replaced by the name the subscriber entered, you'll be sending personalized email when using this.</li>
	  *	<li>[SUBSCRIPTIONS] = This shows current subscriptions for the subscriber and a link to change their subscriptions.</li>
	  * </ul>
	  * </p>
	  * 
	  * @version 0.5.0
	  * @since 0.4.3
	  * @param string $content the content for the mailing
	  * @param array $subscriber the subscriber's information
	  * @param int $listid the newsletter's id
	  * @param int $subscriberid the subscriber's id
	  * @param int $html tells whether this is an html mailing or not
	  * @param string $senddate the date the mailing was sent
	  * @global string filesytem path to main directory
	  * @global string url to main site directory
	  * @global array configuration information
	  * @global class access to the database
	  * @global int Itemid to help with pathway
	  * @return string the updated content for the mailing
	  */
	 function replaceTags($content, $subscriber, $listid, $subscriberid, $html, $senddate = ''){
	  	global $mosConfig_absolute_path, $mosConfig_live_site, $newsletterConfig, $database, $Itemid;
		
		// get subscriber information
		$query = 'SELECT M.name AS name, M.email AS email, N.* FROM `#__mamml_subscribers` AS N, `#__users` AS M WHERE M.id = N.subscriber_id AND N.subscriber_id = ' . $subscriberid;
		$database->setQuery($query);
		$database->loadObject($row);
		echo $database->getErrorMsg();
		// check to see if subscriber is registered or not and change subscriptionslink accordingly
		if ($row->email != $subscriber['subscriber_email']) {
			$query = 'SELECT * FROM `#__mamml_unregistered` WHERE `unregistered_id` = ' . $subscriberid;
			$database->setQuery($query);
			$database->loadObject($row);
			echo $database->getErrorMsg();
			$subscriptionslink = sefRelToAbs($mosConfig_live_site . '/index.php?option=com_mamml&Itemid=' . $Itemid . '&action=unregistered&type=unregistered&userid=' . $subscriberid . '&test=' . md5($row->email));
			$registered = false;
		} else {
			$subscriptionslink = sefRelToAbs($mosConfig_live_site . '/index.php?option=com_mamml&Itemid=' . $Itemid . '&action=registered&type=registered&userid=' . $subscriberid . '&test=' . md5($row->email));
			$registered = true;
		} // end if
		
		// create unregistered confirmation link
		$confirmlink = sefRelToAbs($mosConfig_live_site . '/index.php?option=com_mamml&Itemid=' . $Itemid. '&action=unregistered&type=unregistered&task=confirm&test=' . md5($row->email) . '&userid=' . $subscriberid);
	    
	    if ($html) {
	    	$confirmlink = '<a href="' . $confirmlink . '" target="_blank">' . _CONFIRMLINKTEKST . '</a>';
	    } else {
			$confirmlink = _CONFIRMLINKTEKST . "\n" . $confirmlink;
	    } // end if
		
		// get the letter ids for inclusion in current subscriptions text
		$query = 'SELECT `list_name`, `id` FROM `#__mamml_letters` ORDER BY `id`';
		$database->setQuery($query);
		$letters = $database->loadObjectList();
		$subscriptionstext = '';
		foreach ($letters as $letter) {
			$list = 'list_' . $letter->id;
			if ($row->$list != 0) {
				$subscriptionstext .= '[' . $letter->list_name . '] ';
			} // end if
		} // end foreach
		/*
		for ($i = $j, $n = count($row); $i < $n; $i++) {
			$letter = $letters[$i-$j];	
			if($row[$i] != 0) {
				$subscriptionstext = $subscriptionstext . ' [' . $letter->list_name . ']';
			} // end if
		} // end for
		*/
		// change to html or text specific versions for subscriptions text
		if($html) {
			$subscriptionslink = '<a href="' . $subscriptionslink . '" target="_blank">' . _CHANGESUBSCRIPTIONS . '</a>';
			$subscriptionstext = '<p>' . _SUBSCRIPTIONS . ':<br />' . $subscriptionstext . '</p><p>' . $subscriptionslink . '</p>';
		} else {
			$subscriptionslink = _CHANGESUBSCRIPTIONS . ': ' . $subscriptionslink;
			$subscriptionstext = "\r\n" . _SUBSCRIPTIONS . ': ' . $subscriptionstext . "\r\n\r\n" . $subscriptionslink;
		} // end if
		
		// finally do the tag replacing
	    $content = str_replace('[CONFIRM]', $confirmlink, $content);
	    $content = str_replace('[NAME]', $subscriber['subscriber_name'], $content);
		$content = str_replace('[SUBSCRIPTIONS]', $subscriptionstext, $content);
	    
		// create images to check html read statistics 
	    if (!empty($senddate)) {
			if ($newsletterConfig['enable_statistics'] == '1') {
				if ($newsletterConfig['statistics_per_subscriber'] == '1') {
      				//add image with subscriberid
      				if($html) {
						// check if user is registered or unregistered
						if ($registered) {
							// if registered
        					$content = $content . '<img src="' . sefRelToAbs($mosConfig_live_site . '/index2.php?option=com_mamml&Itemid=' . $Itemid . '&listid=' . $listid . '&action=log&senddate=' . $senddate . '&subscriber=' . $subscriberid . '&type=registered') . '" border="0" width="1" height="1" />';
						} else {
							// if unregistered
							$content = $content . '<img src="' . sefRelToAbs($mosConfig_live_site . '/index2.php?option=com_mamml&Itemid=' . $Itemid . '&listid=' . $listid . '&action=log&senddate=' . $senddate . '&subscriber=' . $subscriberid . '&type=unregistered') . '" border="0" width="1" height="1" />';
						} // end if
					} // end if
    			} else {
      				//add image without subscriberid
      				if ($html) {
        				$content = $content . '<img src="' . sefRelToAbs($mosConfig_live_site . '/index2.php?option=com_mamml&Itemid=' . $Itemid . '&listid=' . $listid . '&action=log&send=' . $senddate) . '" border="0" width="1" height="1" />';
      				} // end if
    			} // end if
  			} // end if
	    } // end if
	    
		if(!$html) {
			$content = str_replace('<br />', "\r\n", $content);
			$content = strip_tags($content);
		} // end if
		return $content;
	 } // end function
 
	 /**
	  * Saves the mailing to the database
	  * 
	  * <p>Appends the content item placeholders to the mailing and creates 
	  * the attachment string and either updates or adds to the database.</p>
	  * 
	  * @version 0.4.8
	  * @since 0.4.3
	  * @global class access to the database
	  * @return boolean to tell whether it was successful
	  */
	 function save(){
		global $database;
	    
		$listid = mosGetParam($_REQUEST, 'listid', 0);
		$subject = mosGetParam($_REQUEST, 'subject', '');
		$content = mosGetParam($_REQUEST, 'content', '', _MOS_ALLOWHTML);
		$html = mosGetParam($_REQUEST, 'html', 0);
		$alt_content = mosGetParam($_REQUEST, 'alt_content', '', _MOS_NOTRIM);
		$new_list = mosGetParam($_REQUEST, 'new_list', 0);
	    // get attachments
	    $attachments = mosGetParam($_REQUEST, 'attachments', 0);
		$attach = '';
	    if($attachments != 0){
			foreach ($attachments as $attachment) {
				$attach .= $attachment . "\n";
			} // end foreach
		} // end if

		// attach content item placeholders to end of message
		$contentitems = mosGetParam($_REQUEST, 'contentitems', 0);
		if($contentitems != 0) {
			$introonly = mosGetParam($_REQUEST, 'introonly', 0);
			foreach ($contentitems as $contentitem) {
				if($introonly != 0) {
					$insert = '{contentitem:' . $contentitem . '|0' . '}';
				} else {
					$insert = '{contentitem:' . $contentitem . '|1' . '}';
				} // end if
				if ((substr_count($content, '{contentitem:' . $contentitem . '|0}') == 0) && (substr_count($content, '{contentitem:' . $contentitem . '|1}') == 0)){
					if ($html != 0) {
						$content .= '<br />' . $insert;
						if(strlen($alt_content) > 0) {
							$alt_content .= "\n\r" . $insert;
						} // end if
					} else {
						$content .= "\n\r" . $insert;
					} // end if
				} // end if
			} // end foreach
		} // end if
		
		// if quotes not already escaped, escape them
		if (!get_magic_quotes_gpc()) {
			$subject = addslashes($subject);
			$content = addslashes($content);
			$alt_content = addslashes($alt_content);
		} // end if
		// check if it is a new letter, and update the database
		if($new_list != 0) {
			$query = 'INSERT INTO #__mamml_mailing (list_id, send_date, list_subject, list_content, list_textonly, attachments, published) 
						VALUES(' . $listid . ', NOW(), \'' . $subject . '\', \'' . $content . '\', \'' . $alt_content . '\', \'' . $attach . '\', 0)';
		} else {
			$query = 'UPDATE `#__mamml_mailing` SET 
						list_subject = \'' . $subject . '\', 
						list_content = \'' . $content . '\', 
						list_textonly = \'' . $alt_content . '\', 
						attachments = \'' . $attach . '\'
						WHERE list_id = \'' . $listid . '\' AND published = 0';      
		} // end if
	    $database->setQuery($query);    
	    $database->query();
		$error = $database->getErrorMsg();
		// check for error
		if (!empty($error)) {
			echo '<p>Error (classes/class.mailing.php->save() line ' . __LINE__ . '): Letter not saved to database. Database error: <br />' . $error . '</p>';
			return false;
		} else {
			return true;
		} // end if
	 } // end function
 
	 /**
	  * Publishes the letter
	  * 
	  * <p>Saves the mailing if it is supposed to do so. Then gets the 
	  * mailing information and subscribers from the database and then sends 
	  * the information to the {@link send()} function. After that it 
	  * updates the mailing to sent if it was successful.</p>
	  * 
	  * @version 0.4.13
	  * @since 0.4.3
	  * @param int $listid the newsletter to work with
	  * @param string $fromname the sender's from name
	  * @param string $fromemail the sender's from email
	  * @param string $frombounce the sender's from bounce address
	  * @param boolean $preview tells whether this is a preview mailing
	  * @global class access to the database
	  * @global class access to subscriber functions
	  * @global int Itemid to help with pathway
	  * @uses save()
	  * @uses subscribers::getSubscribers()
	  * @uses send()
	  * @return boolean whether it was successful or not
	  */
	 function publish($listid, $fromname = '', $fromemail = '', $frombounce = '', $preview = false){
		global $database, $subscribers, $Itemid;
		
		$donotsave = mosGetParam($_REQUEST, 'donotsave', 0);
		// if not preview or not specified not to save, save mailing
		if ((!$preview) && ($donotsave == 0)) {
			$this->save();
		} // end if
    	
		// get letter from database
		$query = 'SELECT `list_content`, `list_subject`, `attachments`, `list_textonly`, `send_date` FROM `#__mamml_mailing` WHERE `list_id` = ' . $listid . ' AND `published` = 0';
		$database->setQuery($query);
		$mailing = $database->loadRow();
		$error = $database->getErrorMsg();
		// quit on error
		if (!empty($error)) {
			echo '<p>Error (classes/class.mailing.php->publish() line ' . __LINE__ . '): Could not get letter from database. Database error: <br />' . $error . '</p>';
			return false;
		} else {
			$mailing[2] = explode("\n", $mailing[2]);
			array_pop($mailing[2]);
			// get list details
			$query = 'SELECT * FROM `#__mamml_letters` WHERE id = ' . $listid;
			$database->setQuery($query);
			$list = $database->loadRow();
			$error = $database->getErrorMsg();
			// quit on error
			if (!empty($error)) {
				echo '<p>Error (classes/class.mailing.php->publish() line ' . __LINE__ . '): Could not get list details from database. Database error: <br />' . $error . '</p>';
				return false;
			} else {
				// if not preview get subscribers, else get preview details to send	
				if(!$preview) {
					$to = array();
					$receivers = $subscribers->getSubscribers($listid, "AND confirmed != 0");
					// quit if no subscribers
					if (count($receivers) == 0) {
						echo '<p>Error (classes/class.mailing.php->publish() line ' . __LINE__ . '): No subscribers for this list found in the database.</p>';
						return false;	
					} else {
						foreach($receivers AS $receiver) {
							$to[] = array('id' => $receiver->id, 'subscriber_email' => $receiver->email, 'subscriber_name' => $receiver->name, 'receive_html' => $receiver->receive_html);
						} // end foreach
						// quit if no subscribers
						if (empty($to)) {
							echo '<p>Error (classes/class.mailing.php->publish() line ' . __LINE__ . '): Error putting subscribers into outgoing array.</p>';
							return false;	
						} // end if
					} // end if
				} else{
					$to[] = 'preview';
					$to[] = array('id' => 0, 'subscriber_email' => $_POST['emailaddress'], 'subscriber_name' => $_POST['name'], 'receive_html' => $_POST['html']);
				} // end if
				// if no change in params return default
				if($fromname == '') {
					$fromname = $list[3];
				} // end if
				if($fromemail == '') {
					$fromemail = $list[4];
				} // end if
				if($frombounce == '') {
					$frombounce = $list[5];
				} // end if
				
				$check = $this->send($mailing[1], $mailing[0], $mailing[3], $to, $fromname, $fromemail, $frombounce, $list[9], $list[6], $mailing[2], $list[0], $Itemid, $mailing[4]);
				// if not preview
				if(!$preview){
					// if messages sent
  					if($check){
						// set message as sent
  						$query = 'UPDATE `#__mamml_mailing` SET `published` = 1 WHERE `list_id` = ' . $listid;
  						$database->setQuery($query);
  						$database->query();
						$error = $database->getErrorMsg();
						// quit on error
						if (!empty($error)) {
							echo '<p>Error (classes/class.mailing.php->publish() line ' . __LINE__ . '): Error setting letter to published. Database error: ' . $error . '</p>';
							return false;
						} else {
							return true;	
						} // end if
  					} else { 
						echo '<p>Error (classes/class.mailing.php->publish() line ' . __LINE__ . '): Error sending messages.</p>';
						return false;
  					} // end if
				} else {
					return $check;
				} // end if
			} // end if
		} // end if
	 } // end function
	
	 /**
	  * Sends the mailing
	  * 
	  * <p>Sets up the PHPMailer class with the correct information. Then it 
	  * sends a message to each subscriber individually after replacing the 
	  * tags with custom information. Then it updates the statistics 
	  * information in the database. Also, outputs a list of the subscribers 
	  * and whether the mailing was sent to them successfully or not. It 
	  * also provides the email throttling options.</p>
	  * 
	  * @version 0.5.0 
	  * @since 0.4.3
	  * @param string $subject the mailing's subject
	  * @param string $content the mailing's content
	  * @param string $textonly the mailing's text-only content
	  * @param array $to subscriber information
	  * @param string $fromname the mailing's from name
	  * @param string $fromemail the mailing's from email
	  * @param string $frombounce the mailing's from bounce
	  * @param int $html tells whether this is an html mailing
	  * @param string $layout the mailing's layout information
	  * @param array $attachments the mailings attachments
	  * @param int $listid the newsletter to work with
	  * @param int $menuitem Itemid to help with pathway
	  * @param string $senddate the date the mailing was sent
	  * @global array configuration options
	  * @global string url to the site's main directory
	  * @global string filesystem path to the site's main directory
	  * @global class access to the database
	  * @global string md5 of email address
	  * @global string the user type
	  * @global int the user's id
	  * @uses PHPMailer
	  * @uses PHPMailer::$PluginDir
	  * @uses PHPMailer::SetLanguage()
	  * @uses PHPMailer::CharSet
	  * @uses PHPMailer::$From
	  * @uses PHPMailer::$FromName
	  * @uses PHPMailer::AddReplyTo()
	  * @uses PHPMailer::$Subject
	  * @uses PHPMailer::$Mailer
	  * @uses PHPMailer::$Sendmail
	  * @uses PHPMailer::IsSMTP()
	  * @uses PHPMailer::$Host
	  * @uses PHPMailer::$SMTPAuth
	  * @uses PHPMailer::$Password
	  * @uses PHPMailer::$Username
	  * @uses PHPMailer::AddAttachment()
	  * @uses getContentItem()
	  * @uses PHPMailer::AddEmbeddedImage()
	  * @uses PHPMailer::$Body
	  * @uses PHPMailer::$AltBody
	  * @uses PHPMailer::AddAddress()
	  * @uses PHPMailer::ClearAddresses()
	  * @uses sendReport()
	  * @return boolean tell whether it was successful 
	  */
	 function send($subject, $content, $textonly, $to, $fromname, $fromemail, $frombounce, $html, $layout = '', $attachments = array(), $listid = 0, $menuitem = 0, $send_date = '') {
		global $newsletterConfig, $mosConfig_live_site, $mosConfig_absolute_path, $database, $test, $type, $userid;
	  
		// Check for safe mode
		if (ini_get('safe_mode')) {
			// Do it the safe mode way
		} else {
			// set time limit to configured number of minutes
			set_time_limit(60 * $newsletterConfig['script_timeout']);
		}
	  
		// if layout set up, replace content marker with content 
		if (!empty($layout)) {
			$content = str_replace('[CONTENT]', $content, $layout);
		} // end if 

		// initialize phpmailer
		$mail = new PHPMailer();
		// set plugin directory
		$mail->PluginDir = $mosConfig_absolute_path . '/administrator/components/com_mamml/classes/';
		// set language to english
		$mail->SetLanguage("en", $mosConfig_absolute_path . '/components/com_mamml/languages/');
		// set character encoding to mambo defualt
		$mail->CharSet = substr_replace(_ISO, '', 0, 8);
	  
		// set from parts
		$mail->From = '' . $fromemail . '';
		$mail->FromName = '' . $fromname . '';
		$mail->AddReplyTo($frombounce);
	  
		// set send method
		switch ($newsletterConfig['emailmethod']){
			case 'mail' :
				$mail->Mailer = 'mail';
				break;
			case 'sendmail':
				$mail->Mailer = 'sendmail';
				$mail->Sendmail = $newsletterConfig['sendmail_path'];
				break;
			case 'smtp':
				$mail->Mailer = 'smtp';
				$mail->IsSMTP();
				$mail->Host = $newsletterConfig['smtp_host'];
        		// set smtp authorization if required
				if((boolean)$newsletterConfig['smtp_auth_required']){
					$mail->SMTPAuth = $newsletterConfig['smtp_auth_required'];
					$mail->Password = $newsletterConfig['smtp_password'];
					$mail->Username = $newsletterConfig['smtp_username'];
				} // end if
				break;
		} // end switch
	  
		// add attachments
		if (sizeof($attachments) > 0) {
			foreach ($attachments AS $attachment) {
				// check if path needs to be added
				if (stristr($mosConfig_absolute_path, $attachment) === false) {
					$mail->AddAttachment($mosConfig_absolute_path . '/images/stories/' . $attachment);
				} else {
					$mail->AddAttachment($attachment);					
				} // end if
			} // end foreach
		} // end if
		
		// change contentitem placeholders to contentitems
		$contentitems = array();
		preg_match_all('/\{contentitem:(.{1,5})\|(.{1})}/', $content, $contentitems, PREG_SET_ORDER);
		foreach ($contentitems as $contentitem) {
			$contentid = $contentitem[1];
			$thecontentitem = $this->getContentItem($contentid);
			if ($contentitem[2] == '1') {
				$content = str_replace($contentitem[0], '<br /><strong>' . $thecontentitem[0] . '</strong><br />' . $thecontentitem[1] . '<br />' . $thecontentitem[2], $content);
				$textonly = str_replace($contentitem[0], "\n\r" . $thecontentitem[0] . "\n\r" . $thecontentitem[1] . "\n\r" . $thecontentitem[2], $textonly);
			} else {
				$content = str_replace($contentitem[0], '<br /><strong>' . $thecontentitem[0] . '</strong><br />' . $thecontentitem[1] . '<br /><a href="' . sefRelToAbs($mosConfig_live_site . '/index.php?option=com_content&task=view&id=' . $contentid) . '">' . _READ_MORE . '</a>', $content);
				$textonly = str_replace($contentitem[0], "\n\r" . $thecontentitem[0] . "\n\r" . $thecontentitem[1] . "\n\r" . _READ_MORE . ' ' . sefRelToAbs($mosConfig_live_site . '/index.php?option=com_content&task=view&id=' . $contentid), $textonly);
			} // end if
			$images = $this->getImageArray($thecontentitem[3]);
			foreach($images as $image) {
				$imagestring = '<img src="/images/stories/' . $image['image'] . '" align="' . $image['align'] . '" alt="' . $image['alttext'] . '" border="' . $image['border'] . '" />';
				$content = preg_replace('/{mosimage}/', $imagestring, $content, 1);
			} // end foreach
		} // end foreach
		
		// change img to inline if option is true, otherwise change to correct paths
		preg_match_all('/<img([^>]*)src="([^">]+)"([^>]*)>/', $content, $images, PREG_SET_ORDER);
		foreach($images as $image) {
			$image[2] = preg_replace('/(\.\.\/)+/', '/', $image[2]);
			$image[2] = str_replace($mosConfig_live_site, '', $image[2]);
			$imagepath = $mosConfig_absolute_path . $image[2];
			$split = explode('/', $image[2]);
			$filename = $split[count($split)-1];
			$split2 = explode('.', $filename);
			$cid = $split2[0];
			if(($newsletterConfig['use_embedded_images'] != 0) && (stristr($image[2], 'http://') === false)) {
				$mail->AddEmbeddedImage($imagepath, $cid, $filename);
				$replacement = '<img' . $image[1] . 'src="cid:' . $cid . '"' . $image[3] . '>';
			} else {
				if (stristr($image[2], 'http://') === false) {
					// remove unneeded directory information
					if (stristr($image[2], '/images/stories/') !== false) {
						$image[2] = stristr($image[2], '/images/stories/');
					} // end if
					$replacement = '<img' . $image[1] . 'src="' . $mosConfig_live_site . $image[2] . '"' . $image[3] . '>';
				} else {
					$replacement = '<img' . $image[1] . 'src="' . $image[2] . '"' . $image[3] . '>';
				} // end if
			} // end if
			$content = str_replace($image[0], $replacement, $content);
		} // end foreach
		
		// if it is not a preview, subscribe, or unsubscribe message
		if(($to[0] != "preview") && ($to[0] != "subscribe") && ($to[0] != "unsubscribe")){
			// set starting time
			$mtime = microtime();
			$mtime = explode(" ",$mtime);
			$mtime = $mtime[1] + $mtime[0];
			$starttime = $mtime;
      
			// initialize counting variables
			$html_sent = 0;
			$text_sent = 0;
			$size = sizeof($to);
			$i = 1;
			// ouput toolbar cancel button handling form
?>
<form action="index2.php" method="post" name="adminForm">
	<input type="hidden" name="option" value="com_mamml" />
	<input type="hidden" name="act" value="mailing" />
	<input type="hidden" name="task" value="" />
</form>
<?php
			// create counter
			print '
					<form name="counterForm">
					' . _SENDINGEMAIL . ': &nbsp;<input type="text" name="teller" value="0" style="border: 0px solid white; font-family: Arial, Helvetica, sans-serif; font-size: 11px;" size="1" /> of ' . $size . '
					</form>';
			// make textonly equal to html if empty
			if(!(strlen($textonly) > 0)) {
				$textonly = $content;
				// fix line breaks
				$textonly = str_replace('<br />', "\r\n", $textonly);
				// fix links
				$textonly = preg_replace('/<a href="([^"]*)"[^>]*>([^<]*)<\/a>/','${2} ${1}', $textonly);
				// remove all <head> information
				$textonly = preg_replace('/<head>.*<\/head>/i', '', $textonly);
			} // end if
			// add subscriptions placeholder to content
			$content .= '<p>[SUBSCRIPTIONS]</p>';
			$textonly .= "\r\n \r\n[SUBSCRIPTIONS]";
		
			// check if we should flush
			$flush = mosGetParam($_REQUEST, 'flush', 0);
			if ($flush != 0) {
				ob_end_flush();
			} // end if
			
			// send to each subscriber
			$skip_subscribers = mosGetParam($_REQUEST, 'skip_subscribers', 0);
			foreach ($to AS $receiver){
				// check if this subscriber should be skipped
				if ($i <= $skip_subscribers) {
					$i++;
					continue;
				} // end if
				// if receive html, otherwise send text
				if($html && (intval($receiver['receive_html']) == 1)){
					$mail->IsHTML(true);
					$ashtml = 1;
					$letterBody = $this->replaceTags($content, $receiver, $listid, $receiver['id'], $ashtml, $send_date);
					$mail->AltBody = $this->replaceTags($textonly, $receiver, $listid, $receiver['id'], 0, $send_date);
					$html_sent++;
				} else{
					$mail->IsHTML(false);
					$mail->AltBody = ''; 
					$ashtml = 0;
					$letterBody = $this->replaceTags($textonly, $receiver, $listid, $receiver['id'], $ashtml, $send_date);
					$text_sent++;
				} // end if

				// set body, receiver, and subject
				$mail->Body = $letterBody;
				$mail->AddAddress($receiver['subscriber_email'], $receiver['subscriber_name']);
				$mail->Subject = str_replace('[NAME]', $receiver['subscriber_name'], $subject);

				// output information
				echo $i . ' ';
				// report error or success
				if (!$mail->Send()) {
					echo '<br />' . _MESSAGENOT . '. <p>';
					echo _MAILERERROR . ': ' . $mail->ErrorInfo . '</p>';
					if($html && (intval($receiver['receive_html']) == 1)){
						$html_sent--;
					} else{
						$text_sent--;
					}
				} else{
					echo '<b>' . $receiver['subscriber_email'] . '</b> -> ' . _MESSAGESENTSUCCESSFULLY . '<br />';
					// update statistics for subscriber
					if ($newsletterConfig['enable_statistics'] == '1') {
						if ($newsletterConfig['statistics_per_subscriber'] == '1') {
							// check if subscriber is registered or unregistered
							$check = strpos($letterBody, 'type=unregistered');
							if ($check === false) {
								// if registered
								$query = 'INSERT INTO `#__mamml_stats_detailed` (`listid`, `send_date`, `subscriber_id`, `html`) VALUES (\'' . $listid . '\', \'' . $send_date . '\', \'' . $receiver['id'] . '\', \'' . $ashtml . '\');';
							} else {
								// if unregistered
								$query = 'INSERT INTO `#__mamml_stats_detailed_unregistered` (`listid`, `send_date`, `subscriber_id`, `html`) VALUES (\'' . $listid . '\', \'' . $send_date . '\', \'' . $receiver['id'] . '\', \'' . $ashtml . '\');';
							} // end if
							$database->setQuery($query);
							$database->query();
						} // end if
					} // end if
				} // end if
				// remove address
				$mail->ClearAddresses();
				// increment number of mails sent
				echo '<script language="javascript">document.counterForm.teller.value=\'' . $i . '\';</script>';
				// flush
				flush();
				// if sent out a multiple of the configured amount of emails, take a break
				if (($i % $newsletterConfig['emails_between_pauses']) == 0) {
					// check if we should sleep or wait for user input
					if (($newsletterConfig['wait_for_user_at_pause'] == 0) || ($flush == 0)) {
						// if sleep, sleep
						echo "--- Waiting " . $newsletterConfig['pause_time']. " seconds ---<br /><br />";
						// flush
						flush();
						// sleep
						sleep($newsletterConfig['pause_time']);
					} else {
						// if wait, wait
?>
<form action="index2.php" method="post">
	<input type="hidden" name="option" value="com_mamml" />
	<input type="hidden" name="act" value="mailing" />
	<input type="hidden" name="task" value="publish" />
	<input type="hidden" name="listid" value="<?php echo $listid; ?>" />
	<input type="hidden" name="skip_subscribers" value="<?php echo $i; ?>" />
	<input type="hidden" name="fromname" value="<?php echo $fromname; ?>" />
	<input type="hidden" name="fromemail" value="<?php echo $fromemail; ?>" />
	<input type="hidden" name="return" value="<?php echo $frombounce; ?>" />
	<input type="hidden" name="donotsave" value="1" />
	<input type="hidden" name="flush" value="1" />
	<input type="submit" name="submit" value="<?php echo _CONTINUE_SENDING; ?>" />
</form>
<?php
						exit;
					} // end if
				} // end if
				$i++;
			} // end foreach
			// update statistics for newsletter
			if($newsletterConfig['enable_statistics'] == '1'){
				$query = 'INSERT INTO `#__mamml_stats_overall` (`listid`, `send_date`, `html_sent`, `text_sent`) 
							VALUES (\'' . $listid . '\', \'' . $send_date . '\', \'' . $html_sent . '\', \'' . $text_sent . '\');';
				$database->setQuery($query);
				$database->query();
			} // end if
		    // set endtime and calculate total time
			$mtime = microtime();
			$mtime = explode(" ",$mtime);
			$mtime = $mtime[1] + $mtime[0];
			$endtime = $mtime;
			$totaltime = number_format($endtime - $starttime, 4, ',', '');
	      
			// send mailing report
			if($newsletterConfig['send_data'] == 1){
				$this->sendReport($fromemail, $totaltime, $html_sent, $text_sent);
			} // end if
	      
			// display statistics
			echo '<br /><b>' . _SENDINGTOOK . ' ' . $totaltime . ' ' . _SECONDS . '</b><br />';
	
			return true;
		} else {
			// check if address was entered
			if(!empty($to)){
				// make textonly equal to content if empty
				if(!(strlen($textonly) > 0)) {
					$textonly = $content;
					// fix line breaks
					$textonly = str_replace('<br />', "\r\n", $textonly);
					// fix links
					$textonly = preg_replace('/<a href="([^"]*)"[^>]*>([^<]*)<\/a>/','${2} ${1}', $textonly);
					// remove all <head> information
					$textonly = preg_replace('/<head>.*<\/head>/i', '', $textonly);
				} // end if
				// if html otherwise textonly
				if($html && (intval($to[1]['receive_html']) == 1)) {
					$mail->IsHTML(true);
					$ashtml = 1;
					$letterBody = $this->replaceTags($content, $to[1], $listid, $to[1]['id'], $ashtml, $send_date);
					$textonly = strip_tags($textonly);
					$mail->AltBody = $this->replaceTags($textonly, $to[1], $listid, $to[1]['id'], 0, $send_date);
				} else{
					$mail->IsHTML(false); 
					$ashtml = 0;
					$textonly = strip_tags($textonly);
					$letterBody = $this->replaceTags($textonly, $to[1], $listid, $to[1]['id'], $ashtml, $send_date);
				} // end if
		  	  
				// add address, body, and subject
				$mail->AddAddress($to[1]['subscriber_email'], $to[1]['subscriber_name']);
				$mail->Body = $letterBody;
				$mail->Subject = str_replace('[NAME]', $to[1]['subscriber_name'], $subject);
		  	  
				// check if mail was sent
				if(!$mail->Send()){
					echo _MESSAGENOT . " <p>";
					echo _MAILERERROR . ": " . $mail->ErrorInfo;
					return false;
				} else {
					return true;
				} // end if
			} else {
				echo _NOADDRESSENTERED;
				return false;
			} // end if
		} // end if	  
	 } // end function
	 /**
	  * Send a report to the developer
	  * 
	  * <p>Sends a report of the mailing to 
	  * {@link mailto:schmalls@gmail.com schmalls@gmail.com} along with 
	  * server and configuration options.</p>
	  * 
	  * @version 0.4.15
	  * @since 0.4.3
	  * @param string $from_email the return email address
	  * @param int $time the time it took to send
	  * @param int $html_sent the number of html mailings sent
	  * @param int $text_sent the number of text mailing sent
	  */
	 function sendReport($from_email, $time, $html_sent, $text_sent) {
		 global $newsletterConfig;
		 
		 $safemode = (ini_get("safe_mode") == 0) ? 'off' : 'on';
		 $numberSubscribers = $text_sent + $html_sent;
		 
		 $content = 'MaMML send mailing report of ' . date("d-m-Y H:i") . "\n\n"; 
		 $content .= "-----------------------------------\n";
		 $content .= "Server: \n";
		 $content .= "-----------------------------------\n\n";
		 $content .= "Software: " . $_SERVER['SERVER_SOFTWARE'] . "\n";
		 $content .= "PHP Version: " . phpversion() . "\n";
		 $content .= "Zend Version: " . zend_version() . "\n";
		 $content .= "Magic_quotes_gpc: " . ini_get("magic_quotes_gpc") . "\n";
		 $content .= "Disable_functions: " . ini_get("disable_functions") . "\n";
		 $content .= "Max_execution_time: " . ini_get("max_execution_time") . "\n";
		 $content .= "Safe_mode: " . $safemode . "\n";
		 $content .= "Memory_limit: " . ini_get("memory_limit") . "\n";
		 $content .= "\n\n";
		 $content .= "-----------------------------------\n";
		 $content .= "MaMML configuration: \n";
		 $content .= "-----------------------------------\n\n";
		 $content .= "Send method: " . $newsletterConfig['emailmethod'] . "\n";
		 if($newsletterConfig['emailmethod'] == 'smtp'){
			 $auth = ($newsletterConfig['smtp_auth_required'] == 1) ? 'yes' : 'no';
			 $content .= "Authentication required: " . $auth . "\n";
		 }
		 $content .= "MaMML Version: 0.5.0\n"; 
		 $content .= "\n\n";
		 $content .= "-----------------------------------\n";
		 $content .= "Mailing options: \n";
		 $content .= "-----------------------------------\n\n";
		 $content .= "Time to send: " . $time . "\n";
		 $content .= "Number of subscribers: " . $numberSubscribers . "\n";
		 $content .= "HTML format: " . $html_sent . "\n";
		 $content .= "Text format: " . $text_sent . "\n";
		 mail("schmalls@gmail.com", "MaMML mailing report", $content, "From: " . $from_email . "");
	 } // end function
	  
 } // end class
?>
