/*
 * Copyright 2020 Wuyi Chen.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kafkadefinitiveguide.consumer.exitpollloop;

import java.time.Duration;
import java.util.Collections;
import java.util.Properties;

import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Przykład wyjścia z pętli odpytywania przez wywołanie wakeup() w zaczepie ShutdownHook
 *
 * @author  Wuyi Chen
 * @date    06/05/2020
 * @version 1.0
 * @since   1.0
 */
public class ExitPollLoopExample {
	private static Logger logger = LoggerFactory.getLogger(ExitPollLoopExample.class);
	
	public static void main(String[] args) {
		Properties props = new Properties();
		props.put("bootstrap.servers",  "localhost:9092");
		props.put("group.id",           "CountryCounter");                                                   // Określamy grupę konsumentów, do której należy instancja klasy KafkaConsumer
		props.put("key.deserializer",   "org.apache.kafka.common.serialization.StringDeserializer");
		props.put("value.deserializer", "org.apache.kafka.common.serialization.StringDeserializer");

		KafkaConsumer<String, String> consumer = new KafkaConsumer<>(props);
		
		final Thread mainThread = Thread.currentThread();

        // Registering a shutdown hook so we can exit cleanly
        Runtime.getRuntime().addShutdownHook(new Thread() {                                                  // ShutdownHook jest uruchamiany w osobnym wątku
            public void run() {
                System.out.println("Rozpoczęcie procesu wyjścia...");

                consumer.wakeup();                                                                           // Można bezpiecznie wywołać wakeup() w kolejnym wątku                                                                    
                try {
                    mainThread.join();
                } catch (InterruptedException e) {
                    e.printStackTrace();
                }
            }
        });
		
		try {
			consumer.subscribe(Collections.singletonList("customerCountries"));                              // Określamy listę tematów subskrybowanych przez instancję klasy KafkaConsumer
			
		    while (true) { 
		        ConsumerRecords<String, String> records = consumer.poll(Duration.ofMillis(100));             // Interwał limitu czasu, który kontroluje, jak długo poll() będzie blokować wykonywanie, jeśli dane nie będą dostępne w buforze konsumenta
		        for (ConsumerRecord<String, String> record : records) {
		            logger.debug("temat = {}, partycja = {}, przesunięcie = {}, klient = {}, kraj = {}",
		                record.topic(), record.partition(), record.offset(), record.key(), record.value());

		        }
		    }
		} finally {
		    consumer.close();                                                                                // Zamykamy konsument, co uruchamia natychmiast równoważenie obciążenia
		}
	}
}
