WczytPUMSWszystko
================
Win-Vector LLC
21.04.2018r.

Dane pobrane 21.04.2018r. z następujących stron:

-   <https://www.census.gov/data/developers/data-sets/acs-1year.2016.html>
-   <https://www.census.gov/programs-surveys/acs/technical-documentation/pums.html>
-   <http://www2.census.gov/programs-surveys/acs/tech_docs/pums/data_dict/PUMSDataDict16.txt>
-   <https://www2.census.gov/programs-surveys/acs/data/pums/2016/1-Year/>

Najpierw wykonaj w powłoce `bash` następujące czynności:

    wget https://www2.census.gov/programs-surveys/acs/data/pums/2016/1-Year/csv_hus.zip
    md5 csv_hus.zip
    # MD5 (csv_hus.zip) = c81d4b96a95d573c1b10fc7f230d5f7a
    wget https://www2.census.gov/programs-surveys/acs/data/pums/2016/1-Year/csv_pus.zip
    md5 csv_pus.zip
    # MD5 (csv_pus.zip) = 06142320c3865620b0630d74d74181db
    wget http://www2.census.gov/programs-surveys/acs/tech_docs/pums/data_dict/PUMSDataDict16.txt
    md5 PUMSDataDict16.txt
    # MD5 (PUMSDataDict16.txt) = 56b4e8fcc7596cc8b69c9e878f2e699aunzip csv_hus.zip
    unzip csv_pus.zip
    gzip -9 ss16*.csv

W ten sposób uzyskujemy skompresowane pliki rozdzielane przecinkami, z następującymi nagłówkami:

-   `ss16husa.csv.gz`
-   `ss16husb.csv.gz`
-   `ss16pusa.csv.gz`
-   `ss16pusb.csv.gz`

Wczytajmy je do bazy danych PostreSQL.

``` r
library("readr")
library("rquery")
library("wrapr")

data_dict <- readRDS("data_dict.RDS")

# nawiązuje połączenie z bazą danych
db <- DBI::dbConnect(RPostgreSQL::PostgreSQL(),
                        host = 'localhost',
                        port = 5432,
                        user = 'johnmount',
                        password = '')
rquery_opts <- dbi_connection_tests(db)
options(rquery_opts)

tabs <- c("ss16husa", "ss16husb",
          "ss16pusa", "ss16pusb") := 
  c("hus", "hus",
    "pus", "pus")

for(nam in c(names(tabs), unique(tabs))) {
  dbi_remove_table(db, nam)
}

tables <- list()

# Wczytuje dane do bazy danych
for(nam in names(tabs)) {
  print(base::date())
  print(nam)
  tab_group <- tabs[[nam]]
  fnam <- paste0(nam, ".csv.gz")

  dat <- read_csv(fnam,
                  col_types = cols(.default = "c"))
  ti <- rquery::dbi_copy_to(db, nam, dat, 
                            overwrite = TRUE, temporary = TRUE)
  tables[[nam]] <- ti
  rm(list = "dat")
}
```

    ## [1] "Sat Jul 14 08:34:52 2018"
    ## [1] "ss16husa"
    ## [1] "Sat Jul 14 08:35:41 2018"
    ## [1] "ss16husb"
    ## [1] "Sat Jul 14 08:36:24 2018"
    ## [1] "ss16pusa"
    ## [1] "Sat Jul 14 08:39:41 2018"
    ## [1] "ss16pusb"

``` r
print(base::date())
```

    ## [1] "Sat Jul 14 08:42:55 2018"

``` r
hd <- data_dict[data_dict$table=="hus", , drop = FALSE]
hm <- data.frame(column_name = hd$field_name, 
                 old_value = hd$field_keys, 
                 new_value = hd$field_values,
                 stringsAsFactors = FALSE)
hq <- unionall(list(tables[["ss16husa"]],
                     tables[["ss16husb"]])) %.>%
  map_column_values(., hm)
cat(format(hq))
```

    ## unionall(.,
    ##   table('"ss16husa"'; 
    ##   RT,
    ##   SERIALNO,
    ##   ST,
    ##   DIVISION,
    ##   PUMA,
    ##   REGION,
    ##   ADJHSG,
    ##   ADJINC,
    ##   WGTP,
    ##   NP,
    ##   TYPE,
    ##   ACCESS,
    ##   ACR,
    ##   AGS,
    ##   BATH,
    ##   BDSP,
    ##   BLD,
    ##   BROADBND,
    ##   COMPOTHX,
    ##   CONP,
    ##   ...),
    ##  table('"ss16husb"'; 
    ##   RT,
    ##   SERIALNO,
    ##   ST,
    ##   DIVISION,
    ##   PUMA,
    ##   REGION,
    ##   ADJHSG,
    ##   ADJINC,
    ##   WGTP,
    ##   NP,
    ##   TYPE,
    ##   ACCESS,
    ##   ACR,
    ##   AGS,
    ##   BATH,
    ##   BDSP,
    ##   BLD,
    ##   BROADBND,
    ##   COMPOTHX,
    ##   CONP,
    ##   ...)) %.>%
    ##  sql_node(.,
    ##           map_column_values(., hm))

``` r
hq %.>%
  op_diagram(.) %.>%
  DiagrammeR::grViz(.) %.>%
  print(.)
materialize(db, hq, "hus")
```

    ## [1] "table('\"hus\"'; RT, SERIALNO, ST, DIVISION, PUMA, REGION, ADJHSG, ADJINC, WGTP, NP, TYPE, ACCESS, ACR, AGS, BATH, BDSP, BLD, BROADBND, COMPOTHX, CONP, ...)"

``` r
pd <- data_dict[data_dict$table=="pus", , drop = FALSE]
pm <- data.frame(column_name = pd$field_name, 
                 old_value = pd$field_keys, 
                 new_value = pd$field_values,
                 stringsAsFactors = FALSE)
pq <- unionall(list(tables[["ss16pusa"]], 
                     tables[["ss16pusb"]])) %.>%
   map_column_values(., pm)
cat(format(pq))
```

    ## unionall(.,
    ##   table('"ss16pusa"'; 
    ##   RT,
    ##   SERIALNO,
    ##   SPORDER,
    ##   PUMA,
    ##   ST,
    ##   ADJINC,
    ##   PWGTP,
    ##   AGEP,
    ##   CIT,
    ##   CITWP,
    ##   COW,
    ##   DDRS,
    ##   DEAR,
    ##   DEYE,
    ##   DOUT,
    ##   DPHY,
    ##   DRAT,
    ##   DRATX,
    ##   DREM,
    ##   ENG,
    ##   ...),
    ##  table('"ss16pusb"'; 
    ##   RT,
    ##   SERIALNO,
    ##   SPORDER,
    ##   PUMA,
    ##   ST,
    ##   ADJINC,
    ##   PWGTP,
    ##   AGEP,
    ##   CIT,
    ##   CITWP,
    ##   COW,
    ##   DDRS,
    ##   DEAR,
    ##   DEYE,
    ##   DOUT,
    ##   DPHY,
    ##   DRAT,
    ##   DRATX,
    ##   DREM,
    ##   ENG,
    ##   ...)) %.>%
    ##  sql_node(.,
    ##           map_column_values(., pm))

``` r
pq %.>%
  op_diagram(.) %.>%
  DiagrammeR::grViz(.) %.>%
  print(.)
materialize(db, pq, "pus")
```

    ## [1] "table('\"pus\"'; RT, SERIALNO, SPORDER, PUMA, ST, ADJINC, PWGTP, AGEP, CIT, CITWP, COW, DDRS, DEAR, DEYE, DOUT, DPHY, DRAT, DRATX, DREM, ENG, ...)"

``` r
for(nam in names(tabs)) {
  dbi_remove_table(db, nam)
}

cdata::qlook(db, "hus")
```

    ## table "hus" PostgreSQLConnection 
    ##  nrow: 1505086 
    ##  NOTE: "obs" below is count of sample, not number of rows of data.
    ## 'data.frame':    10 obs. of  230 variables:
    ##  $ RT         : chr  "Housing Record or Group Quarters Unit" "Housing Record or Group Quarters Unit" "Housing Record or Group Quarters Unit" "Housing Record or Group Quarters Unit" ...
    ##  $ SERIALNO   : chr  "000000064" "000000080" "000000107" "000000134" ...
    ##  $ ST         : chr  "Alabama/AL" "Alabama/AL" "Alabama/AL" "Alabama/AL" ...
    ##  $ DIVISION   : chr  "East South Central (South region)" "East South Central (South region)" "East South Central (South region)" "East South Central (South region)" ...
    ##  $ PUMA       : chr  "00700" "00900" "00302" "00600" ...
    ##  $ REGION     : chr  "South" "South" "South" "South" ...
    ##  $ ADJHSG     : chr  "2016 factor (1.000000)" "2016 factor (1.000000)" "2016 factor (1.000000)" "2016 factor (1.000000)" ...
    ##  $ ADJINC     : chr  "2016 factor(1.007588)" "2016 factor(1.007588)" "2016 factor(1.007588)" "2016 factor(1.007588)" ...
    ##  $ WGTP       : chr  "00097" "00095" "00159" "00122" ...
    ##  $ NP         : chr  "02" "One person record (one person in household or any person in group quarters)" "04" "02" ...
    ##  $ TYPE       : chr  "Housing unit" "Housing unit" "Housing unit" "Housing unit" ...
    ##  $ ACCESS     : chr  "Yes, by paying a cell phone company or Internet service provider" "No access to the Internet at this house, apartment, or mobile home" "Yes, by paying a cell phone company or Internet service provider" "Yes, by paying a cell phone company or Internet service provider" ...
    ##  $ ACR        : chr  "House on less than one acre" "House on one to less than ten acres" "House on less than one acre" "House on less than one acre" ...
    ##  $ AGS        : chr  NA "None" NA NA ...
    ##  $ BATH       : chr  "Yes" "Yes" "Yes" "Yes" ...
    ##  $ BDSP       : chr  "03" "03" "04" "03" ...
    ##  $ BLD        : chr  "One-family house detached" "Mobile home or trailer" "One-family house detached" "One-family house detached" ...
    ##  $ BROADBND   : chr  "No" NA "Yes" "Yes" ...
    ##  $ COMPOTHX   : chr  "No" "No" "No" "No" ...
    ##  $ CONP       : chr  "Not condo" "Not condo" "Not condo" "Not condo" ...
    ##  $ DIALUP     : chr  "No" NA "No" "No" ...
    ##  $ ELEP       : chr  "160" "390" "130" "260" ...
    ##  $ FS         : chr  "No" "No" "No" "No" ...
    ##  $ FULP       : chr  "No charge or these fuels not used" "No charge or these fuels not used" "No charge or these fuels not used" "No charge or these fuels not used" ...
    ##  $ GASP       : chr  "030" "No charge or gas not used" "No charge or gas not used" "No charge or gas not used" ...
    ##  $ HFL        : chr  "Utility gas" "Electricity" "Electricity" "Electricity" ...
    ##  $ HISPEED    : chr  "Yes" NA "Yes" "Yes" ...
    ##  $ INSP       : chr  "1100" "0000" "1200" "1100" ...
    ##  $ LAPTOP     : chr  "Yes" "No" "Yes" "Yes" ...
    ##  $ MHP        : chr  NA "00350" NA NA ...
    ##  $ MRGI       : chr  NA "No, insurance paid separately or no insurance" "Yes, insurance included in payment" NA ...
    ##  $ MRGP       : chr  NA "00180" "00720" NA ...
    ##  $ MRGT       : chr  NA "No, taxes paid separately or taxes not required" "Yes, taxes included in payment" NA ...
    ##  $ MRGX       : chr  "None" "Mortgage, deed of trust, or similar debt" "Mortgage, deed of trust, or similar debt" "None" ...
    ##  $ OTHSVCEX   : chr  "No" NA "No" "No" ...
    ##  $ REFR       : chr  "Yes" "Yes" "Yes" "Yes" ...
    ##  $ RMSP       : chr  "07" "04" "08" "07" ...
    ##  $ RNTM       : chr  NA NA NA NA ...
    ##  $ RNTP       : chr  NA NA NA NA ...
    ##  $ RWAT       : chr  "Yes" "Yes" "Yes" "Yes" ...
    ##  $ RWATPR     : chr  "Case is from the United States, RWATPR not applicable" "Case is from the United States, RWATPR not applicable" "Case is from the United States, RWATPR not applicable" "Case is from the United States, RWATPR not applicable" ...
    ##  $ SATELLITE  : chr  "No" NA "No" "No" ...
    ##  $ SINK       : chr  "Yes" "Yes" "Yes" "Yes" ...
    ##  $ SMARTPHONE : chr  "No" "No" "Yes" "Yes" ...
    ##  $ SMP        : chr  NA NA NA NA ...
    ##  $ STOV       : chr  "Yes" "Yes" "Yes" "Yes" ...
    ##  $ TABLET     : chr  "No" "No" "Yes" "Yes" ...
    ##  $ TEL        : chr  "Yes" "Yes" "Yes" "Yes" ...
    ##  $ TEN        : chr  "Owned free and clear" "Owned with mortgage or loan (include home equity loans)" "Owned with mortgage or loan (include home equity loans)" "Owned free and clear" ...
    ##  $ VACS       : chr  NA NA NA NA ...
    ##  $ VALP       : chr  "0250000" "0010000" "0200000" "0150000" ...
    ##  $ VEH        : chr  "2 vehicles" "1 vehicle" "4 vehicles" "2 vehicles" ...
    ##  $ WATP       : chr  "0800" "0480" "0650" "0590" ...
    ##  $ YBL        : chr  "1950 to 1959" "1990 to 1999" "1960 to 1969" "2005" ...
    ##  $ FES        : chr  "Married-couple family: Neither husband nor wife in LF" NA "Married-couple family: Husband in labor force, wife.not in LF" "Married-couple family: Neither husband nor wife in LF" ...
    ##  $ FFINCP     : chr  "Yes" "No" "No" "No" ...
    ##  $ FGRNTP     : chr  NA NA NA NA ...
    ##  $ FHINCP     : chr  "Yes" "No" "No" "No" ...
    ##  $ FINCP      : chr  "000036610" NA "000065000" "000060900" ...
    ##  $ FPARC      : chr  "No related children" NA "With related children 5 to 17 years only" "No related children" ...
    ##  $ FPLMPRP    : chr  "No" "No" "No" "No" ...
    ##  $ FSMOCP     : chr  "No" "Yes" "No" "No" ...
    ##  $ GRNTP      : chr  NA NA NA NA ...
    ##  $ GRPIP      : chr  NA NA NA NA ...
    ##  $ HHL        : chr  "English only" "English only" "English only" "English only" ...
    ##  $ HHLANP     : chr  "English only household" "English only household" "English only household" "English only household" ...
    ##  $ HHT        : chr  "Married couple household" "Nonfamily household:Female householder:Living alone" "Married couple household" "Married couple household" ...
    ##  $ HINCP      : chr  "000036610" "000045800" "000065000" "000060900" ...
    ##  $ HOTWAT     : chr  "Case is from the United States, HOTWAT not applicable" "Case is from the United States, HOTWAT not applicable" "Case is from the United States, HOTWAT not applicable" "Case is from the United States, HOTWAT not applicable" ...
    ##  $ HUGCL      : chr  "Household without grandparent living with grandchildren" "Household without grandparent living with grandchildren" "Household without grandparent living with grandchildren" "Household without grandparent living with grandchildren" ...
    ##  $ HUPAC      : chr  "No children" "No children" "With children 6 to 17 years only" "No children" ...
    ##  $ HUPAOC     : chr  "No own children present" "No own children present" "Presence of own children 6 to 17 years only" "No own children present" ...
    ##  $ HUPARC     : chr  "No related children present" "No related children present" "Presence of related children 6 to 17 years only" "No related children present" ...
    ##  $ KIT        : chr  "Yes, has stove or range, refrigerator, and sink with a faucet" "Yes, has stove or range, refrigerator, and sink with a faucet" "Yes, has stove or range, refrigerator, and sink with a faucet" "Yes, has stove or range, refrigerator, and sink with a faucet" ...
    ##  $ LNGI       : chr  "At least one person in the household 14 and over speaks English only or speaks English 'very well'" "At least one person in the household 14 and over speaks English only or speaks English 'very well'" "At least one person in the household 14 and over speaks English only or speaks English 'very well'" "At least one person in the household 14 and over speaks English only or speaks English 'very well'" ...
    ##  $ MULTG      : chr  "No, not a multigenerational household" "No, not a multigenerational household" "No, not a multigenerational household" "No, not a multigenerational household" ...
    ##  $ MV         : chr  "30 years or more" "20 to 29 years" "12 months or less" "10 to 19 years" ...
    ##  $ NOC        : chr  "No own children" "No own children" "01" "No own children" ...
    ##  $ NPF        : chr  "02" NA "04" "02" ...
    ##  $ NPP        : chr  "Not a grandparent headed household with no parent present" "Not a grandparent headed household with no parent present" "Not a grandparent headed household with no parent present" "Not a grandparent headed household with no parent present" ...
    ##  $ NR         : chr  "None" "None" "None" "None" ...
    ##  $ NRC        : chr  "No related children" "No related children" "01" "No related children" ...
    ##  $ OCPIP      : chr  "014" "017" "017" "008" ...
    ##  $ PARTNER    : chr  "No unmarried partner in household" "No unmarried partner in household" "No unmarried partner in household" "No unmarried partner in household" ...
    ##  $ PLM        : chr  "Yes, has hot and cold running water, and a bathtub or shower" "Yes, has hot and cold running water, and a bathtub or shower" "Yes, has hot and cold running water, and a bathtub or shower" "Yes, has hot and cold running water, and a bathtub or shower" ...
    ##  $ PLMPRP     : chr  "Case<a0>is from the United States, PLMPRP not applicable" "Case<a0>is from the United States, PLMPRP not applicable" "Case<a0>is from the United States, PLMPRP not applicable" "Case<a0>is from the United States, PLMPRP not applicable" ...
    ##  $ PSF        : chr  "No subfamilies" "No subfamilies" "No subfamilies" "No subfamilies" ...
    ##  $ R18        : chr  "No person under 18 in household" "No person under 18 in household" "1 or more persons under 18 in household" "No person under 18 in household" ...
    ##  $ R60        : chr  "2 or more persons 60 and over" "1 person 60 and over" "No person 60 and over" "2 or more persons 60 and over" ...
    ##  $ R65        : chr  "2 or more persons 65 and over" "1 person 65 and over" "No person 65 and over" "2 or more persons 65 and over" ...
    ##  $ RESMODE    : chr  "Internet" "Mail" "CATI/CAPI" "Internet" ...
    ##  $ SMOCP      : chr  "00413" "00645" "00904" "00428" ...
    ##  $ SMX        : chr  NA "No" "No" NA ...
    ##  $ SRNT       : chr  "A single-family home on 10 or more acres." "A single-family home on 10 or more acres." "A single-family home on 10 or more acres." "A single-family home on 10 or more acres." ...
    ##  $ SSMC       : chr  "Households without a same-sex married couple" "Households without a same-sex married couple" "Households without a same-sex married couple" "Households without a same-sex married couple" ...
    ##  $ SVAL       : chr  "A single-family home on less than 10 acres." "A single-family home on 10 or more acres or any other type of building, including mobile homes, with no regard to acreage." "A single-family home on less than 10 acres." "A single-family home on less than 10 acres." ...
    ##  $ TAXP       : chr  "$ 750 - $ 799" "$ 50 - $ 99" "$1000 - $1099" "$ 300 - $ 349" ...
    ##  $ WIF        : chr  "No workers" NA "1 worker" "No workers" ...
    ##  $ WKEXREL    : chr  "Householder did not work; spouse did not work" NA "Householder worked FT; spouse did not work" "Householder did not work; spouse did not work" ...
    ##  $ WORKSTAT   : chr  "Neither husband nor wife in labor force" NA "Husband in labor force and wife not in labor force, husband employed or in Armed Forces" "Neither husband nor wife in labor force" ...
    ##  $ FACCESSP   : chr  "No" "No" "No" "No" ...
    ##  $ FACRP      : chr  "No" "No" "No" "No" ...
    ##  $ FAGSP      : chr  "No" "No" "No" "No" ...
    ##  $ FBATHP     : chr  "No" "No" "No" "No" ...
    ##  $ FBDSP      : chr  "No" "No" "No" "No" ...
    ##  $ FBLDP      : chr  "No" "No" "No" "No" ...
    ##  $ FBROADBNDP : chr  "No" "No" "No" "No" ...
    ##  $ FCOMPOTHXP : chr  "No" "No" "No" "No" ...
    ##  $ FCONP      : chr  "No" "No" "No" "No" ...
    ##  $ FDIALUPP   : chr  "Yes" "No" "No" "No" ...
    ##  $ FELEP      : chr  "No" "No" "No" "No" ...
    ##  $ FFSP       : chr  "Yes" "No" "No" "No" ...
    ##  $ FFULP      : chr  "No" "Yes" "No" "No" ...
    ##  $ FGASP      : chr  "No" "Yes" "No" "No" ...
    ##  $ FHFLP      : chr  "No" "No" "No" "No" ...
    ##  $ FHISPEEDP  : chr  "Yes" "No" "No" "No" ...
    ##  $ FHOTWATP   : chr  "No" "No" "No" "No" ...
    ##  $ FINSP      : chr  "No" "No" "No" "No" ...
    ##  $ FKITP      : chr  "No" "No" "No" "No" ...
    ##  $ FLAPTOPP   : chr  "No" "No" "No" "No" ...
    ##  $ FMHP       : chr  "No" "Yes" "No" "No" ...
    ##  $ FMRGIP     : chr  "No" "No" "No" "No" ...
    ##  $ FMRGP      : chr  "No" "Yes" "No" "No" ...
    ##  $ FMRGTP     : chr  "No" "No" "No" "No" ...
    ##  $ FMRGXP     : chr  "No" "No" "No" "No" ...
    ##  $ FMVP       : chr  "No" "No" "No" "No" ...
    ##  $ FOTHSVCEXP : chr  "Yes" "No" "No" "No" ...
    ##  $ FPLMP      : chr  "No" "No" "No" "No" ...
    ##  $ FREFRP     : chr  "No" "No" "No" "No" ...
    ##  $ FRMSP      : chr  "No" "No" "No" "No" ...
    ##  $ FRNTMP     : chr  "No" "No" "No" "No" ...
    ##  $ FRNTP      : chr  "No" "No" "No" "No" ...
    ##  $ FRWATP     : chr  "No" "No" "No" "No" ...
    ##  $ FRWATPRP   : chr  "No" "No" "No" "No" ...
    ##  $ FSATELLITEP: chr  "Yes" "No" "No" "No" ...
    ##  $ FSINKP     : chr  "No" "No" "No" "No" ...
    ##  $ FSMARTPHONP: chr  "No" "No" "No" "No" ...
    ##  $ FSMP       : chr  "No" "No" "No" "No" ...
    ##  $ FSMXHP     : chr  "No" "No" "No" "No" ...
    ##  $ FSMXSP     : chr  "No" "No" "No" "No" ...
    ##  $ FSTOVP     : chr  "No" "No" "No" "No" ...
    ##  $ FTABLETP   : chr  "No" "No" "No" "No" ...
    ##  $ FTAXP      : chr  "No" "No" "No" "No" ...
    ##  $ FTELP      : chr  "No" "No" "No" "No" ...
    ##  $ FTENP      : chr  "No" "No" "No" "No" ...
    ##  $ FVACSP     : chr  "No" "No" "No" "No" ...
    ##  $ FVALP      : chr  "No" "No" "No" "No" ...
    ##  $ FVEHP      : chr  "No" "No" "No" "No" ...
    ##  $ FWATP      : chr  "No" "No" "No" "No" ...
    ##  $ FYBLP      : chr  "No" "No" "No" "No" ...
    ##  $ wgtp1      : chr  "00091" "00078" "00159" "00120" ...
    ##  $ wgtp2      : chr  "00093" "00030" "00044" "00115" ...
    ##  $ wgtp3      : chr  "00104" "00088" "00210" "00122" ...
    ##  $ wgtp4      : chr  "00030" "00093" "00058" "00032" ...
    ##  $ wgtp5      : chr  "00121" "00026" "00314" "00046" ...
    ##  $ wgtp6      : chr  "00117" "00081" "00249" "00216" ...
    ##  $ wgtp7      : chr  "00166" "00148" "00185" "00038" ...
    ##  $ wgtp8      : chr  "00029" "00029" "00159" "00146" ...
    ##  $ wgtp9      : chr  "00149" "00025" "00166" "00121" ...
    ##  $ wgtp10     : chr  "00162" "00160" "00040" "00241" ...
    ##  $ wgtp11     : chr  "00083" "00095" "00158" "00187" ...
    ##  $ wgtp12     : chr  "00102" "00097" "00243" "00196" ...
    ##  $ wgtp13     : chr  "00032" "00154" "00156" "00135" ...
    ##  $ wgtp14     : chr  "00025" "00127" "00268" "00097" ...
    ##  $ wgtp15     : chr  "00165" "00186" "00285" "00032" ...
    ##  $ wgtp16     : chr  "00105" "00103" "00042" "00286" ...
    ##  $ wgtp17     : chr  "00091" "00111" "00169" "00117" ...
    ##  $ wgtp18     : chr  "00076" "00101" "00057" "00031" ...
    ##  $ wgtp19     : chr  "00137" "00026" "00169" "00135" ...
    ##  $ wgtp20     : chr  "00032" "00029" "00173" "00166" ...
    ##  $ wgtp21     : chr  "00118" "00092" "00146" "00035" ...
    ##  $ wgtp22     : chr  "00111" "00171" "00258" "00196" ...
    ##  $ wgtp23     : chr  "00108" "00113" "00168" "00209" ...
    ##  $ wgtp24     : chr  "00031" "00092" "00242" "00111" ...
    ##  $ wgtp25     : chr  "00100" "00160" "00040" "00120" ...
    ##  $ wgtp26     : chr  "00079" "00099" "00045" "00127" ...
    ##  $ wgtp27     : chr  "00181" "00029" "00173" "00138" ...
    ##  $ wgtp28     : chr  "00029" "00154" "00167" "00037" ...
    ##  $ wgtp29     : chr  "00179" "00144" "00149" "00168" ...
    ##  $ wgtp30     : chr  "00182" "00034" "00251" "00136" ...
    ##  $ wgtp31     : chr  "00103" "00094" "00144" "00123" ...
    ##  $ wgtp32     : chr  "00117" "00107" "00052" "00108" ...
    ##  $ wgtp33     : chr  "00026" "00028" "00172" "00037" ...
    ##  $ wgtp34     : chr  "00026" "00088" "00049" "00222" ...
    ##  $ wgtp35     : chr  "00157" "00034" "00049" "00129" ...
    ##  $ wgtp36     : chr  "00103" "00078" "00277" "00103" ...
    ##  $ wgtp37     : chr  "00105" "00086" "00150" "00037" ...
    ##  $ wgtp38     : chr  "00083" "00090" "00250" "00115" ...
    ##  $ wgtp39     : chr  "00153" "00141" "00155" "00033" ...
    ##  $ wgtp40     : chr  "00031" "00030" "00181" "00124" ...
    ##  $ wgtp41     : chr  "00109" "00070" "00168" "00133" ...
    ##  $ wgtp42     : chr  "00092" "00174" "00055" "00122" ...
    ##  $ wgtp43     : chr  "00108" "00092" "00155" "00136" ...
    ##  $ wgtp44     : chr  "00028" "00099" "00053" "00037" ...
    ##  $ wgtp45     : chr  "00094" "00164" "00276" "00040" ...
    ##  $ wgtp46     : chr  "00081" "00130" "00276" "00254" ...
    ##  $ wgtp47     : chr  "00144" "00035" "00182" "00036" ...
    ##  $ wgtp48     : chr  "00029" "00197" "00153" "00122" ...
    ##  $ wgtp49     : chr  "00153" "00142" "00147" "00123" ...
    ##  $ wgtp50     : chr  "00162" "00026" "00043" "00204" ...
    ##  $ wgtp51     : chr  "00116" "00099" "00167" "00208" ...
    ##  $ wgtp52     : chr  "00105" "00081" "00287" "00201" ...
    ##  $ wgtp53     : chr  "00033" "00027" "00150" "00122" ...
    ##  $ wgtp54     : chr  "00027" "00110" "00253" "00121" ...
    ##  $ wgtp55     : chr  "00183" "00021" "00284" "00031" ...
    ##  $ wgtp56     : chr  "00091" "00104" "00048" "00222" ...
    ##  $ wgtp57     : chr  "00089" "00111" "00162" "00114" ...
    ##  $ wgtp58     : chr  "00110" "00105" "00049" "00035" ...
    ##  $ wgtp59     : chr  "00158" "00170" "00150" "00114" ...
    ##  $ wgtp60     : chr  "00033" "00151" "00160" "00174" ...
    ##  $ wgtp61     : chr  "00109" "00130" "00168" "00029" ...
    ##  $ wgtp62     : chr  "00085" "00026" "00244" "00233" ...
    ##  $ wgtp63     : chr  "00081" "00109" "00150" "00235" ...
    ##  $ wgtp64     : chr  "00031" "00093" "00231" "00102" ...
    ##  $ wgtp65     : chr  "00096" "00029" "00045" "00114" ...
    ##  $ wgtp66     : chr  "00097" "00087" "00050" "00101" ...
    ##  $ wgtp67     : chr  "00134" "00160" "00153" "00127" ...
    ##  $ wgtp68     : chr  "00026" "00031" "00163" "00037" ...
    ##  $ wgtp69     : chr  "00176" "00030" "00130" "00190" ...
    ##  $ wgtp70     : chr  "00211" "00169" "00263" "00139" ...
    ##  $ wgtp71     : chr  "00094" "00118" "00152" "00129" ...
    ##  $ wgtp72     : chr  "00098" "00087" "00050" "00138" ...
    ##  $ wgtp73     : chr  "00027" "00145" "00200" "00034" ...
    ##  $ wgtp74     : chr  "00031" "00081" "00045" "00231" ...
    ##  $ wgtp75     : chr  "00174" "00143" "00045" "00131" ...
    ##  $ wgtp76     : chr  "00093" "00081" "00313" "00122" ...
    ##  $ wgtp77     : chr  "00111" "00093" "00184" "00029" ...
    ##  $ wgtp78     : chr  "00087" "00082" "00258" "00110" ...
    ##  $ wgtp79     : chr  "00168" "00024" "00150" "00041" ...
    ##  $ wgtp80     : chr  "00021" "00209" "00168" "00127" ...

``` r
cdata::qlook(db, "pus")
```

    ## table "pus" PostgreSQLConnection 
    ##  nrow: 3156487 
    ##  NOTE: "obs" below is count of sample, not number of rows of data.
    ## 'data.frame':    10 obs. of  284 variables:
    ##  $ RT       : chr  "Person Record" "Person Record" "Person Record" "Person Record" ...
    ##  $ SERIALNO : chr  "000000064" "000000064" "000000080" "000000107" ...
    ##  $ SPORDER  : chr  "01" "02" "01" "01" ...
    ##  $ PUMA     : chr  "00700" "00700" "00900" "00302" ...
    ##  $ ST       : chr  "Alabama/AL" "Alabama/AL" "Alabama/AL" "Alabama/AL" ...
    ##  $ ADJINC   : chr  "2016 factor (1.007588)" "2016 factor (1.007588)" "2016 factor (1.007588)" "2016 factor (1.007588)" ...
    ##  $ PWGTP    : chr  "00098" "00089" "00095" "00160" ...
    ##  $ AGEP     : chr  "84" "84" "78" "46" ...
    ##  $ CIT      : chr  "Born in the U.S." "Born in the U.S." "Born in the U.S." "Born in the U.S." ...
    ##  $ CITWP    : chr  NA NA NA NA ...
    ##  $ COW      : chr  NA NA "Employee of a private for-profit company or business, or of an individual, for wages,salary, or commissions" "Employee of a private for-profit company or business, or of an individual, for wages,salary, or commissions" ...
    ##  $ DDRS     : chr  "No" "No" "No" "No" ...
    ##  $ DEAR     : chr  "No" "No" "No" "No" ...
    ##  $ DEYE     : chr  "No" "No" "No" "No" ...
    ##  $ DOUT     : chr  "No" "No" "No" "No" ...
    ##  $ DPHY     : chr  "No" "No" "No" "No" ...
    ##  $ DRAT     : chr  NA NA NA NA ...
    ##  $ DRATX    : chr  NA NA NA "No" ...
    ##  $ DREM     : chr  "No" "No" "No" "No" ...
    ##  $ ENG      : chr  NA NA NA NA ...
    ##  $ FER      : chr  NA NA NA NA ...
    ##  $ GCL      : chr  "No" "No" "No" "No" ...
    ##  $ GCM      : chr  NA NA NA NA ...
    ##  $ GCR      : chr  NA NA NA NA ...
    ##  $ HINS1    : chr  "Yes" "Yes" "Yes" "No" ...
    ##  $ HINS2    : chr  "Yes" "No" "No" "Yes" ...
    ##  $ HINS3    : chr  "Yes" "Yes" "Yes" "No" ...
    ##  $ HINS4    : chr  "No" "No" "No" "No" ...
    ##  $ HINS5    : chr  "No" "No" "No" "No" ...
    ##  $ HINS6    : chr  "No" "No" "No" "No" ...
    ##  $ HINS7    : chr  "No" "No" "No" "No" ...
    ##  $ INTP     : chr  "0000110" "0000000" "0000000" "0000000" ...
    ##  $ JWMNP    : chr  NA NA "050" "020" ...
    ##  $ JWRIP    : chr  NA NA "Drove alone" "Drove alone" ...
    ##  $ JWTR     : chr  NA NA "Car, truck, or van" "Car, truck, or van" ...
    ##  $ LANP     : chr  NA NA NA NA ...
    ##  $ LANX     : chr  "No, speaks only English" "No, speaks only English" "No, speaks only English" "No, speaks only English" ...
    ##  $ MAR      : chr  "Married" "Married" "Divorced" "Married" ...
    ##  $ MARHD    : chr  "No" "No" "No" "No" ...
    ##  $ MARHM    : chr  "No" "No" "No" "No" ...
    ##  $ MARHT    : chr  "One time" "One time" "Two times" "One time" ...
    ##  $ MARHW    : chr  "No" "No" "No" "No" ...
    ##  $ MARHYP   : chr  "1953" "1953" "1994" "1994" ...
    ##  $ MIG      : chr  "Yes, same house (nonmovers)" "Yes, same house (nonmovers)" "Yes, same house (nonmovers)" "No, different house in US or Puerto Rico" ...
    ##  $ MIL      : chr  "Never served in the military" "Never served in the military" "Never served in the military" "Only on active duty for training in Reserves/National Guard" ...
    ##  $ MLPA     : chr  NA NA NA NA ...
    ##  $ MLPB     : chr  NA NA NA NA ...
    ##  $ MLPCD    : chr  NA NA NA NA ...
    ##  $ MLPE     : chr  NA NA NA NA ...
    ##  $ MLPFG    : chr  NA NA NA NA ...
    ##  $ MLPH     : chr  NA NA NA NA ...
    ##  $ MLPI     : chr  NA NA NA NA ...
    ##  $ MLPJ     : chr  NA NA NA NA ...
    ##  $ MLPK     : chr  NA NA NA NA ...
    ##  $ NWAB     : chr  "No" "Did not report" "Did not report" "Did not report" ...
    ##  $ NWAV     : chr  "Did not report" "Did not report" "Did not report" "Did not report" ...
    ##  $ NWLA     : chr  "No" "No" "Did not report" "Did not report" ...
    ##  $ NWLK     : chr  "No" "No" "Did not report" "Did not report" ...
    ##  $ NWRE     : chr  "Did not report" "No" "Did not report" "Did not report" ...
    ##  $ OIP      : chr  "004200" "None" "None" "None" ...
    ##  $ PAP      : chr  "None" "None" "None" "None" ...
    ##  $ RELP     : chr  "Reference person" "Husband/wife" "Reference person" "Reference person" ...
    ##  $ RETP     : chr  "010100" "None" "None" "None" ...
    ##  $ SCH      : chr  "No, has not attended in the last 3 months" "No, has not attended in the last 3 months" "No, has not attended in the last 3 months" "No, has not attended in the last 3 months" ...
    ##  $ SCHG     : chr  NA NA NA NA ...
    ##  $ SCHL     : chr  "1 or more years of college credit, no degree" "Bachelor's degree" "1 or more years of college credit, no degree" "Associate's degree" ...
    ##  $ SEMP     : chr  "0000000" "0000000" "0000000" "0000000" ...
    ##  $ SEX      : chr  "Female" "Male" "Female" "Male" ...
    ##  $ SSIP     : chr  "None" "None" "None" "None" ...
    ##  $ SSP      : chr  "08100" "14100" "18500" "None" ...
    ##  $ WAGP     : chr  "0000000" "0000000" "0027300" "0065000" ...
    ##  $ WKHP     : chr  NA NA "40" "40" ...
    ##  $ WKL      : chr  "Over 5 years ago or never worked" "Over 5 years ago or never worked" "Within the past 12 months" "Within the past 12 months" ...
    ##  $ WKW      : chr  NA NA "50 to 52 weeks worked during past 12 months" "50 to 52 weeks worked during past 12 months" ...
    ##  $ WRK      : chr  "Did not work" "Did not work" "Worked" "Worked" ...
    ##  $ YOEP     : chr  NA NA NA NA ...
    ##  $ ANC      : chr  "Single" "Single" "Not reported" "Single" ...
    ##  $ ANC1P    : chr  "American" "American" "Not reported" "American" ...
    ##  $ ANC2P    : chr  "Not reported" "Not reported" "Not reported" "Not reported" ...
    ##  $ DECADE   : chr  NA NA NA NA ...
    ##  $ DIS      : chr  "Without a disability" "Without a disability" "Without a disability" "Without a disability" ...
    ##  $ DRIVESP  : chr  NA NA "1.000 vehicles (Drove alone)" "1.000 vehicles (Drove alone)" ...
    ##  $ ESP      : chr  NA NA NA NA ...
    ##  $ ESR      : chr  "Not in labor force" "Not in labor force" "Civilian employed, at work" "Civilian employed, at work" ...
    ##  $ FHICOVP  : chr  "Yes" "Yes" "Yes" "No" ...
    ##  $ FOD1P    : chr  NA "THEOLOGY AND RELIGIOUS VOCATIONS" NA NA ...
    ##  $ FOD2P    : chr  NA NA NA NA ...
    ##  $ HICOV    : chr  "With health insurance coverage" "With health insurance coverage" "With health insurance coverage" "With health insurance coverage" ...
    ##  $ HISP     : chr  "Not Spanish/Hispanic/Latino" "Not Spanish/Hispanic/Latino" "Not Spanish/Hispanic/Latino" "Not Spanish/Hispanic/Latino" ...
    ##  $ INDP     : chr  NA NA "MED-HOSPITALS" "PRF-ARCHITECTURAL, ENGINEERING, AND RELATED SERVICES" ...
    ##  $ JWAP     : chr  NA NA "7:20 a.m. to 7:24 a.m." "6:50 a.m. to 6:54 a.m." ...
    ##  $ JWDP     : chr  NA NA "6:30 a.m. to 6:34 a.m." "6:30 a.m. to 6:34 a.m." ...
    ##  $ MIGPUMA  : chr  NA NA NA "00290" ...
    ##  $ MIGSP    : chr  NA NA NA "Alabama/AL" ...
    ##  $ MSP      : chr  "Now married, spouse present" "Now married, spouse present" "Divorced" "Now married, spouse present" ...
    ##  $ NAICSP   : chr  NA NA "MED-HOSPITALS" "PRF-ARCHITECTURAL, ENGINEERING, AND RELATED SERVICES" ...
    ##  $ NATIVITY : chr  "Native" "Native" "Native" "Native" ...
    ##  $ NOP      : chr  NA NA NA NA ...
    ##  $ OC       : chr  "No (includes GQ)" "No (includes GQ)" "No (includes GQ)" "No (includes GQ)" ...
    ##  $ OCCP     : chr  NA NA "OFF-SECRETARIES AND ADMINISTRATIVE ASSISTANTS" "ENG-ENGINEERING TECHNICIANS, EXCEPT DRAFTERS" ...
    ##  $ PAOC     : chr  "Females with no own children" NA "Females with no own children" NA ...
    ##  $ PERNP    : chr  "000000000" "000000000" "000027300" "000065000" ...
    ##  $ PINCP    : chr  "000022510" "000014100" "000045800" "000065000" ...
    ##  $ POBP     : chr  "Alabama/AL" "Alabama/AL" "Georgia/GA" "Minnesota/MN" ...
    ##  $ POVPIP   : chr  "254" "254" "402" "261" ...
    ##  $ POWPUMA  : chr  NA NA "00900" "00290" ...
    ##  $ POWSP    : chr  NA NA "Alabama/AL" "Alabama/AL" ...
    ##  $ PRIVCOV  : chr  "With private health insurance coverage" "With private health insurance coverage" "With private health insurance coverage" "With private health insurance coverage" ...
    ##  $ PUBCOV   : chr  "With public health coverage" "With public health coverage" "With public health coverage" "Without public health coverage" ...
    ##  $ QTRBIR   : chr  "January through March" "January through March" "October through December" "October through December" ...
    ##  $ RAC1P    : chr  "White alone" "White alone" "White alone" "White alone" ...
    ##  $ RAC2P    : chr  "White alone" "White alone" "White alone" "White alone" ...
    ##  $ RAC3P    : chr  "White alone" "White alone" "White alone" "White alone" ...
    ##  $ RACAIAN  : chr  "No" "No" "No" "No" ...
    ##  $ RACASN   : chr  "No" "No" "No" "No" ...
    ##  $ RACBLK   : chr  "No" "No" "No" "No" ...
    ##  $ RACNH    : chr  "No" "No" "No" "No" ...
    ##  $ RACNUM   : chr  "1" "1" "1" "1" ...
    ##  $ RACPI    : chr  "No" "No" "No" "No" ...
    ##  $ RACSOR   : chr  "No" "No" "No" "No" ...
    ##  $ RACWHT   : chr  "Yes" "Yes" "Yes" "Yes" ...
    ##  $ RC       : chr  "No (includes GQ)" "No (includes GQ)" "No (includes GQ)" "No (includes GQ)" ...
    ##  $ SCIENGP  : chr  NA "No" NA NA ...
    ##  $ SCIENGRLP: chr  NA "No" NA NA ...
    ##  $ SFN      : chr  NA NA NA NA ...
    ##  $ SFR      : chr  NA NA NA NA ...
    ##  $ SOCP     : chr  NA NA "OFF-SECRETARIES AND ADMINISTRATIVE ASSISTANTS" "ENG-ENGINEERING TECHNICIANS, EXCEPT DRAFTERS" ...
    ##  $ VPS      : chr  NA NA NA NA ...
    ##  $ WAOB     : chr  "US state (POB = 001-059)" "US state (POB = 001-059)" "US state (POB = 001-059)" "US state (POB = 001-059)" ...
    ##  $ FAGEP    : chr  "No" "No" "No" "No" ...
    ##  $ FANCP    : chr  "No" "No" "No" "No" ...
    ##  $ FCITP    : chr  "No" "No" "No" "No" ...
    ##  $ FCITWP   : chr  "No" "No" "No" "No" ...
    ##  $ FCOWP    : chr  "No" "No" "No" "No" ...
    ##  $ FDDRSP   : chr  "No" "No" "No" "No" ...
    ##  $ FDEARP   : chr  "No" "No" "No" "No" ...
    ##  $ FDEYEP   : chr  "No" "No" "No" "No" ...
    ##  $ FDISP    : chr  "No" "No" "No" "No" ...
    ##  $ FDOUTP   : chr  "No" "No" "No" "No" ...
    ##  $ FDPHYP   : chr  "No" "No" "No" "No" ...
    ##  $ FDRATP   : chr  "No" "No" "No" "No" ...
    ##  $ FDRATXP  : chr  "No" "No" "No" "No" ...
    ##  $ FDREMP   : chr  "No" "No" "No" "No" ...
    ##  $ FENGP    : chr  "No" "No" "No" "No" ...
    ##  $ FESRP    : chr  "No" "No" "No" "No" ...
    ##  $ FFERP    : chr  "No" "No" "No" "No" ...
    ##  $ FFODP    : chr  "No" "No" "No" "No" ...
    ##  $ FGCLP    : chr  "No" "No" "No" "No" ...
    ##  $ FGCMP    : chr  "No" "No" "No" "No" ...
    ##  $ FGCRP    : chr  "No" "No" "No" "No" ...
    ##  $ FHINS1P  : chr  "No" "No" "No" "No" ...
    ##  $ FHINS2P  : chr  "Yes" "Yes" "No" "No" ...
    ##  $ FHINS3C  : chr  "No" "No" "No" NA ...
    ##  $ FHINS3P  : chr  "No" "No" "No" "No" ...
    ##  $ FHINS4C  : chr  NA NA NA NA ...
    ##  $ FHINS4P  : chr  "Yes" "Yes" "Yes" "No" ...
    ##  $ FHINS5C  : chr  NA NA NA NA ...
    ##  $ FHINS5P  : chr  "Yes" "Yes" "Yes" "No" ...
    ##  $ FHINS6P  : chr  "Yes" "Yes" "Yes" "No" ...
    ##  $ FHINS7P  : chr  "Yes" "Yes" "Yes" "No" ...
    ##  $ FHISP    : chr  "No" "No" "No" "No" ...
    ##  $ FINDP    : chr  "No" "No" "No" "No" ...
    ##  $ FINTP    : chr  "Yes" "No" "No" "No" ...
    ##  $ FJWDP    : chr  "No" "No" "No" "No" ...
    ##  $ FJWMNP   : chr  "No" "No" "No" "No" ...
    ##  $ FJWRIP   : chr  "No" "No" "No" "No" ...
    ##  $ FJWTRP   : chr  "No" "No" "No" "No" ...
    ##  $ FLANP    : chr  "No" "No" "No" "No" ...
    ##  $ FLANXP   : chr  "No" "No" "No" "No" ...
    ##  $ FMARHDP  : chr  "No" "No" "No" "No" ...
    ##  $ FMARHMP  : chr  "No" "No" "No" "No" ...
    ##  $ FMARHTP  : chr  "No" "No" "No" "No" ...
    ##  $ FMARHWP  : chr  "No" "No" "No" "No" ...
    ##  $ FMARHYP  : chr  "No" "No" "No" "No" ...
    ##  $ FMARP    : chr  "No" "No" "No" "No" ...
    ##  $ FMIGP    : chr  "No" "No" "Yes" "No" ...
    ##  $ FMIGSP   : chr  "No" "No" "No" "No" ...
    ##  $ FMILPP   : chr  "No" "No" "No" "No" ...
    ##  $ FMILSP   : chr  "No" "No" "No" "No" ...
    ##  $ FOCCP    : chr  "No" "No" "No" "No" ...
    ##  $ FOIP     : chr  "No" "No" "No" "No" ...
    ##  $ FPAP     : chr  "No" "No" "No" "No" ...
    ##  $ FPERNP   : chr  "No" "No" "No" "No" ...
    ##  $ FPINCP   : chr  "Yes" "No" "No" "No" ...
    ##  $ FPOBP    : chr  "No" "No" "No" "No" ...
    ##  $ FPOWSP   : chr  "No" "No" "No" "No" ...
    ##  $ FPRIVCOVP: chr  "Yes" "Yes" "Yes" "No" ...
    ##  $ FPUBCOVP : chr  "Yes" "Yes" "Yes" "No" ...
    ##  $ FRACP    : chr  "No" "No" "No" "No" ...
    ##  $ FRELP    : chr  "No" "No" "No" "No" ...
    ##  $ FRETP    : chr  "No" "No" "No" "No" ...
    ##  $ FSCHGP   : chr  "No" "No" "No" "No" ...
    ##  $ FSCHLP   : chr  "No" "No" "No" "No" ...
    ##  $ FSCHP    : chr  "No" "No" "No" "No" ...
    ##  $ FSEMP    : chr  "No" "No" "No" "No" ...
    ##  $ FSEXP    : chr  "No" "No" "No" "No" ...
    ##  $ FSSIP    : chr  "No" "No" "No" "No" ...
    ##  $ FSSP     : chr  "No" "No" "No" "No" ...
    ##  $ FWAGP    : chr  "No" "No" "No" "No" ...
    ##  $ FWKHP    : chr  "No" "No" "No" "No" ...
    ##  $ FWKLP    : chr  "No" "No" "No" "No" ...
    ##  $ FWKWP    : chr  "No" "No" "No" "No" ...
    ##  $ FWRKP    : chr  "No" "No" "No" "No" ...
    ##  $ FYOEP    : chr  "No" "No" "No" "No" ...
    ##  $ pwgtp1   : chr  "00091" "00103" "00077" "00159" ...
    ##  $ pwgtp2   : chr  "00092" "00101" "00029" "00044" ...
    ##  $ pwgtp3   : chr  "00105" "00088" "00088" "00210" ...
    ##  $ pwgtp4   : chr  "00030" "00029" "00093" "00058" ...
    ##  $ pwgtp5   : chr  "00121" "00097" "00026" "00313" ...
    ##  $ pwgtp6   : chr  "00117" "00098" "00080" "00250" ...
    ##  $ pwgtp7   : chr  "00167" "00151" "00148" "00185" ...
    ##  $ pwgtp8   : chr  "00028" "00030" "00029" "00159" ...
    ##  $ pwgtp9   : chr  "00149" "00185" "00025" "00166" ...
    ##  $ pwgtp10  : chr  "00162" "00131" "00159" "00039" ...
    ##  $ pwgtp11  : chr  "00084" "00073" "00095" "00158" ...
    ##  $ pwgtp12  : chr  "00102" "00100" "00097" "00243" ...
    ##  $ pwgtp13  : chr  "00032" "00029" "00154" "00155" ...
    ##  $ pwgtp14  : chr  "00025" "00022" "00128" "00268" ...
    ##  $ pwgtp15  : chr  "00165" "00125" "00186" "00285" ...
    ##  $ pwgtp16  : chr  "00106" "00080" "00103" "00042" ...
    ##  $ pwgtp17  : chr  "00091" "00078" "00112" "00169" ...
    ##  $ pwgtp18  : chr  "00076" "00073" "00101" "00057" ...
    ##  $ pwgtp19  : chr  "00137" "00152" "00026" "00169" ...
    ##  $ pwgtp20  : chr  "00033" "00036" "00029" "00172" ...
    ##  $ pwgtp21  : chr  "00117" "00092" "00091" "00146" ...
    ##  $ pwgtp22  : chr  "00110" "00101" "00171" "00258" ...
    ##  $ pwgtp23  : chr  "00109" "00110" "00113" "00169" ...
    ##  $ pwgtp24  : chr  "00032" "00026" "00092" "00242" ...
    ##  $ pwgtp25  : chr  "00099" "00106" "00160" "00040" ...
    ##  $ pwgtp26  : chr  "00079" "00082" "00100" "00045" ...
    ##  $ pwgtp27  : chr  "00181" "00132" "00030" "00173" ...
    ##  $ pwgtp28  : chr  "00029" "00024" "00154" "00168" ...
    ##  $ pwgtp29  : chr  "00180" "00165" "00144" "00148" ...
    ##  $ pwgtp30  : chr  "00182" "00172" "00035" "00251" ...
    ##  $ pwgtp31  : chr  "00104" "00102" "00093" "00144" ...
    ##  $ pwgtp32  : chr  "00117" "00084" "00108" "00052" ...
    ##  $ pwgtp33  : chr  "00025" "00023" "00028" "00172" ...
    ##  $ pwgtp34  : chr  "00026" "00024" "00088" "00049" ...
    ##  $ pwgtp35  : chr  "00157" "00149" "00034" "00049" ...
    ##  $ pwgtp36  : chr  "00103" "00092" "00079" "00277" ...
    ##  $ pwgtp37  : chr  "00105" "00100" "00085" "00151" ...
    ##  $ pwgtp38  : chr  "00083" "00085" "00091" "00251" ...
    ##  $ pwgtp39  : chr  "00152" "00143" "00141" "00155" ...
    ##  $ pwgtp40  : chr  "00030" "00025" "00029" "00181" ...
    ##  $ pwgtp41  : chr  "00109" "00098" "00070" "00168" ...
    ##  $ pwgtp42  : chr  "00093" "00078" "00174" "00055" ...
    ##  $ pwgtp43  : chr  "00108" "00084" "00092" "00156" ...
    ##  $ pwgtp44  : chr  "00028" "00027" "00099" "00054" ...
    ##  $ pwgtp45  : chr  "00093" "00081" "00164" "00276" ...
    ##  $ pwgtp46  : chr  "00081" "00084" "00130" "00275" ...
    ##  $ pwgtp47  : chr  "00144" "00141" "00035" "00182" ...
    ##  $ pwgtp48  : chr  "00029" "00033" "00198" "00153" ...
    ##  $ pwgtp49  : chr  "00153" "00140" "00143" "00147" ...
    ##  $ pwgtp50  : chr  "00163" "00148" "00026" "00043" ...
    ##  $ pwgtp51  : chr  "00116" "00105" "00099" "00168" ...
    ##  $ pwgtp52  : chr  "00105" "00112" "00081" "00287" ...
    ##  $ pwgtp53  : chr  "00033" "00027" "00027" "00150" ...
    ##  $ pwgtp54  : chr  "00028" "00024" "00110" "00253" ...
    ##  $ pwgtp55  : chr  "00183" "00161" "00021" "00284" ...
    ##  $ pwgtp56  : chr  "00091" "00094" "00105" "00048" ...
    ##  $ pwgtp57  : chr  "00089" "00071" "00111" "00162" ...
    ##  $ pwgtp58  : chr  "00110" "00095" "00105" "00049" ...
    ##  $ pwgtp59  : chr  "00158" "00161" "00170" "00151" ...
    ##  $ pwgtp60  : chr  "00034" "00025" "00151" "00160" ...
    ##  $ pwgtp61  : chr  "00109" "00093" "00131" "00168" ...
    ##  $ pwgtp62  : chr  "00086" "00076" "00025" "00244" ...
    ##  $ pwgtp63  : chr  "00081" "00074" "00109" "00150" ...
    ##  $ pwgtp64  : chr  "00031" "00024" "00092" "00231" ...
    ##  $ pwgtp65  : chr  "00096" "00089" "00028" "00045" ...
    ##  $ pwgtp66  : chr  "00097" "00095" "00087" "00050" ...
    ##  $ pwgtp67  : chr  "00134" "00130" "00160" "00153" ...
    ##  $ pwgtp68  : chr  "00026" "00027" "00032" "00163" ...
    ##  $ pwgtp69  : chr  "00176" "00145" "00030" "00130" ...
    ##  $ pwgtp70  : chr  "00211" "00142" "00169" "00262" ...
    ##  $ pwgtp71  : chr  "00095" "00097" "00118" "00152" ...
    ##  $ pwgtp72  : chr  "00098" "00091" "00088" "00050" ...
    ##  $ pwgtp73  : chr  "00027" "00030" "00145" "00200" ...
    ##  $ pwgtp74  : chr  "00031" "00027" "00081" "00045" ...
    ##  $ pwgtp75  : chr  "00174" "00141" "00144" "00045" ...
    ##  $ pwgtp76  : chr  "00093" "00093" "00080" "00313" ...
    ##  $ pwgtp77  : chr  "00111" "00098" "00093" "00183" ...
    ##  $ pwgtp78  : chr  "00087" "00100" "00082" "00258" ...
    ##  $ pwgtp79  : chr  "00168" "00177" "00024" "00151" ...
    ##  $ pwgtp80  : chr  "00021" "00025" "00210" "00168" ...

``` r
DBI::dbDisconnect(db)
```

    ## [1] TRUE
