
# Przykładowe kod i dane współtowarzyszące książce "Język R i analiza danych w praktyce. Wydanie II" autorstwa Niny Zumel i Johna Mounta, Helion 2021.


 * Książka (wersja angielska): ["Practical Data Science with R, 2nd Edition" autorstwa Niny Zumel i Johna Mounta, Manning 2019](http://practicaldatascience.com/) (prawa autorskie należą do wydawnictwa Manning Publications Co.; wszystkie prawa zastrzeżone)
 * Strona z materiałami dodatkowymi (wersja oryginalna): [GitHub WinVector/PDSwR2](https://github.com/WinVector/PDSwR2)


## Fragment książki, dla którego przeznaczone są zawarte w tym katalogu kod i dane:
 * Dodatek B. Ważne pojęcia z dziedziny statystyki

## Przykładowe dane biodostępności

Dane te pochodzą z rysunku 4., dostępnego w publicznych materiałach promocycjnych ze strony: [Caco-2 Permeability Assay](http://www.cyprotex.com/admepk/in-vitro-permeability/caco-2-permeability/) i zostały przekształcone w oszacowania numeryczne za pomocą aplikacji [WebPlotDigitizer](http://arohatgi.info/WebPlotDigitizer/).  Takie rozwiązanie może wydawać się dość agresywne, ale tego rodzaju czynności są dopuszczalne w ramach tzw. [fair use](https://pl.wikipedia.org/wiki/Fair_use) (jest to jedno z praw, jakie właściciel praw autorskich musi udzielić w zamian za rządową ochronę praw autorskich; w tym przypadku spełniamy wiele z pojęć/czynników definiujących fair use: wykorzystujemy niewielką część danych faktograficznych w celach badawczych, bez negatywnego wpływu na oryginalne dane). Poza tym analityk danych musi przyzwyczaić się do odrobiny wysiłku w celu zdobycia danych.

Pierwotny rysunek 4. wygląda następująco:

![Rysunek4](Rysunek4.gif)

Wykres ten ukazuje zmierzony współczynnik przepuszczalności komórek [Caco2](http://en.wikipedia.org/wiki/Caco-2), czyli szybkości, z jaką cząsteczka porusza się przez jedną warstwę komórek nowotworowych jelita grubego. Naukowcy mają nadzieję, że nawet mimo tego, iż kultura nieśmiertelnych komórek nie jest zwykłym narządem, to współczynniki przepuszczalności Caco2 pozwalają przewidywać ilość wchłoniętego leku przez jelito cienkie (zanim przejdzie do jelita grubego, które wchłania skomplikowane cząsteczki w mniejszym stopniu od jelita cienkiego). Można traktować test Caco2 jako test przez analogię: wchłanianie na poziomie całego organizmu zostaje oszacowane na podstawie kultury komórkowej za pomocą innych mechanizmów, niż mają miejsce w rzeczywistości. Z tego powodu test Caco2 nie określa własności [LADME](https://pl.wikipedia.org/wiki/LADME) leku, ale traktowany jest jako przydatny skorelowany sygnał (lub cechę), którego warto mierzyć. 

Oszacowane dane z wykresu są przechowywane w pliku [caco2.csv](caco2.csv), z którego otrzymujemy następujący wykres:


```r
library(ggplot2)
d <- read.table('caco2.csv',header=T,sep=',')
ggplot(d,aes(x=Caco2A2BPapp,y=FractionHumanAbsorption)) +
   geom_point() + scale_x_log10() + geom_smooth()
```

```
## geom_smooth: method="auto" and size of largest group is <1000, so using loess. Use 'method = x' to change the smoothing method.
```

![wykres 1.](rysunki/wykres1.png) 


W ten sposób możemy stworzyć bardzo ogólny model logistyczny (sprawdziłaby się tu również regresja beta) łączący współczynnik przepuszczalności Caco2 z wchłanianiem jelitowym w organizmie człowieka.


```r
model <- glm(data=d, FractionHumanAbsorption~log(Caco2A2BPapp),
   family=binomial(link='logit'))
```

```
## Warning: non-integer #successes in a binomial glm!
```

```r
print(summary(model))
```

```
## 
## Call:
## glm(formula = FractionHumanAbsorption ~ log(Caco2A2BPapp), family = binomial(link = "logit"), 
##     data = d)
## 
## Deviance Residuals: 
##     Min       1Q   Median       3Q      Max  
## -0.8834  -0.1819   0.0211   0.3250   0.7046  
## 
## Coefficients:
##                   Estimate Std. Error z value Pr(>|z|)    
## (Intercept)          7.489      1.921    3.90  9.7e-05 ***
## log(Caco2A2BPapp)    0.499      0.144    3.48  0.00051 ***
## ---
## Signif. codes:  0 '***' 0.001 '**' 0.01 '*' 0.05 '.' 0.1 ' ' 1
## 
## (Dispersion parameter for binomial family taken to be 1)
## 
##     Null deviance: 23.4519  on 67  degrees of freedom
## Residual deviance:  8.8853  on 66  degrees of freedom
## AIC: 47.08
## 
## Number of Fisher Scoring iterations: 5
```

```r
d$model <- predict(model,newdata=d,type='response')
ggplot(d) +
   geom_point(aes(x=Caco2A2BPapp,y=FractionHumanAbsorption)) + 
   geom_line(aes(x=Caco2A2BPapp,y=model)) +
   scale_x_log10()
```

![wykres modelu 1.](rysunki/model1.png) 

```r
Intercept <- model$coefficients['(Intercept)']
print(Intercept)
```

```
## (Intercept) 
##       7.489
```

```r
cacoEffect <- model$coefficients['log(Caco2A2BPapp)']
print(cacoEffect)
```

```
## log(Caco2A2BPapp) 
##            0.4992
```


Uzyskujemy oszacowanie, zgodnie z którym zlogarytmowana jednostka wzrostu przepuszczalności cm/s mierzona przez test Caco2 zazwyczaj zwiększa logit ułamka wchłaniania jelita człowieka o wartość 0.4992. Posłużymy się tym do stworzenia celowo nieprawidłowego przykładu syntetycznego.

Chcemy otrzymać zestaw danych przypominający dane uzyskiwane w procesie optymalizowania leku. W tak intenstywnych projektach, jak optymalizowanie leków wcześnie można rozpoznać cząsteczki, które "mają wszystkie typowe cechy" i działają jak należy. W późniejszym fazach cząsteczki zazwyczaj działają jeszcze lepiej (optymalizacja), ale często są pozbawione wielu cech, które na początku projektu były uznawane za istotne (gdyż relacje pomiędzy cechami nie są sztywne i nie zawsze jest możliwe, aby potencjalny lek był jednocześnie najlepszy na kilku różnych płaszczyznach).

Można oczekiwać, że odsetek wchłaniania przez organizm będzie rósł w miarę upływu czasu (gdyż będą opracowywane coraz lepsze leki). Gdyby pierwotne cząsteczki były zoptymalizowane ze względu na przepuszczalność Caco2, to równocześnie powinniśmy zaobserować zmniejszanie się tej wartości w miarę optymalizacji rzeczywistego współczynnika ADME człowieka (gdyż oddalalibyśmy się od cząsteczek zoptymalizowanych pod kątem Caco2). Naiwna analiza danych pochodzących z takich serii danych mogłaby z łatwością sugerować (w wyniku [problemu pominiętych zmiennych](https://pl.wikipedia.org/wiki/Problem_pominiętych_zmiennych)) (zobacz również [zmienne zakłócające](https://pl.wikipedia.org/wiki/Zmienna_zakłócająca) i [zmienne niewygodne](http://en.wikipedia.org/wiki/Nuisance_variable)), że duża wartość współczynnika Caco2 wiąże się z małą wartością współczynnika ADME (gdy w rzeczywistości uważamy, że duża wchłanialność Caco2 wiąże się z dobrą wchłanialnością ADME). Dlatego nie powinniśmy poprzestawać na prostej analizie danych i interpretowaniu wyników na oślep.


Syntetyczny zestaw danych uzyskamy w następujący sposób:

```r
set.seed(2535251)
s <- data.frame(week=1:100)
s$Caco2A2BPapp <- sort(sample(d$Caco2A2BPapp,100,replace=T),decreasing=T)
sigmoid <- function(x) {1/(1+exp(-x))}
s$FractionHumanAbsorption <- 
 sigmoid(Intercept + cacoEffect*log(s$Caco2A2BPapp) + 
   s$week/10 - mean(s$week/10) +
   rnorm(100)/3)
write.table(s,'synth.csv',sep=',',quote=F,row.names=F)
save(s,d,file='synth.RData')
```


Symulujemy tu proces optymalizacji leku pod względem wchłaniania przez układ pokarmowy (przypuszczalnie poprzez wprowadzanie zmian w obszarze bezpośrednio wpływającym na tę własność), przez co w miarę postępów malała wchłanialność Caco2. Pamiętaj, że w jelicie cienkim nie występuje ściśle przyczynowa relacja pomiędzy wchłanialnością Caco2 a wchłanialnością jelita cienkiego, ale są one ze sobą skorelowane poprzez wiele współdzielonych czynników (i nie są identyczne z powodu wielu odrębnych czynników).


```r
ggplot(s,aes(x=week,y=FractionHumanAbsorption)) +
   geom_point() + geom_smooth()
```

```
## geom_smooth: method="auto" and size of largest group is <1000, so using loess. Use 'method = x' to change the smoothing method.
```

![Wykres T](rysunki/wykresT.png) 


W tym zestawie danych wysoka wartość wchłanialności Caco2 jest korzystna dla wchłanialności jelita cienkiego, ale warto pamiętać, że dominują inne zjawiska (zmieniające się w czasie). W miarę upływu czasu zauważamy, że wchłanialność jelita cienkiego rośnie, natomiast w przypadku wchłanialności Caco2 wartość ta maleje: co może prowadzić do mylnego wniosku, że istnieje pomiędzy nimi korelacja odwrotna.
Zakładamy, że wszystkie czynniki zwiększające zarówno wchłanialność Caco2, jak i jelitową zostały wypróbowane na początku projektu, a teraz naukowcy sprawdzają rozwiązania bezpośrednio usprawniające wchłanianie z poziomu jelita cienkiego wiedząc, że będą one osłabiać wchłanialność Caco2. Być może preferowana byłaby duża wartość współczynnika Caco2, ale ma on znaczenie drugorzędne w stosunku do bardziej bezpośredniego pomiaru wchłanialności jelita cienkiego, dlatego badacze są skłonni do coraz większych kompromisów wraz z rozwojem projektu. Zmienna czasu (time/week) tylko uwidacznia rzeczywiste pomijane czynniki.



```r
print(summary(glm(data=s, FractionHumanAbsorption~log(Caco2A2BPapp),
   family=binomial(link='logit'))))
```

```
## Warning: non-integer #successes in a binomial glm!
```

```
## 
## Call:
## glm(formula = FractionHumanAbsorption ~ log(Caco2A2BPapp), family = binomial(link = "logit"), 
##     data = s)
## 
## Deviance Residuals: 
##    Min      1Q  Median      3Q     Max  
## -0.609  -0.246  -0.118   0.202   0.557  
## 
## Coefficients:
##                   Estimate Std. Error z value Pr(>|z|)    
## (Intercept)        -10.003      2.752   -3.64  0.00028 ***
## log(Caco2A2BPapp)   -0.969      0.257   -3.77  0.00016 ***
## ---
## Signif. codes:  0 '***' 0.001 '**' 0.01 '*' 0.05 '.' 0.1 ' ' 1
## 
## (Dispersion parameter for binomial family taken to be 1)
## 
##     Null deviance: 43.7821  on 99  degrees of freedom
## Residual deviance:  9.4621  on 98  degrees of freedom
## AIC: 64.7
## 
## Number of Fisher Scoring iterations: 6
```

```r
ggplot(s,aes(x=Caco2A2BPapp,y=FractionHumanAbsorption)) +
    geom_point() + scale_x_log10() + geom_smooth()
```

```
## geom_smooth: method="auto" and size of largest group is <1000, so using loess. Use 'method = x' to change the smoothing method.
```

![synth P](rysunki/synthP.png) 


Jak widać, celowo wstawiowa dodatnia relacja pomiędzy log(Caco2) a absorpcją jelitową zanika w trakcie prostej analizy. Niepokój powinien wzbudzić niewłaściwy znak współczynnika Caco2. Problem w tym, że zmienna czasu (week) i log(Caco2) są skorelowane ze sobą (ujemnie), zatem jeśli jedna cecha zniknie, można wykorzystać drugą z przeciwnym znakiem jako zastępstwo predykcji (co prowadzi do niezrozumiałych współczynników). Stanowi to przypomnienie, że nawet model uzyskujący dobre przewidywanie nie musi ich wyjaśniać w zrozumiały sposób.

Jednym z rozwiązań tego problemu jest wprowadzenie potencjalnie przydatnych zmiennych i powtórzenie z nimi fazy modelowania. W naszym przykładzie pominiętą zmienną będzie data (często stanowi ona potencjalnie przydatną zmienną).  One way to fix this is to introduce candidate variables and re-try the modeling.  In our example the omitted variable will be date (which is often a good candidate).  Wśród innych sposobów występują takie, jak stosowanie odpowiednich [modeli o stałych efektach](http://en.wikipedia.org/wiki/Fixed_effects_model), [modeli o efektach losowych](http://en.wikipedia.org/wiki/Random_effects_model), kontrolowanie rozkładów apriorycznych, stosowanie modeli hierarchicznych, kontrolowanie regularyzacji lub wymuszanie ortogonalizacji zmiennych.

Zatem aby rozwiązać nasz problem, musimy co najmniej odpowiednio dopasować dane do pominiętej zmiennej:


```r
print(summary(glm(data=s,FractionHumanAbsorption~week+log(Caco2A2BPapp),
   family=binomial(link='logit'))))
```

```
## Warning: non-integer #successes in a binomial glm!
```

```
## 
## Call:
## glm(formula = FractionHumanAbsorption ~ week + log(Caco2A2BPapp), 
##     family = binomial(link = "logit"), data = s)
## 
## Deviance Residuals: 
##     Min       1Q   Median       3Q      Max  
## -0.3474  -0.0568  -0.0010   0.0709   0.3038  
## 
## Coefficients:
##                   Estimate Std. Error z value Pr(>|z|)   
## (Intercept)         3.1413     4.6837    0.67   0.5024   
## week                0.1033     0.0386    2.68   0.0074 **
## log(Caco2A2BPapp)   0.5689     0.5419    1.05   0.2938   
## ---
## Signif. codes:  0 '***' 0.001 '**' 0.01 '*' 0.05 '.' 0.1 ' ' 1
## 
## (Dispersion parameter for binomial family taken to be 1)
## 
##     Null deviance: 43.7821  on 99  degrees of freedom
## Residual deviance:  1.2595  on 97  degrees of freedom
## AIC: 47.82
## 
## Number of Fisher Scoring iterations: 6
```

Uwaga: zgadujemy tu zarówno zmienne, jak również ich przekształcenia (log()), dlatego ten problem warto byłoby spróbować rozwiązać za pomocą uogólnionego modelu addytywnego (z pakietu GAM).

Zwróć uwagę, że uzyskaliśmy rozsądne (ale nie znakomite) oszacowania współczynników dla zmiennych czasu i Caco2 (chociaż nie otrzymaliśmy istotności na poziomie oszacowania zjawiska Caco2). Jednak nie zawsze mamy dostęp do wystarczająco zróżnicowanych danych, aby być w stanie odtworzyć jakieś zjawisko (nawet gdy we właściwy sposób wykorzystujemy pominiętą zmienną). Jeżeli takie rozwiązanie się nie sprawdzi, musimy spróbować wymusić efekty w pominiętej zmiennej. Na tym etapie masz już kłopoty i stosowane metody nie zawsze zwracają to, czego byśmy oczekiwali. Możesz na przykład próbować "wcisnąć" wszystkie wartości objaśniające w zmienną czasu za pomocą polecenia offset (technika ta pozwala dopasować wartości resztowe w wielu różnych okolicznościach; zob. help(offset) i help(formula)):


```r
mF <- glm(data=s, FractionHumanAbsorption~week,
    family=binomial(link='logit'))
```

```
## Warning: non-integer #successes in a binomial glm!
```

```r
weekEffect <- mF$coefficients['week']
print(summary(mF))
```

```
## 
## Call:
## glm(formula = FractionHumanAbsorption ~ week, family = binomial(link = "logit"), 
##     data = s)
## 
## Deviance Residuals: 
##     Min       1Q   Median       3Q      Max  
## -0.3037  -0.1312  -0.0575   0.0776   0.3833  
## 
## Coefficients:
##             Estimate Std. Error z value Pr(>|z|)    
## (Intercept)  -1.8130     0.5499   -3.30  0.00098 ***
## week          0.0676     0.0142    4.77  1.8e-06 ***
## ---
## Signif. codes:  0 '***' 0.001 '**' 0.01 '*' 0.05 '.' 0.1 ' ' 1
## 
## (Dispersion parameter for binomial family taken to be 1)
## 
##     Null deviance: 43.7821  on 99  degrees of freedom
## Residual deviance:  2.3203  on 98  degrees of freedom
## AIC: 49.78
## 
## Number of Fisher Scoring iterations: 6
```

```r
print(summary(glm(data=s,FractionHumanAbsorption~log(Caco2A2BPapp) +
   offset(weekEffect*week),
   family=binomial(link='logit'))))
```

```
## Warning: non-integer #successes in a binomial glm!
## Warning: non-integer #successes in a binomial glm!
```

```
## 
## Call:
## glm(formula = FractionHumanAbsorption ~ log(Caco2A2BPapp) + offset(weekEffect * 
##     week), family = binomial(link = "logit"), data = s)
## 
## Deviance Residuals: 
##     Min       1Q   Median       3Q      Max  
## -0.3274  -0.1184  -0.0209   0.1050   0.3815  
## 
## Coefficients:
##                   Estimate Std. Error z value Pr(>|z|)
## (Intercept)        -0.9283     2.1973   -0.42     0.67
## log(Caco2A2BPapp)   0.0818     0.2007    0.41     0.68
## 
## (Dispersion parameter for binomial family taken to be 1)
## 
##     Null deviance: 2.3203  on 99  degrees of freedom
## Residual deviance: 2.1663  on 98  degrees of freedom
## AIC: 51.41
## 
## Number of Fisher Scoring iterations: 6
```


Zauważ, że w tym przypadku oszacowania zarówno współczynnika czasu, jak i log(Caco2) są znacznie gorsze. Z tego wynika, że polecenie offset nie jest "srebrną kulą" (przynajmniej uzyskało właściwy znak współczynnika log(Caco2), ale nie uzyskało istotności oszacowania).


Polecenie przygotowujące dokument (w powłoce bash):
```
echo "library('knitr'); knit('README.Rmd')" | R --vanilla ; pandoc -o README.html README.md
```
