// Copyright © 2016 Alan A. A. Donovan & Brian W. Kernighan.
// License: https://creativecommons.org/licenses/by-nc-sa/4.0/


// Du3 oblicza wykorzystanie dysku przez pliki w katalogu.
package main

// Wariant du3 trawersuje wszystkie katalogi równolegle.
// Wykorzystuje semafor zliczajacy, który ogranicza współbieżność,
// aby uniknąć otwierania zbyt wielu plików jednocześnie.

import (
	"flag"
	"fmt"
	"io/ioutil"
	"os"
	"path/filepath"
	"sync"
	"time"
)

var vFlag = flag.Bool("v", false, "pokazuje rozszerzone komunikaty postępu")

//!+
func main() {
	// ...określenie korzeni...

	//!-
	flag.Parse()

	// Określa początkowe katalogi.
	roots := flag.Args()
	if len(roots) == 0 {
		roots = []string{"."}
	}

	//!+
	// Trawersuje każdy korzeń drzewa plików równolegle.
	fileSizes := make(chan int64)
	var n sync.WaitGroup
	for _, root := range roots {
		n.Add(1)
		go walkDir(root, &n, fileSizes)
	}
	go func() {
		n.Wait()
		close(fileSizes)
	}()
	//!-

	// Wyświetla wyniki periodycznie.
	var tick <-chan time.Time
	if *vFlag {
		tick = time.Tick(500 * time.Millisecond)
	}
	var nfiles, nbytes int64
loop:
	for {
		select {
		case size, ok := <-fileSizes:
			if !ok {
				break loop // kanał fileSizes został zamknięty
			}
			nfiles++
			nbytes += size
		case <-tick:
			printDiskUsage(nfiles, nbytes)
		}
	}

	printDiskUsage(nfiles, nbytes) // końcowe sumy
	//!+
	// ...pętla select...
}

//!-

func printDiskUsage(nfiles, nbytes int64) {
	fmt.Printf("%d pliki  %.1f GB\n", nfiles, float64(nbytes)/1e9)
}

// walkDir rekurencyjnie przechodzi drzewo plików zakorzenione w dir
// i wysyła rozmiar każdego znalezionego pliku przez kanał fileSizes.
//!+walkDir
func walkDir(dir string, n *sync.WaitGroup, fileSizes chan<- int64) {
	defer n.Done()
	for _, entry := range dirents(dir) {
		if entry.IsDir() {
			n.Add(1)
			subdir := filepath.Join(dir, entry.Name())
			go walkDir(subdir, n, fileSizes)
		} else {
			fileSizes <- entry.Size()
		}
	}
}

//!-walkDir

//!+sema
// sema jest semaforem zliczającym służącym do ograniczania współbieżności w funkcji dirents.
var sema = make(chan struct{}, 20)

// dirents zwraca wpisy katalogu dir.
func dirents(dir string) []os.FileInfo {
	sema <- struct{}{}        // nabycie żetonu
	defer func() { <-sema }() // zwolnienie żetonu
	// ...
	//!-sema

	entries, err := ioutil.ReadDir(dir)
	if err != nil {
		fmt.Fprintf(os.Stderr, "du: %v\n", err)
		return nil
	}
	return entries
}
