/*
 * Copyright 2013 Oracle and/or its affiliates.
 * All rights reserved.  You may not modify, use,
 * reproduce, or distribute this software except in
 * compliance with  the terms of the License at:
 * http://developers.sun.com/license/berkeley_license.html
 */


package com.forest.web;

import com.forest.ejb.CategoryBean;
import com.forest.entity.Category;
import com.forest.web.util.AbstractPaginationHelper;
import com.forest.web.util.JsfUtil;
import com.forest.web.util.PageNavigation;
import java.io.Serializable;
import java.util.ResourceBundle;
import javax.ejb.EJB;
import javax.faces.bean.ManagedBean;
import javax.faces.bean.SessionScoped;
import javax.faces.component.UIComponent;
import javax.faces.context.FacesContext;
import javax.faces.convert.Converter;
import javax.faces.convert.FacesConverter;
import javax.faces.model.DataModel;
import javax.faces.model.ListDataModel;
import javax.faces.model.SelectItem;


@ManagedBean(name = "categoryController")
@SessionScoped
public class CategoryController implements Serializable {
    private static final String BUNDLE = "/Bundle";
    private static final long serialVersionUID = 2310259107429450847L;
    @EJB
    private com.forest.ejb.CategoryBean ejbFacade;
    private AbstractPaginationHelper pagination;
    private Category current;
    private DataModel items = null;
    private int selectedItemIndex;

    public CategoryController() {
    }

    public Category getSelected() {
        if (current == null) {
            current = new Category();
            selectedItemIndex = -1;
        }

        return current;
    }

    private CategoryBean getFacade() {
        return ejbFacade;
    }

    public AbstractPaginationHelper getPagination() {
        if (pagination == null) {
            pagination = new AbstractPaginationHelper(
                        AbstractPaginationHelper.DEFAULT_SIZE) {
                        @Override
                        public int getItemsCount() {
                            return getFacade()
                                       .count();
                        }

                        @Override
                        public DataModel createPageDataModel() {
                            return new ListDataModel(
                                    getFacade().findRange(
                                            new int[] {
                                                getPageFirstItem(),
                                                getPageFirstItem()
                                                + getPageSize()
                                            }));
                        }
                    };
        }

        return pagination;
    }

    public PageNavigation prepareList() {
        recreateModel();

        return PageNavigation.LIST;
    }

    public PageNavigation prepareView() {
        current = (Category) getItems()
                                 .getRowData();
        selectedItemIndex = pagination.getPageFirstItem()
            + getItems()
                  .getRowIndex();

        return PageNavigation.VIEW; //.getText();
    }

    public PageNavigation prepareCreate() {
        current = new Category();
        selectedItemIndex = -1;

        return PageNavigation.CREATE; //.getText();//"Create";
    }

    public PageNavigation create() {
        try {
            getFacade()
                .create(current);
            JsfUtil.addSuccessMessage(
                    ResourceBundle.getBundle(BUNDLE).getString(
                            "CategoryCreated"));
            recreateModel();

            return prepareCreate();
        } catch (Exception e) {
            JsfUtil.addErrorMessage(
                    e,
                    ResourceBundle.getBundle(BUNDLE).getString(
                            "PersistenceErrorOccured"));

            return null;
        }
    }

    public PageNavigation prepareEdit() {
        current = (Category) getItems()
                                 .getRowData();
        selectedItemIndex = pagination.getPageFirstItem()
            + getItems()
                  .getRowIndex();

        return PageNavigation.EDIT;
    }

    public PageNavigation update() {
        try {
            getFacade()
                .edit(current);
            JsfUtil.addSuccessMessage(
                    ResourceBundle.getBundle(BUNDLE).getString(
                            "CategoryUpdated"));

            return PageNavigation.VIEW;
        } catch (Exception e) {
            JsfUtil.addErrorMessage(
                    e,
                    ResourceBundle.getBundle(BUNDLE).getString(
                            "PersistenceErrorOccured"));

            return null;
        }
    }

    public PageNavigation destroy() {
        current = (Category) getItems()
                                 .getRowData();
        selectedItemIndex = pagination.getPageFirstItem()
            + getItems()
                  .getRowIndex();
        performDestroy();
        recreateModel();

        return PageNavigation.LIST;
    }

    public PageNavigation destroyAndView() {
        performDestroy();
        recreateModel();
        updateCurrentItem();

        if (selectedItemIndex >= 0) {
            return PageNavigation.VIEW;
        } else {
            // all items were removed - go back to list
            recreateModel();

            return PageNavigation.LIST;
        }
    }

    private void performDestroy() {
        try {
            getFacade()
                .remove(current);
            JsfUtil.addSuccessMessage(
                    ResourceBundle.getBundle(BUNDLE).getString(
                            "CategoryDeleted"));
        } catch (Exception e) {
            JsfUtil.addErrorMessage(
                    e,
                    ResourceBundle.getBundle(BUNDLE).getString(
                            "PersistenceErrorOccured"));
        }
    }

    private void updateCurrentItem() {
        int count = getFacade()
                        .count();

        if (selectedItemIndex >= count) {
            // selected index cannot be bigger than number of items:
            selectedItemIndex = count - 1;

            // go to previous page if last page disappeared:
            if (pagination.getPageFirstItem() >= count) {
                pagination.previousPage();
            }
        }

        if (selectedItemIndex >= 0) {
            current = getFacade()
                          .findRange(
                        new int[] { selectedItemIndex, selectedItemIndex + 1 })
                          .get(0);
        }
    }

    public DataModel getItems() {
        if (items == null) {
            items = getPagination()
                        .createPageDataModel();
        }

        return items;
    }

    private void recreateModel() {
        items = null;
    }

    public PageNavigation next() {
        getPagination()
            .nextPage();
        recreateModel();

        return PageNavigation.LIST;
    }

    public PageNavigation previous() {
        getPagination()
            .previousPage();
        recreateModel();

        return PageNavigation.LIST;
    }

    public SelectItem[] getItemsAvailableSelectMany() {
        return JsfUtil.getSelectItems(
            ejbFacade.findAll(),
            false);
    }

    public SelectItem[] getItemsAvailableSelectOne() {
        return JsfUtil.getSelectItems(
            ejbFacade.findAll(),
            true);
    }

    @FacesConverter(forClass = Category.class)
    public static class CategoryControllerConverter implements Converter {
        @Override
        public Object getAsObject(
            FacesContext facesContext,
            UIComponent component,
            String value) {
            if ((value == null) || (value.length() == 0)) {
                return null;
            }

            CategoryController controller = (CategoryController) facesContext.getApplication()
                                                                             .getELResolver()
                                                                             .getValue(
                        facesContext.getELContext(),
                        null,
                        "categoryController");

            return controller.ejbFacade.find(getKey(value));
        }

        java.lang.Integer getKey(String value) {
            java.lang.Integer key;
            key = Integer.valueOf(value);

            return key;
        }

        String getStringKey(java.lang.Integer value) {
            StringBuilder sb = new StringBuilder();
            sb.append(value);

            return sb.toString();
        }

        @Override
        public String getAsString(
            FacesContext facesContext,
            UIComponent component,
            Object object) {
            if (object == null) {
                return null;
            }

            if (object instanceof Category) {
                Category o = (Category) object;

                return getStringKey(o.getId());
            } else {
                throw new IllegalArgumentException(
                        "object " + object + " is of type "
                        + object.getClass().getName() + "; expected type: "
                        + CategoryController.class.getName());
            }
        }
    }
}
