package javaxml3;

import java.awt.*;
import java.io.File;
import java.io.IOException;

import javax.swing.*;
import javax.swing.tree.*;
import javax.swing.event.*;
import java.util.Iterator;

// Importy JDOM.
import org.jdom.*;
import org.jdom.input.SAXBuilder;

public class SimpleXPathViewer extends JFrame {

    /** Wewntrzna klasa procedury obsugi zdarze. */
    EventHandler eventHandler = new EventHandler();

    /** Pole tekstowe do wywietlania XPath dla wybranego wierzchoka. */
    private JTextField statusText;

    /** Drzewo JTree do wywietlania wierzchokw dokumentu xml. */
    private JTree jdomTree;

    /** Sprawdza ktry wierzchoek zosta kliknity. */
    private DefaultTreeSelectionModel selectionModel;

    /** Nazwa pliku zawierajcego xml do obejrzenia. */
    private String filename;

    /** Chwilowe obejcie braku wierzchoka tekstowego. */
    private static Element lastElement;

    class EventHandler implements TreeSelectionListener {

        public void valueChanged(TreeSelectionEvent e) {
            TreePath path= selectionModel.getLeadSelectionPath();

            // Jeli tylko zwijasz drzewo, to moe nie by nowej cieki.
            if (path != null) {
                JDOMNode selection=
                    (JDOMNode)((DefaultMutableTreeNode)path.getLastPathComponent())
                        .getUserObject();
                buildXPath(selection);
            }
        };
    };

    public SimpleXPathViewer(String fileName) throws Exception {
        super();
        this.filename = fileName;
        setSize(600, 450);
        initialize();
    }

    private void initialize() throws Exception {
        setTitle("Simple XPath Viewer");

        // Ustawienie UI.
        initConnections();

        // adowanie JDOM Document.
        Document doc = loadDocument(filename);

        // Utworzenie pocztkowego JDOMNode z metody fabryki.
        JDOMNode root = createNode(doc.getRootElement());

        // Utworzenie korzenia drzewa JTree i budowa drzewa z JDOM Document.
        DefaultMutableTreeNode treeNode = 
            new DefaultMutableTreeNode("Document: " + filename);
        buildTree(root, treeNode);

        // Dodanie wierzchoka do modelu drzewa.
        ((DefaultTreeModel)jdomTree.getModel()).setRoot(treeNode);
    }

    private void initConnections() {
        setDefaultCloseOperation(javax.swing.WindowConstants.DISPOSE_ON_CLOSE);

        // Drzewo JTree i panel do jego obejrzenia.
        jdomTree = new JTree();
        jdomTree.setName("JDOM Tree");
        jdomTree.addTreeSelectionListener(eventHandler);
        selectionModel = (DefaultTreeSelectionModel)jdomTree.getSelectionModel();
        getContentPane().add(new JScrollPane(jdomTree), BorderLayout.CENTER);

        // Pole tekstowe do uytku w pasku statusu.
        statusText = new JTextField("Kliknij element, aby zobaczy ciek XPath.");
        JPanel statusBarPane= new JPanel();
        statusBarPane.setLayout(new BorderLayout());
        statusBarPane.add(statusText, BorderLayout.CENTER );
        getContentPane().add(statusBarPane, BorderLayout.SOUTH);
    }

    private Document loadDocument(String filename) throws JDOMException, IOException {
        SAXBuilder builder = new SAXBuilder();
        builder.setIgnoringElementContentWhitespace(true);
        return builder.build(new File(filename));
    }

    private JDOMNode createNode(Object node) {
        if (node instanceof Element) {
            lastElement = (Element)node;
            return new ElementNode((Element)node);
        }

        if (node instanceof Attribute) {
            return new AttributeNode((Attribute)node);
        }

        if (node instanceof Text) {
            return new TextNode((Text)node).setParent(lastElement);
        }

        // Pozostae wierzchoki nie s zaimplementowane.
        return null;
    }

    private void buildTree(JDOMNode node, DefaultMutableTreeNode treeNode) {
        // Jeli to jest spacja, albo jaki nieobsugiwany wierzchoek, to zignoruj to.
        if ((node == null) || (node.toString().trim().equals(""))) {
            return;
        }

        DefaultMutableTreeNode newTreeNode = new DefaultMutableTreeNode(node);

        // Przejcie przez potomkw wierzchoka. 
        Iterator i = node.iterator();
        while (i.hasNext()) {
            // Tworzenie wierzchokw JDOMNode na potomkach i dodawanie do drzewa.
            JDOMNode newNode = createNode(i.next());
            buildTree(newNode, newTreeNode);
        }

        // Po dodaniu wszystkich potomkw, pocz z drzewem.
        treeNode.add(newTreeNode);
    }

    private void buildXPath(JDOMNode node) {
        try {
            statusText.setText(new XPathDisplayNode(node).getXPath());
        } catch (JDOMException e) {
            statusText.setText("Unable to build XPath");
            e.printStackTrace();
        }
    }

    public static void main(java.lang.String[] args) {
        try {
            if (args.length != 1) {
                System.out.println("Usage: java javaxml3.SimpleXPathViewer " + 
                    "[XML Document filename]");
                return;
            }

            /* Tworzenie ramki. */
            SimpleXPathViewer viewer= new SimpleXPathViewer(args[0]);

            /* Dodanie windowListener dla windowClosedEvent. */
            viewer.addWindowListener(new java.awt.event.WindowAdapter() {
                    public void windowClosed(java.awt.event.WindowEvent e) {
                        System.exit(0);
                    };
                });
            viewer.setVisible(true);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}
