package javaxml3;

import java.awt.BorderLayout;
import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import javax.swing.JFrame;
import javax.swing.JScrollPane;
import javax.swing.JTree;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;

import org.xml.sax.Attributes;
import org.xml.sax.ContentHandler;
import org.xml.sax.InputSource;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.DefaultHandler;
import org.xml.sax.helpers.XMLReaderFactory;

class JTreeHandler extends DefaultHandler {

	/** Biecy wierzchoek, do ktrego dodawane bd pod-wierzchoki. */
	private DefaultMutableTreeNode current;

	/** Zapis identyfikatora URI odwzorowa prefiksw */
	private Map namespaceMappings;

	/** Model drzewa, do ktrego dodawane bd wierzchoki. */
	private DefaultTreeModel treeModel;

	public JTreeHandler(DefaultTreeModel treeModel, DefaultMutableTreeNode base) {
		this.treeModel = treeModel;
		this.current = base;
		this.namespaceMappings = new HashMap();
	}

	public void characters(char[] ch, int start, int length)
			throws SAXException {
		String s = new String(ch, start, length);
		DefaultMutableTreeNode data = new DefaultMutableTreeNode(
				"Character Data: '" + s + "'");
		current.add(data);
	}

	public void endElement(String uri, String localName, String qName)
			throws SAXException {
		// Powrt w gr drzewa.
		current = (DefaultMutableTreeNode) current.getParent();

	}

	public void endPrefixMapping(String prefix) throws SAXException {
		for (Iterator i = namespaceMappings.keySet().iterator(); i.hasNext();) {
			String uri = (String) i.next();
			String thisPrefix = (String) namespaceMappings.get(uri);
			if (prefix.equals(thisPrefix)) {
				namespaceMappings.remove(uri);
				break;
			}
		}

	}

	public void ignorableWhitespace(char[] ch, int start, int length)
			throws SAXException {
		DefaultMutableTreeNode ws = new DefaultMutableTreeNode(
				"Whitespace (length = " + length + ")");
		current.add(ws);

	}

	public void processingInstruction(String target, String data)
			throws SAXException {
		DefaultMutableTreeNode pi = new DefaultMutableTreeNode("PI (target = '"
				+ target + "', data = '" + data + "')");
		current.add(pi);

	}

	public void skippedEntity(String name) throws SAXException {
		DefaultMutableTreeNode skipped = new DefaultMutableTreeNode(
				"Skipped Entity: '" + name + "'");
		current.add(skipped);

	}

	public void startElement(String namespaceURI, String localName,
			String qName, Attributes atts) throws SAXException {
		DefaultMutableTreeNode element = new DefaultMutableTreeNode("Element: "
				+ localName);
		current.add(element);
		current = element;

		// Okrelanie przestrzeni nazw.
		if (namespaceURI.length() > 0) {
			String prefix = (String) namespaceMappings.get(namespaceURI);
			if (prefix.equals("")) {
				prefix = "[None]";
			}
			DefaultMutableTreeNode namespace = new DefaultMutableTreeNode(
					"Namespace: prefix = '" + prefix + "', URI = '"
							+ namespaceURI + "'");
			current.add(namespace);
		}

		// Przetwarzanie atrybutw.
		for (int i = 0; i < atts.getLength(); i++) {
			DefaultMutableTreeNode attribute = new DefaultMutableTreeNode(
					"Attribute (name = '" + atts.getLocalName(i)
							+ "', value = '" + atts.getValue(i) + "')");
			String attURI = atts.getURI(i);
			if (attURI.length() > 0) {
				String attPrefix = (String) namespaceMappings.get(attURI);
				if (attPrefix.equals("")) {
					attPrefix = "[None]";
				}
				DefaultMutableTreeNode attNamespace = new DefaultMutableTreeNode(
						"Namespace: prefix = '" + attPrefix + "', URI = '"
								+ attURI + "'");
				attribute.add(attNamespace);
			}
			current.add(attribute);
		}

	}

	public void startPrefixMapping(String prefix, String uri)
			throws SAXException {
		namespaceMappings.put(uri, prefix);

	}
}

public class SAXTreeViewer extends JFrame {

	/** Podstawowe drzewo do wizualizacji. */
	private JTree jTree;

	/** Model drzewa ktry ma zosta uyty. */
	DefaultTreeModel defaultTreeModel;

	public SAXTreeViewer() {
		// Swing
		super("SAX Tree Viewer");
		setSize(600, 450);
	}

	public void init(String xmlURI) throws IOException, SAXException {
		DefaultMutableTreeNode base = new DefaultMutableTreeNode(
				"XML Document: " + xmlURI);

		// Tworzenie modelu drzewa.
		defaultTreeModel = new DefaultTreeModel(base);
		jTree = new JTree(defaultTreeModel);

		// Konstruowanie hierarchii drzewa.
		buildTree(defaultTreeModel, base, xmlURI);

		// Pokazanie wynikw.
		getContentPane().add(new JScrollPane(jTree), BorderLayout.CENTER);
	}

	// Zmienne i metody zwizane z pakietem Swing, wcznie z
	// nastawianiem drzewa JTree i podstawowej powierzchni zawartoci.
	public static void main(String[] args) {
		try {
			if (args.length != 1) {
				System.out.println("Usage: java javaxml3.SAXTreeViewer "
						+ "[XML Document]");
				return;
			}
			SAXTreeViewer viewer = new SAXTreeViewer();
			viewer.init(args[0]);
			viewer.setVisible(true);
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	public void buildTree(DefaultTreeModel treeModel,
			DefaultMutableTreeNode base, String xmlURI) throws IOException,
			SAXException {

		// Create instances needed for parsing
		XMLReader reader = XMLReaderFactory.createXMLReader();

		ContentHandler jTreeHandler = new JTreeHandler(treeModel, base);

		// Rejestracja procedury obsugi encji.
		reader.setEntityResolver(new SimpleEntityResolver());

		// Wczenie walidacji.
		String featureURI = "http://xml.org/sax/features/validation";
		reader.setFeature(featureURI, true);

		// Wczenie walidacji schematw.
		featureURI = "http://apache.org/xml/features/validation/schema";
		reader.setFeature(featureURI, true);

		// Rejestracja procedury obsugi treci.
		reader.setContentHandler(jTreeHandler);

		// Rejestracja procedury obsugi bdw.

		// Analiza skadniowa.
		InputSource inputSource = new InputSource(xmlURI);
		reader.parse(inputSource);
	}

}
