package com.brackeen.javagamebook.graphics;

import java.awt.*;
import java.awt.image.BufferedImage;
import java.io.*;
import javax.swing.ImageIcon;

/**
    Klasa ImageFont umoliwia wczytywanie i rysowanie tekstu
    za pomoc obrazw reprezentujcych znaki.

    Wczytuje wszystkie pliki PNG znajdujce si w danym katalogu.
    Wczytywane pliki maj nazw w postaci "charXX.png", gdzie "XX"
    jest dziesitnym kodem unicode znaku.

    Znaki mog si rni wysokociami i szerokociami.
*/
public class ImageFont {

    public static final int HCENTER = 1;
    public static final int VCENTER = 2;
    public static final int LEFT = 4;
    public static final int RIGHT = 8;
    public static final int TOP = 16;
    public static final int BOTTOM = 32;

    private char firstChar;
    private Image[] characters;
    private Image invalidCharacter;


    /**
        Tworzy nowy obiekt klasy ImageFont bez znakw.
    */
    public ImageFont() {
        this(null);
        firstChar = 0;
        characters = new Image[0];
    }


    /**
        Tworzy nowy obiekt klasy ImageFont i wczytuje obrazy
        reprezentujce znaki z plikw umieszczonych w danym katalogu.
    */
    public ImageFont(String path) {
        if (path != null) {
            load(path);
        }

        // powoduje, e dany znak bdzie wykorzystywany dla niepoprawnych znakw
        invalidCharacter = new BufferedImage(10, 10, BufferedImage.TYPE_INT_RGB);
        Graphics g = invalidCharacter.getGraphics();
        g.setColor(Color.RED);
        g.fillRect(0,0,10,10);
        g.dispose();
    }


    /**
        Wczytuje umieszczone we wskazanym katalogu pliki z obrazami
        dla poszczeglnych znakw. Przykadowo, jeli tym katalogiem
        jest "../fonts/large", metoda przeszukuje go w celu otwarcia
        wszystkich plikw o nazwie "charXX.png", gdzie "XX" jest
        dziesitnym kodem znaku unicode. Katalog nie musi zawiera
        plikw z obrazami dla penego zbioru znakw; moemy np.
        stworzy obrazy tylko dla wielkich liter lub cyfr.
    */
    public void load(String path) throws NumberFormatException {
        // w tym katalogu:
        // wczytaj kady plik .png, ktrego nazwa rozpoczyna si od 'char'
        File dir = new File(path);
        File[] files = dir.listFiles();

        // znajd znaki o minimalnym i maksymalnym kodzie
        char minChar = Character.MAX_VALUE;
        char maxChar = Character.MIN_VALUE;
        for (int i=0; i<files.length; i++) {
            int unicodeValue = getUnicodeValue(files[i]);
            if (unicodeValue != -1) {
                minChar = (char)Math.min(minChar, unicodeValue);
                maxChar = (char)Math.max(maxChar, unicodeValue);
            }
        }

        // wczytaj obraz
        if (minChar < maxChar) {
            firstChar = minChar;
            characters = new Image[maxChar - minChar + 1];
            for (int i=0; i<files.length; i++) {
                int unicodeValue = getUnicodeValue(files[i]);
                if (unicodeValue != -1) {
                    int index = unicodeValue - firstChar;
                    characters[index] = new ImageIcon(files[i].getAbsolutePath()).getImage();
                }
            }

        }

    }


    private int getUnicodeValue(File file) throws NumberFormatException
    {
        String name = file.getName().toLowerCase();
        if (name.startsWith("char") && name.endsWith(".png")) {
            String unicodeString = name.substring(4, name.length() - 4);
            return Integer.parseInt(unicodeString);
        }
        return -1;
    }


    /**
        Zwraca obraz dla okrelonego znaku. Jeli dla danego znaku
        obraz nie istnieje, zwracany jest specjalny obraz dla
        "nieprawidowego" znaku.
    */
    public Image getImage(char ch) {
        int index = ch - firstChar;
        if (index < 0 || index >= characters.length ||
            characters[index] == null)
        {
            return invalidCharacter;
        }
        else {
            return characters[index];
        }
    }


    /**
        Zwraca (wyraon w pikselach) szeroko przekazanego acucha.
    */
    public int stringWidth(String s) {
        int width = 0;
        for (int i=0; i<s.length(); i++) {
            width += charWidth(s.charAt(i));
        }
        return width;
    }


    /**
        Zwraca (wyraon w pikselach) szeroko przekazanego znaku.
    */
    public int charWidth(char ch) {
        return getImage(ch).getWidth(null);
    }


    /**
        Zwraca (wyraon w pikselach) wysoko przekazanego znaku.
    */
    public int charHeight(char ch) {
        return getImage(ch).getHeight(null);
    }


    /**
        Rysuje przekazany acuch w punkcie (x, y).
    */
    public void drawString(Graphics g, String s, int x, int y) {
        drawString(g, s, x, y, LEFT | BOTTOM);
    }


    /**
        Rysuje przekazany acuch w punkcie (x, y).
    */
    public void drawString(Graphics g, String s, int x, int y, int anchor)
    {
        if ((anchor & HCENTER) != 0) {
            x-=stringWidth(s) / 2;
        }
        else if ((anchor & RIGHT) != 0) {
            x-=stringWidth(s);
        }
        // czyci poziome flagi dla rysowanych znakw
        anchor &= ~HCENTER;
        anchor &= ~RIGHT;

        // rysuje znaki
        for (int i=0; i<s.length(); i++) {
            drawChar(g, s.charAt(i), x, y, anchor);
            x+=charWidth(s.charAt(i));
        }
    }


    /**
        Rysuje przekazany znak w punkcie (x, y).
    */
    public void drawChar(Graphics g, char ch, int x, int y) {
        drawChar(g, ch, x, y, LEFT | BOTTOM);
    }


    /**
        Rysuje przekazany znak w punkcie (x, y).
    */
    public void drawChar(Graphics g, char ch, int x, int y, int anchor)
    {
        if ((anchor & HCENTER) != 0) {
            x-=charWidth(ch) / 2;
        }
        else if ((anchor & RIGHT) != 0) {
            x-=charWidth(ch);
        }

        if ((anchor & VCENTER) != 0) {
            y-=charHeight(ch) / 2;
        }
        else if ((anchor & BOTTOM) != 0) {
            y-=charHeight(ch);
        }
        g.drawImage(getImage(ch), x, y, null);
    }
}
