package com.brackeen.javagamebook.graphics3D;

import com.brackeen.javagamebook.math3D.*;

/**
    Klas ScanConverter wykorzystywalimy do rysowania wieloktw
    posortowanych od przodu do tyu (bez nakadania si pikseli). 
    Wielokty s dodawane do listy i przycinane w sposb umoliwiajcy
    ich dopasowanie do wieloktw ju wywietlonych. Przed narysowaniem
    kadej klatki naley wywoa metod clear().
*/
public class SortedScanConverter extends ScanConverter {

    protected static final int DEFAULT_SCANLIST_CAPACITY = 8;

    private SortedScanList[] viewScans;
    private SortedScanList[] polygonScans;
    private boolean sortedMode;

    /**
        Tworzy nowy obiekt klasy SortedScanConverter dla danego okna
        (obiektu klasy ViewWindow). Wasnoci obiektu ViewWindow mog
        si zmienia pomidzy operacjami konwersji cieek. Tryb
        sortowania jest domylnie wyczony, mona go jednak w kadej
        chwili zmieni wywoujc metod setSortedMode().
    */
    public SortedScanConverter(ViewWindow view) {
        super(view);
        sortedMode = false;
    }


    /**
        Czyci biec ciek. Wywouj t metod przed przystpieniem
        do rysowania kadej klatki.
    */
    public void clear() {
        if (viewScans != null) {
            for (int y=0; y<viewScans.length; y++) {
                viewScans[y].clear();
            }
        }
    }


    /**
        Ustawia tryb sortowanie, co oznacza, e ten konwerter cieek
        moe zakada, e wielokty s rysowane od przodu do tyu i
        powinny by przycinane w taki sposb, by pasoway do wieloktw
        wczeniej przekonwertowanych na cieki.
    */
    public void setSortedMode(boolean b) {
        sortedMode = b;
    }


    /**
        Zwraca n-t ciek danego rzdu.
    */
    public Scan getScan(int y, int index) {
        return polygonScans[y].getScan(index);
    }


    /**
        Zwraca liczb cieek w danym rzdzie.
    */
    public int getNumScans(int y) {
        return polygonScans[y].getNumScans();
    }


    /**
        Sprawdza, czy ekran jest wypeniony.
    */
    public boolean isFilled() {
        if (viewScans == null) {
            return false;
        }

        int left = view.getLeftOffset();
        int right = left + view.getWidth() - 1;
        for (int y=view.getTopOffset(); y<viewScans.length; y++) {
            if (!viewScans[y].equals(left, right)) {
                return false;
            }
        }
        return true;
    }


    protected void ensureCapacity() {
        super.ensureCapacity();
        int height = view.getTopOffset() + view.getHeight();
        int oldHeight = (viewScans == null)?0:viewScans.length;
        if (height != oldHeight) {
            SortedScanList[] newViewScans =
                new SortedScanList[height];
            SortedScanList[] newPolygonScans =
                new SortedScanList[height];
            if (oldHeight != 0) {
                System.arraycopy(viewScans, 0, newViewScans, 0,
                    Math.min(height, oldHeight));
                System.arraycopy(polygonScans, 0, newPolygonScans,
                    0, Math.min(height, oldHeight));
            }
            viewScans = newViewScans;
            polygonScans = newPolygonScans;
            for (int i=oldHeight; i<height; i++) {
                viewScans[i] = new SortedScanList();
                polygonScans[i] = new SortedScanList();
            }
        }
    }


    /**
        Przeksztaca wielokt na cieki i  jeli tryb sortedMode jest
        wczony - dodaje cieki do listy i odpowiednio je przycina, by
        piksele nowego wielokta nie przykryway pikseli ju istniejcych.
    */
    public boolean convert(Polygon3D polygon) {
        boolean visible = super.convert(polygon);
        if (!sortedMode || !visible) {
            return visible;
        }

        // przytnij ciek tak, by nie zasaniaa wczeniej dodanych punktw
        visible = false;
        for (int y=getTopBoundary(); y<=getBottomBoundary(); y++) {
            Scan scan = getScan(y);
            SortedScanList diff = polygonScans[y];
            diff.clear();
            if (scan.isValid()) {
                viewScans[y].add(scan.left, scan.right, diff);
                visible |= (polygonScans[y].getNumScans() > 0);
            }
        }

        return visible;

    }


    /**
        Klasa SortedScanList reprezentuje szereg cieek dla poziomego
        rzdu pikseli. Nowe cieki mog by dodawane i przycinane w taki
        sposb, by nie zasaniay istniejcych wczeniej dodanych cieek.
    */
    private static class SortedScanList {
    
        private int length;
        private Scan[] scans;
    
        /**
            Tworzy nowy obiekt klasy SortedScanList o domylnej pojemnoci
            (liczbie cieek na poziomy rzd pikseli).
        */
        public SortedScanList() {
            this(DEFAULT_SCANLIST_CAPACITY);
        }
    
    
        /**
            Tworzy nowy obiekt klasy SortedScanList o podanej pojemnoci
            (liczbie cieek na poziomy rzd pikseli).
        */
        public SortedScanList(int capacity) {
            scans = new Scan[capacity];
            for (int i=0; i<capacity; i++) {
                scans[i] = new Scan();
            }
            length = 0;
        }
    
    
        /**
            Czyci t list cieek.
        */
        public void clear() {
            length = 0;
        }
    
    
        /**
            Zwraca liczb cieek na tej licie.
        */
        public int getNumScans() {
            return length;
        }
    
    
        /**
            Zwraca n-t ciek z tej listy.
        */
        public Scan getScan(int index) {
            return scans[index];
        }
    
    
        /**
            Sprawdza, czy ta lista cieek zawiera tylko jedn ciek i czy ta
            cieka jest rwna przekazanym wartociom argumentw left i right.
        */
        public boolean equals(int left, int right) {
            return (length == 1 && scans[0].equals(left,right));
        }
    
    
        /**
            Dodaje i przycina now ciek w rzdzie, umieszcza widoczny
            fragment (rnic) w okrelonym obiekcie klasy SortedScanList.
        */
        public void add(int left, int right, SortedScanList diff) {
            for (int i=0; i<length && left <= right; i++) {
                Scan scan = scans[i];
                int maxRight = scan.left - 1;
                if (left <= maxRight) {
                    if (right < maxRight) {
                        diff.add(left, right);
                        insert(left, right, i);
                        return;
                    }
                    else {
                        diff.add(left, maxRight);
                        scan.left = left;
                        left = scan.right + 1;
                        if (merge(i)) {
                            i--;
                        }
                    }
                }
                else if (left <= scan.right) {
                    left = scan.right + 1;
                }
            }
            if (left <= right) {
                insert(left, right, length);
                diff.add(left, right);
            }
    
        }
    
        // metody pomocnicze wykorzystywane przez metod add()
    
        private void growCapacity() {
            int capacity = scans.length;
            int newCapacity = capacity*2;
            Scan[] newScans = new Scan[newCapacity];
            System.arraycopy(scans, 0, newScans, 0, capacity);
            for (int i=length; i<newCapacity; i++) {
                newScans[i] = new Scan();
            }
            scans = newScans;
        }
    
        private void add(int left, int right) {
            if (length == scans.length) {
                growCapacity();
            }
            scans[length].setTo(left, right);
            length++;
        }
    
        private void insert(int left, int right, int index) {
            if (index > 0) {
                Scan prevScan = scans[index-1];
                if (prevScan.right == left - 1) {
                    prevScan.right = right;
                    return;
                }
            }
    
            if (length == scans.length) {
                growCapacity();
            }
            Scan last = scans[length];
            last.setTo(left, right);
            for (int i=length; i>index; i--) {
                scans[i] = scans[i-1];
            }
            scans[index] = last;
            length++;
        }
    
        private void remove(int index) {
            Scan removed = scans[index];
            for (int i=index; i<length-1; i++) {
                scans[i] = scans[i+1];
            }
            scans[length-1] = removed;
            length--;
        }
    
        private boolean merge(int index) {
            if (index > 0) {
                Scan prevScan = scans[index-1];
                Scan thisScan = scans[index];
                if (prevScan.right == thisScan.left-1) {
                    prevScan.right = thisScan.right;
                    remove(index);
                    return true;
                }
            }
            return false;
        }

    }

}
