package com.brackeen.javagamebook.game;

import com.brackeen.javagamebook.bsp2D.*;
import com.brackeen.javagamebook.math3D.*;
import com.brackeen.javagamebook.game.GameObject;

/**
    Klasa CollisionDetection obsuguje wykrywanie wzajemnych kolizji
    obiektw klasy GameObjects, oraz kolizji tych obiektw z obiektami
    przechowywanymi w drzewie BSP. W przypadku wykrycia kolizji, obiekt
    GameObject nie zatrzymuje si, tylko przesuwa wzdu obiektu, z
    ktrym koliduje.
*/
public class CollisionDetectionWithSliding
    extends CollisionDetection
{

    private Vector3D scratch = new Vector3D();
    private Vector3D originalLocation = new Vector3D();

    /**
        Tworzy nowy obiekt klasy CollisionDetectionWithSliding
        dla przekazanego drzewa BSP.
    */
    public CollisionDetectionWithSliding(BSPTree bspTree) {
        super(bspTree);
    }


    /**
        Sprawdza, czy obiekt w grze nie koliduje ze cianami reprezentowanymi
        w drzewie BSP. Zwraca pierwsz cian, dla ktrej wystpuje kolizja, lub
        null, jeli obiekt nie koliduje z adn cian. Jeli metoda wykryje
        kolizj, obiekt przemieszcza si wzdu ciany i sprawdza, czy nie
        wystpuj kolejne kolizje. Jeli w czasie ruchu wzdu ciany nastpi
        nowa kolizja, obiekt jest przywracany na swoj pierwotn pozycj.
    */
    public BSPPolygon checkWalls(GameObject object, Vector3D oldLocation, long elapsedTime)
    {
    
        float goalX = object.getX();
        float goalZ = object.getZ();
    
        BSPPolygon wall = super.checkWalls(object, oldLocation, elapsedTime);
        // jeli wykryto kolizj i obiekt si sam nie zatrzyma
        if (wall != null && object.getTransform().isMoving()) {
            float actualX = object.getX();
            float actualZ = object.getZ();
    
            // iloczyn skalarny wektora prostopadego do ciany i linii prowadzcej do celu
            scratch.setTo(actualX, 0, actualZ);
            scratch.subtract(goalX, 0, goalZ);
            float length = scratch.getDotProduct(wall.getNormal());
    
            float slideX = goalX + length * wall.getNormal().x;
            float slideZ = goalZ + length * wall.getNormal().z;
    
            object.getLocation().setTo(slideX, object.getY(), slideZ);
            originalLocation.setTo(oldLocation);
            oldLocation.setTo(actualX, oldLocation.y, actualZ);
    
            // uyj mniejszego promienia otoczenia podczas ruchu wzdu ciany
            PolygonGroupBounds bounds = object.getBounds();
            float originalRadius = bounds.getRadius();
            bounds.setRadius(originalRadius-1);
    
            // sprawd wystpowanie kolizji w czasie ruchu wzdu ciany
            BSPPolygon wall2 = super.checkWalls(object, oldLocation, elapsedTime);
    
            // przywr zmienione parametry
            oldLocation.setTo(originalLocation);
            bounds.setRadius(originalRadius);
    
            if (wall2 != null) {
                object.getLocation().setTo(
                    actualX, object.getY(), actualZ);
                return wall2;
            }
        }
    
        return wall;
    }


    /**
        Sprawdza, czy obiekt koliduje z podog i sufitem.
        Uywa metod object.getFloorHeight() i object.getCeilHeight()
        do pozyskania wartoci definiujcych podog i sufit. Stosuje
        przyspieszenie ziemskie dla obiektu znajdujcego si w
        powietrzu i pynne przyspieszenie w gr dla gracza
        znajdujcego si na wysokoci mniejszej ni wynosi wysoko
        podogi (celem uzyskania efektu pynnego wchodzenia po schodach).
    */
    protected void checkFloorAndCeiling(GameObject object, long elapsedTime)
    {
        float floorHeight = object.getFloorHeight();
        float ceilHeight = object.getCeilHeight();
        float bottomHeight = object.getBounds().getBottomHeight();
        float topHeight = object.getBounds().getTopHeight();
        Vector3D v = object.getTransform().getVelocity();
        Physics physics = Physics.getInstance();
    
        // sprawdza, czy obiekt stoi na pododze
        if (object.getY() + bottomHeight == floorHeight) {
            if (v.y < 0) {
                v.y = 0;
            }
        }
        // sprawdza, czy obiekt nie znajduje si poniej poziomu podogi
        else if (object.getY() + bottomHeight < floorHeight) {
    
            if (!object.isFlying()) {
                // jeli opada
                if (v.y < 0) {
                    object.getListener().notifyFloorCollision(object);
                    v.y = 0;
                    object.getLocation().y = floorHeight - bottomHeight;
                }
                else if (!object.isJumping()) {
                    physics.scootUp(object, elapsedTime);
                }
            }
            else {
                object.getListener().notifyFloorCollision(object);
                v.y = 0;
                object.getLocation().y = floorHeight - bottomHeight;
            }
        }
        // sprawdza, czy obiekt nie uderza w sufit
        else if (object.getY() + topHeight > ceilHeight) {
           object.getListener().notifyCeilingCollision(object);
            if (v.y > 0) {
                v.y = 0;
            }
            object.getLocation().y = ceilHeight - topHeight;
            if (!object.isFlying()) {
                physics.applyGravity(object, elapsedTime);
            }
        }
        // nad podog
        else {
            if (!object.isFlying()) {
                // zatrzymaj ruch po schodach w gr (jeli obiekt jest w takim ruchu)
                if (v.y > 0 && !object.isJumping()) {
                    v.y = 0;
                    object.getLocation().y = floorHeight - bottomHeight;
                }
                else {
                    physics.applyGravity(object, elapsedTime);
                }
            }
        }


    }


    /**
        Obsuguje kolizj obiektu. Obiekt A jest obiektem poruszajcym
        si, Obiekt B jest tym obiektem, z ktrym obiekt A koliduje.
        Obiekt A przesuwa si wzdu krawdzi obiektu B lub (jeli to
        moliwe) przekracza ten obiekt.
    */
    protected boolean handleObjectCollision(GameObject objectA, GameObject objectB, float distSq, float minDistSq, Vector3D oldLocation)
    {
        objectA.getListener().notifyObjectCollision(objectA, objectB);
    
        if (objectB.getPolygonGroup().isEmpty()) {
            return false;
        }
                
        if (objectA.isFlying()) {
            return true;
        }
    
        float stepSize = objectA.getBounds().getTopHeight() / 6;
        Vector3D velocity = objectA.getTransform().getVelocity();
    
        // jeli to moliwe, sta na obiekcie statycznym
        float objectABottom = objectA.getY() + objectA.getBounds().getBottomHeight();
        float objectBTop = objectB.getY() + objectB.getBounds().getTopHeight();
        if (objectABottom + stepSize > objectBTop && objectBTop + objectA.getBounds().getTopHeight() < objectA.getCeilHeight())
        {
            objectA.getLocation().y = (objectBTop - objectA.getBounds().getBottomHeight());
            if (velocity.y < 0) {
                objectA.setJumping(false);
                // uwzgldnij dziaanie siy grawitacji
                velocity.y = -.01f;
            }
            return false;
        }
    
        if (objectA.getX() != oldLocation.x || objectA.getZ() != oldLocation.z)
        {
            // przesuwaj obiekt wzdu krawdzi obiektu statycznego
            float slideDistFactor = (float)Math.sqrt(minDistSq / distSq) - 1;
            scratch.setTo(objectA.getX(), 0, objectA.getZ());
            scratch.subtract(objectB.getX(), 0, objectB.getZ());
            scratch.multiply(slideDistFactor);
            objectA.getLocation().add(scratch);
    
            // jeli obiekt koliduje ze cian, przywr go do pierwotnego pooenia
            if (super.checkWalls(objectA, oldLocation, 0) != null)
            {
                return true;
            }
    
            return false;
        }
    
        return true;
    }



}
