package com.brackeen.javagamebook.math3D;

/**
    Klasa PointLight3D to punktowe rdo wiata o ustalonej
    intensywnoci(midzy 0 a 1) i opcjonalnie odlegoci wygasania,
    ktra pozwala na osabianie wiata wraz z odlegoci.
*/
public class PointLight3D extends Vector3D {

    public static final float NO_DISTANCE_FALLOFF = -1;

    private float intensity;
    private float distanceFalloff;

    /**
        Tworzy nowe wiato PointLight3D w punkcie (0,0,0) o intensywnoci
        1 i niewygasajce wraz z odlegoci.
    */
    public PointLight3D() {
        this(0,0,0, 1, NO_DISTANCE_FALLOFF);
    }


    /**
        Tworzy kopi podanego rda wiata PointLight3D.
    */
    public PointLight3D(PointLight3D p) {
        setTo(p);
    }


    /**
        Tworzy nowe wiato PointLight3D o podanej lokacji
        i intensywnoci. wiato nie wygasa wraz z odlegoci.
    */
    public PointLight3D(float x, float y, float z,
        float intensity)
    {
        this(x, y, z, intensity, NO_DISTANCE_FALLOFF);
    }


    /**
        Tworzy nowe wiato PointLight3D o podanej lokacji
        i intensywnoci. wiato nie wygasa wraz z odlegoci.
    */
    public PointLight3D(float x, float y, float z,
        float intensity, float distanceFalloff)
    {
        setTo(x, y, z);
        setIntensity(intensity);
        setDistanceFalloff(distanceFalloff);
    }


    /**
        Przypisuje temu wiatu PointLight3D t sam lokalizacj, intensywno
        i odlego wygasania, ktre ma podane wiato PointLight3D.
    */
    public void setTo(PointLight3D p) {
        setTo(p.x, p.y, p.z);
        setIntensity(p.getIntensity());
        setDistanceFalloff(p.getDistanceFalloff());
    }


    /**
        Pobiera intensywno wiata z tego rda w 
        okrelonej odlegoci od niego.
    */
    public float getIntensity(float distance) {
        if (distanceFalloff == NO_DISTANCE_FALLOFF) {
            return intensity;
        }
        else if (distance >= distanceFalloff) {
            return 0;
        }
        else {
            return intensity * (distanceFalloff - distance)
                / (distanceFalloff + distance);
        }
    }

    /**
        Pobiera intensywno tego wiata.
    */
    public float getIntensity() {
        return intensity;
    }


    /**
        Ustawia intensywno tego wiata.
    */
    public void setIntensity(float intensity) {
        this.intensity = intensity;
    }


    /**
        Pobiera odlego wygasania. Po jej przekroczeniu
        intensywno wiata spada do zera.
    */
    public float getDistanceFalloff() {
        return distanceFalloff;
    }


    /**
        Ustawia odlego wygasania. Po jej przekroczeniu
        intensywno wiata spada do zera. NO_DISTANCE_FALLOFF
        jesli wiato nie powinno wygasa wraz z odlegoci.
    */
    public void setDistanceFalloff(float distanceFalloff) {
        this.distanceFalloff = distanceFalloff;
    }

}
