package com.brackeen.javagamebook.game;

import com.brackeen.javagamebook.math3D.*;

/**
    Klasa GameObject jest podstaw dla wszystkich typw obiektw
    wystpujcych w grze, ktre s reprezentowane za pomoc obiektu
    klasy PolygonGroup. Przykadowo, GameObject moe by obiektem
    statycznym (np. skrzyni), obiektem w ruchu (np. pociskiem lub
    przeciwnikiem) lub dowolnym innym typem obiektu (np. apteczk).
    Obiekty klasy GameObject mog si znajdowa w trzech podstawowych
    stanach: bezczynnoci (STATE_IDLE), aktywnoci (STATE_ACTIVE) i 
    zniszczenia (STATE_DESTROYED).
*/
public class GameObject {
    protected static final int STATE_IDLE = 0;
    protected static final int STATE_ACTIVE = 1;
    protected static final int STATE_DESTROYED = 2;

    private PolygonGroup polygonGroup;
    private int state;

    /**
        Tworzy nowy GameObject reprezentowany przez przekazany obiekt 
        klasy PolygonGroup. Przekazany obiekt moe mie warto null.
    */
    public GameObject(PolygonGroup polygonGroup) {
        this.polygonGroup = polygonGroup;
        state = STATE_IDLE;
    }


    /**
        Zwraca pooenie tego obiektu GameObject w oparciu o dane
        obiektu klasy Transform3D.
    */
    public Vector3D getLocation() {
        return polygonGroup.getTransform().getLocation();
    }


    /**
        Zwraca obiekt reprezentujcy ruch obiektu w grze.
    */
    public MovingTransform3D getTransform() {
        return polygonGroup.getTransform();
    }


    /**
        Zwraca obiekt klasy PolygonGroup dla tego obiektu w grze.
    */
    public PolygonGroup getPolygonGroup() {
        return polygonGroup;
    }


    /**
        Zwraca wsprzdn X tego obiektu klasy GameObject.
    */
    public float getX() {
        return getLocation().x;
    }


    /**
        Zwraca wsprzdn Y tego obiektu klasy GameObject.
    */
    public float getY() {
        return getLocation().y;
    }


    /**
        Zwraca wsprzdn Z tego obiektu klasy GameObject.
    */
    public float getZ() {
        return getLocation().z;
    }


    /**
        Ustawia stan tego obiektu. Parametr state powinien mie
        warto STATE_IDLE, STATE_ACTIVE lub STATE_DESTROYED.
    */
    protected void setState(int state) {
        this.state = state;
    }


    /**
        Ustawia stan wskazanego obiektu. Dziki temu dowolny obiekt
        GameObject moe zmieni stan dowolnego innego obiektu GameObject.
        Parametr state powinien mie warto STATE_IDLE, STATE_ACTIVE lub
        STATE_DESTROYED.
    */
    protected void setState(GameObject object, int state) {
        object.setState(state);
    }


    /**
        Zwraca true, jeli ten obiekt GameObject jest w stanie bezczynnoci.
    */
    public boolean isIdle() {
        return (state == STATE_IDLE);
    }


    /**
        Zwraca true, jeli ten obiekt GameObject jest w stanie aktywnoci.
    */
    public boolean isActive() {
        return (state == STATE_ACTIVE);
    }


    /**
        Zwraca true, jeli ten obiekt GameObject jest w stanie zniszczenia.
    */
    public boolean isDestroyed() {
        return (state == STATE_DESTROYED);
    }


    /**
        Jeli ten GameObject znajduje si w stanie aktywnoci, ta
        metoda aktualizuje jego obiekt PolygonGroup. W przeciwnym
        przypadku, metoda nie wykonuje adnych dziaa.
    */
    public void update(GameObject player, long elapsedTime) {
        if (isActive()) {
            polygonGroup.update(elapsedTime);
        }
    }


    /**
        Informuje ten GameObject o tym, czy by widoczny, czy nie,
        podczas ostatniej aktualizacji. Domylnie, jeli ten GameObject
        znajduje si w stanie bezczynnoci i otrzymuje informacj, e
        jest widoczny, przechodzi w stan aktywnoci.
    */
    public void notifyVisible(boolean visible) {
        if (visible && isIdle()) {
            state = STATE_ACTIVE;
        }
    }

}
