package com.brackeen.javagamebook.sound;

import java.io.*;
import javax.sound.midi.*;

public class MidiPlayer implements MetaEventListener {

    // Zdarzenie meta MIDI:
    public static final int END_OF_TRACK_MESSAGE = 47;

    private Sequencer sequencer;
    private boolean loop;
    private boolean paused;

    /**
        Tworzy nowy obiekt MidiPlayer.
    */
    public MidiPlayer() {
        try {
            sequencer = MidiSystem.getSequencer();
            sequencer.open();
            sequencer.addMetaEventListener(this);
        }
        catch ( MidiUnavailableException ex) {
            sequencer = null;
        }
    }

    /**
        adowanie sekwencji z systemu plikw. Zwraca warto null, gdy wystpi bd.
    */
    public Sequence getSequence(String filename) {
        try {
            return getSequence(new FileInputStream(filename));
        }
        catch (IOException ex) {
            ex.printStackTrace();
            return null;
        }
    }

    /**
        adowanie sekwencji ze strumienia wejciowego. 
        Zwraca warto null, gdy wystpi bd.
    */
    public Sequence getSequence(InputStream is) {
        try {
            if (!is.markSupported()) {
                is = new BufferedInputStream(is);
            }
            Sequence s = MidiSystem.getSequence(is);
            is.close();
            return s;
        }
        catch (InvalidMidiDataException ex) {
            ex.printStackTrace();
            return null;
        }
        catch (IOException ex) {
            ex.printStackTrace();
            return null;
        }
    }

    /**
        Odtwarza sekwencj, ewentualnie j zaptlajc. Metoda natychmiast 
        koczy dziaanie. Jeeli sekwencja jest nieprawidowa, 
        nie jest odtwarzana.
    */
    public void play(Sequence sequence, boolean loop) {
        if (sequencer != null && sequence != null && sequencer.isOpen()) {
            try {
                sequencer.setSequence(sequence);
                sequencer.start();
                this.loop = loop;
            }
            catch (InvalidMidiDataException ex) {
                ex.printStackTrace();
            }
        }
    }

    /**
        Metoda wywoywana przez system dwikowy w przypadku zajcia 
        zdarzenia meta. W tym przypadku, gdy zdarzeniem jest koniec cieki
        i gdy jest wczone zaptlanie, sekwencja jest uruchamiana od pocztku.
    */
    public void meta(MetaMessage event) {
        if (event.getType() == END_OF_TRACK_MESSAGE) {
            if (sequencer != null && sequencer.isOpen() && loop) {
                sequencer.start();
            }
        }
    }

     /**
       Zatrzymuje sekwenser i ustawia jego pozycj na 0.
    */
    public void stop() {
         if (sequencer != null && sequencer.isOpen()) {
             sequencer.stop();
             sequencer.setMicrosecondPosition(0);
         }
    }

     /**
        Zamyka sekwenser.
    */
    public void close() {
         if (sequencer != null && sequencer.isOpen()) {
             sequencer.close();
         }
    }

    /**
        Pobiera sekwenser.
    */
    public Sequencer getSequencer() {
        return sequencer;
    }

    /**
        Wcza pauz. Muzyka moe nie zatrzyma si natychmiast.
    */
    public void setPaused(boolean paused) {
        if (this.paused != paused && sequencer != null && sequencer.isOpen()) {
            this.paused = paused;
            if (paused) {
                sequencer.stop();
            }
            else {
                sequencer.start();
            }
        }
    }

    /**
        Zwraca stan pauzy.
    */
    public boolean isPaused() {
        return paused;
    }
}
