package com.brackeen.javagamebook.input;

import java.awt.*;
import java.awt.event.*;
import java.util.List;
import java.util.ArrayList;
import javax.swing.SwingUtilities;

/**
    InputManager zarzdza wejciem z klawiatury i zdarze myszy.
    Zdarzenia s kojarzone z obiektami GameAction.
*/
public class InputManager implements KeyListener, MouseListener,
    MouseMotionListener, MouseWheelListener
{
    /**
        Niewidoczny kursor.
    */
    public static final Cursor INVISIBLE_CURSOR =
        Toolkit.getDefaultToolkit().createCustomCursor(
            Toolkit.getDefaultToolkit().getImage(""),
            new Point(0,0),
            "invisible");

    // Kody myszy:
    public static final int MOUSE_MOVE_LEFT = 0;
    public static final int MOUSE_MOVE_RIGHT = 1;
    public static final int MOUSE_MOVE_UP = 2;
    public static final int MOUSE_MOVE_DOWN = 3;
    public static final int MOUSE_WHEEL_UP = 4;
    public static final int MOUSE_WHEEL_DOWN = 5;
    public static final int MOUSE_BUTTON_1 = 6;
    public static final int MOUSE_BUTTON_2 = 7;
    public static final int MOUSE_BUTTON_3 = 8;

    private static final int NUM_MOUSE_CODES = 9;

    // Kody klawiszy s zdefiniowane w pakiecie java.awt.KeyEvent.
    // Wikszo kodw (poza niektrymi, rzadko wykorzystywanymi, 
    // na przykad "alt graph") jest mniejsza ni 600.
    private static final int NUM_KEY_CODES = 600;

    private GameAction[] keyActions =
        new GameAction[NUM_KEY_CODES];
    private GameAction[] mouseActions =
        new GameAction[NUM_MOUSE_CODES];

    private Point mouseLocation;
    private Point centerLocation;
    private Component comp;
    private Robot robot;
    private boolean isRecentering;

    /**
        Tworzy nowy obiekt InputManager, odbierajcy 
        dane wejciowe dla okrelonego komponentu.
    */
    public InputManager(Component comp) {
        this.comp = comp;
        mouseLocation = new Point();
        centerLocation = new Point();

        // Rejestracja klawiszy i nasuchw dla myszy:
        comp.addKeyListener(this);
        comp.addMouseListener(this);
        comp.addMouseMotionListener(this);
        comp.addMouseWheelListener(this);

        // Umoliwia przechwytywanie klawisza TAB oraz pozostaych
        // klawiszy, wykorzystywanych normalnie do zmiany fokusu.
        comp.setFocusTraversalKeysEnabled(false);
    }

    /**
        Ustawia kursor w biecym komponencie wejciowym obiektu InputManager.
    */
    public void setCursor(Cursor cursor) {
        comp.setCursor(cursor);
    }

    /**
        Ustawia tryb wzgldny myszy. W trybie wzgldnym kursor
        myszy jest "zablokowany" w rodku ekranu; mierzone s
        wwczas jedynie przesunicia wskanika. W trybie normalnym 
        kursor myszy moe porusza si swobodnie po ekranie.
    */
    public void setRelativeMouseMode(boolean mode) {
        if (mode == isRelativeMouseMode()) {
            return;
        }

        if (mode) {
            try {
                robot = new Robot();
                recenterMouse();
            }
            catch (AWTException ex) {
                // Nie mona utworzy robota!
                robot = null;
            }
        }
        else {
            robot = null;
        }
    }

    /**
        Sprawdza, czy mysz dziaa w trybie wzgldnym.
    */
    public boolean isRelativeMouseMode() {
        return (robot != null);
    }

    /**
        Kojarzy obiekt GameAction z okrelonym klawiszem. Kody
        klawiszy s zdefiniowane w java.awt.KeyEvent. Jeeli z klawiszem
        skojarzony ju jest obiekt GameAction, nowy obiekt zastpuje go.
    */
    public void mapToKey(GameAction gameAction, int keyCode) {
        keyActions[keyCode] = gameAction;
    }

    /**
        Kojarzy obiekt GameAction z okrelonym zdarzeniem myszy. Kody 
        zdarze s zdefiniowane w InputManager (MOUSE_MOVE_LEFT,
        MOUSE_BUTTON_1, itd.). Jeeli ze zdarzeniem myszy skojarzony 
        Jest ju obiekt GameAction, nowy obiekt go zastpuje.
    */
    public void mapToMouse(GameAction gameAction,
        int mouseCode)
    {
        mouseActions[mouseCode] = gameAction;
    }

    /**
        Usuwa z biecego obiektu GameAction wszystkie 
        skojarzone klawisze i zdarzenia myszy.
    */
    public void clearMap(GameAction gameAction) {
        for (int i=0; i<keyActions.length; i++) {
            if (keyActions[i] == gameAction) {
                keyActions[i] = null;
            }
        }

        for (int i=0; i<mouseActions.length; i++) {
            if (mouseActions[i] == gameAction) {
                mouseActions[i] = null;
            }
        }

        gameAction.reset();
    }

    /**
        Zwraca list nazw klawiszy i zdarze myszy skojarzonych z biecym
        obiektem GameAction. Kada pozycja w List jest typu String.
    */
    public List getMaps(GameAction gameCode) {
        ArrayList list = new ArrayList();

        for (int i=0; i<keyActions.length; i++) {
            if (keyActions[i] == gameCode) {
                list.add(getKeyName(i));
            }
        }

        for (int i=0; i<mouseActions.length; i++) {
            if (mouseActions[i] == gameCode) {
                list.add(getMouseName(i));
            }
        }
        return list;
    }

    /**
        Kasuje stan wszystkich obiektw GameAction, dziki czemu
        dziaaj one tak, jakby zden klawisz nie zosta nacinity.
    */
    public void resetAllGameActions() {
        for (int i=0; i<keyActions.length; i++) {
            if (keyActions[i] != null) {
                keyActions[i].reset();
            }
        }

        for (int i=0; i<mouseActions.length; i++) {
            if (mouseActions[i] != null) {
                mouseActions[i].reset();
            }
        }
    }

    /**
        Pobranie nazwy kodu zdarzenia klawisza.
    */
    public static String getKeyName(int keyCode) {
        return KeyEvent.getKeyText(keyCode);
    }

    /**
        Pobranie nazwy kodu zdarzenia myszy.
    */
    public static String getMouseName(int mouseCode) {
        switch (mouseCode) {
            case MOUSE_MOVE_LEFT: return "Mysz lewo";
            case MOUSE_MOVE_RIGHT: return "Mysz prawo";
            case MOUSE_MOVE_UP: return "Mysz gra";
            case MOUSE_MOVE_DOWN: return "Mysz d";
            case MOUSE_WHEEL_UP: return "Kko myszy gra";
            case MOUSE_WHEEL_DOWN: return "Kko myszy d";
            case MOUSE_BUTTON_1: return "Przycisk myszy 1";
            case MOUSE_BUTTON_2: return "Przycisk myszy 2";
            case MOUSE_BUTTON_3: return "Przycisk myszy 3";
            default: return "Nieznany kod zdarzenia myszy " + mouseCode;
        }
    }

    /**
        Zwraca wsprzdn x kursora myszy.
    */
    public int getMouseX() {
        return mouseLocation.x;
    }


    /**
        Zwraca wsprzdn y kursora myszy.
    */
    public int getMouseY() {
        return mouseLocation.y;
    }


    /**
        Korzysta z klasy Robot do ustawienia kursora myszy na rodku ekranu.
        <p>Naley pamita, e ta funkcja klasy Robot moe 
        nie by dostpna na wszystkich platformach.
    */
    private synchronized void recenterMouse() {
        if (robot != null && comp.isShowing()) {
            centerLocation.x = comp.getWidth() / 2;
            centerLocation.y = comp.getHeight() / 2;
            SwingUtilities.convertPointToScreen(centerLocation,
                comp);
            isRecentering = true;
            robot.mouseMove(centerLocation.x, centerLocation.y);
        }
    }

    private GameAction getKeyAction(KeyEvent e) {
        int keyCode = e.getKeyCode();
        if (keyCode < keyActions.length) {
            return keyActions[keyCode];
        }
        else {
            return null;
        }
    }

    /**
        Zwraca kod zdarzenia myszy dla przycisku 
        skojarzonego z biecym obiektem MouseEvent.
    */
    public static int getMouseButtonCode(MouseEvent e) {
         switch (e.getButton()) {
            case MouseEvent.BUTTON1:
                return MOUSE_BUTTON_1;
            case MouseEvent.BUTTON2:
                return MOUSE_BUTTON_2;
            case MouseEvent.BUTTON3:
                return MOUSE_BUTTON_3;
            default:
                return -1;
        }
    }

    private GameAction getMouseButtonAction(MouseEvent e) {
        int mouseCode = getMouseButtonCode(e);
        if (mouseCode != -1) {
             return mouseActions[mouseCode];
        }
        else {
             return null;
        }
    }

    // Z interfejsu KeyListener:
    public void keyPressed(KeyEvent e) {
        GameAction gameAction = getKeyAction(e);
        if (gameAction != null) {
            gameAction.press();
        }
        // Upewnij si, e zdarzenie nie bdzie dalej obsugiwane:
        e.consume();
    }


    // Z interfejsu KeyListener:
    public void keyReleased(KeyEvent e) {
        GameAction gameAction = getKeyAction(e);
        if (gameAction != null) {
            gameAction.release();
        }
        // Upewnij si, e zdarzenie nie bdzie dalej obsugiwane:
        e.consume();
    }

    // Z interfejsu KeyListener:
    public void keyTyped(KeyEvent e) {
        // Upewnij si, e zdarzenie nie bdzie dalej obsugiwane:
        e.consume();
    }

    // Z interfejsu MouseListener:
    public void mousePressed(MouseEvent e) {
        GameAction gameAction = getMouseButtonAction(e);
        if (gameAction != null) {
            gameAction.press();
        }
    }

    // Z interfejsu MouseListener:
    public void mouseReleased(MouseEvent e) {
        GameAction gameAction = getMouseButtonAction(e);
        if (gameAction != null) {
            gameAction.release();
        }
    }

    // Z interfejsu MouseListener:
    public void mouseClicked(MouseEvent e) {
        // Nic nie wykonuj.
    }

    // Z interfejsu MouseListener:
    public void mouseEntered(MouseEvent e) {
        mouseMoved(e);
    }

    // Z interfejsu MouseListener:
    public void mouseExited(MouseEvent e) {
        mouseMoved(e);
    }

    // Z interfejsu MouseMotionListener:
    public void mouseDragged(MouseEvent e) {
        mouseMoved(e);
    }

    // Z interfejsu MouseMotionListener
    public synchronized void mouseMoved(MouseEvent e) {
        // To zdarzenie jest generowane przez centrowanie myszy  zignoruj je.
        if (isRecentering &&
            centerLocation.x == e.getX() &&
            centerLocation.y == e.getY())
        {
            isRecentering = false;
        }
        else {
            int dx = e.getX() - mouseLocation.x;
            int dy = e.getY() - mouseLocation.y;
            mouseHelper(MOUSE_MOVE_LEFT, MOUSE_MOVE_RIGHT, dx);
            mouseHelper(MOUSE_MOVE_UP, MOUSE_MOVE_DOWN, dy);

            if (isRelativeMouseMode()) {
                recenterMouse();
            }
        }

        mouseLocation.x = e.getX();
        mouseLocation.y = e.getY();

    }

    // Z interfejsu MouseWheelListener:
    public void mouseWheelMoved(MouseWheelEvent e) {
        mouseHelper(MOUSE_WHEEL_UP, MOUSE_WHEEL_DOWN,
            e.getWheelRotation());
    }

    private void mouseHelper(int codeNeg, int codePos,
        int amount)
    {
        GameAction gameAction;
        if (amount < 0) {
            gameAction = mouseActions[codeNeg];
        }
        else {
            gameAction = mouseActions[codePos];
        }
        if (gameAction != null) {
            gameAction.press(Math.abs(amount));
            gameAction.release();
        }
    }
}
