package com.brackeen.javagamebook.input;

/**
    Klasa GameAction jest klas abstrakcyjn dla akcji inicjowanych
    przez uytkownika, na przykad poruszania si lub skakania. Obiekty
    GameAction mog by kojarzone z klawiszami poprzez InputManager.
*/
public class GameAction {

    /**
        Normalne dziaanie. Dopki klawisz jest przycinity,
        metoda isPressed() zwraca warto true.
    */
    public static final int NORMAL = 0;

    /**
        Pocztkowa obsuga przycinicia. Metoda isPressed() zwraca
        warto true jedynie po pierwszym naciniciu klawisza.
    */
    public static final int DETECT_INITAL_PRESS_ONLY = 1;

    private static final int STATE_RELEASED = 0;
    private static final int STATE_PRESSED = 1;
    private static final int STATE_WAITING_FOR_RELEASE = 2;

    private String name;
    private int behavior;
    private int amount;
    private int state;

    /**
        Tworzenie nowego obiektu GameAction z dziaaniem NORMAL.
    */
    public GameAction(String name) {
        this(name, NORMAL);
    }

    /**
        Tworzenie nowego obiektu GameAction z podanym dziaaniem.
    */
    public GameAction(String name, int behavior) {
        this.name = name;
        this.behavior = behavior;
        reset();
    }

    /**
        Zwraca nazw tego obiektu GameAction.
    */
    public String getName() {
        return name;
    }

    /**
        Kasuje stan obiektu GameAction, dziki czemu zachowuje 
        si on tak, jakby nie zosta nacinity klawisz.
    */
    public void reset() {
        state = STATE_RELEASED;
        amount = 0;
    }

    /**
        Programowe przycinicie w biecym obiekcie GameAction. 
        Ma to samo dziaanie, co wywoanie press(), a nastpnie release().
    */
    public synchronized void tap() {
        press();
        release();
    }

    /**
        Sygnalizacja nacinicia klawisza.
    */
    public synchronized void press() {
        press(1);
    }

    /**
        Sygnalizacja nacinicia klawisza okrelon liczb razy
        lub przesunicia myszy o okrelony dystans.
    */
    public synchronized void press(int amount) {
        if (state != STATE_WAITING_FOR_RELEASE) {
            this.amount+=amount;
            state = STATE_PRESSED;
        }
    }

    /**
        Sygnalizuje zwolnienie klawisza.
    */
    public synchronized void release() {
        state = STATE_RELEASED;
    }

    /**
        Zwraca informacj o tym, czy od ostatniego sprawdzenia 
        zosta nacinity jaki klawisz.
    */
    public synchronized boolean isPressed() {
        return (getAmount() != 0);
    }

    /**
        W przypadku klawiszy jest to liczba nacini klawisza 
        od ostatniego sprawdzenia.
        W przypadku przesunicia myszy jest to odlego.
    */
    public synchronized int getAmount() {
        int retVal = amount;
        if (retVal != 0) {
            if (state == STATE_RELEASED) {
                amount = 0;
            }
            else if (behavior == DETECT_INITAL_PRESS_ONLY) {
                state = STATE_WAITING_FOR_RELEASE;
                amount = 0;
            }
        }
        return retVal;
    }
}
