import java.io.*;
import javax.sound.sampled.*;

/**
    Klasa SimpleSoundPlayer obsuguje otwieranie pliku dwikowego
    z systemu plikw i jego pniejsze otwieranie.
*/
public class SimpleSoundPlayer  {

    public static void main(String[] args) {
        // adowanie dwiku.
        SimpleSoundPlayer sound =
            new SimpleSoundPlayer("../sounds/voice.wav");

        // Tworzenie strumienia dwikowego.
        InputStream stream =
            new ByteArrayInputStream(sound.getSamples());

        // Odtwarzanie dwiku.
        sound.play(stream);

        // Koniec.
        System.exit(0);
    }

    private AudioFormat format;
    private byte[] samples;

    /**
        Otwieranie dwiku z pliku.
    */
    public SimpleSoundPlayer(String filename) {
        try {
            // Otwieranie strumienia wejcia dwikowego.
            AudioInputStream stream =
                AudioSystem.getAudioInputStream(
                new File(filename));

            format = stream.getFormat();

            // Pobieranie prbek dwikowych.
            samples = getSamples(stream);
        }
        catch (UnsupportedAudioFileException ex) {
            ex.printStackTrace();
        }
        catch (IOException ex) {
            ex.printStackTrace();
        }
    }

    /**
        Pobieranie prbek dwiku jako tablicy bajtw.
    */
    public byte[] getSamples() {
        return samples;
    }

    /**
        Pobranie prbek z AudioInputStream w postaci tablicy bajtw.
    */
    private byte[] getSamples(AudioInputStream audioStream) {
        // Pobranie liczby bajtw do odczytania.
        int length = (int)(audioStream.getFrameLength() *
            format.getFrameSize());

        // Odczytanie caego strumienia.
        byte[] samples = new byte[length];
        DataInputStream is = new DataInputStream(audioStream);
        try {
            is.readFully(samples);
        }
        catch (IOException ex) {
            ex.printStackTrace();
        }

        // Zwrcenie prbek.
        return samples;
    }

    /**
        Odtwarzanie strumienia. Metoda ta jest blokowana (nie wraca), 
        dopki nie zostanie zakoczenie odtwarzanie dwiku.
    */
    public void play(InputStream source) {

        // Wykorzystujemy krtki - dugoci 100 ms (1/10 sekundy) - bufor 
        // do zmiany strumienia dwiku w czasie rzeczywistym
        int bufferSize = format.getFrameSize() *
            Math.round(format.getSampleRate() / 10);
        byte[] buffer = new byte[bufferSize];

        // Tworzenie obiektu Line do odtwarzania dwiku.
        SourceDataLine line;
        try {
            DataLine.Info info =
                new DataLine.Info(SourceDataLine.class, format);
            line = (SourceDataLine)AudioSystem.getLine(info);
            line.open(format, bufferSize);
        }
        catch (LineUnavailableException ex) {
            ex.printStackTrace();
            return;
        }

        // Uruchomienie odtwarzania.
        line.start();

        // Kopiowanie danych do obiektu Line.
        try {
            int numBytesRead = 0;
            while (numBytesRead != -1) {
                numBytesRead =
                    source.read(buffer, 0, buffer.length);
                if (numBytesRead != -1) {
                   line.write(buffer, 0, numBytesRead);
                }
            }
        }
        catch (IOException ex) {
            ex.printStackTrace();
        }

        // Oczekiwanie na zakoczenie odtwarzania i zamknicie obiektu Line.
        line.drain();
        line.close();

    }
}
