package com.brackeen.javagamebook.math3D;

/**
    Klasa MovingTransform3D jest klas potomn klasy Transform3D
    i obsuguje ruch postpowy i ruch obrotowy wok osi x, y i z.
*/
public class MovingTransform3D extends Transform3D {

    public static final int FOREVER = -1;

    // Vector3D uywany do oblicze
    private static Vector3D temp = new Vector3D();

    // szybko (w jednostkach na milisekund)
    private Vector3D velocity;
    private Movement velocityMovement;

    // prdko ruchu obrotowego (w radianach na milisekund)
    private Movement velocityAngleX;
    private Movement velocityAngleY;
    private Movement velocityAngleZ;

    /**
        Tworzy nowy obiekt MovingTransform3D
    */
    public MovingTransform3D() {
        init();
    }


    /**
        Tworzy nowy obiekt MovingTransform3D w oparciu o te same
        wartoci, ktre okrelono dla Transform3D.
    */
    public MovingTransform3D(Transform3D v) {
        super(v);
        init();
    }


    protected void init() {
        velocity = new Vector3D(0,0,0);
        velocityMovement = new Movement();
        velocityAngleX = new Movement();
        velocityAngleY = new Movement();
        velocityAngleZ = new Movement();
    }


    public Object clone() {
        return new MovingTransform3D(this);
    }


    /**
        Aktualizuje obiekt Transform3D w oparciu o przekazany czas ruchu,
        ktry upyn. Aktualizowane jest pooenie i kty obiektu.
    */
    public void update(long elapsedTime) {
        float delta = velocityMovement.getDistance(elapsedTime);
        if (delta != 0) {
            temp.setTo(velocity);
            temp.multiply(delta);
            location.add(temp);
        }

        rotateAngle(
            velocityAngleX.getDistance(elapsedTime),
            velocityAngleY.getDistance(elapsedTime),
            velocityAngleZ.getDistance(elapsedTime));
    }


    /**
        Zatrzymuje obiekt Transform3D. Wartoci wszystkich wektorw ruchu
        s zerowane.
    */
    public void stop() {
        velocity.setTo(0,0,0);
        velocityMovement.set(0,0);
        velocityAngleX.set(0,0);
        velocityAngleY.set(0,0);
        velocityAngleZ.set(0,0);
    }


    /**
        Ustawia wartoci dla ruchu do okrelonego miejsca
        z okrelon szybkoci.
    */
    public void moveTo(Vector3D destination, float speed) {
        temp.setTo(destination);
        temp.subtract(location);

        // oblicza czas potrzebny na przebycie wyznaczonej 
        // odlegoci
        float distance = temp.length();
        long time = (long)(distance / speed);

        // normalizuje wektor kierunku
        temp.divide(distance);
        temp.multiply(speed);

        setVelocity(temp, time);
    }


    /**
        Zwraca true, jeli obiekt jest w ruchu.
    */
    public boolean isMoving() {
        return !velocityMovement.isStopped() && !velocity.equals(0,0,0);
    }


    /**
        Returns true if currently moving, ignoring the y movement.
    */
    public boolean isMovingIgnoreY() {
        return !velocityMovement.isStopped() &&
            (velocity.x != 0 || velocity.z != 0);
    }


    /**
        Zwraca pozostay czas, w ktrym obiekt bdzie w ruchu.
    */
    public long getRemainingMoveTime() {
        if (!isMoving()) {
            return 0;
        }
        else {
            return velocityMovement.remainingTime;
        }
    }


    /**
        Zwraca wektor ruchu. Jeli wektor jest bezporednio
        modyfikowany, wywoaj metod setVelocity(), aby upewni
        si, e zmiany zostan uwzgldnione.
    */
    public Vector3D getVelocity() {
        return velocity;
    }


    /**
        Ustawia warto szybkoci dla danego wektora.
    */
    public void setVelocity(Vector3D v) {
        setVelocity(v, FOREVER);
    }


    /**
        Ustawia szybko. Warto ta jest automatycznie zerowana
        w momencie, gdy upynie przekazana ilo czasu (liczba
        milisekund). Jeli przekazano warto FOREVER, szybko
        nigdy nie zostanie zredukowana do zera.
    */
    public void setVelocity(Vector3D v, long time) {
        if (velocity != v) {
            velocity.setTo(v);
        }
        if (v.x == 0 && v.y == 0 && v.z == 0) {
            velocityMovement.set(0, 0);
        }
        else {
            velocityMovement.set(1, time);
        }

    }


    /**
        Dodaje przekazan warto do aktualnej szybkoci ruchu.
        Jeli ten obiekt MovingTransform3D jest aktualnie w ruchu,
        jego czas pozostay do zatrzymania nie ulegnie zmianie.
        W przeciwnym przypadku, pozostay czas bdzie mia warto
        FOREVER.
    */
    public void addVelocity(Vector3D v) {
        if (isMoving()) {
            velocity.add(v);
        }
        else {
            setVelocity(v);
        }
    }


    /**
        Obraca obiekt wok osi x do danego kta z okrelon
        prdkoci.
    */
    public void turnXTo(float angleDest, float speed) {
        turnTo(velocityAngleX, getAngleX(), angleDest, speed);
    }


    /**
        Obraca obiekt wok osi y do danego kta z okrelon
        prdkoci.
    */
    public void turnYTo(float angleDest, float speed) {
        turnTo(velocityAngleY, getAngleY(), angleDest, speed);
    }


    /**
        Obraca obiekt wok osi z do danego kta z okrelon
        prdkoci.
    */
    public void turnZTo(float angleDest, float speed) {
        turnTo(velocityAngleZ, getAngleZ(), angleDest, speed);
    }

    /**
        Obraca z okrelon prdkoci obiekt wok osi x a 
        przyjmie pozycj przodem do kierunku danego wektora
        (y, z).
    */
    public void turnXTo(float y, float z, float angleOffset,
        float speed)
    {
        turnXTo((float)Math.atan2(-z,y) + angleOffset, speed);
    }


    /**
        Obraca z okrelon prdkoci obiekt wok osi y a 
        przyjmie pozycj przodem do kierunku danego wektora
        (x, z).
    */
    public void turnYTo(float x, float z, float angleOffset,
        float speed)
    {
        turnYTo((float)Math.atan2(-z,x) + angleOffset, speed);
    }


    /**
        Obraca z okrelon prdkoci obiekt wok osi z a 
        przyjmie pozycj przodem do kierunku danego wektora
        (x, y).
    */
    public void turnZTo(float x, float y, float angleOffset,
        float speed)
    {
        turnZTo((float)Math.atan2(y,x) + angleOffset, speed);
    }


    /**
        Upewnia si, e dany kt naley do przedziau od -pi do pi.
        Zwraca ten kt lub warto poprawion, jeli nie mieci si
        w prawidowym przedziale.
    */
    protected float ensureAngleWithinBounds(float angle) {
        if (angle < -Math.PI || angle > Math.PI) {
            // przeksztaca przedzia do postaci od 0 do 1
            double newAngle = (angle + Math.PI) / (2*Math.PI);
            // poprawia przedzia
            newAngle = newAngle - Math.floor(newAngle);
            // przeksztaca przedzia ponownie do postaci od -pi do pi
            newAngle = Math.PI * (newAngle * 2 - 1);
            return (float)newAngle;
        }
        return angle;
    }


    /**
        Zmienia z okrelon szybkoci kt ruchu od kta startAngle
        do kta endAngle.
    */
    protected void turnTo(Movement movement,
        float startAngle, float endAngle, float speed)
    {
        startAngle = ensureAngleWithinBounds(startAngle);
        endAngle = ensureAngleWithinBounds(endAngle);
        if (startAngle == endAngle) {
            movement.set(0,0);
        }
        else {

            float distanceLeft;
            float distanceRight;
            float pi2 = (float)(2*Math.PI);

            if (startAngle < endAngle) {
                distanceLeft = startAngle - endAngle + pi2;
                distanceRight = endAngle - startAngle;
            }
            else {
                distanceLeft = startAngle - endAngle;
                distanceRight = endAngle - startAngle + pi2;
            }

            if (distanceLeft < distanceRight) {
                speed = -Math.abs(speed);
                movement.set(speed, (long)(distanceLeft / -speed));
            }
            else {
                speed = Math.abs(speed);
                movement.set(speed, (long)(distanceRight / speed));
            }
        }
    }


    /**
        Ustawia szybko ruchu obrotowego wok osi x.
    */
    public void setAngleVelocityX(float speed) {
        setAngleVelocityX(speed, FOREVER);
    }


    /**
        Ustawia szybko ruchu obrotowego wok osi y.
    */
    public void setAngleVelocityY(float speed) {
        setAngleVelocityY(speed, FOREVER);
    }


    /**
        Ustawia szybko ruchu obrotowego wok osi z.
    */
    public void setAngleVelocityZ(float speed) {
        setAngleVelocityZ(speed, FOREVER);
    }


    /**
        Ustawia szybko ruchu obrotowego wok osi x
        na okrelony czas.
    */
    public void setAngleVelocityX(float speed, long time) {
        velocityAngleX.set(speed, time);
    }


    /**
        Ustawia szybko ruchu obrotowego wok osi y
        na okrelony czas.
    */
    public void setAngleVelocityY(float speed, long time) {
        velocityAngleY.set(speed, time);
    }


    /**
        Ustawia szybko ruchu obrotowego wok osi z
        na okrelony czas.
    */
    public void setAngleVelocityZ(float speed, long time) {
        velocityAngleZ.set(speed, time);
    }


    /**
        Ustawia szybko ruchu obrotowego wok osi x
        na okrelony czas.
    */
    public float getAngleVelocityX() {
        return isTurningX()?velocityAngleX.speed:0;
    }


    /**
        Ustawia szybko ruchu obrotowego wok osi y
        na okrelony czas.
    */
    public float getAngleVelocityY() {
        return isTurningY()?velocityAngleY.speed:0;
    }


    /**
        Ustawia szybko ruchu obrotowego wok osi z
        na okrelony czas.
    */
    public float getAngleVelocityZ() {
        return isTurningZ()?velocityAngleZ.speed:0;
    }


    /**
        Zwraca true, jeli obiekt aktualnie obraca si
        wok osi x.
    */
    public boolean isTurningX() {
        return !velocityAngleX.isStopped();
    }


    /**
        Zwraca true, jeli obiekt aktualnie obraca si
        wok osi y.
    */
    public boolean isTurningY() {
        return !velocityAngleY.isStopped();
    }


    /**
        Zwraca true, jeli obiekt aktualnie obraca si
        wok osi z.
    */
    public boolean isTurningZ() {
        return !velocityAngleZ.isStopped();
    }

    /**
        Klasa Movement przechowuje szybko oraz czas, przez jaki
        obiekt ma si porusza z t szybkoci.
    */
    protected static class Movement {
        // zmiana na milisekund
        float speed;
        long remainingTime;

        /**
            Ustawia waciwoci ruchu zgodnie z przekazan szybkoci 
            i iloci czasu (w milisekundach).
        */
        public void set(float speed, long time) {
            this.speed = speed;
            this.remainingTime = time;
        }


        public boolean isStopped() {
            return (speed == 0) || (remainingTime == 0);
        }

        /**
            Zwraca odlego przebyt w okrelonym czasie (liczbie milisekund).
        */
        public float getDistance(long elapsedTime) {
            if (remainingTime == 0) {
                return 0;
            }
            else if (remainingTime != FOREVER) {
                elapsedTime = Math.min(elapsedTime, remainingTime);
                remainingTime-=elapsedTime;
            }
            return speed * elapsedTime;
        }
    }
}

