package com.brackeen.javagamebook.graphics3D.texture;

import java.awt.Color;
import java.awt.image.IndexColorModel;

/**
    Klasa ShadedTexture jest podklas klasy Texture z cieniowanymi
    wariantami tekstury. rdowa tekstura przechowywana jest jako 8-bitowy
    obraz, ktry dla kadego odcienia posiada osobn palet.
*/
public final class ShadedTexture extends Texture {

    public static final int NUM_SHADE_LEVELS = 64;
    public static final int MAX_LEVEL = NUM_SHADE_LEVELS-1;

    private static final int PALETTE_SIZE_BITS = 8;
    private static final int PALETTE_SIZE = 1 << PALETTE_SIZE_BITS;

    private byte[] buffer;
    private IndexColorModel palette;
    private short[] shadeTable;
    private int defaultShadeLevel;
    private int widthBits;
    private int widthMask;
    private int heightBits;
    private int heightMask;

    // wiersz jest zapisywany za pomoc setCurrRow i pobierany za pomoc getColorCurrRow
    private int currRow;

    /**
        Tworzy now tekstur ShadedTexture z podanej palety i 
        8-bitowego bufora obrazu. Szeroko mapy bitowej to 2 
        do potgi widthBits, czyli (1 << widthBits). Podobnie, 
        wysoko mapy bitowej to 2 do potgi heightBits, czyli
        (1 << heightBits). Tekstura cieniowana jest od jej 
        oryginalnego koloru do czerni. 
    */
    public ShadedTexture(byte[] buffer,
        int widthBits, int heightBits,
        IndexColorModel palette)
    {
        this(buffer, widthBits, heightBits, palette, Color.BLACK);
    }


    /**
        Tworzy now tekstur ShadedTexture z podanej palety i 
        8-bitowego bufora obrazu i docelowego odcienia. Szeroko 
        mapy bitowej to 2 do potgi widthBits, czyli (1 << widthBits). 
        Podobnie, wysoko mapy bitowej to 2 do potgi heightBits, 
        czyli (1 << heightBits). Tekstura cieniowana jest od jej 
        oryginalnego koloru do docelowego odcienia. 
    */
    public ShadedTexture(byte[] buffer,
        int widthBits, int heightBits,
        IndexColorModel palette, Color targetShade)
    {
        super(1 << widthBits, 1 << heightBits);
        this.buffer = buffer;
        this.widthBits = widthBits;
        this.heightBits = heightBits;
        this.widthMask = getWidth() - 1;
        this.heightMask = getHeight() - 1;
        this.buffer = buffer;
        this.palette = palette;
        defaultShadeLevel = MAX_LEVEL;

        makeShadeTable(targetShade);
    }


    /**
        Tworzy tablic odcieni dla tej tekstury ShadedTexture. Kada
        pozycja w palecie bdzie stopniowo cieniowana od oryginalnego
        koloru do podanego odcienia.
    */
    public void makeShadeTable(Color targetShade) {

        shadeTable = new short[NUM_SHADE_LEVELS*PALETTE_SIZE];

        for (int level=0; level<NUM_SHADE_LEVELS; level++) {
            for (int i=0; i<palette.getMapSize(); i++) {
                int red = calcColor(palette.getRed(i),
                    targetShade.getRed(), level);
                int green = calcColor(palette.getGreen(i),
                    targetShade.getGreen(), level);
                int blue = calcColor(palette.getBlue(i),
                    targetShade.getBlue(), level);

                int index = level * PALETTE_SIZE + i;
                // RGB 5:6:5
                shadeTable[index] = (short)(
                            ((red >> 3) << 11) |
                            ((green >> 2) << 5) |
                            (blue >> 3));
            }
        }
    }

    private int calcColor(int palColor, int target, int level) {
        return (palColor - target) * (level+1) /
            NUM_SHADE_LEVELS + target;
    }


    /**
        Ustawia domylny poziom cieniowania wykorzystywany, gdy 
        przywoywana jest metoda getColor().
    */
    public void setDefaultShadeLevel(int level) {
        defaultShadeLevel = level;
    }


    /**
        Pobiera domylny poziom cieniowania wykorzystywany, gdy 
        przywoywana jest metoda getColor().
    */
    public int getDefaultShadeLevel() {
        return defaultShadeLevel;
    }


    /**
        Pobiera 16-bitowy kolor tej tekstury Texture w podanej
        lokacji (x,y), wykorzystujc domylny poziom cieniowania.
    */
    public short getColor(int x, int y) {
        return getColor(x, y, defaultShadeLevel);
    }


    /**
        Pobiera 16-bitowy kolor tej tekstury Texture w podanej
        lokacji (x,y), wykorzystujc podany poziom cieniowania.
    */
    public short getColor(int x, int y, int shadeLevel) {
        return shadeTable[(shadeLevel << PALETTE_SIZE_BITS) |
            (0xff & buffer[
            (x & widthMask) |
            ((y & heightMask) << widthBits)])];
    }

    /**
        Ustawia biecy wiersz dla getColorCurrRow(). Wstpnie wylicza
        przesunicie dla tego wiersza.
    */
    public void setCurrRow(int y) {
        currRow = (y & heightMask) << widthBits;
    }


    /**
        Pobiera kolor w podanej lokacji x i o okrelonym poziomie
        cieniowania. Wykorzystywany jest wiersz definiowany 
        w metodzie setCurrRow.
    */
    public short getColorCurrRow(int x, int shadeLevel) {
        return shadeTable[(shadeLevel << PALETTE_SIZE_BITS) |
            (0xff & buffer[(x & widthMask) | currRow])];
    }

}
