package com.brackeen.javagamebook.bsp2D;

import java.io.*;
import java.util.*;
import com.brackeen.javagamebook.math3D.*;
import com.brackeen.javagamebook.game.GameObject;

/**
    Klasa MapLoader wczytuje mapy z pliku tekstowego zbudowanego
    zgodnie z ze specyfikacj plikw OBJ Alias|Wavefront.

    Polecenia pliku MAP:
    <pre>
    v [x] [y] [z]        - Definiuje wierzchoek o wsprzdnych
                           zmiennoprzecinkowych (x,y,z).
    mtllib [filename]    - aduje materiay z zewntrznego pliku 
                           .mtl.
    usemtl [name]        - Wykorzystuje materia o podanej nazwie (adowany
                           z pliku .mtl) na nastpn podog, sufit lub
                           cian.
    ambientLightIntensity
        [value]          - Definiuje intensywno wiata otoczenia 
                           dla kolejnego pokoju, w przedziale od 0 do 1.
    pointlight [v]       - Definiuje punktowe wiato zlokalizowane pod 
        [intensity]        okrelonym wektorem. Opcjonalnie mozna poda
        [falloff]          intensywno wiata i odlego cakowitego 
                           wygasania wiata ze rda.
    player [v] [angle]   - Definiuje pocztkow lokacj gracza i  
                           opcjonalnie pocztkowy kt, mierzony 
                           w radianach wzgldem osi y.
    obj [uniqueName]     - Definiuje obiekt z zewnyrznego pliku
        [filename] [v]     OBJ. Ta unikatowa nazwa pozwala jednoznacznie 
        [angle]            zidentyfikowa obiekt, mozna jej jednak take  
                           przypisa warto "null", jeli nie potrzebujemy
                           unikatowej nazwy. Nazwa pliku odnosi si do
                           zewntrznego pliku OBJ. Opcjonalnie mozna poda 
                           pocztkowy kt, mierzony w radianach 
                           wzgldem osi y.
    room [name]          - Definiuje nowy pokj, opcjonalnie nadajc mu
                           nazw. Pokj skada si z pionowych cian,
                           poziomej podogi i poziomego sufitu.
                           Wklse pomieszczenia nie s na razie 
                           obsugiwane, mona je jednak symulowa
                           za pomoc ssiadujcych ze sob wypukych pokoi.
    floor [height]       - Definiuje wysoko podogi w biecym 
                           pokoju, wykorzystujc biecy materia.
                           Biecemu materiaowi mozna przypisa null
                           i wtedy nie jest tworzona aden wielokt
                           podogi. Podoga moe te znajdowa si nad 
                           sufitem i wtedy zamiast pokoju tworzona jest
                           struktura "filaru" lub "skalnego bloku",
                           niedostpna dla gracza.
    ceil [height]        - Definiuje wysoko sufitu w biezcym pokoju,
                           wykorzystujc w tym celu biezacy materia.
                           Biecemu materiaowi mozna przypisa null
                           i wtedy nie jest tworzona aden wielokt
                           sufitu. Sufit moe te znajdowa si pod 
                           podog i wtedy zamiast pokoju tworzona jest
                           struktura "filaru" lub "skalnego bloku",
                           niedostpna dla gracza.
    wall [x] [z]         - Dwfiniuje wierzchoek ciany w pokoju uywajc
         [bottom] [top]    podanych wsprzdnych x i z.ciany powinny 
                           by definiowane w porzdku zgodnym z ruchem 
                           wskazwek zegara. Jeli nie zostanie zdefinowany
                           "d" i "gra", wykorzystywane s wysokoci 
                           podogii sufitu. Jeli biezcy materia jest
                           null lub d [bottom] jest rwny grze [top],
                           to wielokt ciany nie zostanie utworzony.
    </pre>
*/
public class MapLoader extends ObjectLoader {

    private BSPTreeBuilder builder;
    private Map loadedObjects;
    private Transform3D playerStart;
    private RoomDef currentRoom;
    private List rooms;
    private List mapObjects;

    // uyj osobnego obiektu klasy ObjectLoader dla obiektw
    private ObjectLoader objectLoader;


    /**
        Tworzy nowy obiekt klasy MapLoader wykorzystujcy domylny obiekt BSPTreeBuilder.
    */
    public MapLoader() {
        this(null);
    }


    /**
        Tworzy nowy obiekt klasy MapLoader wykorzystujcy wskazany obiekt BSPTreeBuilder.
        Jeli wskazany obiekt jest rwny null, tworzony jest domylny obiekt BSPTreeBuilder.
    */
    public MapLoader(BSPTreeBuilder builder) {
        if (builder == null) {
            this.builder = new BSPTreeBuilder();
        }
        else {
            this.builder = builder;
        }
        parsers.put("map", new MapLineParser());
        objectLoader = new ObjectLoader();
        loadedObjects = new HashMap();
        rooms = new ArrayList();
        mapObjects = new ArrayList();
    }


    /**
        Wczytuje plik mapy i tworzy drzewo BSP. Tworzone obiekty mona odczytywa za
        pomoc metody getObjectsInMap().
    */
    public BSPTree loadMap(String filename) throws IOException {
        currentRoom = null;
        rooms.clear();
        vertices.clear();
        mapObjects.clear();
        playerStart = new Transform3D();

        path = new File(filename).getParentFile();

        parseFile(filename);

        return createBSPTree();
    }


    /**
        Tworzy drzewo BSP na podstawie pomieszcze zdefiniowanych w pliku mapy.
    */
    protected BSPTree createBSPTree() {
        // odczytaj wszystkie wielokty
        List allPolygons = new ArrayList();
        for (int i=0; i<rooms.size(); i++) {
            RoomDef room = (RoomDef)rooms.get(i);
            allPolygons.addAll(room.createPolygons());
        }

        // buduj drzewo
        BSPTree tree = builder.build(allPolygons);

        // stwrz powierzchnie wieloktw z uwzgldnieniem wiate
        tree.createSurfaces(lights);
        return tree;
    }


    /**
        Zwraca list wszystkich obiektw zdefiniowanych w pliku mapy.
    */
    public List getObjectsInMap() {
        return mapObjects;
    }


    /**
        Zwraca pocztkowe pooenie gracza zdefiniowane w pliku mapy.
    */
    public Transform3D getPlayerStartLocation() {
        return playerStart;
    }


    /**
        Ustawia wiata wykorzystywane dla obiektw OBJ.
    */
    public void setObjectLights(List lights,
        float ambientLightIntensity)
    {
        objectLoader.setLights(lights, ambientLightIntensity);
    }


    /**
        Parsuje pojedynczy wiersz z pliku MAP.
    */
    protected class MapLineParser implements LineParser {

        public void parseLine(String line) throws IOException,
            NoSuchElementException
        {
            StringTokenizer tokenizer = new StringTokenizer(line);
            String command = tokenizer.nextToken();

            if (command.equals("v")) {
                // stwrz nowy wierzchoek
                vertices.add(new Vector3D(
                    Float.parseFloat(tokenizer.nextToken()),
                    Float.parseFloat(tokenizer.nextToken()),
                    Float.parseFloat(tokenizer.nextToken())));
            }
            else if (command.equals("mtllib")) {
                // wczytaj materiay z pliku
                String name = tokenizer.nextToken();
                parseFile(name);
            }
            else if (command.equals("usemtl")) {
                // zdefiniuj biecy materia
                String name = tokenizer.nextToken();
                if ("null".equals(name)) {
                    currentMaterial = new Material();
                }
                else {
                    currentMaterial =
                        (Material)materials.get(name);
                    if (currentMaterial == null) {
                        currentMaterial = new Material();
                        System.out.println("brak materiau: " + name);
                    }
                }
            }
            else if (command.equals("pointlight")) {
                // stwrz wiato punktowe
                Vector3D loc = getVector(tokenizer.nextToken());
                float intensity = 1;
                float falloff = PointLight3D.NO_DISTANCE_FALLOFF;
                if (tokenizer.hasMoreTokens()) {
                    intensity =
                        Float.parseFloat(tokenizer.nextToken());
                }
                if (tokenizer.hasMoreTokens()) {
                    falloff =
                        Float.parseFloat(tokenizer.nextToken());
                }
                lights.add(new PointLight3D(loc.x, loc.y, loc.z,
                    intensity, falloff));
            }
            else if (command.equals("ambientLightIntensity")) {
                // zdefiniuj intensywno wiata otaczajcego
                ambientLightIntensity =
                    Float.parseFloat(tokenizer.nextToken());
            }
            else if (command.equals("player")) {
                // zdefiniuj pocztkowe pooenie gracza
                playerStart.getLocation().setTo(
                    getVector(tokenizer.nextToken()));
                if (tokenizer.hasMoreTokens()) {
                    playerStart.setAngleY(
                        Float.parseFloat(tokenizer.nextToken()));
                }
            }
            else if (command.equals("obj")) {
                // stwrz nowy obiekt
                String uniqueName = tokenizer.nextToken();
                String filename = tokenizer.nextToken();
                // sprawd, czy obiekt nie zosta ju wczytany
                PolygonGroup object =
                    (PolygonGroup)loadedObjects.get(filename);
                if (object == null) {
                    File file = new File(path, filename);
                    String filePath = file.getPath();
                    object = objectLoader.loadObject(filePath);
                    loadedObjects.put(filename, object);
                }
                Vector3D loc = getVector(tokenizer.nextToken());
                PolygonGroup mapObject =
                    (PolygonGroup)object.clone();
                mapObject.getTransform().getLocation().setTo(loc);
                if (!uniqueName.equals("null")) {
                    mapObject.setName(uniqueName);
                }
                if (tokenizer.hasMoreTokens()) {
                    mapObject.getTransform().setAngleY(
                        Float.parseFloat(tokenizer.nextToken()));
                }
                mapObjects.add(mapObject);
            }
            else if (command.equals("trigger")) {
                String uniqueName = tokenizer.nextToken();
                Vector3D loc = getVector(tokenizer.nextToken());
                float r = Float.parseFloat(tokenizer.nextToken());
                GameObject object = new GameObject(
                    new PolygonGroup(uniqueName));
                object.getTransform().getLocation().setTo(loc);
                object.getBounds().setTopHeight(32);
                object.getBounds().setRadius(r);
                mapObjects.add(object);
            }
            else if (command.equals("room")) {
                // rozpocznij wczytywanie nowego pomieszczenia
                currentRoom = new RoomDef(ambientLightIntensity);
                rooms.add(currentRoom);
            }
            else if (command.equals("floor")) {
                // zdefiniuj podog w pomieszczeniu
                float y = Float.parseFloat(tokenizer.nextToken());
                currentRoom.setFloor(y, currentMaterial.texture);
            }
            else if (command.equals("ceil")) {
                // zdefiniuj sufit w pomieszczeniu
                float y = Float.parseFloat(tokenizer.nextToken());
                currentRoom.setCeil(y, currentMaterial.texture);
            }
            else if (command.equals("wall")) {
                // zdefiniuj wierzchoek ciany w pomieszczeniu
                float x = Float.parseFloat(tokenizer.nextToken());
                float z = Float.parseFloat(tokenizer.nextToken());
                if (tokenizer.hasMoreTokens()) {
                    float bottom =
                        Float.parseFloat(tokenizer.nextToken());
                    float top =
                        Float.parseFloat(tokenizer.nextToken());
                    currentRoom.addVertex(x, z, bottom, top,
                        currentMaterial.texture);
                }
                else {
                    currentRoom.addVertex(x, z,
                        currentMaterial.texture);
                }
            }
            else {
                System.out.println("Nieznana instrukcja: " + command);
            }
        }
    }
}
