package com.brackeen.javagamebook.ai.pattern;

import java.util.*;
import com.brackeen.javagamebook.math3D.Vector3D;
import com.brackeen.javagamebook.path.PathFinder;
import com.brackeen.javagamebook.game.GameObject;
import com.brackeen.javagamebook.bsp2D.BSPTree;

/**
    Prosta abstrakcyjna klasa implementujca interfejs PathFinder,
    w ktrej metoda find(Vector3D, Vector3D) zwraca warto null.
    Nie zaimplementowano: AttackPatternSneak and DodgePatternHide
*/
public abstract class AIPattern implements PathFinder {


    protected BSPTree bspTree;

    /**
        Drzewo BSP jest wykorzystywane do otrzymywania waciwych
        wartoci y dla reprezentowanego wiata.
    */
    public AIPattern(BSPTree bspTree) {
        this.bspTree = bspTree;
    }

    public void setBSPTree(BSPTree bspTree) {
        this.bspTree = bspTree;
    }

    /**
        Ta metoda nie jest zaimplementowana.
    */
    public Iterator find(Vector3D start, Vector3D goal) {
        return null;
    }

    public abstract Iterator find(GameObject bot,
        GameObject player);


    /**
        Oblicza wysoko podogi dla okrelonego pooenia. Jeli
        nie mona obliczy wysokoci, wykorzystywana jest warto 
        domylna.
    */
    protected void calcFloorHeight(Vector3D v, float defaultY) {
       BSPTree.Leaf leaf = bspTree.getLeaf(v.x, v.z);
       if (leaf == null || leaf.floorHeight == Float.MIN_VALUE) {
           v.y = defaultY;
       }
       else {
           v.y = leaf.floorHeight;
       }
    }


    /**
        Zwraca pooenie takiego punktu znajdujcego si na
        prostej czcej bota z graczem, ktry znajduje si 
        w okrelonej odlegoci od gracza.
    */
    protected Vector3D getLocationFromPlayer(GameObject bot, GameObject player, float desiredDistSq)
    {
        // oblicza odlego (podniesion do kwadratu)
        float distSq =  bot.getLocation().getDistanceSq(player.getLocation());

        // jeli w promieniu piciu jednostek od gracza,
        // bot jest wystarczajco blisko
        if (Math.abs(desiredDistSq - distSq) < 25) {
            return new Vector3D(bot.getLocation());
        }

        // oblicza wektor od bota do gracza
        Vector3D goal = new Vector3D(bot.getLocation());
        goal.subtract(player.getLocation());

        // oblicza docelow odlego od gracza
        goal.multiply((float)Math.sqrt(desiredDistSq / distSq));

        goal.add(player.getLocation());
        calcFloorHeight(goal, bot.getFloorHeight());

        return goal;
    }

    public String toString() {
        // zwr nazw klasy (bez nazwy pakietu)
        String fullName = getClass().getName();
        int index = fullName.lastIndexOf('.');
        return fullName.substring(index+1);
    }

}
