package com.brackeen.javagamebook.bsp2D;

import com.brackeen.javagamebook.math3D.*;
import com.brackeen.javagamebook.game.GameObjectManager;

/**
    Klasa BSPTreeTraverer odpowiada za przegldanie wzw dwuwymiarowego
    drzewa BSP albo w porzdku poprzeczny, albo w kolejnoci rysowania
    (od przodu do tyu). Odwiedzone wielokty otrzymuj odpowiednie sygnay
    za porednictwem obiektw klasy BSPTreeTraverseListener.
*/
public class BSPTreeTraverser {

    private boolean traversing;
    private float x;
    private float z;
    private GameObjectManager objectManager;
    private BSPTreeTraverseListener listener;

    /**
        Tworzy nowy obiekt BSPTreeTraverser bez obiektu klasy
        BSPTreeTraverseListener.
    */
    public BSPTreeTraverser() {
        this(null);
    }


    /**
        Tworzy nowy obiekt BSPTreeTraverser ze wskazanym obiektem
        klasy BSPTreeTraverseListener.
    */
    public BSPTreeTraverser(BSPTreeTraverseListener listener) {
        setListener(listener);
    }


    /**
        Ustawia obiekt BSPTreeTraverseListener wykorzystywany podczas
        przegldania drzewa.
    */
    public void setListener(BSPTreeTraverseListener listener) {
        this.listener = listener;
    }


    /**
        Ustawia obiekt GameObjectManager. Jeli podczas przegldania
        obiekt GameObjectManager jest rny od null, wwczas wywoywana
        jest metoda markVisible() menadera, ktra okrela widoczne
        czci drzewa.
    */
    public void setGameObjectManager(
        GameObjectManager objectManager)
    {
        this.objectManager = objectManager;
    }


    /**
        Przeglda wzy drzewa w kolejnoci rysowania reprezentowanych przez
        nie wieloktw (od przodu do tyu) dla okrelonej pozycji kamery.
    */
    public void traverse(BSPTree tree, Vector3D viewLocation) {
        x = viewLocation.x;
        z = viewLocation.z;
        traversing = true;
        traverseDrawOrder(tree.getRoot());
    }


    /**
        Przeglda wzy drzewa w porzdku poprzecznym.
    */
    public void traverse(BSPTree tree) {
        traversing = true;
        traverseInOrder(tree.getRoot());
    }


    /**
        Przeglda wze w kolejnoci rysowania (od przodu do tyu) dla
        okrelonej pozycji kamery.
    */
    private void traverseDrawOrder(BSPTree.Node node) {
        if (traversing && node != null) {
            if (node instanceof BSPTree.Leaf) {
                // brak podziaw, tylko obsuga wieloktw
                visitNode(node);
            }
            else if (node.partition.getSideThin(x,z) != BSPLine.BACK) {
                traverseDrawOrder(node.front);
                visitNode(node);
                traverseDrawOrder(node.back);
            }
            else {
                traverseDrawOrder(node.back);
                visitNode(node);
                traverseDrawOrder(node.front);
            }
        }
    }



    /**
        Przeglda wze w porzdku poprzecznym.
    */
    private void traverseInOrder(BSPTree.Node node) {
        if (traversing && node != null) {
            traverseInOrder(node.front);
            visitNode(node);
            traverseInOrder(node.back);
        }
    }


    /**
        Odwiedza wze drzewa BSP. Metoda visitPolygon() interfejsu
        BSPTreeTraverseListener jest wywoywana dla kadego wielokta
        w danym wle.
    */
    private void visitNode(BSPTree.Node node) {
        if (!traversing || node.polygons == null) {
            return;
        }

        boolean isBack = false;
        if (node instanceof BSPTree.Leaf) {
            BSPTree.Leaf leaf = (BSPTree.Leaf)node;
            isBack = leaf.isBack;
            // oznacza ograniczenia tego licia jako obszar widoczny
            // w menaderze obiektw wystpujcych w grze.
            if (objectManager != null && leaf.bounds != null) {
                objectManager.markVisible(leaf.bounds);
            }
        }

        // odwiedza kady wielokt
        for (int i=0; traversing && i<node.polygons.size(); i++) {
            BSPPolygon poly = (BSPPolygon)node.polygons.get(i);
            traversing = listener.visitPolygon(poly, isBack);
        }
    }


}
