package com.brackeen.javagamebook.test;

import java.awt.*;
import javax.swing.ImageIcon;

import com.brackeen.javagamebook.graphics.ScreenManager;

/**
    Prosta klasa abstrakcyjna wykorzystywana do testowania. Klasy pochodne 
    powinny implementowa metod draw().
*/
public abstract class GameCore {

    protected static final int FONT_SIZE = 24;

    private static final DisplayMode POSSIBLE_MODES[] = {
        new DisplayMode(800, 600, 16, 0),
        new DisplayMode(800, 600, 32, 0),
        new DisplayMode(800, 600, 24, 0),
        new DisplayMode(640, 480, 16, 0),
        new DisplayMode(640, 480, 32, 0),
        new DisplayMode(640, 480, 24, 0),
        new DisplayMode(1024, 768, 16, 0),
        new DisplayMode(1024, 768, 32, 0),
        new DisplayMode(1024, 768, 24, 0),
    };

    private boolean isRunning;
    protected ScreenManager screen;


    /**
        Sygnalizuje ptli gry, e czas koczy prac.
    */
    public void stop() {
        isRunning = false;
    }


    /**
        Wywouje init() i gameLoop()
    */
    public void run() {
        try {
            init();
            gameLoop();
        }
        finally {
            screen.restoreScreen();
            lazilyExit();
        }
    }


    /**
        Koczy dziaanie maszyny wirtualnej z wtku demona. Wtek-demon
        czeka 2 sekundy a nastpnie wywouje System.exit(0). Poniewa maszyna
        wirtualna powinna koczyc si gdy dziaaj tylko wtki-demony,
        metoda zapewnia, e System.exit(0) jest wywoywane tylko
        w ostatecznosci. Jest to niezbdne gy dziaa Java Sound.
    */
    public void lazilyExit() {
        Thread thread = new Thread() {
            public void run() {
                // na pocztku czekamy na samodzielne zakoczenie dziaania VM
                try {
                    Thread.sleep(2000);
                }
                catch (InterruptedException ex) { }
                // system nadal dziaa, wic wymuszamy zakoczenie
                System.exit(0);
            }
        };
        thread.setDaemon(true);
        thread.start();
    }


    /**
        Ustawia tryb penoekranowy i inicjuje obiekty.
    */
    public void init() {
        screen = new ScreenManager();
        DisplayMode displayMode =
            screen.findFirstCompatibleMode(POSSIBLE_MODES);
        screen.setFullScreen(displayMode);

        Window window = screen.getFullScreenWindow();
        window.setFont(new Font("Dialog", Font.PLAIN, FONT_SIZE));
        window.setBackground(Color.blue);
        window.setForeground(Color.white);

        isRunning = true;
    }


    public Image loadImage(String fileName) {
        return new ImageIcon(fileName).getImage();
    }


    /**
        Uruchamia ptl gry dziaajc a do wywoania metody stop().
    */
    public void gameLoop() {
        long startTime = System.currentTimeMillis();
        long currTime = startTime;

        while (isRunning) {
            long elapsedTime =
                System.currentTimeMillis() - currTime;
            currTime += elapsedTime;

            // aktualizacja
            update(elapsedTime);

            // rysowanie
            Graphics2D g = screen.getGraphics();
            draw(g);
            g.dispose();
            screen.update();

            // nie usypiamy zadania, praca z pen prdkoci!
            /*try {
                Thread.sleep(20);
            }
            catch (InterruptedException ex) { }*/
        }
    }


    /**
        Aktualizuje stan gry/animacji w oparciu o okres czas jaki min
        od uruchomienia programu.
    */
    public void update(long elapsedTime) {
        // nic nie rb
    }


    /**
        Rysowanie na ekranie. Klasy pochodne musz dziedziczy t metod.
    */
    public abstract void draw(Graphics2D g);
}
