package org.jpwh.test.fetching;

import org.jpwh.env.JPATest;
import org.jpwh.model.fetching.eagerjoin.Bid;
import org.jpwh.model.fetching.eagerjoin.Item;
import org.jpwh.model.fetching.eagerjoin.User;
import org.jpwh.shared.util.CalendarUtil;
import org.jpwh.shared.util.TestData;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.Test;

import javax.persistence.EntityManager;
import javax.transaction.UserTransaction;
import java.math.BigDecimal;

import static org.testng.Assert.assertEquals;
import static org.testng.Assert.assertNotNull;

public class EagerJoin extends JPATest {

    @Override
    public void configurePersistenceUnit() throws Exception {
        configurePersistenceUnit("FetchingEagerJoinPU");
    }

    public FetchTestData storeTestData() throws Exception {
        UserTransaction tx = TM.getUserTransaction();
        tx.begin();
        EntityManager em = JPA.createEntityManager();

        Long[] itemIds = new Long[3];
        Long[] userIds = new Long[3];

        User jandomanski = new User("jandomanski");
        em.persist(jandomanski);
        userIds[0] = jandomanski.getId();

        User janinadomanska = new User("janinadomanska");
        em.persist(janinadomanska);
        userIds[1] = janinadomanska.getId();

        User robertdomanski = new User("robertdomanski");
        em.persist(robertdomanski);
        userIds[2] = robertdomanski.getId();

        Item item = new Item("Przedmiot pierwszy", CalendarUtil.TOMORROW.getTime(), jandomanski);
        em.persist(item);
        itemIds[0] = item.getId();
        for (int i = 1; i <= 3; i++) {
            Bid bid = new Bid(item, robertdomanski, new BigDecimal(9 + i));
            item.getBids().add(bid);
            em.persist(bid);
        }

        item = new Item("Przedmiot drugi", CalendarUtil.TOMORROW.getTime(), jandomanski);
        em.persist(item);
        itemIds[1] = item.getId();
        for (int i = 1; i <= 1; i++) {
            Bid bid = new Bid(item, janinadomanska, new BigDecimal(2 + i));
            item.getBids().add(bid);
            em.persist(bid);
        }

        item = new Item("Przedmiot trzeci", CalendarUtil.AFTER_TOMORROW.getTime(), janinadomanska);
        em.persist(item);
        itemIds[2] = item.getId();

        tx.commit();
        em.close();

        FetchTestData testData = new FetchTestData();
        testData.items = new TestData(itemIds);
        testData.users = new TestData(userIds);
        return testData;
    }

    @Test
    public void fetchEagerJoin() throws Exception {
        FetchTestData testData = storeTestData();

        UserTransaction tx = TM.getUserTransaction();
        try {
            tx.begin();
            EntityManager em = JPA.createEntityManager();

            Long ITEM_ID = testData.items.getFirstId();

            Item item = em.find(Item.class, ITEM_ID);
            // select i.*, u.*, b.*
            //  from ITEM i
            //   left outer join USERS u on u.ID = i.SELLER_ID
            //   left outer join BID b on b.ITEM_ID = i.ID
            //  where i.ID = ?

            em.detach(item); // Pobieranie ukończone. Nie ma już leniwego ładowania

            // W stanie odłączonym oferty są dostępne...
            assertEquals(item.getBids().size(), 3);
            assertNotNull(item.getBids().iterator().next().getAmount());

            // .. dostępny jest też sprzedawca
            assertEquals(item.getSeller().getUsername(), "jandomanski");

            tx.commit();
            em.close();
        } finally {
            TM.rollback();
        }
    }
}
