package org.jpwh.web.jsf;

import org.jpwh.web.dao.ImageDAO;
import org.jpwh.web.dao.ItemDAO;
import org.jpwh.web.dao.UserDAO;
import org.jpwh.web.model.Image;
import org.jpwh.web.model.ImageLookup;
import org.jpwh.web.model.Item;

import javax.enterprise.context.Conversation;
import javax.enterprise.context.ConversationScoped;
import javax.enterprise.event.Observes;
import javax.inject.Inject;
import javax.inject.Named;
import javax.persistence.EntityNotFoundException;
import javax.servlet.http.Part;
import javax.transaction.Transactional;
import java.io.Serializable;
import java.math.BigDecimal;

import static javax.enterprise.event.Reception.IF_EXISTS;

@Named
/* 
    Egzemplarz usługi ma zasięg konwersacji. Domyślnie kontekst konwersacji 
	jest przejściowy i dlatego zachowuje się tak, jakby był usługą o zasięgu żądania. 
 */
@ConversationScoped
/* 
    W przeciwieństwie do implementacji usługi o zasięgu żądania klasa musi implementować 
	interfejs Serializable. Egzemplarz klasy EditItemService może być zapisany w sesji HTTP, 
	a dane tej sesji mogą być zserializowane na dysku, albo przesłane w sieci wewnątrz klastra. 
	W poprzednim rozdziale zastosowaliśmy łatwe podejście polegające na wykorzystaniu 
	komponentu EJB stateful. Powiedzieliśmy: "on nie obsługuje pasywacji".
	Wszystko, co znajduje się w kontekście konwersacji CDI musi pozwalać na pasywację, 
	a tym samym implementować interfejs serializable. 
 */
public class EditItemService implements Serializable {

    /* 
        Wstrzyknięte egzemplarze DAO mają zależny zasięg i implementują interfejs serializable. 
		Można by sądzić, że tak nie jest, ponieważ mają pole <code>EntityManager</code>, 
		które nie jest serializowalne. O tym niedopasowaniu opowiemy za chwilę.      
    */
    @Inject
    ItemDAO itemDAO;

    @Inject
    ImageDAO imageDAO;

    @Inject
    UserDAO userDAO;

    /* 
        API <code>Conversation</code> dostarczane przez kontener. Należy go wywołać 
		w celu zarządzania kontekstem konwersacji. Potrzebujemy tego API, gdy użytkownik 
		po raz pierwszy kliknie przycisk Następny, co spowoduje wypromowanie konwersacji 
		przejściowej do długotrwałej. 
     */
    @Inject
    Conversation conversation;

    /* 
        Stan usługi tworzą: przedmiot, który użytkownik edytuje na stronach kreatora. 
		Zaczynamy od świeżego egzemplarza Item w stanie przejściowym. 
		Jeśli ta usługa zostanie zainicjowana z wykorzystaniem wartości identyfikatora,
		to należy załadować egzemplarz <code>Item</code> wewnątrz metody <code>setItemId()</code>. 
     */
    Long itemId;
    Item item = new Item();

    /* 
	
		Potrzebujemy jej tylko chwilowo, kiedy użytkownik kliknie przycisk 
		<em>Wgraj</em> na stronie "Edytuj zdjęcia". Klasa <code>Part</code> 
		należąca do API Servlet nie implementuje interfejsu Serializable. 
		Występowanie stanów przejściowych w usłudze konwersacyjnej nie jest 
		niczym niezwykłym, ale gdy zachodzi potrzeba ich użycia, 
		wtedy należy je inicjować dla każdego żądania. 
     */
    transient Part imageUploadPart;

    /* 
        Metoda <code>setItemId</code> będzie wywołana tylko wtedy, gdy żądanie 
		zawiera wartość identyfikatora przedmiotu. Dlatego do tej konwersacji 
		istnieją dwa punkty wejścia: z wartością identyfikatora istniejącego przedmiotu,
		albo bez tej wartości.
     */
    public void setItemId(Long itemId) {
        this.itemId = itemId;
        if (item.getId() == null && itemId != null) {
            /* 
                Jeżeli użytkownik edytuje przedmiot, to trzeba go załadować z bazy danych. 
				Nadal bazujemy na kontekście utrwalania o zasięgu żądania, dlatego po 
				zakończeniu obsługi żądania ten egzemplarz <code>Item</code> będzie 
				w stanie odłączonym. Odłączone egzemplarze encji mogą być utrzymywane 
				wewnątrz stanu usługi konwersacyjnej i scalane w chwili, 
				gdy zachodzi potrzeba utrwalenia zmian. 
             */
            item = itemDAO.findById(itemId);
            if (item == null)
                throw new EntityNotFoundException();
        }
    }

    public Conversation getConversation() {
        return conversation;
    }

    public Long getItemId() {
        return itemId;
    }

    public Item getItem() {
        return item;
    }

    public Part getImageUploadPart() {
        return imageUploadPart;
    }

    public void setImageUploadPart(Part imageUploadPart) {
        this.imageUploadPart = imageUploadPart;
    }

    public String editImages() {
        if (conversation.isTransient()) {
            conversation.setTimeout(10 * 60 * 1000); // 10 minut
            conversation.begin();
        }
        return "editItemImages";
    }

    public void uploadImage() throws Exception {
        if (imageUploadPart == null)
            return;

        /* 
            Utworzenie egzemplarza encji <code>Image</code> na podstawie 
			przesłanego formularza multi-part. 
         */
        Image image =
            imageDAO.hydrateImage(imageUploadPart.getInputStream());
        image.setName(imageUploadPart.getSubmittedFileName());
        image.setContentType(imageUploadPart.getContentType());

        /* 
            Trzeba dodać przejściowy egzemplarz <code>Image</code> do egzemplarza 
			<code>Item</code> w stanie przejściowym, albo odłączonym. Ta konwersacja, 
			w miarę dodawania danych zdjęć do stanu konwersacji, a tym samym sesji użytkownika, 
			będzie zużywać coraz więcej pamięci serwera. 
         */
        image.setItem(item);
        item.getImages().add(image);
    }

    /* 
        Interceptor transakcji systemowych opakowuje wywołanie metody.
     */
    @Transactional
    public String submitItem() {
        /* 
            Aby zapisać dane, należy złączyć niezsynchronizowany kontekst utrwalania 
			o zasięgu żądania z transakcją systemową. 
         */
        itemDAO.joinTransaction();

        item.setSeller(userDAO.findById(1l));

        /* 
            Ten obiekt DAO przekształca obiekt <code>Item</code> w stanie przejściowym 
			lub odłączonym na obiekt w stanie utrwalonym. Ponieważ włączyliśmy go 
			z wykorzystaniem reguły kaskadowej wewnątrz adnotacji <code>@OneToMany</code>, 
			to zapisuje także wszystkie nowe przejściowe lub stare odłączone elementy 
			kolekcji <code>Item#images</code>. Zgodnie z kontraktem obiektu DAO 
			zwrócony egzemplarz powinien być traktowany jako stan bieżący.
         */
        item = itemDAO.makePersistent(item);

        /* 
			
            Ręczne zakończenie długotrwałej konwersacji. W gruncie rzeczy 
			to jest degradacja: długotrwająca konwersacja staje się przejściowa; 
			Kontekst konwersacji oraz egzemplarz usługi są niszczone po zakończeniu 
			obsługi żądania. Z sesji użytkownika są usuwane wszystkie elementy stanu konwersacji. 
			
         */
        if (!conversation.isTransient())
            conversation.end();

        /* 
            W JSF to jest przekierowanie po wykonaniu operacji POST strony szczegółów 
			przedmiotu aukcji z nową wartością identyfikatora egzemplarza Item teraz w stanie trwałym. 
         */
        return "auction?id=" + item.getId() + "&faces-redirect=true";
    }

    public String cancel() {
        if (!conversation.isTransient())
            conversation.end();
        return "catalog?faces-redirect=true";
    }

         // Ciekawa sztuczka z CDI: Ten komponent bean odpowie, jeśli ktoś zainicjuje zdarzenie szukania,  
		 // ale nie będzie stworzony, jeśli wcześniej nie istniał.
    public void getConversationalImage(@Observes(notifyObserver = IF_EXISTS)
                                       ImageLookup imageLookup) {
										   
        // Zdjęcia w stanie przejściowym mogą istnieć bez wartości identyfikatora. W związku z tym zakładamy, że
        // identyfikator wyszukiwania właściwie jest indeksem zdjęcia na liście zdjęć

        Image image = getItem().getImagesSorted().get(
            new BigDecimal(imageLookup.getId()).intValueExact()
        );
        if (image != null)
            imageLookup.setImage(image);
    }
}