package org.jpwh.web.jsf;

import org.jpwh.web.dao.BidDAO;
import org.jpwh.web.dao.ItemDAO;
import org.jpwh.web.model.Bid;
import org.jpwh.web.model.Item;

import javax.enterprise.context.RequestScoped;
import javax.inject.Inject;
import javax.inject.Named;
import javax.persistence.EntityNotFoundException;
import javax.transaction.Transactional;
import java.math.BigDecimal;

@Named
/* 
    Dla tego przypadku użycia nie ma potrzeby utrzymywania stanu pomiędzy żądaniami. 
    W momencie renderowania widoku strony aukcji za pomocą żądania GET tworzony jest 
    egzemplarz usługi. W odpowiedzi komponent JSF wiąże parametr żądania poprzez 
    wywołanie metody setId(). Egzemplarz usługi jest niszczony w chwili 
    zakończenia renderowania. Serwer nie utrzymuje żadnego stanu pomiędzy żądaniami.
    W chwili przesłania formularza aukcji, gdy zaczyna się przetwarzanie żądania POST, 
    JSF wywołuje metodę setId(), która wiąże ukryte pole formularza. 
    To pozwala na nowo zainicjować stan usługi.
 */

@RequestScoped
public class AuctionService {

    @Inject
    ItemDAO itemDAO;

    @Inject
    BidDAO bidDAO;

    /* 
       Stan utrzymywany dla każdego żądania zawiera identyfikator wartości egzemplarza Item, 
	   z którym  użytkownik pracuje, egzemplarz Item po załadowaniu z bazy danych, aktualnie 
	   najwyższa kwotę oferty dla tego przedmiotu oraz nową kwotę wprowadzoną przez użytkownika.
     */
    long id;
    Item item;
    BigDecimal highestBidAmount;
    BigDecimal newBidAmount;

    public void setId(long id) {
        this.id = id;
        if (item == null) {
            item = itemDAO.findById(id);
            if (item == null)
                throw new EntityNotFoundException();
            highestBidAmount = itemDAO.getMaxBidAmount(item);
        }
    }

    // Inne p[roste gettery i settery...
    public long getId() {
        return id;
    }

    public Item getItem() {
        return item;
    }

    public BigDecimal getNewBidAmount() {
        return newBidAmount;
    }

    public void setNewBidAmount(BigDecimal newBidAmount) {
        this.newBidAmount = newBidAmount;
    }

    public BigDecimal getHighestBidAmount() {
        return highestBidAmount;
    }

    /* 
        Adnotacja @Transactional jest nowością wprowadzoną w Java EE 7 (począwszy od JTA 1.2)
		i podobna do adnotacji @TransactionAttribute komponentów EJB. Wewnętrznie interceptor 
		opakowuje wywołanie metody w kontekście transakcji systemowej — tak, jak dla metody EJB. 
     */
    @Transactional
    public String placeBid() {
        /* 
            Wykonanie pracy w ramach transakcji i zapisanie nowej oferty w bazie danych. 
			Zapobiega składaniu ofert równolegle. Należy połączyć kontekst utrwalania z transakcją. 
			Nie ma znaczenia, który obiekt DAO wywołamy, ponieważ wszystkie one współdzielą 
			ten sam obiekt <code>EntityManager</code> o zasięgu żądania.
 
        */
        itemDAO.joinTransaction();

        /* 
            Jeżeli w tym samym czasie, w którym użytkownik myślał nad transakcją, i oglądał 
			renderowaną stronę aukcji zostanie zatwierdzona inna transakcja z wyższą ofertą, 
			operacja zakończy się niepowodzeniem, a strona z aukcją zostanie wyrenderowana 
			jeszcze raz z odpowiednim komunikatem. 
         */
        if (!getItem().isValidBidAmount(
                getHighestBidAmount(),
                getNewBidAmount()
            )) {
            ValidationMessages.addFacesMessage("Auction.bid.TooLow");
            return null;
        }

        /* 
            Aby zapobiec równoległemu składaniu ofert, należy wymusić inkrementację wersji 
			egzemplarza Item przy każdej synchronizacji. Jeżeli w tym samym czasie działa inna 
			transakcja, która podczas wykonywania metody setId() ładuje egzemplarz Item 
			w tej samej wersji, i taką samą najwyższą ofertą, to jedna z transakcji wewnątrz 
			metody placeBid() musi zakończyć się niepowodzeniem. 
         */
        itemDAO.checkVersion(getItem(), true);

        bidDAO.makePersistent(new Bid(getNewBidAmount(), getItem()));

        /* 
            To jest prosta operacja komponentu JSF "przekieruj po wykonaniu operacji POST". 
			Dzięki niej użytkownicy mogą bezpiecznie ponownie załadować stronę po złożeniu oferty.
    	*/
        return "auction?id=" + getId() + "&faces-redirect=true";
    }

}
