package org.jpwh.test.filtering;

import org.hibernate.CallbackException;
import org.hibernate.EmptyInterceptor;
import org.hibernate.Session;
import org.hibernate.type.Type;
import org.jpwh.model.filtering.interceptor.AuditLogRecord;
import org.jpwh.model.filtering.interceptor.Auditable;

import java.io.Serializable;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

public class AuditLogInterceptor extends EmptyInterceptor {
    /* 
       Trzeba sięgnąc do bazy danych w celu zapisania dziennika inspekcji, dlatego ten interceptor 
       wymaga obiektu <code>Session</code> frameworka Hibernate. W każdym rekordzie dziennika 
       inspekcji trzeba zapisać identyfikator aktualnie zalogowanego użytkownika. 
       Zmienne <code>inserts</code> i <code>updates</code> to kolekcje, 
       w których ten interceptor będzie przechowywał swój wewnętrzny stan.
     */
    protected Session currentSession;
    protected Long currentUserId;
    protected Set<Auditable> inserts = new HashSet<Auditable>();
    protected Set<Auditable> updates = new HashSet<Auditable>();

    public void setCurrentSession(Session session) {
        this.currentSession = session;
    }

    public void setCurrentUserId(Long currentUserId) {
        this.currentUserId = currentUserId;
    }
    /* 
       Ta metoda jest wywoływana w chwili utrwalania egzemplarza encji.
     */
    public boolean onSave(Object entity, Serializable id,
                          Object[] state, String[] propertyNames, Type[] types)
        throws CallbackException {

        if (entity instanceof Auditable)
            inserts.add((Auditable)entity);

        return false; // Nie modyfikujemy stanu
    }

    /* 
         Ta metoda jest wywoływana w momencie, kiedy podczas synchronizacji kontekstu utrwalania 
         egzemplarz encji zostanie wykryty jako "zabrudzony".
     */
    public boolean onFlushDirty(Object entity, Serializable id,
                                Object[] currentState, Object[] previousState,
                                String[] propertyNames, Type[] types)
        throws CallbackException {

        if (entity instanceof Auditable)
            updates.add((Auditable)entity);

        return false; // Nie zmodyfikowaliśmy właściwości currentState
    }

    /* 
       Ta metoda jest wywoływana w czasie, gdy synchronizacja kontekstu utrwalania została zakończona.
       Tutaj zapisujemy rekordy dziennika inspekcji dla wszystkich operacji wstawiania i aktualizacji zebranych wcześniej.
     */
    public void postFlush(Iterator iterator) throws CallbackException {

        /* 
           Nie mamy prawa dostępu do oryginalnego kontekstu utrwalania, obiektu <code>Session</code>, 
           który w tym momencie uruchamia ten interceptor.
           Obiekt <code>Session</code> podczas wywołań interceptora jest w stanie kruchym.
           Hibernate zezwala na stworzenie nowego obiektu <code>Session</code>, 
           który dziedziczy pewne informacje po oryginalnym obiekcie <code>Session</code> 
           za pomocą metody <code>sessionWithOptions()</code>. 
           Tutaj nowy, tymczasowy obiekt <code>Session</code> działa w tej samej transakcji 
           i w tym samym połączeniu z bazą danych co oryginalny obiekt <code>Session</code>.

         */
        Session tempSession =
            currentSession.sessionWithOptions()
                .transactionContext()
                .connection()
                .openSession();

        try {
            /* 
                  Zapisujemy nowy obiekt <code>AuditLogRecord</code> dla każdej operacji wstawiania 
                  i aktualizacji z wykorzystaniem tymczasowego obiektu <code>Session</code>.             */
            for (Auditable entity : inserts) {
                tempSession.persist(
                    new AuditLogRecord("insert", entity, currentUserId)
                );
            }
            for (Auditable entity : updates) {
                tempSession.persist(
                    new AuditLogRecord("update", entity, currentUserId)
                );
            }

            /* 
               Synchronizujemy i zamykamy tymczasowy obiekt <code>Session</code>
               niezależnie od oryginalnego obiektu <code>Session</code>.
             */
            tempSession.flush();
        } finally {
            tempSession.close();
            inserts.clear();
            updates.clear();
        }
    }
}
